/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about the location of input data.
 * </p>
 * <p>
 * You must specify at least one of the following: <code>S3DataSource</code> or <code>SnsDataSource</code>.
 * </p>
 * <p>
 * Use <code>SnsDataSource</code> to specify an SNS input topic for a streaming labeling job. If you do not specify and
 * SNS input topic ARN, Ground Truth will create a one-time labeling job.
 * </p>
 * <p>
 * Use <code>S3DataSource</code> to specify an input manifest file for both streaming and one-time labeling jobs. Adding
 * an <code>S3DataSource</code> is optional if you use <code>SnsDataSource</code> to create a streaming labeling job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LabelingJobDataSource implements SdkPojo, Serializable,
        ToCopyableBuilder<LabelingJobDataSource.Builder, LabelingJobDataSource> {
    private static final SdkField<LabelingJobS3DataSource> S3_DATA_SOURCE_FIELD = SdkField
            .<LabelingJobS3DataSource> builder(MarshallingType.SDK_POJO).memberName("S3DataSource")
            .getter(getter(LabelingJobDataSource::s3DataSource)).setter(setter(Builder::s3DataSource))
            .constructor(LabelingJobS3DataSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3DataSource").build()).build();

    private static final SdkField<LabelingJobSnsDataSource> SNS_DATA_SOURCE_FIELD = SdkField
            .<LabelingJobSnsDataSource> builder(MarshallingType.SDK_POJO).memberName("SnsDataSource")
            .getter(getter(LabelingJobDataSource::snsDataSource)).setter(setter(Builder::snsDataSource))
            .constructor(LabelingJobSnsDataSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnsDataSource").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_DATA_SOURCE_FIELD,
            SNS_DATA_SOURCE_FIELD));

    private static final long serialVersionUID = 1L;

    private final LabelingJobS3DataSource s3DataSource;

    private final LabelingJobSnsDataSource snsDataSource;

    private LabelingJobDataSource(BuilderImpl builder) {
        this.s3DataSource = builder.s3DataSource;
        this.snsDataSource = builder.snsDataSource;
    }

    /**
     * <p>
     * The Amazon S3 location of the input data objects.
     * </p>
     * 
     * @return The Amazon S3 location of the input data objects.
     */
    public final LabelingJobS3DataSource s3DataSource() {
        return s3DataSource;
    }

    /**
     * <p>
     * An Amazon SNS data source used for streaming labeling jobs. To learn more, see <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/dg/sms-streaming-labeling-job.html#sms-streaming-how-it-works-send-data"
     * >Send Data to a Streaming Labeling Job</a>.
     * </p>
     * 
     * @return An Amazon SNS data source used for streaming labeling jobs. To learn more, see <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/sms-streaming-labeling-job.html#sms-streaming-how-it-works-send-data"
     *         >Send Data to a Streaming Labeling Job</a>.
     */
    public final LabelingJobSnsDataSource snsDataSource() {
        return snsDataSource;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3DataSource());
        hashCode = 31 * hashCode + Objects.hashCode(snsDataSource());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LabelingJobDataSource)) {
            return false;
        }
        LabelingJobDataSource other = (LabelingJobDataSource) obj;
        return Objects.equals(s3DataSource(), other.s3DataSource()) && Objects.equals(snsDataSource(), other.snsDataSource());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LabelingJobDataSource").add("S3DataSource", s3DataSource())
                .add("SnsDataSource", snsDataSource()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3DataSource":
            return Optional.ofNullable(clazz.cast(s3DataSource()));
        case "SnsDataSource":
            return Optional.ofNullable(clazz.cast(snsDataSource()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LabelingJobDataSource, T> g) {
        return obj -> g.apply((LabelingJobDataSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LabelingJobDataSource> {
        /**
         * <p>
         * The Amazon S3 location of the input data objects.
         * </p>
         * 
         * @param s3DataSource
         *        The Amazon S3 location of the input data objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3DataSource(LabelingJobS3DataSource s3DataSource);

        /**
         * <p>
         * The Amazon S3 location of the input data objects.
         * </p>
         * This is a convenience that creates an instance of the {@link LabelingJobS3DataSource.Builder} avoiding the
         * need to create one manually via {@link LabelingJobS3DataSource#builder()}.
         *
         * When the {@link Consumer} completes, {@link LabelingJobS3DataSource.Builder#build()} is called immediately
         * and its result is passed to {@link #s3DataSource(LabelingJobS3DataSource)}.
         * 
         * @param s3DataSource
         *        a consumer that will call methods on {@link LabelingJobS3DataSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3DataSource(LabelingJobS3DataSource)
         */
        default Builder s3DataSource(Consumer<LabelingJobS3DataSource.Builder> s3DataSource) {
            return s3DataSource(LabelingJobS3DataSource.builder().applyMutation(s3DataSource).build());
        }

        /**
         * <p>
         * An Amazon SNS data source used for streaming labeling jobs. To learn more, see <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/sms-streaming-labeling-job.html#sms-streaming-how-it-works-send-data"
         * >Send Data to a Streaming Labeling Job</a>.
         * </p>
         * 
         * @param snsDataSource
         *        An Amazon SNS data source used for streaming labeling jobs. To learn more, see <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/sms-streaming-labeling-job.html#sms-streaming-how-it-works-send-data"
         *        >Send Data to a Streaming Labeling Job</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsDataSource(LabelingJobSnsDataSource snsDataSource);

        /**
         * <p>
         * An Amazon SNS data source used for streaming labeling jobs. To learn more, see <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/sms-streaming-labeling-job.html#sms-streaming-how-it-works-send-data"
         * >Send Data to a Streaming Labeling Job</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link LabelingJobSnsDataSource.Builder} avoiding the
         * need to create one manually via {@link LabelingJobSnsDataSource#builder()}.
         *
         * When the {@link Consumer} completes, {@link LabelingJobSnsDataSource.Builder#build()} is called immediately
         * and its result is passed to {@link #snsDataSource(LabelingJobSnsDataSource)}.
         * 
         * @param snsDataSource
         *        a consumer that will call methods on {@link LabelingJobSnsDataSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #snsDataSource(LabelingJobSnsDataSource)
         */
        default Builder snsDataSource(Consumer<LabelingJobSnsDataSource.Builder> snsDataSource) {
            return snsDataSource(LabelingJobSnsDataSource.builder().applyMutation(snsDataSource).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private LabelingJobS3DataSource s3DataSource;

        private LabelingJobSnsDataSource snsDataSource;

        private BuilderImpl() {
        }

        private BuilderImpl(LabelingJobDataSource model) {
            s3DataSource(model.s3DataSource);
            snsDataSource(model.snsDataSource);
        }

        public final LabelingJobS3DataSource.Builder getS3DataSource() {
            return s3DataSource != null ? s3DataSource.toBuilder() : null;
        }

        @Override
        public final Builder s3DataSource(LabelingJobS3DataSource s3DataSource) {
            this.s3DataSource = s3DataSource;
            return this;
        }

        public final void setS3DataSource(LabelingJobS3DataSource.BuilderImpl s3DataSource) {
            this.s3DataSource = s3DataSource != null ? s3DataSource.build() : null;
        }

        public final LabelingJobSnsDataSource.Builder getSnsDataSource() {
            return snsDataSource != null ? snsDataSource.toBuilder() : null;
        }

        @Override
        public final Builder snsDataSource(LabelingJobSnsDataSource snsDataSource) {
            this.snsDataSource = snsDataSource;
            return this;
        }

        public final void setSnsDataSource(LabelingJobSnsDataSource.BuilderImpl snsDataSource) {
            this.snsDataSource = snsDataSource != null ? snsDataSource.build() : null;
        }

        @Override
        public LabelingJobDataSource build() {
            return new LabelingJobDataSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
