/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration information for the Debugger hook parameters, metric and tensor collections, and storage paths. To
 * learn more about how to configure the <code>DebugHookConfig</code> parameter, see <a
 * href="https://docs.aws.amazon.com/sagemaker/latest/dg/debugger-createtrainingjob-api.html">Use the SageMaker and
 * Debugger Configuration API Operations to Create, Update, and Debug Your Training Job</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DebugHookConfig implements SdkPojo, Serializable, ToCopyableBuilder<DebugHookConfig.Builder, DebugHookConfig> {
    private static final SdkField<String> LOCAL_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LocalPath").getter(getter(DebugHookConfig::localPath)).setter(setter(Builder::localPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocalPath").build()).build();

    private static final SdkField<String> S3_OUTPUT_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3OutputPath").getter(getter(DebugHookConfig::s3OutputPath)).setter(setter(Builder::s3OutputPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3OutputPath").build()).build();

    private static final SdkField<Map<String, String>> HOOK_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("HookParameters")
            .getter(getter(DebugHookConfig::hookParameters))
            .setter(setter(Builder::hookParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HookParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<CollectionConfiguration>> COLLECTION_CONFIGURATIONS_FIELD = SdkField
            .<List<CollectionConfiguration>> builder(MarshallingType.LIST)
            .memberName("CollectionConfigurations")
            .getter(getter(DebugHookConfig::collectionConfigurations))
            .setter(setter(Builder::collectionConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CollectionConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CollectionConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(CollectionConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOCAL_PATH_FIELD,
            S3_OUTPUT_PATH_FIELD, HOOK_PARAMETERS_FIELD, COLLECTION_CONFIGURATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String localPath;

    private final String s3OutputPath;

    private final Map<String, String> hookParameters;

    private final List<CollectionConfiguration> collectionConfigurations;

    private DebugHookConfig(BuilderImpl builder) {
        this.localPath = builder.localPath;
        this.s3OutputPath = builder.s3OutputPath;
        this.hookParameters = builder.hookParameters;
        this.collectionConfigurations = builder.collectionConfigurations;
    }

    /**
     * <p>
     * Path to local storage location for metrics and tensors. Defaults to <code>/opt/ml/output/tensors/</code>.
     * </p>
     * 
     * @return Path to local storage location for metrics and tensors. Defaults to <code>/opt/ml/output/tensors/</code>.
     */
    public final String localPath() {
        return localPath;
    }

    /**
     * <p>
     * Path to Amazon S3 storage location for metrics and tensors.
     * </p>
     * 
     * @return Path to Amazon S3 storage location for metrics and tensors.
     */
    public final String s3OutputPath() {
        return s3OutputPath;
    }

    /**
     * For responses, this returns true if the service returned a value for the HookParameters property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasHookParameters() {
        return hookParameters != null && !(hookParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Configuration information for the Debugger hook parameters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHookParameters} method.
     * </p>
     * 
     * @return Configuration information for the Debugger hook parameters.
     */
    public final Map<String, String> hookParameters() {
        return hookParameters;
    }

    /**
     * For responses, this returns true if the service returned a value for the CollectionConfigurations property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCollectionConfigurations() {
        return collectionConfigurations != null && !(collectionConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Configuration information for Debugger tensor collections. To learn more about how to configure the
     * <code>CollectionConfiguration</code> parameter, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/debugger-createtrainingjob-api.html">Use the SageMaker and
     * Debugger Configuration API Operations to Create, Update, and Debug Your Training Job</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCollectionConfigurations} method.
     * </p>
     * 
     * @return Configuration information for Debugger tensor collections. To learn more about how to configure the
     *         <code>CollectionConfiguration</code> parameter, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/debugger-createtrainingjob-api.html">Use the
     *         SageMaker and Debugger Configuration API Operations to Create, Update, and Debug Your Training Job</a>.
     */
    public final List<CollectionConfiguration> collectionConfigurations() {
        return collectionConfigurations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(localPath());
        hashCode = 31 * hashCode + Objects.hashCode(s3OutputPath());
        hashCode = 31 * hashCode + Objects.hashCode(hasHookParameters() ? hookParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCollectionConfigurations() ? collectionConfigurations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DebugHookConfig)) {
            return false;
        }
        DebugHookConfig other = (DebugHookConfig) obj;
        return Objects.equals(localPath(), other.localPath()) && Objects.equals(s3OutputPath(), other.s3OutputPath())
                && hasHookParameters() == other.hasHookParameters() && Objects.equals(hookParameters(), other.hookParameters())
                && hasCollectionConfigurations() == other.hasCollectionConfigurations()
                && Objects.equals(collectionConfigurations(), other.collectionConfigurations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DebugHookConfig").add("LocalPath", localPath()).add("S3OutputPath", s3OutputPath())
                .add("HookParameters", hasHookParameters() ? hookParameters() : null)
                .add("CollectionConfigurations", hasCollectionConfigurations() ? collectionConfigurations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LocalPath":
            return Optional.ofNullable(clazz.cast(localPath()));
        case "S3OutputPath":
            return Optional.ofNullable(clazz.cast(s3OutputPath()));
        case "HookParameters":
            return Optional.ofNullable(clazz.cast(hookParameters()));
        case "CollectionConfigurations":
            return Optional.ofNullable(clazz.cast(collectionConfigurations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DebugHookConfig, T> g) {
        return obj -> g.apply((DebugHookConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DebugHookConfig> {
        /**
         * <p>
         * Path to local storage location for metrics and tensors. Defaults to <code>/opt/ml/output/tensors/</code>.
         * </p>
         * 
         * @param localPath
         *        Path to local storage location for metrics and tensors. Defaults to
         *        <code>/opt/ml/output/tensors/</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localPath(String localPath);

        /**
         * <p>
         * Path to Amazon S3 storage location for metrics and tensors.
         * </p>
         * 
         * @param s3OutputPath
         *        Path to Amazon S3 storage location for metrics and tensors.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3OutputPath(String s3OutputPath);

        /**
         * <p>
         * Configuration information for the Debugger hook parameters.
         * </p>
         * 
         * @param hookParameters
         *        Configuration information for the Debugger hook parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hookParameters(Map<String, String> hookParameters);

        /**
         * <p>
         * Configuration information for Debugger tensor collections. To learn more about how to configure the
         * <code>CollectionConfiguration</code> parameter, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/debugger-createtrainingjob-api.html">Use the SageMaker
         * and Debugger Configuration API Operations to Create, Update, and Debug Your Training Job</a>.
         * </p>
         * 
         * @param collectionConfigurations
         *        Configuration information for Debugger tensor collections. To learn more about how to configure the
         *        <code>CollectionConfiguration</code> parameter, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/debugger-createtrainingjob-api.html">Use the
         *        SageMaker and Debugger Configuration API Operations to Create, Update, and Debug Your Training
         *        Job</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder collectionConfigurations(Collection<CollectionConfiguration> collectionConfigurations);

        /**
         * <p>
         * Configuration information for Debugger tensor collections. To learn more about how to configure the
         * <code>CollectionConfiguration</code> parameter, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/debugger-createtrainingjob-api.html">Use the SageMaker
         * and Debugger Configuration API Operations to Create, Update, and Debug Your Training Job</a>.
         * </p>
         * 
         * @param collectionConfigurations
         *        Configuration information for Debugger tensor collections. To learn more about how to configure the
         *        <code>CollectionConfiguration</code> parameter, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/debugger-createtrainingjob-api.html">Use the
         *        SageMaker and Debugger Configuration API Operations to Create, Update, and Debug Your Training
         *        Job</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder collectionConfigurations(CollectionConfiguration... collectionConfigurations);

        /**
         * <p>
         * Configuration information for Debugger tensor collections. To learn more about how to configure the
         * <code>CollectionConfiguration</code> parameter, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/debugger-createtrainingjob-api.html">Use the SageMaker
         * and Debugger Configuration API Operations to Create, Update, and Debug Your Training Job</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<CollectionConfiguration>.Builder} avoiding
         * the need to create one manually via {@link List<CollectionConfiguration>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<CollectionConfiguration>.Builder#build()} is called
         * immediately and its result is passed to {@link #collectionConfigurations(List<CollectionConfiguration>)}.
         * 
         * @param collectionConfigurations
         *        a consumer that will call methods on {@link List<CollectionConfiguration>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #collectionConfigurations(List<CollectionConfiguration>)
         */
        Builder collectionConfigurations(Consumer<CollectionConfiguration.Builder>... collectionConfigurations);
    }

    static final class BuilderImpl implements Builder {
        private String localPath;

        private String s3OutputPath;

        private Map<String, String> hookParameters = DefaultSdkAutoConstructMap.getInstance();

        private List<CollectionConfiguration> collectionConfigurations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DebugHookConfig model) {
            localPath(model.localPath);
            s3OutputPath(model.s3OutputPath);
            hookParameters(model.hookParameters);
            collectionConfigurations(model.collectionConfigurations);
        }

        public final String getLocalPath() {
            return localPath;
        }

        public final void setLocalPath(String localPath) {
            this.localPath = localPath;
        }

        @Override
        @Transient
        public final Builder localPath(String localPath) {
            this.localPath = localPath;
            return this;
        }

        public final String getS3OutputPath() {
            return s3OutputPath;
        }

        public final void setS3OutputPath(String s3OutputPath) {
            this.s3OutputPath = s3OutputPath;
        }

        @Override
        @Transient
        public final Builder s3OutputPath(String s3OutputPath) {
            this.s3OutputPath = s3OutputPath;
            return this;
        }

        public final Map<String, String> getHookParameters() {
            if (hookParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return hookParameters;
        }

        public final void setHookParameters(Map<String, String> hookParameters) {
            this.hookParameters = HookParametersCopier.copy(hookParameters);
        }

        @Override
        @Transient
        public final Builder hookParameters(Map<String, String> hookParameters) {
            this.hookParameters = HookParametersCopier.copy(hookParameters);
            return this;
        }

        public final List<CollectionConfiguration.Builder> getCollectionConfigurations() {
            List<CollectionConfiguration.Builder> result = CollectionConfigurationsCopier
                    .copyToBuilder(this.collectionConfigurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCollectionConfigurations(Collection<CollectionConfiguration.BuilderImpl> collectionConfigurations) {
            this.collectionConfigurations = CollectionConfigurationsCopier.copyFromBuilder(collectionConfigurations);
        }

        @Override
        @Transient
        public final Builder collectionConfigurations(Collection<CollectionConfiguration> collectionConfigurations) {
            this.collectionConfigurations = CollectionConfigurationsCopier.copy(collectionConfigurations);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder collectionConfigurations(CollectionConfiguration... collectionConfigurations) {
            collectionConfigurations(Arrays.asList(collectionConfigurations));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder collectionConfigurations(Consumer<CollectionConfiguration.Builder>... collectionConfigurations) {
            collectionConfigurations(Stream.of(collectionConfigurations)
                    .map(c -> CollectionConfiguration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public DebugHookConfig build() {
            return new DebugHookConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
