/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration information for updating the Debugger profile parameters, system and framework metrics configurations,
 * and storage paths.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProfilerConfigForUpdate implements SdkPojo, Serializable,
        ToCopyableBuilder<ProfilerConfigForUpdate.Builder, ProfilerConfigForUpdate> {
    private static final SdkField<String> S3_OUTPUT_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3OutputPath").getter(getter(ProfilerConfigForUpdate::s3OutputPath))
            .setter(setter(Builder::s3OutputPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3OutputPath").build()).build();

    private static final SdkField<Long> PROFILING_INTERVAL_IN_MILLISECONDS_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("ProfilingIntervalInMilliseconds")
            .getter(getter(ProfilerConfigForUpdate::profilingIntervalInMilliseconds))
            .setter(setter(Builder::profilingIntervalInMilliseconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProfilingIntervalInMilliseconds")
                    .build()).build();

    private static final SdkField<Map<String, String>> PROFILING_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ProfilingParameters")
            .getter(getter(ProfilerConfigForUpdate::profilingParameters))
            .setter(setter(Builder::profilingParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProfilingParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Boolean> DISABLE_PROFILER_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DisableProfiler").getter(getter(ProfilerConfigForUpdate::disableProfiler))
            .setter(setter(Builder::disableProfiler))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisableProfiler").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_OUTPUT_PATH_FIELD,
            PROFILING_INTERVAL_IN_MILLISECONDS_FIELD, PROFILING_PARAMETERS_FIELD, DISABLE_PROFILER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String s3OutputPath;

    private final Long profilingIntervalInMilliseconds;

    private final Map<String, String> profilingParameters;

    private final Boolean disableProfiler;

    private ProfilerConfigForUpdate(BuilderImpl builder) {
        this.s3OutputPath = builder.s3OutputPath;
        this.profilingIntervalInMilliseconds = builder.profilingIntervalInMilliseconds;
        this.profilingParameters = builder.profilingParameters;
        this.disableProfiler = builder.disableProfiler;
    }

    /**
     * <p>
     * Path to Amazon S3 storage location for system and framework metrics.
     * </p>
     * 
     * @return Path to Amazon S3 storage location for system and framework metrics.
     */
    public final String s3OutputPath() {
        return s3OutputPath;
    }

    /**
     * <p>
     * A time interval for capturing system metrics in milliseconds. Available values are 100, 200, 500, 1000 (1
     * second), 5000 (5 seconds), and 60000 (1 minute) milliseconds. The default value is 500 milliseconds.
     * </p>
     * 
     * @return A time interval for capturing system metrics in milliseconds. Available values are 100, 200, 500, 1000 (1
     *         second), 5000 (5 seconds), and 60000 (1 minute) milliseconds. The default value is 500 milliseconds.
     */
    public final Long profilingIntervalInMilliseconds() {
        return profilingIntervalInMilliseconds;
    }

    /**
     * For responses, this returns true if the service returned a value for the ProfilingParameters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasProfilingParameters() {
        return profilingParameters != null && !(profilingParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Configuration information for capturing framework metrics. Available key strings for different profiling options
     * are <code>DetailedProfilingConfig</code>, <code>PythonProfilingConfig</code>, and
     * <code>DataLoaderProfilingConfig</code>. The following codes are configuration structures for the
     * <code>ProfilingParameters</code> parameter. To learn more about how to configure the
     * <code>ProfilingParameters</code> parameter, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/debugger-createtrainingjob-api.html">Use the SageMaker and
     * Debugger Configuration API Operations to Create, Update, and Debug Your Training Job</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProfilingParameters} method.
     * </p>
     * 
     * @return Configuration information for capturing framework metrics. Available key strings for different profiling
     *         options are <code>DetailedProfilingConfig</code>, <code>PythonProfilingConfig</code>, and
     *         <code>DataLoaderProfilingConfig</code>. The following codes are configuration structures for the
     *         <code>ProfilingParameters</code> parameter. To learn more about how to configure the
     *         <code>ProfilingParameters</code> parameter, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/debugger-createtrainingjob-api.html">Use the
     *         SageMaker and Debugger Configuration API Operations to Create, Update, and Debug Your Training Job</a>.
     */
    public final Map<String, String> profilingParameters() {
        return profilingParameters;
    }

    /**
     * <p>
     * To disable Debugger monitoring and profiling, set to <code>True</code>.
     * </p>
     * 
     * @return To disable Debugger monitoring and profiling, set to <code>True</code>.
     */
    public final Boolean disableProfiler() {
        return disableProfiler;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3OutputPath());
        hashCode = 31 * hashCode + Objects.hashCode(profilingIntervalInMilliseconds());
        hashCode = 31 * hashCode + Objects.hashCode(hasProfilingParameters() ? profilingParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(disableProfiler());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProfilerConfigForUpdate)) {
            return false;
        }
        ProfilerConfigForUpdate other = (ProfilerConfigForUpdate) obj;
        return Objects.equals(s3OutputPath(), other.s3OutputPath())
                && Objects.equals(profilingIntervalInMilliseconds(), other.profilingIntervalInMilliseconds())
                && hasProfilingParameters() == other.hasProfilingParameters()
                && Objects.equals(profilingParameters(), other.profilingParameters())
                && Objects.equals(disableProfiler(), other.disableProfiler());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProfilerConfigForUpdate").add("S3OutputPath", s3OutputPath())
                .add("ProfilingIntervalInMilliseconds", profilingIntervalInMilliseconds())
                .add("ProfilingParameters", hasProfilingParameters() ? profilingParameters() : null)
                .add("DisableProfiler", disableProfiler()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3OutputPath":
            return Optional.ofNullable(clazz.cast(s3OutputPath()));
        case "ProfilingIntervalInMilliseconds":
            return Optional.ofNullable(clazz.cast(profilingIntervalInMilliseconds()));
        case "ProfilingParameters":
            return Optional.ofNullable(clazz.cast(profilingParameters()));
        case "DisableProfiler":
            return Optional.ofNullable(clazz.cast(disableProfiler()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProfilerConfigForUpdate, T> g) {
        return obj -> g.apply((ProfilerConfigForUpdate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProfilerConfigForUpdate> {
        /**
         * <p>
         * Path to Amazon S3 storage location for system and framework metrics.
         * </p>
         * 
         * @param s3OutputPath
         *        Path to Amazon S3 storage location for system and framework metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3OutputPath(String s3OutputPath);

        /**
         * <p>
         * A time interval for capturing system metrics in milliseconds. Available values are 100, 200, 500, 1000 (1
         * second), 5000 (5 seconds), and 60000 (1 minute) milliseconds. The default value is 500 milliseconds.
         * </p>
         * 
         * @param profilingIntervalInMilliseconds
         *        A time interval for capturing system metrics in milliseconds. Available values are 100, 200, 500, 1000
         *        (1 second), 5000 (5 seconds), and 60000 (1 minute) milliseconds. The default value is 500
         *        milliseconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder profilingIntervalInMilliseconds(Long profilingIntervalInMilliseconds);

        /**
         * <p>
         * Configuration information for capturing framework metrics. Available key strings for different profiling
         * options are <code>DetailedProfilingConfig</code>, <code>PythonProfilingConfig</code>, and
         * <code>DataLoaderProfilingConfig</code>. The following codes are configuration structures for the
         * <code>ProfilingParameters</code> parameter. To learn more about how to configure the
         * <code>ProfilingParameters</code> parameter, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/debugger-createtrainingjob-api.html">Use the SageMaker
         * and Debugger Configuration API Operations to Create, Update, and Debug Your Training Job</a>.
         * </p>
         * 
         * @param profilingParameters
         *        Configuration information for capturing framework metrics. Available key strings for different
         *        profiling options are <code>DetailedProfilingConfig</code>, <code>PythonProfilingConfig</code>, and
         *        <code>DataLoaderProfilingConfig</code>. The following codes are configuration structures for the
         *        <code>ProfilingParameters</code> parameter. To learn more about how to configure the
         *        <code>ProfilingParameters</code> parameter, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/debugger-createtrainingjob-api.html">Use the
         *        SageMaker and Debugger Configuration API Operations to Create, Update, and Debug Your Training
         *        Job</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder profilingParameters(Map<String, String> profilingParameters);

        /**
         * <p>
         * To disable Debugger monitoring and profiling, set to <code>True</code>.
         * </p>
         * 
         * @param disableProfiler
         *        To disable Debugger monitoring and profiling, set to <code>True</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disableProfiler(Boolean disableProfiler);
    }

    static final class BuilderImpl implements Builder {
        private String s3OutputPath;

        private Long profilingIntervalInMilliseconds;

        private Map<String, String> profilingParameters = DefaultSdkAutoConstructMap.getInstance();

        private Boolean disableProfiler;

        private BuilderImpl() {
        }

        private BuilderImpl(ProfilerConfigForUpdate model) {
            s3OutputPath(model.s3OutputPath);
            profilingIntervalInMilliseconds(model.profilingIntervalInMilliseconds);
            profilingParameters(model.profilingParameters);
            disableProfiler(model.disableProfiler);
        }

        public final String getS3OutputPath() {
            return s3OutputPath;
        }

        public final void setS3OutputPath(String s3OutputPath) {
            this.s3OutputPath = s3OutputPath;
        }

        @Override
        @Transient
        public final Builder s3OutputPath(String s3OutputPath) {
            this.s3OutputPath = s3OutputPath;
            return this;
        }

        public final Long getProfilingIntervalInMilliseconds() {
            return profilingIntervalInMilliseconds;
        }

        public final void setProfilingIntervalInMilliseconds(Long profilingIntervalInMilliseconds) {
            this.profilingIntervalInMilliseconds = profilingIntervalInMilliseconds;
        }

        @Override
        @Transient
        public final Builder profilingIntervalInMilliseconds(Long profilingIntervalInMilliseconds) {
            this.profilingIntervalInMilliseconds = profilingIntervalInMilliseconds;
            return this;
        }

        public final Map<String, String> getProfilingParameters() {
            if (profilingParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return profilingParameters;
        }

        public final void setProfilingParameters(Map<String, String> profilingParameters) {
            this.profilingParameters = ProfilingParametersCopier.copy(profilingParameters);
        }

        @Override
        @Transient
        public final Builder profilingParameters(Map<String, String> profilingParameters) {
            this.profilingParameters = ProfilingParametersCopier.copy(profilingParameters);
            return this;
        }

        public final Boolean getDisableProfiler() {
            return disableProfiler;
        }

        public final void setDisableProfiler(Boolean disableProfiler) {
            this.disableProfiler = disableProfiler;
        }

        @Override
        @Transient
        public final Builder disableProfiler(Boolean disableProfiler) {
            this.disableProfiler = disableProfiler;
            return this;
        }

        @Override
        public ProfilerConfigForUpdate build() {
            return new ProfilerConfigForUpdate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
