/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Identifies a model that you want to host and the resources chosen to deploy for hosting it. If you are deploying
 * multiple models, tell Amazon SageMaker how to distribute traffic among the models by specifying variant weights.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProductionVariant implements SdkPojo, Serializable,
        ToCopyableBuilder<ProductionVariant.Builder, ProductionVariant> {
    private static final SdkField<String> VARIANT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VariantName").getter(getter(ProductionVariant::variantName)).setter(setter(Builder::variantName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VariantName").build()).build();

    private static final SdkField<String> MODEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelName").getter(getter(ProductionVariant::modelName)).setter(setter(Builder::modelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelName").build()).build();

    private static final SdkField<Integer> INITIAL_INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("InitialInstanceCount").getter(getter(ProductionVariant::initialInstanceCount))
            .setter(setter(Builder::initialInstanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitialInstanceCount").build())
            .build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(ProductionVariant::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Float> INITIAL_VARIANT_WEIGHT_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("InitialVariantWeight").getter(getter(ProductionVariant::initialVariantWeight))
            .setter(setter(Builder::initialVariantWeight))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitialVariantWeight").build())
            .build();

    private static final SdkField<String> ACCELERATOR_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AcceleratorType").getter(getter(ProductionVariant::acceleratorTypeAsString))
            .setter(setter(Builder::acceleratorType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceleratorType").build()).build();

    private static final SdkField<ProductionVariantCoreDumpConfig> CORE_DUMP_CONFIG_FIELD = SdkField
            .<ProductionVariantCoreDumpConfig> builder(MarshallingType.SDK_POJO).memberName("CoreDumpConfig")
            .getter(getter(ProductionVariant::coreDumpConfig)).setter(setter(Builder::coreDumpConfig))
            .constructor(ProductionVariantCoreDumpConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CoreDumpConfig").build()).build();

    private static final SdkField<ProductionVariantServerlessConfig> SERVERLESS_CONFIG_FIELD = SdkField
            .<ProductionVariantServerlessConfig> builder(MarshallingType.SDK_POJO).memberName("ServerlessConfig")
            .getter(getter(ProductionVariant::serverlessConfig)).setter(setter(Builder::serverlessConfig))
            .constructor(ProductionVariantServerlessConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerlessConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VARIANT_NAME_FIELD,
            MODEL_NAME_FIELD, INITIAL_INSTANCE_COUNT_FIELD, INSTANCE_TYPE_FIELD, INITIAL_VARIANT_WEIGHT_FIELD,
            ACCELERATOR_TYPE_FIELD, CORE_DUMP_CONFIG_FIELD, SERVERLESS_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String variantName;

    private final String modelName;

    private final Integer initialInstanceCount;

    private final String instanceType;

    private final Float initialVariantWeight;

    private final String acceleratorType;

    private final ProductionVariantCoreDumpConfig coreDumpConfig;

    private final ProductionVariantServerlessConfig serverlessConfig;

    private ProductionVariant(BuilderImpl builder) {
        this.variantName = builder.variantName;
        this.modelName = builder.modelName;
        this.initialInstanceCount = builder.initialInstanceCount;
        this.instanceType = builder.instanceType;
        this.initialVariantWeight = builder.initialVariantWeight;
        this.acceleratorType = builder.acceleratorType;
        this.coreDumpConfig = builder.coreDumpConfig;
        this.serverlessConfig = builder.serverlessConfig;
    }

    /**
     * <p>
     * The name of the production variant.
     * </p>
     * 
     * @return The name of the production variant.
     */
    public final String variantName() {
        return variantName;
    }

    /**
     * <p>
     * The name of the model that you want to host. This is the name that you specified when creating the model.
     * </p>
     * 
     * @return The name of the model that you want to host. This is the name that you specified when creating the model.
     */
    public final String modelName() {
        return modelName;
    }

    /**
     * <p>
     * Number of instances to launch initially.
     * </p>
     * 
     * @return Number of instances to launch initially.
     */
    public final Integer initialInstanceCount() {
        return initialInstanceCount;
    }

    /**
     * <p>
     * The ML compute instance type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link ProductionVariantInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The ML compute instance type.
     * @see ProductionVariantInstanceType
     */
    public final ProductionVariantInstanceType instanceType() {
        return ProductionVariantInstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The ML compute instance type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link ProductionVariantInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The ML compute instance type.
     * @see ProductionVariantInstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * Determines initial traffic distribution among all of the models that you specify in the endpoint configuration.
     * The traffic to a production variant is determined by the ratio of the <code>VariantWeight</code> to the sum of
     * all <code>VariantWeight</code> values across all ProductionVariants. If unspecified, it defaults to 1.0.
     * </p>
     * 
     * @return Determines initial traffic distribution among all of the models that you specify in the endpoint
     *         configuration. The traffic to a production variant is determined by the ratio of the
     *         <code>VariantWeight</code> to the sum of all <code>VariantWeight</code> values across all
     *         ProductionVariants. If unspecified, it defaults to 1.0.
     */
    public final Float initialVariantWeight() {
        return initialVariantWeight;
    }

    /**
     * <p>
     * The size of the Elastic Inference (EI) instance to use for the production variant. EI instances provide on-demand
     * GPU computing for inference. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon SageMaker</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #acceleratorType}
     * will return {@link ProductionVariantAcceleratorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #acceleratorTypeAsString}.
     * </p>
     * 
     * @return The size of the Elastic Inference (EI) instance to use for the production variant. EI instances provide
     *         on-demand GPU computing for inference. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon
     *         SageMaker</a>.
     * @see ProductionVariantAcceleratorType
     */
    public final ProductionVariantAcceleratorType acceleratorType() {
        return ProductionVariantAcceleratorType.fromValue(acceleratorType);
    }

    /**
     * <p>
     * The size of the Elastic Inference (EI) instance to use for the production variant. EI instances provide on-demand
     * GPU computing for inference. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon SageMaker</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #acceleratorType}
     * will return {@link ProductionVariantAcceleratorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #acceleratorTypeAsString}.
     * </p>
     * 
     * @return The size of the Elastic Inference (EI) instance to use for the production variant. EI instances provide
     *         on-demand GPU computing for inference. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon
     *         SageMaker</a>.
     * @see ProductionVariantAcceleratorType
     */
    public final String acceleratorTypeAsString() {
        return acceleratorType;
    }

    /**
     * <p>
     * Specifies configuration for a core dump from the model container when the process crashes.
     * </p>
     * 
     * @return Specifies configuration for a core dump from the model container when the process crashes.
     */
    public final ProductionVariantCoreDumpConfig coreDumpConfig() {
        return coreDumpConfig;
    }

    /**
     * <p>
     * The serverless configuration for an endpoint. Specifies a serverless endpoint configuration instead of an
     * instance-based endpoint configuration.
     * </p>
     * <note>
     * <p>
     * Serverless Inference is in preview release for Amazon SageMaker and is subject to change. We do not recommend
     * using this feature in production environments.
     * </p>
     * </note>
     * 
     * @return The serverless configuration for an endpoint. Specifies a serverless endpoint configuration instead of an
     *         instance-based endpoint configuration.</p> <note>
     *         <p>
     *         Serverless Inference is in preview release for Amazon SageMaker and is subject to change. We do not
     *         recommend using this feature in production environments.
     *         </p>
     */
    public final ProductionVariantServerlessConfig serverlessConfig() {
        return serverlessConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(variantName());
        hashCode = 31 * hashCode + Objects.hashCode(modelName());
        hashCode = 31 * hashCode + Objects.hashCode(initialInstanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(initialVariantWeight());
        hashCode = 31 * hashCode + Objects.hashCode(acceleratorTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(coreDumpConfig());
        hashCode = 31 * hashCode + Objects.hashCode(serverlessConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProductionVariant)) {
            return false;
        }
        ProductionVariant other = (ProductionVariant) obj;
        return Objects.equals(variantName(), other.variantName()) && Objects.equals(modelName(), other.modelName())
                && Objects.equals(initialInstanceCount(), other.initialInstanceCount())
                && Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(initialVariantWeight(), other.initialVariantWeight())
                && Objects.equals(acceleratorTypeAsString(), other.acceleratorTypeAsString())
                && Objects.equals(coreDumpConfig(), other.coreDumpConfig())
                && Objects.equals(serverlessConfig(), other.serverlessConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProductionVariant").add("VariantName", variantName()).add("ModelName", modelName())
                .add("InitialInstanceCount", initialInstanceCount()).add("InstanceType", instanceTypeAsString())
                .add("InitialVariantWeight", initialVariantWeight()).add("AcceleratorType", acceleratorTypeAsString())
                .add("CoreDumpConfig", coreDumpConfig()).add("ServerlessConfig", serverlessConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VariantName":
            return Optional.ofNullable(clazz.cast(variantName()));
        case "ModelName":
            return Optional.ofNullable(clazz.cast(modelName()));
        case "InitialInstanceCount":
            return Optional.ofNullable(clazz.cast(initialInstanceCount()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "InitialVariantWeight":
            return Optional.ofNullable(clazz.cast(initialVariantWeight()));
        case "AcceleratorType":
            return Optional.ofNullable(clazz.cast(acceleratorTypeAsString()));
        case "CoreDumpConfig":
            return Optional.ofNullable(clazz.cast(coreDumpConfig()));
        case "ServerlessConfig":
            return Optional.ofNullable(clazz.cast(serverlessConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProductionVariant, T> g) {
        return obj -> g.apply((ProductionVariant) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProductionVariant> {
        /**
         * <p>
         * The name of the production variant.
         * </p>
         * 
         * @param variantName
         *        The name of the production variant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variantName(String variantName);

        /**
         * <p>
         * The name of the model that you want to host. This is the name that you specified when creating the model.
         * </p>
         * 
         * @param modelName
         *        The name of the model that you want to host. This is the name that you specified when creating the
         *        model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelName(String modelName);

        /**
         * <p>
         * Number of instances to launch initially.
         * </p>
         * 
         * @param initialInstanceCount
         *        Number of instances to launch initially.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initialInstanceCount(Integer initialInstanceCount);

        /**
         * <p>
         * The ML compute instance type.
         * </p>
         * 
         * @param instanceType
         *        The ML compute instance type.
         * @see ProductionVariantInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantInstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The ML compute instance type.
         * </p>
         * 
         * @param instanceType
         *        The ML compute instance type.
         * @see ProductionVariantInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantInstanceType
         */
        Builder instanceType(ProductionVariantInstanceType instanceType);

        /**
         * <p>
         * Determines initial traffic distribution among all of the models that you specify in the endpoint
         * configuration. The traffic to a production variant is determined by the ratio of the
         * <code>VariantWeight</code> to the sum of all <code>VariantWeight</code> values across all ProductionVariants.
         * If unspecified, it defaults to 1.0.
         * </p>
         * 
         * @param initialVariantWeight
         *        Determines initial traffic distribution among all of the models that you specify in the endpoint
         *        configuration. The traffic to a production variant is determined by the ratio of the
         *        <code>VariantWeight</code> to the sum of all <code>VariantWeight</code> values across all
         *        ProductionVariants. If unspecified, it defaults to 1.0.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initialVariantWeight(Float initialVariantWeight);

        /**
         * <p>
         * The size of the Elastic Inference (EI) instance to use for the production variant. EI instances provide
         * on-demand GPU computing for inference. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon
         * SageMaker</a>.
         * </p>
         * 
         * @param acceleratorType
         *        The size of the Elastic Inference (EI) instance to use for the production variant. EI instances
         *        provide on-demand GPU computing for inference. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon
         *        SageMaker</a>.
         * @see ProductionVariantAcceleratorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantAcceleratorType
         */
        Builder acceleratorType(String acceleratorType);

        /**
         * <p>
         * The size of the Elastic Inference (EI) instance to use for the production variant. EI instances provide
         * on-demand GPU computing for inference. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon
         * SageMaker</a>.
         * </p>
         * 
         * @param acceleratorType
         *        The size of the Elastic Inference (EI) instance to use for the production variant. EI instances
         *        provide on-demand GPU computing for inference. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon
         *        SageMaker</a>.
         * @see ProductionVariantAcceleratorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantAcceleratorType
         */
        Builder acceleratorType(ProductionVariantAcceleratorType acceleratorType);

        /**
         * <p>
         * Specifies configuration for a core dump from the model container when the process crashes.
         * </p>
         * 
         * @param coreDumpConfig
         *        Specifies configuration for a core dump from the model container when the process crashes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder coreDumpConfig(ProductionVariantCoreDumpConfig coreDumpConfig);

        /**
         * <p>
         * Specifies configuration for a core dump from the model container when the process crashes.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProductionVariantCoreDumpConfig.Builder}
         * avoiding the need to create one manually via {@link ProductionVariantCoreDumpConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProductionVariantCoreDumpConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #coreDumpConfig(ProductionVariantCoreDumpConfig)}.
         * 
         * @param coreDumpConfig
         *        a consumer that will call methods on {@link ProductionVariantCoreDumpConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #coreDumpConfig(ProductionVariantCoreDumpConfig)
         */
        default Builder coreDumpConfig(Consumer<ProductionVariantCoreDumpConfig.Builder> coreDumpConfig) {
            return coreDumpConfig(ProductionVariantCoreDumpConfig.builder().applyMutation(coreDumpConfig).build());
        }

        /**
         * <p>
         * The serverless configuration for an endpoint. Specifies a serverless endpoint configuration instead of an
         * instance-based endpoint configuration.
         * </p>
         * <note>
         * <p>
         * Serverless Inference is in preview release for Amazon SageMaker and is subject to change. We do not recommend
         * using this feature in production environments.
         * </p>
         * </note>
         * 
         * @param serverlessConfig
         *        The serverless configuration for an endpoint. Specifies a serverless endpoint configuration instead of
         *        an instance-based endpoint configuration.</p> <note>
         *        <p>
         *        Serverless Inference is in preview release for Amazon SageMaker and is subject to change. We do not
         *        recommend using this feature in production environments.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverlessConfig(ProductionVariantServerlessConfig serverlessConfig);

        /**
         * <p>
         * The serverless configuration for an endpoint. Specifies a serverless endpoint configuration instead of an
         * instance-based endpoint configuration.
         * </p>
         * <note>
         * <p>
         * Serverless Inference is in preview release for Amazon SageMaker and is subject to change. We do not recommend
         * using this feature in production environments.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link ProductionVariantServerlessConfig.Builder} avoiding the need to create one manually via
         * {@link ProductionVariantServerlessConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProductionVariantServerlessConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #serverlessConfig(ProductionVariantServerlessConfig)}.
         * 
         * @param serverlessConfig
         *        a consumer that will call methods on {@link ProductionVariantServerlessConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serverlessConfig(ProductionVariantServerlessConfig)
         */
        default Builder serverlessConfig(Consumer<ProductionVariantServerlessConfig.Builder> serverlessConfig) {
            return serverlessConfig(ProductionVariantServerlessConfig.builder().applyMutation(serverlessConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String variantName;

        private String modelName;

        private Integer initialInstanceCount;

        private String instanceType;

        private Float initialVariantWeight;

        private String acceleratorType;

        private ProductionVariantCoreDumpConfig coreDumpConfig;

        private ProductionVariantServerlessConfig serverlessConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(ProductionVariant model) {
            variantName(model.variantName);
            modelName(model.modelName);
            initialInstanceCount(model.initialInstanceCount);
            instanceType(model.instanceType);
            initialVariantWeight(model.initialVariantWeight);
            acceleratorType(model.acceleratorType);
            coreDumpConfig(model.coreDumpConfig);
            serverlessConfig(model.serverlessConfig);
        }

        public final String getVariantName() {
            return variantName;
        }

        public final void setVariantName(String variantName) {
            this.variantName = variantName;
        }

        @Override
        @Transient
        public final Builder variantName(String variantName) {
            this.variantName = variantName;
            return this;
        }

        public final String getModelName() {
            return modelName;
        }

        public final void setModelName(String modelName) {
            this.modelName = modelName;
        }

        @Override
        @Transient
        public final Builder modelName(String modelName) {
            this.modelName = modelName;
            return this;
        }

        public final Integer getInitialInstanceCount() {
            return initialInstanceCount;
        }

        public final void setInitialInstanceCount(Integer initialInstanceCount) {
            this.initialInstanceCount = initialInstanceCount;
        }

        @Override
        @Transient
        public final Builder initialInstanceCount(Integer initialInstanceCount) {
            this.initialInstanceCount = initialInstanceCount;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        @Transient
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        @Transient
        public final Builder instanceType(ProductionVariantInstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final Float getInitialVariantWeight() {
            return initialVariantWeight;
        }

        public final void setInitialVariantWeight(Float initialVariantWeight) {
            this.initialVariantWeight = initialVariantWeight;
        }

        @Override
        @Transient
        public final Builder initialVariantWeight(Float initialVariantWeight) {
            this.initialVariantWeight = initialVariantWeight;
            return this;
        }

        public final String getAcceleratorType() {
            return acceleratorType;
        }

        public final void setAcceleratorType(String acceleratorType) {
            this.acceleratorType = acceleratorType;
        }

        @Override
        @Transient
        public final Builder acceleratorType(String acceleratorType) {
            this.acceleratorType = acceleratorType;
            return this;
        }

        @Override
        @Transient
        public final Builder acceleratorType(ProductionVariantAcceleratorType acceleratorType) {
            this.acceleratorType(acceleratorType == null ? null : acceleratorType.toString());
            return this;
        }

        public final ProductionVariantCoreDumpConfig.Builder getCoreDumpConfig() {
            return coreDumpConfig != null ? coreDumpConfig.toBuilder() : null;
        }

        public final void setCoreDumpConfig(ProductionVariantCoreDumpConfig.BuilderImpl coreDumpConfig) {
            this.coreDumpConfig = coreDumpConfig != null ? coreDumpConfig.build() : null;
        }

        @Override
        @Transient
        public final Builder coreDumpConfig(ProductionVariantCoreDumpConfig coreDumpConfig) {
            this.coreDumpConfig = coreDumpConfig;
            return this;
        }

        public final ProductionVariantServerlessConfig.Builder getServerlessConfig() {
            return serverlessConfig != null ? serverlessConfig.toBuilder() : null;
        }

        public final void setServerlessConfig(ProductionVariantServerlessConfig.BuilderImpl serverlessConfig) {
            this.serverlessConfig = serverlessConfig != null ? serverlessConfig.build() : null;
        }

        @Override
        @Transient
        public final Builder serverlessConfig(ProductionVariantServerlessConfig serverlessConfig) {
            this.serverlessConfig = serverlessConfig;
            return this;
        }

        @Override
        public ProductionVariant build() {
            return new ProductionVariant(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
