/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines a named input source, called a channel, to be used by an algorithm.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ChannelSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<ChannelSpecification.Builder, ChannelSpecification> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ChannelSpecification::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ChannelSpecification::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Boolean> IS_REQUIRED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsRequired").getter(getter(ChannelSpecification::isRequired)).setter(setter(Builder::isRequired))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsRequired").build()).build();

    private static final SdkField<List<String>> SUPPORTED_CONTENT_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SupportedContentTypes")
            .getter(getter(ChannelSpecification::supportedContentTypes))
            .setter(setter(Builder::supportedContentTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedContentTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SUPPORTED_COMPRESSION_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SupportedCompressionTypes")
            .getter(getter(ChannelSpecification::supportedCompressionTypesAsStrings))
            .setter(setter(Builder::supportedCompressionTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedCompressionTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SUPPORTED_INPUT_MODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SupportedInputModes")
            .getter(getter(ChannelSpecification::supportedInputModesAsStrings))
            .setter(setter(Builder::supportedInputModesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedInputModes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            IS_REQUIRED_FIELD, SUPPORTED_CONTENT_TYPES_FIELD, SUPPORTED_COMPRESSION_TYPES_FIELD, SUPPORTED_INPUT_MODES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final Boolean isRequired;

    private final List<String> supportedContentTypes;

    private final List<String> supportedCompressionTypes;

    private final List<String> supportedInputModes;

    private ChannelSpecification(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.isRequired = builder.isRequired;
        this.supportedContentTypes = builder.supportedContentTypes;
        this.supportedCompressionTypes = builder.supportedCompressionTypes;
        this.supportedInputModes = builder.supportedInputModes;
    }

    /**
     * <p>
     * The name of the channel.
     * </p>
     * 
     * @return The name of the channel.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A brief description of the channel.
     * </p>
     * 
     * @return A brief description of the channel.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Indicates whether the channel is required by the algorithm.
     * </p>
     * 
     * @return Indicates whether the channel is required by the algorithm.
     */
    public final Boolean isRequired() {
        return isRequired;
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedContentTypes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSupportedContentTypes() {
        return supportedContentTypes != null && !(supportedContentTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The supported MIME types for the data.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedContentTypes} method.
     * </p>
     * 
     * @return The supported MIME types for the data.
     */
    public final List<String> supportedContentTypes() {
        return supportedContentTypes;
    }

    /**
     * <p>
     * The allowed compression types, if data compression is used.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedCompressionTypes} method.
     * </p>
     * 
     * @return The allowed compression types, if data compression is used.
     */
    public final List<CompressionType> supportedCompressionTypes() {
        return CompressionTypesCopier.copyStringToEnum(supportedCompressionTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedCompressionTypes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSupportedCompressionTypes() {
        return supportedCompressionTypes != null && !(supportedCompressionTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The allowed compression types, if data compression is used.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedCompressionTypes} method.
     * </p>
     * 
     * @return The allowed compression types, if data compression is used.
     */
    public final List<String> supportedCompressionTypesAsStrings() {
        return supportedCompressionTypes;
    }

    /**
     * <p>
     * The allowed input mode, either FILE or PIPE.
     * </p>
     * <p>
     * In FILE mode, Amazon SageMaker copies the data from the input source onto the local Amazon Elastic Block Store
     * (Amazon EBS) volumes before starting your training algorithm. This is the most commonly used input mode.
     * </p>
     * <p>
     * In PIPE mode, Amazon SageMaker streams input data from the source directly to your algorithm without using the
     * EBS volume.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedInputModes} method.
     * </p>
     * 
     * @return The allowed input mode, either FILE or PIPE.</p>
     *         <p>
     *         In FILE mode, Amazon SageMaker copies the data from the input source onto the local Amazon Elastic Block
     *         Store (Amazon EBS) volumes before starting your training algorithm. This is the most commonly used input
     *         mode.
     *         </p>
     *         <p>
     *         In PIPE mode, Amazon SageMaker streams input data from the source directly to your algorithm without
     *         using the EBS volume.
     */
    public final List<TrainingInputMode> supportedInputModes() {
        return InputModesCopier.copyStringToEnum(supportedInputModes);
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedInputModes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSupportedInputModes() {
        return supportedInputModes != null && !(supportedInputModes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The allowed input mode, either FILE or PIPE.
     * </p>
     * <p>
     * In FILE mode, Amazon SageMaker copies the data from the input source onto the local Amazon Elastic Block Store
     * (Amazon EBS) volumes before starting your training algorithm. This is the most commonly used input mode.
     * </p>
     * <p>
     * In PIPE mode, Amazon SageMaker streams input data from the source directly to your algorithm without using the
     * EBS volume.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedInputModes} method.
     * </p>
     * 
     * @return The allowed input mode, either FILE or PIPE.</p>
     *         <p>
     *         In FILE mode, Amazon SageMaker copies the data from the input source onto the local Amazon Elastic Block
     *         Store (Amazon EBS) volumes before starting your training algorithm. This is the most commonly used input
     *         mode.
     *         </p>
     *         <p>
     *         In PIPE mode, Amazon SageMaker streams input data from the source directly to your algorithm without
     *         using the EBS volume.
     */
    public final List<String> supportedInputModesAsStrings() {
        return supportedInputModes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(isRequired());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedContentTypes() ? supportedContentTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedCompressionTypes() ? supportedCompressionTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedInputModes() ? supportedInputModesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ChannelSpecification)) {
            return false;
        }
        ChannelSpecification other = (ChannelSpecification) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(isRequired(), other.isRequired())
                && hasSupportedContentTypes() == other.hasSupportedContentTypes()
                && Objects.equals(supportedContentTypes(), other.supportedContentTypes())
                && hasSupportedCompressionTypes() == other.hasSupportedCompressionTypes()
                && Objects.equals(supportedCompressionTypesAsStrings(), other.supportedCompressionTypesAsStrings())
                && hasSupportedInputModes() == other.hasSupportedInputModes()
                && Objects.equals(supportedInputModesAsStrings(), other.supportedInputModesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ChannelSpecification").add("Name", name()).add("Description", description())
                .add("IsRequired", isRequired())
                .add("SupportedContentTypes", hasSupportedContentTypes() ? supportedContentTypes() : null)
                .add("SupportedCompressionTypes", hasSupportedCompressionTypes() ? supportedCompressionTypesAsStrings() : null)
                .add("SupportedInputModes", hasSupportedInputModes() ? supportedInputModesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "IsRequired":
            return Optional.ofNullable(clazz.cast(isRequired()));
        case "SupportedContentTypes":
            return Optional.ofNullable(clazz.cast(supportedContentTypes()));
        case "SupportedCompressionTypes":
            return Optional.ofNullable(clazz.cast(supportedCompressionTypesAsStrings()));
        case "SupportedInputModes":
            return Optional.ofNullable(clazz.cast(supportedInputModesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ChannelSpecification, T> g) {
        return obj -> g.apply((ChannelSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ChannelSpecification> {
        /**
         * <p>
         * The name of the channel.
         * </p>
         * 
         * @param name
         *        The name of the channel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A brief description of the channel.
         * </p>
         * 
         * @param description
         *        A brief description of the channel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Indicates whether the channel is required by the algorithm.
         * </p>
         * 
         * @param isRequired
         *        Indicates whether the channel is required by the algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isRequired(Boolean isRequired);

        /**
         * <p>
         * The supported MIME types for the data.
         * </p>
         * 
         * @param supportedContentTypes
         *        The supported MIME types for the data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedContentTypes(Collection<String> supportedContentTypes);

        /**
         * <p>
         * The supported MIME types for the data.
         * </p>
         * 
         * @param supportedContentTypes
         *        The supported MIME types for the data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedContentTypes(String... supportedContentTypes);

        /**
         * <p>
         * The allowed compression types, if data compression is used.
         * </p>
         * 
         * @param supportedCompressionTypes
         *        The allowed compression types, if data compression is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedCompressionTypesWithStrings(Collection<String> supportedCompressionTypes);

        /**
         * <p>
         * The allowed compression types, if data compression is used.
         * </p>
         * 
         * @param supportedCompressionTypes
         *        The allowed compression types, if data compression is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedCompressionTypesWithStrings(String... supportedCompressionTypes);

        /**
         * <p>
         * The allowed compression types, if data compression is used.
         * </p>
         * 
         * @param supportedCompressionTypes
         *        The allowed compression types, if data compression is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedCompressionTypes(Collection<CompressionType> supportedCompressionTypes);

        /**
         * <p>
         * The allowed compression types, if data compression is used.
         * </p>
         * 
         * @param supportedCompressionTypes
         *        The allowed compression types, if data compression is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedCompressionTypes(CompressionType... supportedCompressionTypes);

        /**
         * <p>
         * The allowed input mode, either FILE or PIPE.
         * </p>
         * <p>
         * In FILE mode, Amazon SageMaker copies the data from the input source onto the local Amazon Elastic Block
         * Store (Amazon EBS) volumes before starting your training algorithm. This is the most commonly used input
         * mode.
         * </p>
         * <p>
         * In PIPE mode, Amazon SageMaker streams input data from the source directly to your algorithm without using
         * the EBS volume.
         * </p>
         * 
         * @param supportedInputModes
         *        The allowed input mode, either FILE or PIPE.</p>
         *        <p>
         *        In FILE mode, Amazon SageMaker copies the data from the input source onto the local Amazon Elastic
         *        Block Store (Amazon EBS) volumes before starting your training algorithm. This is the most commonly
         *        used input mode.
         *        </p>
         *        <p>
         *        In PIPE mode, Amazon SageMaker streams input data from the source directly to your algorithm without
         *        using the EBS volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedInputModesWithStrings(Collection<String> supportedInputModes);

        /**
         * <p>
         * The allowed input mode, either FILE or PIPE.
         * </p>
         * <p>
         * In FILE mode, Amazon SageMaker copies the data from the input source onto the local Amazon Elastic Block
         * Store (Amazon EBS) volumes before starting your training algorithm. This is the most commonly used input
         * mode.
         * </p>
         * <p>
         * In PIPE mode, Amazon SageMaker streams input data from the source directly to your algorithm without using
         * the EBS volume.
         * </p>
         * 
         * @param supportedInputModes
         *        The allowed input mode, either FILE or PIPE.</p>
         *        <p>
         *        In FILE mode, Amazon SageMaker copies the data from the input source onto the local Amazon Elastic
         *        Block Store (Amazon EBS) volumes before starting your training algorithm. This is the most commonly
         *        used input mode.
         *        </p>
         *        <p>
         *        In PIPE mode, Amazon SageMaker streams input data from the source directly to your algorithm without
         *        using the EBS volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedInputModesWithStrings(String... supportedInputModes);

        /**
         * <p>
         * The allowed input mode, either FILE or PIPE.
         * </p>
         * <p>
         * In FILE mode, Amazon SageMaker copies the data from the input source onto the local Amazon Elastic Block
         * Store (Amazon EBS) volumes before starting your training algorithm. This is the most commonly used input
         * mode.
         * </p>
         * <p>
         * In PIPE mode, Amazon SageMaker streams input data from the source directly to your algorithm without using
         * the EBS volume.
         * </p>
         * 
         * @param supportedInputModes
         *        The allowed input mode, either FILE or PIPE.</p>
         *        <p>
         *        In FILE mode, Amazon SageMaker copies the data from the input source onto the local Amazon Elastic
         *        Block Store (Amazon EBS) volumes before starting your training algorithm. This is the most commonly
         *        used input mode.
         *        </p>
         *        <p>
         *        In PIPE mode, Amazon SageMaker streams input data from the source directly to your algorithm without
         *        using the EBS volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedInputModes(Collection<TrainingInputMode> supportedInputModes);

        /**
         * <p>
         * The allowed input mode, either FILE or PIPE.
         * </p>
         * <p>
         * In FILE mode, Amazon SageMaker copies the data from the input source onto the local Amazon Elastic Block
         * Store (Amazon EBS) volumes before starting your training algorithm. This is the most commonly used input
         * mode.
         * </p>
         * <p>
         * In PIPE mode, Amazon SageMaker streams input data from the source directly to your algorithm without using
         * the EBS volume.
         * </p>
         * 
         * @param supportedInputModes
         *        The allowed input mode, either FILE or PIPE.</p>
         *        <p>
         *        In FILE mode, Amazon SageMaker copies the data from the input source onto the local Amazon Elastic
         *        Block Store (Amazon EBS) volumes before starting your training algorithm. This is the most commonly
         *        used input mode.
         *        </p>
         *        <p>
         *        In PIPE mode, Amazon SageMaker streams input data from the source directly to your algorithm without
         *        using the EBS volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedInputModes(TrainingInputMode... supportedInputModes);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private Boolean isRequired;

        private List<String> supportedContentTypes = DefaultSdkAutoConstructList.getInstance();

        private List<String> supportedCompressionTypes = DefaultSdkAutoConstructList.getInstance();

        private List<String> supportedInputModes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ChannelSpecification model) {
            name(model.name);
            description(model.description);
            isRequired(model.isRequired);
            supportedContentTypes(model.supportedContentTypes);
            supportedCompressionTypesWithStrings(model.supportedCompressionTypes);
            supportedInputModesWithStrings(model.supportedInputModes);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        @Transient
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Boolean getIsRequired() {
            return isRequired;
        }

        public final void setIsRequired(Boolean isRequired) {
            this.isRequired = isRequired;
        }

        @Override
        @Transient
        public final Builder isRequired(Boolean isRequired) {
            this.isRequired = isRequired;
            return this;
        }

        public final Collection<String> getSupportedContentTypes() {
            if (supportedContentTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedContentTypes;
        }

        public final void setSupportedContentTypes(Collection<String> supportedContentTypes) {
            this.supportedContentTypes = ContentTypesCopier.copy(supportedContentTypes);
        }

        @Override
        @Transient
        public final Builder supportedContentTypes(Collection<String> supportedContentTypes) {
            this.supportedContentTypes = ContentTypesCopier.copy(supportedContentTypes);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder supportedContentTypes(String... supportedContentTypes) {
            supportedContentTypes(Arrays.asList(supportedContentTypes));
            return this;
        }

        public final Collection<String> getSupportedCompressionTypes() {
            if (supportedCompressionTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedCompressionTypes;
        }

        public final void setSupportedCompressionTypes(Collection<String> supportedCompressionTypes) {
            this.supportedCompressionTypes = CompressionTypesCopier.copy(supportedCompressionTypes);
        }

        @Override
        @Transient
        public final Builder supportedCompressionTypesWithStrings(Collection<String> supportedCompressionTypes) {
            this.supportedCompressionTypes = CompressionTypesCopier.copy(supportedCompressionTypes);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder supportedCompressionTypesWithStrings(String... supportedCompressionTypes) {
            supportedCompressionTypesWithStrings(Arrays.asList(supportedCompressionTypes));
            return this;
        }

        @Override
        @Transient
        public final Builder supportedCompressionTypes(Collection<CompressionType> supportedCompressionTypes) {
            this.supportedCompressionTypes = CompressionTypesCopier.copyEnumToString(supportedCompressionTypes);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder supportedCompressionTypes(CompressionType... supportedCompressionTypes) {
            supportedCompressionTypes(Arrays.asList(supportedCompressionTypes));
            return this;
        }

        public final Collection<String> getSupportedInputModes() {
            if (supportedInputModes instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedInputModes;
        }

        public final void setSupportedInputModes(Collection<String> supportedInputModes) {
            this.supportedInputModes = InputModesCopier.copy(supportedInputModes);
        }

        @Override
        @Transient
        public final Builder supportedInputModesWithStrings(Collection<String> supportedInputModes) {
            this.supportedInputModes = InputModesCopier.copy(supportedInputModes);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder supportedInputModesWithStrings(String... supportedInputModes) {
            supportedInputModesWithStrings(Arrays.asList(supportedInputModes));
            return this;
        }

        @Override
        @Transient
        public final Builder supportedInputModes(Collection<TrainingInputMode> supportedInputModes) {
            this.supportedInputModes = InputModesCopier.copyEnumToString(supportedInputModes);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder supportedInputModes(TrainingInputMode... supportedInputModes) {
            supportedInputModes(Arrays.asList(supportedInputModes));
            return this;
        }

        @Override
        public ChannelSpecification build() {
            return new ChannelSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
