/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An Amazon SageMaker processing job that is used to analyze data and evaluate models. For more information, see <a
 * href="https://docs.aws.amazon.com/sagemaker/latest/dg/processing-job.html">Process Data and Evaluate Models</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProcessingJob implements SdkPojo, Serializable, ToCopyableBuilder<ProcessingJob.Builder, ProcessingJob> {
    private static final SdkField<List<ProcessingInput>> PROCESSING_INPUTS_FIELD = SdkField
            .<List<ProcessingInput>> builder(MarshallingType.LIST)
            .memberName("ProcessingInputs")
            .getter(getter(ProcessingJob::processingInputs))
            .setter(setter(Builder::processingInputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProcessingInputs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ProcessingInput> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProcessingInput::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ProcessingOutputConfig> PROCESSING_OUTPUT_CONFIG_FIELD = SdkField
            .<ProcessingOutputConfig> builder(MarshallingType.SDK_POJO).memberName("ProcessingOutputConfig")
            .getter(getter(ProcessingJob::processingOutputConfig)).setter(setter(Builder::processingOutputConfig))
            .constructor(ProcessingOutputConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProcessingOutputConfig").build())
            .build();

    private static final SdkField<String> PROCESSING_JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProcessingJobName").getter(getter(ProcessingJob::processingJobName))
            .setter(setter(Builder::processingJobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProcessingJobName").build()).build();

    private static final SdkField<ProcessingResources> PROCESSING_RESOURCES_FIELD = SdkField
            .<ProcessingResources> builder(MarshallingType.SDK_POJO).memberName("ProcessingResources")
            .getter(getter(ProcessingJob::processingResources)).setter(setter(Builder::processingResources))
            .constructor(ProcessingResources::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProcessingResources").build())
            .build();

    private static final SdkField<ProcessingStoppingCondition> STOPPING_CONDITION_FIELD = SdkField
            .<ProcessingStoppingCondition> builder(MarshallingType.SDK_POJO).memberName("StoppingCondition")
            .getter(getter(ProcessingJob::stoppingCondition)).setter(setter(Builder::stoppingCondition))
            .constructor(ProcessingStoppingCondition::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StoppingCondition").build()).build();

    private static final SdkField<AppSpecification> APP_SPECIFICATION_FIELD = SdkField
            .<AppSpecification> builder(MarshallingType.SDK_POJO).memberName("AppSpecification")
            .getter(getter(ProcessingJob::appSpecification)).setter(setter(Builder::appSpecification))
            .constructor(AppSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppSpecification").build()).build();

    private static final SdkField<Map<String, String>> ENVIRONMENT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Environment")
            .getter(getter(ProcessingJob::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Environment").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<NetworkConfig> NETWORK_CONFIG_FIELD = SdkField
            .<NetworkConfig> builder(MarshallingType.SDK_POJO).memberName("NetworkConfig")
            .getter(getter(ProcessingJob::networkConfig)).setter(setter(Builder::networkConfig))
            .constructor(NetworkConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkConfig").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleArn").getter(getter(ProcessingJob::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<ExperimentConfig> EXPERIMENT_CONFIG_FIELD = SdkField
            .<ExperimentConfig> builder(MarshallingType.SDK_POJO).memberName("ExperimentConfig")
            .getter(getter(ProcessingJob::experimentConfig)).setter(setter(Builder::experimentConfig))
            .constructor(ExperimentConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExperimentConfig").build()).build();

    private static final SdkField<String> PROCESSING_JOB_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProcessingJobArn").getter(getter(ProcessingJob::processingJobArn))
            .setter(setter(Builder::processingJobArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProcessingJobArn").build()).build();

    private static final SdkField<String> PROCESSING_JOB_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProcessingJobStatus").getter(getter(ProcessingJob::processingJobStatusAsString))
            .setter(setter(Builder::processingJobStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProcessingJobStatus").build())
            .build();

    private static final SdkField<String> EXIT_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExitMessage").getter(getter(ProcessingJob::exitMessage)).setter(setter(Builder::exitMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExitMessage").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(ProcessingJob::failureReason)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final SdkField<Instant> PROCESSING_END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ProcessingEndTime").getter(getter(ProcessingJob::processingEndTime))
            .setter(setter(Builder::processingEndTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProcessingEndTime").build()).build();

    private static final SdkField<Instant> PROCESSING_START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ProcessingStartTime").getter(getter(ProcessingJob::processingStartTime))
            .setter(setter(Builder::processingStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProcessingStartTime").build())
            .build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedTime").getter(getter(ProcessingJob::lastModifiedTime))
            .setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(ProcessingJob::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> MONITORING_SCHEDULE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MonitoringScheduleArn").getter(getter(ProcessingJob::monitoringScheduleArn))
            .setter(setter(Builder::monitoringScheduleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MonitoringScheduleArn").build())
            .build();

    private static final SdkField<String> AUTO_ML_JOB_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AutoMLJobArn").getter(getter(ProcessingJob::autoMLJobArn)).setter(setter(Builder::autoMLJobArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoMLJobArn").build()).build();

    private static final SdkField<String> TRAINING_JOB_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TrainingJobArn").getter(getter(ProcessingJob::trainingJobArn)).setter(setter(Builder::trainingJobArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingJobArn").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(ProcessingJob::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PROCESSING_INPUTS_FIELD,
            PROCESSING_OUTPUT_CONFIG_FIELD, PROCESSING_JOB_NAME_FIELD, PROCESSING_RESOURCES_FIELD, STOPPING_CONDITION_FIELD,
            APP_SPECIFICATION_FIELD, ENVIRONMENT_FIELD, NETWORK_CONFIG_FIELD, ROLE_ARN_FIELD, EXPERIMENT_CONFIG_FIELD,
            PROCESSING_JOB_ARN_FIELD, PROCESSING_JOB_STATUS_FIELD, EXIT_MESSAGE_FIELD, FAILURE_REASON_FIELD,
            PROCESSING_END_TIME_FIELD, PROCESSING_START_TIME_FIELD, LAST_MODIFIED_TIME_FIELD, CREATION_TIME_FIELD,
            MONITORING_SCHEDULE_ARN_FIELD, AUTO_ML_JOB_ARN_FIELD, TRAINING_JOB_ARN_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<ProcessingInput> processingInputs;

    private final ProcessingOutputConfig processingOutputConfig;

    private final String processingJobName;

    private final ProcessingResources processingResources;

    private final ProcessingStoppingCondition stoppingCondition;

    private final AppSpecification appSpecification;

    private final Map<String, String> environment;

    private final NetworkConfig networkConfig;

    private final String roleArn;

    private final ExperimentConfig experimentConfig;

    private final String processingJobArn;

    private final String processingJobStatus;

    private final String exitMessage;

    private final String failureReason;

    private final Instant processingEndTime;

    private final Instant processingStartTime;

    private final Instant lastModifiedTime;

    private final Instant creationTime;

    private final String monitoringScheduleArn;

    private final String autoMLJobArn;

    private final String trainingJobArn;

    private final List<Tag> tags;

    private ProcessingJob(BuilderImpl builder) {
        this.processingInputs = builder.processingInputs;
        this.processingOutputConfig = builder.processingOutputConfig;
        this.processingJobName = builder.processingJobName;
        this.processingResources = builder.processingResources;
        this.stoppingCondition = builder.stoppingCondition;
        this.appSpecification = builder.appSpecification;
        this.environment = builder.environment;
        this.networkConfig = builder.networkConfig;
        this.roleArn = builder.roleArn;
        this.experimentConfig = builder.experimentConfig;
        this.processingJobArn = builder.processingJobArn;
        this.processingJobStatus = builder.processingJobStatus;
        this.exitMessage = builder.exitMessage;
        this.failureReason = builder.failureReason;
        this.processingEndTime = builder.processingEndTime;
        this.processingStartTime = builder.processingStartTime;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.creationTime = builder.creationTime;
        this.monitoringScheduleArn = builder.monitoringScheduleArn;
        this.autoMLJobArn = builder.autoMLJobArn;
        this.trainingJobArn = builder.trainingJobArn;
        this.tags = builder.tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the ProcessingInputs property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasProcessingInputs() {
        return processingInputs != null && !(processingInputs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of input configurations for the processing job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProcessingInputs} method.
     * </p>
     * 
     * @return List of input configurations for the processing job.
     */
    public final List<ProcessingInput> processingInputs() {
        return processingInputs;
    }

    /**
     * Returns the value of the ProcessingOutputConfig property for this object.
     * 
     * @return The value of the ProcessingOutputConfig property for this object.
     */
    public final ProcessingOutputConfig processingOutputConfig() {
        return processingOutputConfig;
    }

    /**
     * <p>
     * The name of the processing job.
     * </p>
     * 
     * @return The name of the processing job.
     */
    public final String processingJobName() {
        return processingJobName;
    }

    /**
     * Returns the value of the ProcessingResources property for this object.
     * 
     * @return The value of the ProcessingResources property for this object.
     */
    public final ProcessingResources processingResources() {
        return processingResources;
    }

    /**
     * Returns the value of the StoppingCondition property for this object.
     * 
     * @return The value of the StoppingCondition property for this object.
     */
    public final ProcessingStoppingCondition stoppingCondition() {
        return stoppingCondition;
    }

    /**
     * Returns the value of the AppSpecification property for this object.
     * 
     * @return The value of the AppSpecification property for this object.
     */
    public final AppSpecification appSpecification() {
        return appSpecification;
    }

    /**
     * For responses, this returns true if the service returned a value for the Environment property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Sets the environment variables in the Docker container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironment} method.
     * </p>
     * 
     * @return Sets the environment variables in the Docker container.
     */
    public final Map<String, String> environment() {
        return environment;
    }

    /**
     * Returns the value of the NetworkConfig property for this object.
     * 
     * @return The value of the NetworkConfig property for this object.
     */
    public final NetworkConfig networkConfig() {
        return networkConfig;
    }

    /**
     * <p>
     * The ARN of the role used to create the processing job.
     * </p>
     * 
     * @return The ARN of the role used to create the processing job.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * Returns the value of the ExperimentConfig property for this object.
     * 
     * @return The value of the ExperimentConfig property for this object.
     */
    public final ExperimentConfig experimentConfig() {
        return experimentConfig;
    }

    /**
     * <p>
     * The ARN of the processing job.
     * </p>
     * 
     * @return The ARN of the processing job.
     */
    public final String processingJobArn() {
        return processingJobArn;
    }

    /**
     * <p>
     * The status of the processing job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #processingJobStatus} will return {@link ProcessingJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #processingJobStatusAsString}.
     * </p>
     * 
     * @return The status of the processing job.
     * @see ProcessingJobStatus
     */
    public final ProcessingJobStatus processingJobStatus() {
        return ProcessingJobStatus.fromValue(processingJobStatus);
    }

    /**
     * <p>
     * The status of the processing job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #processingJobStatus} will return {@link ProcessingJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #processingJobStatusAsString}.
     * </p>
     * 
     * @return The status of the processing job.
     * @see ProcessingJobStatus
     */
    public final String processingJobStatusAsString() {
        return processingJobStatus;
    }

    /**
     * <p>
     * A string, up to one KB in size, that contains metadata from the processing container when the processing job
     * exits.
     * </p>
     * 
     * @return A string, up to one KB in size, that contains metadata from the processing container when the processing
     *         job exits.
     */
    public final String exitMessage() {
        return exitMessage;
    }

    /**
     * <p>
     * A string, up to one KB in size, that contains the reason a processing job failed, if it failed.
     * </p>
     * 
     * @return A string, up to one KB in size, that contains the reason a processing job failed, if it failed.
     */
    public final String failureReason() {
        return failureReason;
    }

    /**
     * <p>
     * The time that the processing job ended.
     * </p>
     * 
     * @return The time that the processing job ended.
     */
    public final Instant processingEndTime() {
        return processingEndTime;
    }

    /**
     * <p>
     * The time that the processing job started.
     * </p>
     * 
     * @return The time that the processing job started.
     */
    public final Instant processingStartTime() {
        return processingStartTime;
    }

    /**
     * <p>
     * The time the processing job was last modified.
     * </p>
     * 
     * @return The time the processing job was last modified.
     */
    public final Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * <p>
     * The time the processing job was created.
     * </p>
     * 
     * @return The time the processing job was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The ARN of a monitoring schedule for an endpoint associated with this processing job.
     * </p>
     * 
     * @return The ARN of a monitoring schedule for an endpoint associated with this processing job.
     */
    public final String monitoringScheduleArn() {
        return monitoringScheduleArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the AutoML job associated with this processing job.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the AutoML job associated with this processing job.
     */
    public final String autoMLJobArn() {
        return autoMLJobArn;
    }

    /**
     * <p>
     * The ARN of the training job associated with this processing job.
     * </p>
     * 
     * @return The ARN of the training job associated with this processing job.
     */
    public final String trainingJobArn() {
        return trainingJobArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of key-value pairs. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html#allocation-whatURL">Using
     * Cost Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return An array of key-value pairs. For more information, see <a
     *         href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html#allocation-whatURL"
     *         >Using Cost Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasProcessingInputs() ? processingInputs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(processingOutputConfig());
        hashCode = 31 * hashCode + Objects.hashCode(processingJobName());
        hashCode = 31 * hashCode + Objects.hashCode(processingResources());
        hashCode = 31 * hashCode + Objects.hashCode(stoppingCondition());
        hashCode = 31 * hashCode + Objects.hashCode(appSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(networkConfig());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(experimentConfig());
        hashCode = 31 * hashCode + Objects.hashCode(processingJobArn());
        hashCode = 31 * hashCode + Objects.hashCode(processingJobStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(exitMessage());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(processingEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(processingStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(monitoringScheduleArn());
        hashCode = 31 * hashCode + Objects.hashCode(autoMLJobArn());
        hashCode = 31 * hashCode + Objects.hashCode(trainingJobArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProcessingJob)) {
            return false;
        }
        ProcessingJob other = (ProcessingJob) obj;
        return hasProcessingInputs() == other.hasProcessingInputs()
                && Objects.equals(processingInputs(), other.processingInputs())
                && Objects.equals(processingOutputConfig(), other.processingOutputConfig())
                && Objects.equals(processingJobName(), other.processingJobName())
                && Objects.equals(processingResources(), other.processingResources())
                && Objects.equals(stoppingCondition(), other.stoppingCondition())
                && Objects.equals(appSpecification(), other.appSpecification()) && hasEnvironment() == other.hasEnvironment()
                && Objects.equals(environment(), other.environment()) && Objects.equals(networkConfig(), other.networkConfig())
                && Objects.equals(roleArn(), other.roleArn()) && Objects.equals(experimentConfig(), other.experimentConfig())
                && Objects.equals(processingJobArn(), other.processingJobArn())
                && Objects.equals(processingJobStatusAsString(), other.processingJobStatusAsString())
                && Objects.equals(exitMessage(), other.exitMessage()) && Objects.equals(failureReason(), other.failureReason())
                && Objects.equals(processingEndTime(), other.processingEndTime())
                && Objects.equals(processingStartTime(), other.processingStartTime())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(monitoringScheduleArn(), other.monitoringScheduleArn())
                && Objects.equals(autoMLJobArn(), other.autoMLJobArn())
                && Objects.equals(trainingJobArn(), other.trainingJobArn()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProcessingJob").add("ProcessingInputs", hasProcessingInputs() ? processingInputs() : null)
                .add("ProcessingOutputConfig", processingOutputConfig()).add("ProcessingJobName", processingJobName())
                .add("ProcessingResources", processingResources()).add("StoppingCondition", stoppingCondition())
                .add("AppSpecification", appSpecification()).add("Environment", hasEnvironment() ? environment() : null)
                .add("NetworkConfig", networkConfig()).add("RoleArn", roleArn()).add("ExperimentConfig", experimentConfig())
                .add("ProcessingJobArn", processingJobArn()).add("ProcessingJobStatus", processingJobStatusAsString())
                .add("ExitMessage", exitMessage()).add("FailureReason", failureReason())
                .add("ProcessingEndTime", processingEndTime()).add("ProcessingStartTime", processingStartTime())
                .add("LastModifiedTime", lastModifiedTime()).add("CreationTime", creationTime())
                .add("MonitoringScheduleArn", monitoringScheduleArn()).add("AutoMLJobArn", autoMLJobArn())
                .add("TrainingJobArn", trainingJobArn()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ProcessingInputs":
            return Optional.ofNullable(clazz.cast(processingInputs()));
        case "ProcessingOutputConfig":
            return Optional.ofNullable(clazz.cast(processingOutputConfig()));
        case "ProcessingJobName":
            return Optional.ofNullable(clazz.cast(processingJobName()));
        case "ProcessingResources":
            return Optional.ofNullable(clazz.cast(processingResources()));
        case "StoppingCondition":
            return Optional.ofNullable(clazz.cast(stoppingCondition()));
        case "AppSpecification":
            return Optional.ofNullable(clazz.cast(appSpecification()));
        case "Environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "NetworkConfig":
            return Optional.ofNullable(clazz.cast(networkConfig()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "ExperimentConfig":
            return Optional.ofNullable(clazz.cast(experimentConfig()));
        case "ProcessingJobArn":
            return Optional.ofNullable(clazz.cast(processingJobArn()));
        case "ProcessingJobStatus":
            return Optional.ofNullable(clazz.cast(processingJobStatusAsString()));
        case "ExitMessage":
            return Optional.ofNullable(clazz.cast(exitMessage()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "ProcessingEndTime":
            return Optional.ofNullable(clazz.cast(processingEndTime()));
        case "ProcessingStartTime":
            return Optional.ofNullable(clazz.cast(processingStartTime()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "MonitoringScheduleArn":
            return Optional.ofNullable(clazz.cast(monitoringScheduleArn()));
        case "AutoMLJobArn":
            return Optional.ofNullable(clazz.cast(autoMLJobArn()));
        case "TrainingJobArn":
            return Optional.ofNullable(clazz.cast(trainingJobArn()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProcessingJob, T> g) {
        return obj -> g.apply((ProcessingJob) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProcessingJob> {
        /**
         * <p>
         * List of input configurations for the processing job.
         * </p>
         * 
         * @param processingInputs
         *        List of input configurations for the processing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processingInputs(Collection<ProcessingInput> processingInputs);

        /**
         * <p>
         * List of input configurations for the processing job.
         * </p>
         * 
         * @param processingInputs
         *        List of input configurations for the processing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processingInputs(ProcessingInput... processingInputs);

        /**
         * <p>
         * List of input configurations for the processing job.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<ProcessingInput>.Builder} avoiding
         * the need to create one manually via {@link List<ProcessingInput>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ProcessingInput>.Builder#build()} is called immediately and
         * its result is passed to {@link #processingInputs(List<ProcessingInput>)}.
         * 
         * @param processingInputs
         *        a consumer that will call methods on {@link List<ProcessingInput>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #processingInputs(List<ProcessingInput>)
         */
        Builder processingInputs(Consumer<ProcessingInput.Builder>... processingInputs);

        /**
         * Sets the value of the ProcessingOutputConfig property for this object.
         *
         * @param processingOutputConfig
         *        The new value for the ProcessingOutputConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processingOutputConfig(ProcessingOutputConfig processingOutputConfig);

        /**
         * Sets the value of the ProcessingOutputConfig property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ProcessingOutputConfig.Builder} avoiding
         * the need to create one manually via {@link ProcessingOutputConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProcessingOutputConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #processingOutputConfig(ProcessingOutputConfig)}.
         * 
         * @param processingOutputConfig
         *        a consumer that will call methods on {@link ProcessingOutputConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #processingOutputConfig(ProcessingOutputConfig)
         */
        default Builder processingOutputConfig(Consumer<ProcessingOutputConfig.Builder> processingOutputConfig) {
            return processingOutputConfig(ProcessingOutputConfig.builder().applyMutation(processingOutputConfig).build());
        }

        /**
         * <p>
         * The name of the processing job.
         * </p>
         * 
         * @param processingJobName
         *        The name of the processing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processingJobName(String processingJobName);

        /**
         * Sets the value of the ProcessingResources property for this object.
         *
         * @param processingResources
         *        The new value for the ProcessingResources property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processingResources(ProcessingResources processingResources);

        /**
         * Sets the value of the ProcessingResources property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ProcessingResources.Builder} avoiding the
         * need to create one manually via {@link ProcessingResources#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProcessingResources.Builder#build()} is called immediately and
         * its result is passed to {@link #processingResources(ProcessingResources)}.
         * 
         * @param processingResources
         *        a consumer that will call methods on {@link ProcessingResources.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #processingResources(ProcessingResources)
         */
        default Builder processingResources(Consumer<ProcessingResources.Builder> processingResources) {
            return processingResources(ProcessingResources.builder().applyMutation(processingResources).build());
        }

        /**
         * Sets the value of the StoppingCondition property for this object.
         *
         * @param stoppingCondition
         *        The new value for the StoppingCondition property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppingCondition(ProcessingStoppingCondition stoppingCondition);

        /**
         * Sets the value of the StoppingCondition property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ProcessingStoppingCondition.Builder}
         * avoiding the need to create one manually via {@link ProcessingStoppingCondition#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProcessingStoppingCondition.Builder#build()} is called
         * immediately and its result is passed to {@link #stoppingCondition(ProcessingStoppingCondition)}.
         * 
         * @param stoppingCondition
         *        a consumer that will call methods on {@link ProcessingStoppingCondition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stoppingCondition(ProcessingStoppingCondition)
         */
        default Builder stoppingCondition(Consumer<ProcessingStoppingCondition.Builder> stoppingCondition) {
            return stoppingCondition(ProcessingStoppingCondition.builder().applyMutation(stoppingCondition).build());
        }

        /**
         * Sets the value of the AppSpecification property for this object.
         *
         * @param appSpecification
         *        The new value for the AppSpecification property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appSpecification(AppSpecification appSpecification);

        /**
         * Sets the value of the AppSpecification property for this object.
         *
         * This is a convenience method that creates an instance of the {@link AppSpecification.Builder} avoiding the
         * need to create one manually via {@link AppSpecification#builder()}.
         *
         * When the {@link Consumer} completes, {@link AppSpecification.Builder#build()} is called immediately and its
         * result is passed to {@link #appSpecification(AppSpecification)}.
         * 
         * @param appSpecification
         *        a consumer that will call methods on {@link AppSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #appSpecification(AppSpecification)
         */
        default Builder appSpecification(Consumer<AppSpecification.Builder> appSpecification) {
            return appSpecification(AppSpecification.builder().applyMutation(appSpecification).build());
        }

        /**
         * <p>
         * Sets the environment variables in the Docker container.
         * </p>
         * 
         * @param environment
         *        Sets the environment variables in the Docker container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Map<String, String> environment);

        /**
         * Sets the value of the NetworkConfig property for this object.
         *
         * @param networkConfig
         *        The new value for the NetworkConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkConfig(NetworkConfig networkConfig);

        /**
         * Sets the value of the NetworkConfig property for this object.
         *
         * This is a convenience method that creates an instance of the {@link NetworkConfig.Builder} avoiding the need
         * to create one manually via {@link NetworkConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link NetworkConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #networkConfig(NetworkConfig)}.
         * 
         * @param networkConfig
         *        a consumer that will call methods on {@link NetworkConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkConfig(NetworkConfig)
         */
        default Builder networkConfig(Consumer<NetworkConfig.Builder> networkConfig) {
            return networkConfig(NetworkConfig.builder().applyMutation(networkConfig).build());
        }

        /**
         * <p>
         * The ARN of the role used to create the processing job.
         * </p>
         * 
         * @param roleArn
         *        The ARN of the role used to create the processing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * Sets the value of the ExperimentConfig property for this object.
         *
         * @param experimentConfig
         *        The new value for the ExperimentConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder experimentConfig(ExperimentConfig experimentConfig);

        /**
         * Sets the value of the ExperimentConfig property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ExperimentConfig.Builder} avoiding the
         * need to create one manually via {@link ExperimentConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ExperimentConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #experimentConfig(ExperimentConfig)}.
         * 
         * @param experimentConfig
         *        a consumer that will call methods on {@link ExperimentConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #experimentConfig(ExperimentConfig)
         */
        default Builder experimentConfig(Consumer<ExperimentConfig.Builder> experimentConfig) {
            return experimentConfig(ExperimentConfig.builder().applyMutation(experimentConfig).build());
        }

        /**
         * <p>
         * The ARN of the processing job.
         * </p>
         * 
         * @param processingJobArn
         *        The ARN of the processing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processingJobArn(String processingJobArn);

        /**
         * <p>
         * The status of the processing job.
         * </p>
         * 
         * @param processingJobStatus
         *        The status of the processing job.
         * @see ProcessingJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProcessingJobStatus
         */
        Builder processingJobStatus(String processingJobStatus);

        /**
         * <p>
         * The status of the processing job.
         * </p>
         * 
         * @param processingJobStatus
         *        The status of the processing job.
         * @see ProcessingJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProcessingJobStatus
         */
        Builder processingJobStatus(ProcessingJobStatus processingJobStatus);

        /**
         * <p>
         * A string, up to one KB in size, that contains metadata from the processing container when the processing job
         * exits.
         * </p>
         * 
         * @param exitMessage
         *        A string, up to one KB in size, that contains metadata from the processing container when the
         *        processing job exits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exitMessage(String exitMessage);

        /**
         * <p>
         * A string, up to one KB in size, that contains the reason a processing job failed, if it failed.
         * </p>
         * 
         * @param failureReason
         *        A string, up to one KB in size, that contains the reason a processing job failed, if it failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * The time that the processing job ended.
         * </p>
         * 
         * @param processingEndTime
         *        The time that the processing job ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processingEndTime(Instant processingEndTime);

        /**
         * <p>
         * The time that the processing job started.
         * </p>
         * 
         * @param processingStartTime
         *        The time that the processing job started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processingStartTime(Instant processingStartTime);

        /**
         * <p>
         * The time the processing job was last modified.
         * </p>
         * 
         * @param lastModifiedTime
         *        The time the processing job was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * The time the processing job was created.
         * </p>
         * 
         * @param creationTime
         *        The time the processing job was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The ARN of a monitoring schedule for an endpoint associated with this processing job.
         * </p>
         * 
         * @param monitoringScheduleArn
         *        The ARN of a monitoring schedule for an endpoint associated with this processing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder monitoringScheduleArn(String monitoringScheduleArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the AutoML job associated with this processing job.
         * </p>
         * 
         * @param autoMLJobArn
         *        The Amazon Resource Name (ARN) of the AutoML job associated with this processing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoMLJobArn(String autoMLJobArn);

        /**
         * <p>
         * The ARN of the training job associated with this processing job.
         * </p>
         * 
         * @param trainingJobArn
         *        The ARN of the training job associated with this processing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingJobArn(String trainingJobArn);

        /**
         * <p>
         * An array of key-value pairs. For more information, see <a
         * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html#allocation-whatURL"
         * >Using Cost Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>.
         * </p>
         * 
         * @param tags
         *        An array of key-value pairs. For more information, see <a href=
         *        "https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html#allocation-whatURL"
         *        >Using Cost Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost Management User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * An array of key-value pairs. For more information, see <a
         * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html#allocation-whatURL"
         * >Using Cost Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>.
         * </p>
         * 
         * @param tags
         *        An array of key-value pairs. For more information, see <a href=
         *        "https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html#allocation-whatURL"
         *        >Using Cost Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost Management User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * An array of key-value pairs. For more information, see <a
         * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html#allocation-whatURL"
         * >Using Cost Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Tag>.Builder} avoiding the need to
         * create one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private List<ProcessingInput> processingInputs = DefaultSdkAutoConstructList.getInstance();

        private ProcessingOutputConfig processingOutputConfig;

        private String processingJobName;

        private ProcessingResources processingResources;

        private ProcessingStoppingCondition stoppingCondition;

        private AppSpecification appSpecification;

        private Map<String, String> environment = DefaultSdkAutoConstructMap.getInstance();

        private NetworkConfig networkConfig;

        private String roleArn;

        private ExperimentConfig experimentConfig;

        private String processingJobArn;

        private String processingJobStatus;

        private String exitMessage;

        private String failureReason;

        private Instant processingEndTime;

        private Instant processingStartTime;

        private Instant lastModifiedTime;

        private Instant creationTime;

        private String monitoringScheduleArn;

        private String autoMLJobArn;

        private String trainingJobArn;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ProcessingJob model) {
            processingInputs(model.processingInputs);
            processingOutputConfig(model.processingOutputConfig);
            processingJobName(model.processingJobName);
            processingResources(model.processingResources);
            stoppingCondition(model.stoppingCondition);
            appSpecification(model.appSpecification);
            environment(model.environment);
            networkConfig(model.networkConfig);
            roleArn(model.roleArn);
            experimentConfig(model.experimentConfig);
            processingJobArn(model.processingJobArn);
            processingJobStatus(model.processingJobStatus);
            exitMessage(model.exitMessage);
            failureReason(model.failureReason);
            processingEndTime(model.processingEndTime);
            processingStartTime(model.processingStartTime);
            lastModifiedTime(model.lastModifiedTime);
            creationTime(model.creationTime);
            monitoringScheduleArn(model.monitoringScheduleArn);
            autoMLJobArn(model.autoMLJobArn);
            trainingJobArn(model.trainingJobArn);
            tags(model.tags);
        }

        public final List<ProcessingInput.Builder> getProcessingInputs() {
            List<ProcessingInput.Builder> result = ProcessingInputsCopier.copyToBuilder(this.processingInputs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setProcessingInputs(Collection<ProcessingInput.BuilderImpl> processingInputs) {
            this.processingInputs = ProcessingInputsCopier.copyFromBuilder(processingInputs);
        }

        @Override
        public final Builder processingInputs(Collection<ProcessingInput> processingInputs) {
            this.processingInputs = ProcessingInputsCopier.copy(processingInputs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder processingInputs(ProcessingInput... processingInputs) {
            processingInputs(Arrays.asList(processingInputs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder processingInputs(Consumer<ProcessingInput.Builder>... processingInputs) {
            processingInputs(Stream.of(processingInputs).map(c -> ProcessingInput.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final ProcessingOutputConfig.Builder getProcessingOutputConfig() {
            return processingOutputConfig != null ? processingOutputConfig.toBuilder() : null;
        }

        public final void setProcessingOutputConfig(ProcessingOutputConfig.BuilderImpl processingOutputConfig) {
            this.processingOutputConfig = processingOutputConfig != null ? processingOutputConfig.build() : null;
        }

        @Override
        public final Builder processingOutputConfig(ProcessingOutputConfig processingOutputConfig) {
            this.processingOutputConfig = processingOutputConfig;
            return this;
        }

        public final String getProcessingJobName() {
            return processingJobName;
        }

        public final void setProcessingJobName(String processingJobName) {
            this.processingJobName = processingJobName;
        }

        @Override
        public final Builder processingJobName(String processingJobName) {
            this.processingJobName = processingJobName;
            return this;
        }

        public final ProcessingResources.Builder getProcessingResources() {
            return processingResources != null ? processingResources.toBuilder() : null;
        }

        public final void setProcessingResources(ProcessingResources.BuilderImpl processingResources) {
            this.processingResources = processingResources != null ? processingResources.build() : null;
        }

        @Override
        public final Builder processingResources(ProcessingResources processingResources) {
            this.processingResources = processingResources;
            return this;
        }

        public final ProcessingStoppingCondition.Builder getStoppingCondition() {
            return stoppingCondition != null ? stoppingCondition.toBuilder() : null;
        }

        public final void setStoppingCondition(ProcessingStoppingCondition.BuilderImpl stoppingCondition) {
            this.stoppingCondition = stoppingCondition != null ? stoppingCondition.build() : null;
        }

        @Override
        public final Builder stoppingCondition(ProcessingStoppingCondition stoppingCondition) {
            this.stoppingCondition = stoppingCondition;
            return this;
        }

        public final AppSpecification.Builder getAppSpecification() {
            return appSpecification != null ? appSpecification.toBuilder() : null;
        }

        public final void setAppSpecification(AppSpecification.BuilderImpl appSpecification) {
            this.appSpecification = appSpecification != null ? appSpecification.build() : null;
        }

        @Override
        public final Builder appSpecification(AppSpecification appSpecification) {
            this.appSpecification = appSpecification;
            return this;
        }

        public final Map<String, String> getEnvironment() {
            if (environment instanceof SdkAutoConstructMap) {
                return null;
            }
            return environment;
        }

        public final void setEnvironment(Map<String, String> environment) {
            this.environment = ProcessingEnvironmentMapCopier.copy(environment);
        }

        @Override
        public final Builder environment(Map<String, String> environment) {
            this.environment = ProcessingEnvironmentMapCopier.copy(environment);
            return this;
        }

        public final NetworkConfig.Builder getNetworkConfig() {
            return networkConfig != null ? networkConfig.toBuilder() : null;
        }

        public final void setNetworkConfig(NetworkConfig.BuilderImpl networkConfig) {
            this.networkConfig = networkConfig != null ? networkConfig.build() : null;
        }

        @Override
        public final Builder networkConfig(NetworkConfig networkConfig) {
            this.networkConfig = networkConfig;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final ExperimentConfig.Builder getExperimentConfig() {
            return experimentConfig != null ? experimentConfig.toBuilder() : null;
        }

        public final void setExperimentConfig(ExperimentConfig.BuilderImpl experimentConfig) {
            this.experimentConfig = experimentConfig != null ? experimentConfig.build() : null;
        }

        @Override
        public final Builder experimentConfig(ExperimentConfig experimentConfig) {
            this.experimentConfig = experimentConfig;
            return this;
        }

        public final String getProcessingJobArn() {
            return processingJobArn;
        }

        public final void setProcessingJobArn(String processingJobArn) {
            this.processingJobArn = processingJobArn;
        }

        @Override
        public final Builder processingJobArn(String processingJobArn) {
            this.processingJobArn = processingJobArn;
            return this;
        }

        public final String getProcessingJobStatus() {
            return processingJobStatus;
        }

        public final void setProcessingJobStatus(String processingJobStatus) {
            this.processingJobStatus = processingJobStatus;
        }

        @Override
        public final Builder processingJobStatus(String processingJobStatus) {
            this.processingJobStatus = processingJobStatus;
            return this;
        }

        @Override
        public final Builder processingJobStatus(ProcessingJobStatus processingJobStatus) {
            this.processingJobStatus(processingJobStatus == null ? null : processingJobStatus.toString());
            return this;
        }

        public final String getExitMessage() {
            return exitMessage;
        }

        public final void setExitMessage(String exitMessage) {
            this.exitMessage = exitMessage;
        }

        @Override
        public final Builder exitMessage(String exitMessage) {
            this.exitMessage = exitMessage;
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final Instant getProcessingEndTime() {
            return processingEndTime;
        }

        public final void setProcessingEndTime(Instant processingEndTime) {
            this.processingEndTime = processingEndTime;
        }

        @Override
        public final Builder processingEndTime(Instant processingEndTime) {
            this.processingEndTime = processingEndTime;
            return this;
        }

        public final Instant getProcessingStartTime() {
            return processingStartTime;
        }

        public final void setProcessingStartTime(Instant processingStartTime) {
            this.processingStartTime = processingStartTime;
        }

        @Override
        public final Builder processingStartTime(Instant processingStartTime) {
            this.processingStartTime = processingStartTime;
            return this;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getMonitoringScheduleArn() {
            return monitoringScheduleArn;
        }

        public final void setMonitoringScheduleArn(String monitoringScheduleArn) {
            this.monitoringScheduleArn = monitoringScheduleArn;
        }

        @Override
        public final Builder monitoringScheduleArn(String monitoringScheduleArn) {
            this.monitoringScheduleArn = monitoringScheduleArn;
            return this;
        }

        public final String getAutoMLJobArn() {
            return autoMLJobArn;
        }

        public final void setAutoMLJobArn(String autoMLJobArn) {
            this.autoMLJobArn = autoMLJobArn;
        }

        @Override
        public final Builder autoMLJobArn(String autoMLJobArn) {
            this.autoMLJobArn = autoMLJobArn;
            return this;
        }

        public final String getTrainingJobArn() {
            return trainingJobArn;
        }

        public final void setTrainingJobArn(String trainingJobArn) {
            this.trainingJobArn = trainingJobArn;
        }

        @Override
        public final Builder trainingJobArn(String trainingJobArn) {
            this.trainingJobArn = trainingJobArn;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ProcessingJob build() {
            return new ProcessingJob(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
