/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the results of a processing job. The processing output must specify exactly one of either
 * <code>S3Output</code> or <code>FeatureStoreOutput</code> types.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProcessingOutput implements SdkPojo, Serializable,
        ToCopyableBuilder<ProcessingOutput.Builder, ProcessingOutput> {
    private static final SdkField<String> OUTPUT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputName").getter(getter(ProcessingOutput::outputName)).setter(setter(Builder::outputName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputName").build()).build();

    private static final SdkField<ProcessingS3Output> S3_OUTPUT_FIELD = SdkField
            .<ProcessingS3Output> builder(MarshallingType.SDK_POJO).memberName("S3Output")
            .getter(getter(ProcessingOutput::s3Output)).setter(setter(Builder::s3Output))
            .constructor(ProcessingS3Output::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Output").build()).build();

    private static final SdkField<ProcessingFeatureStoreOutput> FEATURE_STORE_OUTPUT_FIELD = SdkField
            .<ProcessingFeatureStoreOutput> builder(MarshallingType.SDK_POJO).memberName("FeatureStoreOutput")
            .getter(getter(ProcessingOutput::featureStoreOutput)).setter(setter(Builder::featureStoreOutput))
            .constructor(ProcessingFeatureStoreOutput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeatureStoreOutput").build())
            .build();

    private static final SdkField<Boolean> APP_MANAGED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AppManaged").getter(getter(ProcessingOutput::appManaged)).setter(setter(Builder::appManaged))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppManaged").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OUTPUT_NAME_FIELD,
            S3_OUTPUT_FIELD, FEATURE_STORE_OUTPUT_FIELD, APP_MANAGED_FIELD));

    private static final long serialVersionUID = 1L;

    private final String outputName;

    private final ProcessingS3Output s3Output;

    private final ProcessingFeatureStoreOutput featureStoreOutput;

    private final Boolean appManaged;

    private ProcessingOutput(BuilderImpl builder) {
        this.outputName = builder.outputName;
        this.s3Output = builder.s3Output;
        this.featureStoreOutput = builder.featureStoreOutput;
        this.appManaged = builder.appManaged;
    }

    /**
     * <p>
     * The name for the processing job output.
     * </p>
     * 
     * @return The name for the processing job output.
     */
    public final String outputName() {
        return outputName;
    }

    /**
     * <p>
     * Configuration for processing job outputs in Amazon S3.
     * </p>
     * 
     * @return Configuration for processing job outputs in Amazon S3.
     */
    public final ProcessingS3Output s3Output() {
        return s3Output;
    }

    /**
     * <p>
     * Configuration for processing job outputs in Amazon SageMaker Feature Store. This processing output type is only
     * supported when <code>AppManaged</code> is specified.
     * </p>
     * 
     * @return Configuration for processing job outputs in Amazon SageMaker Feature Store. This processing output type
     *         is only supported when <code>AppManaged</code> is specified.
     */
    public final ProcessingFeatureStoreOutput featureStoreOutput() {
        return featureStoreOutput;
    }

    /**
     * <p>
     * When <code>True</code>, output operations such as data upload are managed natively by the processing job
     * application. When <code>False</code> (default), output operations are managed by Amazon SageMaker.
     * </p>
     * 
     * @return When <code>True</code>, output operations such as data upload are managed natively by the processing job
     *         application. When <code>False</code> (default), output operations are managed by Amazon SageMaker.
     */
    public final Boolean appManaged() {
        return appManaged;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(outputName());
        hashCode = 31 * hashCode + Objects.hashCode(s3Output());
        hashCode = 31 * hashCode + Objects.hashCode(featureStoreOutput());
        hashCode = 31 * hashCode + Objects.hashCode(appManaged());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProcessingOutput)) {
            return false;
        }
        ProcessingOutput other = (ProcessingOutput) obj;
        return Objects.equals(outputName(), other.outputName()) && Objects.equals(s3Output(), other.s3Output())
                && Objects.equals(featureStoreOutput(), other.featureStoreOutput())
                && Objects.equals(appManaged(), other.appManaged());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProcessingOutput").add("OutputName", outputName()).add("S3Output", s3Output())
                .add("FeatureStoreOutput", featureStoreOutput()).add("AppManaged", appManaged()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OutputName":
            return Optional.ofNullable(clazz.cast(outputName()));
        case "S3Output":
            return Optional.ofNullable(clazz.cast(s3Output()));
        case "FeatureStoreOutput":
            return Optional.ofNullable(clazz.cast(featureStoreOutput()));
        case "AppManaged":
            return Optional.ofNullable(clazz.cast(appManaged()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProcessingOutput, T> g) {
        return obj -> g.apply((ProcessingOutput) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProcessingOutput> {
        /**
         * <p>
         * The name for the processing job output.
         * </p>
         * 
         * @param outputName
         *        The name for the processing job output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputName(String outputName);

        /**
         * <p>
         * Configuration for processing job outputs in Amazon S3.
         * </p>
         * 
         * @param s3Output
         *        Configuration for processing job outputs in Amazon S3.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Output(ProcessingS3Output s3Output);

        /**
         * <p>
         * Configuration for processing job outputs in Amazon S3.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProcessingS3Output.Builder} avoiding the
         * need to create one manually via {@link ProcessingS3Output#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProcessingS3Output.Builder#build()} is called immediately and its
         * result is passed to {@link #s3Output(ProcessingS3Output)}.
         * 
         * @param s3Output
         *        a consumer that will call methods on {@link ProcessingS3Output.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Output(ProcessingS3Output)
         */
        default Builder s3Output(Consumer<ProcessingS3Output.Builder> s3Output) {
            return s3Output(ProcessingS3Output.builder().applyMutation(s3Output).build());
        }

        /**
         * <p>
         * Configuration for processing job outputs in Amazon SageMaker Feature Store. This processing output type is
         * only supported when <code>AppManaged</code> is specified.
         * </p>
         * 
         * @param featureStoreOutput
         *        Configuration for processing job outputs in Amazon SageMaker Feature Store. This processing output
         *        type is only supported when <code>AppManaged</code> is specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureStoreOutput(ProcessingFeatureStoreOutput featureStoreOutput);

        /**
         * <p>
         * Configuration for processing job outputs in Amazon SageMaker Feature Store. This processing output type is
         * only supported when <code>AppManaged</code> is specified.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProcessingFeatureStoreOutput.Builder}
         * avoiding the need to create one manually via {@link ProcessingFeatureStoreOutput#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProcessingFeatureStoreOutput.Builder#build()} is called
         * immediately and its result is passed to {@link #featureStoreOutput(ProcessingFeatureStoreOutput)}.
         * 
         * @param featureStoreOutput
         *        a consumer that will call methods on {@link ProcessingFeatureStoreOutput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #featureStoreOutput(ProcessingFeatureStoreOutput)
         */
        default Builder featureStoreOutput(Consumer<ProcessingFeatureStoreOutput.Builder> featureStoreOutput) {
            return featureStoreOutput(ProcessingFeatureStoreOutput.builder().applyMutation(featureStoreOutput).build());
        }

        /**
         * <p>
         * When <code>True</code>, output operations such as data upload are managed natively by the processing job
         * application. When <code>False</code> (default), output operations are managed by Amazon SageMaker.
         * </p>
         * 
         * @param appManaged
         *        When <code>True</code>, output operations such as data upload are managed natively by the processing
         *        job application. When <code>False</code> (default), output operations are managed by Amazon SageMaker.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appManaged(Boolean appManaged);
    }

    static final class BuilderImpl implements Builder {
        private String outputName;

        private ProcessingS3Output s3Output;

        private ProcessingFeatureStoreOutput featureStoreOutput;

        private Boolean appManaged;

        private BuilderImpl() {
        }

        private BuilderImpl(ProcessingOutput model) {
            outputName(model.outputName);
            s3Output(model.s3Output);
            featureStoreOutput(model.featureStoreOutput);
            appManaged(model.appManaged);
        }

        public final String getOutputName() {
            return outputName;
        }

        public final void setOutputName(String outputName) {
            this.outputName = outputName;
        }

        @Override
        public final Builder outputName(String outputName) {
            this.outputName = outputName;
            return this;
        }

        public final ProcessingS3Output.Builder getS3Output() {
            return s3Output != null ? s3Output.toBuilder() : null;
        }

        public final void setS3Output(ProcessingS3Output.BuilderImpl s3Output) {
            this.s3Output = s3Output != null ? s3Output.build() : null;
        }

        @Override
        public final Builder s3Output(ProcessingS3Output s3Output) {
            this.s3Output = s3Output;
            return this;
        }

        public final ProcessingFeatureStoreOutput.Builder getFeatureStoreOutput() {
            return featureStoreOutput != null ? featureStoreOutput.toBuilder() : null;
        }

        public final void setFeatureStoreOutput(ProcessingFeatureStoreOutput.BuilderImpl featureStoreOutput) {
            this.featureStoreOutput = featureStoreOutput != null ? featureStoreOutput.build() : null;
        }

        @Override
        public final Builder featureStoreOutput(ProcessingFeatureStoreOutput featureStoreOutput) {
            this.featureStoreOutput = featureStoreOutput;
            return this;
        }

        public final Boolean getAppManaged() {
            return appManaged;
        }

        public final void setAppManaged(Boolean appManaged) {
            this.appManaged = appManaged;
        }

        @Override
        public final Builder appManaged(Boolean appManaged) {
            this.appManaged = appManaged;
            return this;
        }

        @Override
        public ProcessingOutput build() {
            return new ProcessingOutput(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
