/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the configuration for a hyperparameter tuning job that uses one or more previous hyperparameter tuning jobs
 * as a starting point. The results of previous tuning jobs are used to inform which combinations of hyperparameters to
 * search over in the new tuning job.
 * </p>
 * <p>
 * All training jobs launched by the new hyperparameter tuning job are evaluated by using the objective metric, and the
 * training job that performs the best is compared to the best training jobs from the parent tuning jobs. From these,
 * the training job that performs the best as measured by the objective metric is returned as the overall best training
 * job.
 * </p>
 * <note>
 * <p>
 * All training jobs launched by parent hyperparameter tuning jobs and the new hyperparameter tuning jobs count against
 * the limit of training jobs for the tuning job.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HyperParameterTuningJobWarmStartConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<HyperParameterTuningJobWarmStartConfig.Builder, HyperParameterTuningJobWarmStartConfig> {
    private static final SdkField<List<ParentHyperParameterTuningJob>> PARENT_HYPER_PARAMETER_TUNING_JOBS_FIELD = SdkField
            .<List<ParentHyperParameterTuningJob>> builder(MarshallingType.LIST)
            .memberName("ParentHyperParameterTuningJobs")
            .getter(getter(HyperParameterTuningJobWarmStartConfig::parentHyperParameterTuningJobs))
            .setter(setter(Builder::parentHyperParameterTuningJobs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParentHyperParameterTuningJobs")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ParentHyperParameterTuningJob> builder(MarshallingType.SDK_POJO)
                                            .constructor(ParentHyperParameterTuningJob::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> WARM_START_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WarmStartType").getter(getter(HyperParameterTuningJobWarmStartConfig::warmStartTypeAsString))
            .setter(setter(Builder::warmStartType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WarmStartType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            PARENT_HYPER_PARAMETER_TUNING_JOBS_FIELD, WARM_START_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<ParentHyperParameterTuningJob> parentHyperParameterTuningJobs;

    private final String warmStartType;

    private HyperParameterTuningJobWarmStartConfig(BuilderImpl builder) {
        this.parentHyperParameterTuningJobs = builder.parentHyperParameterTuningJobs;
        this.warmStartType = builder.warmStartType;
    }

    /**
     * For responses, this returns true if the service returned a value for the ParentHyperParameterTuningJobs property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasParentHyperParameterTuningJobs() {
        return parentHyperParameterTuningJobs != null && !(parentHyperParameterTuningJobs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of hyperparameter tuning jobs that are used as the starting point for the new hyperparameter tuning job.
     * For more information about warm starting a hyperparameter tuning job, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-warm-start.html">Using a Previous
     * Hyperparameter Tuning Job as a Starting Point</a>.
     * </p>
     * <p>
     * Hyperparameter tuning jobs created before October 1, 2018 cannot be used as parent jobs for warm start tuning
     * jobs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParentHyperParameterTuningJobs} method.
     * </p>
     * 
     * @return An array of hyperparameter tuning jobs that are used as the starting point for the new hyperparameter
     *         tuning job. For more information about warm starting a hyperparameter tuning job, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-warm-start.html">Using a
     *         Previous Hyperparameter Tuning Job as a Starting Point</a>.</p>
     *         <p>
     *         Hyperparameter tuning jobs created before October 1, 2018 cannot be used as parent jobs for warm start
     *         tuning jobs.
     */
    public final List<ParentHyperParameterTuningJob> parentHyperParameterTuningJobs() {
        return parentHyperParameterTuningJobs;
    }

    /**
     * <p>
     * Specifies one of the following:
     * </p>
     * <dl>
     * <dt>IDENTICAL_DATA_AND_ALGORITHM</dt>
     * <dd>
     * <p>
     * The new hyperparameter tuning job uses the same input data and training image as the parent tuning jobs. You can
     * change the hyperparameter ranges to search and the maximum number of training jobs that the hyperparameter tuning
     * job launches. You cannot use a new version of the training algorithm, unless the changes in the new version do
     * not affect the algorithm itself. For example, changes that improve logging or adding support for a different data
     * format are allowed. You can also change hyperparameters from tunable to static, and from static to tunable, but
     * the total number of static plus tunable hyperparameters must remain the same as it is in all parent jobs. The
     * objective metric for the new tuning job must be the same as for all parent jobs.
     * </p>
     * </dd>
     * <dt>TRANSFER_LEARNING</dt>
     * <dd>
     * <p>
     * The new hyperparameter tuning job can include input data, hyperparameter ranges, maximum number of concurrent
     * training jobs, and maximum number of training jobs that are different than those of its parent hyperparameter
     * tuning jobs. The training image can also be a different version from the version used in the parent
     * hyperparameter tuning job. You can also change hyperparameters from tunable to static, and from static to
     * tunable, but the total number of static plus tunable hyperparameters must remain the same as it is in all parent
     * jobs. The objective metric for the new tuning job must be the same as for all parent jobs.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #warmStartType}
     * will return {@link HyperParameterTuningJobWarmStartType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #warmStartTypeAsString}.
     * </p>
     * 
     * @return Specifies one of the following:</p>
     *         <dl>
     *         <dt>IDENTICAL_DATA_AND_ALGORITHM</dt>
     *         <dd>
     *         <p>
     *         The new hyperparameter tuning job uses the same input data and training image as the parent tuning jobs.
     *         You can change the hyperparameter ranges to search and the maximum number of training jobs that the
     *         hyperparameter tuning job launches. You cannot use a new version of the training algorithm, unless the
     *         changes in the new version do not affect the algorithm itself. For example, changes that improve logging
     *         or adding support for a different data format are allowed. You can also change hyperparameters from
     *         tunable to static, and from static to tunable, but the total number of static plus tunable
     *         hyperparameters must remain the same as it is in all parent jobs. The objective metric for the new tuning
     *         job must be the same as for all parent jobs.
     *         </p>
     *         </dd>
     *         <dt>TRANSFER_LEARNING</dt>
     *         <dd>
     *         <p>
     *         The new hyperparameter tuning job can include input data, hyperparameter ranges, maximum number of
     *         concurrent training jobs, and maximum number of training jobs that are different than those of its parent
     *         hyperparameter tuning jobs. The training image can also be a different version from the version used in
     *         the parent hyperparameter tuning job. You can also change hyperparameters from tunable to static, and
     *         from static to tunable, but the total number of static plus tunable hyperparameters must remain the same
     *         as it is in all parent jobs. The objective metric for the new tuning job must be the same as for all
     *         parent jobs.
     *         </p>
     *         </dd>
     * @see HyperParameterTuningJobWarmStartType
     */
    public final HyperParameterTuningJobWarmStartType warmStartType() {
        return HyperParameterTuningJobWarmStartType.fromValue(warmStartType);
    }

    /**
     * <p>
     * Specifies one of the following:
     * </p>
     * <dl>
     * <dt>IDENTICAL_DATA_AND_ALGORITHM</dt>
     * <dd>
     * <p>
     * The new hyperparameter tuning job uses the same input data and training image as the parent tuning jobs. You can
     * change the hyperparameter ranges to search and the maximum number of training jobs that the hyperparameter tuning
     * job launches. You cannot use a new version of the training algorithm, unless the changes in the new version do
     * not affect the algorithm itself. For example, changes that improve logging or adding support for a different data
     * format are allowed. You can also change hyperparameters from tunable to static, and from static to tunable, but
     * the total number of static plus tunable hyperparameters must remain the same as it is in all parent jobs. The
     * objective metric for the new tuning job must be the same as for all parent jobs.
     * </p>
     * </dd>
     * <dt>TRANSFER_LEARNING</dt>
     * <dd>
     * <p>
     * The new hyperparameter tuning job can include input data, hyperparameter ranges, maximum number of concurrent
     * training jobs, and maximum number of training jobs that are different than those of its parent hyperparameter
     * tuning jobs. The training image can also be a different version from the version used in the parent
     * hyperparameter tuning job. You can also change hyperparameters from tunable to static, and from static to
     * tunable, but the total number of static plus tunable hyperparameters must remain the same as it is in all parent
     * jobs. The objective metric for the new tuning job must be the same as for all parent jobs.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #warmStartType}
     * will return {@link HyperParameterTuningJobWarmStartType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #warmStartTypeAsString}.
     * </p>
     * 
     * @return Specifies one of the following:</p>
     *         <dl>
     *         <dt>IDENTICAL_DATA_AND_ALGORITHM</dt>
     *         <dd>
     *         <p>
     *         The new hyperparameter tuning job uses the same input data and training image as the parent tuning jobs.
     *         You can change the hyperparameter ranges to search and the maximum number of training jobs that the
     *         hyperparameter tuning job launches. You cannot use a new version of the training algorithm, unless the
     *         changes in the new version do not affect the algorithm itself. For example, changes that improve logging
     *         or adding support for a different data format are allowed. You can also change hyperparameters from
     *         tunable to static, and from static to tunable, but the total number of static plus tunable
     *         hyperparameters must remain the same as it is in all parent jobs. The objective metric for the new tuning
     *         job must be the same as for all parent jobs.
     *         </p>
     *         </dd>
     *         <dt>TRANSFER_LEARNING</dt>
     *         <dd>
     *         <p>
     *         The new hyperparameter tuning job can include input data, hyperparameter ranges, maximum number of
     *         concurrent training jobs, and maximum number of training jobs that are different than those of its parent
     *         hyperparameter tuning jobs. The training image can also be a different version from the version used in
     *         the parent hyperparameter tuning job. You can also change hyperparameters from tunable to static, and
     *         from static to tunable, but the total number of static plus tunable hyperparameters must remain the same
     *         as it is in all parent jobs. The objective metric for the new tuning job must be the same as for all
     *         parent jobs.
     *         </p>
     *         </dd>
     * @see HyperParameterTuningJobWarmStartType
     */
    public final String warmStartTypeAsString() {
        return warmStartType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode
                + Objects.hashCode(hasParentHyperParameterTuningJobs() ? parentHyperParameterTuningJobs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(warmStartTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HyperParameterTuningJobWarmStartConfig)) {
            return false;
        }
        HyperParameterTuningJobWarmStartConfig other = (HyperParameterTuningJobWarmStartConfig) obj;
        return hasParentHyperParameterTuningJobs() == other.hasParentHyperParameterTuningJobs()
                && Objects.equals(parentHyperParameterTuningJobs(), other.parentHyperParameterTuningJobs())
                && Objects.equals(warmStartTypeAsString(), other.warmStartTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("HyperParameterTuningJobWarmStartConfig")
                .add("ParentHyperParameterTuningJobs",
                        hasParentHyperParameterTuningJobs() ? parentHyperParameterTuningJobs() : null)
                .add("WarmStartType", warmStartTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ParentHyperParameterTuningJobs":
            return Optional.ofNullable(clazz.cast(parentHyperParameterTuningJobs()));
        case "WarmStartType":
            return Optional.ofNullable(clazz.cast(warmStartTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<HyperParameterTuningJobWarmStartConfig, T> g) {
        return obj -> g.apply((HyperParameterTuningJobWarmStartConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HyperParameterTuningJobWarmStartConfig> {
        /**
         * <p>
         * An array of hyperparameter tuning jobs that are used as the starting point for the new hyperparameter tuning
         * job. For more information about warm starting a hyperparameter tuning job, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-warm-start.html">Using a
         * Previous Hyperparameter Tuning Job as a Starting Point</a>.
         * </p>
         * <p>
         * Hyperparameter tuning jobs created before October 1, 2018 cannot be used as parent jobs for warm start tuning
         * jobs.
         * </p>
         * 
         * @param parentHyperParameterTuningJobs
         *        An array of hyperparameter tuning jobs that are used as the starting point for the new hyperparameter
         *        tuning job. For more information about warm starting a hyperparameter tuning job, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-warm-start.html">Using a
         *        Previous Hyperparameter Tuning Job as a Starting Point</a>.</p>
         *        <p>
         *        Hyperparameter tuning jobs created before October 1, 2018 cannot be used as parent jobs for warm start
         *        tuning jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentHyperParameterTuningJobs(Collection<ParentHyperParameterTuningJob> parentHyperParameterTuningJobs);

        /**
         * <p>
         * An array of hyperparameter tuning jobs that are used as the starting point for the new hyperparameter tuning
         * job. For more information about warm starting a hyperparameter tuning job, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-warm-start.html">Using a
         * Previous Hyperparameter Tuning Job as a Starting Point</a>.
         * </p>
         * <p>
         * Hyperparameter tuning jobs created before October 1, 2018 cannot be used as parent jobs for warm start tuning
         * jobs.
         * </p>
         * 
         * @param parentHyperParameterTuningJobs
         *        An array of hyperparameter tuning jobs that are used as the starting point for the new hyperparameter
         *        tuning job. For more information about warm starting a hyperparameter tuning job, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-warm-start.html">Using a
         *        Previous Hyperparameter Tuning Job as a Starting Point</a>.</p>
         *        <p>
         *        Hyperparameter tuning jobs created before October 1, 2018 cannot be used as parent jobs for warm start
         *        tuning jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentHyperParameterTuningJobs(ParentHyperParameterTuningJob... parentHyperParameterTuningJobs);

        /**
         * <p>
         * An array of hyperparameter tuning jobs that are used as the starting point for the new hyperparameter tuning
         * job. For more information about warm starting a hyperparameter tuning job, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-warm-start.html">Using a
         * Previous Hyperparameter Tuning Job as a Starting Point</a>.
         * </p>
         * <p>
         * Hyperparameter tuning jobs created before October 1, 2018 cannot be used as parent jobs for warm start tuning
         * jobs.
         * </p>
         * This is a convenience method that creates an instance of the {@link List
         * <ParentHyperParameterTuningJob>.Builder} avoiding the need to create one manually via {@link List
         * <ParentHyperParameterTuningJob>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ParentHyperParameterTuningJob>.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #parentHyperParameterTuningJobs(List<ParentHyperParameterTuningJob>)}.
         * 
         * @param parentHyperParameterTuningJobs
         *        a consumer that will call methods on {@link List<ParentHyperParameterTuningJob>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parentHyperParameterTuningJobs(List<ParentHyperParameterTuningJob>)
         */
        Builder parentHyperParameterTuningJobs(Consumer<ParentHyperParameterTuningJob.Builder>... parentHyperParameterTuningJobs);

        /**
         * <p>
         * Specifies one of the following:
         * </p>
         * <dl>
         * <dt>IDENTICAL_DATA_AND_ALGORITHM</dt>
         * <dd>
         * <p>
         * The new hyperparameter tuning job uses the same input data and training image as the parent tuning jobs. You
         * can change the hyperparameter ranges to search and the maximum number of training jobs that the
         * hyperparameter tuning job launches. You cannot use a new version of the training algorithm, unless the
         * changes in the new version do not affect the algorithm itself. For example, changes that improve logging or
         * adding support for a different data format are allowed. You can also change hyperparameters from tunable to
         * static, and from static to tunable, but the total number of static plus tunable hyperparameters must remain
         * the same as it is in all parent jobs. The objective metric for the new tuning job must be the same as for all
         * parent jobs.
         * </p>
         * </dd>
         * <dt>TRANSFER_LEARNING</dt>
         * <dd>
         * <p>
         * The new hyperparameter tuning job can include input data, hyperparameter ranges, maximum number of concurrent
         * training jobs, and maximum number of training jobs that are different than those of its parent hyperparameter
         * tuning jobs. The training image can also be a different version from the version used in the parent
         * hyperparameter tuning job. You can also change hyperparameters from tunable to static, and from static to
         * tunable, but the total number of static plus tunable hyperparameters must remain the same as it is in all
         * parent jobs. The objective metric for the new tuning job must be the same as for all parent jobs.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param warmStartType
         *        Specifies one of the following:</p>
         *        <dl>
         *        <dt>IDENTICAL_DATA_AND_ALGORITHM</dt>
         *        <dd>
         *        <p>
         *        The new hyperparameter tuning job uses the same input data and training image as the parent tuning
         *        jobs. You can change the hyperparameter ranges to search and the maximum number of training jobs that
         *        the hyperparameter tuning job launches. You cannot use a new version of the training algorithm, unless
         *        the changes in the new version do not affect the algorithm itself. For example, changes that improve
         *        logging or adding support for a different data format are allowed. You can also change hyperparameters
         *        from tunable to static, and from static to tunable, but the total number of static plus tunable
         *        hyperparameters must remain the same as it is in all parent jobs. The objective metric for the new
         *        tuning job must be the same as for all parent jobs.
         *        </p>
         *        </dd>
         *        <dt>TRANSFER_LEARNING</dt>
         *        <dd>
         *        <p>
         *        The new hyperparameter tuning job can include input data, hyperparameter ranges, maximum number of
         *        concurrent training jobs, and maximum number of training jobs that are different than those of its
         *        parent hyperparameter tuning jobs. The training image can also be a different version from the version
         *        used in the parent hyperparameter tuning job. You can also change hyperparameters from tunable to
         *        static, and from static to tunable, but the total number of static plus tunable hyperparameters must
         *        remain the same as it is in all parent jobs. The objective metric for the new tuning job must be the
         *        same as for all parent jobs.
         *        </p>
         *        </dd>
         * @see HyperParameterTuningJobWarmStartType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HyperParameterTuningJobWarmStartType
         */
        Builder warmStartType(String warmStartType);

        /**
         * <p>
         * Specifies one of the following:
         * </p>
         * <dl>
         * <dt>IDENTICAL_DATA_AND_ALGORITHM</dt>
         * <dd>
         * <p>
         * The new hyperparameter tuning job uses the same input data and training image as the parent tuning jobs. You
         * can change the hyperparameter ranges to search and the maximum number of training jobs that the
         * hyperparameter tuning job launches. You cannot use a new version of the training algorithm, unless the
         * changes in the new version do not affect the algorithm itself. For example, changes that improve logging or
         * adding support for a different data format are allowed. You can also change hyperparameters from tunable to
         * static, and from static to tunable, but the total number of static plus tunable hyperparameters must remain
         * the same as it is in all parent jobs. The objective metric for the new tuning job must be the same as for all
         * parent jobs.
         * </p>
         * </dd>
         * <dt>TRANSFER_LEARNING</dt>
         * <dd>
         * <p>
         * The new hyperparameter tuning job can include input data, hyperparameter ranges, maximum number of concurrent
         * training jobs, and maximum number of training jobs that are different than those of its parent hyperparameter
         * tuning jobs. The training image can also be a different version from the version used in the parent
         * hyperparameter tuning job. You can also change hyperparameters from tunable to static, and from static to
         * tunable, but the total number of static plus tunable hyperparameters must remain the same as it is in all
         * parent jobs. The objective metric for the new tuning job must be the same as for all parent jobs.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param warmStartType
         *        Specifies one of the following:</p>
         *        <dl>
         *        <dt>IDENTICAL_DATA_AND_ALGORITHM</dt>
         *        <dd>
         *        <p>
         *        The new hyperparameter tuning job uses the same input data and training image as the parent tuning
         *        jobs. You can change the hyperparameter ranges to search and the maximum number of training jobs that
         *        the hyperparameter tuning job launches. You cannot use a new version of the training algorithm, unless
         *        the changes in the new version do not affect the algorithm itself. For example, changes that improve
         *        logging or adding support for a different data format are allowed. You can also change hyperparameters
         *        from tunable to static, and from static to tunable, but the total number of static plus tunable
         *        hyperparameters must remain the same as it is in all parent jobs. The objective metric for the new
         *        tuning job must be the same as for all parent jobs.
         *        </p>
         *        </dd>
         *        <dt>TRANSFER_LEARNING</dt>
         *        <dd>
         *        <p>
         *        The new hyperparameter tuning job can include input data, hyperparameter ranges, maximum number of
         *        concurrent training jobs, and maximum number of training jobs that are different than those of its
         *        parent hyperparameter tuning jobs. The training image can also be a different version from the version
         *        used in the parent hyperparameter tuning job. You can also change hyperparameters from tunable to
         *        static, and from static to tunable, but the total number of static plus tunable hyperparameters must
         *        remain the same as it is in all parent jobs. The objective metric for the new tuning job must be the
         *        same as for all parent jobs.
         *        </p>
         *        </dd>
         * @see HyperParameterTuningJobWarmStartType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HyperParameterTuningJobWarmStartType
         */
        Builder warmStartType(HyperParameterTuningJobWarmStartType warmStartType);
    }

    static final class BuilderImpl implements Builder {
        private List<ParentHyperParameterTuningJob> parentHyperParameterTuningJobs = DefaultSdkAutoConstructList.getInstance();

        private String warmStartType;

        private BuilderImpl() {
        }

        private BuilderImpl(HyperParameterTuningJobWarmStartConfig model) {
            parentHyperParameterTuningJobs(model.parentHyperParameterTuningJobs);
            warmStartType(model.warmStartType);
        }

        public final List<ParentHyperParameterTuningJob.Builder> getParentHyperParameterTuningJobs() {
            List<ParentHyperParameterTuningJob.Builder> result = ParentHyperParameterTuningJobsCopier
                    .copyToBuilder(this.parentHyperParameterTuningJobs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setParentHyperParameterTuningJobs(
                Collection<ParentHyperParameterTuningJob.BuilderImpl> parentHyperParameterTuningJobs) {
            this.parentHyperParameterTuningJobs = ParentHyperParameterTuningJobsCopier
                    .copyFromBuilder(parentHyperParameterTuningJobs);
        }

        @Override
        public final Builder parentHyperParameterTuningJobs(
                Collection<ParentHyperParameterTuningJob> parentHyperParameterTuningJobs) {
            this.parentHyperParameterTuningJobs = ParentHyperParameterTuningJobsCopier.copy(parentHyperParameterTuningJobs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parentHyperParameterTuningJobs(ParentHyperParameterTuningJob... parentHyperParameterTuningJobs) {
            parentHyperParameterTuningJobs(Arrays.asList(parentHyperParameterTuningJobs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parentHyperParameterTuningJobs(
                Consumer<ParentHyperParameterTuningJob.Builder>... parentHyperParameterTuningJobs) {
            parentHyperParameterTuningJobs(Stream.of(parentHyperParameterTuningJobs)
                    .map(c -> ParentHyperParameterTuningJob.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getWarmStartType() {
            return warmStartType;
        }

        public final void setWarmStartType(String warmStartType) {
            this.warmStartType = warmStartType;
        }

        @Override
        public final Builder warmStartType(String warmStartType) {
            this.warmStartType = warmStartType;
            return this;
        }

        @Override
        public final Builder warmStartType(HyperParameterTuningJobWarmStartType warmStartType) {
            this.warmStartType(warmStartType == null ? null : warmStartType.toString());
            return this;
        }

        @Override
        public HyperParameterTuningJobWarmStartConfig build() {
            return new HyperParameterTuningJobWarmStartConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
