/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes weight and capacities for a production variant associated with an endpoint. If you sent a request to the
 * <code>UpdateEndpointWeightsAndCapacities</code> API and the endpoint status is <code>Updating</code>, you get
 * different desired and current values.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProductionVariantSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<ProductionVariantSummary.Builder, ProductionVariantSummary> {
    private static final SdkField<String> VARIANT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VariantName").getter(getter(ProductionVariantSummary::variantName)).setter(setter(Builder::variantName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VariantName").build()).build();

    private static final SdkField<List<DeployedImage>> DEPLOYED_IMAGES_FIELD = SdkField
            .<List<DeployedImage>> builder(MarshallingType.LIST)
            .memberName("DeployedImages")
            .getter(getter(ProductionVariantSummary::deployedImages))
            .setter(setter(Builder::deployedImages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeployedImages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DeployedImage> builder(MarshallingType.SDK_POJO)
                                            .constructor(DeployedImage::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Float> CURRENT_WEIGHT_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("CurrentWeight").getter(getter(ProductionVariantSummary::currentWeight))
            .setter(setter(Builder::currentWeight))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CurrentWeight").build()).build();

    private static final SdkField<Float> DESIRED_WEIGHT_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("DesiredWeight").getter(getter(ProductionVariantSummary::desiredWeight))
            .setter(setter(Builder::desiredWeight))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DesiredWeight").build()).build();

    private static final SdkField<Integer> CURRENT_INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("CurrentInstanceCount").getter(getter(ProductionVariantSummary::currentInstanceCount))
            .setter(setter(Builder::currentInstanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CurrentInstanceCount").build())
            .build();

    private static final SdkField<Integer> DESIRED_INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("DesiredInstanceCount").getter(getter(ProductionVariantSummary::desiredInstanceCount))
            .setter(setter(Builder::desiredInstanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DesiredInstanceCount").build())
            .build();

    private static final SdkField<List<ProductionVariantStatus>> VARIANT_STATUS_FIELD = SdkField
            .<List<ProductionVariantStatus>> builder(MarshallingType.LIST)
            .memberName("VariantStatus")
            .getter(getter(ProductionVariantSummary::variantStatus))
            .setter(setter(Builder::variantStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VariantStatus").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ProductionVariantStatus> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProductionVariantStatus::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ProductionVariantServerlessConfig> CURRENT_SERVERLESS_CONFIG_FIELD = SdkField
            .<ProductionVariantServerlessConfig> builder(MarshallingType.SDK_POJO).memberName("CurrentServerlessConfig")
            .getter(getter(ProductionVariantSummary::currentServerlessConfig)).setter(setter(Builder::currentServerlessConfig))
            .constructor(ProductionVariantServerlessConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CurrentServerlessConfig").build())
            .build();

    private static final SdkField<ProductionVariantServerlessConfig> DESIRED_SERVERLESS_CONFIG_FIELD = SdkField
            .<ProductionVariantServerlessConfig> builder(MarshallingType.SDK_POJO).memberName("DesiredServerlessConfig")
            .getter(getter(ProductionVariantSummary::desiredServerlessConfig)).setter(setter(Builder::desiredServerlessConfig))
            .constructor(ProductionVariantServerlessConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DesiredServerlessConfig").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(VARIANT_NAME_FIELD, DEPLOYED_IMAGES_FIELD, CURRENT_WEIGHT_FIELD,
                    DESIRED_WEIGHT_FIELD, CURRENT_INSTANCE_COUNT_FIELD, DESIRED_INSTANCE_COUNT_FIELD, VARIANT_STATUS_FIELD,
                    CURRENT_SERVERLESS_CONFIG_FIELD, DESIRED_SERVERLESS_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String variantName;

    private final List<DeployedImage> deployedImages;

    private final Float currentWeight;

    private final Float desiredWeight;

    private final Integer currentInstanceCount;

    private final Integer desiredInstanceCount;

    private final List<ProductionVariantStatus> variantStatus;

    private final ProductionVariantServerlessConfig currentServerlessConfig;

    private final ProductionVariantServerlessConfig desiredServerlessConfig;

    private ProductionVariantSummary(BuilderImpl builder) {
        this.variantName = builder.variantName;
        this.deployedImages = builder.deployedImages;
        this.currentWeight = builder.currentWeight;
        this.desiredWeight = builder.desiredWeight;
        this.currentInstanceCount = builder.currentInstanceCount;
        this.desiredInstanceCount = builder.desiredInstanceCount;
        this.variantStatus = builder.variantStatus;
        this.currentServerlessConfig = builder.currentServerlessConfig;
        this.desiredServerlessConfig = builder.desiredServerlessConfig;
    }

    /**
     * <p>
     * The name of the variant.
     * </p>
     * 
     * @return The name of the variant.
     */
    public final String variantName() {
        return variantName;
    }

    /**
     * For responses, this returns true if the service returned a value for the DeployedImages property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDeployedImages() {
        return deployedImages != null && !(deployedImages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>DeployedImage</code> objects that specify the Amazon EC2 Container Registry paths of the
     * inference images deployed on instances of this <code>ProductionVariant</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDeployedImages} method.
     * </p>
     * 
     * @return An array of <code>DeployedImage</code> objects that specify the Amazon EC2 Container Registry paths of
     *         the inference images deployed on instances of this <code>ProductionVariant</code>.
     */
    public final List<DeployedImage> deployedImages() {
        return deployedImages;
    }

    /**
     * <p>
     * The weight associated with the variant.
     * </p>
     * 
     * @return The weight associated with the variant.
     */
    public final Float currentWeight() {
        return currentWeight;
    }

    /**
     * <p>
     * The requested weight, as specified in the <code>UpdateEndpointWeightsAndCapacities</code> request.
     * </p>
     * 
     * @return The requested weight, as specified in the <code>UpdateEndpointWeightsAndCapacities</code> request.
     */
    public final Float desiredWeight() {
        return desiredWeight;
    }

    /**
     * <p>
     * The number of instances associated with the variant.
     * </p>
     * 
     * @return The number of instances associated with the variant.
     */
    public final Integer currentInstanceCount() {
        return currentInstanceCount;
    }

    /**
     * <p>
     * The number of instances requested in the <code>UpdateEndpointWeightsAndCapacities</code> request.
     * </p>
     * 
     * @return The number of instances requested in the <code>UpdateEndpointWeightsAndCapacities</code> request.
     */
    public final Integer desiredInstanceCount() {
        return desiredInstanceCount;
    }

    /**
     * For responses, this returns true if the service returned a value for the VariantStatus property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVariantStatus() {
        return variantStatus != null && !(variantStatus instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The endpoint variant status which describes the current deployment stage status or operational status.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVariantStatus} method.
     * </p>
     * 
     * @return The endpoint variant status which describes the current deployment stage status or operational status.
     */
    public final List<ProductionVariantStatus> variantStatus() {
        return variantStatus;
    }

    /**
     * <p>
     * The serverless configuration for the endpoint.
     * </p>
     * 
     * @return The serverless configuration for the endpoint.
     */
    public final ProductionVariantServerlessConfig currentServerlessConfig() {
        return currentServerlessConfig;
    }

    /**
     * <p>
     * The serverless configuration requested for the endpoint update.
     * </p>
     * 
     * @return The serverless configuration requested for the endpoint update.
     */
    public final ProductionVariantServerlessConfig desiredServerlessConfig() {
        return desiredServerlessConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(variantName());
        hashCode = 31 * hashCode + Objects.hashCode(hasDeployedImages() ? deployedImages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(currentWeight());
        hashCode = 31 * hashCode + Objects.hashCode(desiredWeight());
        hashCode = 31 * hashCode + Objects.hashCode(currentInstanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(desiredInstanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(hasVariantStatus() ? variantStatus() : null);
        hashCode = 31 * hashCode + Objects.hashCode(currentServerlessConfig());
        hashCode = 31 * hashCode + Objects.hashCode(desiredServerlessConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProductionVariantSummary)) {
            return false;
        }
        ProductionVariantSummary other = (ProductionVariantSummary) obj;
        return Objects.equals(variantName(), other.variantName()) && hasDeployedImages() == other.hasDeployedImages()
                && Objects.equals(deployedImages(), other.deployedImages())
                && Objects.equals(currentWeight(), other.currentWeight())
                && Objects.equals(desiredWeight(), other.desiredWeight())
                && Objects.equals(currentInstanceCount(), other.currentInstanceCount())
                && Objects.equals(desiredInstanceCount(), other.desiredInstanceCount())
                && hasVariantStatus() == other.hasVariantStatus() && Objects.equals(variantStatus(), other.variantStatus())
                && Objects.equals(currentServerlessConfig(), other.currentServerlessConfig())
                && Objects.equals(desiredServerlessConfig(), other.desiredServerlessConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProductionVariantSummary").add("VariantName", variantName())
                .add("DeployedImages", hasDeployedImages() ? deployedImages() : null).add("CurrentWeight", currentWeight())
                .add("DesiredWeight", desiredWeight()).add("CurrentInstanceCount", currentInstanceCount())
                .add("DesiredInstanceCount", desiredInstanceCount())
                .add("VariantStatus", hasVariantStatus() ? variantStatus() : null)
                .add("CurrentServerlessConfig", currentServerlessConfig())
                .add("DesiredServerlessConfig", desiredServerlessConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VariantName":
            return Optional.ofNullable(clazz.cast(variantName()));
        case "DeployedImages":
            return Optional.ofNullable(clazz.cast(deployedImages()));
        case "CurrentWeight":
            return Optional.ofNullable(clazz.cast(currentWeight()));
        case "DesiredWeight":
            return Optional.ofNullable(clazz.cast(desiredWeight()));
        case "CurrentInstanceCount":
            return Optional.ofNullable(clazz.cast(currentInstanceCount()));
        case "DesiredInstanceCount":
            return Optional.ofNullable(clazz.cast(desiredInstanceCount()));
        case "VariantStatus":
            return Optional.ofNullable(clazz.cast(variantStatus()));
        case "CurrentServerlessConfig":
            return Optional.ofNullable(clazz.cast(currentServerlessConfig()));
        case "DesiredServerlessConfig":
            return Optional.ofNullable(clazz.cast(desiredServerlessConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProductionVariantSummary, T> g) {
        return obj -> g.apply((ProductionVariantSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProductionVariantSummary> {
        /**
         * <p>
         * The name of the variant.
         * </p>
         * 
         * @param variantName
         *        The name of the variant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variantName(String variantName);

        /**
         * <p>
         * An array of <code>DeployedImage</code> objects that specify the Amazon EC2 Container Registry paths of the
         * inference images deployed on instances of this <code>ProductionVariant</code>.
         * </p>
         * 
         * @param deployedImages
         *        An array of <code>DeployedImage</code> objects that specify the Amazon EC2 Container Registry paths of
         *        the inference images deployed on instances of this <code>ProductionVariant</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deployedImages(Collection<DeployedImage> deployedImages);

        /**
         * <p>
         * An array of <code>DeployedImage</code> objects that specify the Amazon EC2 Container Registry paths of the
         * inference images deployed on instances of this <code>ProductionVariant</code>.
         * </p>
         * 
         * @param deployedImages
         *        An array of <code>DeployedImage</code> objects that specify the Amazon EC2 Container Registry paths of
         *        the inference images deployed on instances of this <code>ProductionVariant</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deployedImages(DeployedImage... deployedImages);

        /**
         * <p>
         * An array of <code>DeployedImage</code> objects that specify the Amazon EC2 Container Registry paths of the
         * inference images deployed on instances of this <code>ProductionVariant</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<DeployedImage>.Builder} avoiding the
         * need to create one manually via {@link List<DeployedImage>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DeployedImage>.Builder#build()} is called immediately and
         * its result is passed to {@link #deployedImages(List<DeployedImage>)}.
         * 
         * @param deployedImages
         *        a consumer that will call methods on {@link List<DeployedImage>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deployedImages(List<DeployedImage>)
         */
        Builder deployedImages(Consumer<DeployedImage.Builder>... deployedImages);

        /**
         * <p>
         * The weight associated with the variant.
         * </p>
         * 
         * @param currentWeight
         *        The weight associated with the variant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentWeight(Float currentWeight);

        /**
         * <p>
         * The requested weight, as specified in the <code>UpdateEndpointWeightsAndCapacities</code> request.
         * </p>
         * 
         * @param desiredWeight
         *        The requested weight, as specified in the <code>UpdateEndpointWeightsAndCapacities</code> request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredWeight(Float desiredWeight);

        /**
         * <p>
         * The number of instances associated with the variant.
         * </p>
         * 
         * @param currentInstanceCount
         *        The number of instances associated with the variant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentInstanceCount(Integer currentInstanceCount);

        /**
         * <p>
         * The number of instances requested in the <code>UpdateEndpointWeightsAndCapacities</code> request.
         * </p>
         * 
         * @param desiredInstanceCount
         *        The number of instances requested in the <code>UpdateEndpointWeightsAndCapacities</code> request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredInstanceCount(Integer desiredInstanceCount);

        /**
         * <p>
         * The endpoint variant status which describes the current deployment stage status or operational status.
         * </p>
         * 
         * @param variantStatus
         *        The endpoint variant status which describes the current deployment stage status or operational status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variantStatus(Collection<ProductionVariantStatus> variantStatus);

        /**
         * <p>
         * The endpoint variant status which describes the current deployment stage status or operational status.
         * </p>
         * 
         * @param variantStatus
         *        The endpoint variant status which describes the current deployment stage status or operational status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variantStatus(ProductionVariantStatus... variantStatus);

        /**
         * <p>
         * The endpoint variant status which describes the current deployment stage status or operational status.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<ProductionVariantStatus>.Builder}
         * avoiding the need to create one manually via {@link List<ProductionVariantStatus>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ProductionVariantStatus>.Builder#build()} is called
         * immediately and its result is passed to {@link #variantStatus(List<ProductionVariantStatus>)}.
         * 
         * @param variantStatus
         *        a consumer that will call methods on {@link List<ProductionVariantStatus>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #variantStatus(List<ProductionVariantStatus>)
         */
        Builder variantStatus(Consumer<ProductionVariantStatus.Builder>... variantStatus);

        /**
         * <p>
         * The serverless configuration for the endpoint.
         * </p>
         * 
         * @param currentServerlessConfig
         *        The serverless configuration for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentServerlessConfig(ProductionVariantServerlessConfig currentServerlessConfig);

        /**
         * <p>
         * The serverless configuration for the endpoint.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ProductionVariantServerlessConfig.Builder} avoiding the need to create one manually via
         * {@link ProductionVariantServerlessConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProductionVariantServerlessConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #currentServerlessConfig(ProductionVariantServerlessConfig)}.
         * 
         * @param currentServerlessConfig
         *        a consumer that will call methods on {@link ProductionVariantServerlessConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #currentServerlessConfig(ProductionVariantServerlessConfig)
         */
        default Builder currentServerlessConfig(Consumer<ProductionVariantServerlessConfig.Builder> currentServerlessConfig) {
            return currentServerlessConfig(ProductionVariantServerlessConfig.builder().applyMutation(currentServerlessConfig)
                    .build());
        }

        /**
         * <p>
         * The serverless configuration requested for the endpoint update.
         * </p>
         * 
         * @param desiredServerlessConfig
         *        The serverless configuration requested for the endpoint update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredServerlessConfig(ProductionVariantServerlessConfig desiredServerlessConfig);

        /**
         * <p>
         * The serverless configuration requested for the endpoint update.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ProductionVariantServerlessConfig.Builder} avoiding the need to create one manually via
         * {@link ProductionVariantServerlessConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProductionVariantServerlessConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #desiredServerlessConfig(ProductionVariantServerlessConfig)}.
         * 
         * @param desiredServerlessConfig
         *        a consumer that will call methods on {@link ProductionVariantServerlessConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #desiredServerlessConfig(ProductionVariantServerlessConfig)
         */
        default Builder desiredServerlessConfig(Consumer<ProductionVariantServerlessConfig.Builder> desiredServerlessConfig) {
            return desiredServerlessConfig(ProductionVariantServerlessConfig.builder().applyMutation(desiredServerlessConfig)
                    .build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String variantName;

        private List<DeployedImage> deployedImages = DefaultSdkAutoConstructList.getInstance();

        private Float currentWeight;

        private Float desiredWeight;

        private Integer currentInstanceCount;

        private Integer desiredInstanceCount;

        private List<ProductionVariantStatus> variantStatus = DefaultSdkAutoConstructList.getInstance();

        private ProductionVariantServerlessConfig currentServerlessConfig;

        private ProductionVariantServerlessConfig desiredServerlessConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(ProductionVariantSummary model) {
            variantName(model.variantName);
            deployedImages(model.deployedImages);
            currentWeight(model.currentWeight);
            desiredWeight(model.desiredWeight);
            currentInstanceCount(model.currentInstanceCount);
            desiredInstanceCount(model.desiredInstanceCount);
            variantStatus(model.variantStatus);
            currentServerlessConfig(model.currentServerlessConfig);
            desiredServerlessConfig(model.desiredServerlessConfig);
        }

        public final String getVariantName() {
            return variantName;
        }

        public final void setVariantName(String variantName) {
            this.variantName = variantName;
        }

        @Override
        public final Builder variantName(String variantName) {
            this.variantName = variantName;
            return this;
        }

        public final List<DeployedImage.Builder> getDeployedImages() {
            List<DeployedImage.Builder> result = DeployedImagesCopier.copyToBuilder(this.deployedImages);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDeployedImages(Collection<DeployedImage.BuilderImpl> deployedImages) {
            this.deployedImages = DeployedImagesCopier.copyFromBuilder(deployedImages);
        }

        @Override
        public final Builder deployedImages(Collection<DeployedImage> deployedImages) {
            this.deployedImages = DeployedImagesCopier.copy(deployedImages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder deployedImages(DeployedImage... deployedImages) {
            deployedImages(Arrays.asList(deployedImages));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder deployedImages(Consumer<DeployedImage.Builder>... deployedImages) {
            deployedImages(Stream.of(deployedImages).map(c -> DeployedImage.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Float getCurrentWeight() {
            return currentWeight;
        }

        public final void setCurrentWeight(Float currentWeight) {
            this.currentWeight = currentWeight;
        }

        @Override
        public final Builder currentWeight(Float currentWeight) {
            this.currentWeight = currentWeight;
            return this;
        }

        public final Float getDesiredWeight() {
            return desiredWeight;
        }

        public final void setDesiredWeight(Float desiredWeight) {
            this.desiredWeight = desiredWeight;
        }

        @Override
        public final Builder desiredWeight(Float desiredWeight) {
            this.desiredWeight = desiredWeight;
            return this;
        }

        public final Integer getCurrentInstanceCount() {
            return currentInstanceCount;
        }

        public final void setCurrentInstanceCount(Integer currentInstanceCount) {
            this.currentInstanceCount = currentInstanceCount;
        }

        @Override
        public final Builder currentInstanceCount(Integer currentInstanceCount) {
            this.currentInstanceCount = currentInstanceCount;
            return this;
        }

        public final Integer getDesiredInstanceCount() {
            return desiredInstanceCount;
        }

        public final void setDesiredInstanceCount(Integer desiredInstanceCount) {
            this.desiredInstanceCount = desiredInstanceCount;
        }

        @Override
        public final Builder desiredInstanceCount(Integer desiredInstanceCount) {
            this.desiredInstanceCount = desiredInstanceCount;
            return this;
        }

        public final List<ProductionVariantStatus.Builder> getVariantStatus() {
            List<ProductionVariantStatus.Builder> result = ProductionVariantStatusListCopier.copyToBuilder(this.variantStatus);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVariantStatus(Collection<ProductionVariantStatus.BuilderImpl> variantStatus) {
            this.variantStatus = ProductionVariantStatusListCopier.copyFromBuilder(variantStatus);
        }

        @Override
        public final Builder variantStatus(Collection<ProductionVariantStatus> variantStatus) {
            this.variantStatus = ProductionVariantStatusListCopier.copy(variantStatus);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variantStatus(ProductionVariantStatus... variantStatus) {
            variantStatus(Arrays.asList(variantStatus));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variantStatus(Consumer<ProductionVariantStatus.Builder>... variantStatus) {
            variantStatus(Stream.of(variantStatus).map(c -> ProductionVariantStatus.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final ProductionVariantServerlessConfig.Builder getCurrentServerlessConfig() {
            return currentServerlessConfig != null ? currentServerlessConfig.toBuilder() : null;
        }

        public final void setCurrentServerlessConfig(ProductionVariantServerlessConfig.BuilderImpl currentServerlessConfig) {
            this.currentServerlessConfig = currentServerlessConfig != null ? currentServerlessConfig.build() : null;
        }

        @Override
        public final Builder currentServerlessConfig(ProductionVariantServerlessConfig currentServerlessConfig) {
            this.currentServerlessConfig = currentServerlessConfig;
            return this;
        }

        public final ProductionVariantServerlessConfig.Builder getDesiredServerlessConfig() {
            return desiredServerlessConfig != null ? desiredServerlessConfig.toBuilder() : null;
        }

        public final void setDesiredServerlessConfig(ProductionVariantServerlessConfig.BuilderImpl desiredServerlessConfig) {
            this.desiredServerlessConfig = desiredServerlessConfig != null ? desiredServerlessConfig.build() : null;
        }

        @Override
        public final Builder desiredServerlessConfig(ProductionVariantServerlessConfig desiredServerlessConfig) {
            this.desiredServerlessConfig = desiredServerlessConfig;
            return this;
        }

        @Override
        public ProductionVariantSummary build() {
            return new ProductionVariantSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
