/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration information for SageMaker Debugger rules for debugging. To learn more about how to configure the
 * <code>DebugRuleConfiguration</code> parameter, see <a
 * href="https://docs.aws.amazon.com/sagemaker/latest/dg/debugger-createtrainingjob-api.html">Use the SageMaker and
 * Debugger Configuration API Operations to Create, Update, and Debug Your Training Job</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DebugRuleConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<DebugRuleConfiguration.Builder, DebugRuleConfiguration> {
    private static final SdkField<String> RULE_CONFIGURATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleConfigurationName").getter(getter(DebugRuleConfiguration::ruleConfigurationName))
            .setter(setter(Builder::ruleConfigurationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleConfigurationName").build())
            .build();

    private static final SdkField<String> LOCAL_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LocalPath").getter(getter(DebugRuleConfiguration::localPath)).setter(setter(Builder::localPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocalPath").build()).build();

    private static final SdkField<String> S3_OUTPUT_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3OutputPath").getter(getter(DebugRuleConfiguration::s3OutputPath))
            .setter(setter(Builder::s3OutputPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3OutputPath").build()).build();

    private static final SdkField<String> RULE_EVALUATOR_IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleEvaluatorImage").getter(getter(DebugRuleConfiguration::ruleEvaluatorImage))
            .setter(setter(Builder::ruleEvaluatorImage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleEvaluatorImage").build())
            .build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(DebugRuleConfiguration::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Integer> VOLUME_SIZE_IN_GB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("VolumeSizeInGB").getter(getter(DebugRuleConfiguration::volumeSizeInGB))
            .setter(setter(Builder::volumeSizeInGB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeSizeInGB").build()).build();

    private static final SdkField<Map<String, String>> RULE_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("RuleParameters")
            .getter(getter(DebugRuleConfiguration::ruleParameters))
            .setter(setter(Builder::ruleParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RULE_CONFIGURATION_NAME_FIELD,
            LOCAL_PATH_FIELD, S3_OUTPUT_PATH_FIELD, RULE_EVALUATOR_IMAGE_FIELD, INSTANCE_TYPE_FIELD, VOLUME_SIZE_IN_GB_FIELD,
            RULE_PARAMETERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ruleConfigurationName;

    private final String localPath;

    private final String s3OutputPath;

    private final String ruleEvaluatorImage;

    private final String instanceType;

    private final Integer volumeSizeInGB;

    private final Map<String, String> ruleParameters;

    private DebugRuleConfiguration(BuilderImpl builder) {
        this.ruleConfigurationName = builder.ruleConfigurationName;
        this.localPath = builder.localPath;
        this.s3OutputPath = builder.s3OutputPath;
        this.ruleEvaluatorImage = builder.ruleEvaluatorImage;
        this.instanceType = builder.instanceType;
        this.volumeSizeInGB = builder.volumeSizeInGB;
        this.ruleParameters = builder.ruleParameters;
    }

    /**
     * <p>
     * The name of the rule configuration. It must be unique relative to other rule configuration names.
     * </p>
     * 
     * @return The name of the rule configuration. It must be unique relative to other rule configuration names.
     */
    public final String ruleConfigurationName() {
        return ruleConfigurationName;
    }

    /**
     * <p>
     * Path to local storage location for output of rules. Defaults to <code>/opt/ml/processing/output/rule/</code>.
     * </p>
     * 
     * @return Path to local storage location for output of rules. Defaults to
     *         <code>/opt/ml/processing/output/rule/</code>.
     */
    public final String localPath() {
        return localPath;
    }

    /**
     * <p>
     * Path to Amazon S3 storage location for rules.
     * </p>
     * 
     * @return Path to Amazon S3 storage location for rules.
     */
    public final String s3OutputPath() {
        return s3OutputPath;
    }

    /**
     * <p>
     * The Amazon Elastic Container (ECR) Image for the managed rule evaluation.
     * </p>
     * 
     * @return The Amazon Elastic Container (ECR) Image for the managed rule evaluation.
     */
    public final String ruleEvaluatorImage() {
        return ruleEvaluatorImage;
    }

    /**
     * <p>
     * The instance type to deploy a Debugger custom rule for debugging a training job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link ProcessingInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type to deploy a Debugger custom rule for debugging a training job.
     * @see ProcessingInstanceType
     */
    public final ProcessingInstanceType instanceType() {
        return ProcessingInstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The instance type to deploy a Debugger custom rule for debugging a training job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link ProcessingInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type to deploy a Debugger custom rule for debugging a training job.
     * @see ProcessingInstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * The size, in GB, of the ML storage volume attached to the processing instance.
     * </p>
     * 
     * @return The size, in GB, of the ML storage volume attached to the processing instance.
     */
    public final Integer volumeSizeInGB() {
        return volumeSizeInGB;
    }

    /**
     * For responses, this returns true if the service returned a value for the RuleParameters property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRuleParameters() {
        return ruleParameters != null && !(ruleParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Runtime configuration for rule container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRuleParameters} method.
     * </p>
     * 
     * @return Runtime configuration for rule container.
     */
    public final Map<String, String> ruleParameters() {
        return ruleParameters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ruleConfigurationName());
        hashCode = 31 * hashCode + Objects.hashCode(localPath());
        hashCode = 31 * hashCode + Objects.hashCode(s3OutputPath());
        hashCode = 31 * hashCode + Objects.hashCode(ruleEvaluatorImage());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(volumeSizeInGB());
        hashCode = 31 * hashCode + Objects.hashCode(hasRuleParameters() ? ruleParameters() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DebugRuleConfiguration)) {
            return false;
        }
        DebugRuleConfiguration other = (DebugRuleConfiguration) obj;
        return Objects.equals(ruleConfigurationName(), other.ruleConfigurationName())
                && Objects.equals(localPath(), other.localPath()) && Objects.equals(s3OutputPath(), other.s3OutputPath())
                && Objects.equals(ruleEvaluatorImage(), other.ruleEvaluatorImage())
                && Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(volumeSizeInGB(), other.volumeSizeInGB()) && hasRuleParameters() == other.hasRuleParameters()
                && Objects.equals(ruleParameters(), other.ruleParameters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DebugRuleConfiguration").add("RuleConfigurationName", ruleConfigurationName())
                .add("LocalPath", localPath()).add("S3OutputPath", s3OutputPath())
                .add("RuleEvaluatorImage", ruleEvaluatorImage()).add("InstanceType", instanceTypeAsString())
                .add("VolumeSizeInGB", volumeSizeInGB()).add("RuleParameters", hasRuleParameters() ? ruleParameters() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RuleConfigurationName":
            return Optional.ofNullable(clazz.cast(ruleConfigurationName()));
        case "LocalPath":
            return Optional.ofNullable(clazz.cast(localPath()));
        case "S3OutputPath":
            return Optional.ofNullable(clazz.cast(s3OutputPath()));
        case "RuleEvaluatorImage":
            return Optional.ofNullable(clazz.cast(ruleEvaluatorImage()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "VolumeSizeInGB":
            return Optional.ofNullable(clazz.cast(volumeSizeInGB()));
        case "RuleParameters":
            return Optional.ofNullable(clazz.cast(ruleParameters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DebugRuleConfiguration, T> g) {
        return obj -> g.apply((DebugRuleConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DebugRuleConfiguration> {
        /**
         * <p>
         * The name of the rule configuration. It must be unique relative to other rule configuration names.
         * </p>
         * 
         * @param ruleConfigurationName
         *        The name of the rule configuration. It must be unique relative to other rule configuration names.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleConfigurationName(String ruleConfigurationName);

        /**
         * <p>
         * Path to local storage location for output of rules. Defaults to <code>/opt/ml/processing/output/rule/</code>.
         * </p>
         * 
         * @param localPath
         *        Path to local storage location for output of rules. Defaults to
         *        <code>/opt/ml/processing/output/rule/</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localPath(String localPath);

        /**
         * <p>
         * Path to Amazon S3 storage location for rules.
         * </p>
         * 
         * @param s3OutputPath
         *        Path to Amazon S3 storage location for rules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3OutputPath(String s3OutputPath);

        /**
         * <p>
         * The Amazon Elastic Container (ECR) Image for the managed rule evaluation.
         * </p>
         * 
         * @param ruleEvaluatorImage
         *        The Amazon Elastic Container (ECR) Image for the managed rule evaluation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleEvaluatorImage(String ruleEvaluatorImage);

        /**
         * <p>
         * The instance type to deploy a Debugger custom rule for debugging a training job.
         * </p>
         * 
         * @param instanceType
         *        The instance type to deploy a Debugger custom rule for debugging a training job.
         * @see ProcessingInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProcessingInstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The instance type to deploy a Debugger custom rule for debugging a training job.
         * </p>
         * 
         * @param instanceType
         *        The instance type to deploy a Debugger custom rule for debugging a training job.
         * @see ProcessingInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProcessingInstanceType
         */
        Builder instanceType(ProcessingInstanceType instanceType);

        /**
         * <p>
         * The size, in GB, of the ML storage volume attached to the processing instance.
         * </p>
         * 
         * @param volumeSizeInGB
         *        The size, in GB, of the ML storage volume attached to the processing instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeSizeInGB(Integer volumeSizeInGB);

        /**
         * <p>
         * Runtime configuration for rule container.
         * </p>
         * 
         * @param ruleParameters
         *        Runtime configuration for rule container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleParameters(Map<String, String> ruleParameters);
    }

    static final class BuilderImpl implements Builder {
        private String ruleConfigurationName;

        private String localPath;

        private String s3OutputPath;

        private String ruleEvaluatorImage;

        private String instanceType;

        private Integer volumeSizeInGB;

        private Map<String, String> ruleParameters = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DebugRuleConfiguration model) {
            ruleConfigurationName(model.ruleConfigurationName);
            localPath(model.localPath);
            s3OutputPath(model.s3OutputPath);
            ruleEvaluatorImage(model.ruleEvaluatorImage);
            instanceType(model.instanceType);
            volumeSizeInGB(model.volumeSizeInGB);
            ruleParameters(model.ruleParameters);
        }

        public final String getRuleConfigurationName() {
            return ruleConfigurationName;
        }

        public final void setRuleConfigurationName(String ruleConfigurationName) {
            this.ruleConfigurationName = ruleConfigurationName;
        }

        @Override
        public final Builder ruleConfigurationName(String ruleConfigurationName) {
            this.ruleConfigurationName = ruleConfigurationName;
            return this;
        }

        public final String getLocalPath() {
            return localPath;
        }

        public final void setLocalPath(String localPath) {
            this.localPath = localPath;
        }

        @Override
        public final Builder localPath(String localPath) {
            this.localPath = localPath;
            return this;
        }

        public final String getS3OutputPath() {
            return s3OutputPath;
        }

        public final void setS3OutputPath(String s3OutputPath) {
            this.s3OutputPath = s3OutputPath;
        }

        @Override
        public final Builder s3OutputPath(String s3OutputPath) {
            this.s3OutputPath = s3OutputPath;
            return this;
        }

        public final String getRuleEvaluatorImage() {
            return ruleEvaluatorImage;
        }

        public final void setRuleEvaluatorImage(String ruleEvaluatorImage) {
            this.ruleEvaluatorImage = ruleEvaluatorImage;
        }

        @Override
        public final Builder ruleEvaluatorImage(String ruleEvaluatorImage) {
            this.ruleEvaluatorImage = ruleEvaluatorImage;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(ProcessingInstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final Integer getVolumeSizeInGB() {
            return volumeSizeInGB;
        }

        public final void setVolumeSizeInGB(Integer volumeSizeInGB) {
            this.volumeSizeInGB = volumeSizeInGB;
        }

        @Override
        public final Builder volumeSizeInGB(Integer volumeSizeInGB) {
            this.volumeSizeInGB = volumeSizeInGB;
            return this;
        }

        public final Map<String, String> getRuleParameters() {
            if (ruleParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return ruleParameters;
        }

        public final void setRuleParameters(Map<String, String> ruleParameters) {
            this.ruleParameters = RuleParametersCopier.copy(ruleParameters);
        }

        @Override
        public final Builder ruleParameters(Map<String, String> ruleParameters) {
            this.ruleParameters = RuleParametersCopier.copy(ruleParameters);
            return this;
        }

        @Override
        public DebugRuleConfiguration build() {
            return new DebugRuleConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
