/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The inference configuration parameter for the model container.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ClarifyInferenceConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<ClarifyInferenceConfig.Builder, ClarifyInferenceConfig> {
    private static final SdkField<String> FEATURES_ATTRIBUTE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FeaturesAttribute").getter(getter(ClarifyInferenceConfig::featuresAttribute))
            .setter(setter(Builder::featuresAttribute))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeaturesAttribute").build()).build();

    private static final SdkField<String> CONTENT_TEMPLATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContentTemplate").getter(getter(ClarifyInferenceConfig::contentTemplate))
            .setter(setter(Builder::contentTemplate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentTemplate").build()).build();

    private static final SdkField<Integer> MAX_RECORD_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxRecordCount").getter(getter(ClarifyInferenceConfig::maxRecordCount))
            .setter(setter(Builder::maxRecordCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxRecordCount").build()).build();

    private static final SdkField<Integer> MAX_PAYLOAD_IN_MB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxPayloadInMB").getter(getter(ClarifyInferenceConfig::maxPayloadInMB))
            .setter(setter(Builder::maxPayloadInMB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxPayloadInMB").build()).build();

    private static final SdkField<Integer> PROBABILITY_INDEX_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ProbabilityIndex").getter(getter(ClarifyInferenceConfig::probabilityIndex))
            .setter(setter(Builder::probabilityIndex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProbabilityIndex").build()).build();

    private static final SdkField<Integer> LABEL_INDEX_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("LabelIndex").getter(getter(ClarifyInferenceConfig::labelIndex)).setter(setter(Builder::labelIndex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LabelIndex").build()).build();

    private static final SdkField<String> PROBABILITY_ATTRIBUTE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProbabilityAttribute").getter(getter(ClarifyInferenceConfig::probabilityAttribute))
            .setter(setter(Builder::probabilityAttribute))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProbabilityAttribute").build())
            .build();

    private static final SdkField<String> LABEL_ATTRIBUTE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LabelAttribute").getter(getter(ClarifyInferenceConfig::labelAttribute))
            .setter(setter(Builder::labelAttribute))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LabelAttribute").build()).build();

    private static final SdkField<List<String>> LABEL_HEADERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LabelHeaders")
            .getter(getter(ClarifyInferenceConfig::labelHeaders))
            .setter(setter(Builder::labelHeaders))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LabelHeaders").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> FEATURE_HEADERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("FeatureHeaders")
            .getter(getter(ClarifyInferenceConfig::featureHeaders))
            .setter(setter(Builder::featureHeaders))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeatureHeaders").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> FEATURE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("FeatureTypes")
            .getter(getter(ClarifyInferenceConfig::featureTypesAsStrings))
            .setter(setter(Builder::featureTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeatureTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FEATURES_ATTRIBUTE_FIELD,
            CONTENT_TEMPLATE_FIELD, MAX_RECORD_COUNT_FIELD, MAX_PAYLOAD_IN_MB_FIELD, PROBABILITY_INDEX_FIELD, LABEL_INDEX_FIELD,
            PROBABILITY_ATTRIBUTE_FIELD, LABEL_ATTRIBUTE_FIELD, LABEL_HEADERS_FIELD, FEATURE_HEADERS_FIELD, FEATURE_TYPES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String featuresAttribute;

    private final String contentTemplate;

    private final Integer maxRecordCount;

    private final Integer maxPayloadInMB;

    private final Integer probabilityIndex;

    private final Integer labelIndex;

    private final String probabilityAttribute;

    private final String labelAttribute;

    private final List<String> labelHeaders;

    private final List<String> featureHeaders;

    private final List<String> featureTypes;

    private ClarifyInferenceConfig(BuilderImpl builder) {
        this.featuresAttribute = builder.featuresAttribute;
        this.contentTemplate = builder.contentTemplate;
        this.maxRecordCount = builder.maxRecordCount;
        this.maxPayloadInMB = builder.maxPayloadInMB;
        this.probabilityIndex = builder.probabilityIndex;
        this.labelIndex = builder.labelIndex;
        this.probabilityAttribute = builder.probabilityAttribute;
        this.labelAttribute = builder.labelAttribute;
        this.labelHeaders = builder.labelHeaders;
        this.featureHeaders = builder.featureHeaders;
        this.featureTypes = builder.featureTypes;
    }

    /**
     * <p>
     * Provides the JMESPath expression to extract the features from a model container input in JSON Lines format. For
     * example, if <code>FeaturesAttribute</code> is the JMESPath expression <code>'myfeatures'</code>, it extracts a
     * list of features <code>[1,2,3]</code> from request data <code>'{"myfeatures":[1,2,3}'</code>.
     * </p>
     * 
     * @return Provides the JMESPath expression to extract the features from a model container input in JSON Lines
     *         format. For example, if <code>FeaturesAttribute</code> is the JMESPath expression
     *         <code>'myfeatures'</code>, it extracts a list of features <code>[1,2,3]</code> from request data
     *         <code>'{"myfeatures":[1,2,3}'</code>.
     */
    public final String featuresAttribute() {
        return featuresAttribute;
    }

    /**
     * <p>
     * A template string used to format a JSON record into an acceptable model container input. For example, a
     * <code>ContentTemplate</code> string <code>'{"myfeatures":$features}'</code> will format a list of features
     * <code>[1,2,3]</code> into the record string <code>'{"myfeatures":[1,2,3]}'</code>. Required only when the model
     * container input is in JSON Lines format.
     * </p>
     * 
     * @return A template string used to format a JSON record into an acceptable model container input. For example, a
     *         <code>ContentTemplate</code> string <code>'{"myfeatures":$features}'</code> will format a list of
     *         features <code>[1,2,3]</code> into the record string <code>'{"myfeatures":[1,2,3]}'</code>. Required only
     *         when the model container input is in JSON Lines format.
     */
    public final String contentTemplate() {
        return contentTemplate;
    }

    /**
     * <p>
     * The maximum number of records in a request that the model container can process when querying the model container
     * for the predictions of a <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-create-endpoint.html#clarify-online-explainability-create-endpoint-synthetic"
     * >synthetic dataset</a>. A record is a unit of input data that inference can be made on, for example, a single
     * line in CSV data. If <code>MaxRecordCount</code> is <code>1</code>, the model container expects one record per
     * request. A value of 2 or greater means that the model expects batch requests, which can reduce overhead and speed
     * up the inferencing process. If this parameter is not provided, the explainer will tune the record count per
     * request according to the model container's capacity at runtime.
     * </p>
     * 
     * @return The maximum number of records in a request that the model container can process when querying the model
     *         container for the predictions of a <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-create-endpoint.html#clarify-online-explainability-create-endpoint-synthetic"
     *         >synthetic dataset</a>. A record is a unit of input data that inference can be made on, for example, a
     *         single line in CSV data. If <code>MaxRecordCount</code> is <code>1</code>, the model container expects
     *         one record per request. A value of 2 or greater means that the model expects batch requests, which can
     *         reduce overhead and speed up the inferencing process. If this parameter is not provided, the explainer
     *         will tune the record count per request according to the model container's capacity at runtime.
     */
    public final Integer maxRecordCount() {
        return maxRecordCount;
    }

    /**
     * <p>
     * The maximum payload size (MB) allowed of a request from the explainer to the model container. Defaults to
     * <code>6</code> MB.
     * </p>
     * 
     * @return The maximum payload size (MB) allowed of a request from the explainer to the model container. Defaults to
     *         <code>6</code> MB.
     */
    public final Integer maxPayloadInMB() {
        return maxPayloadInMB;
    }

    /**
     * <p>
     * A zero-based index used to extract a probability value (score) or list from model container output in CSV format.
     * If this value is not provided, the entire model container output will be treated as a probability value (score)
     * or list.
     * </p>
     * <p>
     * <b>Example for a single class model:</b> If the model container output consists of a string-formatted prediction
     * label followed by its probability: <code>'1,0.6'</code>, set <code>ProbabilityIndex</code> to <code>1</code> to
     * select the probability value <code>0.6</code>.
     * </p>
     * <p>
     * <b>Example for a multiclass model:</b> If the model container output consists of a string-formatted prediction
     * label followed by its probability: <code>'"[\'cat\',\'dog\',\'fish\']","[0.1,0.6,0.3]"'</code>, set
     * <code>ProbabilityIndex</code> to <code>1</code> to select the probability values <code>[0.1,0.6,0.3]</code>.
     * </p>
     * 
     * @return A zero-based index used to extract a probability value (score) or list from model container output in CSV
     *         format. If this value is not provided, the entire model container output will be treated as a probability
     *         value (score) or list.</p>
     *         <p>
     *         <b>Example for a single class model:</b> If the model container output consists of a string-formatted
     *         prediction label followed by its probability: <code>'1,0.6'</code>, set <code>ProbabilityIndex</code> to
     *         <code>1</code> to select the probability value <code>0.6</code>.
     *         </p>
     *         <p>
     *         <b>Example for a multiclass model:</b> If the model container output consists of a string-formatted
     *         prediction label followed by its probability: <code>'"[\'cat\',\'dog\',\'fish\']","[0.1,0.6,0.3]"'</code>
     *         , set <code>ProbabilityIndex</code> to <code>1</code> to select the probability values
     *         <code>[0.1,0.6,0.3]</code>.
     */
    public final Integer probabilityIndex() {
        return probabilityIndex;
    }

    /**
     * <p>
     * A zero-based index used to extract a label header or list of label headers from model container output in CSV
     * format.
     * </p>
     * <p>
     * <b>Example for a multiclass model:</b> If the model container output consists of label headers followed by
     * probabilities: <code>'"[\'cat\',\'dog\',\'fish\']","[0.1,0.6,0.3]"'</code>, set <code>LabelIndex</code> to
     * <code>0</code> to select the label headers <code>['cat','dog','fish']</code>.
     * </p>
     * 
     * @return A zero-based index used to extract a label header or list of label headers from model container output in
     *         CSV format.</p>
     *         <p>
     *         <b>Example for a multiclass model:</b> If the model container output consists of label headers followed
     *         by probabilities: <code>'"[\'cat\',\'dog\',\'fish\']","[0.1,0.6,0.3]"'</code>, set
     *         <code>LabelIndex</code> to <code>0</code> to select the label headers <code>['cat','dog','fish']</code>.
     */
    public final Integer labelIndex() {
        return labelIndex;
    }

    /**
     * <p>
     * A JMESPath expression used to extract the probability (or score) from the model container output if the model
     * container is in JSON Lines format.
     * </p>
     * <p>
     * <b>Example</b>: If the model container output of a single request is
     * <code>'{"predicted_label":1,"probability":0.6}'</code>, then set <code>ProbabilityAttribute</code> to
     * <code>'probability'</code>.
     * </p>
     * 
     * @return A JMESPath expression used to extract the probability (or score) from the model container output if the
     *         model container is in JSON Lines format.</p>
     *         <p>
     *         <b>Example</b>: If the model container output of a single request is
     *         <code>'{"predicted_label":1,"probability":0.6}'</code>, then set <code>ProbabilityAttribute</code> to
     *         <code>'probability'</code>.
     */
    public final String probabilityAttribute() {
        return probabilityAttribute;
    }

    /**
     * <p>
     * A JMESPath expression used to locate the list of label headers in the model container output.
     * </p>
     * <p>
     * <b>Example</b>: If the model container output of a batch request is
     * <code>'{"labels":["cat","dog","fish"],"probability":[0.6,0.3,0.1]}'</code>, then set <code>LabelAttribute</code>
     * to <code>'labels'</code> to extract the list of label headers <code>["cat","dog","fish"]</code>
     * </p>
     * 
     * @return A JMESPath expression used to locate the list of label headers in the model container output.</p>
     *         <p>
     *         <b>Example</b>: If the model container output of a batch request is
     *         <code>'{"labels":["cat","dog","fish"],"probability":[0.6,0.3,0.1]}'</code>, then set
     *         <code>LabelAttribute</code> to <code>'labels'</code> to extract the list of label headers
     *         <code>["cat","dog","fish"]</code>
     */
    public final String labelAttribute() {
        return labelAttribute;
    }

    /**
     * For responses, this returns true if the service returned a value for the LabelHeaders property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLabelHeaders() {
        return labelHeaders != null && !(labelHeaders instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * For multiclass classification problems, the label headers are the names of the classes. Otherwise, the label
     * header is the name of the predicted label. These are used to help readability for the output of the
     * <code>InvokeEndpoint</code> API. See the <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
     * >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information. If there are
     * no label headers in the model container output, provide them manually using this parameter.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLabelHeaders} method.
     * </p>
     * 
     * @return For multiclass classification problems, the label headers are the names of the classes. Otherwise, the
     *         label header is the name of the predicted label. These are used to help readability for the output of the
     *         <code>InvokeEndpoint</code> API. See the <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
     *         >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information. If
     *         there are no label headers in the model container output, provide them manually using this parameter.
     */
    public final List<String> labelHeaders() {
        return labelHeaders;
    }

    /**
     * For responses, this returns true if the service returned a value for the FeatureHeaders property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFeatureHeaders() {
        return featureHeaders != null && !(featureHeaders instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The names of the features. If provided, these are included in the endpoint response payload to help readability
     * of the <code>InvokeEndpoint</code> output. See the <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
     * >Response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeatureHeaders} method.
     * </p>
     * 
     * @return The names of the features. If provided, these are included in the endpoint response payload to help
     *         readability of the <code>InvokeEndpoint</code> output. See the <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
     *         >Response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
     */
    public final List<String> featureHeaders() {
        return featureHeaders;
    }

    /**
     * <p>
     * A list of data types of the features (optional). Applicable only to NLP explainability. If provided,
     * <code>FeatureTypes</code> must have at least one <code>'text'</code> string (for example, <code>['text']</code>).
     * If <code>FeatureTypes</code> is not provided, the explainer infers the feature types based on the baseline data.
     * The feature types are included in the endpoint response payload. For additional information see the <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
     * >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeatureTypes} method.
     * </p>
     * 
     * @return A list of data types of the features (optional). Applicable only to NLP explainability. If provided,
     *         <code>FeatureTypes</code> must have at least one <code>'text'</code> string (for example,
     *         <code>['text']</code>). If <code>FeatureTypes</code> is not provided, the explainer infers the feature
     *         types based on the baseline data. The feature types are included in the endpoint response payload. For
     *         additional information see the <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
     *         >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
     */
    public final List<ClarifyFeatureType> featureTypes() {
        return ClarifyFeatureTypesCopier.copyStringToEnum(featureTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the FeatureTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFeatureTypes() {
        return featureTypes != null && !(featureTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of data types of the features (optional). Applicable only to NLP explainability. If provided,
     * <code>FeatureTypes</code> must have at least one <code>'text'</code> string (for example, <code>['text']</code>).
     * If <code>FeatureTypes</code> is not provided, the explainer infers the feature types based on the baseline data.
     * The feature types are included in the endpoint response payload. For additional information see the <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
     * >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeatureTypes} method.
     * </p>
     * 
     * @return A list of data types of the features (optional). Applicable only to NLP explainability. If provided,
     *         <code>FeatureTypes</code> must have at least one <code>'text'</code> string (for example,
     *         <code>['text']</code>). If <code>FeatureTypes</code> is not provided, the explainer infers the feature
     *         types based on the baseline data. The feature types are included in the endpoint response payload. For
     *         additional information see the <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
     *         >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
     */
    public final List<String> featureTypesAsStrings() {
        return featureTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(featuresAttribute());
        hashCode = 31 * hashCode + Objects.hashCode(contentTemplate());
        hashCode = 31 * hashCode + Objects.hashCode(maxRecordCount());
        hashCode = 31 * hashCode + Objects.hashCode(maxPayloadInMB());
        hashCode = 31 * hashCode + Objects.hashCode(probabilityIndex());
        hashCode = 31 * hashCode + Objects.hashCode(labelIndex());
        hashCode = 31 * hashCode + Objects.hashCode(probabilityAttribute());
        hashCode = 31 * hashCode + Objects.hashCode(labelAttribute());
        hashCode = 31 * hashCode + Objects.hashCode(hasLabelHeaders() ? labelHeaders() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasFeatureHeaders() ? featureHeaders() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasFeatureTypes() ? featureTypesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClarifyInferenceConfig)) {
            return false;
        }
        ClarifyInferenceConfig other = (ClarifyInferenceConfig) obj;
        return Objects.equals(featuresAttribute(), other.featuresAttribute())
                && Objects.equals(contentTemplate(), other.contentTemplate())
                && Objects.equals(maxRecordCount(), other.maxRecordCount())
                && Objects.equals(maxPayloadInMB(), other.maxPayloadInMB())
                && Objects.equals(probabilityIndex(), other.probabilityIndex())
                && Objects.equals(labelIndex(), other.labelIndex())
                && Objects.equals(probabilityAttribute(), other.probabilityAttribute())
                && Objects.equals(labelAttribute(), other.labelAttribute()) && hasLabelHeaders() == other.hasLabelHeaders()
                && Objects.equals(labelHeaders(), other.labelHeaders()) && hasFeatureHeaders() == other.hasFeatureHeaders()
                && Objects.equals(featureHeaders(), other.featureHeaders()) && hasFeatureTypes() == other.hasFeatureTypes()
                && Objects.equals(featureTypesAsStrings(), other.featureTypesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ClarifyInferenceConfig").add("FeaturesAttribute", featuresAttribute())
                .add("ContentTemplate", contentTemplate()).add("MaxRecordCount", maxRecordCount())
                .add("MaxPayloadInMB", maxPayloadInMB()).add("ProbabilityIndex", probabilityIndex())
                .add("LabelIndex", labelIndex()).add("ProbabilityAttribute", probabilityAttribute())
                .add("LabelAttribute", labelAttribute()).add("LabelHeaders", hasLabelHeaders() ? labelHeaders() : null)
                .add("FeatureHeaders", hasFeatureHeaders() ? featureHeaders() : null)
                .add("FeatureTypes", hasFeatureTypes() ? featureTypesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FeaturesAttribute":
            return Optional.ofNullable(clazz.cast(featuresAttribute()));
        case "ContentTemplate":
            return Optional.ofNullable(clazz.cast(contentTemplate()));
        case "MaxRecordCount":
            return Optional.ofNullable(clazz.cast(maxRecordCount()));
        case "MaxPayloadInMB":
            return Optional.ofNullable(clazz.cast(maxPayloadInMB()));
        case "ProbabilityIndex":
            return Optional.ofNullable(clazz.cast(probabilityIndex()));
        case "LabelIndex":
            return Optional.ofNullable(clazz.cast(labelIndex()));
        case "ProbabilityAttribute":
            return Optional.ofNullable(clazz.cast(probabilityAttribute()));
        case "LabelAttribute":
            return Optional.ofNullable(clazz.cast(labelAttribute()));
        case "LabelHeaders":
            return Optional.ofNullable(clazz.cast(labelHeaders()));
        case "FeatureHeaders":
            return Optional.ofNullable(clazz.cast(featureHeaders()));
        case "FeatureTypes":
            return Optional.ofNullable(clazz.cast(featureTypesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ClarifyInferenceConfig, T> g) {
        return obj -> g.apply((ClarifyInferenceConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ClarifyInferenceConfig> {
        /**
         * <p>
         * Provides the JMESPath expression to extract the features from a model container input in JSON Lines format.
         * For example, if <code>FeaturesAttribute</code> is the JMESPath expression <code>'myfeatures'</code>, it
         * extracts a list of features <code>[1,2,3]</code> from request data <code>'{"myfeatures":[1,2,3}'</code>.
         * </p>
         * 
         * @param featuresAttribute
         *        Provides the JMESPath expression to extract the features from a model container input in JSON Lines
         *        format. For example, if <code>FeaturesAttribute</code> is the JMESPath expression
         *        <code>'myfeatures'</code>, it extracts a list of features <code>[1,2,3]</code> from request data
         *        <code>'{"myfeatures":[1,2,3}'</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featuresAttribute(String featuresAttribute);

        /**
         * <p>
         * A template string used to format a JSON record into an acceptable model container input. For example, a
         * <code>ContentTemplate</code> string <code>'{"myfeatures":$features}'</code> will format a list of features
         * <code>[1,2,3]</code> into the record string <code>'{"myfeatures":[1,2,3]}'</code>. Required only when the
         * model container input is in JSON Lines format.
         * </p>
         * 
         * @param contentTemplate
         *        A template string used to format a JSON record into an acceptable model container input. For example,
         *        a <code>ContentTemplate</code> string <code>'{"myfeatures":$features}'</code> will format a list of
         *        features <code>[1,2,3]</code> into the record string <code>'{"myfeatures":[1,2,3]}'</code>. Required
         *        only when the model container input is in JSON Lines format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentTemplate(String contentTemplate);

        /**
         * <p>
         * The maximum number of records in a request that the model container can process when querying the model
         * container for the predictions of a <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-create-endpoint.html#clarify-online-explainability-create-endpoint-synthetic"
         * >synthetic dataset</a>. A record is a unit of input data that inference can be made on, for example, a single
         * line in CSV data. If <code>MaxRecordCount</code> is <code>1</code>, the model container expects one record
         * per request. A value of 2 or greater means that the model expects batch requests, which can reduce overhead
         * and speed up the inferencing process. If this parameter is not provided, the explainer will tune the record
         * count per request according to the model container's capacity at runtime.
         * </p>
         * 
         * @param maxRecordCount
         *        The maximum number of records in a request that the model container can process when querying the
         *        model container for the predictions of a <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-create-endpoint.html#clarify-online-explainability-create-endpoint-synthetic"
         *        >synthetic dataset</a>. A record is a unit of input data that inference can be made on, for example, a
         *        single line in CSV data. If <code>MaxRecordCount</code> is <code>1</code>, the model container expects
         *        one record per request. A value of 2 or greater means that the model expects batch requests, which can
         *        reduce overhead and speed up the inferencing process. If this parameter is not provided, the explainer
         *        will tune the record count per request according to the model container's capacity at runtime.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxRecordCount(Integer maxRecordCount);

        /**
         * <p>
         * The maximum payload size (MB) allowed of a request from the explainer to the model container. Defaults to
         * <code>6</code> MB.
         * </p>
         * 
         * @param maxPayloadInMB
         *        The maximum payload size (MB) allowed of a request from the explainer to the model container. Defaults
         *        to <code>6</code> MB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxPayloadInMB(Integer maxPayloadInMB);

        /**
         * <p>
         * A zero-based index used to extract a probability value (score) or list from model container output in CSV
         * format. If this value is not provided, the entire model container output will be treated as a probability
         * value (score) or list.
         * </p>
         * <p>
         * <b>Example for a single class model:</b> If the model container output consists of a string-formatted
         * prediction label followed by its probability: <code>'1,0.6'</code>, set <code>ProbabilityIndex</code> to
         * <code>1</code> to select the probability value <code>0.6</code>.
         * </p>
         * <p>
         * <b>Example for a multiclass model:</b> If the model container output consists of a string-formatted
         * prediction label followed by its probability: <code>'"[\'cat\',\'dog\',\'fish\']","[0.1,0.6,0.3]"'</code>,
         * set <code>ProbabilityIndex</code> to <code>1</code> to select the probability values
         * <code>[0.1,0.6,0.3]</code>.
         * </p>
         * 
         * @param probabilityIndex
         *        A zero-based index used to extract a probability value (score) or list from model container output in
         *        CSV format. If this value is not provided, the entire model container output will be treated as a
         *        probability value (score) or list.</p>
         *        <p>
         *        <b>Example for a single class model:</b> If the model container output consists of a string-formatted
         *        prediction label followed by its probability: <code>'1,0.6'</code>, set <code>ProbabilityIndex</code>
         *        to <code>1</code> to select the probability value <code>0.6</code>.
         *        </p>
         *        <p>
         *        <b>Example for a multiclass model:</b> If the model container output consists of a string-formatted
         *        prediction label followed by its probability:
         *        <code>'"[\'cat\',\'dog\',\'fish\']","[0.1,0.6,0.3]"'</code>, set <code>ProbabilityIndex</code> to
         *        <code>1</code> to select the probability values <code>[0.1,0.6,0.3]</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder probabilityIndex(Integer probabilityIndex);

        /**
         * <p>
         * A zero-based index used to extract a label header or list of label headers from model container output in CSV
         * format.
         * </p>
         * <p>
         * <b>Example for a multiclass model:</b> If the model container output consists of label headers followed by
         * probabilities: <code>'"[\'cat\',\'dog\',\'fish\']","[0.1,0.6,0.3]"'</code>, set <code>LabelIndex</code> to
         * <code>0</code> to select the label headers <code>['cat','dog','fish']</code>.
         * </p>
         * 
         * @param labelIndex
         *        A zero-based index used to extract a label header or list of label headers from model container output
         *        in CSV format.</p>
         *        <p>
         *        <b>Example for a multiclass model:</b> If the model container output consists of label headers
         *        followed by probabilities: <code>'"[\'cat\',\'dog\',\'fish\']","[0.1,0.6,0.3]"'</code>, set
         *        <code>LabelIndex</code> to <code>0</code> to select the label headers
         *        <code>['cat','dog','fish']</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labelIndex(Integer labelIndex);

        /**
         * <p>
         * A JMESPath expression used to extract the probability (or score) from the model container output if the model
         * container is in JSON Lines format.
         * </p>
         * <p>
         * <b>Example</b>: If the model container output of a single request is
         * <code>'{"predicted_label":1,"probability":0.6}'</code>, then set <code>ProbabilityAttribute</code> to
         * <code>'probability'</code>.
         * </p>
         * 
         * @param probabilityAttribute
         *        A JMESPath expression used to extract the probability (or score) from the model container output if
         *        the model container is in JSON Lines format.</p>
         *        <p>
         *        <b>Example</b>: If the model container output of a single request is
         *        <code>'{"predicted_label":1,"probability":0.6}'</code>, then set <code>ProbabilityAttribute</code> to
         *        <code>'probability'</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder probabilityAttribute(String probabilityAttribute);

        /**
         * <p>
         * A JMESPath expression used to locate the list of label headers in the model container output.
         * </p>
         * <p>
         * <b>Example</b>: If the model container output of a batch request is
         * <code>'{"labels":["cat","dog","fish"],"probability":[0.6,0.3,0.1]}'</code>, then set
         * <code>LabelAttribute</code> to <code>'labels'</code> to extract the list of label headers
         * <code>["cat","dog","fish"]</code>
         * </p>
         * 
         * @param labelAttribute
         *        A JMESPath expression used to locate the list of label headers in the model container output.</p>
         *        <p>
         *        <b>Example</b>: If the model container output of a batch request is
         *        <code>'{"labels":["cat","dog","fish"],"probability":[0.6,0.3,0.1]}'</code>, then set
         *        <code>LabelAttribute</code> to <code>'labels'</code> to extract the list of label headers
         *        <code>["cat","dog","fish"]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labelAttribute(String labelAttribute);

        /**
         * <p>
         * For multiclass classification problems, the label headers are the names of the classes. Otherwise, the label
         * header is the name of the predicted label. These are used to help readability for the output of the
         * <code>InvokeEndpoint</code> API. See the <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         * >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information. If there
         * are no label headers in the model container output, provide them manually using this parameter.
         * </p>
         * 
         * @param labelHeaders
         *        For multiclass classification problems, the label headers are the names of the classes. Otherwise, the
         *        label header is the name of the predicted label. These are used to help readability for the output of
         *        the <code>InvokeEndpoint</code> API. See the <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         *        >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information. If
         *        there are no label headers in the model container output, provide them manually using this parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labelHeaders(Collection<String> labelHeaders);

        /**
         * <p>
         * For multiclass classification problems, the label headers are the names of the classes. Otherwise, the label
         * header is the name of the predicted label. These are used to help readability for the output of the
         * <code>InvokeEndpoint</code> API. See the <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         * >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information. If there
         * are no label headers in the model container output, provide them manually using this parameter.
         * </p>
         * 
         * @param labelHeaders
         *        For multiclass classification problems, the label headers are the names of the classes. Otherwise, the
         *        label header is the name of the predicted label. These are used to help readability for the output of
         *        the <code>InvokeEndpoint</code> API. See the <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         *        >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information. If
         *        there are no label headers in the model container output, provide them manually using this parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labelHeaders(String... labelHeaders);

        /**
         * <p>
         * The names of the features. If provided, these are included in the endpoint response payload to help
         * readability of the <code>InvokeEndpoint</code> output. See the <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         * >Response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
         * </p>
         * 
         * @param featureHeaders
         *        The names of the features. If provided, these are included in the endpoint response payload to help
         *        readability of the <code>InvokeEndpoint</code> output. See the <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         *        >Response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureHeaders(Collection<String> featureHeaders);

        /**
         * <p>
         * The names of the features. If provided, these are included in the endpoint response payload to help
         * readability of the <code>InvokeEndpoint</code> output. See the <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         * >Response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
         * </p>
         * 
         * @param featureHeaders
         *        The names of the features. If provided, these are included in the endpoint response payload to help
         *        readability of the <code>InvokeEndpoint</code> output. See the <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         *        >Response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureHeaders(String... featureHeaders);

        /**
         * <p>
         * A list of data types of the features (optional). Applicable only to NLP explainability. If provided,
         * <code>FeatureTypes</code> must have at least one <code>'text'</code> string (for example,
         * <code>['text']</code>). If <code>FeatureTypes</code> is not provided, the explainer infers the feature types
         * based on the baseline data. The feature types are included in the endpoint response payload. For additional
         * information see the <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         * >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
         * </p>
         * 
         * @param featureTypes
         *        A list of data types of the features (optional). Applicable only to NLP explainability. If provided,
         *        <code>FeatureTypes</code> must have at least one <code>'text'</code> string (for example,
         *        <code>['text']</code>). If <code>FeatureTypes</code> is not provided, the explainer infers the feature
         *        types based on the baseline data. The feature types are included in the endpoint response payload. For
         *        additional information see the <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         *        >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTypesWithStrings(Collection<String> featureTypes);

        /**
         * <p>
         * A list of data types of the features (optional). Applicable only to NLP explainability. If provided,
         * <code>FeatureTypes</code> must have at least one <code>'text'</code> string (for example,
         * <code>['text']</code>). If <code>FeatureTypes</code> is not provided, the explainer infers the feature types
         * based on the baseline data. The feature types are included in the endpoint response payload. For additional
         * information see the <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         * >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
         * </p>
         * 
         * @param featureTypes
         *        A list of data types of the features (optional). Applicable only to NLP explainability. If provided,
         *        <code>FeatureTypes</code> must have at least one <code>'text'</code> string (for example,
         *        <code>['text']</code>). If <code>FeatureTypes</code> is not provided, the explainer infers the feature
         *        types based on the baseline data. The feature types are included in the endpoint response payload. For
         *        additional information see the <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         *        >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTypesWithStrings(String... featureTypes);

        /**
         * <p>
         * A list of data types of the features (optional). Applicable only to NLP explainability. If provided,
         * <code>FeatureTypes</code> must have at least one <code>'text'</code> string (for example,
         * <code>['text']</code>). If <code>FeatureTypes</code> is not provided, the explainer infers the feature types
         * based on the baseline data. The feature types are included in the endpoint response payload. For additional
         * information see the <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         * >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
         * </p>
         * 
         * @param featureTypes
         *        A list of data types of the features (optional). Applicable only to NLP explainability. If provided,
         *        <code>FeatureTypes</code> must have at least one <code>'text'</code> string (for example,
         *        <code>['text']</code>). If <code>FeatureTypes</code> is not provided, the explainer infers the feature
         *        types based on the baseline data. The feature types are included in the endpoint response payload. For
         *        additional information see the <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         *        >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTypes(Collection<ClarifyFeatureType> featureTypes);

        /**
         * <p>
         * A list of data types of the features (optional). Applicable only to NLP explainability. If provided,
         * <code>FeatureTypes</code> must have at least one <code>'text'</code> string (for example,
         * <code>['text']</code>). If <code>FeatureTypes</code> is not provided, the explainer infers the feature types
         * based on the baseline data. The feature types are included in the endpoint response payload. For additional
         * information see the <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         * >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
         * </p>
         * 
         * @param featureTypes
         *        A list of data types of the features (optional). Applicable only to NLP explainability. If provided,
         *        <code>FeatureTypes</code> must have at least one <code>'text'</code> string (for example,
         *        <code>['text']</code>). If <code>FeatureTypes</code> is not provided, the explainer infers the feature
         *        types based on the baseline data. The feature types are included in the endpoint response payload. For
         *        additional information see the <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-invoke-endpoint.html#clarify-online-explainability-response"
         *        >response</a> section under <b>Invoke the endpoint</b> in the Developer Guide for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTypes(ClarifyFeatureType... featureTypes);
    }

    static final class BuilderImpl implements Builder {
        private String featuresAttribute;

        private String contentTemplate;

        private Integer maxRecordCount;

        private Integer maxPayloadInMB;

        private Integer probabilityIndex;

        private Integer labelIndex;

        private String probabilityAttribute;

        private String labelAttribute;

        private List<String> labelHeaders = DefaultSdkAutoConstructList.getInstance();

        private List<String> featureHeaders = DefaultSdkAutoConstructList.getInstance();

        private List<String> featureTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ClarifyInferenceConfig model) {
            featuresAttribute(model.featuresAttribute);
            contentTemplate(model.contentTemplate);
            maxRecordCount(model.maxRecordCount);
            maxPayloadInMB(model.maxPayloadInMB);
            probabilityIndex(model.probabilityIndex);
            labelIndex(model.labelIndex);
            probabilityAttribute(model.probabilityAttribute);
            labelAttribute(model.labelAttribute);
            labelHeaders(model.labelHeaders);
            featureHeaders(model.featureHeaders);
            featureTypesWithStrings(model.featureTypes);
        }

        public final String getFeaturesAttribute() {
            return featuresAttribute;
        }

        public final void setFeaturesAttribute(String featuresAttribute) {
            this.featuresAttribute = featuresAttribute;
        }

        @Override
        public final Builder featuresAttribute(String featuresAttribute) {
            this.featuresAttribute = featuresAttribute;
            return this;
        }

        public final String getContentTemplate() {
            return contentTemplate;
        }

        public final void setContentTemplate(String contentTemplate) {
            this.contentTemplate = contentTemplate;
        }

        @Override
        public final Builder contentTemplate(String contentTemplate) {
            this.contentTemplate = contentTemplate;
            return this;
        }

        public final Integer getMaxRecordCount() {
            return maxRecordCount;
        }

        public final void setMaxRecordCount(Integer maxRecordCount) {
            this.maxRecordCount = maxRecordCount;
        }

        @Override
        public final Builder maxRecordCount(Integer maxRecordCount) {
            this.maxRecordCount = maxRecordCount;
            return this;
        }

        public final Integer getMaxPayloadInMB() {
            return maxPayloadInMB;
        }

        public final void setMaxPayloadInMB(Integer maxPayloadInMB) {
            this.maxPayloadInMB = maxPayloadInMB;
        }

        @Override
        public final Builder maxPayloadInMB(Integer maxPayloadInMB) {
            this.maxPayloadInMB = maxPayloadInMB;
            return this;
        }

        public final Integer getProbabilityIndex() {
            return probabilityIndex;
        }

        public final void setProbabilityIndex(Integer probabilityIndex) {
            this.probabilityIndex = probabilityIndex;
        }

        @Override
        public final Builder probabilityIndex(Integer probabilityIndex) {
            this.probabilityIndex = probabilityIndex;
            return this;
        }

        public final Integer getLabelIndex() {
            return labelIndex;
        }

        public final void setLabelIndex(Integer labelIndex) {
            this.labelIndex = labelIndex;
        }

        @Override
        public final Builder labelIndex(Integer labelIndex) {
            this.labelIndex = labelIndex;
            return this;
        }

        public final String getProbabilityAttribute() {
            return probabilityAttribute;
        }

        public final void setProbabilityAttribute(String probabilityAttribute) {
            this.probabilityAttribute = probabilityAttribute;
        }

        @Override
        public final Builder probabilityAttribute(String probabilityAttribute) {
            this.probabilityAttribute = probabilityAttribute;
            return this;
        }

        public final String getLabelAttribute() {
            return labelAttribute;
        }

        public final void setLabelAttribute(String labelAttribute) {
            this.labelAttribute = labelAttribute;
        }

        @Override
        public final Builder labelAttribute(String labelAttribute) {
            this.labelAttribute = labelAttribute;
            return this;
        }

        public final Collection<String> getLabelHeaders() {
            if (labelHeaders instanceof SdkAutoConstructList) {
                return null;
            }
            return labelHeaders;
        }

        public final void setLabelHeaders(Collection<String> labelHeaders) {
            this.labelHeaders = ClarifyLabelHeadersCopier.copy(labelHeaders);
        }

        @Override
        public final Builder labelHeaders(Collection<String> labelHeaders) {
            this.labelHeaders = ClarifyLabelHeadersCopier.copy(labelHeaders);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder labelHeaders(String... labelHeaders) {
            labelHeaders(Arrays.asList(labelHeaders));
            return this;
        }

        public final Collection<String> getFeatureHeaders() {
            if (featureHeaders instanceof SdkAutoConstructList) {
                return null;
            }
            return featureHeaders;
        }

        public final void setFeatureHeaders(Collection<String> featureHeaders) {
            this.featureHeaders = ClarifyFeatureHeadersCopier.copy(featureHeaders);
        }

        @Override
        public final Builder featureHeaders(Collection<String> featureHeaders) {
            this.featureHeaders = ClarifyFeatureHeadersCopier.copy(featureHeaders);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder featureHeaders(String... featureHeaders) {
            featureHeaders(Arrays.asList(featureHeaders));
            return this;
        }

        public final Collection<String> getFeatureTypes() {
            if (featureTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return featureTypes;
        }

        public final void setFeatureTypes(Collection<String> featureTypes) {
            this.featureTypes = ClarifyFeatureTypesCopier.copy(featureTypes);
        }

        @Override
        public final Builder featureTypesWithStrings(Collection<String> featureTypes) {
            this.featureTypes = ClarifyFeatureTypesCopier.copy(featureTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder featureTypesWithStrings(String... featureTypes) {
            featureTypesWithStrings(Arrays.asList(featureTypes));
            return this;
        }

        @Override
        public final Builder featureTypes(Collection<ClarifyFeatureType> featureTypes) {
            this.featureTypes = ClarifyFeatureTypesCopier.copyEnumToString(featureTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder featureTypes(ClarifyFeatureType... featureTypes) {
            featureTypes(Arrays.asList(featureTypes));
            return this;
        }

        @Override
        public ClarifyInferenceConfig build() {
            return new ClarifyInferenceConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
