/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration to control how SageMaker captures inference data.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataCaptureConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<DataCaptureConfig.Builder, DataCaptureConfig> {
    private static final SdkField<Boolean> ENABLE_CAPTURE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableCapture").getter(getter(DataCaptureConfig::enableCapture)).setter(setter(Builder::enableCapture))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableCapture").build()).build();

    private static final SdkField<Integer> INITIAL_SAMPLING_PERCENTAGE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("InitialSamplingPercentage")
            .getter(getter(DataCaptureConfig::initialSamplingPercentage)).setter(setter(Builder::initialSamplingPercentage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitialSamplingPercentage").build())
            .build();

    private static final SdkField<String> DESTINATION_S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DestinationS3Uri").getter(getter(DataCaptureConfig::destinationS3Uri))
            .setter(setter(Builder::destinationS3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationS3Uri").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(DataCaptureConfig::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<List<CaptureOption>> CAPTURE_OPTIONS_FIELD = SdkField
            .<List<CaptureOption>> builder(MarshallingType.LIST)
            .memberName("CaptureOptions")
            .getter(getter(DataCaptureConfig::captureOptions))
            .setter(setter(Builder::captureOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CaptureOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CaptureOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(CaptureOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<CaptureContentTypeHeader> CAPTURE_CONTENT_TYPE_HEADER_FIELD = SdkField
            .<CaptureContentTypeHeader> builder(MarshallingType.SDK_POJO).memberName("CaptureContentTypeHeader")
            .getter(getter(DataCaptureConfig::captureContentTypeHeader)).setter(setter(Builder::captureContentTypeHeader))
            .constructor(CaptureContentTypeHeader::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CaptureContentTypeHeader").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENABLE_CAPTURE_FIELD,
            INITIAL_SAMPLING_PERCENTAGE_FIELD, DESTINATION_S3_URI_FIELD, KMS_KEY_ID_FIELD, CAPTURE_OPTIONS_FIELD,
            CAPTURE_CONTENT_TYPE_HEADER_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean enableCapture;

    private final Integer initialSamplingPercentage;

    private final String destinationS3Uri;

    private final String kmsKeyId;

    private final List<CaptureOption> captureOptions;

    private final CaptureContentTypeHeader captureContentTypeHeader;

    private DataCaptureConfig(BuilderImpl builder) {
        this.enableCapture = builder.enableCapture;
        this.initialSamplingPercentage = builder.initialSamplingPercentage;
        this.destinationS3Uri = builder.destinationS3Uri;
        this.kmsKeyId = builder.kmsKeyId;
        this.captureOptions = builder.captureOptions;
        this.captureContentTypeHeader = builder.captureContentTypeHeader;
    }

    /**
     * <p>
     * Whether data capture should be enabled or disabled (defaults to enabled).
     * </p>
     * 
     * @return Whether data capture should be enabled or disabled (defaults to enabled).
     */
    public final Boolean enableCapture() {
        return enableCapture;
    }

    /**
     * <p>
     * The percentage of requests SageMaker will capture. A lower value is recommended for Endpoints with high traffic.
     * </p>
     * 
     * @return The percentage of requests SageMaker will capture. A lower value is recommended for Endpoints with high
     *         traffic.
     */
    public final Integer initialSamplingPercentage() {
        return initialSamplingPercentage;
    }

    /**
     * <p>
     * The Amazon S3 location used to capture the data.
     * </p>
     * 
     * @return The Amazon S3 location used to capture the data.
     */
    public final String destinationS3Uri() {
        return destinationS3Uri;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of a Amazon Web Services Key Management Service key that SageMaker uses to encrypt
     * data on the storage volume attached to the ML compute instance that hosts the endpoint.
     * </p>
     * <p>
     * The KmsKeyId can be any of the following formats:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN: <code>arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name: <code>alias/ExampleAlias</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name ARN: <code>arn:aws:kms:us-west-2:111122223333:alias/ExampleAlias</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The Amazon Resource Name (ARN) of a Amazon Web Services Key Management Service key that SageMaker uses to
     *         encrypt data on the storage volume attached to the ML compute instance that hosts the endpoint.</p>
     *         <p>
     *         The KmsKeyId can be any of the following formats:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key ARN: <code>arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias name: <code>alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias name ARN: <code>arn:aws:kms:us-west-2:111122223333:alias/ExampleAlias</code>
     *         </p>
     *         </li>
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * For responses, this returns true if the service returned a value for the CaptureOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCaptureOptions() {
        return captureOptions != null && !(captureOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies data Model Monitor will capture. You can configure whether to collect only input, only output, or both
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCaptureOptions} method.
     * </p>
     * 
     * @return Specifies data Model Monitor will capture. You can configure whether to collect only input, only output,
     *         or both
     */
    public final List<CaptureOption> captureOptions() {
        return captureOptions;
    }

    /**
     * <p>
     * Configuration specifying how to treat different headers. If no headers are specified SageMaker will by default
     * base64 encode when capturing the data.
     * </p>
     * 
     * @return Configuration specifying how to treat different headers. If no headers are specified SageMaker will by
     *         default base64 encode when capturing the data.
     */
    public final CaptureContentTypeHeader captureContentTypeHeader() {
        return captureContentTypeHeader;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(enableCapture());
        hashCode = 31 * hashCode + Objects.hashCode(initialSamplingPercentage());
        hashCode = 31 * hashCode + Objects.hashCode(destinationS3Uri());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(hasCaptureOptions() ? captureOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(captureContentTypeHeader());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataCaptureConfig)) {
            return false;
        }
        DataCaptureConfig other = (DataCaptureConfig) obj;
        return Objects.equals(enableCapture(), other.enableCapture())
                && Objects.equals(initialSamplingPercentage(), other.initialSamplingPercentage())
                && Objects.equals(destinationS3Uri(), other.destinationS3Uri()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && hasCaptureOptions() == other.hasCaptureOptions() && Objects.equals(captureOptions(), other.captureOptions())
                && Objects.equals(captureContentTypeHeader(), other.captureContentTypeHeader());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataCaptureConfig").add("EnableCapture", enableCapture())
                .add("InitialSamplingPercentage", initialSamplingPercentage()).add("DestinationS3Uri", destinationS3Uri())
                .add("KmsKeyId", kmsKeyId()).add("CaptureOptions", hasCaptureOptions() ? captureOptions() : null)
                .add("CaptureContentTypeHeader", captureContentTypeHeader()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EnableCapture":
            return Optional.ofNullable(clazz.cast(enableCapture()));
        case "InitialSamplingPercentage":
            return Optional.ofNullable(clazz.cast(initialSamplingPercentage()));
        case "DestinationS3Uri":
            return Optional.ofNullable(clazz.cast(destinationS3Uri()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "CaptureOptions":
            return Optional.ofNullable(clazz.cast(captureOptions()));
        case "CaptureContentTypeHeader":
            return Optional.ofNullable(clazz.cast(captureContentTypeHeader()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DataCaptureConfig, T> g) {
        return obj -> g.apply((DataCaptureConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataCaptureConfig> {
        /**
         * <p>
         * Whether data capture should be enabled or disabled (defaults to enabled).
         * </p>
         * 
         * @param enableCapture
         *        Whether data capture should be enabled or disabled (defaults to enabled).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableCapture(Boolean enableCapture);

        /**
         * <p>
         * The percentage of requests SageMaker will capture. A lower value is recommended for Endpoints with high
         * traffic.
         * </p>
         * 
         * @param initialSamplingPercentage
         *        The percentage of requests SageMaker will capture. A lower value is recommended for Endpoints with
         *        high traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initialSamplingPercentage(Integer initialSamplingPercentage);

        /**
         * <p>
         * The Amazon S3 location used to capture the data.
         * </p>
         * 
         * @param destinationS3Uri
         *        The Amazon S3 location used to capture the data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationS3Uri(String destinationS3Uri);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of a Amazon Web Services Key Management Service key that SageMaker uses to
         * encrypt data on the storage volume attached to the ML compute instance that hosts the endpoint.
         * </p>
         * <p>
         * The KmsKeyId can be any of the following formats:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Key ARN: <code>arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias name: <code>alias/ExampleAlias</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias name ARN: <code>arn:aws:kms:us-west-2:111122223333:alias/ExampleAlias</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param kmsKeyId
         *        The Amazon Resource Name (ARN) of a Amazon Web Services Key Management Service key that SageMaker uses
         *        to encrypt data on the storage volume attached to the ML compute instance that hosts the endpoint.</p>
         *        <p>
         *        The KmsKeyId can be any of the following formats:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key ARN: <code>arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias name: <code>alias/ExampleAlias</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias name ARN: <code>arn:aws:kms:us-west-2:111122223333:alias/ExampleAlias</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * Specifies data Model Monitor will capture. You can configure whether to collect only input, only output, or
         * both
         * </p>
         * 
         * @param captureOptions
         *        Specifies data Model Monitor will capture. You can configure whether to collect only input, only
         *        output, or both
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder captureOptions(Collection<CaptureOption> captureOptions);

        /**
         * <p>
         * Specifies data Model Monitor will capture. You can configure whether to collect only input, only output, or
         * both
         * </p>
         * 
         * @param captureOptions
         *        Specifies data Model Monitor will capture. You can configure whether to collect only input, only
         *        output, or both
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder captureOptions(CaptureOption... captureOptions);

        /**
         * <p>
         * Specifies data Model Monitor will capture. You can configure whether to collect only input, only output, or
         * both
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.CaptureOption.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.sagemaker.model.CaptureOption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.CaptureOption.Builder#build()} is called immediately
         * and its result is passed to {@link #captureOptions(List<CaptureOption>)}.
         * 
         * @param captureOptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.CaptureOption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #captureOptions(java.util.Collection<CaptureOption>)
         */
        Builder captureOptions(Consumer<CaptureOption.Builder>... captureOptions);

        /**
         * <p>
         * Configuration specifying how to treat different headers. If no headers are specified SageMaker will by
         * default base64 encode when capturing the data.
         * </p>
         * 
         * @param captureContentTypeHeader
         *        Configuration specifying how to treat different headers. If no headers are specified SageMaker will by
         *        default base64 encode when capturing the data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder captureContentTypeHeader(CaptureContentTypeHeader captureContentTypeHeader);

        /**
         * <p>
         * Configuration specifying how to treat different headers. If no headers are specified SageMaker will by
         * default base64 encode when capturing the data.
         * </p>
         * This is a convenience method that creates an instance of the {@link CaptureContentTypeHeader.Builder}
         * avoiding the need to create one manually via {@link CaptureContentTypeHeader#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CaptureContentTypeHeader.Builder#build()} is called immediately
         * and its result is passed to {@link #captureContentTypeHeader(CaptureContentTypeHeader)}.
         * 
         * @param captureContentTypeHeader
         *        a consumer that will call methods on {@link CaptureContentTypeHeader.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #captureContentTypeHeader(CaptureContentTypeHeader)
         */
        default Builder captureContentTypeHeader(Consumer<CaptureContentTypeHeader.Builder> captureContentTypeHeader) {
            return captureContentTypeHeader(CaptureContentTypeHeader.builder().applyMutation(captureContentTypeHeader).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Boolean enableCapture;

        private Integer initialSamplingPercentage;

        private String destinationS3Uri;

        private String kmsKeyId;

        private List<CaptureOption> captureOptions = DefaultSdkAutoConstructList.getInstance();

        private CaptureContentTypeHeader captureContentTypeHeader;

        private BuilderImpl() {
        }

        private BuilderImpl(DataCaptureConfig model) {
            enableCapture(model.enableCapture);
            initialSamplingPercentage(model.initialSamplingPercentage);
            destinationS3Uri(model.destinationS3Uri);
            kmsKeyId(model.kmsKeyId);
            captureOptions(model.captureOptions);
            captureContentTypeHeader(model.captureContentTypeHeader);
        }

        public final Boolean getEnableCapture() {
            return enableCapture;
        }

        public final void setEnableCapture(Boolean enableCapture) {
            this.enableCapture = enableCapture;
        }

        @Override
        public final Builder enableCapture(Boolean enableCapture) {
            this.enableCapture = enableCapture;
            return this;
        }

        public final Integer getInitialSamplingPercentage() {
            return initialSamplingPercentage;
        }

        public final void setInitialSamplingPercentage(Integer initialSamplingPercentage) {
            this.initialSamplingPercentage = initialSamplingPercentage;
        }

        @Override
        public final Builder initialSamplingPercentage(Integer initialSamplingPercentage) {
            this.initialSamplingPercentage = initialSamplingPercentage;
            return this;
        }

        public final String getDestinationS3Uri() {
            return destinationS3Uri;
        }

        public final void setDestinationS3Uri(String destinationS3Uri) {
            this.destinationS3Uri = destinationS3Uri;
        }

        @Override
        public final Builder destinationS3Uri(String destinationS3Uri) {
            this.destinationS3Uri = destinationS3Uri;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final List<CaptureOption.Builder> getCaptureOptions() {
            List<CaptureOption.Builder> result = CaptureOptionListCopier.copyToBuilder(this.captureOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCaptureOptions(Collection<CaptureOption.BuilderImpl> captureOptions) {
            this.captureOptions = CaptureOptionListCopier.copyFromBuilder(captureOptions);
        }

        @Override
        public final Builder captureOptions(Collection<CaptureOption> captureOptions) {
            this.captureOptions = CaptureOptionListCopier.copy(captureOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder captureOptions(CaptureOption... captureOptions) {
            captureOptions(Arrays.asList(captureOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder captureOptions(Consumer<CaptureOption.Builder>... captureOptions) {
            captureOptions(Stream.of(captureOptions).map(c -> CaptureOption.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final CaptureContentTypeHeader.Builder getCaptureContentTypeHeader() {
            return captureContentTypeHeader != null ? captureContentTypeHeader.toBuilder() : null;
        }

        public final void setCaptureContentTypeHeader(CaptureContentTypeHeader.BuilderImpl captureContentTypeHeader) {
            this.captureContentTypeHeader = captureContentTypeHeader != null ? captureContentTypeHeader.build() : null;
        }

        @Override
        public final Builder captureContentTypeHeader(CaptureContentTypeHeader captureContentTypeHeader) {
            this.captureContentTypeHeader = captureContentTypeHeader;
            return this;
        }

        @Override
        public DataCaptureConfig build() {
            return new DataCaptureConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
