/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configures a hyperparameter tuning job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HyperParameterTuningJobConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<HyperParameterTuningJobConfig.Builder, HyperParameterTuningJobConfig> {
    private static final SdkField<String> STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Strategy").getter(getter(HyperParameterTuningJobConfig::strategyAsString))
            .setter(setter(Builder::strategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Strategy").build()).build();

    private static final SdkField<HyperParameterTuningJobStrategyConfig> STRATEGY_CONFIG_FIELD = SdkField
            .<HyperParameterTuningJobStrategyConfig> builder(MarshallingType.SDK_POJO).memberName("StrategyConfig")
            .getter(getter(HyperParameterTuningJobConfig::strategyConfig)).setter(setter(Builder::strategyConfig))
            .constructor(HyperParameterTuningJobStrategyConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StrategyConfig").build()).build();

    private static final SdkField<HyperParameterTuningJobObjective> HYPER_PARAMETER_TUNING_JOB_OBJECTIVE_FIELD = SdkField
            .<HyperParameterTuningJobObjective> builder(MarshallingType.SDK_POJO)
            .memberName("HyperParameterTuningJobObjective")
            .getter(getter(HyperParameterTuningJobConfig::hyperParameterTuningJobObjective))
            .setter(setter(Builder::hyperParameterTuningJobObjective))
            .constructor(HyperParameterTuningJobObjective::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HyperParameterTuningJobObjective")
                    .build()).build();

    private static final SdkField<ResourceLimits> RESOURCE_LIMITS_FIELD = SdkField
            .<ResourceLimits> builder(MarshallingType.SDK_POJO).memberName("ResourceLimits")
            .getter(getter(HyperParameterTuningJobConfig::resourceLimits)).setter(setter(Builder::resourceLimits))
            .constructor(ResourceLimits::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceLimits").build()).build();

    private static final SdkField<ParameterRanges> PARAMETER_RANGES_FIELD = SdkField
            .<ParameterRanges> builder(MarshallingType.SDK_POJO).memberName("ParameterRanges")
            .getter(getter(HyperParameterTuningJobConfig::parameterRanges)).setter(setter(Builder::parameterRanges))
            .constructor(ParameterRanges::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParameterRanges").build()).build();

    private static final SdkField<String> TRAINING_JOB_EARLY_STOPPING_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TrainingJobEarlyStoppingType")
            .getter(getter(HyperParameterTuningJobConfig::trainingJobEarlyStoppingTypeAsString))
            .setter(setter(Builder::trainingJobEarlyStoppingType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingJobEarlyStoppingType")
                    .build()).build();

    private static final SdkField<TuningJobCompletionCriteria> TUNING_JOB_COMPLETION_CRITERIA_FIELD = SdkField
            .<TuningJobCompletionCriteria> builder(MarshallingType.SDK_POJO)
            .memberName("TuningJobCompletionCriteria")
            .getter(getter(HyperParameterTuningJobConfig::tuningJobCompletionCriteria))
            .setter(setter(Builder::tuningJobCompletionCriteria))
            .constructor(TuningJobCompletionCriteria::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TuningJobCompletionCriteria")
                    .build()).build();

    private static final SdkField<Integer> RANDOM_SEED_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RandomSeed").getter(getter(HyperParameterTuningJobConfig::randomSeed))
            .setter(setter(Builder::randomSeed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RandomSeed").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STRATEGY_FIELD,
            STRATEGY_CONFIG_FIELD, HYPER_PARAMETER_TUNING_JOB_OBJECTIVE_FIELD, RESOURCE_LIMITS_FIELD, PARAMETER_RANGES_FIELD,
            TRAINING_JOB_EARLY_STOPPING_TYPE_FIELD, TUNING_JOB_COMPLETION_CRITERIA_FIELD, RANDOM_SEED_FIELD));

    private static final long serialVersionUID = 1L;

    private final String strategy;

    private final HyperParameterTuningJobStrategyConfig strategyConfig;

    private final HyperParameterTuningJobObjective hyperParameterTuningJobObjective;

    private final ResourceLimits resourceLimits;

    private final ParameterRanges parameterRanges;

    private final String trainingJobEarlyStoppingType;

    private final TuningJobCompletionCriteria tuningJobCompletionCriteria;

    private final Integer randomSeed;

    private HyperParameterTuningJobConfig(BuilderImpl builder) {
        this.strategy = builder.strategy;
        this.strategyConfig = builder.strategyConfig;
        this.hyperParameterTuningJobObjective = builder.hyperParameterTuningJobObjective;
        this.resourceLimits = builder.resourceLimits;
        this.parameterRanges = builder.parameterRanges;
        this.trainingJobEarlyStoppingType = builder.trainingJobEarlyStoppingType;
        this.tuningJobCompletionCriteria = builder.tuningJobCompletionCriteria;
        this.randomSeed = builder.randomSeed;
    }

    /**
     * <p>
     * Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the training job
     * it launches. For information about search strategies, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
     * Hyperparameter Tuning Works</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #strategy} will
     * return {@link HyperParameterTuningJobStrategyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #strategyAsString}.
     * </p>
     * 
     * @return Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the
     *         training job it launches. For information about search strategies, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
     *         Hyperparameter Tuning Works</a>.
     * @see HyperParameterTuningJobStrategyType
     */
    public final HyperParameterTuningJobStrategyType strategy() {
        return HyperParameterTuningJobStrategyType.fromValue(strategy);
    }

    /**
     * <p>
     * Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the training job
     * it launches. For information about search strategies, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
     * Hyperparameter Tuning Works</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #strategy} will
     * return {@link HyperParameterTuningJobStrategyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #strategyAsString}.
     * </p>
     * 
     * @return Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the
     *         training job it launches. For information about search strategies, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
     *         Hyperparameter Tuning Works</a>.
     * @see HyperParameterTuningJobStrategyType
     */
    public final String strategyAsString() {
        return strategy;
    }

    /**
     * <p>
     * The configuration for the <code>Hyperband</code> optimization strategy. This parameter should be provided only if
     * <code>Hyperband</code> is selected as the strategy for <code>HyperParameterTuningJobConfig</code>.
     * </p>
     * 
     * @return The configuration for the <code>Hyperband</code> optimization strategy. This parameter should be provided
     *         only if <code>Hyperband</code> is selected as the strategy for <code>HyperParameterTuningJobConfig</code>
     *         .
     */
    public final HyperParameterTuningJobStrategyConfig strategyConfig() {
        return strategyConfig;
    }

    /**
     * <p>
     * The <a>HyperParameterTuningJobObjective</a> specifies the objective metric used to evaluate the performance of
     * training jobs launched by this tuning job.
     * </p>
     * 
     * @return The <a>HyperParameterTuningJobObjective</a> specifies the objective metric used to evaluate the
     *         performance of training jobs launched by this tuning job.
     */
    public final HyperParameterTuningJobObjective hyperParameterTuningJobObjective() {
        return hyperParameterTuningJobObjective;
    }

    /**
     * <p>
     * The <a>ResourceLimits</a> object that specifies the maximum number of training and parallel training jobs that
     * can be used for this hyperparameter tuning job.
     * </p>
     * 
     * @return The <a>ResourceLimits</a> object that specifies the maximum number of training and parallel training jobs
     *         that can be used for this hyperparameter tuning job.
     */
    public final ResourceLimits resourceLimits() {
        return resourceLimits;
    }

    /**
     * <p>
     * The <a>ParameterRanges</a> object that specifies the ranges of hyperparameters that this tuning job searches over
     * to find the optimal configuration for the highest model performance against .your chosen objective metric.
     * </p>
     * 
     * @return The <a>ParameterRanges</a> object that specifies the ranges of hyperparameters that this tuning job
     *         searches over to find the optimal configuration for the highest model performance against .your chosen
     *         objective metric.
     */
    public final ParameterRanges parameterRanges() {
        return parameterRanges;
    }

    /**
     * <p>
     * Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job. Because the
     * <code>Hyperband</code> strategy has its own advanced internal early stopping mechanism,
     * <code>TrainingJobEarlyStoppingType</code> must be <code>OFF</code> to use <code>Hyperband</code>. This parameter
     * can take on one of the following values (the default value is <code>OFF</code>):
     * </p>
     * <dl>
     * <dt>OFF</dt>
     * <dd>
     * <p>
     * Training jobs launched by the hyperparameter tuning job do not use early stopping.
     * </p>
     * </dd>
     * <dt>AUTO</dt>
     * <dd>
     * <p>
     * SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to perform better
     * than previously completed training jobs. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop Training
     * Jobs Early</a>.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #trainingJobEarlyStoppingType} will return {@link TrainingJobEarlyStoppingType#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #trainingJobEarlyStoppingTypeAsString}.
     * </p>
     * 
     * @return Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job.
     *         Because the <code>Hyperband</code> strategy has its own advanced internal early stopping mechanism,
     *         <code>TrainingJobEarlyStoppingType</code> must be <code>OFF</code> to use <code>Hyperband</code>. This
     *         parameter can take on one of the following values (the default value is <code>OFF</code>):</p>
     *         <dl>
     *         <dt>OFF</dt>
     *         <dd>
     *         <p>
     *         Training jobs launched by the hyperparameter tuning job do not use early stopping.
     *         </p>
     *         </dd>
     *         <dt>AUTO</dt>
     *         <dd>
     *         <p>
     *         SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to perform
     *         better than previously completed training jobs. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop
     *         Training Jobs Early</a>.
     *         </p>
     *         </dd>
     * @see TrainingJobEarlyStoppingType
     */
    public final TrainingJobEarlyStoppingType trainingJobEarlyStoppingType() {
        return TrainingJobEarlyStoppingType.fromValue(trainingJobEarlyStoppingType);
    }

    /**
     * <p>
     * Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job. Because the
     * <code>Hyperband</code> strategy has its own advanced internal early stopping mechanism,
     * <code>TrainingJobEarlyStoppingType</code> must be <code>OFF</code> to use <code>Hyperband</code>. This parameter
     * can take on one of the following values (the default value is <code>OFF</code>):
     * </p>
     * <dl>
     * <dt>OFF</dt>
     * <dd>
     * <p>
     * Training jobs launched by the hyperparameter tuning job do not use early stopping.
     * </p>
     * </dd>
     * <dt>AUTO</dt>
     * <dd>
     * <p>
     * SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to perform better
     * than previously completed training jobs. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop Training
     * Jobs Early</a>.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #trainingJobEarlyStoppingType} will return {@link TrainingJobEarlyStoppingType#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #trainingJobEarlyStoppingTypeAsString}.
     * </p>
     * 
     * @return Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job.
     *         Because the <code>Hyperband</code> strategy has its own advanced internal early stopping mechanism,
     *         <code>TrainingJobEarlyStoppingType</code> must be <code>OFF</code> to use <code>Hyperband</code>. This
     *         parameter can take on one of the following values (the default value is <code>OFF</code>):</p>
     *         <dl>
     *         <dt>OFF</dt>
     *         <dd>
     *         <p>
     *         Training jobs launched by the hyperparameter tuning job do not use early stopping.
     *         </p>
     *         </dd>
     *         <dt>AUTO</dt>
     *         <dd>
     *         <p>
     *         SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to perform
     *         better than previously completed training jobs. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop
     *         Training Jobs Early</a>.
     *         </p>
     *         </dd>
     * @see TrainingJobEarlyStoppingType
     */
    public final String trainingJobEarlyStoppingTypeAsString() {
        return trainingJobEarlyStoppingType;
    }

    /**
     * <p>
     * The tuning job's completion criteria.
     * </p>
     * 
     * @return The tuning job's completion criteria.
     */
    public final TuningJobCompletionCriteria tuningJobCompletionCriteria() {
        return tuningJobCompletionCriteria;
    }

    /**
     * <p>
     * A value used to initialize a pseudo-random number generator. Setting a random seed and using the same seed later
     * for the same tuning job will allow hyperparameter optimization to find more a consistent hyperparameter
     * configuration between the two runs.
     * </p>
     * 
     * @return A value used to initialize a pseudo-random number generator. Setting a random seed and using the same
     *         seed later for the same tuning job will allow hyperparameter optimization to find more a consistent
     *         hyperparameter configuration between the two runs.
     */
    public final Integer randomSeed() {
        return randomSeed;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(strategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(strategyConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hyperParameterTuningJobObjective());
        hashCode = 31 * hashCode + Objects.hashCode(resourceLimits());
        hashCode = 31 * hashCode + Objects.hashCode(parameterRanges());
        hashCode = 31 * hashCode + Objects.hashCode(trainingJobEarlyStoppingTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tuningJobCompletionCriteria());
        hashCode = 31 * hashCode + Objects.hashCode(randomSeed());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HyperParameterTuningJobConfig)) {
            return false;
        }
        HyperParameterTuningJobConfig other = (HyperParameterTuningJobConfig) obj;
        return Objects.equals(strategyAsString(), other.strategyAsString())
                && Objects.equals(strategyConfig(), other.strategyConfig())
                && Objects.equals(hyperParameterTuningJobObjective(), other.hyperParameterTuningJobObjective())
                && Objects.equals(resourceLimits(), other.resourceLimits())
                && Objects.equals(parameterRanges(), other.parameterRanges())
                && Objects.equals(trainingJobEarlyStoppingTypeAsString(), other.trainingJobEarlyStoppingTypeAsString())
                && Objects.equals(tuningJobCompletionCriteria(), other.tuningJobCompletionCriteria())
                && Objects.equals(randomSeed(), other.randomSeed());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HyperParameterTuningJobConfig").add("Strategy", strategyAsString())
                .add("StrategyConfig", strategyConfig())
                .add("HyperParameterTuningJobObjective", hyperParameterTuningJobObjective())
                .add("ResourceLimits", resourceLimits()).add("ParameterRanges", parameterRanges())
                .add("TrainingJobEarlyStoppingType", trainingJobEarlyStoppingTypeAsString())
                .add("TuningJobCompletionCriteria", tuningJobCompletionCriteria()).add("RandomSeed", randomSeed()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Strategy":
            return Optional.ofNullable(clazz.cast(strategyAsString()));
        case "StrategyConfig":
            return Optional.ofNullable(clazz.cast(strategyConfig()));
        case "HyperParameterTuningJobObjective":
            return Optional.ofNullable(clazz.cast(hyperParameterTuningJobObjective()));
        case "ResourceLimits":
            return Optional.ofNullable(clazz.cast(resourceLimits()));
        case "ParameterRanges":
            return Optional.ofNullable(clazz.cast(parameterRanges()));
        case "TrainingJobEarlyStoppingType":
            return Optional.ofNullable(clazz.cast(trainingJobEarlyStoppingTypeAsString()));
        case "TuningJobCompletionCriteria":
            return Optional.ofNullable(clazz.cast(tuningJobCompletionCriteria()));
        case "RandomSeed":
            return Optional.ofNullable(clazz.cast(randomSeed()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<HyperParameterTuningJobConfig, T> g) {
        return obj -> g.apply((HyperParameterTuningJobConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HyperParameterTuningJobConfig> {
        /**
         * <p>
         * Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the training
         * job it launches. For information about search strategies, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
         * Hyperparameter Tuning Works</a>.
         * </p>
         * 
         * @param strategy
         *        Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the
         *        training job it launches. For information about search strategies, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
         *        Hyperparameter Tuning Works</a>.
         * @see HyperParameterTuningJobStrategyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HyperParameterTuningJobStrategyType
         */
        Builder strategy(String strategy);

        /**
         * <p>
         * Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the training
         * job it launches. For information about search strategies, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
         * Hyperparameter Tuning Works</a>.
         * </p>
         * 
         * @param strategy
         *        Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the
         *        training job it launches. For information about search strategies, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
         *        Hyperparameter Tuning Works</a>.
         * @see HyperParameterTuningJobStrategyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HyperParameterTuningJobStrategyType
         */
        Builder strategy(HyperParameterTuningJobStrategyType strategy);

        /**
         * <p>
         * The configuration for the <code>Hyperband</code> optimization strategy. This parameter should be provided
         * only if <code>Hyperband</code> is selected as the strategy for <code>HyperParameterTuningJobConfig</code>.
         * </p>
         * 
         * @param strategyConfig
         *        The configuration for the <code>Hyperband</code> optimization strategy. This parameter should be
         *        provided only if <code>Hyperband</code> is selected as the strategy for
         *        <code>HyperParameterTuningJobConfig</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder strategyConfig(HyperParameterTuningJobStrategyConfig strategyConfig);

        /**
         * <p>
         * The configuration for the <code>Hyperband</code> optimization strategy. This parameter should be provided
         * only if <code>Hyperband</code> is selected as the strategy for <code>HyperParameterTuningJobConfig</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link HyperParameterTuningJobStrategyConfig.Builder} avoiding the need to create one manually via
         * {@link HyperParameterTuningJobStrategyConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link HyperParameterTuningJobStrategyConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #strategyConfig(HyperParameterTuningJobStrategyConfig)}.
         * 
         * @param strategyConfig
         *        a consumer that will call methods on {@link HyperParameterTuningJobStrategyConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #strategyConfig(HyperParameterTuningJobStrategyConfig)
         */
        default Builder strategyConfig(Consumer<HyperParameterTuningJobStrategyConfig.Builder> strategyConfig) {
            return strategyConfig(HyperParameterTuningJobStrategyConfig.builder().applyMutation(strategyConfig).build());
        }

        /**
         * <p>
         * The <a>HyperParameterTuningJobObjective</a> specifies the objective metric used to evaluate the performance
         * of training jobs launched by this tuning job.
         * </p>
         * 
         * @param hyperParameterTuningJobObjective
         *        The <a>HyperParameterTuningJobObjective</a> specifies the objective metric used to evaluate the
         *        performance of training jobs launched by this tuning job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hyperParameterTuningJobObjective(HyperParameterTuningJobObjective hyperParameterTuningJobObjective);

        /**
         * <p>
         * The <a>HyperParameterTuningJobObjective</a> specifies the objective metric used to evaluate the performance
         * of training jobs launched by this tuning job.
         * </p>
         * This is a convenience method that creates an instance of the {@link HyperParameterTuningJobObjective.Builder}
         * avoiding the need to create one manually via {@link HyperParameterTuningJobObjective#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link HyperParameterTuningJobObjective.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #hyperParameterTuningJobObjective(HyperParameterTuningJobObjective)}.
         * 
         * @param hyperParameterTuningJobObjective
         *        a consumer that will call methods on {@link HyperParameterTuningJobObjective.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hyperParameterTuningJobObjective(HyperParameterTuningJobObjective)
         */
        default Builder hyperParameterTuningJobObjective(
                Consumer<HyperParameterTuningJobObjective.Builder> hyperParameterTuningJobObjective) {
            return hyperParameterTuningJobObjective(HyperParameterTuningJobObjective.builder()
                    .applyMutation(hyperParameterTuningJobObjective).build());
        }

        /**
         * <p>
         * The <a>ResourceLimits</a> object that specifies the maximum number of training and parallel training jobs
         * that can be used for this hyperparameter tuning job.
         * </p>
         * 
         * @param resourceLimits
         *        The <a>ResourceLimits</a> object that specifies the maximum number of training and parallel training
         *        jobs that can be used for this hyperparameter tuning job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceLimits(ResourceLimits resourceLimits);

        /**
         * <p>
         * The <a>ResourceLimits</a> object that specifies the maximum number of training and parallel training jobs
         * that can be used for this hyperparameter tuning job.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResourceLimits.Builder} avoiding the need
         * to create one manually via {@link ResourceLimits#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceLimits.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceLimits(ResourceLimits)}.
         * 
         * @param resourceLimits
         *        a consumer that will call methods on {@link ResourceLimits.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceLimits(ResourceLimits)
         */
        default Builder resourceLimits(Consumer<ResourceLimits.Builder> resourceLimits) {
            return resourceLimits(ResourceLimits.builder().applyMutation(resourceLimits).build());
        }

        /**
         * <p>
         * The <a>ParameterRanges</a> object that specifies the ranges of hyperparameters that this tuning job searches
         * over to find the optimal configuration for the highest model performance against .your chosen objective
         * metric.
         * </p>
         * 
         * @param parameterRanges
         *        The <a>ParameterRanges</a> object that specifies the ranges of hyperparameters that this tuning job
         *        searches over to find the optimal configuration for the highest model performance against .your chosen
         *        objective metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterRanges(ParameterRanges parameterRanges);

        /**
         * <p>
         * The <a>ParameterRanges</a> object that specifies the ranges of hyperparameters that this tuning job searches
         * over to find the optimal configuration for the highest model performance against .your chosen objective
         * metric.
         * </p>
         * This is a convenience method that creates an instance of the {@link ParameterRanges.Builder} avoiding the
         * need to create one manually via {@link ParameterRanges#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ParameterRanges.Builder#build()} is called immediately and its
         * result is passed to {@link #parameterRanges(ParameterRanges)}.
         * 
         * @param parameterRanges
         *        a consumer that will call methods on {@link ParameterRanges.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameterRanges(ParameterRanges)
         */
        default Builder parameterRanges(Consumer<ParameterRanges.Builder> parameterRanges) {
            return parameterRanges(ParameterRanges.builder().applyMutation(parameterRanges).build());
        }

        /**
         * <p>
         * Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job. Because
         * the <code>Hyperband</code> strategy has its own advanced internal early stopping mechanism,
         * <code>TrainingJobEarlyStoppingType</code> must be <code>OFF</code> to use <code>Hyperband</code>. This
         * parameter can take on one of the following values (the default value is <code>OFF</code>):
         * </p>
         * <dl>
         * <dt>OFF</dt>
         * <dd>
         * <p>
         * Training jobs launched by the hyperparameter tuning job do not use early stopping.
         * </p>
         * </dd>
         * <dt>AUTO</dt>
         * <dd>
         * <p>
         * SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to perform
         * better than previously completed training jobs. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop
         * Training Jobs Early</a>.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param trainingJobEarlyStoppingType
         *        Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job.
         *        Because the <code>Hyperband</code> strategy has its own advanced internal early stopping mechanism,
         *        <code>TrainingJobEarlyStoppingType</code> must be <code>OFF</code> to use <code>Hyperband</code>. This
         *        parameter can take on one of the following values (the default value is <code>OFF</code>):</p>
         *        <dl>
         *        <dt>OFF</dt>
         *        <dd>
         *        <p>
         *        Training jobs launched by the hyperparameter tuning job do not use early stopping.
         *        </p>
         *        </dd>
         *        <dt>AUTO</dt>
         *        <dd>
         *        <p>
         *        SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to
         *        perform better than previously completed training jobs. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop
         *        Training Jobs Early</a>.
         *        </p>
         *        </dd>
         * @see TrainingJobEarlyStoppingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingJobEarlyStoppingType
         */
        Builder trainingJobEarlyStoppingType(String trainingJobEarlyStoppingType);

        /**
         * <p>
         * Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job. Because
         * the <code>Hyperband</code> strategy has its own advanced internal early stopping mechanism,
         * <code>TrainingJobEarlyStoppingType</code> must be <code>OFF</code> to use <code>Hyperband</code>. This
         * parameter can take on one of the following values (the default value is <code>OFF</code>):
         * </p>
         * <dl>
         * <dt>OFF</dt>
         * <dd>
         * <p>
         * Training jobs launched by the hyperparameter tuning job do not use early stopping.
         * </p>
         * </dd>
         * <dt>AUTO</dt>
         * <dd>
         * <p>
         * SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to perform
         * better than previously completed training jobs. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop
         * Training Jobs Early</a>.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param trainingJobEarlyStoppingType
         *        Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job.
         *        Because the <code>Hyperband</code> strategy has its own advanced internal early stopping mechanism,
         *        <code>TrainingJobEarlyStoppingType</code> must be <code>OFF</code> to use <code>Hyperband</code>. This
         *        parameter can take on one of the following values (the default value is <code>OFF</code>):</p>
         *        <dl>
         *        <dt>OFF</dt>
         *        <dd>
         *        <p>
         *        Training jobs launched by the hyperparameter tuning job do not use early stopping.
         *        </p>
         *        </dd>
         *        <dt>AUTO</dt>
         *        <dd>
         *        <p>
         *        SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to
         *        perform better than previously completed training jobs. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop
         *        Training Jobs Early</a>.
         *        </p>
         *        </dd>
         * @see TrainingJobEarlyStoppingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingJobEarlyStoppingType
         */
        Builder trainingJobEarlyStoppingType(TrainingJobEarlyStoppingType trainingJobEarlyStoppingType);

        /**
         * <p>
         * The tuning job's completion criteria.
         * </p>
         * 
         * @param tuningJobCompletionCriteria
         *        The tuning job's completion criteria.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tuningJobCompletionCriteria(TuningJobCompletionCriteria tuningJobCompletionCriteria);

        /**
         * <p>
         * The tuning job's completion criteria.
         * </p>
         * This is a convenience method that creates an instance of the {@link TuningJobCompletionCriteria.Builder}
         * avoiding the need to create one manually via {@link TuningJobCompletionCriteria#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TuningJobCompletionCriteria.Builder#build()} is called
         * immediately and its result is passed to {@link #tuningJobCompletionCriteria(TuningJobCompletionCriteria)}.
         * 
         * @param tuningJobCompletionCriteria
         *        a consumer that will call methods on {@link TuningJobCompletionCriteria.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tuningJobCompletionCriteria(TuningJobCompletionCriteria)
         */
        default Builder tuningJobCompletionCriteria(Consumer<TuningJobCompletionCriteria.Builder> tuningJobCompletionCriteria) {
            return tuningJobCompletionCriteria(TuningJobCompletionCriteria.builder().applyMutation(tuningJobCompletionCriteria)
                    .build());
        }

        /**
         * <p>
         * A value used to initialize a pseudo-random number generator. Setting a random seed and using the same seed
         * later for the same tuning job will allow hyperparameter optimization to find more a consistent hyperparameter
         * configuration between the two runs.
         * </p>
         * 
         * @param randomSeed
         *        A value used to initialize a pseudo-random number generator. Setting a random seed and using the same
         *        seed later for the same tuning job will allow hyperparameter optimization to find more a consistent
         *        hyperparameter configuration between the two runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder randomSeed(Integer randomSeed);
    }

    static final class BuilderImpl implements Builder {
        private String strategy;

        private HyperParameterTuningJobStrategyConfig strategyConfig;

        private HyperParameterTuningJobObjective hyperParameterTuningJobObjective;

        private ResourceLimits resourceLimits;

        private ParameterRanges parameterRanges;

        private String trainingJobEarlyStoppingType;

        private TuningJobCompletionCriteria tuningJobCompletionCriteria;

        private Integer randomSeed;

        private BuilderImpl() {
        }

        private BuilderImpl(HyperParameterTuningJobConfig model) {
            strategy(model.strategy);
            strategyConfig(model.strategyConfig);
            hyperParameterTuningJobObjective(model.hyperParameterTuningJobObjective);
            resourceLimits(model.resourceLimits);
            parameterRanges(model.parameterRanges);
            trainingJobEarlyStoppingType(model.trainingJobEarlyStoppingType);
            tuningJobCompletionCriteria(model.tuningJobCompletionCriteria);
            randomSeed(model.randomSeed);
        }

        public final String getStrategy() {
            return strategy;
        }

        public final void setStrategy(String strategy) {
            this.strategy = strategy;
        }

        @Override
        public final Builder strategy(String strategy) {
            this.strategy = strategy;
            return this;
        }

        @Override
        public final Builder strategy(HyperParameterTuningJobStrategyType strategy) {
            this.strategy(strategy == null ? null : strategy.toString());
            return this;
        }

        public final HyperParameterTuningJobStrategyConfig.Builder getStrategyConfig() {
            return strategyConfig != null ? strategyConfig.toBuilder() : null;
        }

        public final void setStrategyConfig(HyperParameterTuningJobStrategyConfig.BuilderImpl strategyConfig) {
            this.strategyConfig = strategyConfig != null ? strategyConfig.build() : null;
        }

        @Override
        public final Builder strategyConfig(HyperParameterTuningJobStrategyConfig strategyConfig) {
            this.strategyConfig = strategyConfig;
            return this;
        }

        public final HyperParameterTuningJobObjective.Builder getHyperParameterTuningJobObjective() {
            return hyperParameterTuningJobObjective != null ? hyperParameterTuningJobObjective.toBuilder() : null;
        }

        public final void setHyperParameterTuningJobObjective(
                HyperParameterTuningJobObjective.BuilderImpl hyperParameterTuningJobObjective) {
            this.hyperParameterTuningJobObjective = hyperParameterTuningJobObjective != null ? hyperParameterTuningJobObjective
                    .build() : null;
        }

        @Override
        public final Builder hyperParameterTuningJobObjective(HyperParameterTuningJobObjective hyperParameterTuningJobObjective) {
            this.hyperParameterTuningJobObjective = hyperParameterTuningJobObjective;
            return this;
        }

        public final ResourceLimits.Builder getResourceLimits() {
            return resourceLimits != null ? resourceLimits.toBuilder() : null;
        }

        public final void setResourceLimits(ResourceLimits.BuilderImpl resourceLimits) {
            this.resourceLimits = resourceLimits != null ? resourceLimits.build() : null;
        }

        @Override
        public final Builder resourceLimits(ResourceLimits resourceLimits) {
            this.resourceLimits = resourceLimits;
            return this;
        }

        public final ParameterRanges.Builder getParameterRanges() {
            return parameterRanges != null ? parameterRanges.toBuilder() : null;
        }

        public final void setParameterRanges(ParameterRanges.BuilderImpl parameterRanges) {
            this.parameterRanges = parameterRanges != null ? parameterRanges.build() : null;
        }

        @Override
        public final Builder parameterRanges(ParameterRanges parameterRanges) {
            this.parameterRanges = parameterRanges;
            return this;
        }

        public final String getTrainingJobEarlyStoppingType() {
            return trainingJobEarlyStoppingType;
        }

        public final void setTrainingJobEarlyStoppingType(String trainingJobEarlyStoppingType) {
            this.trainingJobEarlyStoppingType = trainingJobEarlyStoppingType;
        }

        @Override
        public final Builder trainingJobEarlyStoppingType(String trainingJobEarlyStoppingType) {
            this.trainingJobEarlyStoppingType = trainingJobEarlyStoppingType;
            return this;
        }

        @Override
        public final Builder trainingJobEarlyStoppingType(TrainingJobEarlyStoppingType trainingJobEarlyStoppingType) {
            this.trainingJobEarlyStoppingType(trainingJobEarlyStoppingType == null ? null : trainingJobEarlyStoppingType
                    .toString());
            return this;
        }

        public final TuningJobCompletionCriteria.Builder getTuningJobCompletionCriteria() {
            return tuningJobCompletionCriteria != null ? tuningJobCompletionCriteria.toBuilder() : null;
        }

        public final void setTuningJobCompletionCriteria(TuningJobCompletionCriteria.BuilderImpl tuningJobCompletionCriteria) {
            this.tuningJobCompletionCriteria = tuningJobCompletionCriteria != null ? tuningJobCompletionCriteria.build() : null;
        }

        @Override
        public final Builder tuningJobCompletionCriteria(TuningJobCompletionCriteria tuningJobCompletionCriteria) {
            this.tuningJobCompletionCriteria = tuningJobCompletionCriteria;
            return this;
        }

        public final Integer getRandomSeed() {
            return randomSeed;
        }

        public final void setRandomSeed(Integer randomSeed) {
            this.randomSeed = randomSeed;
        }

        @Override
        public final Builder randomSeed(Integer randomSeed) {
            this.randomSeed = randomSeed;
            return this;
        }

        @Override
        public HyperParameterTuningJobConfig build() {
            return new HyperParameterTuningJobConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
