/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the training algorithm to use in a <a>CreateTrainingJob</a> request.
 * </p>
 * <p>
 * For more information about algorithms provided by SageMaker, see <a
 * href="https://docs.aws.amazon.com/sagemaker/latest/dg/algos.html">Algorithms</a>. For information about using your
 * own algorithms, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own
 * Algorithms with Amazon SageMaker</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AlgorithmSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<AlgorithmSpecification.Builder, AlgorithmSpecification> {
    private static final SdkField<String> TRAINING_IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TrainingImage").getter(getter(AlgorithmSpecification::trainingImage))
            .setter(setter(Builder::trainingImage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingImage").build()).build();

    private static final SdkField<String> ALGORITHM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlgorithmName").getter(getter(AlgorithmSpecification::algorithmName))
            .setter(setter(Builder::algorithmName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlgorithmName").build()).build();

    private static final SdkField<String> TRAINING_INPUT_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TrainingInputMode").getter(getter(AlgorithmSpecification::trainingInputModeAsString))
            .setter(setter(Builder::trainingInputMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingInputMode").build()).build();

    private static final SdkField<List<MetricDefinition>> METRIC_DEFINITIONS_FIELD = SdkField
            .<List<MetricDefinition>> builder(MarshallingType.LIST)
            .memberName("MetricDefinitions")
            .getter(getter(AlgorithmSpecification::metricDefinitions))
            .setter(setter(Builder::metricDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricDefinitions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ENABLE_SAGE_MAKER_METRICS_TIME_SERIES_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableSageMakerMetricsTimeSeries")
            .getter(getter(AlgorithmSpecification::enableSageMakerMetricsTimeSeries))
            .setter(setter(Builder::enableSageMakerMetricsTimeSeries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableSageMakerMetricsTimeSeries")
                    .build()).build();

    private static final SdkField<List<String>> CONTAINER_ENTRYPOINT_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ContainerEntrypoint")
            .getter(getter(AlgorithmSpecification::containerEntrypoint))
            .setter(setter(Builder::containerEntrypoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerEntrypoint").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> CONTAINER_ARGUMENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ContainerArguments")
            .getter(getter(AlgorithmSpecification::containerArguments))
            .setter(setter(Builder::containerArguments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerArguments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRAINING_IMAGE_FIELD,
            ALGORITHM_NAME_FIELD, TRAINING_INPUT_MODE_FIELD, METRIC_DEFINITIONS_FIELD,
            ENABLE_SAGE_MAKER_METRICS_TIME_SERIES_FIELD, CONTAINER_ENTRYPOINT_FIELD, CONTAINER_ARGUMENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String trainingImage;

    private final String algorithmName;

    private final String trainingInputMode;

    private final List<MetricDefinition> metricDefinitions;

    private final Boolean enableSageMakerMetricsTimeSeries;

    private final List<String> containerEntrypoint;

    private final List<String> containerArguments;

    private AlgorithmSpecification(BuilderImpl builder) {
        this.trainingImage = builder.trainingImage;
        this.algorithmName = builder.algorithmName;
        this.trainingInputMode = builder.trainingInputMode;
        this.metricDefinitions = builder.metricDefinitions;
        this.enableSageMakerMetricsTimeSeries = builder.enableSageMakerMetricsTimeSeries;
        this.containerEntrypoint = builder.containerEntrypoint;
        this.containerArguments = builder.containerArguments;
    }

    /**
     * <p>
     * The registry path of the Docker image that contains the training algorithm. For information about docker registry
     * paths for SageMaker built-in algorithms, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-algo-docker-registry-paths.html">Docker Registry
     * Paths and Example Code</a> in the <i>Amazon SageMaker developer guide</i>. SageMaker supports both
     * <code>registry/repository[:tag]</code> and <code>registry/repository[@digest]</code> image path formats. For more
     * information about using your custom training container, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own Algorithms with Amazon
     * SageMaker</a>.
     * </p>
     * <note>
     * <p>
     * You must specify either the algorithm name to the <code>AlgorithmName</code> parameter or the image URI of the
     * algorithm container to the <code>TrainingImage</code> parameter.
     * </p>
     * <p>
     * For more information, see the note in the <code>AlgorithmName</code> parameter description.
     * </p>
     * </note>
     * 
     * @return The registry path of the Docker image that contains the training algorithm. For information about docker
     *         registry paths for SageMaker built-in algorithms, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-algo-docker-registry-paths.html">Docker
     *         Registry Paths and Example Code</a> in the <i>Amazon SageMaker developer guide</i>. SageMaker supports
     *         both <code>registry/repository[:tag]</code> and <code>registry/repository[@digest]</code> image path
     *         formats. For more information about using your custom training container, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own Algorithms
     *         with Amazon SageMaker</a>.</p> <note>
     *         <p>
     *         You must specify either the algorithm name to the <code>AlgorithmName</code> parameter or the image URI
     *         of the algorithm container to the <code>TrainingImage</code> parameter.
     *         </p>
     *         <p>
     *         For more information, see the note in the <code>AlgorithmName</code> parameter description.
     *         </p>
     */
    public final String trainingImage() {
        return trainingImage;
    }

    /**
     * <p>
     * The name of the algorithm resource to use for the training job. This must be an algorithm resource that you
     * created or subscribe to on Amazon Web Services Marketplace.
     * </p>
     * <note>
     * <p>
     * You must specify either the algorithm name to the <code>AlgorithmName</code> parameter or the image URI of the
     * algorithm container to the <code>TrainingImage</code> parameter.
     * </p>
     * <p>
     * Note that the <code>AlgorithmName</code> parameter is mutually exclusive with the <code>TrainingImage</code>
     * parameter. If you specify a value for the <code>AlgorithmName</code> parameter, you can't specify a value for
     * <code>TrainingImage</code>, and vice versa.
     * </p>
     * <p>
     * If you specify values for both parameters, the training job might break; if you don't specify any value for both
     * parameters, the training job might raise a <code>null</code> error.
     * </p>
     * </note>
     * 
     * @return The name of the algorithm resource to use for the training job. This must be an algorithm resource that
     *         you created or subscribe to on Amazon Web Services Marketplace.</p> <note>
     *         <p>
     *         You must specify either the algorithm name to the <code>AlgorithmName</code> parameter or the image URI
     *         of the algorithm container to the <code>TrainingImage</code> parameter.
     *         </p>
     *         <p>
     *         Note that the <code>AlgorithmName</code> parameter is mutually exclusive with the
     *         <code>TrainingImage</code> parameter. If you specify a value for the <code>AlgorithmName</code>
     *         parameter, you can't specify a value for <code>TrainingImage</code>, and vice versa.
     *         </p>
     *         <p>
     *         If you specify values for both parameters, the training job might break; if you don't specify any value
     *         for both parameters, the training job might raise a <code>null</code> error.
     *         </p>
     */
    public final String algorithmName() {
        return algorithmName;
    }

    /**
     * Returns the value of the TrainingInputMode property for this object.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trainingInputMode}
     * will return {@link TrainingInputMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #trainingInputModeAsString}.
     * </p>
     * 
     * @return The value of the TrainingInputMode property for this object.
     * @see TrainingInputMode
     */
    public final TrainingInputMode trainingInputMode() {
        return TrainingInputMode.fromValue(trainingInputMode);
    }

    /**
     * Returns the value of the TrainingInputMode property for this object.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trainingInputMode}
     * will return {@link TrainingInputMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #trainingInputModeAsString}.
     * </p>
     * 
     * @return The value of the TrainingInputMode property for this object.
     * @see TrainingInputMode
     */
    public final String trainingInputModeAsString() {
        return trainingInputMode;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricDefinitions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMetricDefinitions() {
        return metricDefinitions != null && !(metricDefinitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of metric definition objects. Each object specifies the metric name and regular expressions used to parse
     * algorithm logs. SageMaker publishes each metric to Amazon CloudWatch.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricDefinitions} method.
     * </p>
     * 
     * @return A list of metric definition objects. Each object specifies the metric name and regular expressions used
     *         to parse algorithm logs. SageMaker publishes each metric to Amazon CloudWatch.
     */
    public final List<MetricDefinition> metricDefinitions() {
        return metricDefinitions;
    }

    /**
     * <p>
     * To generate and save time-series metrics during training, set to <code>true</code>. The default is
     * <code>false</code> and time-series metrics aren't generated except in the following cases:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You use one of the SageMaker built-in algorithms
     * </p>
     * </li>
     * <li>
     * <p>
     * You use one of the following <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/pre-built-containers-frameworks-deep-learning.html"
     * >Prebuilt SageMaker Docker Images</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Tensorflow (version &gt;= 1.15)
     * </p>
     * </li>
     * <li>
     * <p>
     * MXNet (version &gt;= 1.6)
     * </p>
     * </li>
     * <li>
     * <p>
     * PyTorch (version &gt;= 1.3)
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * You specify at least one <a>MetricDefinition</a>
     * </p>
     * </li>
     * </ul>
     * 
     * @return To generate and save time-series metrics during training, set to <code>true</code>. The default is
     *         <code>false</code> and time-series metrics aren't generated except in the following cases:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         You use one of the SageMaker built-in algorithms
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You use one of the following <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/pre-built-containers-frameworks-deep-learning.html"
     *         >Prebuilt SageMaker Docker Images</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Tensorflow (version &gt;= 1.15)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MXNet (version &gt;= 1.6)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PyTorch (version &gt;= 1.3)
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         You specify at least one <a>MetricDefinition</a>
     *         </p>
     *         </li>
     */
    public final Boolean enableSageMakerMetricsTimeSeries() {
        return enableSageMakerMetricsTimeSeries;
    }

    /**
     * For responses, this returns true if the service returned a value for the ContainerEntrypoint property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasContainerEntrypoint() {
        return containerEntrypoint != null && !(containerEntrypoint instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The <a href="https://docs.docker.com/engine/reference/builder/">entrypoint script for a Docker container</a> used
     * to run a training job. This script takes precedence over the default train processing instructions. See <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How Amazon
     * SageMaker Runs Your Training Image</a> for more information.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContainerEntrypoint} method.
     * </p>
     * 
     * @return The <a href="https://docs.docker.com/engine/reference/builder/">entrypoint script for a Docker
     *         container</a> used to run a training job. This script takes precedence over the default train processing
     *         instructions. See <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
     *         Amazon SageMaker Runs Your Training Image</a> for more information.
     */
    public final List<String> containerEntrypoint() {
        return containerEntrypoint;
    }

    /**
     * For responses, this returns true if the service returned a value for the ContainerArguments property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasContainerArguments() {
        return containerArguments != null && !(containerArguments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The arguments for a container used to run a training job. See <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How Amazon
     * SageMaker Runs Your Training Image</a> for additional information.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContainerArguments} method.
     * </p>
     * 
     * @return The arguments for a container used to run a training job. See <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
     *         Amazon SageMaker Runs Your Training Image</a> for additional information.
     */
    public final List<String> containerArguments() {
        return containerArguments;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(trainingImage());
        hashCode = 31 * hashCode + Objects.hashCode(algorithmName());
        hashCode = 31 * hashCode + Objects.hashCode(trainingInputModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricDefinitions() ? metricDefinitions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(enableSageMakerMetricsTimeSeries());
        hashCode = 31 * hashCode + Objects.hashCode(hasContainerEntrypoint() ? containerEntrypoint() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasContainerArguments() ? containerArguments() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AlgorithmSpecification)) {
            return false;
        }
        AlgorithmSpecification other = (AlgorithmSpecification) obj;
        return Objects.equals(trainingImage(), other.trainingImage()) && Objects.equals(algorithmName(), other.algorithmName())
                && Objects.equals(trainingInputModeAsString(), other.trainingInputModeAsString())
                && hasMetricDefinitions() == other.hasMetricDefinitions()
                && Objects.equals(metricDefinitions(), other.metricDefinitions())
                && Objects.equals(enableSageMakerMetricsTimeSeries(), other.enableSageMakerMetricsTimeSeries())
                && hasContainerEntrypoint() == other.hasContainerEntrypoint()
                && Objects.equals(containerEntrypoint(), other.containerEntrypoint())
                && hasContainerArguments() == other.hasContainerArguments()
                && Objects.equals(containerArguments(), other.containerArguments());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AlgorithmSpecification").add("TrainingImage", trainingImage())
                .add("AlgorithmName", algorithmName()).add("TrainingInputMode", trainingInputModeAsString())
                .add("MetricDefinitions", hasMetricDefinitions() ? metricDefinitions() : null)
                .add("EnableSageMakerMetricsTimeSeries", enableSageMakerMetricsTimeSeries())
                .add("ContainerEntrypoint", hasContainerEntrypoint() ? containerEntrypoint() : null)
                .add("ContainerArguments", hasContainerArguments() ? containerArguments() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TrainingImage":
            return Optional.ofNullable(clazz.cast(trainingImage()));
        case "AlgorithmName":
            return Optional.ofNullable(clazz.cast(algorithmName()));
        case "TrainingInputMode":
            return Optional.ofNullable(clazz.cast(trainingInputModeAsString()));
        case "MetricDefinitions":
            return Optional.ofNullable(clazz.cast(metricDefinitions()));
        case "EnableSageMakerMetricsTimeSeries":
            return Optional.ofNullable(clazz.cast(enableSageMakerMetricsTimeSeries()));
        case "ContainerEntrypoint":
            return Optional.ofNullable(clazz.cast(containerEntrypoint()));
        case "ContainerArguments":
            return Optional.ofNullable(clazz.cast(containerArguments()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AlgorithmSpecification, T> g) {
        return obj -> g.apply((AlgorithmSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AlgorithmSpecification> {
        /**
         * <p>
         * The registry path of the Docker image that contains the training algorithm. For information about docker
         * registry paths for SageMaker built-in algorithms, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-algo-docker-registry-paths.html">Docker
         * Registry Paths and Example Code</a> in the <i>Amazon SageMaker developer guide</i>. SageMaker supports both
         * <code>registry/repository[:tag]</code> and <code>registry/repository[@digest]</code> image path formats. For
         * more information about using your custom training container, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own Algorithms with
         * Amazon SageMaker</a>.
         * </p>
         * <note>
         * <p>
         * You must specify either the algorithm name to the <code>AlgorithmName</code> parameter or the image URI of
         * the algorithm container to the <code>TrainingImage</code> parameter.
         * </p>
         * <p>
         * For more information, see the note in the <code>AlgorithmName</code> parameter description.
         * </p>
         * </note>
         * 
         * @param trainingImage
         *        The registry path of the Docker image that contains the training algorithm. For information about
         *        docker registry paths for SageMaker built-in algorithms, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-algo-docker-registry-paths.html"
         *        >Docker Registry Paths and Example Code</a> in the <i>Amazon SageMaker developer guide</i>. SageMaker
         *        supports both <code>registry/repository[:tag]</code> and <code>registry/repository[@digest]</code>
         *        image path formats. For more information about using your custom training container, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own Algorithms
         *        with Amazon SageMaker</a>.</p> <note>
         *        <p>
         *        You must specify either the algorithm name to the <code>AlgorithmName</code> parameter or the image
         *        URI of the algorithm container to the <code>TrainingImage</code> parameter.
         *        </p>
         *        <p>
         *        For more information, see the note in the <code>AlgorithmName</code> parameter description.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingImage(String trainingImage);

        /**
         * <p>
         * The name of the algorithm resource to use for the training job. This must be an algorithm resource that you
         * created or subscribe to on Amazon Web Services Marketplace.
         * </p>
         * <note>
         * <p>
         * You must specify either the algorithm name to the <code>AlgorithmName</code> parameter or the image URI of
         * the algorithm container to the <code>TrainingImage</code> parameter.
         * </p>
         * <p>
         * Note that the <code>AlgorithmName</code> parameter is mutually exclusive with the <code>TrainingImage</code>
         * parameter. If you specify a value for the <code>AlgorithmName</code> parameter, you can't specify a value for
         * <code>TrainingImage</code>, and vice versa.
         * </p>
         * <p>
         * If you specify values for both parameters, the training job might break; if you don't specify any value for
         * both parameters, the training job might raise a <code>null</code> error.
         * </p>
         * </note>
         * 
         * @param algorithmName
         *        The name of the algorithm resource to use for the training job. This must be an algorithm resource
         *        that you created or subscribe to on Amazon Web Services Marketplace.</p> <note>
         *        <p>
         *        You must specify either the algorithm name to the <code>AlgorithmName</code> parameter or the image
         *        URI of the algorithm container to the <code>TrainingImage</code> parameter.
         *        </p>
         *        <p>
         *        Note that the <code>AlgorithmName</code> parameter is mutually exclusive with the
         *        <code>TrainingImage</code> parameter. If you specify a value for the <code>AlgorithmName</code>
         *        parameter, you can't specify a value for <code>TrainingImage</code>, and vice versa.
         *        </p>
         *        <p>
         *        If you specify values for both parameters, the training job might break; if you don't specify any
         *        value for both parameters, the training job might raise a <code>null</code> error.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder algorithmName(String algorithmName);

        /**
         * Sets the value of the TrainingInputMode property for this object.
         *
         * @param trainingInputMode
         *        The new value for the TrainingInputMode property for this object.
         * @see TrainingInputMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInputMode
         */
        Builder trainingInputMode(String trainingInputMode);

        /**
         * Sets the value of the TrainingInputMode property for this object.
         *
         * @param trainingInputMode
         *        The new value for the TrainingInputMode property for this object.
         * @see TrainingInputMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInputMode
         */
        Builder trainingInputMode(TrainingInputMode trainingInputMode);

        /**
         * <p>
         * A list of metric definition objects. Each object specifies the metric name and regular expressions used to
         * parse algorithm logs. SageMaker publishes each metric to Amazon CloudWatch.
         * </p>
         * 
         * @param metricDefinitions
         *        A list of metric definition objects. Each object specifies the metric name and regular expressions
         *        used to parse algorithm logs. SageMaker publishes each metric to Amazon CloudWatch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricDefinitions(Collection<MetricDefinition> metricDefinitions);

        /**
         * <p>
         * A list of metric definition objects. Each object specifies the metric name and regular expressions used to
         * parse algorithm logs. SageMaker publishes each metric to Amazon CloudWatch.
         * </p>
         * 
         * @param metricDefinitions
         *        A list of metric definition objects. Each object specifies the metric name and regular expressions
         *        used to parse algorithm logs. SageMaker publishes each metric to Amazon CloudWatch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricDefinitions(MetricDefinition... metricDefinitions);

        /**
         * <p>
         * A list of metric definition objects. Each object specifies the metric name and regular expressions used to
         * parse algorithm logs. SageMaker publishes each metric to Amazon CloudWatch.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.MetricDefinition.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.sagemaker.model.MetricDefinition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.MetricDefinition.Builder#build()} is called
         * immediately and its result is passed to {@link #metricDefinitions(List<MetricDefinition>)}.
         * 
         * @param metricDefinitions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.MetricDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricDefinitions(java.util.Collection<MetricDefinition>)
         */
        Builder metricDefinitions(Consumer<MetricDefinition.Builder>... metricDefinitions);

        /**
         * <p>
         * To generate and save time-series metrics during training, set to <code>true</code>. The default is
         * <code>false</code> and time-series metrics aren't generated except in the following cases:
         * </p>
         * <ul>
         * <li>
         * <p>
         * You use one of the SageMaker built-in algorithms
         * </p>
         * </li>
         * <li>
         * <p>
         * You use one of the following <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/pre-built-containers-frameworks-deep-learning.html"
         * >Prebuilt SageMaker Docker Images</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Tensorflow (version &gt;= 1.15)
         * </p>
         * </li>
         * <li>
         * <p>
         * MXNet (version &gt;= 1.6)
         * </p>
         * </li>
         * <li>
         * <p>
         * PyTorch (version &gt;= 1.3)
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * You specify at least one <a>MetricDefinition</a>
         * </p>
         * </li>
         * </ul>
         * 
         * @param enableSageMakerMetricsTimeSeries
         *        To generate and save time-series metrics during training, set to <code>true</code>. The default is
         *        <code>false</code> and time-series metrics aren't generated except in the following cases:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        You use one of the SageMaker built-in algorithms
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You use one of the following <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/pre-built-containers-frameworks-deep-learning.html"
         *        >Prebuilt SageMaker Docker Images</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Tensorflow (version &gt;= 1.15)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MXNet (version &gt;= 1.6)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PyTorch (version &gt;= 1.3)
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        You specify at least one <a>MetricDefinition</a>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableSageMakerMetricsTimeSeries(Boolean enableSageMakerMetricsTimeSeries);

        /**
         * <p>
         * The <a href="https://docs.docker.com/engine/reference/builder/">entrypoint script for a Docker container</a>
         * used to run a training job. This script takes precedence over the default train processing instructions. See
         * <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
         * Amazon SageMaker Runs Your Training Image</a> for more information.
         * </p>
         * 
         * @param containerEntrypoint
         *        The <a href="https://docs.docker.com/engine/reference/builder/">entrypoint script for a Docker
         *        container</a> used to run a training job. This script takes precedence over the default train
         *        processing instructions. See <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html"
         *        >How Amazon SageMaker Runs Your Training Image</a> for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerEntrypoint(Collection<String> containerEntrypoint);

        /**
         * <p>
         * The <a href="https://docs.docker.com/engine/reference/builder/">entrypoint script for a Docker container</a>
         * used to run a training job. This script takes precedence over the default train processing instructions. See
         * <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
         * Amazon SageMaker Runs Your Training Image</a> for more information.
         * </p>
         * 
         * @param containerEntrypoint
         *        The <a href="https://docs.docker.com/engine/reference/builder/">entrypoint script for a Docker
         *        container</a> used to run a training job. This script takes precedence over the default train
         *        processing instructions. See <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html"
         *        >How Amazon SageMaker Runs Your Training Image</a> for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerEntrypoint(String... containerEntrypoint);

        /**
         * <p>
         * The arguments for a container used to run a training job. See <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
         * Amazon SageMaker Runs Your Training Image</a> for additional information.
         * </p>
         * 
         * @param containerArguments
         *        The arguments for a container used to run a training job. See <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html"
         *        >How Amazon SageMaker Runs Your Training Image</a> for additional information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerArguments(Collection<String> containerArguments);

        /**
         * <p>
         * The arguments for a container used to run a training job. See <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
         * Amazon SageMaker Runs Your Training Image</a> for additional information.
         * </p>
         * 
         * @param containerArguments
         *        The arguments for a container used to run a training job. See <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html"
         *        >How Amazon SageMaker Runs Your Training Image</a> for additional information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerArguments(String... containerArguments);
    }

    static final class BuilderImpl implements Builder {
        private String trainingImage;

        private String algorithmName;

        private String trainingInputMode;

        private List<MetricDefinition> metricDefinitions = DefaultSdkAutoConstructList.getInstance();

        private Boolean enableSageMakerMetricsTimeSeries;

        private List<String> containerEntrypoint = DefaultSdkAutoConstructList.getInstance();

        private List<String> containerArguments = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AlgorithmSpecification model) {
            trainingImage(model.trainingImage);
            algorithmName(model.algorithmName);
            trainingInputMode(model.trainingInputMode);
            metricDefinitions(model.metricDefinitions);
            enableSageMakerMetricsTimeSeries(model.enableSageMakerMetricsTimeSeries);
            containerEntrypoint(model.containerEntrypoint);
            containerArguments(model.containerArguments);
        }

        public final String getTrainingImage() {
            return trainingImage;
        }

        public final void setTrainingImage(String trainingImage) {
            this.trainingImage = trainingImage;
        }

        @Override
        public final Builder trainingImage(String trainingImage) {
            this.trainingImage = trainingImage;
            return this;
        }

        public final String getAlgorithmName() {
            return algorithmName;
        }

        public final void setAlgorithmName(String algorithmName) {
            this.algorithmName = algorithmName;
        }

        @Override
        public final Builder algorithmName(String algorithmName) {
            this.algorithmName = algorithmName;
            return this;
        }

        public final String getTrainingInputMode() {
            return trainingInputMode;
        }

        public final void setTrainingInputMode(String trainingInputMode) {
            this.trainingInputMode = trainingInputMode;
        }

        @Override
        public final Builder trainingInputMode(String trainingInputMode) {
            this.trainingInputMode = trainingInputMode;
            return this;
        }

        @Override
        public final Builder trainingInputMode(TrainingInputMode trainingInputMode) {
            this.trainingInputMode(trainingInputMode == null ? null : trainingInputMode.toString());
            return this;
        }

        public final List<MetricDefinition.Builder> getMetricDefinitions() {
            List<MetricDefinition.Builder> result = MetricDefinitionListCopier.copyToBuilder(this.metricDefinitions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMetricDefinitions(Collection<MetricDefinition.BuilderImpl> metricDefinitions) {
            this.metricDefinitions = MetricDefinitionListCopier.copyFromBuilder(metricDefinitions);
        }

        @Override
        public final Builder metricDefinitions(Collection<MetricDefinition> metricDefinitions) {
            this.metricDefinitions = MetricDefinitionListCopier.copy(metricDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricDefinitions(MetricDefinition... metricDefinitions) {
            metricDefinitions(Arrays.asList(metricDefinitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricDefinitions(Consumer<MetricDefinition.Builder>... metricDefinitions) {
            metricDefinitions(Stream.of(metricDefinitions).map(c -> MetricDefinition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getEnableSageMakerMetricsTimeSeries() {
            return enableSageMakerMetricsTimeSeries;
        }

        public final void setEnableSageMakerMetricsTimeSeries(Boolean enableSageMakerMetricsTimeSeries) {
            this.enableSageMakerMetricsTimeSeries = enableSageMakerMetricsTimeSeries;
        }

        @Override
        public final Builder enableSageMakerMetricsTimeSeries(Boolean enableSageMakerMetricsTimeSeries) {
            this.enableSageMakerMetricsTimeSeries = enableSageMakerMetricsTimeSeries;
            return this;
        }

        public final Collection<String> getContainerEntrypoint() {
            if (containerEntrypoint instanceof SdkAutoConstructList) {
                return null;
            }
            return containerEntrypoint;
        }

        public final void setContainerEntrypoint(Collection<String> containerEntrypoint) {
            this.containerEntrypoint = TrainingContainerEntrypointCopier.copy(containerEntrypoint);
        }

        @Override
        public final Builder containerEntrypoint(Collection<String> containerEntrypoint) {
            this.containerEntrypoint = TrainingContainerEntrypointCopier.copy(containerEntrypoint);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder containerEntrypoint(String... containerEntrypoint) {
            containerEntrypoint(Arrays.asList(containerEntrypoint));
            return this;
        }

        public final Collection<String> getContainerArguments() {
            if (containerArguments instanceof SdkAutoConstructList) {
                return null;
            }
            return containerArguments;
        }

        public final void setContainerArguments(Collection<String> containerArguments) {
            this.containerArguments = TrainingContainerArgumentsCopier.copy(containerArguments);
        }

        @Override
        public final Builder containerArguments(Collection<String> containerArguments) {
            this.containerArguments = TrainingContainerArgumentsCopier.copy(containerArguments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder containerArguments(String... containerArguments) {
            containerArguments(Arrays.asList(containerArguments));
            return this;
        }

        @Override
        public AlgorithmSpecification build() {
            return new AlgorithmSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
