/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the Amazon S3 data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoMLS3DataSource implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoMLS3DataSource.Builder, AutoMLS3DataSource> {
    private static final SdkField<String> S3_DATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3DataType").getter(getter(AutoMLS3DataSource::s3DataTypeAsString)).setter(setter(Builder::s3DataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3DataType").build()).build();

    private static final SdkField<String> S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("S3Uri")
            .getter(getter(AutoMLS3DataSource::s3Uri)).setter(setter(Builder::s3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Uri").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_DATA_TYPE_FIELD,
            S3_URI_FIELD));

    private static final long serialVersionUID = 1L;

    private final String s3DataType;

    private final String s3Uri;

    private AutoMLS3DataSource(BuilderImpl builder) {
        this.s3DataType = builder.s3DataType;
        this.s3Uri = builder.s3Uri;
    }

    /**
     * <p>
     * The data type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all objects
     * that match the specified key name prefix for model training.
     * </p>
     * <p>
     * The <code>S3Prefix</code> should have the following format:
     * </p>
     * <p>
     * <code>s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER-OR-FILE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
     * containing a list of object keys that you want SageMaker to use for model training.
     * </p>
     * <p>
     * A <code>ManifestFile</code> should have the format shown below:
     * </p>
     * <p>
     * <code>[ {"prefix": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/DOC-EXAMPLE-PREFIX/"}, </code>
     * </p>
     * <p>
     * <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-1",</code>
     * </p>
     * <p>
     * <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-2",</code>
     * </p>
     * <p>
     * <code>... "DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-N" ]</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an augmented
     * manifest file in JSON lines format. This file contains the data you want to use for model training.
     * <code>AugmentedManifestFile</code> is available for V2 API jobs only (for example, for jobs created by calling
     * <code>CreateAutoMLJobV2</code>).
     * </p>
     * <p>
     * Here is a minimal, single-record example of an <code>AugmentedManifestFile</code>:
     * </p>
     * <p>
     * <code>{"source-ref": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/cats/cat.jpg",</code>
     * </p>
     * <p>
     * <code>"label-metadata": {"class-name": "cat"</code>
     * </p>
     * <p>
     * For more information on <code>AugmentedManifestFile</code>, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/augmented-manifest.html">Provide Dataset Metadata to
     * Training Jobs with an Augmented Manifest File</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3DataType} will
     * return {@link AutoMLS3DataType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3DataTypeAsString}.
     * </p>
     * 
     * @return The data type. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all
     *         objects that match the specified key name prefix for model training.
     *         </p>
     *         <p>
     *         The <code>S3Prefix</code> should have the following format:
     *         </p>
     *         <p>
     *         <code>s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER-OR-FILE</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
     *         containing a list of object keys that you want SageMaker to use for model training.
     *         </p>
     *         <p>
     *         A <code>ManifestFile</code> should have the format shown below:
     *         </p>
     *         <p>
     *         <code>[ {"prefix": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/DOC-EXAMPLE-PREFIX/"}, </code>
     *         </p>
     *         <p>
     *         <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-1",</code>
     *         </p>
     *         <p>
     *         <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-2",</code>
     *         </p>
     *         <p>
     *         <code>... "DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-N" ]</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an
     *         augmented manifest file in JSON lines format. This file contains the data you want to use for model
     *         training. <code>AugmentedManifestFile</code> is available for V2 API jobs only (for example, for jobs
     *         created by calling <code>CreateAutoMLJobV2</code>).
     *         </p>
     *         <p>
     *         Here is a minimal, single-record example of an <code>AugmentedManifestFile</code>:
     *         </p>
     *         <p>
     *         <code>{"source-ref": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/cats/cat.jpg",</code>
     *         </p>
     *         <p>
     *         <code>"label-metadata": {"class-name": "cat"</code>
     *         </p>
     *         <p>
     *         For more information on <code>AugmentedManifestFile</code>, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/augmented-manifest.html">Provide Dataset Metadata
     *         to Training Jobs with an Augmented Manifest File</a>.
     *         </p>
     *         </li>
     * @see AutoMLS3DataType
     */
    public final AutoMLS3DataType s3DataType() {
        return AutoMLS3DataType.fromValue(s3DataType);
    }

    /**
     * <p>
     * The data type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all objects
     * that match the specified key name prefix for model training.
     * </p>
     * <p>
     * The <code>S3Prefix</code> should have the following format:
     * </p>
     * <p>
     * <code>s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER-OR-FILE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
     * containing a list of object keys that you want SageMaker to use for model training.
     * </p>
     * <p>
     * A <code>ManifestFile</code> should have the format shown below:
     * </p>
     * <p>
     * <code>[ {"prefix": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/DOC-EXAMPLE-PREFIX/"}, </code>
     * </p>
     * <p>
     * <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-1",</code>
     * </p>
     * <p>
     * <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-2",</code>
     * </p>
     * <p>
     * <code>... "DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-N" ]</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an augmented
     * manifest file in JSON lines format. This file contains the data you want to use for model training.
     * <code>AugmentedManifestFile</code> is available for V2 API jobs only (for example, for jobs created by calling
     * <code>CreateAutoMLJobV2</code>).
     * </p>
     * <p>
     * Here is a minimal, single-record example of an <code>AugmentedManifestFile</code>:
     * </p>
     * <p>
     * <code>{"source-ref": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/cats/cat.jpg",</code>
     * </p>
     * <p>
     * <code>"label-metadata": {"class-name": "cat"</code>
     * </p>
     * <p>
     * For more information on <code>AugmentedManifestFile</code>, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/augmented-manifest.html">Provide Dataset Metadata to
     * Training Jobs with an Augmented Manifest File</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3DataType} will
     * return {@link AutoMLS3DataType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3DataTypeAsString}.
     * </p>
     * 
     * @return The data type. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all
     *         objects that match the specified key name prefix for model training.
     *         </p>
     *         <p>
     *         The <code>S3Prefix</code> should have the following format:
     *         </p>
     *         <p>
     *         <code>s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER-OR-FILE</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
     *         containing a list of object keys that you want SageMaker to use for model training.
     *         </p>
     *         <p>
     *         A <code>ManifestFile</code> should have the format shown below:
     *         </p>
     *         <p>
     *         <code>[ {"prefix": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/DOC-EXAMPLE-PREFIX/"}, </code>
     *         </p>
     *         <p>
     *         <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-1",</code>
     *         </p>
     *         <p>
     *         <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-2",</code>
     *         </p>
     *         <p>
     *         <code>... "DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-N" ]</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an
     *         augmented manifest file in JSON lines format. This file contains the data you want to use for model
     *         training. <code>AugmentedManifestFile</code> is available for V2 API jobs only (for example, for jobs
     *         created by calling <code>CreateAutoMLJobV2</code>).
     *         </p>
     *         <p>
     *         Here is a minimal, single-record example of an <code>AugmentedManifestFile</code>:
     *         </p>
     *         <p>
     *         <code>{"source-ref": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/cats/cat.jpg",</code>
     *         </p>
     *         <p>
     *         <code>"label-metadata": {"class-name": "cat"</code>
     *         </p>
     *         <p>
     *         For more information on <code>AugmentedManifestFile</code>, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/augmented-manifest.html">Provide Dataset Metadata
     *         to Training Jobs with an Augmented Manifest File</a>.
     *         </p>
     *         </li>
     * @see AutoMLS3DataType
     */
    public final String s3DataTypeAsString() {
        return s3DataType;
    }

    /**
     * <p>
     * The URL to the Amazon S3 data source. The Uri refers to the Amazon S3 prefix or ManifestFile depending on the
     * data type.
     * </p>
     * 
     * @return The URL to the Amazon S3 data source. The Uri refers to the Amazon S3 prefix or ManifestFile depending on
     *         the data type.
     */
    public final String s3Uri() {
        return s3Uri;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3DataTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(s3Uri());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoMLS3DataSource)) {
            return false;
        }
        AutoMLS3DataSource other = (AutoMLS3DataSource) obj;
        return Objects.equals(s3DataTypeAsString(), other.s3DataTypeAsString()) && Objects.equals(s3Uri(), other.s3Uri());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AutoMLS3DataSource").add("S3DataType", s3DataTypeAsString()).add("S3Uri", s3Uri()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3DataType":
            return Optional.ofNullable(clazz.cast(s3DataTypeAsString()));
        case "S3Uri":
            return Optional.ofNullable(clazz.cast(s3Uri()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutoMLS3DataSource, T> g) {
        return obj -> g.apply((AutoMLS3DataSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoMLS3DataSource> {
        /**
         * <p>
         * The data type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all
         * objects that match the specified key name prefix for model training.
         * </p>
         * <p>
         * The <code>S3Prefix</code> should have the following format:
         * </p>
         * <p>
         * <code>s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER-OR-FILE</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
         * containing a list of object keys that you want SageMaker to use for model training.
         * </p>
         * <p>
         * A <code>ManifestFile</code> should have the format shown below:
         * </p>
         * <p>
         * <code>[ {"prefix": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/DOC-EXAMPLE-PREFIX/"}, </code>
         * </p>
         * <p>
         * <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-1",</code>
         * </p>
         * <p>
         * <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-2",</code>
         * </p>
         * <p>
         * <code>... "DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-N" ]</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an
         * augmented manifest file in JSON lines format. This file contains the data you want to use for model training.
         * <code>AugmentedManifestFile</code> is available for V2 API jobs only (for example, for jobs created by
         * calling <code>CreateAutoMLJobV2</code>).
         * </p>
         * <p>
         * Here is a minimal, single-record example of an <code>AugmentedManifestFile</code>:
         * </p>
         * <p>
         * <code>{"source-ref": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/cats/cat.jpg",</code>
         * </p>
         * <p>
         * <code>"label-metadata": {"class-name": "cat"</code>
         * </p>
         * <p>
         * For more information on <code>AugmentedManifestFile</code>, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/augmented-manifest.html">Provide Dataset Metadata to
         * Training Jobs with an Augmented Manifest File</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param s3DataType
         *        The data type. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses
         *        all objects that match the specified key name prefix for model training.
         *        </p>
         *        <p>
         *        The <code>S3Prefix</code> should have the following format:
         *        </p>
         *        <p>
         *        <code>s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER-OR-FILE</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest
         *        file containing a list of object keys that you want SageMaker to use for model training.
         *        </p>
         *        <p>
         *        A <code>ManifestFile</code> should have the format shown below:
         *        </p>
         *        <p>
         *        <code>[ {"prefix": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/DOC-EXAMPLE-PREFIX/"}, </code>
         *        </p>
         *        <p>
         *        <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-1",</code>
         *        </p>
         *        <p>
         *        <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-2",</code>
         *        </p>
         *        <p>
         *        <code>... "DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-N" ]</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an
         *        augmented manifest file in JSON lines format. This file contains the data you want to use for model
         *        training. <code>AugmentedManifestFile</code> is available for V2 API jobs only (for example, for jobs
         *        created by calling <code>CreateAutoMLJobV2</code>).
         *        </p>
         *        <p>
         *        Here is a minimal, single-record example of an <code>AugmentedManifestFile</code>:
         *        </p>
         *        <p>
         *        <code>{"source-ref": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/cats/cat.jpg",</code>
         *        </p>
         *        <p>
         *        <code>"label-metadata": {"class-name": "cat"</code>
         *        </p>
         *        <p>
         *        For more information on <code>AugmentedManifestFile</code>, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/augmented-manifest.html">Provide Dataset
         *        Metadata to Training Jobs with an Augmented Manifest File</a>.
         *        </p>
         *        </li>
         * @see AutoMLS3DataType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoMLS3DataType
         */
        Builder s3DataType(String s3DataType);

        /**
         * <p>
         * The data type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all
         * objects that match the specified key name prefix for model training.
         * </p>
         * <p>
         * The <code>S3Prefix</code> should have the following format:
         * </p>
         * <p>
         * <code>s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER-OR-FILE</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
         * containing a list of object keys that you want SageMaker to use for model training.
         * </p>
         * <p>
         * A <code>ManifestFile</code> should have the format shown below:
         * </p>
         * <p>
         * <code>[ {"prefix": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/DOC-EXAMPLE-PREFIX/"}, </code>
         * </p>
         * <p>
         * <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-1",</code>
         * </p>
         * <p>
         * <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-2",</code>
         * </p>
         * <p>
         * <code>... "DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-N" ]</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an
         * augmented manifest file in JSON lines format. This file contains the data you want to use for model training.
         * <code>AugmentedManifestFile</code> is available for V2 API jobs only (for example, for jobs created by
         * calling <code>CreateAutoMLJobV2</code>).
         * </p>
         * <p>
         * Here is a minimal, single-record example of an <code>AugmentedManifestFile</code>:
         * </p>
         * <p>
         * <code>{"source-ref": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/cats/cat.jpg",</code>
         * </p>
         * <p>
         * <code>"label-metadata": {"class-name": "cat"</code>
         * </p>
         * <p>
         * For more information on <code>AugmentedManifestFile</code>, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/augmented-manifest.html">Provide Dataset Metadata to
         * Training Jobs with an Augmented Manifest File</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param s3DataType
         *        The data type. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses
         *        all objects that match the specified key name prefix for model training.
         *        </p>
         *        <p>
         *        The <code>S3Prefix</code> should have the following format:
         *        </p>
         *        <p>
         *        <code>s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER-OR-FILE</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest
         *        file containing a list of object keys that you want SageMaker to use for model training.
         *        </p>
         *        <p>
         *        A <code>ManifestFile</code> should have the format shown below:
         *        </p>
         *        <p>
         *        <code>[ {"prefix": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/DOC-EXAMPLE-PREFIX/"}, </code>
         *        </p>
         *        <p>
         *        <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-1",</code>
         *        </p>
         *        <p>
         *        <code>"DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-2",</code>
         *        </p>
         *        <p>
         *        <code>... "DOC-EXAMPLE-RELATIVE-PATH/DOC-EXAMPLE-FOLDER/DATA-N" ]</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an
         *        augmented manifest file in JSON lines format. This file contains the data you want to use for model
         *        training. <code>AugmentedManifestFile</code> is available for V2 API jobs only (for example, for jobs
         *        created by calling <code>CreateAutoMLJobV2</code>).
         *        </p>
         *        <p>
         *        Here is a minimal, single-record example of an <code>AugmentedManifestFile</code>:
         *        </p>
         *        <p>
         *        <code>{"source-ref": "s3://DOC-EXAMPLE-BUCKET/DOC-EXAMPLE-FOLDER/cats/cat.jpg",</code>
         *        </p>
         *        <p>
         *        <code>"label-metadata": {"class-name": "cat"</code>
         *        </p>
         *        <p>
         *        For more information on <code>AugmentedManifestFile</code>, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/augmented-manifest.html">Provide Dataset
         *        Metadata to Training Jobs with an Augmented Manifest File</a>.
         *        </p>
         *        </li>
         * @see AutoMLS3DataType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoMLS3DataType
         */
        Builder s3DataType(AutoMLS3DataType s3DataType);

        /**
         * <p>
         * The URL to the Amazon S3 data source. The Uri refers to the Amazon S3 prefix or ManifestFile depending on the
         * data type.
         * </p>
         * 
         * @param s3Uri
         *        The URL to the Amazon S3 data source. The Uri refers to the Amazon S3 prefix or ManifestFile depending
         *        on the data type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Uri(String s3Uri);
    }

    static final class BuilderImpl implements Builder {
        private String s3DataType;

        private String s3Uri;

        private BuilderImpl() {
        }

        private BuilderImpl(AutoMLS3DataSource model) {
            s3DataType(model.s3DataType);
            s3Uri(model.s3Uri);
        }

        public final String getS3DataType() {
            return s3DataType;
        }

        public final void setS3DataType(String s3DataType) {
            this.s3DataType = s3DataType;
        }

        @Override
        public final Builder s3DataType(String s3DataType) {
            this.s3DataType = s3DataType;
            return this;
        }

        @Override
        public final Builder s3DataType(AutoMLS3DataType s3DataType) {
            this.s3DataType(s3DataType == null ? null : s3DataType.toString());
            return this;
        }

        public final String getS3Uri() {
            return s3Uri;
        }

        public final void setS3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
        }

        @Override
        public final Builder s3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
            return this;
        }

        @Override
        public AutoMLS3DataSource build() {
            return new AutoMLS3DataSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
