/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines a hyperparameter to be used by an algorithm.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HyperParameterSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<HyperParameterSpecification.Builder, HyperParameterSpecification> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(HyperParameterSpecification::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(HyperParameterSpecification::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(HyperParameterSpecification::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<ParameterRange> RANGE_FIELD = SdkField.<ParameterRange> builder(MarshallingType.SDK_POJO)
            .memberName("Range").getter(getter(HyperParameterSpecification::range)).setter(setter(Builder::range))
            .constructor(ParameterRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Range").build()).build();

    private static final SdkField<Boolean> IS_TUNABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsTunable").getter(getter(HyperParameterSpecification::isTunable)).setter(setter(Builder::isTunable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsTunable").build()).build();

    private static final SdkField<Boolean> IS_REQUIRED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsRequired").getter(getter(HyperParameterSpecification::isRequired)).setter(setter(Builder::isRequired))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsRequired").build()).build();

    private static final SdkField<String> DEFAULT_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DefaultValue").getter(getter(HyperParameterSpecification::defaultValue))
            .setter(setter(Builder::defaultValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultValue").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            TYPE_FIELD, RANGE_FIELD, IS_TUNABLE_FIELD, IS_REQUIRED_FIELD, DEFAULT_VALUE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final String type;

    private final ParameterRange range;

    private final Boolean isTunable;

    private final Boolean isRequired;

    private final String defaultValue;

    private HyperParameterSpecification(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.type = builder.type;
        this.range = builder.range;
        this.isTunable = builder.isTunable;
        this.isRequired = builder.isRequired;
        this.defaultValue = builder.defaultValue;
    }

    /**
     * <p>
     * The name of this hyperparameter. The name must be unique.
     * </p>
     * 
     * @return The name of this hyperparameter. The name must be unique.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A brief description of the hyperparameter.
     * </p>
     * 
     * @return A brief description of the hyperparameter.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The type of this hyperparameter. The valid types are <code>Integer</code>, <code>Continuous</code>,
     * <code>Categorical</code>, and <code>FreeText</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of this hyperparameter. The valid types are <code>Integer</code>, <code>Continuous</code>,
     *         <code>Categorical</code>, and <code>FreeText</code>.
     * @see ParameterType
     */
    public final ParameterType type() {
        return ParameterType.fromValue(type);
    }

    /**
     * <p>
     * The type of this hyperparameter. The valid types are <code>Integer</code>, <code>Continuous</code>,
     * <code>Categorical</code>, and <code>FreeText</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of this hyperparameter. The valid types are <code>Integer</code>, <code>Continuous</code>,
     *         <code>Categorical</code>, and <code>FreeText</code>.
     * @see ParameterType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The allowed range for this hyperparameter.
     * </p>
     * 
     * @return The allowed range for this hyperparameter.
     */
    public final ParameterRange range() {
        return range;
    }

    /**
     * <p>
     * Indicates whether this hyperparameter is tunable in a hyperparameter tuning job.
     * </p>
     * 
     * @return Indicates whether this hyperparameter is tunable in a hyperparameter tuning job.
     */
    public final Boolean isTunable() {
        return isTunable;
    }

    /**
     * <p>
     * Indicates whether this hyperparameter is required.
     * </p>
     * 
     * @return Indicates whether this hyperparameter is required.
     */
    public final Boolean isRequired() {
        return isRequired;
    }

    /**
     * <p>
     * The default value for this hyperparameter. If a default value is specified, a hyperparameter cannot be required.
     * </p>
     * 
     * @return The default value for this hyperparameter. If a default value is specified, a hyperparameter cannot be
     *         required.
     */
    public final String defaultValue() {
        return defaultValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(range());
        hashCode = 31 * hashCode + Objects.hashCode(isTunable());
        hashCode = 31 * hashCode + Objects.hashCode(isRequired());
        hashCode = 31 * hashCode + Objects.hashCode(defaultValue());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HyperParameterSpecification)) {
            return false;
        }
        HyperParameterSpecification other = (HyperParameterSpecification) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(range(), other.range())
                && Objects.equals(isTunable(), other.isTunable()) && Objects.equals(isRequired(), other.isRequired())
                && Objects.equals(defaultValue(), other.defaultValue());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HyperParameterSpecification").add("Name", name()).add("Description", description())
                .add("Type", typeAsString()).add("Range", range()).add("IsTunable", isTunable()).add("IsRequired", isRequired())
                .add("DefaultValue", defaultValue()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Range":
            return Optional.ofNullable(clazz.cast(range()));
        case "IsTunable":
            return Optional.ofNullable(clazz.cast(isTunable()));
        case "IsRequired":
            return Optional.ofNullable(clazz.cast(isRequired()));
        case "DefaultValue":
            return Optional.ofNullable(clazz.cast(defaultValue()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<HyperParameterSpecification, T> g) {
        return obj -> g.apply((HyperParameterSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HyperParameterSpecification> {
        /**
         * <p>
         * The name of this hyperparameter. The name must be unique.
         * </p>
         * 
         * @param name
         *        The name of this hyperparameter. The name must be unique.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A brief description of the hyperparameter.
         * </p>
         * 
         * @param description
         *        A brief description of the hyperparameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The type of this hyperparameter. The valid types are <code>Integer</code>, <code>Continuous</code>,
         * <code>Categorical</code>, and <code>FreeText</code>.
         * </p>
         * 
         * @param type
         *        The type of this hyperparameter. The valid types are <code>Integer</code>, <code>Continuous</code>,
         *        <code>Categorical</code>, and <code>FreeText</code>.
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of this hyperparameter. The valid types are <code>Integer</code>, <code>Continuous</code>,
         * <code>Categorical</code>, and <code>FreeText</code>.
         * </p>
         * 
         * @param type
         *        The type of this hyperparameter. The valid types are <code>Integer</code>, <code>Continuous</code>,
         *        <code>Categorical</code>, and <code>FreeText</code>.
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(ParameterType type);

        /**
         * <p>
         * The allowed range for this hyperparameter.
         * </p>
         * 
         * @param range
         *        The allowed range for this hyperparameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder range(ParameterRange range);

        /**
         * <p>
         * The allowed range for this hyperparameter.
         * </p>
         * This is a convenience method that creates an instance of the {@link ParameterRange.Builder} avoiding the need
         * to create one manually via {@link ParameterRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ParameterRange.Builder#build()} is called immediately and its
         * result is passed to {@link #range(ParameterRange)}.
         * 
         * @param range
         *        a consumer that will call methods on {@link ParameterRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #range(ParameterRange)
         */
        default Builder range(Consumer<ParameterRange.Builder> range) {
            return range(ParameterRange.builder().applyMutation(range).build());
        }

        /**
         * <p>
         * Indicates whether this hyperparameter is tunable in a hyperparameter tuning job.
         * </p>
         * 
         * @param isTunable
         *        Indicates whether this hyperparameter is tunable in a hyperparameter tuning job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isTunable(Boolean isTunable);

        /**
         * <p>
         * Indicates whether this hyperparameter is required.
         * </p>
         * 
         * @param isRequired
         *        Indicates whether this hyperparameter is required.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isRequired(Boolean isRequired);

        /**
         * <p>
         * The default value for this hyperparameter. If a default value is specified, a hyperparameter cannot be
         * required.
         * </p>
         * 
         * @param defaultValue
         *        The default value for this hyperparameter. If a default value is specified, a hyperparameter cannot be
         *        required.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultValue(String defaultValue);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private String type;

        private ParameterRange range;

        private Boolean isTunable;

        private Boolean isRequired;

        private String defaultValue;

        private BuilderImpl() {
        }

        private BuilderImpl(HyperParameterSpecification model) {
            name(model.name);
            description(model.description);
            type(model.type);
            range(model.range);
            isTunable(model.isTunable);
            isRequired(model.isRequired);
            defaultValue(model.defaultValue);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ParameterType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final ParameterRange.Builder getRange() {
            return range != null ? range.toBuilder() : null;
        }

        public final void setRange(ParameterRange.BuilderImpl range) {
            this.range = range != null ? range.build() : null;
        }

        @Override
        public final Builder range(ParameterRange range) {
            this.range = range;
            return this;
        }

        public final Boolean getIsTunable() {
            return isTunable;
        }

        public final void setIsTunable(Boolean isTunable) {
            this.isTunable = isTunable;
        }

        @Override
        public final Builder isTunable(Boolean isTunable) {
            this.isTunable = isTunable;
            return this;
        }

        public final Boolean getIsRequired() {
            return isRequired;
        }

        public final void setIsRequired(Boolean isRequired) {
            this.isRequired = isRequired;
        }

        @Override
        public final Builder isRequired(Boolean isRequired) {
            this.isRequired = isRequired;
            return this;
        }

        public final String getDefaultValue() {
            return defaultValue;
        }

        public final void setDefaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        @Override
        public final Builder defaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
            return this;
        }

        @Override
        public HyperParameterSpecification build() {
            return new HyperParameterSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
