/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the configurations of selected devices.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeviceSelectionConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<DeviceSelectionConfig.Builder, DeviceSelectionConfig> {
    private static final SdkField<String> DEVICE_SUBSET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeviceSubsetType").getter(getter(DeviceSelectionConfig::deviceSubsetTypeAsString))
            .setter(setter(Builder::deviceSubsetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeviceSubsetType").build()).build();

    private static final SdkField<Integer> PERCENTAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Percentage").getter(getter(DeviceSelectionConfig::percentage)).setter(setter(Builder::percentage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Percentage").build()).build();

    private static final SdkField<List<String>> DEVICE_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DeviceNames")
            .getter(getter(DeviceSelectionConfig::deviceNames))
            .setter(setter(Builder::deviceNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeviceNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DEVICE_NAME_CONTAINS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeviceNameContains").getter(getter(DeviceSelectionConfig::deviceNameContains))
            .setter(setter(Builder::deviceNameContains))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeviceNameContains").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DEVICE_SUBSET_TYPE_FIELD,
            PERCENTAGE_FIELD, DEVICE_NAMES_FIELD, DEVICE_NAME_CONTAINS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String deviceSubsetType;

    private final Integer percentage;

    private final List<String> deviceNames;

    private final String deviceNameContains;

    private DeviceSelectionConfig(BuilderImpl builder) {
        this.deviceSubsetType = builder.deviceSubsetType;
        this.percentage = builder.percentage;
        this.deviceNames = builder.deviceNames;
        this.deviceNameContains = builder.deviceNameContains;
    }

    /**
     * <p>
     * Type of device subsets to deploy to the current stage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deviceSubsetType}
     * will return {@link DeviceSubsetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #deviceSubsetTypeAsString}.
     * </p>
     * 
     * @return Type of device subsets to deploy to the current stage.
     * @see DeviceSubsetType
     */
    public final DeviceSubsetType deviceSubsetType() {
        return DeviceSubsetType.fromValue(deviceSubsetType);
    }

    /**
     * <p>
     * Type of device subsets to deploy to the current stage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deviceSubsetType}
     * will return {@link DeviceSubsetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #deviceSubsetTypeAsString}.
     * </p>
     * 
     * @return Type of device subsets to deploy to the current stage.
     * @see DeviceSubsetType
     */
    public final String deviceSubsetTypeAsString() {
        return deviceSubsetType;
    }

    /**
     * <p>
     * Percentage of devices in the fleet to deploy to the current stage.
     * </p>
     * 
     * @return Percentage of devices in the fleet to deploy to the current stage.
     */
    public final Integer percentage() {
        return percentage;
    }

    /**
     * For responses, this returns true if the service returned a value for the DeviceNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDeviceNames() {
        return deviceNames != null && !(deviceNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of devices chosen to deploy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDeviceNames} method.
     * </p>
     * 
     * @return List of devices chosen to deploy.
     */
    public final List<String> deviceNames() {
        return deviceNames;
    }

    /**
     * <p>
     * A filter to select devices with names containing this name.
     * </p>
     * 
     * @return A filter to select devices with names containing this name.
     */
    public final String deviceNameContains() {
        return deviceNameContains;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deviceSubsetTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(percentage());
        hashCode = 31 * hashCode + Objects.hashCode(hasDeviceNames() ? deviceNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(deviceNameContains());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeviceSelectionConfig)) {
            return false;
        }
        DeviceSelectionConfig other = (DeviceSelectionConfig) obj;
        return Objects.equals(deviceSubsetTypeAsString(), other.deviceSubsetTypeAsString())
                && Objects.equals(percentage(), other.percentage()) && hasDeviceNames() == other.hasDeviceNames()
                && Objects.equals(deviceNames(), other.deviceNames())
                && Objects.equals(deviceNameContains(), other.deviceNameContains());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeviceSelectionConfig").add("DeviceSubsetType", deviceSubsetTypeAsString())
                .add("Percentage", percentage()).add("DeviceNames", hasDeviceNames() ? deviceNames() : null)
                .add("DeviceNameContains", deviceNameContains()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DeviceSubsetType":
            return Optional.ofNullable(clazz.cast(deviceSubsetTypeAsString()));
        case "Percentage":
            return Optional.ofNullable(clazz.cast(percentage()));
        case "DeviceNames":
            return Optional.ofNullable(clazz.cast(deviceNames()));
        case "DeviceNameContains":
            return Optional.ofNullable(clazz.cast(deviceNameContains()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeviceSelectionConfig, T> g) {
        return obj -> g.apply((DeviceSelectionConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeviceSelectionConfig> {
        /**
         * <p>
         * Type of device subsets to deploy to the current stage.
         * </p>
         * 
         * @param deviceSubsetType
         *        Type of device subsets to deploy to the current stage.
         * @see DeviceSubsetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeviceSubsetType
         */
        Builder deviceSubsetType(String deviceSubsetType);

        /**
         * <p>
         * Type of device subsets to deploy to the current stage.
         * </p>
         * 
         * @param deviceSubsetType
         *        Type of device subsets to deploy to the current stage.
         * @see DeviceSubsetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeviceSubsetType
         */
        Builder deviceSubsetType(DeviceSubsetType deviceSubsetType);

        /**
         * <p>
         * Percentage of devices in the fleet to deploy to the current stage.
         * </p>
         * 
         * @param percentage
         *        Percentage of devices in the fleet to deploy to the current stage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentage(Integer percentage);

        /**
         * <p>
         * List of devices chosen to deploy.
         * </p>
         * 
         * @param deviceNames
         *        List of devices chosen to deploy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviceNames(Collection<String> deviceNames);

        /**
         * <p>
         * List of devices chosen to deploy.
         * </p>
         * 
         * @param deviceNames
         *        List of devices chosen to deploy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviceNames(String... deviceNames);

        /**
         * <p>
         * A filter to select devices with names containing this name.
         * </p>
         * 
         * @param deviceNameContains
         *        A filter to select devices with names containing this name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviceNameContains(String deviceNameContains);
    }

    static final class BuilderImpl implements Builder {
        private String deviceSubsetType;

        private Integer percentage;

        private List<String> deviceNames = DefaultSdkAutoConstructList.getInstance();

        private String deviceNameContains;

        private BuilderImpl() {
        }

        private BuilderImpl(DeviceSelectionConfig model) {
            deviceSubsetType(model.deviceSubsetType);
            percentage(model.percentage);
            deviceNames(model.deviceNames);
            deviceNameContains(model.deviceNameContains);
        }

        public final String getDeviceSubsetType() {
            return deviceSubsetType;
        }

        public final void setDeviceSubsetType(String deviceSubsetType) {
            this.deviceSubsetType = deviceSubsetType;
        }

        @Override
        public final Builder deviceSubsetType(String deviceSubsetType) {
            this.deviceSubsetType = deviceSubsetType;
            return this;
        }

        @Override
        public final Builder deviceSubsetType(DeviceSubsetType deviceSubsetType) {
            this.deviceSubsetType(deviceSubsetType == null ? null : deviceSubsetType.toString());
            return this;
        }

        public final Integer getPercentage() {
            return percentage;
        }

        public final void setPercentage(Integer percentage) {
            this.percentage = percentage;
        }

        @Override
        public final Builder percentage(Integer percentage) {
            this.percentage = percentage;
            return this;
        }

        public final Collection<String> getDeviceNames() {
            if (deviceNames instanceof SdkAutoConstructList) {
                return null;
            }
            return deviceNames;
        }

        public final void setDeviceNames(Collection<String> deviceNames) {
            this.deviceNames = DeviceNamesCopier.copy(deviceNames);
        }

        @Override
        public final Builder deviceNames(Collection<String> deviceNames) {
            this.deviceNames = DeviceNamesCopier.copy(deviceNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder deviceNames(String... deviceNames) {
            deviceNames(Arrays.asList(deviceNames));
            return this;
        }

        public final String getDeviceNameContains() {
            return deviceNameContains;
        }

        public final void setDeviceNameContains(String deviceNameContains) {
            this.deviceNameContains = deviceNameContains;
        }

        @Override
        public final Builder deviceNameContains(String deviceNameContains) {
            this.deviceNameContains = deviceNameContains;
            return this;
        }

        @Override
        public DeviceSelectionConfig build() {
            return new DeviceSelectionConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
