/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.sagemaker.SageMakerAsyncClient;
import software.amazon.awssdk.services.sagemaker.model.DescribeEndpointRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeEndpointResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeImageRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeImageResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeImageVersionRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeImageVersionResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeNotebookInstanceRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeNotebookInstanceResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeProcessingJobRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeProcessingJobResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeTrainingJobRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeTrainingJobResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeTransformJobRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeTransformJobResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
public interface SageMakerAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link SageMakerAsyncClient#describeEndpoint} API until the desired condition {@code EndpointDeleted} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeEndpointRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeEndpointResponse>> waitUntilEndpointDeleted(
            DescribeEndpointRequest describeEndpointRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeEndpoint} API until the desired condition {@code EndpointDeleted} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeEndpointRequest#builder()}
     *
     * @param describeEndpointRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeEndpointResponse>> waitUntilEndpointDeleted(
            Consumer<DescribeEndpointRequest.Builder> describeEndpointRequest) {
        return waitUntilEndpointDeleted(DescribeEndpointRequest.builder().applyMutation(describeEndpointRequest).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeEndpoint} API until the desired condition {@code EndpointDeleted} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeEndpointRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeEndpointResponse>> waitUntilEndpointDeleted(
            DescribeEndpointRequest describeEndpointRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeEndpoint} API until the desired condition {@code EndpointDeleted} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeEndpointRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeEndpointResponse>> waitUntilEndpointDeleted(
            Consumer<DescribeEndpointRequest.Builder> describeEndpointRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilEndpointDeleted(DescribeEndpointRequest.builder().applyMutation(describeEndpointRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeEndpoint} API until the desired condition {@code EndpointInService} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeEndpointRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeEndpointResponse>> waitUntilEndpointInService(
            DescribeEndpointRequest describeEndpointRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeEndpoint} API until the desired condition {@code EndpointInService} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeEndpointRequest#builder()}
     *
     * @param describeEndpointRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeEndpointResponse>> waitUntilEndpointInService(
            Consumer<DescribeEndpointRequest.Builder> describeEndpointRequest) {
        return waitUntilEndpointInService(DescribeEndpointRequest.builder().applyMutation(describeEndpointRequest).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeEndpoint} API until the desired condition {@code EndpointInService} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeEndpointRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeEndpointResponse>> waitUntilEndpointInService(
            DescribeEndpointRequest describeEndpointRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeEndpoint} API until the desired condition {@code EndpointInService} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeEndpointRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeEndpointResponse>> waitUntilEndpointInService(
            Consumer<DescribeEndpointRequest.Builder> describeEndpointRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilEndpointInService(DescribeEndpointRequest.builder().applyMutation(describeEndpointRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImage} API until the desired condition {@code ImageCreated} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeImageRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageCreated(
            DescribeImageRequest describeImageRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImage} API until the desired condition {@code ImageCreated} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeImageRequest#builder()}
     *
     * @param describeImageRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageCreated(
            Consumer<DescribeImageRequest.Builder> describeImageRequest) {
        return waitUntilImageCreated(DescribeImageRequest.builder().applyMutation(describeImageRequest).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImage} API until the desired condition {@code ImageCreated} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeImageRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageCreated(
            DescribeImageRequest describeImageRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImage} API until the desired condition {@code ImageCreated} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeImageRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageCreated(
            Consumer<DescribeImageRequest.Builder> describeImageRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilImageCreated(DescribeImageRequest.builder().applyMutation(describeImageRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImage} API until the desired condition {@code ImageDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeImageRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageDeleted(
            DescribeImageRequest describeImageRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImage} API until the desired condition {@code ImageDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeImageRequest#builder()}
     *
     * @param describeImageRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageDeleted(
            Consumer<DescribeImageRequest.Builder> describeImageRequest) {
        return waitUntilImageDeleted(DescribeImageRequest.builder().applyMutation(describeImageRequest).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImage} API until the desired condition {@code ImageDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeImageRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageDeleted(
            DescribeImageRequest describeImageRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImage} API until the desired condition {@code ImageDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeImageRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageDeleted(
            Consumer<DescribeImageRequest.Builder> describeImageRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilImageDeleted(DescribeImageRequest.builder().applyMutation(describeImageRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImage} API until the desired condition {@code ImageUpdated} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeImageRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageUpdated(
            DescribeImageRequest describeImageRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImage} API until the desired condition {@code ImageUpdated} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeImageRequest#builder()}
     *
     * @param describeImageRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageUpdated(
            Consumer<DescribeImageRequest.Builder> describeImageRequest) {
        return waitUntilImageUpdated(DescribeImageRequest.builder().applyMutation(describeImageRequest).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImage} API until the desired condition {@code ImageUpdated} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeImageRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageUpdated(
            DescribeImageRequest describeImageRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImage} API until the desired condition {@code ImageUpdated} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeImageRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageUpdated(
            Consumer<DescribeImageRequest.Builder> describeImageRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilImageUpdated(DescribeImageRequest.builder().applyMutation(describeImageRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImageVersion} API until the desired condition
     * {@code ImageVersionCreated} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeImageVersionRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeImageVersionResponse>> waitUntilImageVersionCreated(
            DescribeImageVersionRequest describeImageVersionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImageVersion} API until the desired condition
     * {@code ImageVersionCreated} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeImageVersionRequest#builder()}
     *
     * @param describeImageVersionRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageVersionResponse>> waitUntilImageVersionCreated(
            Consumer<DescribeImageVersionRequest.Builder> describeImageVersionRequest) {
        return waitUntilImageVersionCreated(DescribeImageVersionRequest.builder().applyMutation(describeImageVersionRequest)
                .build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImageVersion} API until the desired condition
     * {@code ImageVersionCreated} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeImageVersionRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageVersionResponse>> waitUntilImageVersionCreated(
            DescribeImageVersionRequest describeImageVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImageVersion} API until the desired condition
     * {@code ImageVersionCreated} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeImageVersionRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageVersionResponse>> waitUntilImageVersionCreated(
            Consumer<DescribeImageVersionRequest.Builder> describeImageVersionRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilImageVersionCreated(DescribeImageVersionRequest.builder().applyMutation(describeImageVersionRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImageVersion} API until the desired condition
     * {@code ImageVersionDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeImageVersionRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeImageVersionResponse>> waitUntilImageVersionDeleted(
            DescribeImageVersionRequest describeImageVersionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImageVersion} API until the desired condition
     * {@code ImageVersionDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeImageVersionRequest#builder()}
     *
     * @param describeImageVersionRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageVersionResponse>> waitUntilImageVersionDeleted(
            Consumer<DescribeImageVersionRequest.Builder> describeImageVersionRequest) {
        return waitUntilImageVersionDeleted(DescribeImageVersionRequest.builder().applyMutation(describeImageVersionRequest)
                .build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImageVersion} API until the desired condition
     * {@code ImageVersionDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeImageVersionRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageVersionResponse>> waitUntilImageVersionDeleted(
            DescribeImageVersionRequest describeImageVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeImageVersion} API until the desired condition
     * {@code ImageVersionDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeImageVersionRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeImageVersionResponse>> waitUntilImageVersionDeleted(
            Consumer<DescribeImageVersionRequest.Builder> describeImageVersionRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilImageVersionDeleted(DescribeImageVersionRequest.builder().applyMutation(describeImageVersionRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeNotebookInstance} API until the desired condition
     * {@code NotebookInstanceDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeNotebookInstanceRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceDeleted(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeNotebookInstance} API until the desired condition
     * {@code NotebookInstanceDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeNotebookInstanceRequest#builder()}
     *
     * @param describeNotebookInstanceRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceDeleted(
            Consumer<DescribeNotebookInstanceRequest.Builder> describeNotebookInstanceRequest) {
        return waitUntilNotebookInstanceDeleted(DescribeNotebookInstanceRequest.builder()
                .applyMutation(describeNotebookInstanceRequest).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeNotebookInstance} API until the desired condition
     * {@code NotebookInstanceDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeNotebookInstanceRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceDeleted(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeNotebookInstance} API until the desired condition
     * {@code NotebookInstanceDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeNotebookInstanceRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceDeleted(
            Consumer<DescribeNotebookInstanceRequest.Builder> describeNotebookInstanceRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilNotebookInstanceDeleted(
                DescribeNotebookInstanceRequest.builder().applyMutation(describeNotebookInstanceRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeNotebookInstance} API until the desired condition
     * {@code NotebookInstanceInService} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeNotebookInstanceRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceInService(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeNotebookInstance} API until the desired condition
     * {@code NotebookInstanceInService} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeNotebookInstanceRequest#builder()}
     *
     * @param describeNotebookInstanceRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceInService(
            Consumer<DescribeNotebookInstanceRequest.Builder> describeNotebookInstanceRequest) {
        return waitUntilNotebookInstanceInService(DescribeNotebookInstanceRequest.builder()
                .applyMutation(describeNotebookInstanceRequest).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeNotebookInstance} API until the desired condition
     * {@code NotebookInstanceInService} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeNotebookInstanceRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceInService(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeNotebookInstance} API until the desired condition
     * {@code NotebookInstanceInService} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeNotebookInstanceRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceInService(
            Consumer<DescribeNotebookInstanceRequest.Builder> describeNotebookInstanceRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilNotebookInstanceInService(
                DescribeNotebookInstanceRequest.builder().applyMutation(describeNotebookInstanceRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeNotebookInstance} API until the desired condition
     * {@code NotebookInstanceStopped} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeNotebookInstanceRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceStopped(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeNotebookInstance} API until the desired condition
     * {@code NotebookInstanceStopped} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeNotebookInstanceRequest#builder()}
     *
     * @param describeNotebookInstanceRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceStopped(
            Consumer<DescribeNotebookInstanceRequest.Builder> describeNotebookInstanceRequest) {
        return waitUntilNotebookInstanceStopped(DescribeNotebookInstanceRequest.builder()
                .applyMutation(describeNotebookInstanceRequest).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeNotebookInstance} API until the desired condition
     * {@code NotebookInstanceStopped} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeNotebookInstanceRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceStopped(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeNotebookInstance} API until the desired condition
     * {@code NotebookInstanceStopped} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeNotebookInstanceRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceStopped(
            Consumer<DescribeNotebookInstanceRequest.Builder> describeNotebookInstanceRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilNotebookInstanceStopped(
                DescribeNotebookInstanceRequest.builder().applyMutation(describeNotebookInstanceRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeProcessingJob} API until the desired condition
     * {@code ProcessingJobCompletedOrStopped} is met, or until it is determined that the resource will never enter into
     * the desired state
     *
     * @param describeProcessingJobRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeProcessingJobResponse>> waitUntilProcessingJobCompletedOrStopped(
            DescribeProcessingJobRequest describeProcessingJobRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeProcessingJob} API until the desired condition
     * {@code ProcessingJobCompletedOrStopped} is met, or until it is determined that the resource will never enter into
     * the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeProcessingJobRequest#builder()}
     *
     * @param describeProcessingJobRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeProcessingJobResponse>> waitUntilProcessingJobCompletedOrStopped(
            Consumer<DescribeProcessingJobRequest.Builder> describeProcessingJobRequest) {
        return waitUntilProcessingJobCompletedOrStopped(DescribeProcessingJobRequest.builder()
                .applyMutation(describeProcessingJobRequest).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeProcessingJob} API until the desired condition
     * {@code ProcessingJobCompletedOrStopped} is met, or until it is determined that the resource will never enter into
     * the desired state
     *
     * @param describeProcessingJobRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeProcessingJobResponse>> waitUntilProcessingJobCompletedOrStopped(
            DescribeProcessingJobRequest describeProcessingJobRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeProcessingJob} API until the desired condition
     * {@code ProcessingJobCompletedOrStopped} is met, or until it is determined that the resource will never enter into
     * the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeProcessingJobRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeProcessingJobResponse>> waitUntilProcessingJobCompletedOrStopped(
            Consumer<DescribeProcessingJobRequest.Builder> describeProcessingJobRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilProcessingJobCompletedOrStopped(
                DescribeProcessingJobRequest.builder().applyMutation(describeProcessingJobRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeTrainingJob} API until the desired condition
     * {@code TrainingJobCompletedOrStopped} is met, or until it is determined that the resource will never enter into
     * the desired state
     *
     * @param describeTrainingJobRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeTrainingJobResponse>> waitUntilTrainingJobCompletedOrStopped(
            DescribeTrainingJobRequest describeTrainingJobRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeTrainingJob} API until the desired condition
     * {@code TrainingJobCompletedOrStopped} is met, or until it is determined that the resource will never enter into
     * the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeTrainingJobRequest#builder()}
     *
     * @param describeTrainingJobRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeTrainingJobResponse>> waitUntilTrainingJobCompletedOrStopped(
            Consumer<DescribeTrainingJobRequest.Builder> describeTrainingJobRequest) {
        return waitUntilTrainingJobCompletedOrStopped(DescribeTrainingJobRequest.builder()
                .applyMutation(describeTrainingJobRequest).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeTrainingJob} API until the desired condition
     * {@code TrainingJobCompletedOrStopped} is met, or until it is determined that the resource will never enter into
     * the desired state
     *
     * @param describeTrainingJobRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeTrainingJobResponse>> waitUntilTrainingJobCompletedOrStopped(
            DescribeTrainingJobRequest describeTrainingJobRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeTrainingJob} API until the desired condition
     * {@code TrainingJobCompletedOrStopped} is met, or until it is determined that the resource will never enter into
     * the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeTrainingJobRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeTrainingJobResponse>> waitUntilTrainingJobCompletedOrStopped(
            Consumer<DescribeTrainingJobRequest.Builder> describeTrainingJobRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilTrainingJobCompletedOrStopped(
                DescribeTrainingJobRequest.builder().applyMutation(describeTrainingJobRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeTransformJob} API until the desired condition
     * {@code TransformJobCompletedOrStopped} is met, or until it is determined that the resource will never enter into
     * the desired state
     *
     * @param describeTransformJobRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeTransformJobResponse>> waitUntilTransformJobCompletedOrStopped(
            DescribeTransformJobRequest describeTransformJobRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeTransformJob} API until the desired condition
     * {@code TransformJobCompletedOrStopped} is met, or until it is determined that the resource will never enter into
     * the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeTransformJobRequest#builder()}
     *
     * @param describeTransformJobRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeTransformJobResponse>> waitUntilTransformJobCompletedOrStopped(
            Consumer<DescribeTransformJobRequest.Builder> describeTransformJobRequest) {
        return waitUntilTransformJobCompletedOrStopped(DescribeTransformJobRequest.builder()
                .applyMutation(describeTransformJobRequest).build());
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeTransformJob} API until the desired condition
     * {@code TransformJobCompletedOrStopped} is met, or until it is determined that the resource will never enter into
     * the desired state
     *
     * @param describeTransformJobRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeTransformJobResponse>> waitUntilTransformJobCompletedOrStopped(
            DescribeTransformJobRequest describeTransformJobRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SageMakerAsyncClient#describeTransformJob} API until the desired condition
     * {@code TransformJobCompletedOrStopped} is met, or until it is determined that the resource will never enter into
     * the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeTransformJobRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeTransformJobResponse>> waitUntilTransformJobCompletedOrStopped(
            Consumer<DescribeTransformJobRequest.Builder> describeTransformJobRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilTransformJobCompletedOrStopped(
                DescribeTransformJobRequest.builder().applyMutation(describeTransformJobRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link SageMakerAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultSageMakerAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link SageMakerAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link SageMakerAsyncClient} will be created to poll resources. It is recommended to share a single
     * instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link SageMakerAsyncWaiter}
     */
    static SageMakerAsyncWaiter create() {
        return DefaultSageMakerAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link SageMakerAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(SageMakerAsyncClient client);

        /**
         * Builds an instance of {@link SageMakerAsyncWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link SageMakerAsyncWaiter}
         */
        SageMakerAsyncWaiter build();
    }
}
