/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The collection of settings used by an AutoML job V2 for the time-series forecasting problem type.
 * </p>
 * <note>
 * <p>
 * The <code>TimeSeriesForecastingJobConfig</code> problem type is only available in private beta. Contact Amazon Web
 * Services Support or your account manager to learn more about access privileges.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TimeSeriesForecastingJobConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<TimeSeriesForecastingJobConfig.Builder, TimeSeriesForecastingJobConfig> {
    private static final SdkField<String> FEATURE_SPECIFICATION_S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FeatureSpecificationS3Uri").getter(getter(TimeSeriesForecastingJobConfig::featureSpecificationS3Uri))
            .setter(setter(Builder::featureSpecificationS3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeatureSpecificationS3Uri").build())
            .build();

    private static final SdkField<AutoMLJobCompletionCriteria> COMPLETION_CRITERIA_FIELD = SdkField
            .<AutoMLJobCompletionCriteria> builder(MarshallingType.SDK_POJO).memberName("CompletionCriteria")
            .getter(getter(TimeSeriesForecastingJobConfig::completionCriteria)).setter(setter(Builder::completionCriteria))
            .constructor(AutoMLJobCompletionCriteria::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionCriteria").build())
            .build();

    private static final SdkField<String> FORECAST_FREQUENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ForecastFrequency").getter(getter(TimeSeriesForecastingJobConfig::forecastFrequency))
            .setter(setter(Builder::forecastFrequency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForecastFrequency").build()).build();

    private static final SdkField<Integer> FORECAST_HORIZON_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ForecastHorizon").getter(getter(TimeSeriesForecastingJobConfig::forecastHorizon))
            .setter(setter(Builder::forecastHorizon))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForecastHorizon").build()).build();

    private static final SdkField<List<String>> FORECAST_QUANTILES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ForecastQuantiles")
            .getter(getter(TimeSeriesForecastingJobConfig::forecastQuantiles))
            .setter(setter(Builder::forecastQuantiles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForecastQuantiles").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<TimeSeriesTransformations> TRANSFORMATIONS_FIELD = SdkField
            .<TimeSeriesTransformations> builder(MarshallingType.SDK_POJO).memberName("Transformations")
            .getter(getter(TimeSeriesForecastingJobConfig::transformations)).setter(setter(Builder::transformations))
            .constructor(TimeSeriesTransformations::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Transformations").build()).build();

    private static final SdkField<TimeSeriesConfig> TIME_SERIES_CONFIG_FIELD = SdkField
            .<TimeSeriesConfig> builder(MarshallingType.SDK_POJO).memberName("TimeSeriesConfig")
            .getter(getter(TimeSeriesForecastingJobConfig::timeSeriesConfig)).setter(setter(Builder::timeSeriesConfig))
            .constructor(TimeSeriesConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeSeriesConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            FEATURE_SPECIFICATION_S3_URI_FIELD, COMPLETION_CRITERIA_FIELD, FORECAST_FREQUENCY_FIELD, FORECAST_HORIZON_FIELD,
            FORECAST_QUANTILES_FIELD, TRANSFORMATIONS_FIELD, TIME_SERIES_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String featureSpecificationS3Uri;

    private final AutoMLJobCompletionCriteria completionCriteria;

    private final String forecastFrequency;

    private final Integer forecastHorizon;

    private final List<String> forecastQuantiles;

    private final TimeSeriesTransformations transformations;

    private final TimeSeriesConfig timeSeriesConfig;

    private TimeSeriesForecastingJobConfig(BuilderImpl builder) {
        this.featureSpecificationS3Uri = builder.featureSpecificationS3Uri;
        this.completionCriteria = builder.completionCriteria;
        this.forecastFrequency = builder.forecastFrequency;
        this.forecastHorizon = builder.forecastHorizon;
        this.forecastQuantiles = builder.forecastQuantiles;
        this.transformations = builder.transformations;
        this.timeSeriesConfig = builder.timeSeriesConfig;
    }

    /**
     * <p>
     * A URL to the Amazon S3 data source containing additional selected features that complement the target, itemID,
     * timestamp, and grouped columns set in <code>TimeSeriesConfig</code>. When not provided, the AutoML job V2
     * includes all the columns from the original dataset that are not already declared in <code>TimeSeriesConfig</code>
     * . If provided, the AutoML job V2 only considers these additional columns as a complement to the ones declared in
     * <code>TimeSeriesConfig</code>.
     * </p>
     * <p>
     * You can input <code>FeatureAttributeNames</code> (optional) in JSON format as shown below:
     * </p>
     * <p>
     * <code>{ "FeatureAttributeNames":["col1", "col2", ...] }</code>.
     * </p>
     * <p>
     * You can also specify the data type of the feature (optional) in the format shown below:
     * </p>
     * <p>
     * <code>{ "FeatureDataTypes":{"col1":"numeric", "col2":"categorical" ... } }</code>
     * </p>
     * <p>
     * Autopilot supports the following data types: <code>numeric</code>, <code>categorical</code>, <code>text</code>,
     * and <code>datetime</code>.
     * </p>
     * <note>
     * <p>
     * These column keys must not include any column set in <code>TimeSeriesConfig</code>.
     * </p>
     * </note>
     * <p>
     * When not provided, the AutoML job V2 includes all the columns from the original dataset that are not already
     * declared in <code>TimeSeriesConfig</code>. If provided, the AutoML job V2 only considers these additional columns
     * as a complement to the ones declared in <code>TimeSeriesConfig</code>.
     * </p>
     * <p>
     * Autopilot supports the following data types: <code>numeric</code>, <code>categorical</code>, <code>text</code>,
     * and <code>datetime</code>.
     * </p>
     * 
     * @return A URL to the Amazon S3 data source containing additional selected features that complement the target,
     *         itemID, timestamp, and grouped columns set in <code>TimeSeriesConfig</code>. When not provided, the
     *         AutoML job V2 includes all the columns from the original dataset that are not already declared in
     *         <code>TimeSeriesConfig</code>. If provided, the AutoML job V2 only considers these additional columns as
     *         a complement to the ones declared in <code>TimeSeriesConfig</code>.</p>
     *         <p>
     *         You can input <code>FeatureAttributeNames</code> (optional) in JSON format as shown below:
     *         </p>
     *         <p>
     *         <code>{ "FeatureAttributeNames":["col1", "col2", ...] }</code>.
     *         </p>
     *         <p>
     *         You can also specify the data type of the feature (optional) in the format shown below:
     *         </p>
     *         <p>
     *         <code>{ "FeatureDataTypes":{"col1":"numeric", "col2":"categorical" ... } }</code>
     *         </p>
     *         <p>
     *         Autopilot supports the following data types: <code>numeric</code>, <code>categorical</code>,
     *         <code>text</code>, and <code>datetime</code>.
     *         </p>
     *         <note>
     *         <p>
     *         These column keys must not include any column set in <code>TimeSeriesConfig</code>.
     *         </p>
     *         </note>
     *         <p>
     *         When not provided, the AutoML job V2 includes all the columns from the original dataset that are not
     *         already declared in <code>TimeSeriesConfig</code>. If provided, the AutoML job V2 only considers these
     *         additional columns as a complement to the ones declared in <code>TimeSeriesConfig</code>.
     *         </p>
     *         <p>
     *         Autopilot supports the following data types: <code>numeric</code>, <code>categorical</code>,
     *         <code>text</code>, and <code>datetime</code>.
     */
    public final String featureSpecificationS3Uri() {
        return featureSpecificationS3Uri;
    }

    /**
     * Returns the value of the CompletionCriteria property for this object.
     * 
     * @return The value of the CompletionCriteria property for this object.
     */
    public final AutoMLJobCompletionCriteria completionCriteria() {
        return completionCriteria;
    }

    /**
     * <p>
     * The frequency of predictions in a forecast.
     * </p>
     * <p>
     * Valid intervals are an integer followed by Y (Year), M (Month), W (Week), D (Day), H (Hour), and min (Minute).
     * For example, <code>1D</code> indicates every day and <code>15min</code> indicates every 15 minutes. The value of
     * a frequency must not overlap with the next larger frequency. For example, you must use a frequency of
     * <code>1H</code> instead of <code>60min</code>.
     * </p>
     * <p>
     * The valid values for each frequency are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Minute - 1-59
     * </p>
     * </li>
     * <li>
     * <p>
     * Hour - 1-23
     * </p>
     * </li>
     * <li>
     * <p>
     * Day - 1-6
     * </p>
     * </li>
     * <li>
     * <p>
     * Week - 1-4
     * </p>
     * </li>
     * <li>
     * <p>
     * Month - 1-11
     * </p>
     * </li>
     * <li>
     * <p>
     * Year - 1
     * </p>
     * </li>
     * </ul>
     * 
     * @return The frequency of predictions in a forecast.</p>
     *         <p>
     *         Valid intervals are an integer followed by Y (Year), M (Month), W (Week), D (Day), H (Hour), and min
     *         (Minute). For example, <code>1D</code> indicates every day and <code>15min</code> indicates every 15
     *         minutes. The value of a frequency must not overlap with the next larger frequency. For example, you must
     *         use a frequency of <code>1H</code> instead of <code>60min</code>.
     *         </p>
     *         <p>
     *         The valid values for each frequency are the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Minute - 1-59
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Hour - 1-23
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Day - 1-6
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Week - 1-4
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Month - 1-11
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Year - 1
     *         </p>
     *         </li>
     */
    public final String forecastFrequency() {
        return forecastFrequency;
    }

    /**
     * <p>
     * The number of time-steps that the model predicts. The forecast horizon is also called the prediction length. The
     * maximum forecast horizon is the lesser of 500 time-steps or 1/4 of the time-steps in the dataset.
     * </p>
     * 
     * @return The number of time-steps that the model predicts. The forecast horizon is also called the prediction
     *         length. The maximum forecast horizon is the lesser of 500 time-steps or 1/4 of the time-steps in the
     *         dataset.
     */
    public final Integer forecastHorizon() {
        return forecastHorizon;
    }

    /**
     * For responses, this returns true if the service returned a value for the ForecastQuantiles property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasForecastQuantiles() {
        return forecastQuantiles != null && !(forecastQuantiles instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The quantiles used to train the model for forecasts at a specified quantile. You can specify quantiles from
     * <code>0.01</code> (p1) to <code>0.99</code> (p99), by increments of 0.01 or higher. Up to five forecast quantiles
     * can be specified. When <code>ForecastQuantiles</code> is not provided, the AutoML job uses the quantiles p10,
     * p50, and p90 as default.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasForecastQuantiles} method.
     * </p>
     * 
     * @return The quantiles used to train the model for forecasts at a specified quantile. You can specify quantiles
     *         from <code>0.01</code> (p1) to <code>0.99</code> (p99), by increments of 0.01 or higher. Up to five
     *         forecast quantiles can be specified. When <code>ForecastQuantiles</code> is not provided, the AutoML job
     *         uses the quantiles p10, p50, and p90 as default.
     */
    public final List<String> forecastQuantiles() {
        return forecastQuantiles;
    }

    /**
     * <p>
     * The transformations modifying specific attributes of the time-series, such as filling strategies for missing
     * values.
     * </p>
     * 
     * @return The transformations modifying specific attributes of the time-series, such as filling strategies for
     *         missing values.
     */
    public final TimeSeriesTransformations transformations() {
        return transformations;
    }

    /**
     * <p>
     * The collection of components that defines the time-series.
     * </p>
     * 
     * @return The collection of components that defines the time-series.
     */
    public final TimeSeriesConfig timeSeriesConfig() {
        return timeSeriesConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(featureSpecificationS3Uri());
        hashCode = 31 * hashCode + Objects.hashCode(completionCriteria());
        hashCode = 31 * hashCode + Objects.hashCode(forecastFrequency());
        hashCode = 31 * hashCode + Objects.hashCode(forecastHorizon());
        hashCode = 31 * hashCode + Objects.hashCode(hasForecastQuantiles() ? forecastQuantiles() : null);
        hashCode = 31 * hashCode + Objects.hashCode(transformations());
        hashCode = 31 * hashCode + Objects.hashCode(timeSeriesConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TimeSeriesForecastingJobConfig)) {
            return false;
        }
        TimeSeriesForecastingJobConfig other = (TimeSeriesForecastingJobConfig) obj;
        return Objects.equals(featureSpecificationS3Uri(), other.featureSpecificationS3Uri())
                && Objects.equals(completionCriteria(), other.completionCriteria())
                && Objects.equals(forecastFrequency(), other.forecastFrequency())
                && Objects.equals(forecastHorizon(), other.forecastHorizon())
                && hasForecastQuantiles() == other.hasForecastQuantiles()
                && Objects.equals(forecastQuantiles(), other.forecastQuantiles())
                && Objects.equals(transformations(), other.transformations())
                && Objects.equals(timeSeriesConfig(), other.timeSeriesConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TimeSeriesForecastingJobConfig").add("FeatureSpecificationS3Uri", featureSpecificationS3Uri())
                .add("CompletionCriteria", completionCriteria()).add("ForecastFrequency", forecastFrequency())
                .add("ForecastHorizon", forecastHorizon())
                .add("ForecastQuantiles", hasForecastQuantiles() ? forecastQuantiles() : null)
                .add("Transformations", transformations()).add("TimeSeriesConfig", timeSeriesConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FeatureSpecificationS3Uri":
            return Optional.ofNullable(clazz.cast(featureSpecificationS3Uri()));
        case "CompletionCriteria":
            return Optional.ofNullable(clazz.cast(completionCriteria()));
        case "ForecastFrequency":
            return Optional.ofNullable(clazz.cast(forecastFrequency()));
        case "ForecastHorizon":
            return Optional.ofNullable(clazz.cast(forecastHorizon()));
        case "ForecastQuantiles":
            return Optional.ofNullable(clazz.cast(forecastQuantiles()));
        case "Transformations":
            return Optional.ofNullable(clazz.cast(transformations()));
        case "TimeSeriesConfig":
            return Optional.ofNullable(clazz.cast(timeSeriesConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TimeSeriesForecastingJobConfig, T> g) {
        return obj -> g.apply((TimeSeriesForecastingJobConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TimeSeriesForecastingJobConfig> {
        /**
         * <p>
         * A URL to the Amazon S3 data source containing additional selected features that complement the target,
         * itemID, timestamp, and grouped columns set in <code>TimeSeriesConfig</code>. When not provided, the AutoML
         * job V2 includes all the columns from the original dataset that are not already declared in
         * <code>TimeSeriesConfig</code>. If provided, the AutoML job V2 only considers these additional columns as a
         * complement to the ones declared in <code>TimeSeriesConfig</code>.
         * </p>
         * <p>
         * You can input <code>FeatureAttributeNames</code> (optional) in JSON format as shown below:
         * </p>
         * <p>
         * <code>{ "FeatureAttributeNames":["col1", "col2", ...] }</code>.
         * </p>
         * <p>
         * You can also specify the data type of the feature (optional) in the format shown below:
         * </p>
         * <p>
         * <code>{ "FeatureDataTypes":{"col1":"numeric", "col2":"categorical" ... } }</code>
         * </p>
         * <p>
         * Autopilot supports the following data types: <code>numeric</code>, <code>categorical</code>,
         * <code>text</code>, and <code>datetime</code>.
         * </p>
         * <note>
         * <p>
         * These column keys must not include any column set in <code>TimeSeriesConfig</code>.
         * </p>
         * </note>
         * <p>
         * When not provided, the AutoML job V2 includes all the columns from the original dataset that are not already
         * declared in <code>TimeSeriesConfig</code>. If provided, the AutoML job V2 only considers these additional
         * columns as a complement to the ones declared in <code>TimeSeriesConfig</code>.
         * </p>
         * <p>
         * Autopilot supports the following data types: <code>numeric</code>, <code>categorical</code>,
         * <code>text</code>, and <code>datetime</code>.
         * </p>
         * 
         * @param featureSpecificationS3Uri
         *        A URL to the Amazon S3 data source containing additional selected features that complement the target,
         *        itemID, timestamp, and grouped columns set in <code>TimeSeriesConfig</code>. When not provided, the
         *        AutoML job V2 includes all the columns from the original dataset that are not already declared in
         *        <code>TimeSeriesConfig</code>. If provided, the AutoML job V2 only considers these additional columns
         *        as a complement to the ones declared in <code>TimeSeriesConfig</code>.</p>
         *        <p>
         *        You can input <code>FeatureAttributeNames</code> (optional) in JSON format as shown below:
         *        </p>
         *        <p>
         *        <code>{ "FeatureAttributeNames":["col1", "col2", ...] }</code>.
         *        </p>
         *        <p>
         *        You can also specify the data type of the feature (optional) in the format shown below:
         *        </p>
         *        <p>
         *        <code>{ "FeatureDataTypes":{"col1":"numeric", "col2":"categorical" ... } }</code>
         *        </p>
         *        <p>
         *        Autopilot supports the following data types: <code>numeric</code>, <code>categorical</code>,
         *        <code>text</code>, and <code>datetime</code>.
         *        </p>
         *        <note>
         *        <p>
         *        These column keys must not include any column set in <code>TimeSeriesConfig</code>.
         *        </p>
         *        </note>
         *        <p>
         *        When not provided, the AutoML job V2 includes all the columns from the original dataset that are not
         *        already declared in <code>TimeSeriesConfig</code>. If provided, the AutoML job V2 only considers these
         *        additional columns as a complement to the ones declared in <code>TimeSeriesConfig</code>.
         *        </p>
         *        <p>
         *        Autopilot supports the following data types: <code>numeric</code>, <code>categorical</code>,
         *        <code>text</code>, and <code>datetime</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureSpecificationS3Uri(String featureSpecificationS3Uri);

        /**
         * Sets the value of the CompletionCriteria property for this object.
         *
         * @param completionCriteria
         *        The new value for the CompletionCriteria property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionCriteria(AutoMLJobCompletionCriteria completionCriteria);

        /**
         * Sets the value of the CompletionCriteria property for this object.
         *
         * This is a convenience method that creates an instance of the {@link AutoMLJobCompletionCriteria.Builder}
         * avoiding the need to create one manually via {@link AutoMLJobCompletionCriteria#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoMLJobCompletionCriteria.Builder#build()} is called
         * immediately and its result is passed to {@link #completionCriteria(AutoMLJobCompletionCriteria)}.
         * 
         * @param completionCriteria
         *        a consumer that will call methods on {@link AutoMLJobCompletionCriteria.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #completionCriteria(AutoMLJobCompletionCriteria)
         */
        default Builder completionCriteria(Consumer<AutoMLJobCompletionCriteria.Builder> completionCriteria) {
            return completionCriteria(AutoMLJobCompletionCriteria.builder().applyMutation(completionCriteria).build());
        }

        /**
         * <p>
         * The frequency of predictions in a forecast.
         * </p>
         * <p>
         * Valid intervals are an integer followed by Y (Year), M (Month), W (Week), D (Day), H (Hour), and min
         * (Minute). For example, <code>1D</code> indicates every day and <code>15min</code> indicates every 15 minutes.
         * The value of a frequency must not overlap with the next larger frequency. For example, you must use a
         * frequency of <code>1H</code> instead of <code>60min</code>.
         * </p>
         * <p>
         * The valid values for each frequency are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Minute - 1-59
         * </p>
         * </li>
         * <li>
         * <p>
         * Hour - 1-23
         * </p>
         * </li>
         * <li>
         * <p>
         * Day - 1-6
         * </p>
         * </li>
         * <li>
         * <p>
         * Week - 1-4
         * </p>
         * </li>
         * <li>
         * <p>
         * Month - 1-11
         * </p>
         * </li>
         * <li>
         * <p>
         * Year - 1
         * </p>
         * </li>
         * </ul>
         * 
         * @param forecastFrequency
         *        The frequency of predictions in a forecast.</p>
         *        <p>
         *        Valid intervals are an integer followed by Y (Year), M (Month), W (Week), D (Day), H (Hour), and min
         *        (Minute). For example, <code>1D</code> indicates every day and <code>15min</code> indicates every 15
         *        minutes. The value of a frequency must not overlap with the next larger frequency. For example, you
         *        must use a frequency of <code>1H</code> instead of <code>60min</code>.
         *        </p>
         *        <p>
         *        The valid values for each frequency are the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Minute - 1-59
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Hour - 1-23
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Day - 1-6
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Week - 1-4
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Month - 1-11
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Year - 1
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastFrequency(String forecastFrequency);

        /**
         * <p>
         * The number of time-steps that the model predicts. The forecast horizon is also called the prediction length.
         * The maximum forecast horizon is the lesser of 500 time-steps or 1/4 of the time-steps in the dataset.
         * </p>
         * 
         * @param forecastHorizon
         *        The number of time-steps that the model predicts. The forecast horizon is also called the prediction
         *        length. The maximum forecast horizon is the lesser of 500 time-steps or 1/4 of the time-steps in the
         *        dataset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastHorizon(Integer forecastHorizon);

        /**
         * <p>
         * The quantiles used to train the model for forecasts at a specified quantile. You can specify quantiles from
         * <code>0.01</code> (p1) to <code>0.99</code> (p99), by increments of 0.01 or higher. Up to five forecast
         * quantiles can be specified. When <code>ForecastQuantiles</code> is not provided, the AutoML job uses the
         * quantiles p10, p50, and p90 as default.
         * </p>
         * 
         * @param forecastQuantiles
         *        The quantiles used to train the model for forecasts at a specified quantile. You can specify quantiles
         *        from <code>0.01</code> (p1) to <code>0.99</code> (p99), by increments of 0.01 or higher. Up to five
         *        forecast quantiles can be specified. When <code>ForecastQuantiles</code> is not provided, the AutoML
         *        job uses the quantiles p10, p50, and p90 as default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastQuantiles(Collection<String> forecastQuantiles);

        /**
         * <p>
         * The quantiles used to train the model for forecasts at a specified quantile. You can specify quantiles from
         * <code>0.01</code> (p1) to <code>0.99</code> (p99), by increments of 0.01 or higher. Up to five forecast
         * quantiles can be specified. When <code>ForecastQuantiles</code> is not provided, the AutoML job uses the
         * quantiles p10, p50, and p90 as default.
         * </p>
         * 
         * @param forecastQuantiles
         *        The quantiles used to train the model for forecasts at a specified quantile. You can specify quantiles
         *        from <code>0.01</code> (p1) to <code>0.99</code> (p99), by increments of 0.01 or higher. Up to five
         *        forecast quantiles can be specified. When <code>ForecastQuantiles</code> is not provided, the AutoML
         *        job uses the quantiles p10, p50, and p90 as default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastQuantiles(String... forecastQuantiles);

        /**
         * <p>
         * The transformations modifying specific attributes of the time-series, such as filling strategies for missing
         * values.
         * </p>
         * 
         * @param transformations
         *        The transformations modifying specific attributes of the time-series, such as filling strategies for
         *        missing values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transformations(TimeSeriesTransformations transformations);

        /**
         * <p>
         * The transformations modifying specific attributes of the time-series, such as filling strategies for missing
         * values.
         * </p>
         * This is a convenience method that creates an instance of the {@link TimeSeriesTransformations.Builder}
         * avoiding the need to create one manually via {@link TimeSeriesTransformations#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TimeSeriesTransformations.Builder#build()} is called immediately
         * and its result is passed to {@link #transformations(TimeSeriesTransformations)}.
         * 
         * @param transformations
         *        a consumer that will call methods on {@link TimeSeriesTransformations.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transformations(TimeSeriesTransformations)
         */
        default Builder transformations(Consumer<TimeSeriesTransformations.Builder> transformations) {
            return transformations(TimeSeriesTransformations.builder().applyMutation(transformations).build());
        }

        /**
         * <p>
         * The collection of components that defines the time-series.
         * </p>
         * 
         * @param timeSeriesConfig
         *        The collection of components that defines the time-series.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeSeriesConfig(TimeSeriesConfig timeSeriesConfig);

        /**
         * <p>
         * The collection of components that defines the time-series.
         * </p>
         * This is a convenience method that creates an instance of the {@link TimeSeriesConfig.Builder} avoiding the
         * need to create one manually via {@link TimeSeriesConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TimeSeriesConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #timeSeriesConfig(TimeSeriesConfig)}.
         * 
         * @param timeSeriesConfig
         *        a consumer that will call methods on {@link TimeSeriesConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #timeSeriesConfig(TimeSeriesConfig)
         */
        default Builder timeSeriesConfig(Consumer<TimeSeriesConfig.Builder> timeSeriesConfig) {
            return timeSeriesConfig(TimeSeriesConfig.builder().applyMutation(timeSeriesConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String featureSpecificationS3Uri;

        private AutoMLJobCompletionCriteria completionCriteria;

        private String forecastFrequency;

        private Integer forecastHorizon;

        private List<String> forecastQuantiles = DefaultSdkAutoConstructList.getInstance();

        private TimeSeriesTransformations transformations;

        private TimeSeriesConfig timeSeriesConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(TimeSeriesForecastingJobConfig model) {
            featureSpecificationS3Uri(model.featureSpecificationS3Uri);
            completionCriteria(model.completionCriteria);
            forecastFrequency(model.forecastFrequency);
            forecastHorizon(model.forecastHorizon);
            forecastQuantiles(model.forecastQuantiles);
            transformations(model.transformations);
            timeSeriesConfig(model.timeSeriesConfig);
        }

        public final String getFeatureSpecificationS3Uri() {
            return featureSpecificationS3Uri;
        }

        public final void setFeatureSpecificationS3Uri(String featureSpecificationS3Uri) {
            this.featureSpecificationS3Uri = featureSpecificationS3Uri;
        }

        @Override
        public final Builder featureSpecificationS3Uri(String featureSpecificationS3Uri) {
            this.featureSpecificationS3Uri = featureSpecificationS3Uri;
            return this;
        }

        public final AutoMLJobCompletionCriteria.Builder getCompletionCriteria() {
            return completionCriteria != null ? completionCriteria.toBuilder() : null;
        }

        public final void setCompletionCriteria(AutoMLJobCompletionCriteria.BuilderImpl completionCriteria) {
            this.completionCriteria = completionCriteria != null ? completionCriteria.build() : null;
        }

        @Override
        public final Builder completionCriteria(AutoMLJobCompletionCriteria completionCriteria) {
            this.completionCriteria = completionCriteria;
            return this;
        }

        public final String getForecastFrequency() {
            return forecastFrequency;
        }

        public final void setForecastFrequency(String forecastFrequency) {
            this.forecastFrequency = forecastFrequency;
        }

        @Override
        public final Builder forecastFrequency(String forecastFrequency) {
            this.forecastFrequency = forecastFrequency;
            return this;
        }

        public final Integer getForecastHorizon() {
            return forecastHorizon;
        }

        public final void setForecastHorizon(Integer forecastHorizon) {
            this.forecastHorizon = forecastHorizon;
        }

        @Override
        public final Builder forecastHorizon(Integer forecastHorizon) {
            this.forecastHorizon = forecastHorizon;
            return this;
        }

        public final Collection<String> getForecastQuantiles() {
            if (forecastQuantiles instanceof SdkAutoConstructList) {
                return null;
            }
            return forecastQuantiles;
        }

        public final void setForecastQuantiles(Collection<String> forecastQuantiles) {
            this.forecastQuantiles = ForecastQuantilesCopier.copy(forecastQuantiles);
        }

        @Override
        public final Builder forecastQuantiles(Collection<String> forecastQuantiles) {
            this.forecastQuantiles = ForecastQuantilesCopier.copy(forecastQuantiles);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder forecastQuantiles(String... forecastQuantiles) {
            forecastQuantiles(Arrays.asList(forecastQuantiles));
            return this;
        }

        public final TimeSeriesTransformations.Builder getTransformations() {
            return transformations != null ? transformations.toBuilder() : null;
        }

        public final void setTransformations(TimeSeriesTransformations.BuilderImpl transformations) {
            this.transformations = transformations != null ? transformations.build() : null;
        }

        @Override
        public final Builder transformations(TimeSeriesTransformations transformations) {
            this.transformations = transformations;
            return this;
        }

        public final TimeSeriesConfig.Builder getTimeSeriesConfig() {
            return timeSeriesConfig != null ? timeSeriesConfig.toBuilder() : null;
        }

        public final void setTimeSeriesConfig(TimeSeriesConfig.BuilderImpl timeSeriesConfig) {
            this.timeSeriesConfig = timeSeriesConfig != null ? timeSeriesConfig.build() : null;
        }

        @Override
        public final Builder timeSeriesConfig(TimeSeriesConfig timeSeriesConfig) {
            this.timeSeriesConfig = timeSeriesConfig;
            return this;
        }

        @Override
        public TimeSeriesForecastingJobConfig build() {
            return new TimeSeriesForecastingJobConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
