/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a limit to how long a model training job or model compilation job can run. It also specifies how long a
 * managed spot training job has to complete. When the job reaches the time limit, SageMaker ends the training or
 * compilation job. Use this API to cap model training costs.
 * </p>
 * <p>
 * To stop a training job, SageMaker sends the algorithm the <code>SIGTERM</code> signal, which delays job termination
 * for 120 seconds. Algorithms can use this 120-second window to save the model artifacts, so the results of training
 * are not lost.
 * </p>
 * <p>
 * The training algorithms provided by SageMaker automatically save the intermediate results of a model training job
 * when possible. This attempt to save artifacts is only a best effort case as model might not be in a state from which
 * it can be saved. For example, if training has just started, the model might not be ready to save. When saved, this
 * intermediate data is a valid model artifact. You can use it to create a model with <code>CreateModel</code>.
 * </p>
 * <note>
 * <p>
 * The Neural Topic Model (NTM) currently does not support saving intermediate model artifacts. When training NTMs, make
 * sure that the maximum runtime is sufficient for the training job to complete.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StoppingCondition implements SdkPojo, Serializable,
        ToCopyableBuilder<StoppingCondition.Builder, StoppingCondition> {
    private static final SdkField<Integer> MAX_RUNTIME_IN_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxRuntimeInSeconds").getter(getter(StoppingCondition::maxRuntimeInSeconds))
            .setter(setter(Builder::maxRuntimeInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxRuntimeInSeconds").build())
            .build();

    private static final SdkField<Integer> MAX_WAIT_TIME_IN_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxWaitTimeInSeconds").getter(getter(StoppingCondition::maxWaitTimeInSeconds))
            .setter(setter(Builder::maxWaitTimeInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxWaitTimeInSeconds").build())
            .build();

    private static final SdkField<Integer> MAX_PENDING_TIME_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("MaxPendingTimeInSeconds")
            .getter(getter(StoppingCondition::maxPendingTimeInSeconds)).setter(setter(Builder::maxPendingTimeInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxPendingTimeInSeconds").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MAX_RUNTIME_IN_SECONDS_FIELD,
            MAX_WAIT_TIME_IN_SECONDS_FIELD, MAX_PENDING_TIME_IN_SECONDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer maxRuntimeInSeconds;

    private final Integer maxWaitTimeInSeconds;

    private final Integer maxPendingTimeInSeconds;

    private StoppingCondition(BuilderImpl builder) {
        this.maxRuntimeInSeconds = builder.maxRuntimeInSeconds;
        this.maxWaitTimeInSeconds = builder.maxWaitTimeInSeconds;
        this.maxPendingTimeInSeconds = builder.maxPendingTimeInSeconds;
    }

    /**
     * <p>
     * The maximum length of time, in seconds, that a training or compilation job can run before it is stopped.
     * </p>
     * <p>
     * For compilation jobs, if the job does not complete during this time, a <code>TimeOut</code> error is generated.
     * We recommend starting with 900 seconds and increasing as necessary based on your model.
     * </p>
     * <p>
     * For all other jobs, if the job does not complete during this time, SageMaker ends the job. When
     * <code>RetryStrategy</code> is specified in the job request, <code>MaxRuntimeInSeconds</code> specifies the
     * maximum time for all of the attempts in total, not each individual attempt. The default value is 1 day. The
     * maximum value is 28 days.
     * </p>
     * <p>
     * The maximum time that a <code>TrainingJob</code> can run in total, including any time spent publishing metrics or
     * archiving and uploading models after it has been stopped, is 30 days.
     * </p>
     * 
     * @return The maximum length of time, in seconds, that a training or compilation job can run before it is
     *         stopped.</p>
     *         <p>
     *         For compilation jobs, if the job does not complete during this time, a <code>TimeOut</code> error is
     *         generated. We recommend starting with 900 seconds and increasing as necessary based on your model.
     *         </p>
     *         <p>
     *         For all other jobs, if the job does not complete during this time, SageMaker ends the job. When
     *         <code>RetryStrategy</code> is specified in the job request, <code>MaxRuntimeInSeconds</code> specifies
     *         the maximum time for all of the attempts in total, not each individual attempt. The default value is 1
     *         day. The maximum value is 28 days.
     *         </p>
     *         <p>
     *         The maximum time that a <code>TrainingJob</code> can run in total, including any time spent publishing
     *         metrics or archiving and uploading models after it has been stopped, is 30 days.
     */
    public final Integer maxRuntimeInSeconds() {
        return maxRuntimeInSeconds;
    }

    /**
     * <p>
     * The maximum length of time, in seconds, that a managed Spot training job has to complete. It is the amount of
     * time spent waiting for Spot capacity plus the amount of time the job can run. It must be equal to or greater than
     * <code>MaxRuntimeInSeconds</code>. If the job does not complete during this time, SageMaker ends the job.
     * </p>
     * <p>
     * When <code>RetryStrategy</code> is specified in the job request, <code>MaxWaitTimeInSeconds</code> specifies the
     * maximum time for all of the attempts in total, not each individual attempt.
     * </p>
     * 
     * @return The maximum length of time, in seconds, that a managed Spot training job has to complete. It is the
     *         amount of time spent waiting for Spot capacity plus the amount of time the job can run. It must be equal
     *         to or greater than <code>MaxRuntimeInSeconds</code>. If the job does not complete during this time,
     *         SageMaker ends the job.</p>
     *         <p>
     *         When <code>RetryStrategy</code> is specified in the job request, <code>MaxWaitTimeInSeconds</code>
     *         specifies the maximum time for all of the attempts in total, not each individual attempt.
     */
    public final Integer maxWaitTimeInSeconds() {
        return maxWaitTimeInSeconds;
    }

    /**
     * <p>
     * The maximum pending time in seconds.
     * </p>
     * 
     * @return The maximum pending time in seconds.
     */
    public final Integer maxPendingTimeInSeconds() {
        return maxPendingTimeInSeconds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(maxRuntimeInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(maxWaitTimeInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(maxPendingTimeInSeconds());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StoppingCondition)) {
            return false;
        }
        StoppingCondition other = (StoppingCondition) obj;
        return Objects.equals(maxRuntimeInSeconds(), other.maxRuntimeInSeconds())
                && Objects.equals(maxWaitTimeInSeconds(), other.maxWaitTimeInSeconds())
                && Objects.equals(maxPendingTimeInSeconds(), other.maxPendingTimeInSeconds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StoppingCondition").add("MaxRuntimeInSeconds", maxRuntimeInSeconds())
                .add("MaxWaitTimeInSeconds", maxWaitTimeInSeconds()).add("MaxPendingTimeInSeconds", maxPendingTimeInSeconds())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MaxRuntimeInSeconds":
            return Optional.ofNullable(clazz.cast(maxRuntimeInSeconds()));
        case "MaxWaitTimeInSeconds":
            return Optional.ofNullable(clazz.cast(maxWaitTimeInSeconds()));
        case "MaxPendingTimeInSeconds":
            return Optional.ofNullable(clazz.cast(maxPendingTimeInSeconds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StoppingCondition, T> g) {
        return obj -> g.apply((StoppingCondition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StoppingCondition> {
        /**
         * <p>
         * The maximum length of time, in seconds, that a training or compilation job can run before it is stopped.
         * </p>
         * <p>
         * For compilation jobs, if the job does not complete during this time, a <code>TimeOut</code> error is
         * generated. We recommend starting with 900 seconds and increasing as necessary based on your model.
         * </p>
         * <p>
         * For all other jobs, if the job does not complete during this time, SageMaker ends the job. When
         * <code>RetryStrategy</code> is specified in the job request, <code>MaxRuntimeInSeconds</code> specifies the
         * maximum time for all of the attempts in total, not each individual attempt. The default value is 1 day. The
         * maximum value is 28 days.
         * </p>
         * <p>
         * The maximum time that a <code>TrainingJob</code> can run in total, including any time spent publishing
         * metrics or archiving and uploading models after it has been stopped, is 30 days.
         * </p>
         * 
         * @param maxRuntimeInSeconds
         *        The maximum length of time, in seconds, that a training or compilation job can run before it is
         *        stopped.</p>
         *        <p>
         *        For compilation jobs, if the job does not complete during this time, a <code>TimeOut</code> error is
         *        generated. We recommend starting with 900 seconds and increasing as necessary based on your model.
         *        </p>
         *        <p>
         *        For all other jobs, if the job does not complete during this time, SageMaker ends the job. When
         *        <code>RetryStrategy</code> is specified in the job request, <code>MaxRuntimeInSeconds</code> specifies
         *        the maximum time for all of the attempts in total, not each individual attempt. The default value is 1
         *        day. The maximum value is 28 days.
         *        </p>
         *        <p>
         *        The maximum time that a <code>TrainingJob</code> can run in total, including any time spent publishing
         *        metrics or archiving and uploading models after it has been stopped, is 30 days.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxRuntimeInSeconds(Integer maxRuntimeInSeconds);

        /**
         * <p>
         * The maximum length of time, in seconds, that a managed Spot training job has to complete. It is the amount of
         * time spent waiting for Spot capacity plus the amount of time the job can run. It must be equal to or greater
         * than <code>MaxRuntimeInSeconds</code>. If the job does not complete during this time, SageMaker ends the job.
         * </p>
         * <p>
         * When <code>RetryStrategy</code> is specified in the job request, <code>MaxWaitTimeInSeconds</code> specifies
         * the maximum time for all of the attempts in total, not each individual attempt.
         * </p>
         * 
         * @param maxWaitTimeInSeconds
         *        The maximum length of time, in seconds, that a managed Spot training job has to complete. It is the
         *        amount of time spent waiting for Spot capacity plus the amount of time the job can run. It must be
         *        equal to or greater than <code>MaxRuntimeInSeconds</code>. If the job does not complete during this
         *        time, SageMaker ends the job.</p>
         *        <p>
         *        When <code>RetryStrategy</code> is specified in the job request, <code>MaxWaitTimeInSeconds</code>
         *        specifies the maximum time for all of the attempts in total, not each individual attempt.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxWaitTimeInSeconds(Integer maxWaitTimeInSeconds);

        /**
         * <p>
         * The maximum pending time in seconds.
         * </p>
         * 
         * @param maxPendingTimeInSeconds
         *        The maximum pending time in seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxPendingTimeInSeconds(Integer maxPendingTimeInSeconds);
    }

    static final class BuilderImpl implements Builder {
        private Integer maxRuntimeInSeconds;

        private Integer maxWaitTimeInSeconds;

        private Integer maxPendingTimeInSeconds;

        private BuilderImpl() {
        }

        private BuilderImpl(StoppingCondition model) {
            maxRuntimeInSeconds(model.maxRuntimeInSeconds);
            maxWaitTimeInSeconds(model.maxWaitTimeInSeconds);
            maxPendingTimeInSeconds(model.maxPendingTimeInSeconds);
        }

        public final Integer getMaxRuntimeInSeconds() {
            return maxRuntimeInSeconds;
        }

        public final void setMaxRuntimeInSeconds(Integer maxRuntimeInSeconds) {
            this.maxRuntimeInSeconds = maxRuntimeInSeconds;
        }

        @Override
        public final Builder maxRuntimeInSeconds(Integer maxRuntimeInSeconds) {
            this.maxRuntimeInSeconds = maxRuntimeInSeconds;
            return this;
        }

        public final Integer getMaxWaitTimeInSeconds() {
            return maxWaitTimeInSeconds;
        }

        public final void setMaxWaitTimeInSeconds(Integer maxWaitTimeInSeconds) {
            this.maxWaitTimeInSeconds = maxWaitTimeInSeconds;
        }

        @Override
        public final Builder maxWaitTimeInSeconds(Integer maxWaitTimeInSeconds) {
            this.maxWaitTimeInSeconds = maxWaitTimeInSeconds;
            return this;
        }

        public final Integer getMaxPendingTimeInSeconds() {
            return maxPendingTimeInSeconds;
        }

        public final void setMaxPendingTimeInSeconds(Integer maxPendingTimeInSeconds) {
            this.maxPendingTimeInSeconds = maxPendingTimeInSeconds;
        }

        @Override
        public final Builder maxPendingTimeInSeconds(Integer maxPendingTimeInSeconds) {
            this.maxPendingTimeInSeconds = maxPendingTimeInSeconds;
            return this;
        }

        @Override
        public StoppingCondition build() {
            return new StoppingCondition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
