/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the container, as part of model definition.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerDefinition.Builder, ContainerDefinition> {
    private static final SdkField<String> CONTAINER_HOSTNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContainerHostname").getter(getter(ContainerDefinition::containerHostname))
            .setter(setter(Builder::containerHostname))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerHostname").build()).build();

    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Image")
            .getter(getter(ContainerDefinition::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Image").build()).build();

    private static final SdkField<ImageConfig> IMAGE_CONFIG_FIELD = SdkField.<ImageConfig> builder(MarshallingType.SDK_POJO)
            .memberName("ImageConfig").getter(getter(ContainerDefinition::imageConfig)).setter(setter(Builder::imageConfig))
            .constructor(ImageConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageConfig").build()).build();

    private static final SdkField<String> MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Mode")
            .getter(getter(ContainerDefinition::modeAsString)).setter(setter(Builder::mode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mode").build()).build();

    private static final SdkField<String> MODEL_DATA_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelDataUrl").getter(getter(ContainerDefinition::modelDataUrl)).setter(setter(Builder::modelDataUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelDataUrl").build()).build();

    private static final SdkField<ModelDataSource> MODEL_DATA_SOURCE_FIELD = SdkField
            .<ModelDataSource> builder(MarshallingType.SDK_POJO).memberName("ModelDataSource")
            .getter(getter(ContainerDefinition::modelDataSource)).setter(setter(Builder::modelDataSource))
            .constructor(ModelDataSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelDataSource").build()).build();

    private static final SdkField<Map<String, String>> ENVIRONMENT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Environment")
            .getter(getter(ContainerDefinition::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Environment").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> MODEL_PACKAGE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelPackageName").getter(getter(ContainerDefinition::modelPackageName))
            .setter(setter(Builder::modelPackageName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelPackageName").build()).build();

    private static final SdkField<String> INFERENCE_SPECIFICATION_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InferenceSpecificationName")
            .getter(getter(ContainerDefinition::inferenceSpecificationName))
            .setter(setter(Builder::inferenceSpecificationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InferenceSpecificationName").build())
            .build();

    private static final SdkField<MultiModelConfig> MULTI_MODEL_CONFIG_FIELD = SdkField
            .<MultiModelConfig> builder(MarshallingType.SDK_POJO).memberName("MultiModelConfig")
            .getter(getter(ContainerDefinition::multiModelConfig)).setter(setter(Builder::multiModelConfig))
            .constructor(MultiModelConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MultiModelConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_HOSTNAME_FIELD,
            IMAGE_FIELD, IMAGE_CONFIG_FIELD, MODE_FIELD, MODEL_DATA_URL_FIELD, MODEL_DATA_SOURCE_FIELD, ENVIRONMENT_FIELD,
            MODEL_PACKAGE_NAME_FIELD, INFERENCE_SPECIFICATION_NAME_FIELD, MULTI_MODEL_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String containerHostname;

    private final String image;

    private final ImageConfig imageConfig;

    private final String mode;

    private final String modelDataUrl;

    private final ModelDataSource modelDataSource;

    private final Map<String, String> environment;

    private final String modelPackageName;

    private final String inferenceSpecificationName;

    private final MultiModelConfig multiModelConfig;

    private ContainerDefinition(BuilderImpl builder) {
        this.containerHostname = builder.containerHostname;
        this.image = builder.image;
        this.imageConfig = builder.imageConfig;
        this.mode = builder.mode;
        this.modelDataUrl = builder.modelDataUrl;
        this.modelDataSource = builder.modelDataSource;
        this.environment = builder.environment;
        this.modelPackageName = builder.modelPackageName;
        this.inferenceSpecificationName = builder.inferenceSpecificationName;
        this.multiModelConfig = builder.multiModelConfig;
    }

    /**
     * <p>
     * This parameter is ignored for models that contain only a <code>PrimaryContainer</code>.
     * </p>
     * <p>
     * When a <code>ContainerDefinition</code> is part of an inference pipeline, the value of the parameter uniquely
     * identifies the container for the purposes of logging and metrics. For information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/inference-pipeline-logs-metrics.html">Use Logs and Metrics
     * to Monitor an Inference Pipeline</a>. If you don't specify a value for this parameter for a
     * <code>ContainerDefinition</code> that is part of an inference pipeline, a unique name is automatically assigned
     * based on the position of the <code>ContainerDefinition</code> in the pipeline. If you specify a value for the
     * <code>ContainerHostName</code> for any <code>ContainerDefinition</code> that is part of an inference pipeline,
     * you must specify a value for the <code>ContainerHostName</code> parameter of every
     * <code>ContainerDefinition</code> in that pipeline.
     * </p>
     * 
     * @return This parameter is ignored for models that contain only a <code>PrimaryContainer</code>.</p>
     *         <p>
     *         When a <code>ContainerDefinition</code> is part of an inference pipeline, the value of the parameter
     *         uniquely identifies the container for the purposes of logging and metrics. For information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/inference-pipeline-logs-metrics.html">Use Logs and
     *         Metrics to Monitor an Inference Pipeline</a>. If you don't specify a value for this parameter for a
     *         <code>ContainerDefinition</code> that is part of an inference pipeline, a unique name is automatically
     *         assigned based on the position of the <code>ContainerDefinition</code> in the pipeline. If you specify a
     *         value for the <code>ContainerHostName</code> for any <code>ContainerDefinition</code> that is part of an
     *         inference pipeline, you must specify a value for the <code>ContainerHostName</code> parameter of every
     *         <code>ContainerDefinition</code> in that pipeline.
     */
    public final String containerHostname() {
        return containerHostname;
    }

    /**
     * <p>
     * The path where inference code is stored. This can be either in Amazon EC2 Container Registry or in a Docker
     * registry that is accessible from the same VPC that you configure for your endpoint. If you are using your own
     * custom algorithm instead of an algorithm provided by SageMaker, the inference code must meet SageMaker
     * requirements. SageMaker supports both <code>registry/repository[:tag]</code> and
     * <code>registry/repository[@digest]</code> image path formats. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own Algorithms with Amazon
     * SageMaker</a>.
     * </p>
     * <note>
     * <p>
     * The model artifacts in an Amazon S3 bucket and the Docker image for inference container in Amazon EC2 Container
     * Registry must be in the same region as the model or endpoint you are creating.
     * </p>
     * </note>
     * 
     * @return The path where inference code is stored. This can be either in Amazon EC2 Container Registry or in a
     *         Docker registry that is accessible from the same VPC that you configure for your endpoint. If you are
     *         using your own custom algorithm instead of an algorithm provided by SageMaker, the inference code must
     *         meet SageMaker requirements. SageMaker supports both <code>registry/repository[:tag]</code> and
     *         <code>registry/repository[@digest]</code> image path formats. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own Algorithms
     *         with Amazon SageMaker</a>. </p> <note>
     *         <p>
     *         The model artifacts in an Amazon S3 bucket and the Docker image for inference container in Amazon EC2
     *         Container Registry must be in the same region as the model or endpoint you are creating.
     *         </p>
     */
    public final String image() {
        return image;
    }

    /**
     * <p>
     * Specifies whether the model container is in Amazon ECR or a private Docker registry accessible from your Amazon
     * Virtual Private Cloud (VPC). For information about storing containers in a private Docker registry, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-containers-inference-private.html">Use a
     * Private Docker Registry for Real-Time Inference Containers</a>.
     * </p>
     * <note>
     * <p>
     * The model artifacts in an Amazon S3 bucket and the Docker image for inference container in Amazon EC2 Container
     * Registry must be in the same region as the model or endpoint you are creating.
     * </p>
     * </note>
     * 
     * @return Specifies whether the model container is in Amazon ECR or a private Docker registry accessible from your
     *         Amazon Virtual Private Cloud (VPC). For information about storing containers in a private Docker
     *         registry, see <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-containers-inference-private.html">Use a
     *         Private Docker Registry for Real-Time Inference Containers</a>. </p> <note>
     *         <p>
     *         The model artifacts in an Amazon S3 bucket and the Docker image for inference container in Amazon EC2
     *         Container Registry must be in the same region as the model or endpoint you are creating.
     *         </p>
     */
    public final ImageConfig imageConfig() {
        return imageConfig;
    }

    /**
     * <p>
     * Whether the container hosts a single model or multiple models.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link ContainerMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return Whether the container hosts a single model or multiple models.
     * @see ContainerMode
     */
    public final ContainerMode mode() {
        return ContainerMode.fromValue(mode);
    }

    /**
     * <p>
     * Whether the container hosts a single model or multiple models.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link ContainerMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return Whether the container hosts a single model or multiple models.
     * @see ContainerMode
     */
    public final String modeAsString() {
        return mode;
    }

    /**
     * <p>
     * The S3 path where the model artifacts, which result from model training, are stored. This path must point to a
     * single gzip compressed tar archive (.tar.gz suffix). The S3 path is required for SageMaker built-in algorithms,
     * but not if you use your own algorithms. For more information on built-in algorithms, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-algo-docker-registry-paths.html">Common
     * Parameters</a>.
     * </p>
     * <note>
     * <p>
     * The model artifacts must be in an S3 bucket that is in the same region as the model or endpoint you are creating.
     * </p>
     * </note>
     * <p>
     * If you provide a value for this parameter, SageMaker uses Amazon Web Services Security Token Service to download
     * model artifacts from the S3 path you provide. Amazon Web Services STS is activated in your Amazon Web Services
     * account by default. If you previously deactivated Amazon Web Services STS for a region, you need to reactivate
     * Amazon Web Services STS for that region. For more information, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp_enable-regions.html">Activating and
     * Deactivating Amazon Web Services STS in an Amazon Web Services Region</a> in the <i>Amazon Web Services Identity
     * and Access Management User Guide</i>.
     * </p>
     * <important>
     * <p>
     * If you use a built-in algorithm to create a model, SageMaker requires that you provide a S3 path to the model
     * artifacts in <code>ModelDataUrl</code>.
     * </p>
     * </important>
     * 
     * @return The S3 path where the model artifacts, which result from model training, are stored. This path must point
     *         to a single gzip compressed tar archive (.tar.gz suffix). The S3 path is required for SageMaker built-in
     *         algorithms, but not if you use your own algorithms. For more information on built-in algorithms, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-algo-docker-registry-paths.html">Common
     *         Parameters</a>. </p> <note>
     *         <p>
     *         The model artifacts must be in an S3 bucket that is in the same region as the model or endpoint you are
     *         creating.
     *         </p>
     *         </note>
     *         <p>
     *         If you provide a value for this parameter, SageMaker uses Amazon Web Services Security Token Service to
     *         download model artifacts from the S3 path you provide. Amazon Web Services STS is activated in your
     *         Amazon Web Services account by default. If you previously deactivated Amazon Web Services STS for a
     *         region, you need to reactivate Amazon Web Services STS for that region. For more information, see <a
     *         href=
     *         "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp_enable-regions.html">Activating and
     *         Deactivating Amazon Web Services STS in an Amazon Web Services Region</a> in the <i>Amazon Web Services
     *         Identity and Access Management User Guide</i>.
     *         </p>
     *         <important>
     *         <p>
     *         If you use a built-in algorithm to create a model, SageMaker requires that you provide a S3 path to the
     *         model artifacts in <code>ModelDataUrl</code>.
     *         </p>
     */
    public final String modelDataUrl() {
        return modelDataUrl;
    }

    /**
     * <p>
     * Specifies the location of ML model data to deploy.
     * </p>
     * <note>
     * <p>
     * Currently you cannot use <code>ModelDataSource</code> in conjunction with SageMaker batch transform, SageMaker
     * serverless endpoints, SageMaker multi-model endpoints, and SageMaker Marketplace.
     * </p>
     * </note>
     * 
     * @return Specifies the location of ML model data to deploy.</p> <note>
     *         <p>
     *         Currently you cannot use <code>ModelDataSource</code> in conjunction with SageMaker batch transform,
     *         SageMaker serverless endpoints, SageMaker multi-model endpoints, and SageMaker Marketplace.
     *         </p>
     */
    public final ModelDataSource modelDataSource() {
        return modelDataSource;
    }

    /**
     * For responses, this returns true if the service returned a value for the Environment property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The environment variables to set in the Docker container. Each key and value in the <code>Environment</code>
     * string to string map can have length of up to 1024. We support up to 16 entries in the map.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironment} method.
     * </p>
     * 
     * @return The environment variables to set in the Docker container. Each key and value in the
     *         <code>Environment</code> string to string map can have length of up to 1024. We support up to 16 entries
     *         in the map.
     */
    public final Map<String, String> environment() {
        return environment;
    }

    /**
     * <p>
     * The name or Amazon Resource Name (ARN) of the model package to use to create the model.
     * </p>
     * 
     * @return The name or Amazon Resource Name (ARN) of the model package to use to create the model.
     */
    public final String modelPackageName() {
        return modelPackageName;
    }

    /**
     * <p>
     * The inference specification name in the model package version.
     * </p>
     * 
     * @return The inference specification name in the model package version.
     */
    public final String inferenceSpecificationName() {
        return inferenceSpecificationName;
    }

    /**
     * <p>
     * Specifies additional configuration for multi-model endpoints.
     * </p>
     * 
     * @return Specifies additional configuration for multi-model endpoints.
     */
    public final MultiModelConfig multiModelConfig() {
        return multiModelConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerHostname());
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(imageConfig());
        hashCode = 31 * hashCode + Objects.hashCode(modeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(modelDataUrl());
        hashCode = 31 * hashCode + Objects.hashCode(modelDataSource());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(modelPackageName());
        hashCode = 31 * hashCode + Objects.hashCode(inferenceSpecificationName());
        hashCode = 31 * hashCode + Objects.hashCode(multiModelConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerDefinition)) {
            return false;
        }
        ContainerDefinition other = (ContainerDefinition) obj;
        return Objects.equals(containerHostname(), other.containerHostname()) && Objects.equals(image(), other.image())
                && Objects.equals(imageConfig(), other.imageConfig()) && Objects.equals(modeAsString(), other.modeAsString())
                && Objects.equals(modelDataUrl(), other.modelDataUrl())
                && Objects.equals(modelDataSource(), other.modelDataSource()) && hasEnvironment() == other.hasEnvironment()
                && Objects.equals(environment(), other.environment())
                && Objects.equals(modelPackageName(), other.modelPackageName())
                && Objects.equals(inferenceSpecificationName(), other.inferenceSpecificationName())
                && Objects.equals(multiModelConfig(), other.multiModelConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerDefinition").add("ContainerHostname", containerHostname()).add("Image", image())
                .add("ImageConfig", imageConfig()).add("Mode", modeAsString()).add("ModelDataUrl", modelDataUrl())
                .add("ModelDataSource", modelDataSource()).add("Environment", hasEnvironment() ? environment() : null)
                .add("ModelPackageName", modelPackageName()).add("InferenceSpecificationName", inferenceSpecificationName())
                .add("MultiModelConfig", multiModelConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContainerHostname":
            return Optional.ofNullable(clazz.cast(containerHostname()));
        case "Image":
            return Optional.ofNullable(clazz.cast(image()));
        case "ImageConfig":
            return Optional.ofNullable(clazz.cast(imageConfig()));
        case "Mode":
            return Optional.ofNullable(clazz.cast(modeAsString()));
        case "ModelDataUrl":
            return Optional.ofNullable(clazz.cast(modelDataUrl()));
        case "ModelDataSource":
            return Optional.ofNullable(clazz.cast(modelDataSource()));
        case "Environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "ModelPackageName":
            return Optional.ofNullable(clazz.cast(modelPackageName()));
        case "InferenceSpecificationName":
            return Optional.ofNullable(clazz.cast(inferenceSpecificationName()));
        case "MultiModelConfig":
            return Optional.ofNullable(clazz.cast(multiModelConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerDefinition, T> g) {
        return obj -> g.apply((ContainerDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerDefinition> {
        /**
         * <p>
         * This parameter is ignored for models that contain only a <code>PrimaryContainer</code>.
         * </p>
         * <p>
         * When a <code>ContainerDefinition</code> is part of an inference pipeline, the value of the parameter uniquely
         * identifies the container for the purposes of logging and metrics. For information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/inference-pipeline-logs-metrics.html">Use Logs and
         * Metrics to Monitor an Inference Pipeline</a>. If you don't specify a value for this parameter for a
         * <code>ContainerDefinition</code> that is part of an inference pipeline, a unique name is automatically
         * assigned based on the position of the <code>ContainerDefinition</code> in the pipeline. If you specify a
         * value for the <code>ContainerHostName</code> for any <code>ContainerDefinition</code> that is part of an
         * inference pipeline, you must specify a value for the <code>ContainerHostName</code> parameter of every
         * <code>ContainerDefinition</code> in that pipeline.
         * </p>
         * 
         * @param containerHostname
         *        This parameter is ignored for models that contain only a <code>PrimaryContainer</code>.</p>
         *        <p>
         *        When a <code>ContainerDefinition</code> is part of an inference pipeline, the value of the parameter
         *        uniquely identifies the container for the purposes of logging and metrics. For information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/inference-pipeline-logs-metrics.html">Use Logs
         *        and Metrics to Monitor an Inference Pipeline</a>. If you don't specify a value for this parameter for
         *        a <code>ContainerDefinition</code> that is part of an inference pipeline, a unique name is
         *        automatically assigned based on the position of the <code>ContainerDefinition</code> in the pipeline.
         *        If you specify a value for the <code>ContainerHostName</code> for any <code>ContainerDefinition</code>
         *        that is part of an inference pipeline, you must specify a value for the <code>ContainerHostName</code>
         *        parameter of every <code>ContainerDefinition</code> in that pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerHostname(String containerHostname);

        /**
         * <p>
         * The path where inference code is stored. This can be either in Amazon EC2 Container Registry or in a Docker
         * registry that is accessible from the same VPC that you configure for your endpoint. If you are using your own
         * custom algorithm instead of an algorithm provided by SageMaker, the inference code must meet SageMaker
         * requirements. SageMaker supports both <code>registry/repository[:tag]</code> and
         * <code>registry/repository[@digest]</code> image path formats. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own Algorithms with
         * Amazon SageMaker</a>.
         * </p>
         * <note>
         * <p>
         * The model artifacts in an Amazon S3 bucket and the Docker image for inference container in Amazon EC2
         * Container Registry must be in the same region as the model or endpoint you are creating.
         * </p>
         * </note>
         * 
         * @param image
         *        The path where inference code is stored. This can be either in Amazon EC2 Container Registry or in a
         *        Docker registry that is accessible from the same VPC that you configure for your endpoint. If you are
         *        using your own custom algorithm instead of an algorithm provided by SageMaker, the inference code must
         *        meet SageMaker requirements. SageMaker supports both <code>registry/repository[:tag]</code> and
         *        <code>registry/repository[@digest]</code> image path formats. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own Algorithms
         *        with Amazon SageMaker</a>. </p> <note>
         *        <p>
         *        The model artifacts in an Amazon S3 bucket and the Docker image for inference container in Amazon EC2
         *        Container Registry must be in the same region as the model or endpoint you are creating.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * Specifies whether the model container is in Amazon ECR or a private Docker registry accessible from your
         * Amazon Virtual Private Cloud (VPC). For information about storing containers in a private Docker registry,
         * see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-containers-inference-private.html">Use
         * a Private Docker Registry for Real-Time Inference Containers</a>.
         * </p>
         * <note>
         * <p>
         * The model artifacts in an Amazon S3 bucket and the Docker image for inference container in Amazon EC2
         * Container Registry must be in the same region as the model or endpoint you are creating.
         * </p>
         * </note>
         * 
         * @param imageConfig
         *        Specifies whether the model container is in Amazon ECR or a private Docker registry accessible from
         *        your Amazon Virtual Private Cloud (VPC). For information about storing containers in a private Docker
         *        registry, see <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-containers-inference-private.html"
         *        >Use a Private Docker Registry for Real-Time Inference Containers</a>. </p> <note>
         *        <p>
         *        The model artifacts in an Amazon S3 bucket and the Docker image for inference container in Amazon EC2
         *        Container Registry must be in the same region as the model or endpoint you are creating.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageConfig(ImageConfig imageConfig);

        /**
         * <p>
         * Specifies whether the model container is in Amazon ECR or a private Docker registry accessible from your
         * Amazon Virtual Private Cloud (VPC). For information about storing containers in a private Docker registry,
         * see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-containers-inference-private.html">Use
         * a Private Docker Registry for Real-Time Inference Containers</a>.
         * </p>
         * <note>
         * <p>
         * The model artifacts in an Amazon S3 bucket and the Docker image for inference container in Amazon EC2
         * Container Registry must be in the same region as the model or endpoint you are creating.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link ImageConfig.Builder} avoiding the
         * need to create one manually via {@link ImageConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImageConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #imageConfig(ImageConfig)}.
         * 
         * @param imageConfig
         *        a consumer that will call methods on {@link ImageConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageConfig(ImageConfig)
         */
        default Builder imageConfig(Consumer<ImageConfig.Builder> imageConfig) {
            return imageConfig(ImageConfig.builder().applyMutation(imageConfig).build());
        }

        /**
         * <p>
         * Whether the container hosts a single model or multiple models.
         * </p>
         * 
         * @param mode
         *        Whether the container hosts a single model or multiple models.
         * @see ContainerMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerMode
         */
        Builder mode(String mode);

        /**
         * <p>
         * Whether the container hosts a single model or multiple models.
         * </p>
         * 
         * @param mode
         *        Whether the container hosts a single model or multiple models.
         * @see ContainerMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerMode
         */
        Builder mode(ContainerMode mode);

        /**
         * <p>
         * The S3 path where the model artifacts, which result from model training, are stored. This path must point to
         * a single gzip compressed tar archive (.tar.gz suffix). The S3 path is required for SageMaker built-in
         * algorithms, but not if you use your own algorithms. For more information on built-in algorithms, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-algo-docker-registry-paths.html">Common
         * Parameters</a>.
         * </p>
         * <note>
         * <p>
         * The model artifacts must be in an S3 bucket that is in the same region as the model or endpoint you are
         * creating.
         * </p>
         * </note>
         * <p>
         * If you provide a value for this parameter, SageMaker uses Amazon Web Services Security Token Service to
         * download model artifacts from the S3 path you provide. Amazon Web Services STS is activated in your Amazon
         * Web Services account by default. If you previously deactivated Amazon Web Services STS for a region, you need
         * to reactivate Amazon Web Services STS for that region. For more information, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp_enable-regions.html">Activating
         * and Deactivating Amazon Web Services STS in an Amazon Web Services Region</a> in the <i>Amazon Web Services
         * Identity and Access Management User Guide</i>.
         * </p>
         * <important>
         * <p>
         * If you use a built-in algorithm to create a model, SageMaker requires that you provide a S3 path to the model
         * artifacts in <code>ModelDataUrl</code>.
         * </p>
         * </important>
         * 
         * @param modelDataUrl
         *        The S3 path where the model artifacts, which result from model training, are stored. This path must
         *        point to a single gzip compressed tar archive (.tar.gz suffix). The S3 path is required for SageMaker
         *        built-in algorithms, but not if you use your own algorithms. For more information on built-in
         *        algorithms, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-algo-docker-registry-paths.html"
         *        >Common Parameters</a>. </p> <note>
         *        <p>
         *        The model artifacts must be in an S3 bucket that is in the same region as the model or endpoint you
         *        are creating.
         *        </p>
         *        </note>
         *        <p>
         *        If you provide a value for this parameter, SageMaker uses Amazon Web Services Security Token Service
         *        to download model artifacts from the S3 path you provide. Amazon Web Services STS is activated in your
         *        Amazon Web Services account by default. If you previously deactivated Amazon Web Services STS for a
         *        region, you need to reactivate Amazon Web Services STS for that region. For more information, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp_enable-regions.html">
         *        Activating and Deactivating Amazon Web Services STS in an Amazon Web Services Region</a> in the
         *        <i>Amazon Web Services Identity and Access Management User Guide</i>.
         *        </p>
         *        <important>
         *        <p>
         *        If you use a built-in algorithm to create a model, SageMaker requires that you provide a S3 path to
         *        the model artifacts in <code>ModelDataUrl</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelDataUrl(String modelDataUrl);

        /**
         * <p>
         * Specifies the location of ML model data to deploy.
         * </p>
         * <note>
         * <p>
         * Currently you cannot use <code>ModelDataSource</code> in conjunction with SageMaker batch transform,
         * SageMaker serverless endpoints, SageMaker multi-model endpoints, and SageMaker Marketplace.
         * </p>
         * </note>
         * 
         * @param modelDataSource
         *        Specifies the location of ML model data to deploy.</p> <note>
         *        <p>
         *        Currently you cannot use <code>ModelDataSource</code> in conjunction with SageMaker batch transform,
         *        SageMaker serverless endpoints, SageMaker multi-model endpoints, and SageMaker Marketplace.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelDataSource(ModelDataSource modelDataSource);

        /**
         * <p>
         * Specifies the location of ML model data to deploy.
         * </p>
         * <note>
         * <p>
         * Currently you cannot use <code>ModelDataSource</code> in conjunction with SageMaker batch transform,
         * SageMaker serverless endpoints, SageMaker multi-model endpoints, and SageMaker Marketplace.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link ModelDataSource.Builder} avoiding
         * the need to create one manually via {@link ModelDataSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ModelDataSource.Builder#build()} is called immediately and its
         * result is passed to {@link #modelDataSource(ModelDataSource)}.
         * 
         * @param modelDataSource
         *        a consumer that will call methods on {@link ModelDataSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modelDataSource(ModelDataSource)
         */
        default Builder modelDataSource(Consumer<ModelDataSource.Builder> modelDataSource) {
            return modelDataSource(ModelDataSource.builder().applyMutation(modelDataSource).build());
        }

        /**
         * <p>
         * The environment variables to set in the Docker container. Each key and value in the <code>Environment</code>
         * string to string map can have length of up to 1024. We support up to 16 entries in the map.
         * </p>
         * 
         * @param environment
         *        The environment variables to set in the Docker container. Each key and value in the
         *        <code>Environment</code> string to string map can have length of up to 1024. We support up to 16
         *        entries in the map.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Map<String, String> environment);

        /**
         * <p>
         * The name or Amazon Resource Name (ARN) of the model package to use to create the model.
         * </p>
         * 
         * @param modelPackageName
         *        The name or Amazon Resource Name (ARN) of the model package to use to create the model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelPackageName(String modelPackageName);

        /**
         * <p>
         * The inference specification name in the model package version.
         * </p>
         * 
         * @param inferenceSpecificationName
         *        The inference specification name in the model package version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferenceSpecificationName(String inferenceSpecificationName);

        /**
         * <p>
         * Specifies additional configuration for multi-model endpoints.
         * </p>
         * 
         * @param multiModelConfig
         *        Specifies additional configuration for multi-model endpoints.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder multiModelConfig(MultiModelConfig multiModelConfig);

        /**
         * <p>
         * Specifies additional configuration for multi-model endpoints.
         * </p>
         * This is a convenience method that creates an instance of the {@link MultiModelConfig.Builder} avoiding the
         * need to create one manually via {@link MultiModelConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MultiModelConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #multiModelConfig(MultiModelConfig)}.
         * 
         * @param multiModelConfig
         *        a consumer that will call methods on {@link MultiModelConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #multiModelConfig(MultiModelConfig)
         */
        default Builder multiModelConfig(Consumer<MultiModelConfig.Builder> multiModelConfig) {
            return multiModelConfig(MultiModelConfig.builder().applyMutation(multiModelConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String containerHostname;

        private String image;

        private ImageConfig imageConfig;

        private String mode;

        private String modelDataUrl;

        private ModelDataSource modelDataSource;

        private Map<String, String> environment = DefaultSdkAutoConstructMap.getInstance();

        private String modelPackageName;

        private String inferenceSpecificationName;

        private MultiModelConfig multiModelConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerDefinition model) {
            containerHostname(model.containerHostname);
            image(model.image);
            imageConfig(model.imageConfig);
            mode(model.mode);
            modelDataUrl(model.modelDataUrl);
            modelDataSource(model.modelDataSource);
            environment(model.environment);
            modelPackageName(model.modelPackageName);
            inferenceSpecificationName(model.inferenceSpecificationName);
            multiModelConfig(model.multiModelConfig);
        }

        public final String getContainerHostname() {
            return containerHostname;
        }

        public final void setContainerHostname(String containerHostname) {
            this.containerHostname = containerHostname;
        }

        @Override
        public final Builder containerHostname(String containerHostname) {
            this.containerHostname = containerHostname;
            return this;
        }

        public final String getImage() {
            return image;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final ImageConfig.Builder getImageConfig() {
            return imageConfig != null ? imageConfig.toBuilder() : null;
        }

        public final void setImageConfig(ImageConfig.BuilderImpl imageConfig) {
            this.imageConfig = imageConfig != null ? imageConfig.build() : null;
        }

        @Override
        public final Builder imageConfig(ImageConfig imageConfig) {
            this.imageConfig = imageConfig;
            return this;
        }

        public final String getMode() {
            return mode;
        }

        public final void setMode(String mode) {
            this.mode = mode;
        }

        @Override
        public final Builder mode(String mode) {
            this.mode = mode;
            return this;
        }

        @Override
        public final Builder mode(ContainerMode mode) {
            this.mode(mode == null ? null : mode.toString());
            return this;
        }

        public final String getModelDataUrl() {
            return modelDataUrl;
        }

        public final void setModelDataUrl(String modelDataUrl) {
            this.modelDataUrl = modelDataUrl;
        }

        @Override
        public final Builder modelDataUrl(String modelDataUrl) {
            this.modelDataUrl = modelDataUrl;
            return this;
        }

        public final ModelDataSource.Builder getModelDataSource() {
            return modelDataSource != null ? modelDataSource.toBuilder() : null;
        }

        public final void setModelDataSource(ModelDataSource.BuilderImpl modelDataSource) {
            this.modelDataSource = modelDataSource != null ? modelDataSource.build() : null;
        }

        @Override
        public final Builder modelDataSource(ModelDataSource modelDataSource) {
            this.modelDataSource = modelDataSource;
            return this;
        }

        public final Map<String, String> getEnvironment() {
            if (environment instanceof SdkAutoConstructMap) {
                return null;
            }
            return environment;
        }

        public final void setEnvironment(Map<String, String> environment) {
            this.environment = EnvironmentMapCopier.copy(environment);
        }

        @Override
        public final Builder environment(Map<String, String> environment) {
            this.environment = EnvironmentMapCopier.copy(environment);
            return this;
        }

        public final String getModelPackageName() {
            return modelPackageName;
        }

        public final void setModelPackageName(String modelPackageName) {
            this.modelPackageName = modelPackageName;
        }

        @Override
        public final Builder modelPackageName(String modelPackageName) {
            this.modelPackageName = modelPackageName;
            return this;
        }

        public final String getInferenceSpecificationName() {
            return inferenceSpecificationName;
        }

        public final void setInferenceSpecificationName(String inferenceSpecificationName) {
            this.inferenceSpecificationName = inferenceSpecificationName;
        }

        @Override
        public final Builder inferenceSpecificationName(String inferenceSpecificationName) {
            this.inferenceSpecificationName = inferenceSpecificationName;
            return this;
        }

        public final MultiModelConfig.Builder getMultiModelConfig() {
            return multiModelConfig != null ? multiModelConfig.toBuilder() : null;
        }

        public final void setMultiModelConfig(MultiModelConfig.BuilderImpl multiModelConfig) {
            this.multiModelConfig = multiModelConfig != null ? multiModelConfig.build() : null;
        }

        @Override
        public final Builder multiModelConfig(MultiModelConfig multiModelConfig) {
            this.multiModelConfig = multiModelConfig;
            return this;
        }

        @Override
        public ContainerDefinition build() {
            return new ContainerDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
