/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for the <a
 * href="https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-feature-attribute-shap-baselines.html">SHAP
 * baseline</a> (also called the background or reference dataset) of the Kernal SHAP algorithm.
 * </p>
 * <note>
 * <ul>
 * <li>
 * <p>
 * The number of records in the baseline data determines the size of the synthetic dataset, which has an impact on
 * latency of explainability requests. For more information, see the <b>Synthetic data</b> of <a
 * href="https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-online-explainability-create-endpoint.html">Configure
 * and create an endpoint</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>ShapBaseline</code> and <code>ShapBaselineUri</code> are mutually exclusive parameters. One or the either is
 * required to configure a SHAP baseline.
 * </p>
 * </li>
 * </ul>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ClarifyShapBaselineConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<ClarifyShapBaselineConfig.Builder, ClarifyShapBaselineConfig> {
    private static final SdkField<String> MIME_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MimeType").getter(getter(ClarifyShapBaselineConfig::mimeType)).setter(setter(Builder::mimeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MimeType").build()).build();

    private static final SdkField<String> SHAP_BASELINE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ShapBaseline").getter(getter(ClarifyShapBaselineConfig::shapBaseline))
            .setter(setter(Builder::shapBaseline))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShapBaseline").build()).build();

    private static final SdkField<String> SHAP_BASELINE_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ShapBaselineUri").getter(getter(ClarifyShapBaselineConfig::shapBaselineUri))
            .setter(setter(Builder::shapBaselineUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShapBaselineUri").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MIME_TYPE_FIELD,
            SHAP_BASELINE_FIELD, SHAP_BASELINE_URI_FIELD));

    private static final long serialVersionUID = 1L;

    private final String mimeType;

    private final String shapBaseline;

    private final String shapBaselineUri;

    private ClarifyShapBaselineConfig(BuilderImpl builder) {
        this.mimeType = builder.mimeType;
        this.shapBaseline = builder.shapBaseline;
        this.shapBaselineUri = builder.shapBaselineUri;
    }

    /**
     * <p>
     * The MIME type of the baseline data. Choose from <code>'text/csv'</code> or <code>'application/jsonlines'</code>.
     * Defaults to <code>'text/csv'</code>.
     * </p>
     * 
     * @return The MIME type of the baseline data. Choose from <code>'text/csv'</code> or
     *         <code>'application/jsonlines'</code>. Defaults to <code>'text/csv'</code>.
     */
    public final String mimeType() {
        return mimeType;
    }

    /**
     * <p>
     * The inline SHAP baseline data in string format. <code>ShapBaseline</code> can have one or multiple records to be
     * used as the baseline dataset. The format of the SHAP baseline file should be the same format as the training
     * dataset. For example, if the training dataset is in CSV format and each record contains four features, and all
     * features are numerical, then the format of the baseline data should also share these characteristics. For natural
     * language processing (NLP) of text columns, the baseline value should be the value used to replace the unit of
     * text specified by the <code>Granularity</code> of the <code>TextConfig</code> parameter. The size limit for
     * <code>ShapBasline</code> is 4 KB. Use the <code>ShapBaselineUri</code> parameter if you want to provide more than
     * 4 KB of baseline data.
     * </p>
     * 
     * @return The inline SHAP baseline data in string format. <code>ShapBaseline</code> can have one or multiple
     *         records to be used as the baseline dataset. The format of the SHAP baseline file should be the same
     *         format as the training dataset. For example, if the training dataset is in CSV format and each record
     *         contains four features, and all features are numerical, then the format of the baseline data should also
     *         share these characteristics. For natural language processing (NLP) of text columns, the baseline value
     *         should be the value used to replace the unit of text specified by the <code>Granularity</code> of the
     *         <code>TextConfig</code> parameter. The size limit for <code>ShapBasline</code> is 4 KB. Use the
     *         <code>ShapBaselineUri</code> parameter if you want to provide more than 4 KB of baseline data.
     */
    public final String shapBaseline() {
        return shapBaseline;
    }

    /**
     * <p>
     * The uniform resource identifier (URI) of the S3 bucket where the SHAP baseline file is stored. The format of the
     * SHAP baseline file should be the same format as the format of the training dataset. For example, if the training
     * dataset is in CSV format, and each record in the training dataset has four features, and all features are
     * numerical, then the baseline file should also have this same format. Each record should contain only the
     * features. If you are using a virtual private cloud (VPC), the <code>ShapBaselineUri</code> should be accessible
     * to the VPC. For more information about setting up endpoints with Amazon Virtual Private Cloud, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/infrastructure-give-access.html">Give SageMaker access to
     * Resources in your Amazon Virtual Private Cloud</a>.
     * </p>
     * 
     * @return The uniform resource identifier (URI) of the S3 bucket where the SHAP baseline file is stored. The format
     *         of the SHAP baseline file should be the same format as the format of the training dataset. For example,
     *         if the training dataset is in CSV format, and each record in the training dataset has four features, and
     *         all features are numerical, then the baseline file should also have this same format. Each record should
     *         contain only the features. If you are using a virtual private cloud (VPC), the
     *         <code>ShapBaselineUri</code> should be accessible to the VPC. For more information about setting up
     *         endpoints with Amazon Virtual Private Cloud, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/infrastructure-give-access.html">Give SageMaker
     *         access to Resources in your Amazon Virtual Private Cloud</a>.
     */
    public final String shapBaselineUri() {
        return shapBaselineUri;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(mimeType());
        hashCode = 31 * hashCode + Objects.hashCode(shapBaseline());
        hashCode = 31 * hashCode + Objects.hashCode(shapBaselineUri());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClarifyShapBaselineConfig)) {
            return false;
        }
        ClarifyShapBaselineConfig other = (ClarifyShapBaselineConfig) obj;
        return Objects.equals(mimeType(), other.mimeType()) && Objects.equals(shapBaseline(), other.shapBaseline())
                && Objects.equals(shapBaselineUri(), other.shapBaselineUri());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ClarifyShapBaselineConfig").add("MimeType", mimeType()).add("ShapBaseline", shapBaseline())
                .add("ShapBaselineUri", shapBaselineUri()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MimeType":
            return Optional.ofNullable(clazz.cast(mimeType()));
        case "ShapBaseline":
            return Optional.ofNullable(clazz.cast(shapBaseline()));
        case "ShapBaselineUri":
            return Optional.ofNullable(clazz.cast(shapBaselineUri()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ClarifyShapBaselineConfig, T> g) {
        return obj -> g.apply((ClarifyShapBaselineConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ClarifyShapBaselineConfig> {
        /**
         * <p>
         * The MIME type of the baseline data. Choose from <code>'text/csv'</code> or
         * <code>'application/jsonlines'</code>. Defaults to <code>'text/csv'</code>.
         * </p>
         * 
         * @param mimeType
         *        The MIME type of the baseline data. Choose from <code>'text/csv'</code> or
         *        <code>'application/jsonlines'</code>. Defaults to <code>'text/csv'</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mimeType(String mimeType);

        /**
         * <p>
         * The inline SHAP baseline data in string format. <code>ShapBaseline</code> can have one or multiple records to
         * be used as the baseline dataset. The format of the SHAP baseline file should be the same format as the
         * training dataset. For example, if the training dataset is in CSV format and each record contains four
         * features, and all features are numerical, then the format of the baseline data should also share these
         * characteristics. For natural language processing (NLP) of text columns, the baseline value should be the
         * value used to replace the unit of text specified by the <code>Granularity</code> of the
         * <code>TextConfig</code> parameter. The size limit for <code>ShapBasline</code> is 4 KB. Use the
         * <code>ShapBaselineUri</code> parameter if you want to provide more than 4 KB of baseline data.
         * </p>
         * 
         * @param shapBaseline
         *        The inline SHAP baseline data in string format. <code>ShapBaseline</code> can have one or multiple
         *        records to be used as the baseline dataset. The format of the SHAP baseline file should be the same
         *        format as the training dataset. For example, if the training dataset is in CSV format and each record
         *        contains four features, and all features are numerical, then the format of the baseline data should
         *        also share these characteristics. For natural language processing (NLP) of text columns, the baseline
         *        value should be the value used to replace the unit of text specified by the <code>Granularity</code>
         *        of the <code>TextConfig</code> parameter. The size limit for <code>ShapBasline</code> is 4 KB. Use the
         *        <code>ShapBaselineUri</code> parameter if you want to provide more than 4 KB of baseline data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shapBaseline(String shapBaseline);

        /**
         * <p>
         * The uniform resource identifier (URI) of the S3 bucket where the SHAP baseline file is stored. The format of
         * the SHAP baseline file should be the same format as the format of the training dataset. For example, if the
         * training dataset is in CSV format, and each record in the training dataset has four features, and all
         * features are numerical, then the baseline file should also have this same format. Each record should contain
         * only the features. If you are using a virtual private cloud (VPC), the <code>ShapBaselineUri</code> should be
         * accessible to the VPC. For more information about setting up endpoints with Amazon Virtual Private Cloud, see
         * <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/infrastructure-give-access.html">Give SageMaker
         * access to Resources in your Amazon Virtual Private Cloud</a>.
         * </p>
         * 
         * @param shapBaselineUri
         *        The uniform resource identifier (URI) of the S3 bucket where the SHAP baseline file is stored. The
         *        format of the SHAP baseline file should be the same format as the format of the training dataset. For
         *        example, if the training dataset is in CSV format, and each record in the training dataset has four
         *        features, and all features are numerical, then the baseline file should also have this same format.
         *        Each record should contain only the features. If you are using a virtual private cloud (VPC), the
         *        <code>ShapBaselineUri</code> should be accessible to the VPC. For more information about setting up
         *        endpoints with Amazon Virtual Private Cloud, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/infrastructure-give-access.html">Give SageMaker
         *        access to Resources in your Amazon Virtual Private Cloud</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shapBaselineUri(String shapBaselineUri);
    }

    static final class BuilderImpl implements Builder {
        private String mimeType;

        private String shapBaseline;

        private String shapBaselineUri;

        private BuilderImpl() {
        }

        private BuilderImpl(ClarifyShapBaselineConfig model) {
            mimeType(model.mimeType);
            shapBaseline(model.shapBaseline);
            shapBaselineUri(model.shapBaselineUri);
        }

        public final String getMimeType() {
            return mimeType;
        }

        public final void setMimeType(String mimeType) {
            this.mimeType = mimeType;
        }

        @Override
        public final Builder mimeType(String mimeType) {
            this.mimeType = mimeType;
            return this;
        }

        public final String getShapBaseline() {
            return shapBaseline;
        }

        public final void setShapBaseline(String shapBaseline) {
            this.shapBaseline = shapBaseline;
        }

        @Override
        public final Builder shapBaseline(String shapBaseline) {
            this.shapBaseline = shapBaseline;
            return this;
        }

        public final String getShapBaselineUri() {
            return shapBaselineUri;
        }

        public final void setShapBaselineUri(String shapBaselineUri) {
            this.shapBaselineUri = shapBaselineUri;
        }

        @Override
        public final Builder shapBaselineUri(String shapBaselineUri) {
            this.shapBaselineUri = shapBaselineUri;
            return this;
        }

        @Override
        public ClarifyShapBaselineConfig build() {
            return new ClarifyShapBaselineConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
