/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.sagemaker.SageMakerAsyncClient;
import software.amazon.awssdk.services.sagemaker.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.sagemaker.model.DescribeEndpointRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeEndpointResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeImageRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeImageResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeImageVersionRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeImageVersionResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeNotebookInstanceRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeNotebookInstanceResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeProcessingJobRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeProcessingJobResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeTrainingJobRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeTrainingJobResponse;
import software.amazon.awssdk.services.sagemaker.model.DescribeTransformJobRequest;
import software.amazon.awssdk.services.sagemaker.model.DescribeTransformJobResponse;
import software.amazon.awssdk.services.sagemaker.model.SageMakerRequest;
import software.amazon.awssdk.services.sagemaker.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultSageMakerAsyncWaiter implements SageMakerAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final SageMakerAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeNotebookInstanceResponse> notebookInstanceInServiceWaiter;

    private final AsyncWaiter<DescribeNotebookInstanceResponse> notebookInstanceStoppedWaiter;

    private final AsyncWaiter<DescribeNotebookInstanceResponse> notebookInstanceDeletedWaiter;

    private final AsyncWaiter<DescribeTrainingJobResponse> trainingJobCompletedOrStoppedWaiter;

    private final AsyncWaiter<DescribeEndpointResponse> endpointInServiceWaiter;

    private final AsyncWaiter<DescribeEndpointResponse> endpointDeletedWaiter;

    private final AsyncWaiter<DescribeTransformJobResponse> transformJobCompletedOrStoppedWaiter;

    private final AsyncWaiter<DescribeProcessingJobResponse> processingJobCompletedOrStoppedWaiter;

    private final AsyncWaiter<DescribeImageResponse> imageCreatedWaiter;

    private final AsyncWaiter<DescribeImageResponse> imageUpdatedWaiter;

    private final AsyncWaiter<DescribeImageResponse> imageDeletedWaiter;

    private final AsyncWaiter<DescribeImageVersionResponse> imageVersionCreatedWaiter;

    private final AsyncWaiter<DescribeImageVersionResponse> imageVersionDeletedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultSageMakerAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = SageMakerAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.notebookInstanceInServiceWaiter = AsyncWaiter.builder(DescribeNotebookInstanceResponse.class)
                .acceptors(notebookInstanceInServiceWaiterAcceptors())
                .overrideConfiguration(notebookInstanceInServiceWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.notebookInstanceStoppedWaiter = AsyncWaiter.builder(DescribeNotebookInstanceResponse.class)
                .acceptors(notebookInstanceStoppedWaiterAcceptors())
                .overrideConfiguration(notebookInstanceStoppedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.notebookInstanceDeletedWaiter = AsyncWaiter.builder(DescribeNotebookInstanceResponse.class)
                .acceptors(notebookInstanceDeletedWaiterAcceptors())
                .overrideConfiguration(notebookInstanceDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.trainingJobCompletedOrStoppedWaiter = AsyncWaiter.builder(DescribeTrainingJobResponse.class)
                .acceptors(trainingJobCompletedOrStoppedWaiterAcceptors())
                .overrideConfiguration(trainingJobCompletedOrStoppedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.endpointInServiceWaiter = AsyncWaiter.builder(DescribeEndpointResponse.class)
                .acceptors(endpointInServiceWaiterAcceptors())
                .overrideConfiguration(endpointInServiceWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.endpointDeletedWaiter = AsyncWaiter.builder(DescribeEndpointResponse.class)
                .acceptors(endpointDeletedWaiterAcceptors())
                .overrideConfiguration(endpointDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.transformJobCompletedOrStoppedWaiter = AsyncWaiter.builder(DescribeTransformJobResponse.class)
                .acceptors(transformJobCompletedOrStoppedWaiterAcceptors())
                .overrideConfiguration(transformJobCompletedOrStoppedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.processingJobCompletedOrStoppedWaiter = AsyncWaiter.builder(DescribeProcessingJobResponse.class)
                .acceptors(processingJobCompletedOrStoppedWaiterAcceptors())
                .overrideConfiguration(processingJobCompletedOrStoppedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.imageCreatedWaiter = AsyncWaiter.builder(DescribeImageResponse.class).acceptors(imageCreatedWaiterAcceptors())
                .overrideConfiguration(imageCreatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.imageUpdatedWaiter = AsyncWaiter.builder(DescribeImageResponse.class).acceptors(imageUpdatedWaiterAcceptors())
                .overrideConfiguration(imageUpdatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.imageDeletedWaiter = AsyncWaiter.builder(DescribeImageResponse.class).acceptors(imageDeletedWaiterAcceptors())
                .overrideConfiguration(imageDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.imageVersionCreatedWaiter = AsyncWaiter.builder(DescribeImageVersionResponse.class)
                .acceptors(imageVersionCreatedWaiterAcceptors())
                .overrideConfiguration(imageVersionCreatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.imageVersionDeletedWaiter = AsyncWaiter.builder(DescribeImageVersionResponse.class)
                .acceptors(imageVersionDeletedWaiterAcceptors())
                .overrideConfiguration(imageVersionDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEndpointResponse>> waitUntilEndpointDeleted(
            DescribeEndpointRequest describeEndpointRequest) {
        return endpointDeletedWaiter.runAsync(() -> client.describeEndpoint(applyWaitersUserAgent(describeEndpointRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEndpointResponse>> waitUntilEndpointDeleted(
            DescribeEndpointRequest describeEndpointRequest, WaiterOverrideConfiguration overrideConfig) {
        return endpointDeletedWaiter.runAsync(() -> client.describeEndpoint(applyWaitersUserAgent(describeEndpointRequest)),
                endpointDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEndpointResponse>> waitUntilEndpointInService(
            DescribeEndpointRequest describeEndpointRequest) {
        return endpointInServiceWaiter.runAsync(() -> client.describeEndpoint(applyWaitersUserAgent(describeEndpointRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEndpointResponse>> waitUntilEndpointInService(
            DescribeEndpointRequest describeEndpointRequest, WaiterOverrideConfiguration overrideConfig) {
        return endpointInServiceWaiter.runAsync(() -> client.describeEndpoint(applyWaitersUserAgent(describeEndpointRequest)),
                endpointInServiceWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageCreated(
            DescribeImageRequest describeImageRequest) {
        return imageCreatedWaiter.runAsync(() -> client.describeImage(applyWaitersUserAgent(describeImageRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageCreated(
            DescribeImageRequest describeImageRequest, WaiterOverrideConfiguration overrideConfig) {
        return imageCreatedWaiter.runAsync(() -> client.describeImage(applyWaitersUserAgent(describeImageRequest)),
                imageCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageDeleted(
            DescribeImageRequest describeImageRequest) {
        return imageDeletedWaiter.runAsync(() -> client.describeImage(applyWaitersUserAgent(describeImageRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageDeleted(
            DescribeImageRequest describeImageRequest, WaiterOverrideConfiguration overrideConfig) {
        return imageDeletedWaiter.runAsync(() -> client.describeImage(applyWaitersUserAgent(describeImageRequest)),
                imageDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageUpdated(
            DescribeImageRequest describeImageRequest) {
        return imageUpdatedWaiter.runAsync(() -> client.describeImage(applyWaitersUserAgent(describeImageRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImageResponse>> waitUntilImageUpdated(
            DescribeImageRequest describeImageRequest, WaiterOverrideConfiguration overrideConfig) {
        return imageUpdatedWaiter.runAsync(() -> client.describeImage(applyWaitersUserAgent(describeImageRequest)),
                imageUpdatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImageVersionResponse>> waitUntilImageVersionCreated(
            DescribeImageVersionRequest describeImageVersionRequest) {
        return imageVersionCreatedWaiter.runAsync(() -> client
                .describeImageVersion(applyWaitersUserAgent(describeImageVersionRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImageVersionResponse>> waitUntilImageVersionCreated(
            DescribeImageVersionRequest describeImageVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        return imageVersionCreatedWaiter.runAsync(
                () -> client.describeImageVersion(applyWaitersUserAgent(describeImageVersionRequest)),
                imageVersionCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImageVersionResponse>> waitUntilImageVersionDeleted(
            DescribeImageVersionRequest describeImageVersionRequest) {
        return imageVersionDeletedWaiter.runAsync(() -> client
                .describeImageVersion(applyWaitersUserAgent(describeImageVersionRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImageVersionResponse>> waitUntilImageVersionDeleted(
            DescribeImageVersionRequest describeImageVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        return imageVersionDeletedWaiter.runAsync(
                () -> client.describeImageVersion(applyWaitersUserAgent(describeImageVersionRequest)),
                imageVersionDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceDeleted(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest) {
        return notebookInstanceDeletedWaiter.runAsync(() -> client
                .describeNotebookInstance(applyWaitersUserAgent(describeNotebookInstanceRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceDeleted(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest, WaiterOverrideConfiguration overrideConfig) {
        return notebookInstanceDeletedWaiter.runAsync(
                () -> client.describeNotebookInstance(applyWaitersUserAgent(describeNotebookInstanceRequest)),
                notebookInstanceDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceInService(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest) {
        return notebookInstanceInServiceWaiter.runAsync(() -> client
                .describeNotebookInstance(applyWaitersUserAgent(describeNotebookInstanceRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceInService(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest, WaiterOverrideConfiguration overrideConfig) {
        return notebookInstanceInServiceWaiter.runAsync(
                () -> client.describeNotebookInstance(applyWaitersUserAgent(describeNotebookInstanceRequest)),
                notebookInstanceInServiceWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceStopped(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest) {
        return notebookInstanceStoppedWaiter.runAsync(() -> client
                .describeNotebookInstance(applyWaitersUserAgent(describeNotebookInstanceRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNotebookInstanceResponse>> waitUntilNotebookInstanceStopped(
            DescribeNotebookInstanceRequest describeNotebookInstanceRequest, WaiterOverrideConfiguration overrideConfig) {
        return notebookInstanceStoppedWaiter.runAsync(
                () -> client.describeNotebookInstance(applyWaitersUserAgent(describeNotebookInstanceRequest)),
                notebookInstanceStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeProcessingJobResponse>> waitUntilProcessingJobCompletedOrStopped(
            DescribeProcessingJobRequest describeProcessingJobRequest) {
        return processingJobCompletedOrStoppedWaiter.runAsync(() -> client
                .describeProcessingJob(applyWaitersUserAgent(describeProcessingJobRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeProcessingJobResponse>> waitUntilProcessingJobCompletedOrStopped(
            DescribeProcessingJobRequest describeProcessingJobRequest, WaiterOverrideConfiguration overrideConfig) {
        return processingJobCompletedOrStoppedWaiter.runAsync(
                () -> client.describeProcessingJob(applyWaitersUserAgent(describeProcessingJobRequest)),
                processingJobCompletedOrStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTrainingJobResponse>> waitUntilTrainingJobCompletedOrStopped(
            DescribeTrainingJobRequest describeTrainingJobRequest) {
        return trainingJobCompletedOrStoppedWaiter.runAsync(() -> client
                .describeTrainingJob(applyWaitersUserAgent(describeTrainingJobRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTrainingJobResponse>> waitUntilTrainingJobCompletedOrStopped(
            DescribeTrainingJobRequest describeTrainingJobRequest, WaiterOverrideConfiguration overrideConfig) {
        return trainingJobCompletedOrStoppedWaiter.runAsync(
                () -> client.describeTrainingJob(applyWaitersUserAgent(describeTrainingJobRequest)),
                trainingJobCompletedOrStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTransformJobResponse>> waitUntilTransformJobCompletedOrStopped(
            DescribeTransformJobRequest describeTransformJobRequest) {
        return transformJobCompletedOrStoppedWaiter.runAsync(() -> client
                .describeTransformJob(applyWaitersUserAgent(describeTransformJobRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTransformJobResponse>> waitUntilTransformJobCompletedOrStopped(
            DescribeTransformJobRequest describeTransformJobRequest, WaiterOverrideConfiguration overrideConfig) {
        return transformJobCompletedOrStoppedWaiter.runAsync(
                () -> client.describeTransformJob(applyWaitersUserAgent(describeTransformJobRequest)),
                transformJobCompletedOrStoppedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeNotebookInstanceResponse>> notebookInstanceInServiceWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNotebookInstanceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("NotebookInstanceStatus").value(), "InService");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("NotebookInstanceStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (NotebookInstanceStatus=Failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNotebookInstanceResponse>> notebookInstanceStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNotebookInstanceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("NotebookInstanceStatus").value(), "Stopped");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("NotebookInstanceStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (NotebookInstanceStatus=Failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNotebookInstanceResponse>> notebookInstanceDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNotebookInstanceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("NotebookInstanceStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (NotebookInstanceStatus=Failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTrainingJobResponse>> trainingJobCompletedOrStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTrainingJobResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("TrainingJobStatus").value(), "Completed");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("TrainingJobStatus").value(), "Stopped");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("TrainingJobStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (TrainingJobStatus=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException"),
                "A waiter acceptor was matched on error condition (ValidationException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeEndpointResponse>> endpointInServiceWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeEndpointResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("EndpointStatus").value(), "InService");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("EndpointStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (EndpointStatus=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException"),
                "A waiter acceptor was matched on error condition (ValidationException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeEndpointResponse>> endpointDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeEndpointResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("EndpointStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (EndpointStatus=Failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTransformJobResponse>> transformJobCompletedOrStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTransformJobResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("TransformJobStatus").value(), "Completed");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("TransformJobStatus").value(), "Stopped");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("TransformJobStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (TransformJobStatus=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException"),
                "A waiter acceptor was matched on error condition (ValidationException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeProcessingJobResponse>> processingJobCompletedOrStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeProcessingJobResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ProcessingJobStatus").value(), "Completed");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ProcessingJobStatus").value(), "Stopped");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ProcessingJobStatus").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ProcessingJobStatus=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException"),
                "A waiter acceptor was matched on error condition (ValidationException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImageResponse>> imageCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImageResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ImageStatus").value(), "CREATED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ImageStatus").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ImageStatus=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException"),
                "A waiter acceptor was matched on error condition (ValidationException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImageResponse>> imageUpdatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImageResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ImageStatus").value(), "CREATED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ImageStatus").value(), "UPDATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ImageStatus=UPDATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException"),
                "A waiter acceptor was matched on error condition (ValidationException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImageResponse>> imageDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImageResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ImageStatus").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ImageStatus=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException"),
                "A waiter acceptor was matched on error condition (ValidationException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImageVersionResponse>> imageVersionCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImageVersionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ImageVersionStatus").value(), "CREATED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ImageVersionStatus").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ImageVersionStatus=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException"),
                "A waiter acceptor was matched on error condition (ValidationException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImageVersionResponse>> imageVersionDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImageVersionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ImageVersionStatus").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ImageVersionStatus=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationException"),
                "A waiter acceptor was matched on error condition (ValidationException) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration notebookInstanceInServiceWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration notebookInstanceStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration notebookInstanceDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration trainingJobCompletedOrStoppedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(120)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration endpointInServiceWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration endpointDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration transformJobCompletedOrStoppedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration processingJobCompletedOrStoppedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration imageCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration imageUpdatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration imageDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration imageVersionCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration imageVersionDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static SageMakerAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends SageMakerRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements SageMakerAsyncWaiter.Builder {
        private SageMakerAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public SageMakerAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public SageMakerAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public SageMakerAsyncWaiter.Builder client(SageMakerAsyncClient client) {
            this.client = client;
            return this;
        }

        public SageMakerAsyncWaiter build() {
            return new DefaultSageMakerAsyncWaiter(this);
        }
    }
}
