/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateInferenceExperimentRequest extends SageMakerRequest implements
        ToCopyableBuilder<CreateInferenceExperimentRequest.Builder, CreateInferenceExperimentRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreateInferenceExperimentRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(CreateInferenceExperimentRequest::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<InferenceExperimentSchedule> SCHEDULE_FIELD = SdkField
            .<InferenceExperimentSchedule> builder(MarshallingType.SDK_POJO).memberName("Schedule")
            .getter(getter(CreateInferenceExperimentRequest::schedule)).setter(setter(Builder::schedule))
            .constructor(InferenceExperimentSchedule::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Schedule").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(CreateInferenceExperimentRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleArn").getter(getter(CreateInferenceExperimentRequest::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<String> ENDPOINT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointName").getter(getter(CreateInferenceExperimentRequest::endpointName))
            .setter(setter(Builder::endpointName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointName").build()).build();

    private static final SdkField<List<ModelVariantConfig>> MODEL_VARIANTS_FIELD = SdkField
            .<List<ModelVariantConfig>> builder(MarshallingType.LIST)
            .memberName("ModelVariants")
            .getter(getter(CreateInferenceExperimentRequest::modelVariants))
            .setter(setter(Builder::modelVariants))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelVariants").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ModelVariantConfig> builder(MarshallingType.SDK_POJO)
                                            .constructor(ModelVariantConfig::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<InferenceExperimentDataStorageConfig> DATA_STORAGE_CONFIG_FIELD = SdkField
            .<InferenceExperimentDataStorageConfig> builder(MarshallingType.SDK_POJO).memberName("DataStorageConfig")
            .getter(getter(CreateInferenceExperimentRequest::dataStorageConfig)).setter(setter(Builder::dataStorageConfig))
            .constructor(InferenceExperimentDataStorageConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataStorageConfig").build()).build();

    private static final SdkField<ShadowModeConfig> SHADOW_MODE_CONFIG_FIELD = SdkField
            .<ShadowModeConfig> builder(MarshallingType.SDK_POJO).memberName("ShadowModeConfig")
            .getter(getter(CreateInferenceExperimentRequest::shadowModeConfig)).setter(setter(Builder::shadowModeConfig))
            .constructor(ShadowModeConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShadowModeConfig").build()).build();

    private static final SdkField<String> KMS_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KmsKey")
            .getter(getter(CreateInferenceExperimentRequest::kmsKey)).setter(setter(Builder::kmsKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKey").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateInferenceExperimentRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TYPE_FIELD,
            SCHEDULE_FIELD, DESCRIPTION_FIELD, ROLE_ARN_FIELD, ENDPOINT_NAME_FIELD, MODEL_VARIANTS_FIELD,
            DATA_STORAGE_CONFIG_FIELD, SHADOW_MODE_CONFIG_FIELD, KMS_KEY_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Name", NAME_FIELD);
                    put("Type", TYPE_FIELD);
                    put("Schedule", SCHEDULE_FIELD);
                    put("Description", DESCRIPTION_FIELD);
                    put("RoleArn", ROLE_ARN_FIELD);
                    put("EndpointName", ENDPOINT_NAME_FIELD);
                    put("ModelVariants", MODEL_VARIANTS_FIELD);
                    put("DataStorageConfig", DATA_STORAGE_CONFIG_FIELD);
                    put("ShadowModeConfig", SHADOW_MODE_CONFIG_FIELD);
                    put("KmsKey", KMS_KEY_FIELD);
                    put("Tags", TAGS_FIELD);
                }
            });

    private final String name;

    private final String type;

    private final InferenceExperimentSchedule schedule;

    private final String description;

    private final String roleArn;

    private final String endpointName;

    private final List<ModelVariantConfig> modelVariants;

    private final InferenceExperimentDataStorageConfig dataStorageConfig;

    private final ShadowModeConfig shadowModeConfig;

    private final String kmsKey;

    private final List<Tag> tags;

    private CreateInferenceExperimentRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.type = builder.type;
        this.schedule = builder.schedule;
        this.description = builder.description;
        this.roleArn = builder.roleArn;
        this.endpointName = builder.endpointName;
        this.modelVariants = builder.modelVariants;
        this.dataStorageConfig = builder.dataStorageConfig;
        this.shadowModeConfig = builder.shadowModeConfig;
        this.kmsKey = builder.kmsKey;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name for the inference experiment.
     * </p>
     * 
     * @return The name for the inference experiment.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of the inference experiment that you want to run. The following types of experiments are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ShadowMode</code>: You can use this type to validate a shadow variant. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/shadow-tests.html">Shadow tests</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link InferenceExperimentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the inference experiment that you want to run. The following types of experiments are
     *         possible: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ShadowMode</code>: You can use this type to validate a shadow variant. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/shadow-tests.html">Shadow tests</a>.
     *         </p>
     *         </li>
     * @see InferenceExperimentType
     */
    public final InferenceExperimentType type() {
        return InferenceExperimentType.fromValue(type);
    }

    /**
     * <p>
     * The type of the inference experiment that you want to run. The following types of experiments are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ShadowMode</code>: You can use this type to validate a shadow variant. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/shadow-tests.html">Shadow tests</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link InferenceExperimentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the inference experiment that you want to run. The following types of experiments are
     *         possible: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ShadowMode</code>: You can use this type to validate a shadow variant. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/shadow-tests.html">Shadow tests</a>.
     *         </p>
     *         </li>
     * @see InferenceExperimentType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The duration for which you want the inference experiment to run. If you don't specify this field, the experiment
     * automatically starts immediately upon creation and concludes after 7 days.
     * </p>
     * 
     * @return The duration for which you want the inference experiment to run. If you don't specify this field, the
     *         experiment automatically starts immediately upon creation and concludes after 7 days.
     */
    public final InferenceExperimentSchedule schedule() {
        return schedule;
    }

    /**
     * <p>
     * A description for the inference experiment.
     * </p>
     * 
     * @return A description for the inference experiment.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The ARN of the IAM role that Amazon SageMaker can assume to access model artifacts and container images, and
     * manage Amazon SageMaker Inference endpoints for model deployment.
     * </p>
     * 
     * @return The ARN of the IAM role that Amazon SageMaker can assume to access model artifacts and container images,
     *         and manage Amazon SageMaker Inference endpoints for model deployment.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The name of the Amazon SageMaker endpoint on which you want to run the inference experiment.
     * </p>
     * 
     * @return The name of the Amazon SageMaker endpoint on which you want to run the inference experiment.
     */
    public final String endpointName() {
        return endpointName;
    }

    /**
     * For responses, this returns true if the service returned a value for the ModelVariants property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasModelVariants() {
        return modelVariants != null && !(modelVariants instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>ModelVariantConfig</code> objects. There is one for each variant in the inference experiment.
     * Each <code>ModelVariantConfig</code> object in the array describes the infrastructure configuration for the
     * corresponding variant.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasModelVariants} method.
     * </p>
     * 
     * @return An array of <code>ModelVariantConfig</code> objects. There is one for each variant in the inference
     *         experiment. Each <code>ModelVariantConfig</code> object in the array describes the infrastructure
     *         configuration for the corresponding variant.
     */
    public final List<ModelVariantConfig> modelVariants() {
        return modelVariants;
    }

    /**
     * <p>
     * The Amazon S3 location and configuration for storing inference request and response data.
     * </p>
     * <p>
     * This is an optional parameter that you can use for data capture. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor-data-capture.html">Capture data</a>.
     * </p>
     * 
     * @return The Amazon S3 location and configuration for storing inference request and response data. </p>
     *         <p>
     *         This is an optional parameter that you can use for data capture. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor-data-capture.html">Capture data</a>.
     */
    public final InferenceExperimentDataStorageConfig dataStorageConfig() {
        return dataStorageConfig;
    }

    /**
     * <p>
     * The configuration of <code>ShadowMode</code> inference experiment type. Use this field to specify a production
     * variant which takes all the inference requests, and a shadow variant to which Amazon SageMaker replicates a
     * percentage of the inference requests. For the shadow variant also specify the percentage of requests that Amazon
     * SageMaker replicates.
     * </p>
     * 
     * @return The configuration of <code>ShadowMode</code> inference experiment type. Use this field to specify a
     *         production variant which takes all the inference requests, and a shadow variant to which Amazon SageMaker
     *         replicates a percentage of the inference requests. For the shadow variant also specify the percentage of
     *         requests that Amazon SageMaker replicates.
     */
    public final ShadowModeConfig shadowModeConfig() {
        return shadowModeConfig;
    }

    /**
     * <p>
     * The Amazon Web Services Key Management Service (Amazon Web Services KMS) key that Amazon SageMaker uses to
     * encrypt data on the storage volume attached to the ML compute instance that hosts the endpoint. The
     * <code>KmsKey</code> can be any of the following formats:
     * </p>
     * <ul>
     * <li>
     * <p>
     * KMS key ID
     * </p>
     * <p>
     * <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Resource Name (ARN) of a KMS key
     * </p>
     * <p>
     * <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * KMS key Alias
     * </p>
     * <p>
     * <code>"alias/ExampleAlias"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Resource Name (ARN) of a KMS key Alias
     * </p>
     * <p>
     * <code>"arn:aws:kms:us-west-2:111122223333:alias/ExampleAlias"</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you use a KMS key ID or an alias of your KMS key, the Amazon SageMaker execution role must include permissions
     * to call <code>kms:Encrypt</code>. If you don't provide a KMS key ID, Amazon SageMaker uses the default KMS key
     * for Amazon S3 for your role's account. Amazon SageMaker uses server-side encryption with KMS managed keys for
     * <code>OutputDataConfig</code>. If you use a bucket policy with an <code>s3:PutObject</code> permission that only
     * allows objects with server-side encryption, set the condition key of <code>s3:x-amz-server-side-encryption</code>
     * to <code>"aws:kms"</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html">KMS managed Encryption Keys</a> in
     * the <i>Amazon Simple Storage Service Developer Guide.</i>
     * </p>
     * <p>
     * The KMS key policy must grant permission to the IAM role that you specify in your <code>CreateEndpoint</code> and
     * <code>UpdateEndpoint</code> requests. For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html">Using Key Policies in Amazon Web
     * Services KMS</a> in the <i>Amazon Web Services Key Management Service Developer Guide</i>.
     * </p>
     * 
     * @return The Amazon Web Services Key Management Service (Amazon Web Services KMS) key that Amazon SageMaker uses
     *         to encrypt data on the storage volume attached to the ML compute instance that hosts the endpoint. The
     *         <code>KmsKey</code> can be any of the following formats: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         KMS key ID
     *         </p>
     *         <p>
     *         <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Resource Name (ARN) of a KMS key
     *         </p>
     *         <p>
     *         <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         KMS key Alias
     *         </p>
     *         <p>
     *         <code>"alias/ExampleAlias"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Resource Name (ARN) of a KMS key Alias
     *         </p>
     *         <p>
     *         <code>"arn:aws:kms:us-west-2:111122223333:alias/ExampleAlias"</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you use a KMS key ID or an alias of your KMS key, the Amazon SageMaker execution role must include
     *         permissions to call <code>kms:Encrypt</code>. If you don't provide a KMS key ID, Amazon SageMaker uses
     *         the default KMS key for Amazon S3 for your role's account. Amazon SageMaker uses server-side encryption
     *         with KMS managed keys for <code>OutputDataConfig</code>. If you use a bucket policy with an
     *         <code>s3:PutObject</code> permission that only allows objects with server-side encryption, set the
     *         condition key of <code>s3:x-amz-server-side-encryption</code> to <code>"aws:kms"</code>. For more
     *         information, see <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html">KMS
     *         managed Encryption Keys</a> in the <i>Amazon Simple Storage Service Developer Guide.</i>
     *         </p>
     *         <p>
     *         The KMS key policy must grant permission to the IAM role that you specify in your
     *         <code>CreateEndpoint</code> and <code>UpdateEndpoint</code> requests. For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html">Using Key Policies in
     *         Amazon Web Services KMS</a> in the <i>Amazon Web Services Key Management Service Developer Guide</i>.
     */
    public final String kmsKey() {
        return kmsKey;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Array of key-value pairs. You can use tags to categorize your Amazon Web Services resources in different ways,
     * for example, by purpose, owner, or environment. For more information, see <a
     * href="https://docs.aws.amazon.com/ARG/latest/userguide/tagging.html">Tagging your Amazon Web Services
     * Resources</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Array of key-value pairs. You can use tags to categorize your Amazon Web Services resources in different
     *         ways, for example, by purpose, owner, or environment. For more information, see <a
     *         href="https://docs.aws.amazon.com/ARG/latest/userguide/tagging.html">Tagging your Amazon Web Services
     *         Resources</a>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(schedule());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(endpointName());
        hashCode = 31 * hashCode + Objects.hashCode(hasModelVariants() ? modelVariants() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dataStorageConfig());
        hashCode = 31 * hashCode + Objects.hashCode(shadowModeConfig());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKey());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateInferenceExperimentRequest)) {
            return false;
        }
        CreateInferenceExperimentRequest other = (CreateInferenceExperimentRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(schedule(), other.schedule()) && Objects.equals(description(), other.description())
                && Objects.equals(roleArn(), other.roleArn()) && Objects.equals(endpointName(), other.endpointName())
                && hasModelVariants() == other.hasModelVariants() && Objects.equals(modelVariants(), other.modelVariants())
                && Objects.equals(dataStorageConfig(), other.dataStorageConfig())
                && Objects.equals(shadowModeConfig(), other.shadowModeConfig()) && Objects.equals(kmsKey(), other.kmsKey())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateInferenceExperimentRequest").add("Name", name()).add("Type", typeAsString())
                .add("Schedule", schedule()).add("Description", description()).add("RoleArn", roleArn())
                .add("EndpointName", endpointName()).add("ModelVariants", hasModelVariants() ? modelVariants() : null)
                .add("DataStorageConfig", dataStorageConfig()).add("ShadowModeConfig", shadowModeConfig())
                .add("KmsKey", kmsKey()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Schedule":
            return Optional.ofNullable(clazz.cast(schedule()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "EndpointName":
            return Optional.ofNullable(clazz.cast(endpointName()));
        case "ModelVariants":
            return Optional.ofNullable(clazz.cast(modelVariants()));
        case "DataStorageConfig":
            return Optional.ofNullable(clazz.cast(dataStorageConfig()));
        case "ShadowModeConfig":
            return Optional.ofNullable(clazz.cast(shadowModeConfig()));
        case "KmsKey":
            return Optional.ofNullable(clazz.cast(kmsKey()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<CreateInferenceExperimentRequest, T> g) {
        return obj -> g.apply((CreateInferenceExperimentRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SageMakerRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, CreateInferenceExperimentRequest> {
        /**
         * <p>
         * The name for the inference experiment.
         * </p>
         * 
         * @param name
         *        The name for the inference experiment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of the inference experiment that you want to run. The following types of experiments are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ShadowMode</code>: You can use this type to validate a shadow variant. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/shadow-tests.html">Shadow tests</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of the inference experiment that you want to run. The following types of experiments are
         *        possible: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ShadowMode</code>: You can use this type to validate a shadow variant. For more information, see
         *        <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/shadow-tests.html">Shadow tests</a>.
         *        </p>
         *        </li>
         * @see InferenceExperimentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InferenceExperimentType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the inference experiment that you want to run. The following types of experiments are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ShadowMode</code>: You can use this type to validate a shadow variant. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/shadow-tests.html">Shadow tests</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of the inference experiment that you want to run. The following types of experiments are
         *        possible: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ShadowMode</code>: You can use this type to validate a shadow variant. For more information, see
         *        <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/shadow-tests.html">Shadow tests</a>.
         *        </p>
         *        </li>
         * @see InferenceExperimentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InferenceExperimentType
         */
        Builder type(InferenceExperimentType type);

        /**
         * <p>
         * The duration for which you want the inference experiment to run. If you don't specify this field, the
         * experiment automatically starts immediately upon creation and concludes after 7 days.
         * </p>
         * 
         * @param schedule
         *        The duration for which you want the inference experiment to run. If you don't specify this field, the
         *        experiment automatically starts immediately upon creation and concludes after 7 days.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schedule(InferenceExperimentSchedule schedule);

        /**
         * <p>
         * The duration for which you want the inference experiment to run. If you don't specify this field, the
         * experiment automatically starts immediately upon creation and concludes after 7 days.
         * </p>
         * This is a convenience method that creates an instance of the {@link InferenceExperimentSchedule.Builder}
         * avoiding the need to create one manually via {@link InferenceExperimentSchedule#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InferenceExperimentSchedule.Builder#build()} is called
         * immediately and its result is passed to {@link #schedule(InferenceExperimentSchedule)}.
         * 
         * @param schedule
         *        a consumer that will call methods on {@link InferenceExperimentSchedule.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schedule(InferenceExperimentSchedule)
         */
        default Builder schedule(Consumer<InferenceExperimentSchedule.Builder> schedule) {
            return schedule(InferenceExperimentSchedule.builder().applyMutation(schedule).build());
        }

        /**
         * <p>
         * A description for the inference experiment.
         * </p>
         * 
         * @param description
         *        A description for the inference experiment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The ARN of the IAM role that Amazon SageMaker can assume to access model artifacts and container images, and
         * manage Amazon SageMaker Inference endpoints for model deployment.
         * </p>
         * 
         * @param roleArn
         *        The ARN of the IAM role that Amazon SageMaker can assume to access model artifacts and container
         *        images, and manage Amazon SageMaker Inference endpoints for model deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The name of the Amazon SageMaker endpoint on which you want to run the inference experiment.
         * </p>
         * 
         * @param endpointName
         *        The name of the Amazon SageMaker endpoint on which you want to run the inference experiment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointName(String endpointName);

        /**
         * <p>
         * An array of <code>ModelVariantConfig</code> objects. There is one for each variant in the inference
         * experiment. Each <code>ModelVariantConfig</code> object in the array describes the infrastructure
         * configuration for the corresponding variant.
         * </p>
         * 
         * @param modelVariants
         *        An array of <code>ModelVariantConfig</code> objects. There is one for each variant in the inference
         *        experiment. Each <code>ModelVariantConfig</code> object in the array describes the infrastructure
         *        configuration for the corresponding variant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelVariants(Collection<ModelVariantConfig> modelVariants);

        /**
         * <p>
         * An array of <code>ModelVariantConfig</code> objects. There is one for each variant in the inference
         * experiment. Each <code>ModelVariantConfig</code> object in the array describes the infrastructure
         * configuration for the corresponding variant.
         * </p>
         * 
         * @param modelVariants
         *        An array of <code>ModelVariantConfig</code> objects. There is one for each variant in the inference
         *        experiment. Each <code>ModelVariantConfig</code> object in the array describes the infrastructure
         *        configuration for the corresponding variant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelVariants(ModelVariantConfig... modelVariants);

        /**
         * <p>
         * An array of <code>ModelVariantConfig</code> objects. There is one for each variant in the inference
         * experiment. Each <code>ModelVariantConfig</code> object in the array describes the infrastructure
         * configuration for the corresponding variant.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.ModelVariantConfig.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.sagemaker.model.ModelVariantConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.ModelVariantConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #modelVariants(List<ModelVariantConfig>)}.
         * 
         * @param modelVariants
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.ModelVariantConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modelVariants(java.util.Collection<ModelVariantConfig>)
         */
        Builder modelVariants(Consumer<ModelVariantConfig.Builder>... modelVariants);

        /**
         * <p>
         * The Amazon S3 location and configuration for storing inference request and response data.
         * </p>
         * <p>
         * This is an optional parameter that you can use for data capture. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor-data-capture.html">Capture data</a>.
         * </p>
         * 
         * @param dataStorageConfig
         *        The Amazon S3 location and configuration for storing inference request and response data. </p>
         *        <p>
         *        This is an optional parameter that you can use for data capture. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor-data-capture.html">Capture
         *        data</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataStorageConfig(InferenceExperimentDataStorageConfig dataStorageConfig);

        /**
         * <p>
         * The Amazon S3 location and configuration for storing inference request and response data.
         * </p>
         * <p>
         * This is an optional parameter that you can use for data capture. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor-data-capture.html">Capture data</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link InferenceExperimentDataStorageConfig.Builder} avoiding the need to create one manually via
         * {@link InferenceExperimentDataStorageConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InferenceExperimentDataStorageConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #dataStorageConfig(InferenceExperimentDataStorageConfig)}.
         * 
         * @param dataStorageConfig
         *        a consumer that will call methods on {@link InferenceExperimentDataStorageConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataStorageConfig(InferenceExperimentDataStorageConfig)
         */
        default Builder dataStorageConfig(Consumer<InferenceExperimentDataStorageConfig.Builder> dataStorageConfig) {
            return dataStorageConfig(InferenceExperimentDataStorageConfig.builder().applyMutation(dataStorageConfig).build());
        }

        /**
         * <p>
         * The configuration of <code>ShadowMode</code> inference experiment type. Use this field to specify a
         * production variant which takes all the inference requests, and a shadow variant to which Amazon SageMaker
         * replicates a percentage of the inference requests. For the shadow variant also specify the percentage of
         * requests that Amazon SageMaker replicates.
         * </p>
         * 
         * @param shadowModeConfig
         *        The configuration of <code>ShadowMode</code> inference experiment type. Use this field to specify a
         *        production variant which takes all the inference requests, and a shadow variant to which Amazon
         *        SageMaker replicates a percentage of the inference requests. For the shadow variant also specify the
         *        percentage of requests that Amazon SageMaker replicates.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shadowModeConfig(ShadowModeConfig shadowModeConfig);

        /**
         * <p>
         * The configuration of <code>ShadowMode</code> inference experiment type. Use this field to specify a
         * production variant which takes all the inference requests, and a shadow variant to which Amazon SageMaker
         * replicates a percentage of the inference requests. For the shadow variant also specify the percentage of
         * requests that Amazon SageMaker replicates.
         * </p>
         * This is a convenience method that creates an instance of the {@link ShadowModeConfig.Builder} avoiding the
         * need to create one manually via {@link ShadowModeConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ShadowModeConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #shadowModeConfig(ShadowModeConfig)}.
         * 
         * @param shadowModeConfig
         *        a consumer that will call methods on {@link ShadowModeConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #shadowModeConfig(ShadowModeConfig)
         */
        default Builder shadowModeConfig(Consumer<ShadowModeConfig.Builder> shadowModeConfig) {
            return shadowModeConfig(ShadowModeConfig.builder().applyMutation(shadowModeConfig).build());
        }

        /**
         * <p>
         * The Amazon Web Services Key Management Service (Amazon Web Services KMS) key that Amazon SageMaker uses to
         * encrypt data on the storage volume attached to the ML compute instance that hosts the endpoint. The
         * <code>KmsKey</code> can be any of the following formats:
         * </p>
         * <ul>
         * <li>
         * <p>
         * KMS key ID
         * </p>
         * <p>
         * <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Resource Name (ARN) of a KMS key
         * </p>
         * <p>
         * <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * KMS key Alias
         * </p>
         * <p>
         * <code>"alias/ExampleAlias"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Resource Name (ARN) of a KMS key Alias
         * </p>
         * <p>
         * <code>"arn:aws:kms:us-west-2:111122223333:alias/ExampleAlias"</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you use a KMS key ID or an alias of your KMS key, the Amazon SageMaker execution role must include
         * permissions to call <code>kms:Encrypt</code>. If you don't provide a KMS key ID, Amazon SageMaker uses the
         * default KMS key for Amazon S3 for your role's account. Amazon SageMaker uses server-side encryption with KMS
         * managed keys for <code>OutputDataConfig</code>. If you use a bucket policy with an <code>s3:PutObject</code>
         * permission that only allows objects with server-side encryption, set the condition key of
         * <code>s3:x-amz-server-side-encryption</code> to <code>"aws:kms"</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html">KMS managed Encryption
         * Keys</a> in the <i>Amazon Simple Storage Service Developer Guide.</i>
         * </p>
         * <p>
         * The KMS key policy must grant permission to the IAM role that you specify in your <code>CreateEndpoint</code>
         * and <code>UpdateEndpoint</code> requests. For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html">Using Key Policies in Amazon
         * Web Services KMS</a> in the <i>Amazon Web Services Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param kmsKey
         *        The Amazon Web Services Key Management Service (Amazon Web Services KMS) key that Amazon SageMaker
         *        uses to encrypt data on the storage volume attached to the ML compute instance that hosts the
         *        endpoint. The <code>KmsKey</code> can be any of the following formats: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        KMS key ID
         *        </p>
         *        <p>
         *        <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Resource Name (ARN) of a KMS key
         *        </p>
         *        <p>
         *        <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        KMS key Alias
         *        </p>
         *        <p>
         *        <code>"alias/ExampleAlias"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Resource Name (ARN) of a KMS key Alias
         *        </p>
         *        <p>
         *        <code>"arn:aws:kms:us-west-2:111122223333:alias/ExampleAlias"</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you use a KMS key ID or an alias of your KMS key, the Amazon SageMaker execution role must include
         *        permissions to call <code>kms:Encrypt</code>. If you don't provide a KMS key ID, Amazon SageMaker uses
         *        the default KMS key for Amazon S3 for your role's account. Amazon SageMaker uses server-side
         *        encryption with KMS managed keys for <code>OutputDataConfig</code>. If you use a bucket policy with an
         *        <code>s3:PutObject</code> permission that only allows objects with server-side encryption, set the
         *        condition key of <code>s3:x-amz-server-side-encryption</code> to <code>"aws:kms"</code>. For more
         *        information, see <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html">KMS
         *        managed Encryption Keys</a> in the <i>Amazon Simple Storage Service Developer Guide.</i>
         *        </p>
         *        <p>
         *        The KMS key policy must grant permission to the IAM role that you specify in your
         *        <code>CreateEndpoint</code> and <code>UpdateEndpoint</code> requests. For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html">Using Key Policies in
         *        Amazon Web Services KMS</a> in the <i>Amazon Web Services Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKey(String kmsKey);

        /**
         * <p>
         * Array of key-value pairs. You can use tags to categorize your Amazon Web Services resources in different
         * ways, for example, by purpose, owner, or environment. For more information, see <a
         * href="https://docs.aws.amazon.com/ARG/latest/userguide/tagging.html">Tagging your Amazon Web Services
         * Resources</a>.
         * </p>
         * 
         * @param tags
         *        Array of key-value pairs. You can use tags to categorize your Amazon Web Services resources in
         *        different ways, for example, by purpose, owner, or environment. For more information, see <a
         *        href="https://docs.aws.amazon.com/ARG/latest/userguide/tagging.html">Tagging your Amazon Web Services
         *        Resources</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Array of key-value pairs. You can use tags to categorize your Amazon Web Services resources in different
         * ways, for example, by purpose, owner, or environment. For more information, see <a
         * href="https://docs.aws.amazon.com/ARG/latest/userguide/tagging.html">Tagging your Amazon Web Services
         * Resources</a>.
         * </p>
         * 
         * @param tags
         *        Array of key-value pairs. You can use tags to categorize your Amazon Web Services resources in
         *        different ways, for example, by purpose, owner, or environment. For more information, see <a
         *        href="https://docs.aws.amazon.com/ARG/latest/userguide/tagging.html">Tagging your Amazon Web Services
         *        Resources</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Array of key-value pairs. You can use tags to categorize your Amazon Web Services resources in different
         * ways, for example, by purpose, owner, or environment. For more information, see <a
         * href="https://docs.aws.amazon.com/ARG/latest/userguide/tagging.html">Tagging your Amazon Web Services
         * Resources</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.sagemaker.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SageMakerRequest.BuilderImpl implements Builder {
        private String name;

        private String type;

        private InferenceExperimentSchedule schedule;

        private String description;

        private String roleArn;

        private String endpointName;

        private List<ModelVariantConfig> modelVariants = DefaultSdkAutoConstructList.getInstance();

        private InferenceExperimentDataStorageConfig dataStorageConfig;

        private ShadowModeConfig shadowModeConfig;

        private String kmsKey;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateInferenceExperimentRequest model) {
            super(model);
            name(model.name);
            type(model.type);
            schedule(model.schedule);
            description(model.description);
            roleArn(model.roleArn);
            endpointName(model.endpointName);
            modelVariants(model.modelVariants);
            dataStorageConfig(model.dataStorageConfig);
            shadowModeConfig(model.shadowModeConfig);
            kmsKey(model.kmsKey);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(InferenceExperimentType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final InferenceExperimentSchedule.Builder getSchedule() {
            return schedule != null ? schedule.toBuilder() : null;
        }

        public final void setSchedule(InferenceExperimentSchedule.BuilderImpl schedule) {
            this.schedule = schedule != null ? schedule.build() : null;
        }

        @Override
        public final Builder schedule(InferenceExperimentSchedule schedule) {
            this.schedule = schedule;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getEndpointName() {
            return endpointName;
        }

        public final void setEndpointName(String endpointName) {
            this.endpointName = endpointName;
        }

        @Override
        public final Builder endpointName(String endpointName) {
            this.endpointName = endpointName;
            return this;
        }

        public final List<ModelVariantConfig.Builder> getModelVariants() {
            List<ModelVariantConfig.Builder> result = ModelVariantConfigListCopier.copyToBuilder(this.modelVariants);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setModelVariants(Collection<ModelVariantConfig.BuilderImpl> modelVariants) {
            this.modelVariants = ModelVariantConfigListCopier.copyFromBuilder(modelVariants);
        }

        @Override
        public final Builder modelVariants(Collection<ModelVariantConfig> modelVariants) {
            this.modelVariants = ModelVariantConfigListCopier.copy(modelVariants);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder modelVariants(ModelVariantConfig... modelVariants) {
            modelVariants(Arrays.asList(modelVariants));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder modelVariants(Consumer<ModelVariantConfig.Builder>... modelVariants) {
            modelVariants(Stream.of(modelVariants).map(c -> ModelVariantConfig.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final InferenceExperimentDataStorageConfig.Builder getDataStorageConfig() {
            return dataStorageConfig != null ? dataStorageConfig.toBuilder() : null;
        }

        public final void setDataStorageConfig(InferenceExperimentDataStorageConfig.BuilderImpl dataStorageConfig) {
            this.dataStorageConfig = dataStorageConfig != null ? dataStorageConfig.build() : null;
        }

        @Override
        public final Builder dataStorageConfig(InferenceExperimentDataStorageConfig dataStorageConfig) {
            this.dataStorageConfig = dataStorageConfig;
            return this;
        }

        public final ShadowModeConfig.Builder getShadowModeConfig() {
            return shadowModeConfig != null ? shadowModeConfig.toBuilder() : null;
        }

        public final void setShadowModeConfig(ShadowModeConfig.BuilderImpl shadowModeConfig) {
            this.shadowModeConfig = shadowModeConfig != null ? shadowModeConfig.build() : null;
        }

        @Override
        public final Builder shadowModeConfig(ShadowModeConfig shadowModeConfig) {
            this.shadowModeConfig = shadowModeConfig;
            return this;
        }

        public final String getKmsKey() {
            return kmsKey;
        }

        public final void setKmsKey(String kmsKey) {
            this.kmsKey = kmsKey;
        }

        @Override
        public final Builder kmsKey(String kmsKey) {
            this.kmsKey = kmsKey;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateInferenceExperimentRequest build() {
            return new CreateInferenceExperimentRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
