/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for <code>Hyperband</code>, a multi-fidelity based hyperparameter tuning strategy.
 * <code>Hyperband</code> uses the final and intermediate results of a training job to dynamically allocate resources to
 * utilized hyperparameter configurations while automatically stopping under-performing configurations. This parameter
 * should be provided only if <code>Hyperband</code> is selected as the <code>StrategyConfig</code> under the
 * <code>HyperParameterTuningJobConfig</code> API.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HyperbandStrategyConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<HyperbandStrategyConfig.Builder, HyperbandStrategyConfig> {
    private static final SdkField<Integer> MIN_RESOURCE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinResource").getter(getter(HyperbandStrategyConfig::minResource)).setter(setter(Builder::minResource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinResource").build()).build();

    private static final SdkField<Integer> MAX_RESOURCE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxResource").getter(getter(HyperbandStrategyConfig::maxResource)).setter(setter(Builder::maxResource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxResource").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MIN_RESOURCE_FIELD,
            MAX_RESOURCE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("MinResource", MIN_RESOURCE_FIELD);
                    put("MaxResource", MAX_RESOURCE_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final Integer minResource;

    private final Integer maxResource;

    private HyperbandStrategyConfig(BuilderImpl builder) {
        this.minResource = builder.minResource;
        this.maxResource = builder.maxResource;
    }

    /**
     * <p>
     * The minimum number of resources (such as epochs) that can be used by a training job launched by a hyperparameter
     * tuning job. If the value for <code>MinResource</code> has not been reached, the training job is not stopped by
     * <code>Hyperband</code>.
     * </p>
     * 
     * @return The minimum number of resources (such as epochs) that can be used by a training job launched by a
     *         hyperparameter tuning job. If the value for <code>MinResource</code> has not been reached, the training
     *         job is not stopped by <code>Hyperband</code>.
     */
    public final Integer minResource() {
        return minResource;
    }

    /**
     * <p>
     * The maximum number of resources (such as epochs) that can be used by a training job launched by a hyperparameter
     * tuning job. Once a job reaches the <code>MaxResource</code> value, it is stopped. If a value for
     * <code>MaxResource</code> is not provided, and <code>Hyperband</code> is selected as the hyperparameter tuning
     * strategy, <code>HyperbandTraining</code> attempts to infer <code>MaxResource</code> from the following keys (if
     * present) in <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_HyperParameterTrainingJobDefinition.html#sagemaker-Type-HyperParameterTrainingJobDefinition-StaticHyperParameters"
     * >StaticsHyperParameters</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>epochs</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>numepochs</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>n-epochs</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>n_epochs</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>num_epochs</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If <code>HyperbandStrategyConfig</code> is unable to infer a value for <code>MaxResource</code>, it generates a
     * validation error. The maximum value is 20,000 epochs. All metrics that correspond to an objective metric are used
     * to derive <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">early stopping
     * decisions</a>. For <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/distributed-training.html">distributed</a> training jobs,
     * ensure that duplicate metrics are not printed in the logs across the individual nodes in a training job. If
     * multiple nodes are publishing duplicate or incorrect metrics, training jobs may make an incorrect stopping
     * decision and stop the job prematurely.
     * </p>
     * 
     * @return The maximum number of resources (such as epochs) that can be used by a training job launched by a
     *         hyperparameter tuning job. Once a job reaches the <code>MaxResource</code> value, it is stopped. If a
     *         value for <code>MaxResource</code> is not provided, and <code>Hyperband</code> is selected as the
     *         hyperparameter tuning strategy, <code>HyperbandTraining</code> attempts to infer <code>MaxResource</code>
     *         from the following keys (if present) in <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_HyperParameterTrainingJobDefinition.html#sagemaker-Type-HyperParameterTrainingJobDefinition-StaticHyperParameters"
     *         >StaticsHyperParameters</a>:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>epochs</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>numepochs</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>n-epochs</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>n_epochs</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>num_epochs</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If <code>HyperbandStrategyConfig</code> is unable to infer a value for <code>MaxResource</code>, it
     *         generates a validation error. The maximum value is 20,000 epochs. All metrics that correspond to an
     *         objective metric are used to derive <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">early
     *         stopping decisions</a>. For <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/distributed-training.html">distributed</a> training
     *         jobs, ensure that duplicate metrics are not printed in the logs across the individual nodes in a training
     *         job. If multiple nodes are publishing duplicate or incorrect metrics, training jobs may make an incorrect
     *         stopping decision and stop the job prematurely.
     */
    public final Integer maxResource() {
        return maxResource;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(minResource());
        hashCode = 31 * hashCode + Objects.hashCode(maxResource());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HyperbandStrategyConfig)) {
            return false;
        }
        HyperbandStrategyConfig other = (HyperbandStrategyConfig) obj;
        return Objects.equals(minResource(), other.minResource()) && Objects.equals(maxResource(), other.maxResource());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HyperbandStrategyConfig").add("MinResource", minResource()).add("MaxResource", maxResource())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MinResource":
            return Optional.ofNullable(clazz.cast(minResource()));
        case "MaxResource":
            return Optional.ofNullable(clazz.cast(maxResource()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<HyperbandStrategyConfig, T> g) {
        return obj -> g.apply((HyperbandStrategyConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HyperbandStrategyConfig> {
        /**
         * <p>
         * The minimum number of resources (such as epochs) that can be used by a training job launched by a
         * hyperparameter tuning job. If the value for <code>MinResource</code> has not been reached, the training job
         * is not stopped by <code>Hyperband</code>.
         * </p>
         * 
         * @param minResource
         *        The minimum number of resources (such as epochs) that can be used by a training job launched by a
         *        hyperparameter tuning job. If the value for <code>MinResource</code> has not been reached, the
         *        training job is not stopped by <code>Hyperband</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minResource(Integer minResource);

        /**
         * <p>
         * The maximum number of resources (such as epochs) that can be used by a training job launched by a
         * hyperparameter tuning job. Once a job reaches the <code>MaxResource</code> value, it is stopped. If a value
         * for <code>MaxResource</code> is not provided, and <code>Hyperband</code> is selected as the hyperparameter
         * tuning strategy, <code>HyperbandTraining</code> attempts to infer <code>MaxResource</code> from the following
         * keys (if present) in <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_HyperParameterTrainingJobDefinition.html#sagemaker-Type-HyperParameterTrainingJobDefinition-StaticHyperParameters"
         * >StaticsHyperParameters</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>epochs</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>numepochs</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>n-epochs</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>n_epochs</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>num_epochs</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * If <code>HyperbandStrategyConfig</code> is unable to infer a value for <code>MaxResource</code>, it generates
         * a validation error. The maximum value is 20,000 epochs. All metrics that correspond to an objective metric
         * are used to derive <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">early
         * stopping decisions</a>. For <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/distributed-training.html">distributed</a> training
         * jobs, ensure that duplicate metrics are not printed in the logs across the individual nodes in a training
         * job. If multiple nodes are publishing duplicate or incorrect metrics, training jobs may make an incorrect
         * stopping decision and stop the job prematurely.
         * </p>
         * 
         * @param maxResource
         *        The maximum number of resources (such as epochs) that can be used by a training job launched by a
         *        hyperparameter tuning job. Once a job reaches the <code>MaxResource</code> value, it is stopped. If a
         *        value for <code>MaxResource</code> is not provided, and <code>Hyperband</code> is selected as the
         *        hyperparameter tuning strategy, <code>HyperbandTraining</code> attempts to infer
         *        <code>MaxResource</code> from the following keys (if present) in <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_HyperParameterTrainingJobDefinition.html#sagemaker-Type-HyperParameterTrainingJobDefinition-StaticHyperParameters"
         *        >StaticsHyperParameters</a>:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>epochs</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>numepochs</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>n-epochs</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>n_epochs</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>num_epochs</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If <code>HyperbandStrategyConfig</code> is unable to infer a value for <code>MaxResource</code>, it
         *        generates a validation error. The maximum value is 20,000 epochs. All metrics that correspond to an
         *        objective metric are used to derive <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html"
         *        >early stopping decisions</a>. For <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/distributed-training.html">distributed</a>
         *        training jobs, ensure that duplicate metrics are not printed in the logs across the individual nodes
         *        in a training job. If multiple nodes are publishing duplicate or incorrect metrics, training jobs may
         *        make an incorrect stopping decision and stop the job prematurely.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxResource(Integer maxResource);
    }

    static final class BuilderImpl implements Builder {
        private Integer minResource;

        private Integer maxResource;

        private BuilderImpl() {
        }

        private BuilderImpl(HyperbandStrategyConfig model) {
            minResource(model.minResource);
            maxResource(model.maxResource);
        }

        public final Integer getMinResource() {
            return minResource;
        }

        public final void setMinResource(Integer minResource) {
            this.minResource = minResource;
        }

        @Override
        public final Builder minResource(Integer minResource) {
            this.minResource = minResource;
            return this;
        }

        public final Integer getMaxResource() {
            return maxResource;
        }

        public final void setMaxResource(Integer maxResource) {
            this.maxResource = maxResource;
        }

        @Override
        public final Builder maxResource(Integer maxResource) {
            this.maxResource = maxResource;
            return this;
        }

        @Override
        public HyperbandStrategyConfig build() {
            return new HyperbandStrategyConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
