/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A list of nested <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_Filter.html">Filter</a>
 * objects. A resource must satisfy the conditions of all filters to be included in the results returned from the <a
 * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_Search.html">Search</a> API.
 * </p>
 * <p>
 * For example, to filter on a training job's <code>InputDataConfig</code> property with a specific channel name and
 * <code>S3Uri</code> prefix, define the following filters:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>'{Name:"InputDataConfig.ChannelName", "Operator":"Equals", "Value":"train"}',</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>'{Name:"InputDataConfig.DataSource.S3DataSource.S3Uri", "Operator":"Contains", "Value":"mybucket/catdata"}'</code>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NestedFilters implements SdkPojo, Serializable, ToCopyableBuilder<NestedFilters.Builder, NestedFilters> {
    private static final SdkField<String> NESTED_PROPERTY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NestedPropertyName").getter(getter(NestedFilters::nestedPropertyName))
            .setter(setter(Builder::nestedPropertyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NestedPropertyName").build())
            .build();

    private static final SdkField<List<Filter>> FILTERS_FIELD = SdkField
            .<List<Filter>> builder(MarshallingType.LIST)
            .memberName("Filters")
            .getter(getter(NestedFilters::filters))
            .setter(setter(Builder::filters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Filters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Filter> builder(MarshallingType.SDK_POJO)
                                            .constructor(Filter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NESTED_PROPERTY_NAME_FIELD,
            FILTERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("NestedPropertyName", NESTED_PROPERTY_NAME_FIELD);
                    put("Filters", FILTERS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String nestedPropertyName;

    private final List<Filter> filters;

    private NestedFilters(BuilderImpl builder) {
        this.nestedPropertyName = builder.nestedPropertyName;
        this.filters = builder.filters;
    }

    /**
     * <p>
     * The name of the property to use in the nested filters. The value must match a listed property name, such as
     * <code>InputDataConfig</code>.
     * </p>
     * 
     * @return The name of the property to use in the nested filters. The value must match a listed property name, such
     *         as <code>InputDataConfig</code>.
     */
    public final String nestedPropertyName() {
        return nestedPropertyName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Filters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFilters() {
        return filters != null && !(filters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of filters. Each filter acts on a property. Filters must contain at least one <code>Filters</code> value.
     * For example, a <code>NestedFilters</code> call might include a filter on the <code>PropertyName</code> parameter
     * of the <code>InputDataConfig</code> property: <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFilters} method.
     * </p>
     * 
     * @return A list of filters. Each filter acts on a property. Filters must contain at least one <code>Filters</code>
     *         value. For example, a <code>NestedFilters</code> call might include a filter on the
     *         <code>PropertyName</code> parameter of the <code>InputDataConfig</code> property:
     *         <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
     */
    public final List<Filter> filters() {
        return filters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nestedPropertyName());
        hashCode = 31 * hashCode + Objects.hashCode(hasFilters() ? filters() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NestedFilters)) {
            return false;
        }
        NestedFilters other = (NestedFilters) obj;
        return Objects.equals(nestedPropertyName(), other.nestedPropertyName()) && hasFilters() == other.hasFilters()
                && Objects.equals(filters(), other.filters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NestedFilters").add("NestedPropertyName", nestedPropertyName())
                .add("Filters", hasFilters() ? filters() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NestedPropertyName":
            return Optional.ofNullable(clazz.cast(nestedPropertyName()));
        case "Filters":
            return Optional.ofNullable(clazz.cast(filters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<NestedFilters, T> g) {
        return obj -> g.apply((NestedFilters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NestedFilters> {
        /**
         * <p>
         * The name of the property to use in the nested filters. The value must match a listed property name, such as
         * <code>InputDataConfig</code>.
         * </p>
         * 
         * @param nestedPropertyName
         *        The name of the property to use in the nested filters. The value must match a listed property name,
         *        such as <code>InputDataConfig</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nestedPropertyName(String nestedPropertyName);

        /**
         * <p>
         * A list of filters. Each filter acts on a property. Filters must contain at least one <code>Filters</code>
         * value. For example, a <code>NestedFilters</code> call might include a filter on the <code>PropertyName</code>
         * parameter of the <code>InputDataConfig</code> property:
         * <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
         * </p>
         * 
         * @param filters
         *        A list of filters. Each filter acts on a property. Filters must contain at least one
         *        <code>Filters</code> value. For example, a <code>NestedFilters</code> call might include a filter on
         *        the <code>PropertyName</code> parameter of the <code>InputDataConfig</code> property:
         *        <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Collection<Filter> filters);

        /**
         * <p>
         * A list of filters. Each filter acts on a property. Filters must contain at least one <code>Filters</code>
         * value. For example, a <code>NestedFilters</code> call might include a filter on the <code>PropertyName</code>
         * parameter of the <code>InputDataConfig</code> property:
         * <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
         * </p>
         * 
         * @param filters
         *        A list of filters. Each filter acts on a property. Filters must contain at least one
         *        <code>Filters</code> value. For example, a <code>NestedFilters</code> call might include a filter on
         *        the <code>PropertyName</code> parameter of the <code>InputDataConfig</code> property:
         *        <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Filter... filters);

        /**
         * <p>
         * A list of filters. Each filter acts on a property. Filters must contain at least one <code>Filters</code>
         * value. For example, a <code>NestedFilters</code> call might include a filter on the <code>PropertyName</code>
         * parameter of the <code>InputDataConfig</code> property:
         * <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.Filter.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.sagemaker.model.Filter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.Filter.Builder#build()} is called immediately and its
         * result is passed to {@link #filters(List<Filter>)}.
         * 
         * @param filters
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.Filter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filters(java.util.Collection<Filter>)
         */
        Builder filters(Consumer<Filter.Builder>... filters);
    }

    static final class BuilderImpl implements Builder {
        private String nestedPropertyName;

        private List<Filter> filters = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NestedFilters model) {
            nestedPropertyName(model.nestedPropertyName);
            filters(model.filters);
        }

        public final String getNestedPropertyName() {
            return nestedPropertyName;
        }

        public final void setNestedPropertyName(String nestedPropertyName) {
            this.nestedPropertyName = nestedPropertyName;
        }

        @Override
        public final Builder nestedPropertyName(String nestedPropertyName) {
            this.nestedPropertyName = nestedPropertyName;
            return this;
        }

        public final List<Filter.Builder> getFilters() {
            List<Filter.Builder> result = FilterListCopier.copyToBuilder(this.filters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFilters(Collection<Filter.BuilderImpl> filters) {
            this.filters = FilterListCopier.copyFromBuilder(filters);
        }

        @Override
        public final Builder filters(Collection<Filter> filters) {
            this.filters = FilterListCopier.copy(filters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Filter... filters) {
            filters(Arrays.asList(filters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Consumer<Filter.Builder>... filters) {
            filters(Stream.of(filters).map(c -> Filter.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public NestedFilters build() {
            return new NestedFilters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
