/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the resources, including machine learning (ML) compute instances and ML storage volumes, to use for model
 * training.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceConfig implements SdkPojo, Serializable, ToCopyableBuilder<ResourceConfig.Builder, ResourceConfig> {
    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(ResourceConfig::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Integer> INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstanceCount").getter(getter(ResourceConfig::instanceCount)).setter(setter(Builder::instanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceCount").build()).build();

    private static final SdkField<Integer> VOLUME_SIZE_IN_GB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("VolumeSizeInGB").getter(getter(ResourceConfig::volumeSizeInGB)).setter(setter(Builder::volumeSizeInGB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeSizeInGB").build()).build();

    private static final SdkField<String> VOLUME_KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VolumeKmsKeyId").getter(getter(ResourceConfig::volumeKmsKeyId)).setter(setter(Builder::volumeKmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeKmsKeyId").build()).build();

    private static final SdkField<Integer> KEEP_ALIVE_PERIOD_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("KeepAlivePeriodInSeconds")
            .getter(getter(ResourceConfig::keepAlivePeriodInSeconds)).setter(setter(Builder::keepAlivePeriodInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeepAlivePeriodInSeconds").build())
            .build();

    private static final SdkField<List<InstanceGroup>> INSTANCE_GROUPS_FIELD = SdkField
            .<List<InstanceGroup>> builder(MarshallingType.LIST)
            .memberName("InstanceGroups")
            .getter(getter(ResourceConfig::instanceGroups))
            .setter(setter(Builder::instanceGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InstanceGroup> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceGroup::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TRAINING_PLAN_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TrainingPlanArn").getter(getter(ResourceConfig::trainingPlanArn))
            .setter(setter(Builder::trainingPlanArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingPlanArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_TYPE_FIELD,
            INSTANCE_COUNT_FIELD, VOLUME_SIZE_IN_GB_FIELD, VOLUME_KMS_KEY_ID_FIELD, KEEP_ALIVE_PERIOD_IN_SECONDS_FIELD,
            INSTANCE_GROUPS_FIELD, TRAINING_PLAN_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("InstanceType", INSTANCE_TYPE_FIELD);
                    put("InstanceCount", INSTANCE_COUNT_FIELD);
                    put("VolumeSizeInGB", VOLUME_SIZE_IN_GB_FIELD);
                    put("VolumeKmsKeyId", VOLUME_KMS_KEY_ID_FIELD);
                    put("KeepAlivePeriodInSeconds", KEEP_ALIVE_PERIOD_IN_SECONDS_FIELD);
                    put("InstanceGroups", INSTANCE_GROUPS_FIELD);
                    put("TrainingPlanArn", TRAINING_PLAN_ARN_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String instanceType;

    private final Integer instanceCount;

    private final Integer volumeSizeInGB;

    private final String volumeKmsKeyId;

    private final Integer keepAlivePeriodInSeconds;

    private final List<InstanceGroup> instanceGroups;

    private final String trainingPlanArn;

    private ResourceConfig(BuilderImpl builder) {
        this.instanceType = builder.instanceType;
        this.instanceCount = builder.instanceCount;
        this.volumeSizeInGB = builder.volumeSizeInGB;
        this.volumeKmsKeyId = builder.volumeKmsKeyId;
        this.keepAlivePeriodInSeconds = builder.keepAlivePeriodInSeconds;
        this.instanceGroups = builder.instanceGroups;
        this.trainingPlanArn = builder.trainingPlanArn;
    }

    /**
     * <p>
     * The ML compute instance type.
     * </p>
     * <note>
     * <p>
     * SageMaker Training on Amazon Elastic Compute Cloud (EC2) P4de instances is in preview release starting December
     * 9th, 2022.
     * </p>
     * <p>
     * <a href="http://aws.amazon.com/ec2/instance-types/p4/">Amazon EC2 P4de instances</a> (currently in preview) are
     * powered by 8 NVIDIA A100 GPUs with 80GB high-performance HBM2e GPU memory, which accelerate the speed of training
     * ML models that need to be trained on large datasets of high-resolution data. In this preview release, Amazon
     * SageMaker supports ML training jobs on P4de instances (<code>ml.p4de.24xlarge</code>) to reduce model training
     * time. The <code>ml.p4de.24xlarge</code> instances are available in the following Amazon Web Services Regions.
     * </p>
     * <ul>
     * <li>
     * <p>
     * US East (N. Virginia) (us-east-1)
     * </p>
     * </li>
     * <li>
     * <p>
     * US West (Oregon) (us-west-2)
     * </p>
     * </li>
     * </ul>
     * <p>
     * To request quota limit increase and start using P4de instances, contact the SageMaker Training service team
     * through your account team.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link TrainingInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The ML compute instance type. </p> <note>
     *         <p>
     *         SageMaker Training on Amazon Elastic Compute Cloud (EC2) P4de instances is in preview release starting
     *         December 9th, 2022.
     *         </p>
     *         <p>
     *         <a href="http://aws.amazon.com/ec2/instance-types/p4/">Amazon EC2 P4de instances</a> (currently in
     *         preview) are powered by 8 NVIDIA A100 GPUs with 80GB high-performance HBM2e GPU memory, which accelerate
     *         the speed of training ML models that need to be trained on large datasets of high-resolution data. In
     *         this preview release, Amazon SageMaker supports ML training jobs on P4de instances (
     *         <code>ml.p4de.24xlarge</code>) to reduce model training time. The <code>ml.p4de.24xlarge</code> instances
     *         are available in the following Amazon Web Services Regions.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         US East (N. Virginia) (us-east-1)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         US West (Oregon) (us-west-2)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To request quota limit increase and start using P4de instances, contact the SageMaker Training service
     *         team through your account team.
     *         </p>
     * @see TrainingInstanceType
     */
    public final TrainingInstanceType instanceType() {
        return TrainingInstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The ML compute instance type.
     * </p>
     * <note>
     * <p>
     * SageMaker Training on Amazon Elastic Compute Cloud (EC2) P4de instances is in preview release starting December
     * 9th, 2022.
     * </p>
     * <p>
     * <a href="http://aws.amazon.com/ec2/instance-types/p4/">Amazon EC2 P4de instances</a> (currently in preview) are
     * powered by 8 NVIDIA A100 GPUs with 80GB high-performance HBM2e GPU memory, which accelerate the speed of training
     * ML models that need to be trained on large datasets of high-resolution data. In this preview release, Amazon
     * SageMaker supports ML training jobs on P4de instances (<code>ml.p4de.24xlarge</code>) to reduce model training
     * time. The <code>ml.p4de.24xlarge</code> instances are available in the following Amazon Web Services Regions.
     * </p>
     * <ul>
     * <li>
     * <p>
     * US East (N. Virginia) (us-east-1)
     * </p>
     * </li>
     * <li>
     * <p>
     * US West (Oregon) (us-west-2)
     * </p>
     * </li>
     * </ul>
     * <p>
     * To request quota limit increase and start using P4de instances, contact the SageMaker Training service team
     * through your account team.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link TrainingInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The ML compute instance type. </p> <note>
     *         <p>
     *         SageMaker Training on Amazon Elastic Compute Cloud (EC2) P4de instances is in preview release starting
     *         December 9th, 2022.
     *         </p>
     *         <p>
     *         <a href="http://aws.amazon.com/ec2/instance-types/p4/">Amazon EC2 P4de instances</a> (currently in
     *         preview) are powered by 8 NVIDIA A100 GPUs with 80GB high-performance HBM2e GPU memory, which accelerate
     *         the speed of training ML models that need to be trained on large datasets of high-resolution data. In
     *         this preview release, Amazon SageMaker supports ML training jobs on P4de instances (
     *         <code>ml.p4de.24xlarge</code>) to reduce model training time. The <code>ml.p4de.24xlarge</code> instances
     *         are available in the following Amazon Web Services Regions.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         US East (N. Virginia) (us-east-1)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         US West (Oregon) (us-west-2)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To request quota limit increase and start using P4de instances, contact the SageMaker Training service
     *         team through your account team.
     *         </p>
     * @see TrainingInstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * The number of ML compute instances to use. For distributed training, provide a value greater than 1.
     * </p>
     * 
     * @return The number of ML compute instances to use. For distributed training, provide a value greater than 1.
     */
    public final Integer instanceCount() {
        return instanceCount;
    }

    /**
     * <p>
     * The size of the ML storage volume that you want to provision.
     * </p>
     * <p>
     * ML storage volumes store model artifacts and incremental states. Training algorithms might also use the ML
     * storage volume for scratch space. If you want to store the training data in the ML storage volume, choose
     * <code>File</code> as the <code>TrainingInputMode</code> in the algorithm specification.
     * </p>
     * <p>
     * When using an ML instance with <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ssd-instance-store.html#nvme-ssd-volumes">NVMe SSD
     * volumes</a>, SageMaker doesn't provision Amazon EBS General Purpose SSD (gp2) storage. Available storage is fixed
     * to the NVMe-type instance's storage capacity. SageMaker configures storage paths for training datasets,
     * checkpoints, model artifacts, and outputs to use the entire capacity of the instance storage. For example, ML
     * instance families with the NVMe-type instance storage include <code>ml.p4d</code>, <code>ml.g4dn</code>, and
     * <code>ml.g5</code>.
     * </p>
     * <p>
     * When using an ML instance with the EBS-only storage option and without instance storage, you must define the size
     * of EBS volume through <code>VolumeSizeInGB</code> in the <code>ResourceConfig</code> API. For example, ML
     * instance families that use EBS volumes include <code>ml.c5</code> and <code>ml.p2</code>.
     * </p>
     * <p>
     * To look up instance types and their instance storage types and volumes, see <a
     * href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a>.
     * </p>
     * <p>
     * To find the default local paths defined by the SageMaker training platform, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-train-storage.html">Amazon SageMaker Training Storage
     * Folders for Training Datasets, Checkpoints, Model Artifacts, and Outputs</a>.
     * </p>
     * 
     * @return The size of the ML storage volume that you want to provision. </p>
     *         <p>
     *         ML storage volumes store model artifacts and incremental states. Training algorithms might also use the
     *         ML storage volume for scratch space. If you want to store the training data in the ML storage volume,
     *         choose <code>File</code> as the <code>TrainingInputMode</code> in the algorithm specification.
     *         </p>
     *         <p>
     *         When using an ML instance with <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ssd-instance-store.html#nvme-ssd-volumes">NVMe
     *         SSD volumes</a>, SageMaker doesn't provision Amazon EBS General Purpose SSD (gp2) storage. Available
     *         storage is fixed to the NVMe-type instance's storage capacity. SageMaker configures storage paths for
     *         training datasets, checkpoints, model artifacts, and outputs to use the entire capacity of the instance
     *         storage. For example, ML instance families with the NVMe-type instance storage include
     *         <code>ml.p4d</code>, <code>ml.g4dn</code>, and <code>ml.g5</code>.
     *         </p>
     *         <p>
     *         When using an ML instance with the EBS-only storage option and without instance storage, you must define
     *         the size of EBS volume through <code>VolumeSizeInGB</code> in the <code>ResourceConfig</code> API. For
     *         example, ML instance families that use EBS volumes include <code>ml.c5</code> and <code>ml.p2</code>.
     *         </p>
     *         <p>
     *         To look up instance types and their instance storage types and volumes, see <a
     *         href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a>.
     *         </p>
     *         <p>
     *         To find the default local paths defined by the SageMaker training platform, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-train-storage.html">Amazon SageMaker Training
     *         Storage Folders for Training Datasets, Checkpoints, Model Artifacts, and Outputs</a>.
     */
    public final Integer volumeSizeInGB() {
        return volumeSizeInGB;
    }

    /**
     * <p>
     * The Amazon Web Services KMS key that SageMaker uses to encrypt data on the storage volume attached to the ML
     * compute instance(s) that run the training job.
     * </p>
     * <note>
     * <p>
     * Certain Nitro-based instances include local storage, dependent on the instance type. Local storage volumes are
     * encrypted using a hardware module on the instance. You can't request a <code>VolumeKmsKeyId</code> when using an
     * instance type with local storage.
     * </p>
     * <p>
     * For a list of instance types that support local instance storage, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html#instance-store-volumes">Instance
     * Store Volumes</a>.
     * </p>
     * <p>
     * For more information about local instance storage encryption, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ssd-instance-store.html">SSD Instance Store
     * Volumes</a>.
     * </p>
     * </note>
     * <p>
     * The <code>VolumeKmsKeyId</code> can be in any of the following formats:
     * </p>
     * <ul>
     * <li>
     * <p>
     * // KMS Key ID
     * </p>
     * <p>
     * <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * // Amazon Resource Name (ARN) of a KMS Key
     * </p>
     * <p>
     * <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The Amazon Web Services KMS key that SageMaker uses to encrypt data on the storage volume attached to the
     *         ML compute instance(s) that run the training job.</p> <note>
     *         <p>
     *         Certain Nitro-based instances include local storage, dependent on the instance type. Local storage
     *         volumes are encrypted using a hardware module on the instance. You can't request a
     *         <code>VolumeKmsKeyId</code> when using an instance type with local storage.
     *         </p>
     *         <p>
     *         For a list of instance types that support local instance storage, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html#instance-store-volumes"
     *         >Instance Store Volumes</a>.
     *         </p>
     *         <p>
     *         For more information about local instance storage encryption, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ssd-instance-store.html">SSD Instance Store
     *         Volumes</a>.
     *         </p>
     *         </note>
     *         <p>
     *         The <code>VolumeKmsKeyId</code> can be in any of the following formats:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         // KMS Key ID
     *         </p>
     *         <p>
     *         <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         // Amazon Resource Name (ARN) of a KMS Key
     *         </p>
     *         <p>
     *         <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     *         </p>
     *         </li>
     */
    public final String volumeKmsKeyId() {
        return volumeKmsKeyId;
    }

    /**
     * <p>
     * The duration of time in seconds to retain configured resources in a warm pool for subsequent training jobs.
     * </p>
     * 
     * @return The duration of time in seconds to retain configured resources in a warm pool for subsequent training
     *         jobs.
     */
    public final Integer keepAlivePeriodInSeconds() {
        return keepAlivePeriodInSeconds;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInstanceGroups() {
        return instanceGroups != null && !(instanceGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The configuration of a heterogeneous cluster in JSON format.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceGroups} method.
     * </p>
     * 
     * @return The configuration of a heterogeneous cluster in JSON format.
     */
    public final List<InstanceGroup> instanceGroups() {
        return instanceGroups;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN); of the training plan to use for this resource configuration.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN); of the training plan to use for this resource configuration.
     */
    public final String trainingPlanArn() {
        return trainingPlanArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(volumeSizeInGB());
        hashCode = 31 * hashCode + Objects.hashCode(volumeKmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(keepAlivePeriodInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceGroups() ? instanceGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(trainingPlanArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceConfig)) {
            return false;
        }
        ResourceConfig other = (ResourceConfig) obj;
        return Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(instanceCount(), other.instanceCount())
                && Objects.equals(volumeSizeInGB(), other.volumeSizeInGB())
                && Objects.equals(volumeKmsKeyId(), other.volumeKmsKeyId())
                && Objects.equals(keepAlivePeriodInSeconds(), other.keepAlivePeriodInSeconds())
                && hasInstanceGroups() == other.hasInstanceGroups() && Objects.equals(instanceGroups(), other.instanceGroups())
                && Objects.equals(trainingPlanArn(), other.trainingPlanArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceConfig").add("InstanceType", instanceTypeAsString())
                .add("InstanceCount", instanceCount()).add("VolumeSizeInGB", volumeSizeInGB())
                .add("VolumeKmsKeyId", volumeKmsKeyId()).add("KeepAlivePeriodInSeconds", keepAlivePeriodInSeconds())
                .add("InstanceGroups", hasInstanceGroups() ? instanceGroups() : null).add("TrainingPlanArn", trainingPlanArn())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "InstanceCount":
            return Optional.ofNullable(clazz.cast(instanceCount()));
        case "VolumeSizeInGB":
            return Optional.ofNullable(clazz.cast(volumeSizeInGB()));
        case "VolumeKmsKeyId":
            return Optional.ofNullable(clazz.cast(volumeKmsKeyId()));
        case "KeepAlivePeriodInSeconds":
            return Optional.ofNullable(clazz.cast(keepAlivePeriodInSeconds()));
        case "InstanceGroups":
            return Optional.ofNullable(clazz.cast(instanceGroups()));
        case "TrainingPlanArn":
            return Optional.ofNullable(clazz.cast(trainingPlanArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<ResourceConfig, T> g) {
        return obj -> g.apply((ResourceConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceConfig> {
        /**
         * <p>
         * The ML compute instance type.
         * </p>
         * <note>
         * <p>
         * SageMaker Training on Amazon Elastic Compute Cloud (EC2) P4de instances is in preview release starting
         * December 9th, 2022.
         * </p>
         * <p>
         * <a href="http://aws.amazon.com/ec2/instance-types/p4/">Amazon EC2 P4de instances</a> (currently in preview)
         * are powered by 8 NVIDIA A100 GPUs with 80GB high-performance HBM2e GPU memory, which accelerate the speed of
         * training ML models that need to be trained on large datasets of high-resolution data. In this preview
         * release, Amazon SageMaker supports ML training jobs on P4de instances (<code>ml.p4de.24xlarge</code>) to
         * reduce model training time. The <code>ml.p4de.24xlarge</code> instances are available in the following Amazon
         * Web Services Regions.
         * </p>
         * <ul>
         * <li>
         * <p>
         * US East (N. Virginia) (us-east-1)
         * </p>
         * </li>
         * <li>
         * <p>
         * US West (Oregon) (us-west-2)
         * </p>
         * </li>
         * </ul>
         * <p>
         * To request quota limit increase and start using P4de instances, contact the SageMaker Training service team
         * through your account team.
         * </p>
         * </note>
         * 
         * @param instanceType
         *        The ML compute instance type. </p> <note>
         *        <p>
         *        SageMaker Training on Amazon Elastic Compute Cloud (EC2) P4de instances is in preview release starting
         *        December 9th, 2022.
         *        </p>
         *        <p>
         *        <a href="http://aws.amazon.com/ec2/instance-types/p4/">Amazon EC2 P4de instances</a> (currently in
         *        preview) are powered by 8 NVIDIA A100 GPUs with 80GB high-performance HBM2e GPU memory, which
         *        accelerate the speed of training ML models that need to be trained on large datasets of
         *        high-resolution data. In this preview release, Amazon SageMaker supports ML training jobs on P4de
         *        instances (<code>ml.p4de.24xlarge</code>) to reduce model training time. The
         *        <code>ml.p4de.24xlarge</code> instances are available in the following Amazon Web Services Regions.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        US East (N. Virginia) (us-east-1)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        US West (Oregon) (us-west-2)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To request quota limit increase and start using P4de instances, contact the SageMaker Training service
         *        team through your account team.
         *        </p>
         * @see TrainingInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The ML compute instance type.
         * </p>
         * <note>
         * <p>
         * SageMaker Training on Amazon Elastic Compute Cloud (EC2) P4de instances is in preview release starting
         * December 9th, 2022.
         * </p>
         * <p>
         * <a href="http://aws.amazon.com/ec2/instance-types/p4/">Amazon EC2 P4de instances</a> (currently in preview)
         * are powered by 8 NVIDIA A100 GPUs with 80GB high-performance HBM2e GPU memory, which accelerate the speed of
         * training ML models that need to be trained on large datasets of high-resolution data. In this preview
         * release, Amazon SageMaker supports ML training jobs on P4de instances (<code>ml.p4de.24xlarge</code>) to
         * reduce model training time. The <code>ml.p4de.24xlarge</code> instances are available in the following Amazon
         * Web Services Regions.
         * </p>
         * <ul>
         * <li>
         * <p>
         * US East (N. Virginia) (us-east-1)
         * </p>
         * </li>
         * <li>
         * <p>
         * US West (Oregon) (us-west-2)
         * </p>
         * </li>
         * </ul>
         * <p>
         * To request quota limit increase and start using P4de instances, contact the SageMaker Training service team
         * through your account team.
         * </p>
         * </note>
         * 
         * @param instanceType
         *        The ML compute instance type. </p> <note>
         *        <p>
         *        SageMaker Training on Amazon Elastic Compute Cloud (EC2) P4de instances is in preview release starting
         *        December 9th, 2022.
         *        </p>
         *        <p>
         *        <a href="http://aws.amazon.com/ec2/instance-types/p4/">Amazon EC2 P4de instances</a> (currently in
         *        preview) are powered by 8 NVIDIA A100 GPUs with 80GB high-performance HBM2e GPU memory, which
         *        accelerate the speed of training ML models that need to be trained on large datasets of
         *        high-resolution data. In this preview release, Amazon SageMaker supports ML training jobs on P4de
         *        instances (<code>ml.p4de.24xlarge</code>) to reduce model training time. The
         *        <code>ml.p4de.24xlarge</code> instances are available in the following Amazon Web Services Regions.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        US East (N. Virginia) (us-east-1)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        US West (Oregon) (us-west-2)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To request quota limit increase and start using P4de instances, contact the SageMaker Training service
         *        team through your account team.
         *        </p>
         * @see TrainingInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInstanceType
         */
        Builder instanceType(TrainingInstanceType instanceType);

        /**
         * <p>
         * The number of ML compute instances to use. For distributed training, provide a value greater than 1.
         * </p>
         * 
         * @param instanceCount
         *        The number of ML compute instances to use. For distributed training, provide a value greater than 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceCount(Integer instanceCount);

        /**
         * <p>
         * The size of the ML storage volume that you want to provision.
         * </p>
         * <p>
         * ML storage volumes store model artifacts and incremental states. Training algorithms might also use the ML
         * storage volume for scratch space. If you want to store the training data in the ML storage volume, choose
         * <code>File</code> as the <code>TrainingInputMode</code> in the algorithm specification.
         * </p>
         * <p>
         * When using an ML instance with <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ssd-instance-store.html#nvme-ssd-volumes">NVMe SSD
         * volumes</a>, SageMaker doesn't provision Amazon EBS General Purpose SSD (gp2) storage. Available storage is
         * fixed to the NVMe-type instance's storage capacity. SageMaker configures storage paths for training datasets,
         * checkpoints, model artifacts, and outputs to use the entire capacity of the instance storage. For example, ML
         * instance families with the NVMe-type instance storage include <code>ml.p4d</code>, <code>ml.g4dn</code>, and
         * <code>ml.g5</code>.
         * </p>
         * <p>
         * When using an ML instance with the EBS-only storage option and without instance storage, you must define the
         * size of EBS volume through <code>VolumeSizeInGB</code> in the <code>ResourceConfig</code> API. For example,
         * ML instance families that use EBS volumes include <code>ml.c5</code> and <code>ml.p2</code>.
         * </p>
         * <p>
         * To look up instance types and their instance storage types and volumes, see <a
         * href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a>.
         * </p>
         * <p>
         * To find the default local paths defined by the SageMaker training platform, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-train-storage.html">Amazon SageMaker Training
         * Storage Folders for Training Datasets, Checkpoints, Model Artifacts, and Outputs</a>.
         * </p>
         * 
         * @param volumeSizeInGB
         *        The size of the ML storage volume that you want to provision. </p>
         *        <p>
         *        ML storage volumes store model artifacts and incremental states. Training algorithms might also use
         *        the ML storage volume for scratch space. If you want to store the training data in the ML storage
         *        volume, choose <code>File</code> as the <code>TrainingInputMode</code> in the algorithm specification.
         *        </p>
         *        <p>
         *        When using an ML instance with <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ssd-instance-store.html#nvme-ssd-volumes"
         *        >NVMe SSD volumes</a>, SageMaker doesn't provision Amazon EBS General Purpose SSD (gp2) storage.
         *        Available storage is fixed to the NVMe-type instance's storage capacity. SageMaker configures storage
         *        paths for training datasets, checkpoints, model artifacts, and outputs to use the entire capacity of
         *        the instance storage. For example, ML instance families with the NVMe-type instance storage include
         *        <code>ml.p4d</code>, <code>ml.g4dn</code>, and <code>ml.g5</code>.
         *        </p>
         *        <p>
         *        When using an ML instance with the EBS-only storage option and without instance storage, you must
         *        define the size of EBS volume through <code>VolumeSizeInGB</code> in the <code>ResourceConfig</code>
         *        API. For example, ML instance families that use EBS volumes include <code>ml.c5</code> and
         *        <code>ml.p2</code>.
         *        </p>
         *        <p>
         *        To look up instance types and their instance storage types and volumes, see <a
         *        href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a>.
         *        </p>
         *        <p>
         *        To find the default local paths defined by the SageMaker training platform, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-train-storage.html">Amazon SageMaker
         *        Training Storage Folders for Training Datasets, Checkpoints, Model Artifacts, and Outputs</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeSizeInGB(Integer volumeSizeInGB);

        /**
         * <p>
         * The Amazon Web Services KMS key that SageMaker uses to encrypt data on the storage volume attached to the ML
         * compute instance(s) that run the training job.
         * </p>
         * <note>
         * <p>
         * Certain Nitro-based instances include local storage, dependent on the instance type. Local storage volumes
         * are encrypted using a hardware module on the instance. You can't request a <code>VolumeKmsKeyId</code> when
         * using an instance type with local storage.
         * </p>
         * <p>
         * For a list of instance types that support local instance storage, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html#instance-store-volumes"
         * >Instance Store Volumes</a>.
         * </p>
         * <p>
         * For more information about local instance storage encryption, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ssd-instance-store.html">SSD Instance Store
         * Volumes</a>.
         * </p>
         * </note>
         * <p>
         * The <code>VolumeKmsKeyId</code> can be in any of the following formats:
         * </p>
         * <ul>
         * <li>
         * <p>
         * // KMS Key ID
         * </p>
         * <p>
         * <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * // Amazon Resource Name (ARN) of a KMS Key
         * </p>
         * <p>
         * <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param volumeKmsKeyId
         *        The Amazon Web Services KMS key that SageMaker uses to encrypt data on the storage volume attached to
         *        the ML compute instance(s) that run the training job.</p> <note>
         *        <p>
         *        Certain Nitro-based instances include local storage, dependent on the instance type. Local storage
         *        volumes are encrypted using a hardware module on the instance. You can't request a
         *        <code>VolumeKmsKeyId</code> when using an instance type with local storage.
         *        </p>
         *        <p>
         *        For a list of instance types that support local instance storage, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html#instance-store-volumes"
         *        >Instance Store Volumes</a>.
         *        </p>
         *        <p>
         *        For more information about local instance storage encryption, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ssd-instance-store.html">SSD Instance Store
         *        Volumes</a>.
         *        </p>
         *        </note>
         *        <p>
         *        The <code>VolumeKmsKeyId</code> can be in any of the following formats:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        // KMS Key ID
         *        </p>
         *        <p>
         *        <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        // Amazon Resource Name (ARN) of a KMS Key
         *        </p>
         *        <p>
         *        <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeKmsKeyId(String volumeKmsKeyId);

        /**
         * <p>
         * The duration of time in seconds to retain configured resources in a warm pool for subsequent training jobs.
         * </p>
         * 
         * @param keepAlivePeriodInSeconds
         *        The duration of time in seconds to retain configured resources in a warm pool for subsequent training
         *        jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keepAlivePeriodInSeconds(Integer keepAlivePeriodInSeconds);

        /**
         * <p>
         * The configuration of a heterogeneous cluster in JSON format.
         * </p>
         * 
         * @param instanceGroups
         *        The configuration of a heterogeneous cluster in JSON format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGroups(Collection<InstanceGroup> instanceGroups);

        /**
         * <p>
         * The configuration of a heterogeneous cluster in JSON format.
         * </p>
         * 
         * @param instanceGroups
         *        The configuration of a heterogeneous cluster in JSON format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGroups(InstanceGroup... instanceGroups);

        /**
         * <p>
         * The configuration of a heterogeneous cluster in JSON format.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.InstanceGroup.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.sagemaker.model.InstanceGroup#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.InstanceGroup.Builder#build()} is called immediately
         * and its result is passed to {@link #instanceGroups(List<InstanceGroup>)}.
         * 
         * @param instanceGroups
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.InstanceGroup.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceGroups(java.util.Collection<InstanceGroup>)
         */
        Builder instanceGroups(Consumer<InstanceGroup.Builder>... instanceGroups);

        /**
         * <p>
         * The Amazon Resource Name (ARN); of the training plan to use for this resource configuration.
         * </p>
         * 
         * @param trainingPlanArn
         *        The Amazon Resource Name (ARN); of the training plan to use for this resource configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingPlanArn(String trainingPlanArn);
    }

    static final class BuilderImpl implements Builder {
        private String instanceType;

        private Integer instanceCount;

        private Integer volumeSizeInGB;

        private String volumeKmsKeyId;

        private Integer keepAlivePeriodInSeconds;

        private List<InstanceGroup> instanceGroups = DefaultSdkAutoConstructList.getInstance();

        private String trainingPlanArn;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceConfig model) {
            instanceType(model.instanceType);
            instanceCount(model.instanceCount);
            volumeSizeInGB(model.volumeSizeInGB);
            volumeKmsKeyId(model.volumeKmsKeyId);
            keepAlivePeriodInSeconds(model.keepAlivePeriodInSeconds);
            instanceGroups(model.instanceGroups);
            trainingPlanArn(model.trainingPlanArn);
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(TrainingInstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final Integer getInstanceCount() {
            return instanceCount;
        }

        public final void setInstanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
        }

        @Override
        public final Builder instanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
            return this;
        }

        public final Integer getVolumeSizeInGB() {
            return volumeSizeInGB;
        }

        public final void setVolumeSizeInGB(Integer volumeSizeInGB) {
            this.volumeSizeInGB = volumeSizeInGB;
        }

        @Override
        public final Builder volumeSizeInGB(Integer volumeSizeInGB) {
            this.volumeSizeInGB = volumeSizeInGB;
            return this;
        }

        public final String getVolumeKmsKeyId() {
            return volumeKmsKeyId;
        }

        public final void setVolumeKmsKeyId(String volumeKmsKeyId) {
            this.volumeKmsKeyId = volumeKmsKeyId;
        }

        @Override
        public final Builder volumeKmsKeyId(String volumeKmsKeyId) {
            this.volumeKmsKeyId = volumeKmsKeyId;
            return this;
        }

        public final Integer getKeepAlivePeriodInSeconds() {
            return keepAlivePeriodInSeconds;
        }

        public final void setKeepAlivePeriodInSeconds(Integer keepAlivePeriodInSeconds) {
            this.keepAlivePeriodInSeconds = keepAlivePeriodInSeconds;
        }

        @Override
        public final Builder keepAlivePeriodInSeconds(Integer keepAlivePeriodInSeconds) {
            this.keepAlivePeriodInSeconds = keepAlivePeriodInSeconds;
            return this;
        }

        public final List<InstanceGroup.Builder> getInstanceGroups() {
            List<InstanceGroup.Builder> result = InstanceGroupsCopier.copyToBuilder(this.instanceGroups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInstanceGroups(Collection<InstanceGroup.BuilderImpl> instanceGroups) {
            this.instanceGroups = InstanceGroupsCopier.copyFromBuilder(instanceGroups);
        }

        @Override
        public final Builder instanceGroups(Collection<InstanceGroup> instanceGroups) {
            this.instanceGroups = InstanceGroupsCopier.copy(instanceGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceGroups(InstanceGroup... instanceGroups) {
            instanceGroups(Arrays.asList(instanceGroups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceGroups(Consumer<InstanceGroup.Builder>... instanceGroups) {
            instanceGroups(Stream.of(instanceGroups).map(c -> InstanceGroup.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getTrainingPlanArn() {
            return trainingPlanArn;
        }

        public final void setTrainingPlanArn(String trainingPlanArn) {
            this.trainingPlanArn = trainingPlanArn;
        }

        @Override
        public final Builder trainingPlanArn(String trainingPlanArn) {
            this.trainingPlanArn = trainingPlanArn;
            return this;
        }

        @Override
        public ResourceConfig build() {
            return new ResourceConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
