/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the S3 location of ML model data to deploy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3ModelDataSource implements SdkPojo, Serializable,
        ToCopyableBuilder<S3ModelDataSource.Builder, S3ModelDataSource> {
    private static final SdkField<String> S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("S3Uri")
            .getter(getter(S3ModelDataSource::s3Uri)).setter(setter(Builder::s3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Uri").build()).build();

    private static final SdkField<String> S3_DATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3DataType").getter(getter(S3ModelDataSource::s3DataTypeAsString)).setter(setter(Builder::s3DataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3DataType").build()).build();

    private static final SdkField<String> COMPRESSION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CompressionType").getter(getter(S3ModelDataSource::compressionTypeAsString))
            .setter(setter(Builder::compressionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompressionType").build()).build();

    private static final SdkField<ModelAccessConfig> MODEL_ACCESS_CONFIG_FIELD = SdkField
            .<ModelAccessConfig> builder(MarshallingType.SDK_POJO).memberName("ModelAccessConfig")
            .getter(getter(S3ModelDataSource::modelAccessConfig)).setter(setter(Builder::modelAccessConfig))
            .constructor(ModelAccessConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelAccessConfig").build()).build();

    private static final SdkField<InferenceHubAccessConfig> HUB_ACCESS_CONFIG_FIELD = SdkField
            .<InferenceHubAccessConfig> builder(MarshallingType.SDK_POJO).memberName("HubAccessConfig")
            .getter(getter(S3ModelDataSource::hubAccessConfig)).setter(setter(Builder::hubAccessConfig))
            .constructor(InferenceHubAccessConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HubAccessConfig").build()).build();

    private static final SdkField<String> MANIFEST_S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ManifestS3Uri").getter(getter(S3ModelDataSource::manifestS3Uri)).setter(setter(Builder::manifestS3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManifestS3Uri").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(S3_URI_FIELD, S3_DATA_TYPE_FIELD, COMPRESSION_TYPE_FIELD, MODEL_ACCESS_CONFIG_FIELD, HUB_ACCESS_CONFIG_FIELD,
                    MANIFEST_S3_URI_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("S3Uri", S3_URI_FIELD);
                    put("S3DataType", S3_DATA_TYPE_FIELD);
                    put("CompressionType", COMPRESSION_TYPE_FIELD);
                    put("ModelAccessConfig", MODEL_ACCESS_CONFIG_FIELD);
                    put("HubAccessConfig", HUB_ACCESS_CONFIG_FIELD);
                    put("ManifestS3Uri", MANIFEST_S3_URI_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String s3Uri;

    private final String s3DataType;

    private final String compressionType;

    private final ModelAccessConfig modelAccessConfig;

    private final InferenceHubAccessConfig hubAccessConfig;

    private final String manifestS3Uri;

    private S3ModelDataSource(BuilderImpl builder) {
        this.s3Uri = builder.s3Uri;
        this.s3DataType = builder.s3DataType;
        this.compressionType = builder.compressionType;
        this.modelAccessConfig = builder.modelAccessConfig;
        this.hubAccessConfig = builder.hubAccessConfig;
        this.manifestS3Uri = builder.manifestS3Uri;
    }

    /**
     * <p>
     * Specifies the S3 path of ML model data to deploy.
     * </p>
     * 
     * @return Specifies the S3 path of ML model data to deploy.
     */
    public final String s3Uri() {
        return s3Uri;
    }

    /**
     * <p>
     * Specifies the type of ML model data to deploy.
     * </p>
     * <p>
     * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all objects
     * that match the specified key name prefix as part of the ML model data to deploy. A valid key name prefix
     * identified by <code>S3Uri</code> always ends with a forward slash (/).
     * </p>
     * <p>
     * If you choose <code>S3Object</code>, <code>S3Uri</code> identifies an object that is the ML model data to deploy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3DataType} will
     * return {@link S3ModelDataType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3DataTypeAsString}.
     * </p>
     * 
     * @return Specifies the type of ML model data to deploy.</p>
     *         <p>
     *         If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all
     *         objects that match the specified key name prefix as part of the ML model data to deploy. A valid key name
     *         prefix identified by <code>S3Uri</code> always ends with a forward slash (/).
     *         </p>
     *         <p>
     *         If you choose <code>S3Object</code>, <code>S3Uri</code> identifies an object that is the ML model data to
     *         deploy.
     * @see S3ModelDataType
     */
    public final S3ModelDataType s3DataType() {
        return S3ModelDataType.fromValue(s3DataType);
    }

    /**
     * <p>
     * Specifies the type of ML model data to deploy.
     * </p>
     * <p>
     * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all objects
     * that match the specified key name prefix as part of the ML model data to deploy. A valid key name prefix
     * identified by <code>S3Uri</code> always ends with a forward slash (/).
     * </p>
     * <p>
     * If you choose <code>S3Object</code>, <code>S3Uri</code> identifies an object that is the ML model data to deploy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3DataType} will
     * return {@link S3ModelDataType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3DataTypeAsString}.
     * </p>
     * 
     * @return Specifies the type of ML model data to deploy.</p>
     *         <p>
     *         If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all
     *         objects that match the specified key name prefix as part of the ML model data to deploy. A valid key name
     *         prefix identified by <code>S3Uri</code> always ends with a forward slash (/).
     *         </p>
     *         <p>
     *         If you choose <code>S3Object</code>, <code>S3Uri</code> identifies an object that is the ML model data to
     *         deploy.
     * @see S3ModelDataType
     */
    public final String s3DataTypeAsString() {
        return s3DataType;
    }

    /**
     * <p>
     * Specifies how the ML model data is prepared.
     * </p>
     * <p>
     * If you choose <code>Gzip</code> and choose <code>S3Object</code> as the value of <code>S3DataType</code>,
     * <code>S3Uri</code> identifies an object that is a gzip-compressed TAR archive. SageMaker will attempt to
     * decompress and untar the object during model deployment.
     * </p>
     * <p>
     * If you choose <code>None</code> and chooose <code>S3Object</code> as the value of <code>S3DataType</code>,
     * <code>S3Uri</code> identifies an object that represents an uncompressed ML model to deploy.
     * </p>
     * <p>
     * If you choose None and choose <code>S3Prefix</code> as the value of <code>S3DataType</code>, <code>S3Uri</code>
     * identifies a key name prefix, under which all objects represents the uncompressed ML model to deploy.
     * </p>
     * <p>
     * If you choose None, then SageMaker will follow rules below when creating model data files under /opt/ml/model
     * directory for use by your inference code:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you choose <code>S3Object</code> as the value of <code>S3DataType</code>, then SageMaker will split the key of
     * the S3 object referenced by <code>S3Uri</code> by slash (/), and use the last part as the filename of the file
     * holding the content of the S3 object.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you choose <code>S3Prefix</code> as the value of <code>S3DataType</code>, then for each S3 object under the
     * key name pefix referenced by <code>S3Uri</code>, SageMaker will trim its key by the prefix, and use the remainder
     * as the path (relative to <code>/opt/ml/model</code>) of the file holding the content of the S3 object. SageMaker
     * will split the remainder by slash (/), using intermediate parts as directory names and the last part as filename
     * of the file holding the content of the S3 object.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do not use any of the following as file names or directory names:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An empty or blank string
     * </p>
     * </li>
     * <li>
     * <p>
     * A string which contains null bytes
     * </p>
     * </li>
     * <li>
     * <p>
     * A string longer than 255 bytes
     * </p>
     * </li>
     * <li>
     * <p>
     * A single dot (<code>.</code>)
     * </p>
     * </li>
     * <li>
     * <p>
     * A double dot (<code>..</code>)
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Ambiguous file names will result in model deployment failure. For example, if your uncompressed ML model consists
     * of two S3 objects <code>s3://mybucket/model/weights</code> and <code>s3://mybucket/model/weights/part1</code> and
     * you specify <code>s3://mybucket/model/</code> as the value of <code>S3Uri</code> and <code>S3Prefix</code> as the
     * value of <code>S3DataType</code>, then it will result in name clash between <code>/opt/ml/model/weights</code> (a
     * regular file) and <code>/opt/ml/model/weights/</code> (a directory).
     * </p>
     * </li>
     * <li>
     * <p>
     * Do not organize the model artifacts in <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-folders.html">S3 console using folders</a>.
     * When you create a folder in S3 console, S3 creates a 0-byte object with a key set to the folder name you provide.
     * They key of the 0-byte object ends with a slash (/) which violates SageMaker restrictions on model artifact file
     * names, leading to model deployment failure.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compressionType}
     * will return {@link ModelCompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #compressionTypeAsString}.
     * </p>
     * 
     * @return Specifies how the ML model data is prepared.</p>
     *         <p>
     *         If you choose <code>Gzip</code> and choose <code>S3Object</code> as the value of <code>S3DataType</code>,
     *         <code>S3Uri</code> identifies an object that is a gzip-compressed TAR archive. SageMaker will attempt to
     *         decompress and untar the object during model deployment.
     *         </p>
     *         <p>
     *         If you choose <code>None</code> and chooose <code>S3Object</code> as the value of <code>S3DataType</code>, <code>S3Uri</code> identifies an object that represents an uncompressed ML model to deploy.
     *         </p>
     *         <p>
     *         If you choose None and choose <code>S3Prefix</code> as the value of <code>S3DataType</code>,
     *         <code>S3Uri</code> identifies a key name prefix, under which all objects represents the uncompressed ML
     *         model to deploy.
     *         </p>
     *         <p>
     *         If you choose None, then SageMaker will follow rules below when creating model data files under
     *         /opt/ml/model directory for use by your inference code:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you choose <code>S3Object</code> as the value of <code>S3DataType</code>, then SageMaker will split
     *         the key of the S3 object referenced by <code>S3Uri</code> by slash (/), and use the last part as the
     *         filename of the file holding the content of the S3 object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you choose <code>S3Prefix</code> as the value of <code>S3DataType</code>, then for each S3 object
     *         under the key name pefix referenced by <code>S3Uri</code>, SageMaker will trim its key by the prefix, and
     *         use the remainder as the path (relative to <code>/opt/ml/model</code>) of the file holding the content of
     *         the S3 object. SageMaker will split the remainder by slash (/), using intermediate parts as directory
     *         names and the last part as filename of the file holding the content of the S3 object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do not use any of the following as file names or directory names:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         An empty or blank string
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A string which contains null bytes
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A string longer than 255 bytes
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A single dot (<code>.</code>)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A double dot (<code>..</code>)
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Ambiguous file names will result in model deployment failure. For example, if your uncompressed ML model
     *         consists of two S3 objects <code>s3://mybucket/model/weights</code> and
     *         <code>s3://mybucket/model/weights/part1</code> and you specify <code>s3://mybucket/model/</code> as the
     *         value of <code>S3Uri</code> and <code>S3Prefix</code> as the value of <code>S3DataType</code>, then it
     *         will result in name clash between <code>/opt/ml/model/weights</code> (a regular file) and
     *         <code>/opt/ml/model/weights/</code> (a directory).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do not organize the model artifacts in <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-folders.html">S3 console using
     *         folders</a>. When you create a folder in S3 console, S3 creates a 0-byte object with a key set to the
     *         folder name you provide. They key of the 0-byte object ends with a slash (/) which violates SageMaker
     *         restrictions on model artifact file names, leading to model deployment failure.
     *         </p>
     *         </li>
     * @see ModelCompressionType
     */
    public final ModelCompressionType compressionType() {
        return ModelCompressionType.fromValue(compressionType);
    }

    /**
     * <p>
     * Specifies how the ML model data is prepared.
     * </p>
     * <p>
     * If you choose <code>Gzip</code> and choose <code>S3Object</code> as the value of <code>S3DataType</code>,
     * <code>S3Uri</code> identifies an object that is a gzip-compressed TAR archive. SageMaker will attempt to
     * decompress and untar the object during model deployment.
     * </p>
     * <p>
     * If you choose <code>None</code> and chooose <code>S3Object</code> as the value of <code>S3DataType</code>,
     * <code>S3Uri</code> identifies an object that represents an uncompressed ML model to deploy.
     * </p>
     * <p>
     * If you choose None and choose <code>S3Prefix</code> as the value of <code>S3DataType</code>, <code>S3Uri</code>
     * identifies a key name prefix, under which all objects represents the uncompressed ML model to deploy.
     * </p>
     * <p>
     * If you choose None, then SageMaker will follow rules below when creating model data files under /opt/ml/model
     * directory for use by your inference code:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you choose <code>S3Object</code> as the value of <code>S3DataType</code>, then SageMaker will split the key of
     * the S3 object referenced by <code>S3Uri</code> by slash (/), and use the last part as the filename of the file
     * holding the content of the S3 object.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you choose <code>S3Prefix</code> as the value of <code>S3DataType</code>, then for each S3 object under the
     * key name pefix referenced by <code>S3Uri</code>, SageMaker will trim its key by the prefix, and use the remainder
     * as the path (relative to <code>/opt/ml/model</code>) of the file holding the content of the S3 object. SageMaker
     * will split the remainder by slash (/), using intermediate parts as directory names and the last part as filename
     * of the file holding the content of the S3 object.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do not use any of the following as file names or directory names:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An empty or blank string
     * </p>
     * </li>
     * <li>
     * <p>
     * A string which contains null bytes
     * </p>
     * </li>
     * <li>
     * <p>
     * A string longer than 255 bytes
     * </p>
     * </li>
     * <li>
     * <p>
     * A single dot (<code>.</code>)
     * </p>
     * </li>
     * <li>
     * <p>
     * A double dot (<code>..</code>)
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Ambiguous file names will result in model deployment failure. For example, if your uncompressed ML model consists
     * of two S3 objects <code>s3://mybucket/model/weights</code> and <code>s3://mybucket/model/weights/part1</code> and
     * you specify <code>s3://mybucket/model/</code> as the value of <code>S3Uri</code> and <code>S3Prefix</code> as the
     * value of <code>S3DataType</code>, then it will result in name clash between <code>/opt/ml/model/weights</code> (a
     * regular file) and <code>/opt/ml/model/weights/</code> (a directory).
     * </p>
     * </li>
     * <li>
     * <p>
     * Do not organize the model artifacts in <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-folders.html">S3 console using folders</a>.
     * When you create a folder in S3 console, S3 creates a 0-byte object with a key set to the folder name you provide.
     * They key of the 0-byte object ends with a slash (/) which violates SageMaker restrictions on model artifact file
     * names, leading to model deployment failure.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compressionType}
     * will return {@link ModelCompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #compressionTypeAsString}.
     * </p>
     * 
     * @return Specifies how the ML model data is prepared.</p>
     *         <p>
     *         If you choose <code>Gzip</code> and choose <code>S3Object</code> as the value of <code>S3DataType</code>,
     *         <code>S3Uri</code> identifies an object that is a gzip-compressed TAR archive. SageMaker will attempt to
     *         decompress and untar the object during model deployment.
     *         </p>
     *         <p>
     *         If you choose <code>None</code> and chooose <code>S3Object</code> as the value of <code>S3DataType</code>, <code>S3Uri</code> identifies an object that represents an uncompressed ML model to deploy.
     *         </p>
     *         <p>
     *         If you choose None and choose <code>S3Prefix</code> as the value of <code>S3DataType</code>,
     *         <code>S3Uri</code> identifies a key name prefix, under which all objects represents the uncompressed ML
     *         model to deploy.
     *         </p>
     *         <p>
     *         If you choose None, then SageMaker will follow rules below when creating model data files under
     *         /opt/ml/model directory for use by your inference code:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you choose <code>S3Object</code> as the value of <code>S3DataType</code>, then SageMaker will split
     *         the key of the S3 object referenced by <code>S3Uri</code> by slash (/), and use the last part as the
     *         filename of the file holding the content of the S3 object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you choose <code>S3Prefix</code> as the value of <code>S3DataType</code>, then for each S3 object
     *         under the key name pefix referenced by <code>S3Uri</code>, SageMaker will trim its key by the prefix, and
     *         use the remainder as the path (relative to <code>/opt/ml/model</code>) of the file holding the content of
     *         the S3 object. SageMaker will split the remainder by slash (/), using intermediate parts as directory
     *         names and the last part as filename of the file holding the content of the S3 object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do not use any of the following as file names or directory names:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         An empty or blank string
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A string which contains null bytes
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A string longer than 255 bytes
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A single dot (<code>.</code>)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A double dot (<code>..</code>)
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Ambiguous file names will result in model deployment failure. For example, if your uncompressed ML model
     *         consists of two S3 objects <code>s3://mybucket/model/weights</code> and
     *         <code>s3://mybucket/model/weights/part1</code> and you specify <code>s3://mybucket/model/</code> as the
     *         value of <code>S3Uri</code> and <code>S3Prefix</code> as the value of <code>S3DataType</code>, then it
     *         will result in name clash between <code>/opt/ml/model/weights</code> (a regular file) and
     *         <code>/opt/ml/model/weights/</code> (a directory).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do not organize the model artifacts in <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-folders.html">S3 console using
     *         folders</a>. When you create a folder in S3 console, S3 creates a 0-byte object with a key set to the
     *         folder name you provide. They key of the 0-byte object ends with a slash (/) which violates SageMaker
     *         restrictions on model artifact file names, leading to model deployment failure.
     *         </p>
     *         </li>
     * @see ModelCompressionType
     */
    public final String compressionTypeAsString() {
        return compressionType;
    }

    /**
     * <p>
     * Specifies the access configuration file for the ML model. You can explicitly accept the model end-user license
     * agreement (EULA) within the <code>ModelAccessConfig</code>. You are responsible for reviewing and complying with
     * any applicable license terms and making sure they are acceptable for your use case before downloading or using a
     * model.
     * </p>
     * 
     * @return Specifies the access configuration file for the ML model. You can explicitly accept the model end-user
     *         license agreement (EULA) within the <code>ModelAccessConfig</code>. You are responsible for reviewing and
     *         complying with any applicable license terms and making sure they are acceptable for your use case before
     *         downloading or using a model.
     */
    public final ModelAccessConfig modelAccessConfig() {
        return modelAccessConfig;
    }

    /**
     * <p>
     * Configuration information for hub access.
     * </p>
     * 
     * @return Configuration information for hub access.
     */
    public final InferenceHubAccessConfig hubAccessConfig() {
        return hubAccessConfig;
    }

    /**
     * <p>
     * The Amazon S3 URI of the manifest file. The manifest file is a CSV file that stores the artifact locations.
     * </p>
     * 
     * @return The Amazon S3 URI of the manifest file. The manifest file is a CSV file that stores the artifact
     *         locations.
     */
    public final String manifestS3Uri() {
        return manifestS3Uri;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3Uri());
        hashCode = 31 * hashCode + Objects.hashCode(s3DataTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(compressionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(modelAccessConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hubAccessConfig());
        hashCode = 31 * hashCode + Objects.hashCode(manifestS3Uri());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3ModelDataSource)) {
            return false;
        }
        S3ModelDataSource other = (S3ModelDataSource) obj;
        return Objects.equals(s3Uri(), other.s3Uri()) && Objects.equals(s3DataTypeAsString(), other.s3DataTypeAsString())
                && Objects.equals(compressionTypeAsString(), other.compressionTypeAsString())
                && Objects.equals(modelAccessConfig(), other.modelAccessConfig())
                && Objects.equals(hubAccessConfig(), other.hubAccessConfig())
                && Objects.equals(manifestS3Uri(), other.manifestS3Uri());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3ModelDataSource").add("S3Uri", s3Uri()).add("S3DataType", s3DataTypeAsString())
                .add("CompressionType", compressionTypeAsString()).add("ModelAccessConfig", modelAccessConfig())
                .add("HubAccessConfig", hubAccessConfig()).add("ManifestS3Uri", manifestS3Uri()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3Uri":
            return Optional.ofNullable(clazz.cast(s3Uri()));
        case "S3DataType":
            return Optional.ofNullable(clazz.cast(s3DataTypeAsString()));
        case "CompressionType":
            return Optional.ofNullable(clazz.cast(compressionTypeAsString()));
        case "ModelAccessConfig":
            return Optional.ofNullable(clazz.cast(modelAccessConfig()));
        case "HubAccessConfig":
            return Optional.ofNullable(clazz.cast(hubAccessConfig()));
        case "ManifestS3Uri":
            return Optional.ofNullable(clazz.cast(manifestS3Uri()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<S3ModelDataSource, T> g) {
        return obj -> g.apply((S3ModelDataSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3ModelDataSource> {
        /**
         * <p>
         * Specifies the S3 path of ML model data to deploy.
         * </p>
         * 
         * @param s3Uri
         *        Specifies the S3 path of ML model data to deploy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Uri(String s3Uri);

        /**
         * <p>
         * Specifies the type of ML model data to deploy.
         * </p>
         * <p>
         * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all
         * objects that match the specified key name prefix as part of the ML model data to deploy. A valid key name
         * prefix identified by <code>S3Uri</code> always ends with a forward slash (/).
         * </p>
         * <p>
         * If you choose <code>S3Object</code>, <code>S3Uri</code> identifies an object that is the ML model data to
         * deploy.
         * </p>
         * 
         * @param s3DataType
         *        Specifies the type of ML model data to deploy.</p>
         *        <p>
         *        If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses
         *        all objects that match the specified key name prefix as part of the ML model data to deploy. A valid
         *        key name prefix identified by <code>S3Uri</code> always ends with a forward slash (/).
         *        </p>
         *        <p>
         *        If you choose <code>S3Object</code>, <code>S3Uri</code> identifies an object that is the ML model data
         *        to deploy.
         * @see S3ModelDataType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ModelDataType
         */
        Builder s3DataType(String s3DataType);

        /**
         * <p>
         * Specifies the type of ML model data to deploy.
         * </p>
         * <p>
         * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all
         * objects that match the specified key name prefix as part of the ML model data to deploy. A valid key name
         * prefix identified by <code>S3Uri</code> always ends with a forward slash (/).
         * </p>
         * <p>
         * If you choose <code>S3Object</code>, <code>S3Uri</code> identifies an object that is the ML model data to
         * deploy.
         * </p>
         * 
         * @param s3DataType
         *        Specifies the type of ML model data to deploy.</p>
         *        <p>
         *        If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses
         *        all objects that match the specified key name prefix as part of the ML model data to deploy. A valid
         *        key name prefix identified by <code>S3Uri</code> always ends with a forward slash (/).
         *        </p>
         *        <p>
         *        If you choose <code>S3Object</code>, <code>S3Uri</code> identifies an object that is the ML model data
         *        to deploy.
         * @see S3ModelDataType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ModelDataType
         */
        Builder s3DataType(S3ModelDataType s3DataType);

        /**
         * <p>
         * Specifies how the ML model data is prepared.
         * </p>
         * <p>
         * If you choose <code>Gzip</code> and choose <code>S3Object</code> as the value of <code>S3DataType</code>,
         * <code>S3Uri</code> identifies an object that is a gzip-compressed TAR archive. SageMaker will attempt to
         * decompress and untar the object during model deployment.
         * </p>
         * <p>
         * If you choose <code>None</code> and chooose <code>S3Object</code> as the value of <code>S3DataType</code>,
         * <code>S3Uri</code> identifies an object that represents an uncompressed ML model to deploy.
         * </p>
         * <p>
         * If you choose None and choose <code>S3Prefix</code> as the value of <code>S3DataType</code>,
         * <code>S3Uri</code> identifies a key name prefix, under which all objects represents the uncompressed ML model
         * to deploy.
         * </p>
         * <p>
         * If you choose None, then SageMaker will follow rules below when creating model data files under /opt/ml/model
         * directory for use by your inference code:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you choose <code>S3Object</code> as the value of <code>S3DataType</code>, then SageMaker will split the
         * key of the S3 object referenced by <code>S3Uri</code> by slash (/), and use the last part as the filename of
         * the file holding the content of the S3 object.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you choose <code>S3Prefix</code> as the value of <code>S3DataType</code>, then for each S3 object under
         * the key name pefix referenced by <code>S3Uri</code>, SageMaker will trim its key by the prefix, and use the
         * remainder as the path (relative to <code>/opt/ml/model</code>) of the file holding the content of the S3
         * object. SageMaker will split the remainder by slash (/), using intermediate parts as directory names and the
         * last part as filename of the file holding the content of the S3 object.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use any of the following as file names or directory names:
         * </p>
         * <ul>
         * <li>
         * <p>
         * An empty or blank string
         * </p>
         * </li>
         * <li>
         * <p>
         * A string which contains null bytes
         * </p>
         * </li>
         * <li>
         * <p>
         * A string longer than 255 bytes
         * </p>
         * </li>
         * <li>
         * <p>
         * A single dot (<code>.</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * A double dot (<code>..</code>)
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Ambiguous file names will result in model deployment failure. For example, if your uncompressed ML model
         * consists of two S3 objects <code>s3://mybucket/model/weights</code> and
         * <code>s3://mybucket/model/weights/part1</code> and you specify <code>s3://mybucket/model/</code> as the value
         * of <code>S3Uri</code> and <code>S3Prefix</code> as the value of <code>S3DataType</code>, then it will result
         * in name clash between <code>/opt/ml/model/weights</code> (a regular file) and
         * <code>/opt/ml/model/weights/</code> (a directory).
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not organize the model artifacts in <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-folders.html">S3 console using folders</a>.
         * When you create a folder in S3 console, S3 creates a 0-byte object with a key set to the folder name you
         * provide. They key of the 0-byte object ends with a slash (/) which violates SageMaker restrictions on model
         * artifact file names, leading to model deployment failure.
         * </p>
         * </li>
         * </ul>
         * 
         * @param compressionType
         *        Specifies how the ML model data is prepared.</p>
         *        <p>
         *        If you choose <code>Gzip</code> and choose <code>S3Object</code> as the value of
         *        <code>S3DataType</code>, <code>S3Uri</code> identifies an object that is a gzip-compressed TAR
         *        archive. SageMaker will attempt to decompress and untar the object during model deployment.
         *        </p>
         *        <p>
         *        If you choose <code>None</code> and chooose <code>S3Object</code> as the value of
         *        <code>S3DataType</code>, <code>S3Uri</code> identifies an object that represents an uncompressed ML
         *        model to deploy.
         *        </p>
         *        <p>
         *        If you choose None and choose <code>S3Prefix</code> as the value of <code>S3DataType</code>,
         *        <code>S3Uri</code> identifies a key name prefix, under which all objects represents the uncompressed
         *        ML model to deploy.
         *        </p>
         *        <p>
         *        If you choose None, then SageMaker will follow rules below when creating model data files under
         *        /opt/ml/model directory for use by your inference code:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you choose <code>S3Object</code> as the value of <code>S3DataType</code>, then SageMaker will split
         *        the key of the S3 object referenced by <code>S3Uri</code> by slash (/), and use the last part as the
         *        filename of the file holding the content of the S3 object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you choose <code>S3Prefix</code> as the value of <code>S3DataType</code>, then for each S3 object
         *        under the key name pefix referenced by <code>S3Uri</code>, SageMaker will trim its key by the prefix,
         *        and use the remainder as the path (relative to <code>/opt/ml/model</code>) of the file holding the
         *        content of the S3 object. SageMaker will split the remainder by slash (/), using intermediate parts as
         *        directory names and the last part as filename of the file holding the content of the S3 object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use any of the following as file names or directory names:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        An empty or blank string
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A string which contains null bytes
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A string longer than 255 bytes
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A single dot (<code>.</code>)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A double dot (<code>..</code>)
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Ambiguous file names will result in model deployment failure. For example, if your uncompressed ML
         *        model consists of two S3 objects <code>s3://mybucket/model/weights</code> and
         *        <code>s3://mybucket/model/weights/part1</code> and you specify <code>s3://mybucket/model/</code> as
         *        the value of <code>S3Uri</code> and <code>S3Prefix</code> as the value of <code>S3DataType</code>,
         *        then it will result in name clash between <code>/opt/ml/model/weights</code> (a regular file) and
         *        <code>/opt/ml/model/weights/</code> (a directory).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not organize the model artifacts in <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-folders.html">S3 console using
         *        folders</a>. When you create a folder in S3 console, S3 creates a 0-byte object with a key set to the
         *        folder name you provide. They key of the 0-byte object ends with a slash (/) which violates SageMaker
         *        restrictions on model artifact file names, leading to model deployment failure.
         *        </p>
         *        </li>
         * @see ModelCompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ModelCompressionType
         */
        Builder compressionType(String compressionType);

        /**
         * <p>
         * Specifies how the ML model data is prepared.
         * </p>
         * <p>
         * If you choose <code>Gzip</code> and choose <code>S3Object</code> as the value of <code>S3DataType</code>,
         * <code>S3Uri</code> identifies an object that is a gzip-compressed TAR archive. SageMaker will attempt to
         * decompress and untar the object during model deployment.
         * </p>
         * <p>
         * If you choose <code>None</code> and chooose <code>S3Object</code> as the value of <code>S3DataType</code>,
         * <code>S3Uri</code> identifies an object that represents an uncompressed ML model to deploy.
         * </p>
         * <p>
         * If you choose None and choose <code>S3Prefix</code> as the value of <code>S3DataType</code>,
         * <code>S3Uri</code> identifies a key name prefix, under which all objects represents the uncompressed ML model
         * to deploy.
         * </p>
         * <p>
         * If you choose None, then SageMaker will follow rules below when creating model data files under /opt/ml/model
         * directory for use by your inference code:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you choose <code>S3Object</code> as the value of <code>S3DataType</code>, then SageMaker will split the
         * key of the S3 object referenced by <code>S3Uri</code> by slash (/), and use the last part as the filename of
         * the file holding the content of the S3 object.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you choose <code>S3Prefix</code> as the value of <code>S3DataType</code>, then for each S3 object under
         * the key name pefix referenced by <code>S3Uri</code>, SageMaker will trim its key by the prefix, and use the
         * remainder as the path (relative to <code>/opt/ml/model</code>) of the file holding the content of the S3
         * object. SageMaker will split the remainder by slash (/), using intermediate parts as directory names and the
         * last part as filename of the file holding the content of the S3 object.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use any of the following as file names or directory names:
         * </p>
         * <ul>
         * <li>
         * <p>
         * An empty or blank string
         * </p>
         * </li>
         * <li>
         * <p>
         * A string which contains null bytes
         * </p>
         * </li>
         * <li>
         * <p>
         * A string longer than 255 bytes
         * </p>
         * </li>
         * <li>
         * <p>
         * A single dot (<code>.</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * A double dot (<code>..</code>)
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Ambiguous file names will result in model deployment failure. For example, if your uncompressed ML model
         * consists of two S3 objects <code>s3://mybucket/model/weights</code> and
         * <code>s3://mybucket/model/weights/part1</code> and you specify <code>s3://mybucket/model/</code> as the value
         * of <code>S3Uri</code> and <code>S3Prefix</code> as the value of <code>S3DataType</code>, then it will result
         * in name clash between <code>/opt/ml/model/weights</code> (a regular file) and
         * <code>/opt/ml/model/weights/</code> (a directory).
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not organize the model artifacts in <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-folders.html">S3 console using folders</a>.
         * When you create a folder in S3 console, S3 creates a 0-byte object with a key set to the folder name you
         * provide. They key of the 0-byte object ends with a slash (/) which violates SageMaker restrictions on model
         * artifact file names, leading to model deployment failure.
         * </p>
         * </li>
         * </ul>
         * 
         * @param compressionType
         *        Specifies how the ML model data is prepared.</p>
         *        <p>
         *        If you choose <code>Gzip</code> and choose <code>S3Object</code> as the value of
         *        <code>S3DataType</code>, <code>S3Uri</code> identifies an object that is a gzip-compressed TAR
         *        archive. SageMaker will attempt to decompress and untar the object during model deployment.
         *        </p>
         *        <p>
         *        If you choose <code>None</code> and chooose <code>S3Object</code> as the value of
         *        <code>S3DataType</code>, <code>S3Uri</code> identifies an object that represents an uncompressed ML
         *        model to deploy.
         *        </p>
         *        <p>
         *        If you choose None and choose <code>S3Prefix</code> as the value of <code>S3DataType</code>,
         *        <code>S3Uri</code> identifies a key name prefix, under which all objects represents the uncompressed
         *        ML model to deploy.
         *        </p>
         *        <p>
         *        If you choose None, then SageMaker will follow rules below when creating model data files under
         *        /opt/ml/model directory for use by your inference code:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you choose <code>S3Object</code> as the value of <code>S3DataType</code>, then SageMaker will split
         *        the key of the S3 object referenced by <code>S3Uri</code> by slash (/), and use the last part as the
         *        filename of the file holding the content of the S3 object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you choose <code>S3Prefix</code> as the value of <code>S3DataType</code>, then for each S3 object
         *        under the key name pefix referenced by <code>S3Uri</code>, SageMaker will trim its key by the prefix,
         *        and use the remainder as the path (relative to <code>/opt/ml/model</code>) of the file holding the
         *        content of the S3 object. SageMaker will split the remainder by slash (/), using intermediate parts as
         *        directory names and the last part as filename of the file holding the content of the S3 object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use any of the following as file names or directory names:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        An empty or blank string
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A string which contains null bytes
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A string longer than 255 bytes
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A single dot (<code>.</code>)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A double dot (<code>..</code>)
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Ambiguous file names will result in model deployment failure. For example, if your uncompressed ML
         *        model consists of two S3 objects <code>s3://mybucket/model/weights</code> and
         *        <code>s3://mybucket/model/weights/part1</code> and you specify <code>s3://mybucket/model/</code> as
         *        the value of <code>S3Uri</code> and <code>S3Prefix</code> as the value of <code>S3DataType</code>,
         *        then it will result in name clash between <code>/opt/ml/model/weights</code> (a regular file) and
         *        <code>/opt/ml/model/weights/</code> (a directory).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not organize the model artifacts in <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-folders.html">S3 console using
         *        folders</a>. When you create a folder in S3 console, S3 creates a 0-byte object with a key set to the
         *        folder name you provide. They key of the 0-byte object ends with a slash (/) which violates SageMaker
         *        restrictions on model artifact file names, leading to model deployment failure.
         *        </p>
         *        </li>
         * @see ModelCompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ModelCompressionType
         */
        Builder compressionType(ModelCompressionType compressionType);

        /**
         * <p>
         * Specifies the access configuration file for the ML model. You can explicitly accept the model end-user
         * license agreement (EULA) within the <code>ModelAccessConfig</code>. You are responsible for reviewing and
         * complying with any applicable license terms and making sure they are acceptable for your use case before
         * downloading or using a model.
         * </p>
         * 
         * @param modelAccessConfig
         *        Specifies the access configuration file for the ML model. You can explicitly accept the model end-user
         *        license agreement (EULA) within the <code>ModelAccessConfig</code>. You are responsible for reviewing
         *        and complying with any applicable license terms and making sure they are acceptable for your use case
         *        before downloading or using a model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelAccessConfig(ModelAccessConfig modelAccessConfig);

        /**
         * <p>
         * Specifies the access configuration file for the ML model. You can explicitly accept the model end-user
         * license agreement (EULA) within the <code>ModelAccessConfig</code>. You are responsible for reviewing and
         * complying with any applicable license terms and making sure they are acceptable for your use case before
         * downloading or using a model.
         * </p>
         * This is a convenience method that creates an instance of the {@link ModelAccessConfig.Builder} avoiding the
         * need to create one manually via {@link ModelAccessConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ModelAccessConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #modelAccessConfig(ModelAccessConfig)}.
         * 
         * @param modelAccessConfig
         *        a consumer that will call methods on {@link ModelAccessConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modelAccessConfig(ModelAccessConfig)
         */
        default Builder modelAccessConfig(Consumer<ModelAccessConfig.Builder> modelAccessConfig) {
            return modelAccessConfig(ModelAccessConfig.builder().applyMutation(modelAccessConfig).build());
        }

        /**
         * <p>
         * Configuration information for hub access.
         * </p>
         * 
         * @param hubAccessConfig
         *        Configuration information for hub access.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hubAccessConfig(InferenceHubAccessConfig hubAccessConfig);

        /**
         * <p>
         * Configuration information for hub access.
         * </p>
         * This is a convenience method that creates an instance of the {@link InferenceHubAccessConfig.Builder}
         * avoiding the need to create one manually via {@link InferenceHubAccessConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InferenceHubAccessConfig.Builder#build()} is called immediately
         * and its result is passed to {@link #hubAccessConfig(InferenceHubAccessConfig)}.
         * 
         * @param hubAccessConfig
         *        a consumer that will call methods on {@link InferenceHubAccessConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hubAccessConfig(InferenceHubAccessConfig)
         */
        default Builder hubAccessConfig(Consumer<InferenceHubAccessConfig.Builder> hubAccessConfig) {
            return hubAccessConfig(InferenceHubAccessConfig.builder().applyMutation(hubAccessConfig).build());
        }

        /**
         * <p>
         * The Amazon S3 URI of the manifest file. The manifest file is a CSV file that stores the artifact locations.
         * </p>
         * 
         * @param manifestS3Uri
         *        The Amazon S3 URI of the manifest file. The manifest file is a CSV file that stores the artifact
         *        locations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder manifestS3Uri(String manifestS3Uri);
    }

    static final class BuilderImpl implements Builder {
        private String s3Uri;

        private String s3DataType;

        private String compressionType;

        private ModelAccessConfig modelAccessConfig;

        private InferenceHubAccessConfig hubAccessConfig;

        private String manifestS3Uri;

        private BuilderImpl() {
        }

        private BuilderImpl(S3ModelDataSource model) {
            s3Uri(model.s3Uri);
            s3DataType(model.s3DataType);
            compressionType(model.compressionType);
            modelAccessConfig(model.modelAccessConfig);
            hubAccessConfig(model.hubAccessConfig);
            manifestS3Uri(model.manifestS3Uri);
        }

        public final String getS3Uri() {
            return s3Uri;
        }

        public final void setS3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
        }

        @Override
        public final Builder s3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
            return this;
        }

        public final String getS3DataType() {
            return s3DataType;
        }

        public final void setS3DataType(String s3DataType) {
            this.s3DataType = s3DataType;
        }

        @Override
        public final Builder s3DataType(String s3DataType) {
            this.s3DataType = s3DataType;
            return this;
        }

        @Override
        public final Builder s3DataType(S3ModelDataType s3DataType) {
            this.s3DataType(s3DataType == null ? null : s3DataType.toString());
            return this;
        }

        public final String getCompressionType() {
            return compressionType;
        }

        public final void setCompressionType(String compressionType) {
            this.compressionType = compressionType;
        }

        @Override
        public final Builder compressionType(String compressionType) {
            this.compressionType = compressionType;
            return this;
        }

        @Override
        public final Builder compressionType(ModelCompressionType compressionType) {
            this.compressionType(compressionType == null ? null : compressionType.toString());
            return this;
        }

        public final ModelAccessConfig.Builder getModelAccessConfig() {
            return modelAccessConfig != null ? modelAccessConfig.toBuilder() : null;
        }

        public final void setModelAccessConfig(ModelAccessConfig.BuilderImpl modelAccessConfig) {
            this.modelAccessConfig = modelAccessConfig != null ? modelAccessConfig.build() : null;
        }

        @Override
        public final Builder modelAccessConfig(ModelAccessConfig modelAccessConfig) {
            this.modelAccessConfig = modelAccessConfig;
            return this;
        }

        public final InferenceHubAccessConfig.Builder getHubAccessConfig() {
            return hubAccessConfig != null ? hubAccessConfig.toBuilder() : null;
        }

        public final void setHubAccessConfig(InferenceHubAccessConfig.BuilderImpl hubAccessConfig) {
            this.hubAccessConfig = hubAccessConfig != null ? hubAccessConfig.build() : null;
        }

        @Override
        public final Builder hubAccessConfig(InferenceHubAccessConfig hubAccessConfig) {
            this.hubAccessConfig = hubAccessConfig;
            return this;
        }

        public final String getManifestS3Uri() {
            return manifestS3Uri;
        }

        public final void setManifestS3Uri(String manifestS3Uri) {
            this.manifestS3Uri = manifestS3Uri;
        }

        @Override
        public final Builder manifestS3Uri(String manifestS3Uri) {
            this.manifestS3Uri = manifestS3Uri;
            return this;
        }

        @Override
        public S3ModelDataSource build() {
            return new S3ModelDataSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
