/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A collection of settings that apply to users in a domain. These settings are specified when the
 * <code>CreateUserProfile</code> API is called, and as <code>DefaultUserSettings</code> when the
 * <code>CreateDomain</code> API is called.
 * </p>
 * <p>
 * <code>SecurityGroups</code> is aggregated when specified in both calls. For all other settings in
 * <code>UserSettings</code>, the values specified in <code>CreateUserProfile</code> take precedence over those
 * specified in <code>CreateDomain</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UserSettings implements SdkPojo, Serializable, ToCopyableBuilder<UserSettings.Builder, UserSettings> {
    private static final SdkField<String> EXECUTION_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionRole").getter(getter(UserSettings::executionRole)).setter(setter(Builder::executionRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionRole").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroups")
            .getter(getter(UserSettings::securityGroups))
            .setter(setter(Builder::securityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<SharingSettings> SHARING_SETTINGS_FIELD = SdkField
            .<SharingSettings> builder(MarshallingType.SDK_POJO).memberName("SharingSettings")
            .getter(getter(UserSettings::sharingSettings)).setter(setter(Builder::sharingSettings))
            .constructor(SharingSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SharingSettings").build()).build();

    private static final SdkField<JupyterServerAppSettings> JUPYTER_SERVER_APP_SETTINGS_FIELD = SdkField
            .<JupyterServerAppSettings> builder(MarshallingType.SDK_POJO).memberName("JupyterServerAppSettings")
            .getter(getter(UserSettings::jupyterServerAppSettings)).setter(setter(Builder::jupyterServerAppSettings))
            .constructor(JupyterServerAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JupyterServerAppSettings").build())
            .build();

    private static final SdkField<KernelGatewayAppSettings> KERNEL_GATEWAY_APP_SETTINGS_FIELD = SdkField
            .<KernelGatewayAppSettings> builder(MarshallingType.SDK_POJO).memberName("KernelGatewayAppSettings")
            .getter(getter(UserSettings::kernelGatewayAppSettings)).setter(setter(Builder::kernelGatewayAppSettings))
            .constructor(KernelGatewayAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KernelGatewayAppSettings").build())
            .build();

    private static final SdkField<TensorBoardAppSettings> TENSOR_BOARD_APP_SETTINGS_FIELD = SdkField
            .<TensorBoardAppSettings> builder(MarshallingType.SDK_POJO).memberName("TensorBoardAppSettings")
            .getter(getter(UserSettings::tensorBoardAppSettings)).setter(setter(Builder::tensorBoardAppSettings))
            .constructor(TensorBoardAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TensorBoardAppSettings").build())
            .build();

    private static final SdkField<RStudioServerProAppSettings> R_STUDIO_SERVER_PRO_APP_SETTINGS_FIELD = SdkField
            .<RStudioServerProAppSettings> builder(MarshallingType.SDK_POJO)
            .memberName("RStudioServerProAppSettings")
            .getter(getter(UserSettings::rStudioServerProAppSettings))
            .setter(setter(Builder::rStudioServerProAppSettings))
            .constructor(RStudioServerProAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RStudioServerProAppSettings")
                    .build()).build();

    private static final SdkField<RSessionAppSettings> R_SESSION_APP_SETTINGS_FIELD = SdkField
            .<RSessionAppSettings> builder(MarshallingType.SDK_POJO).memberName("RSessionAppSettings")
            .getter(getter(UserSettings::rSessionAppSettings)).setter(setter(Builder::rSessionAppSettings))
            .constructor(RSessionAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RSessionAppSettings").build())
            .build();

    private static final SdkField<CanvasAppSettings> CANVAS_APP_SETTINGS_FIELD = SdkField
            .<CanvasAppSettings> builder(MarshallingType.SDK_POJO).memberName("CanvasAppSettings")
            .getter(getter(UserSettings::canvasAppSettings)).setter(setter(Builder::canvasAppSettings))
            .constructor(CanvasAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CanvasAppSettings").build()).build();

    private static final SdkField<CodeEditorAppSettings> CODE_EDITOR_APP_SETTINGS_FIELD = SdkField
            .<CodeEditorAppSettings> builder(MarshallingType.SDK_POJO).memberName("CodeEditorAppSettings")
            .getter(getter(UserSettings::codeEditorAppSettings)).setter(setter(Builder::codeEditorAppSettings))
            .constructor(CodeEditorAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CodeEditorAppSettings").build())
            .build();

    private static final SdkField<JupyterLabAppSettings> JUPYTER_LAB_APP_SETTINGS_FIELD = SdkField
            .<JupyterLabAppSettings> builder(MarshallingType.SDK_POJO).memberName("JupyterLabAppSettings")
            .getter(getter(UserSettings::jupyterLabAppSettings)).setter(setter(Builder::jupyterLabAppSettings))
            .constructor(JupyterLabAppSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JupyterLabAppSettings").build())
            .build();

    private static final SdkField<DefaultSpaceStorageSettings> SPACE_STORAGE_SETTINGS_FIELD = SdkField
            .<DefaultSpaceStorageSettings> builder(MarshallingType.SDK_POJO).memberName("SpaceStorageSettings")
            .getter(getter(UserSettings::spaceStorageSettings)).setter(setter(Builder::spaceStorageSettings))
            .constructor(DefaultSpaceStorageSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SpaceStorageSettings").build())
            .build();

    private static final SdkField<String> DEFAULT_LANDING_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DefaultLandingUri").getter(getter(UserSettings::defaultLandingUri))
            .setter(setter(Builder::defaultLandingUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultLandingUri").build()).build();

    private static final SdkField<String> STUDIO_WEB_PORTAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StudioWebPortal").getter(getter(UserSettings::studioWebPortalAsString))
            .setter(setter(Builder::studioWebPortal))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StudioWebPortal").build()).build();

    private static final SdkField<CustomPosixUserConfig> CUSTOM_POSIX_USER_CONFIG_FIELD = SdkField
            .<CustomPosixUserConfig> builder(MarshallingType.SDK_POJO).memberName("CustomPosixUserConfig")
            .getter(getter(UserSettings::customPosixUserConfig)).setter(setter(Builder::customPosixUserConfig))
            .constructor(CustomPosixUserConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomPosixUserConfig").build())
            .build();

    private static final SdkField<List<CustomFileSystemConfig>> CUSTOM_FILE_SYSTEM_CONFIGS_FIELD = SdkField
            .<List<CustomFileSystemConfig>> builder(MarshallingType.LIST)
            .memberName("CustomFileSystemConfigs")
            .getter(getter(UserSettings::customFileSystemConfigs))
            .setter(setter(Builder::customFileSystemConfigs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomFileSystemConfigs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CustomFileSystemConfig> builder(MarshallingType.SDK_POJO)
                                            .constructor(CustomFileSystemConfig::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<StudioWebPortalSettings> STUDIO_WEB_PORTAL_SETTINGS_FIELD = SdkField
            .<StudioWebPortalSettings> builder(MarshallingType.SDK_POJO).memberName("StudioWebPortalSettings")
            .getter(getter(UserSettings::studioWebPortalSettings)).setter(setter(Builder::studioWebPortalSettings))
            .constructor(StudioWebPortalSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StudioWebPortalSettings").build())
            .build();

    private static final SdkField<String> AUTO_MOUNT_HOME_EFS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AutoMountHomeEFS").getter(getter(UserSettings::autoMountHomeEFSAsString))
            .setter(setter(Builder::autoMountHomeEFS))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoMountHomeEFS").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXECUTION_ROLE_FIELD,
            SECURITY_GROUPS_FIELD, SHARING_SETTINGS_FIELD, JUPYTER_SERVER_APP_SETTINGS_FIELD, KERNEL_GATEWAY_APP_SETTINGS_FIELD,
            TENSOR_BOARD_APP_SETTINGS_FIELD, R_STUDIO_SERVER_PRO_APP_SETTINGS_FIELD, R_SESSION_APP_SETTINGS_FIELD,
            CANVAS_APP_SETTINGS_FIELD, CODE_EDITOR_APP_SETTINGS_FIELD, JUPYTER_LAB_APP_SETTINGS_FIELD,
            SPACE_STORAGE_SETTINGS_FIELD, DEFAULT_LANDING_URI_FIELD, STUDIO_WEB_PORTAL_FIELD, CUSTOM_POSIX_USER_CONFIG_FIELD,
            CUSTOM_FILE_SYSTEM_CONFIGS_FIELD, STUDIO_WEB_PORTAL_SETTINGS_FIELD, AUTO_MOUNT_HOME_EFS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("ExecutionRole", EXECUTION_ROLE_FIELD);
                    put("SecurityGroups", SECURITY_GROUPS_FIELD);
                    put("SharingSettings", SHARING_SETTINGS_FIELD);
                    put("JupyterServerAppSettings", JUPYTER_SERVER_APP_SETTINGS_FIELD);
                    put("KernelGatewayAppSettings", KERNEL_GATEWAY_APP_SETTINGS_FIELD);
                    put("TensorBoardAppSettings", TENSOR_BOARD_APP_SETTINGS_FIELD);
                    put("RStudioServerProAppSettings", R_STUDIO_SERVER_PRO_APP_SETTINGS_FIELD);
                    put("RSessionAppSettings", R_SESSION_APP_SETTINGS_FIELD);
                    put("CanvasAppSettings", CANVAS_APP_SETTINGS_FIELD);
                    put("CodeEditorAppSettings", CODE_EDITOR_APP_SETTINGS_FIELD);
                    put("JupyterLabAppSettings", JUPYTER_LAB_APP_SETTINGS_FIELD);
                    put("SpaceStorageSettings", SPACE_STORAGE_SETTINGS_FIELD);
                    put("DefaultLandingUri", DEFAULT_LANDING_URI_FIELD);
                    put("StudioWebPortal", STUDIO_WEB_PORTAL_FIELD);
                    put("CustomPosixUserConfig", CUSTOM_POSIX_USER_CONFIG_FIELD);
                    put("CustomFileSystemConfigs", CUSTOM_FILE_SYSTEM_CONFIGS_FIELD);
                    put("StudioWebPortalSettings", STUDIO_WEB_PORTAL_SETTINGS_FIELD);
                    put("AutoMountHomeEFS", AUTO_MOUNT_HOME_EFS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String executionRole;

    private final List<String> securityGroups;

    private final SharingSettings sharingSettings;

    private final JupyterServerAppSettings jupyterServerAppSettings;

    private final KernelGatewayAppSettings kernelGatewayAppSettings;

    private final TensorBoardAppSettings tensorBoardAppSettings;

    private final RStudioServerProAppSettings rStudioServerProAppSettings;

    private final RSessionAppSettings rSessionAppSettings;

    private final CanvasAppSettings canvasAppSettings;

    private final CodeEditorAppSettings codeEditorAppSettings;

    private final JupyterLabAppSettings jupyterLabAppSettings;

    private final DefaultSpaceStorageSettings spaceStorageSettings;

    private final String defaultLandingUri;

    private final String studioWebPortal;

    private final CustomPosixUserConfig customPosixUserConfig;

    private final List<CustomFileSystemConfig> customFileSystemConfigs;

    private final StudioWebPortalSettings studioWebPortalSettings;

    private final String autoMountHomeEFS;

    private UserSettings(BuilderImpl builder) {
        this.executionRole = builder.executionRole;
        this.securityGroups = builder.securityGroups;
        this.sharingSettings = builder.sharingSettings;
        this.jupyterServerAppSettings = builder.jupyterServerAppSettings;
        this.kernelGatewayAppSettings = builder.kernelGatewayAppSettings;
        this.tensorBoardAppSettings = builder.tensorBoardAppSettings;
        this.rStudioServerProAppSettings = builder.rStudioServerProAppSettings;
        this.rSessionAppSettings = builder.rSessionAppSettings;
        this.canvasAppSettings = builder.canvasAppSettings;
        this.codeEditorAppSettings = builder.codeEditorAppSettings;
        this.jupyterLabAppSettings = builder.jupyterLabAppSettings;
        this.spaceStorageSettings = builder.spaceStorageSettings;
        this.defaultLandingUri = builder.defaultLandingUri;
        this.studioWebPortal = builder.studioWebPortal;
        this.customPosixUserConfig = builder.customPosixUserConfig;
        this.customFileSystemConfigs = builder.customFileSystemConfigs;
        this.studioWebPortalSettings = builder.studioWebPortalSettings;
        this.autoMountHomeEFS = builder.autoMountHomeEFS;
    }

    /**
     * <p>
     * The execution role for the user.
     * </p>
     * <p>
     * SageMaker applies this setting only to private spaces that the user creates in the domain. SageMaker doesn't
     * apply this setting to shared spaces.
     * </p>
     * 
     * @return The execution role for the user.</p>
     *         <p>
     *         SageMaker applies this setting only to private spaces that the user creates in the domain. SageMaker
     *         doesn't apply this setting to shared spaces.
     */
    public final String executionRole() {
        return executionRole;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroups() {
        return securityGroups != null && !(securityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The security groups for the Amazon Virtual Private Cloud (VPC) that the domain uses for communication.
     * </p>
     * <p>
     * Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
     * <code>PublicInternetOnly</code>.
     * </p>
     * <p>
     * Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to <code>VpcOnly</code>, unless
     * specified as part of the <code>DefaultUserSettings</code> for the domain.
     * </p>
     * <p>
     * Amazon SageMaker adds a security group to allow NFS traffic from Amazon SageMaker Studio. Therefore, the number
     * of security groups that you can specify is one less than the maximum number shown.
     * </p>
     * <p>
     * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker doesn't
     * apply these settings to shared spaces.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroups} method.
     * </p>
     * 
     * @return The security groups for the Amazon Virtual Private Cloud (VPC) that the domain uses for
     *         communication.</p>
     *         <p>
     *         Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
     *         <code>PublicInternetOnly</code>.
     *         </p>
     *         <p>
     *         Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to <code>VpcOnly</code>
     *         , unless specified as part of the <code>DefaultUserSettings</code> for the domain.
     *         </p>
     *         <p>
     *         Amazon SageMaker adds a security group to allow NFS traffic from Amazon SageMaker Studio. Therefore, the
     *         number of security groups that you can specify is one less than the maximum number shown.
     *         </p>
     *         <p>
     *         SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
     *         doesn't apply these settings to shared spaces.
     */
    public final List<String> securityGroups() {
        return securityGroups;
    }

    /**
     * <p>
     * Specifies options for sharing Amazon SageMaker Studio notebooks.
     * </p>
     * 
     * @return Specifies options for sharing Amazon SageMaker Studio notebooks.
     */
    public final SharingSettings sharingSettings() {
        return sharingSettings;
    }

    /**
     * <p>
     * The Jupyter server's app settings.
     * </p>
     * 
     * @return The Jupyter server's app settings.
     */
    public final JupyterServerAppSettings jupyterServerAppSettings() {
        return jupyterServerAppSettings;
    }

    /**
     * <p>
     * The kernel gateway app settings.
     * </p>
     * 
     * @return The kernel gateway app settings.
     */
    public final KernelGatewayAppSettings kernelGatewayAppSettings() {
        return kernelGatewayAppSettings;
    }

    /**
     * <p>
     * The TensorBoard app settings.
     * </p>
     * 
     * @return The TensorBoard app settings.
     */
    public final TensorBoardAppSettings tensorBoardAppSettings() {
        return tensorBoardAppSettings;
    }

    /**
     * <p>
     * A collection of settings that configure user interaction with the <code>RStudioServerPro</code> app.
     * </p>
     * 
     * @return A collection of settings that configure user interaction with the <code>RStudioServerPro</code> app.
     */
    public final RStudioServerProAppSettings rStudioServerProAppSettings() {
        return rStudioServerProAppSettings;
    }

    /**
     * <p>
     * A collection of settings that configure the <code>RSessionGateway</code> app.
     * </p>
     * 
     * @return A collection of settings that configure the <code>RSessionGateway</code> app.
     */
    public final RSessionAppSettings rSessionAppSettings() {
        return rSessionAppSettings;
    }

    /**
     * <p>
     * The Canvas app settings.
     * </p>
     * <p>
     * SageMaker applies these settings only to private spaces that SageMaker creates for the Canvas app.
     * </p>
     * 
     * @return The Canvas app settings.</p>
     *         <p>
     *         SageMaker applies these settings only to private spaces that SageMaker creates for the Canvas app.
     */
    public final CanvasAppSettings canvasAppSettings() {
        return canvasAppSettings;
    }

    /**
     * <p>
     * The Code Editor application settings.
     * </p>
     * <p>
     * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker doesn't
     * apply these settings to shared spaces.
     * </p>
     * 
     * @return The Code Editor application settings.</p>
     *         <p>
     *         SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
     *         doesn't apply these settings to shared spaces.
     */
    public final CodeEditorAppSettings codeEditorAppSettings() {
        return codeEditorAppSettings;
    }

    /**
     * <p>
     * The settings for the JupyterLab application.
     * </p>
     * <p>
     * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker doesn't
     * apply these settings to shared spaces.
     * </p>
     * 
     * @return The settings for the JupyterLab application.</p>
     *         <p>
     *         SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
     *         doesn't apply these settings to shared spaces.
     */
    public final JupyterLabAppSettings jupyterLabAppSettings() {
        return jupyterLabAppSettings;
    }

    /**
     * <p>
     * The storage settings for a space.
     * </p>
     * <p>
     * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker doesn't
     * apply these settings to shared spaces.
     * </p>
     * 
     * @return The storage settings for a space.</p>
     *         <p>
     *         SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
     *         doesn't apply these settings to shared spaces.
     */
    public final DefaultSpaceStorageSettings spaceStorageSettings() {
        return spaceStorageSettings;
    }

    /**
     * <p>
     * The default experience that the user is directed to when accessing the domain. The supported values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>studio::</code>: Indicates that Studio is the default experience. This value can only be passed if
     * <code>StudioWebPortal</code> is set to <code>ENABLED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>app:JupyterServer:</code>: Indicates that Studio Classic is the default experience.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The default experience that the user is directed to when accessing the domain. The supported values
     *         are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>studio::</code>: Indicates that Studio is the default experience. This value can only be passed if
     *         <code>StudioWebPortal</code> is set to <code>ENABLED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>app:JupyterServer:</code>: Indicates that Studio Classic is the default experience.
     *         </p>
     *         </li>
     */
    public final String defaultLandingUri() {
        return defaultLandingUri;
    }

    /**
     * <p>
     * Whether the user can access Studio. If this value is set to <code>DISABLED</code>, the user cannot access Studio,
     * even if that is the default experience for the domain.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #studioWebPortal}
     * will return {@link StudioWebPortal#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #studioWebPortalAsString}.
     * </p>
     * 
     * @return Whether the user can access Studio. If this value is set to <code>DISABLED</code>, the user cannot access
     *         Studio, even if that is the default experience for the domain.
     * @see StudioWebPortal
     */
    public final StudioWebPortal studioWebPortal() {
        return StudioWebPortal.fromValue(studioWebPortal);
    }

    /**
     * <p>
     * Whether the user can access Studio. If this value is set to <code>DISABLED</code>, the user cannot access Studio,
     * even if that is the default experience for the domain.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #studioWebPortal}
     * will return {@link StudioWebPortal#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #studioWebPortalAsString}.
     * </p>
     * 
     * @return Whether the user can access Studio. If this value is set to <code>DISABLED</code>, the user cannot access
     *         Studio, even if that is the default experience for the domain.
     * @see StudioWebPortal
     */
    public final String studioWebPortalAsString() {
        return studioWebPortal;
    }

    /**
     * <p>
     * Details about the POSIX identity that is used for file system operations.
     * </p>
     * <p>
     * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker doesn't
     * apply these settings to shared spaces.
     * </p>
     * 
     * @return Details about the POSIX identity that is used for file system operations.</p>
     *         <p>
     *         SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
     *         doesn't apply these settings to shared spaces.
     */
    public final CustomPosixUserConfig customPosixUserConfig() {
        return customPosixUserConfig;
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomFileSystemConfigs property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCustomFileSystemConfigs() {
        return customFileSystemConfigs != null && !(customFileSystemConfigs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The settings for assigning a custom file system to a user profile. Permitted users can access this file system in
     * Amazon SageMaker Studio.
     * </p>
     * <p>
     * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker doesn't
     * apply these settings to shared spaces.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomFileSystemConfigs} method.
     * </p>
     * 
     * @return The settings for assigning a custom file system to a user profile. Permitted users can access this file
     *         system in Amazon SageMaker Studio.</p>
     *         <p>
     *         SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
     *         doesn't apply these settings to shared spaces.
     */
    public final List<CustomFileSystemConfig> customFileSystemConfigs() {
        return customFileSystemConfigs;
    }

    /**
     * <p>
     * Studio settings. If these settings are applied on a user level, they take priority over the settings applied on a
     * domain level.
     * </p>
     * 
     * @return Studio settings. If these settings are applied on a user level, they take priority over the settings
     *         applied on a domain level.
     */
    public final StudioWebPortalSettings studioWebPortalSettings() {
        return studioWebPortalSettings;
    }

    /**
     * <p>
     * Indicates whether auto-mounting of an EFS volume is supported for the user profile. The
     * <code>DefaultAsDomain</code> value is only supported for user profiles. Do not use the
     * <code>DefaultAsDomain</code> value when setting this parameter for a domain.
     * </p>
     * <p>
     * SageMaker applies this setting only to private spaces that the user creates in the domain. SageMaker doesn't
     * apply this setting to shared spaces.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #autoMountHomeEFS}
     * will return {@link AutoMountHomeEFS#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #autoMountHomeEFSAsString}.
     * </p>
     * 
     * @return Indicates whether auto-mounting of an EFS volume is supported for the user profile. The
     *         <code>DefaultAsDomain</code> value is only supported for user profiles. Do not use the
     *         <code>DefaultAsDomain</code> value when setting this parameter for a domain.</p>
     *         <p>
     *         SageMaker applies this setting only to private spaces that the user creates in the domain. SageMaker
     *         doesn't apply this setting to shared spaces.
     * @see AutoMountHomeEFS
     */
    public final AutoMountHomeEFS autoMountHomeEFS() {
        return AutoMountHomeEFS.fromValue(autoMountHomeEFS);
    }

    /**
     * <p>
     * Indicates whether auto-mounting of an EFS volume is supported for the user profile. The
     * <code>DefaultAsDomain</code> value is only supported for user profiles. Do not use the
     * <code>DefaultAsDomain</code> value when setting this parameter for a domain.
     * </p>
     * <p>
     * SageMaker applies this setting only to private spaces that the user creates in the domain. SageMaker doesn't
     * apply this setting to shared spaces.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #autoMountHomeEFS}
     * will return {@link AutoMountHomeEFS#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #autoMountHomeEFSAsString}.
     * </p>
     * 
     * @return Indicates whether auto-mounting of an EFS volume is supported for the user profile. The
     *         <code>DefaultAsDomain</code> value is only supported for user profiles. Do not use the
     *         <code>DefaultAsDomain</code> value when setting this parameter for a domain.</p>
     *         <p>
     *         SageMaker applies this setting only to private spaces that the user creates in the domain. SageMaker
     *         doesn't apply this setting to shared spaces.
     * @see AutoMountHomeEFS
     */
    public final String autoMountHomeEFSAsString() {
        return autoMountHomeEFS;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(executionRole());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroups() ? securityGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sharingSettings());
        hashCode = 31 * hashCode + Objects.hashCode(jupyterServerAppSettings());
        hashCode = 31 * hashCode + Objects.hashCode(kernelGatewayAppSettings());
        hashCode = 31 * hashCode + Objects.hashCode(tensorBoardAppSettings());
        hashCode = 31 * hashCode + Objects.hashCode(rStudioServerProAppSettings());
        hashCode = 31 * hashCode + Objects.hashCode(rSessionAppSettings());
        hashCode = 31 * hashCode + Objects.hashCode(canvasAppSettings());
        hashCode = 31 * hashCode + Objects.hashCode(codeEditorAppSettings());
        hashCode = 31 * hashCode + Objects.hashCode(jupyterLabAppSettings());
        hashCode = 31 * hashCode + Objects.hashCode(spaceStorageSettings());
        hashCode = 31 * hashCode + Objects.hashCode(defaultLandingUri());
        hashCode = 31 * hashCode + Objects.hashCode(studioWebPortalAsString());
        hashCode = 31 * hashCode + Objects.hashCode(customPosixUserConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomFileSystemConfigs() ? customFileSystemConfigs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(studioWebPortalSettings());
        hashCode = 31 * hashCode + Objects.hashCode(autoMountHomeEFSAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UserSettings)) {
            return false;
        }
        UserSettings other = (UserSettings) obj;
        return Objects.equals(executionRole(), other.executionRole()) && hasSecurityGroups() == other.hasSecurityGroups()
                && Objects.equals(securityGroups(), other.securityGroups())
                && Objects.equals(sharingSettings(), other.sharingSettings())
                && Objects.equals(jupyterServerAppSettings(), other.jupyterServerAppSettings())
                && Objects.equals(kernelGatewayAppSettings(), other.kernelGatewayAppSettings())
                && Objects.equals(tensorBoardAppSettings(), other.tensorBoardAppSettings())
                && Objects.equals(rStudioServerProAppSettings(), other.rStudioServerProAppSettings())
                && Objects.equals(rSessionAppSettings(), other.rSessionAppSettings())
                && Objects.equals(canvasAppSettings(), other.canvasAppSettings())
                && Objects.equals(codeEditorAppSettings(), other.codeEditorAppSettings())
                && Objects.equals(jupyterLabAppSettings(), other.jupyterLabAppSettings())
                && Objects.equals(spaceStorageSettings(), other.spaceStorageSettings())
                && Objects.equals(defaultLandingUri(), other.defaultLandingUri())
                && Objects.equals(studioWebPortalAsString(), other.studioWebPortalAsString())
                && Objects.equals(customPosixUserConfig(), other.customPosixUserConfig())
                && hasCustomFileSystemConfigs() == other.hasCustomFileSystemConfigs()
                && Objects.equals(customFileSystemConfigs(), other.customFileSystemConfigs())
                && Objects.equals(studioWebPortalSettings(), other.studioWebPortalSettings())
                && Objects.equals(autoMountHomeEFSAsString(), other.autoMountHomeEFSAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UserSettings").add("ExecutionRole", executionRole())
                .add("SecurityGroups", hasSecurityGroups() ? securityGroups() : null).add("SharingSettings", sharingSettings())
                .add("JupyterServerAppSettings", jupyterServerAppSettings())
                .add("KernelGatewayAppSettings", kernelGatewayAppSettings())
                .add("TensorBoardAppSettings", tensorBoardAppSettings())
                .add("RStudioServerProAppSettings", rStudioServerProAppSettings())
                .add("RSessionAppSettings", rSessionAppSettings()).add("CanvasAppSettings", canvasAppSettings())
                .add("CodeEditorAppSettings", codeEditorAppSettings()).add("JupyterLabAppSettings", jupyterLabAppSettings())
                .add("SpaceStorageSettings", spaceStorageSettings()).add("DefaultLandingUri", defaultLandingUri())
                .add("StudioWebPortal", studioWebPortalAsString()).add("CustomPosixUserConfig", customPosixUserConfig())
                .add("CustomFileSystemConfigs", hasCustomFileSystemConfigs() ? customFileSystemConfigs() : null)
                .add("StudioWebPortalSettings", studioWebPortalSettings()).add("AutoMountHomeEFS", autoMountHomeEFSAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ExecutionRole":
            return Optional.ofNullable(clazz.cast(executionRole()));
        case "SecurityGroups":
            return Optional.ofNullable(clazz.cast(securityGroups()));
        case "SharingSettings":
            return Optional.ofNullable(clazz.cast(sharingSettings()));
        case "JupyterServerAppSettings":
            return Optional.ofNullable(clazz.cast(jupyterServerAppSettings()));
        case "KernelGatewayAppSettings":
            return Optional.ofNullable(clazz.cast(kernelGatewayAppSettings()));
        case "TensorBoardAppSettings":
            return Optional.ofNullable(clazz.cast(tensorBoardAppSettings()));
        case "RStudioServerProAppSettings":
            return Optional.ofNullable(clazz.cast(rStudioServerProAppSettings()));
        case "RSessionAppSettings":
            return Optional.ofNullable(clazz.cast(rSessionAppSettings()));
        case "CanvasAppSettings":
            return Optional.ofNullable(clazz.cast(canvasAppSettings()));
        case "CodeEditorAppSettings":
            return Optional.ofNullable(clazz.cast(codeEditorAppSettings()));
        case "JupyterLabAppSettings":
            return Optional.ofNullable(clazz.cast(jupyterLabAppSettings()));
        case "SpaceStorageSettings":
            return Optional.ofNullable(clazz.cast(spaceStorageSettings()));
        case "DefaultLandingUri":
            return Optional.ofNullable(clazz.cast(defaultLandingUri()));
        case "StudioWebPortal":
            return Optional.ofNullable(clazz.cast(studioWebPortalAsString()));
        case "CustomPosixUserConfig":
            return Optional.ofNullable(clazz.cast(customPosixUserConfig()));
        case "CustomFileSystemConfigs":
            return Optional.ofNullable(clazz.cast(customFileSystemConfigs()));
        case "StudioWebPortalSettings":
            return Optional.ofNullable(clazz.cast(studioWebPortalSettings()));
        case "AutoMountHomeEFS":
            return Optional.ofNullable(clazz.cast(autoMountHomeEFSAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<UserSettings, T> g) {
        return obj -> g.apply((UserSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UserSettings> {
        /**
         * <p>
         * The execution role for the user.
         * </p>
         * <p>
         * SageMaker applies this setting only to private spaces that the user creates in the domain. SageMaker doesn't
         * apply this setting to shared spaces.
         * </p>
         * 
         * @param executionRole
         *        The execution role for the user.</p>
         *        <p>
         *        SageMaker applies this setting only to private spaces that the user creates in the domain. SageMaker
         *        doesn't apply this setting to shared spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRole(String executionRole);

        /**
         * <p>
         * The security groups for the Amazon Virtual Private Cloud (VPC) that the domain uses for communication.
         * </p>
         * <p>
         * Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
         * <code>PublicInternetOnly</code>.
         * </p>
         * <p>
         * Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to <code>VpcOnly</code>,
         * unless specified as part of the <code>DefaultUserSettings</code> for the domain.
         * </p>
         * <p>
         * Amazon SageMaker adds a security group to allow NFS traffic from Amazon SageMaker Studio. Therefore, the
         * number of security groups that you can specify is one less than the maximum number shown.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         * doesn't apply these settings to shared spaces.
         * </p>
         * 
         * @param securityGroups
         *        The security groups for the Amazon Virtual Private Cloud (VPC) that the domain uses for
         *        communication.</p>
         *        <p>
         *        Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
         *        <code>PublicInternetOnly</code>.
         *        </p>
         *        <p>
         *        Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
         *        <code>VpcOnly</code>, unless specified as part of the <code>DefaultUserSettings</code> for the domain.
         *        </p>
         *        <p>
         *        Amazon SageMaker adds a security group to allow NFS traffic from Amazon SageMaker Studio. Therefore,
         *        the number of security groups that you can specify is one less than the maximum number shown.
         *        </p>
         *        <p>
         *        SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         *        doesn't apply these settings to shared spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(Collection<String> securityGroups);

        /**
         * <p>
         * The security groups for the Amazon Virtual Private Cloud (VPC) that the domain uses for communication.
         * </p>
         * <p>
         * Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
         * <code>PublicInternetOnly</code>.
         * </p>
         * <p>
         * Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to <code>VpcOnly</code>,
         * unless specified as part of the <code>DefaultUserSettings</code> for the domain.
         * </p>
         * <p>
         * Amazon SageMaker adds a security group to allow NFS traffic from Amazon SageMaker Studio. Therefore, the
         * number of security groups that you can specify is one less than the maximum number shown.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         * doesn't apply these settings to shared spaces.
         * </p>
         * 
         * @param securityGroups
         *        The security groups for the Amazon Virtual Private Cloud (VPC) that the domain uses for
         *        communication.</p>
         *        <p>
         *        Optional when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
         *        <code>PublicInternetOnly</code>.
         *        </p>
         *        <p>
         *        Required when the <code>CreateDomain.AppNetworkAccessType</code> parameter is set to
         *        <code>VpcOnly</code>, unless specified as part of the <code>DefaultUserSettings</code> for the domain.
         *        </p>
         *        <p>
         *        Amazon SageMaker adds a security group to allow NFS traffic from Amazon SageMaker Studio. Therefore,
         *        the number of security groups that you can specify is one less than the maximum number shown.
         *        </p>
         *        <p>
         *        SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         *        doesn't apply these settings to shared spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(String... securityGroups);

        /**
         * <p>
         * Specifies options for sharing Amazon SageMaker Studio notebooks.
         * </p>
         * 
         * @param sharingSettings
         *        Specifies options for sharing Amazon SageMaker Studio notebooks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sharingSettings(SharingSettings sharingSettings);

        /**
         * <p>
         * Specifies options for sharing Amazon SageMaker Studio notebooks.
         * </p>
         * This is a convenience method that creates an instance of the {@link SharingSettings.Builder} avoiding the
         * need to create one manually via {@link SharingSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SharingSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #sharingSettings(SharingSettings)}.
         * 
         * @param sharingSettings
         *        a consumer that will call methods on {@link SharingSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sharingSettings(SharingSettings)
         */
        default Builder sharingSettings(Consumer<SharingSettings.Builder> sharingSettings) {
            return sharingSettings(SharingSettings.builder().applyMutation(sharingSettings).build());
        }

        /**
         * <p>
         * The Jupyter server's app settings.
         * </p>
         * 
         * @param jupyterServerAppSettings
         *        The Jupyter server's app settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jupyterServerAppSettings(JupyterServerAppSettings jupyterServerAppSettings);

        /**
         * <p>
         * The Jupyter server's app settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link JupyterServerAppSettings.Builder}
         * avoiding the need to create one manually via {@link JupyterServerAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JupyterServerAppSettings.Builder#build()} is called immediately
         * and its result is passed to {@link #jupyterServerAppSettings(JupyterServerAppSettings)}.
         * 
         * @param jupyterServerAppSettings
         *        a consumer that will call methods on {@link JupyterServerAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jupyterServerAppSettings(JupyterServerAppSettings)
         */
        default Builder jupyterServerAppSettings(Consumer<JupyterServerAppSettings.Builder> jupyterServerAppSettings) {
            return jupyterServerAppSettings(JupyterServerAppSettings.builder().applyMutation(jupyterServerAppSettings).build());
        }

        /**
         * <p>
         * The kernel gateway app settings.
         * </p>
         * 
         * @param kernelGatewayAppSettings
         *        The kernel gateway app settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kernelGatewayAppSettings(KernelGatewayAppSettings kernelGatewayAppSettings);

        /**
         * <p>
         * The kernel gateway app settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link KernelGatewayAppSettings.Builder}
         * avoiding the need to create one manually via {@link KernelGatewayAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KernelGatewayAppSettings.Builder#build()} is called immediately
         * and its result is passed to {@link #kernelGatewayAppSettings(KernelGatewayAppSettings)}.
         * 
         * @param kernelGatewayAppSettings
         *        a consumer that will call methods on {@link KernelGatewayAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kernelGatewayAppSettings(KernelGatewayAppSettings)
         */
        default Builder kernelGatewayAppSettings(Consumer<KernelGatewayAppSettings.Builder> kernelGatewayAppSettings) {
            return kernelGatewayAppSettings(KernelGatewayAppSettings.builder().applyMutation(kernelGatewayAppSettings).build());
        }

        /**
         * <p>
         * The TensorBoard app settings.
         * </p>
         * 
         * @param tensorBoardAppSettings
         *        The TensorBoard app settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tensorBoardAppSettings(TensorBoardAppSettings tensorBoardAppSettings);

        /**
         * <p>
         * The TensorBoard app settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link TensorBoardAppSettings.Builder} avoiding
         * the need to create one manually via {@link TensorBoardAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TensorBoardAppSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #tensorBoardAppSettings(TensorBoardAppSettings)}.
         * 
         * @param tensorBoardAppSettings
         *        a consumer that will call methods on {@link TensorBoardAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tensorBoardAppSettings(TensorBoardAppSettings)
         */
        default Builder tensorBoardAppSettings(Consumer<TensorBoardAppSettings.Builder> tensorBoardAppSettings) {
            return tensorBoardAppSettings(TensorBoardAppSettings.builder().applyMutation(tensorBoardAppSettings).build());
        }

        /**
         * <p>
         * A collection of settings that configure user interaction with the <code>RStudioServerPro</code> app.
         * </p>
         * 
         * @param rStudioServerProAppSettings
         *        A collection of settings that configure user interaction with the <code>RStudioServerPro</code> app.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rStudioServerProAppSettings(RStudioServerProAppSettings rStudioServerProAppSettings);

        /**
         * <p>
         * A collection of settings that configure user interaction with the <code>RStudioServerPro</code> app.
         * </p>
         * This is a convenience method that creates an instance of the {@link RStudioServerProAppSettings.Builder}
         * avoiding the need to create one manually via {@link RStudioServerProAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RStudioServerProAppSettings.Builder#build()} is called
         * immediately and its result is passed to {@link #rStudioServerProAppSettings(RStudioServerProAppSettings)}.
         * 
         * @param rStudioServerProAppSettings
         *        a consumer that will call methods on {@link RStudioServerProAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rStudioServerProAppSettings(RStudioServerProAppSettings)
         */
        default Builder rStudioServerProAppSettings(Consumer<RStudioServerProAppSettings.Builder> rStudioServerProAppSettings) {
            return rStudioServerProAppSettings(RStudioServerProAppSettings.builder().applyMutation(rStudioServerProAppSettings)
                    .build());
        }

        /**
         * <p>
         * A collection of settings that configure the <code>RSessionGateway</code> app.
         * </p>
         * 
         * @param rSessionAppSettings
         *        A collection of settings that configure the <code>RSessionGateway</code> app.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rSessionAppSettings(RSessionAppSettings rSessionAppSettings);

        /**
         * <p>
         * A collection of settings that configure the <code>RSessionGateway</code> app.
         * </p>
         * This is a convenience method that creates an instance of the {@link RSessionAppSettings.Builder} avoiding the
         * need to create one manually via {@link RSessionAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RSessionAppSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #rSessionAppSettings(RSessionAppSettings)}.
         * 
         * @param rSessionAppSettings
         *        a consumer that will call methods on {@link RSessionAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rSessionAppSettings(RSessionAppSettings)
         */
        default Builder rSessionAppSettings(Consumer<RSessionAppSettings.Builder> rSessionAppSettings) {
            return rSessionAppSettings(RSessionAppSettings.builder().applyMutation(rSessionAppSettings).build());
        }

        /**
         * <p>
         * The Canvas app settings.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that SageMaker creates for the Canvas app.
         * </p>
         * 
         * @param canvasAppSettings
         *        The Canvas app settings.</p>
         *        <p>
         *        SageMaker applies these settings only to private spaces that SageMaker creates for the Canvas app.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder canvasAppSettings(CanvasAppSettings canvasAppSettings);

        /**
         * <p>
         * The Canvas app settings.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that SageMaker creates for the Canvas app.
         * </p>
         * This is a convenience method that creates an instance of the {@link CanvasAppSettings.Builder} avoiding the
         * need to create one manually via {@link CanvasAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CanvasAppSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #canvasAppSettings(CanvasAppSettings)}.
         * 
         * @param canvasAppSettings
         *        a consumer that will call methods on {@link CanvasAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #canvasAppSettings(CanvasAppSettings)
         */
        default Builder canvasAppSettings(Consumer<CanvasAppSettings.Builder> canvasAppSettings) {
            return canvasAppSettings(CanvasAppSettings.builder().applyMutation(canvasAppSettings).build());
        }

        /**
         * <p>
         * The Code Editor application settings.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         * doesn't apply these settings to shared spaces.
         * </p>
         * 
         * @param codeEditorAppSettings
         *        The Code Editor application settings.</p>
         *        <p>
         *        SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         *        doesn't apply these settings to shared spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder codeEditorAppSettings(CodeEditorAppSettings codeEditorAppSettings);

        /**
         * <p>
         * The Code Editor application settings.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         * doesn't apply these settings to shared spaces.
         * </p>
         * This is a convenience method that creates an instance of the {@link CodeEditorAppSettings.Builder} avoiding
         * the need to create one manually via {@link CodeEditorAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CodeEditorAppSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #codeEditorAppSettings(CodeEditorAppSettings)}.
         * 
         * @param codeEditorAppSettings
         *        a consumer that will call methods on {@link CodeEditorAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #codeEditorAppSettings(CodeEditorAppSettings)
         */
        default Builder codeEditorAppSettings(Consumer<CodeEditorAppSettings.Builder> codeEditorAppSettings) {
            return codeEditorAppSettings(CodeEditorAppSettings.builder().applyMutation(codeEditorAppSettings).build());
        }

        /**
         * <p>
         * The settings for the JupyterLab application.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         * doesn't apply these settings to shared spaces.
         * </p>
         * 
         * @param jupyterLabAppSettings
         *        The settings for the JupyterLab application.</p>
         *        <p>
         *        SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         *        doesn't apply these settings to shared spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jupyterLabAppSettings(JupyterLabAppSettings jupyterLabAppSettings);

        /**
         * <p>
         * The settings for the JupyterLab application.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         * doesn't apply these settings to shared spaces.
         * </p>
         * This is a convenience method that creates an instance of the {@link JupyterLabAppSettings.Builder} avoiding
         * the need to create one manually via {@link JupyterLabAppSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JupyterLabAppSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #jupyterLabAppSettings(JupyterLabAppSettings)}.
         * 
         * @param jupyterLabAppSettings
         *        a consumer that will call methods on {@link JupyterLabAppSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jupyterLabAppSettings(JupyterLabAppSettings)
         */
        default Builder jupyterLabAppSettings(Consumer<JupyterLabAppSettings.Builder> jupyterLabAppSettings) {
            return jupyterLabAppSettings(JupyterLabAppSettings.builder().applyMutation(jupyterLabAppSettings).build());
        }

        /**
         * <p>
         * The storage settings for a space.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         * doesn't apply these settings to shared spaces.
         * </p>
         * 
         * @param spaceStorageSettings
         *        The storage settings for a space.</p>
         *        <p>
         *        SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         *        doesn't apply these settings to shared spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spaceStorageSettings(DefaultSpaceStorageSettings spaceStorageSettings);

        /**
         * <p>
         * The storage settings for a space.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         * doesn't apply these settings to shared spaces.
         * </p>
         * This is a convenience method that creates an instance of the {@link DefaultSpaceStorageSettings.Builder}
         * avoiding the need to create one manually via {@link DefaultSpaceStorageSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DefaultSpaceStorageSettings.Builder#build()} is called
         * immediately and its result is passed to {@link #spaceStorageSettings(DefaultSpaceStorageSettings)}.
         * 
         * @param spaceStorageSettings
         *        a consumer that will call methods on {@link DefaultSpaceStorageSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #spaceStorageSettings(DefaultSpaceStorageSettings)
         */
        default Builder spaceStorageSettings(Consumer<DefaultSpaceStorageSettings.Builder> spaceStorageSettings) {
            return spaceStorageSettings(DefaultSpaceStorageSettings.builder().applyMutation(spaceStorageSettings).build());
        }

        /**
         * <p>
         * The default experience that the user is directed to when accessing the domain. The supported values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>studio::</code>: Indicates that Studio is the default experience. This value can only be passed if
         * <code>StudioWebPortal</code> is set to <code>ENABLED</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>app:JupyterServer:</code>: Indicates that Studio Classic is the default experience.
         * </p>
         * </li>
         * </ul>
         * 
         * @param defaultLandingUri
         *        The default experience that the user is directed to when accessing the domain. The supported values
         *        are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>studio::</code>: Indicates that Studio is the default experience. This value can only be passed
         *        if <code>StudioWebPortal</code> is set to <code>ENABLED</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>app:JupyterServer:</code>: Indicates that Studio Classic is the default experience.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultLandingUri(String defaultLandingUri);

        /**
         * <p>
         * Whether the user can access Studio. If this value is set to <code>DISABLED</code>, the user cannot access
         * Studio, even if that is the default experience for the domain.
         * </p>
         * 
         * @param studioWebPortal
         *        Whether the user can access Studio. If this value is set to <code>DISABLED</code>, the user cannot
         *        access Studio, even if that is the default experience for the domain.
         * @see StudioWebPortal
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StudioWebPortal
         */
        Builder studioWebPortal(String studioWebPortal);

        /**
         * <p>
         * Whether the user can access Studio. If this value is set to <code>DISABLED</code>, the user cannot access
         * Studio, even if that is the default experience for the domain.
         * </p>
         * 
         * @param studioWebPortal
         *        Whether the user can access Studio. If this value is set to <code>DISABLED</code>, the user cannot
         *        access Studio, even if that is the default experience for the domain.
         * @see StudioWebPortal
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StudioWebPortal
         */
        Builder studioWebPortal(StudioWebPortal studioWebPortal);

        /**
         * <p>
         * Details about the POSIX identity that is used for file system operations.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         * doesn't apply these settings to shared spaces.
         * </p>
         * 
         * @param customPosixUserConfig
         *        Details about the POSIX identity that is used for file system operations.</p>
         *        <p>
         *        SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         *        doesn't apply these settings to shared spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customPosixUserConfig(CustomPosixUserConfig customPosixUserConfig);

        /**
         * <p>
         * Details about the POSIX identity that is used for file system operations.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         * doesn't apply these settings to shared spaces.
         * </p>
         * This is a convenience method that creates an instance of the {@link CustomPosixUserConfig.Builder} avoiding
         * the need to create one manually via {@link CustomPosixUserConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CustomPosixUserConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #customPosixUserConfig(CustomPosixUserConfig)}.
         * 
         * @param customPosixUserConfig
         *        a consumer that will call methods on {@link CustomPosixUserConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customPosixUserConfig(CustomPosixUserConfig)
         */
        default Builder customPosixUserConfig(Consumer<CustomPosixUserConfig.Builder> customPosixUserConfig) {
            return customPosixUserConfig(CustomPosixUserConfig.builder().applyMutation(customPosixUserConfig).build());
        }

        /**
         * <p>
         * The settings for assigning a custom file system to a user profile. Permitted users can access this file
         * system in Amazon SageMaker Studio.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         * doesn't apply these settings to shared spaces.
         * </p>
         * 
         * @param customFileSystemConfigs
         *        The settings for assigning a custom file system to a user profile. Permitted users can access this
         *        file system in Amazon SageMaker Studio.</p>
         *        <p>
         *        SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         *        doesn't apply these settings to shared spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customFileSystemConfigs(Collection<CustomFileSystemConfig> customFileSystemConfigs);

        /**
         * <p>
         * The settings for assigning a custom file system to a user profile. Permitted users can access this file
         * system in Amazon SageMaker Studio.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         * doesn't apply these settings to shared spaces.
         * </p>
         * 
         * @param customFileSystemConfigs
         *        The settings for assigning a custom file system to a user profile. Permitted users can access this
         *        file system in Amazon SageMaker Studio.</p>
         *        <p>
         *        SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         *        doesn't apply these settings to shared spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customFileSystemConfigs(CustomFileSystemConfig... customFileSystemConfigs);

        /**
         * <p>
         * The settings for assigning a custom file system to a user profile. Permitted users can access this file
         * system in Amazon SageMaker Studio.
         * </p>
         * <p>
         * SageMaker applies these settings only to private spaces that the user creates in the domain. SageMaker
         * doesn't apply these settings to shared spaces.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.CustomFileSystemConfig.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.sagemaker.model.CustomFileSystemConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.CustomFileSystemConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #customFileSystemConfigs(List<CustomFileSystemConfig>)}.
         * 
         * @param customFileSystemConfigs
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.CustomFileSystemConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customFileSystemConfigs(java.util.Collection<CustomFileSystemConfig>)
         */
        Builder customFileSystemConfigs(Consumer<CustomFileSystemConfig.Builder>... customFileSystemConfigs);

        /**
         * <p>
         * Studio settings. If these settings are applied on a user level, they take priority over the settings applied
         * on a domain level.
         * </p>
         * 
         * @param studioWebPortalSettings
         *        Studio settings. If these settings are applied on a user level, they take priority over the settings
         *        applied on a domain level.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder studioWebPortalSettings(StudioWebPortalSettings studioWebPortalSettings);

        /**
         * <p>
         * Studio settings. If these settings are applied on a user level, they take priority over the settings applied
         * on a domain level.
         * </p>
         * This is a convenience method that creates an instance of the {@link StudioWebPortalSettings.Builder} avoiding
         * the need to create one manually via {@link StudioWebPortalSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StudioWebPortalSettings.Builder#build()} is called immediately
         * and its result is passed to {@link #studioWebPortalSettings(StudioWebPortalSettings)}.
         * 
         * @param studioWebPortalSettings
         *        a consumer that will call methods on {@link StudioWebPortalSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #studioWebPortalSettings(StudioWebPortalSettings)
         */
        default Builder studioWebPortalSettings(Consumer<StudioWebPortalSettings.Builder> studioWebPortalSettings) {
            return studioWebPortalSettings(StudioWebPortalSettings.builder().applyMutation(studioWebPortalSettings).build());
        }

        /**
         * <p>
         * Indicates whether auto-mounting of an EFS volume is supported for the user profile. The
         * <code>DefaultAsDomain</code> value is only supported for user profiles. Do not use the
         * <code>DefaultAsDomain</code> value when setting this parameter for a domain.
         * </p>
         * <p>
         * SageMaker applies this setting only to private spaces that the user creates in the domain. SageMaker doesn't
         * apply this setting to shared spaces.
         * </p>
         * 
         * @param autoMountHomeEFS
         *        Indicates whether auto-mounting of an EFS volume is supported for the user profile. The
         *        <code>DefaultAsDomain</code> value is only supported for user profiles. Do not use the
         *        <code>DefaultAsDomain</code> value when setting this parameter for a domain.</p>
         *        <p>
         *        SageMaker applies this setting only to private spaces that the user creates in the domain. SageMaker
         *        doesn't apply this setting to shared spaces.
         * @see AutoMountHomeEFS
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoMountHomeEFS
         */
        Builder autoMountHomeEFS(String autoMountHomeEFS);

        /**
         * <p>
         * Indicates whether auto-mounting of an EFS volume is supported for the user profile. The
         * <code>DefaultAsDomain</code> value is only supported for user profiles. Do not use the
         * <code>DefaultAsDomain</code> value when setting this parameter for a domain.
         * </p>
         * <p>
         * SageMaker applies this setting only to private spaces that the user creates in the domain. SageMaker doesn't
         * apply this setting to shared spaces.
         * </p>
         * 
         * @param autoMountHomeEFS
         *        Indicates whether auto-mounting of an EFS volume is supported for the user profile. The
         *        <code>DefaultAsDomain</code> value is only supported for user profiles. Do not use the
         *        <code>DefaultAsDomain</code> value when setting this parameter for a domain.</p>
         *        <p>
         *        SageMaker applies this setting only to private spaces that the user creates in the domain. SageMaker
         *        doesn't apply this setting to shared spaces.
         * @see AutoMountHomeEFS
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoMountHomeEFS
         */
        Builder autoMountHomeEFS(AutoMountHomeEFS autoMountHomeEFS);
    }

    static final class BuilderImpl implements Builder {
        private String executionRole;

        private List<String> securityGroups = DefaultSdkAutoConstructList.getInstance();

        private SharingSettings sharingSettings;

        private JupyterServerAppSettings jupyterServerAppSettings;

        private KernelGatewayAppSettings kernelGatewayAppSettings;

        private TensorBoardAppSettings tensorBoardAppSettings;

        private RStudioServerProAppSettings rStudioServerProAppSettings;

        private RSessionAppSettings rSessionAppSettings;

        private CanvasAppSettings canvasAppSettings;

        private CodeEditorAppSettings codeEditorAppSettings;

        private JupyterLabAppSettings jupyterLabAppSettings;

        private DefaultSpaceStorageSettings spaceStorageSettings;

        private String defaultLandingUri;

        private String studioWebPortal;

        private CustomPosixUserConfig customPosixUserConfig;

        private List<CustomFileSystemConfig> customFileSystemConfigs = DefaultSdkAutoConstructList.getInstance();

        private StudioWebPortalSettings studioWebPortalSettings;

        private String autoMountHomeEFS;

        private BuilderImpl() {
        }

        private BuilderImpl(UserSettings model) {
            executionRole(model.executionRole);
            securityGroups(model.securityGroups);
            sharingSettings(model.sharingSettings);
            jupyterServerAppSettings(model.jupyterServerAppSettings);
            kernelGatewayAppSettings(model.kernelGatewayAppSettings);
            tensorBoardAppSettings(model.tensorBoardAppSettings);
            rStudioServerProAppSettings(model.rStudioServerProAppSettings);
            rSessionAppSettings(model.rSessionAppSettings);
            canvasAppSettings(model.canvasAppSettings);
            codeEditorAppSettings(model.codeEditorAppSettings);
            jupyterLabAppSettings(model.jupyterLabAppSettings);
            spaceStorageSettings(model.spaceStorageSettings);
            defaultLandingUri(model.defaultLandingUri);
            studioWebPortal(model.studioWebPortal);
            customPosixUserConfig(model.customPosixUserConfig);
            customFileSystemConfigs(model.customFileSystemConfigs);
            studioWebPortalSettings(model.studioWebPortalSettings);
            autoMountHomeEFS(model.autoMountHomeEFS);
        }

        public final String getExecutionRole() {
            return executionRole;
        }

        public final void setExecutionRole(String executionRole) {
            this.executionRole = executionRole;
        }

        @Override
        public final Builder executionRole(String executionRole) {
            this.executionRole = executionRole;
            return this;
        }

        public final Collection<String> getSecurityGroups() {
            if (securityGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroups;
        }

        public final void setSecurityGroups(Collection<String> securityGroups) {
            this.securityGroups = SecurityGroupIdsCopier.copy(securityGroups);
        }

        @Override
        public final Builder securityGroups(Collection<String> securityGroups) {
            this.securityGroups = SecurityGroupIdsCopier.copy(securityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroups(String... securityGroups) {
            securityGroups(Arrays.asList(securityGroups));
            return this;
        }

        public final SharingSettings.Builder getSharingSettings() {
            return sharingSettings != null ? sharingSettings.toBuilder() : null;
        }

        public final void setSharingSettings(SharingSettings.BuilderImpl sharingSettings) {
            this.sharingSettings = sharingSettings != null ? sharingSettings.build() : null;
        }

        @Override
        public final Builder sharingSettings(SharingSettings sharingSettings) {
            this.sharingSettings = sharingSettings;
            return this;
        }

        public final JupyterServerAppSettings.Builder getJupyterServerAppSettings() {
            return jupyterServerAppSettings != null ? jupyterServerAppSettings.toBuilder() : null;
        }

        public final void setJupyterServerAppSettings(JupyterServerAppSettings.BuilderImpl jupyterServerAppSettings) {
            this.jupyterServerAppSettings = jupyterServerAppSettings != null ? jupyterServerAppSettings.build() : null;
        }

        @Override
        public final Builder jupyterServerAppSettings(JupyterServerAppSettings jupyterServerAppSettings) {
            this.jupyterServerAppSettings = jupyterServerAppSettings;
            return this;
        }

        public final KernelGatewayAppSettings.Builder getKernelGatewayAppSettings() {
            return kernelGatewayAppSettings != null ? kernelGatewayAppSettings.toBuilder() : null;
        }

        public final void setKernelGatewayAppSettings(KernelGatewayAppSettings.BuilderImpl kernelGatewayAppSettings) {
            this.kernelGatewayAppSettings = kernelGatewayAppSettings != null ? kernelGatewayAppSettings.build() : null;
        }

        @Override
        public final Builder kernelGatewayAppSettings(KernelGatewayAppSettings kernelGatewayAppSettings) {
            this.kernelGatewayAppSettings = kernelGatewayAppSettings;
            return this;
        }

        public final TensorBoardAppSettings.Builder getTensorBoardAppSettings() {
            return tensorBoardAppSettings != null ? tensorBoardAppSettings.toBuilder() : null;
        }

        public final void setTensorBoardAppSettings(TensorBoardAppSettings.BuilderImpl tensorBoardAppSettings) {
            this.tensorBoardAppSettings = tensorBoardAppSettings != null ? tensorBoardAppSettings.build() : null;
        }

        @Override
        public final Builder tensorBoardAppSettings(TensorBoardAppSettings tensorBoardAppSettings) {
            this.tensorBoardAppSettings = tensorBoardAppSettings;
            return this;
        }

        public final RStudioServerProAppSettings.Builder getRStudioServerProAppSettings() {
            return rStudioServerProAppSettings != null ? rStudioServerProAppSettings.toBuilder() : null;
        }

        public final void setRStudioServerProAppSettings(RStudioServerProAppSettings.BuilderImpl rStudioServerProAppSettings) {
            this.rStudioServerProAppSettings = rStudioServerProAppSettings != null ? rStudioServerProAppSettings.build() : null;
        }

        @Override
        public final Builder rStudioServerProAppSettings(RStudioServerProAppSettings rStudioServerProAppSettings) {
            this.rStudioServerProAppSettings = rStudioServerProAppSettings;
            return this;
        }

        public final RSessionAppSettings.Builder getRSessionAppSettings() {
            return rSessionAppSettings != null ? rSessionAppSettings.toBuilder() : null;
        }

        public final void setRSessionAppSettings(RSessionAppSettings.BuilderImpl rSessionAppSettings) {
            this.rSessionAppSettings = rSessionAppSettings != null ? rSessionAppSettings.build() : null;
        }

        @Override
        public final Builder rSessionAppSettings(RSessionAppSettings rSessionAppSettings) {
            this.rSessionAppSettings = rSessionAppSettings;
            return this;
        }

        public final CanvasAppSettings.Builder getCanvasAppSettings() {
            return canvasAppSettings != null ? canvasAppSettings.toBuilder() : null;
        }

        public final void setCanvasAppSettings(CanvasAppSettings.BuilderImpl canvasAppSettings) {
            this.canvasAppSettings = canvasAppSettings != null ? canvasAppSettings.build() : null;
        }

        @Override
        public final Builder canvasAppSettings(CanvasAppSettings canvasAppSettings) {
            this.canvasAppSettings = canvasAppSettings;
            return this;
        }

        public final CodeEditorAppSettings.Builder getCodeEditorAppSettings() {
            return codeEditorAppSettings != null ? codeEditorAppSettings.toBuilder() : null;
        }

        public final void setCodeEditorAppSettings(CodeEditorAppSettings.BuilderImpl codeEditorAppSettings) {
            this.codeEditorAppSettings = codeEditorAppSettings != null ? codeEditorAppSettings.build() : null;
        }

        @Override
        public final Builder codeEditorAppSettings(CodeEditorAppSettings codeEditorAppSettings) {
            this.codeEditorAppSettings = codeEditorAppSettings;
            return this;
        }

        public final JupyterLabAppSettings.Builder getJupyterLabAppSettings() {
            return jupyterLabAppSettings != null ? jupyterLabAppSettings.toBuilder() : null;
        }

        public final void setJupyterLabAppSettings(JupyterLabAppSettings.BuilderImpl jupyterLabAppSettings) {
            this.jupyterLabAppSettings = jupyterLabAppSettings != null ? jupyterLabAppSettings.build() : null;
        }

        @Override
        public final Builder jupyterLabAppSettings(JupyterLabAppSettings jupyterLabAppSettings) {
            this.jupyterLabAppSettings = jupyterLabAppSettings;
            return this;
        }

        public final DefaultSpaceStorageSettings.Builder getSpaceStorageSettings() {
            return spaceStorageSettings != null ? spaceStorageSettings.toBuilder() : null;
        }

        public final void setSpaceStorageSettings(DefaultSpaceStorageSettings.BuilderImpl spaceStorageSettings) {
            this.spaceStorageSettings = spaceStorageSettings != null ? spaceStorageSettings.build() : null;
        }

        @Override
        public final Builder spaceStorageSettings(DefaultSpaceStorageSettings spaceStorageSettings) {
            this.spaceStorageSettings = spaceStorageSettings;
            return this;
        }

        public final String getDefaultLandingUri() {
            return defaultLandingUri;
        }

        public final void setDefaultLandingUri(String defaultLandingUri) {
            this.defaultLandingUri = defaultLandingUri;
        }

        @Override
        public final Builder defaultLandingUri(String defaultLandingUri) {
            this.defaultLandingUri = defaultLandingUri;
            return this;
        }

        public final String getStudioWebPortal() {
            return studioWebPortal;
        }

        public final void setStudioWebPortal(String studioWebPortal) {
            this.studioWebPortal = studioWebPortal;
        }

        @Override
        public final Builder studioWebPortal(String studioWebPortal) {
            this.studioWebPortal = studioWebPortal;
            return this;
        }

        @Override
        public final Builder studioWebPortal(StudioWebPortal studioWebPortal) {
            this.studioWebPortal(studioWebPortal == null ? null : studioWebPortal.toString());
            return this;
        }

        public final CustomPosixUserConfig.Builder getCustomPosixUserConfig() {
            return customPosixUserConfig != null ? customPosixUserConfig.toBuilder() : null;
        }

        public final void setCustomPosixUserConfig(CustomPosixUserConfig.BuilderImpl customPosixUserConfig) {
            this.customPosixUserConfig = customPosixUserConfig != null ? customPosixUserConfig.build() : null;
        }

        @Override
        public final Builder customPosixUserConfig(CustomPosixUserConfig customPosixUserConfig) {
            this.customPosixUserConfig = customPosixUserConfig;
            return this;
        }

        public final List<CustomFileSystemConfig.Builder> getCustomFileSystemConfigs() {
            List<CustomFileSystemConfig.Builder> result = CustomFileSystemConfigsCopier
                    .copyToBuilder(this.customFileSystemConfigs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCustomFileSystemConfigs(Collection<CustomFileSystemConfig.BuilderImpl> customFileSystemConfigs) {
            this.customFileSystemConfigs = CustomFileSystemConfigsCopier.copyFromBuilder(customFileSystemConfigs);
        }

        @Override
        public final Builder customFileSystemConfigs(Collection<CustomFileSystemConfig> customFileSystemConfigs) {
            this.customFileSystemConfigs = CustomFileSystemConfigsCopier.copy(customFileSystemConfigs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder customFileSystemConfigs(CustomFileSystemConfig... customFileSystemConfigs) {
            customFileSystemConfigs(Arrays.asList(customFileSystemConfigs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder customFileSystemConfigs(Consumer<CustomFileSystemConfig.Builder>... customFileSystemConfigs) {
            customFileSystemConfigs(Stream.of(customFileSystemConfigs)
                    .map(c -> CustomFileSystemConfig.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final StudioWebPortalSettings.Builder getStudioWebPortalSettings() {
            return studioWebPortalSettings != null ? studioWebPortalSettings.toBuilder() : null;
        }

        public final void setStudioWebPortalSettings(StudioWebPortalSettings.BuilderImpl studioWebPortalSettings) {
            this.studioWebPortalSettings = studioWebPortalSettings != null ? studioWebPortalSettings.build() : null;
        }

        @Override
        public final Builder studioWebPortalSettings(StudioWebPortalSettings studioWebPortalSettings) {
            this.studioWebPortalSettings = studioWebPortalSettings;
            return this;
        }

        public final String getAutoMountHomeEFS() {
            return autoMountHomeEFS;
        }

        public final void setAutoMountHomeEFS(String autoMountHomeEFS) {
            this.autoMountHomeEFS = autoMountHomeEFS;
        }

        @Override
        public final Builder autoMountHomeEFS(String autoMountHomeEFS) {
            this.autoMountHomeEFS = autoMountHomeEFS;
            return this;
        }

        @Override
        public final Builder autoMountHomeEFS(AutoMountHomeEFS autoMountHomeEFS) {
            this.autoMountHomeEFS(autoMountHomeEFS == null ? null : autoMountHomeEFS.toString());
            return this;
        }

        @Override
        public UserSettings build() {
            return new UserSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
