/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A single private workforce, which is automatically created when you create your first private work team. You can
 * create one private work force in each Amazon Web Services Region. By default, any workforce-related API operation
 * used in a specific region will apply to the workforce created in that region. To learn how to create a private
 * workforce, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/sms-workforce-create-private.html">Create a
 * Private Workforce</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Workforce implements SdkPojo, Serializable, ToCopyableBuilder<Workforce.Builder, Workforce> {
    private static final SdkField<String> WORKFORCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkforceName").getter(getter(Workforce::workforceName)).setter(setter(Builder::workforceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkforceName").build()).build();

    private static final SdkField<String> WORKFORCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkforceArn").getter(getter(Workforce::workforceArn)).setter(setter(Builder::workforceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkforceArn").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedDate").getter(getter(Workforce::lastUpdatedDate)).setter(setter(Builder::lastUpdatedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedDate").build()).build();

    private static final SdkField<SourceIpConfig> SOURCE_IP_CONFIG_FIELD = SdkField
            .<SourceIpConfig> builder(MarshallingType.SDK_POJO).memberName("SourceIpConfig")
            .getter(getter(Workforce::sourceIpConfig)).setter(setter(Builder::sourceIpConfig))
            .constructor(SourceIpConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceIpConfig").build()).build();

    private static final SdkField<String> SUB_DOMAIN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubDomain").getter(getter(Workforce::subDomain)).setter(setter(Builder::subDomain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubDomain").build()).build();

    private static final SdkField<CognitoConfig> COGNITO_CONFIG_FIELD = SdkField
            .<CognitoConfig> builder(MarshallingType.SDK_POJO).memberName("CognitoConfig")
            .getter(getter(Workforce::cognitoConfig)).setter(setter(Builder::cognitoConfig)).constructor(CognitoConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CognitoConfig").build()).build();

    private static final SdkField<OidcConfigForResponse> OIDC_CONFIG_FIELD = SdkField
            .<OidcConfigForResponse> builder(MarshallingType.SDK_POJO).memberName("OidcConfig")
            .getter(getter(Workforce::oidcConfig)).setter(setter(Builder::oidcConfig))
            .constructor(OidcConfigForResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OidcConfig").build()).build();

    private static final SdkField<Instant> CREATE_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateDate").getter(getter(Workforce::createDate)).setter(setter(Builder::createDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateDate").build()).build();

    private static final SdkField<WorkforceVpcConfigResponse> WORKFORCE_VPC_CONFIG_FIELD = SdkField
            .<WorkforceVpcConfigResponse> builder(MarshallingType.SDK_POJO).memberName("WorkforceVpcConfig")
            .getter(getter(Workforce::workforceVpcConfig)).setter(setter(Builder::workforceVpcConfig))
            .constructor(WorkforceVpcConfigResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkforceVpcConfig").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Workforce::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(Workforce::failureReason)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(WORKFORCE_NAME_FIELD,
            WORKFORCE_ARN_FIELD, LAST_UPDATED_DATE_FIELD, SOURCE_IP_CONFIG_FIELD, SUB_DOMAIN_FIELD, COGNITO_CONFIG_FIELD,
            OIDC_CONFIG_FIELD, CREATE_DATE_FIELD, WORKFORCE_VPC_CONFIG_FIELD, STATUS_FIELD, FAILURE_REASON_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("WorkforceName", WORKFORCE_NAME_FIELD);
                    put("WorkforceArn", WORKFORCE_ARN_FIELD);
                    put("LastUpdatedDate", LAST_UPDATED_DATE_FIELD);
                    put("SourceIpConfig", SOURCE_IP_CONFIG_FIELD);
                    put("SubDomain", SUB_DOMAIN_FIELD);
                    put("CognitoConfig", COGNITO_CONFIG_FIELD);
                    put("OidcConfig", OIDC_CONFIG_FIELD);
                    put("CreateDate", CREATE_DATE_FIELD);
                    put("WorkforceVpcConfig", WORKFORCE_VPC_CONFIG_FIELD);
                    put("Status", STATUS_FIELD);
                    put("FailureReason", FAILURE_REASON_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String workforceName;

    private final String workforceArn;

    private final Instant lastUpdatedDate;

    private final SourceIpConfig sourceIpConfig;

    private final String subDomain;

    private final CognitoConfig cognitoConfig;

    private final OidcConfigForResponse oidcConfig;

    private final Instant createDate;

    private final WorkforceVpcConfigResponse workforceVpcConfig;

    private final String status;

    private final String failureReason;

    private Workforce(BuilderImpl builder) {
        this.workforceName = builder.workforceName;
        this.workforceArn = builder.workforceArn;
        this.lastUpdatedDate = builder.lastUpdatedDate;
        this.sourceIpConfig = builder.sourceIpConfig;
        this.subDomain = builder.subDomain;
        this.cognitoConfig = builder.cognitoConfig;
        this.oidcConfig = builder.oidcConfig;
        this.createDate = builder.createDate;
        this.workforceVpcConfig = builder.workforceVpcConfig;
        this.status = builder.status;
        this.failureReason = builder.failureReason;
    }

    /**
     * <p>
     * The name of the private workforce.
     * </p>
     * 
     * @return The name of the private workforce.
     */
    public final String workforceName() {
        return workforceName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the private workforce.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the private workforce.
     */
    public final String workforceArn() {
        return workforceArn;
    }

    /**
     * <p>
     * The most recent date that <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateWorkforce.html">UpdateWorkforce</a> was
     * used to successfully add one or more IP address ranges (<a
     * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">CIDRs</a>) to a private workforce's
     * allow list.
     * </p>
     * 
     * @return The most recent date that <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateWorkforce.html"
     *         >UpdateWorkforce</a> was used to successfully add one or more IP address ranges (<a
     *         href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">CIDRs</a>) to a private
     *         workforce's allow list.
     */
    public final Instant lastUpdatedDate() {
        return lastUpdatedDate;
    }

    /**
     * <p>
     * A list of one to ten IP address ranges (<a
     * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">CIDRs</a>) to be added to the workforce
     * allow list. By default, a workforce isn't restricted to specific IP addresses.
     * </p>
     * 
     * @return A list of one to ten IP address ranges (<a
     *         href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">CIDRs</a>) to be added to the
     *         workforce allow list. By default, a workforce isn't restricted to specific IP addresses.
     */
    public final SourceIpConfig sourceIpConfig() {
        return sourceIpConfig;
    }

    /**
     * <p>
     * The subdomain for your OIDC Identity Provider.
     * </p>
     * 
     * @return The subdomain for your OIDC Identity Provider.
     */
    public final String subDomain() {
        return subDomain;
    }

    /**
     * <p>
     * The configuration of an Amazon Cognito workforce. A single Cognito workforce is created using and corresponds to
     * a single <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html">
     * Amazon Cognito user pool</a>.
     * </p>
     * 
     * @return The configuration of an Amazon Cognito workforce. A single Cognito workforce is created using and
     *         corresponds to a single <a
     *         href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html"> Amazon
     *         Cognito user pool</a>.
     */
    public final CognitoConfig cognitoConfig() {
        return cognitoConfig;
    }

    /**
     * <p>
     * The configuration of an OIDC Identity Provider (IdP) private workforce.
     * </p>
     * 
     * @return The configuration of an OIDC Identity Provider (IdP) private workforce.
     */
    public final OidcConfigForResponse oidcConfig() {
        return oidcConfig;
    }

    /**
     * <p>
     * The date that the workforce is created.
     * </p>
     * 
     * @return The date that the workforce is created.
     */
    public final Instant createDate() {
        return createDate;
    }

    /**
     * <p>
     * The configuration of a VPC workforce.
     * </p>
     * 
     * @return The configuration of a VPC workforce.
     */
    public final WorkforceVpcConfigResponse workforceVpcConfig() {
        return workforceVpcConfig;
    }

    /**
     * <p>
     * The status of your workforce.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link WorkforceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of your workforce.
     * @see WorkforceStatus
     */
    public final WorkforceStatus status() {
        return WorkforceStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of your workforce.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link WorkforceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of your workforce.
     * @see WorkforceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The reason your workforce failed.
     * </p>
     * 
     * @return The reason your workforce failed.
     */
    public final String failureReason() {
        return failureReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(workforceName());
        hashCode = 31 * hashCode + Objects.hashCode(workforceArn());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedDate());
        hashCode = 31 * hashCode + Objects.hashCode(sourceIpConfig());
        hashCode = 31 * hashCode + Objects.hashCode(subDomain());
        hashCode = 31 * hashCode + Objects.hashCode(cognitoConfig());
        hashCode = 31 * hashCode + Objects.hashCode(oidcConfig());
        hashCode = 31 * hashCode + Objects.hashCode(createDate());
        hashCode = 31 * hashCode + Objects.hashCode(workforceVpcConfig());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Workforce)) {
            return false;
        }
        Workforce other = (Workforce) obj;
        return Objects.equals(workforceName(), other.workforceName()) && Objects.equals(workforceArn(), other.workforceArn())
                && Objects.equals(lastUpdatedDate(), other.lastUpdatedDate())
                && Objects.equals(sourceIpConfig(), other.sourceIpConfig()) && Objects.equals(subDomain(), other.subDomain())
                && Objects.equals(cognitoConfig(), other.cognitoConfig()) && Objects.equals(oidcConfig(), other.oidcConfig())
                && Objects.equals(createDate(), other.createDate())
                && Objects.equals(workforceVpcConfig(), other.workforceVpcConfig())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(failureReason(), other.failureReason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Workforce").add("WorkforceName", workforceName()).add("WorkforceArn", workforceArn())
                .add("LastUpdatedDate", lastUpdatedDate()).add("SourceIpConfig", sourceIpConfig()).add("SubDomain", subDomain())
                .add("CognitoConfig", cognitoConfig()).add("OidcConfig", oidcConfig()).add("CreateDate", createDate())
                .add("WorkforceVpcConfig", workforceVpcConfig()).add("Status", statusAsString())
                .add("FailureReason", failureReason()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WorkforceName":
            return Optional.ofNullable(clazz.cast(workforceName()));
        case "WorkforceArn":
            return Optional.ofNullable(clazz.cast(workforceArn()));
        case "LastUpdatedDate":
            return Optional.ofNullable(clazz.cast(lastUpdatedDate()));
        case "SourceIpConfig":
            return Optional.ofNullable(clazz.cast(sourceIpConfig()));
        case "SubDomain":
            return Optional.ofNullable(clazz.cast(subDomain()));
        case "CognitoConfig":
            return Optional.ofNullable(clazz.cast(cognitoConfig()));
        case "OidcConfig":
            return Optional.ofNullable(clazz.cast(oidcConfig()));
        case "CreateDate":
            return Optional.ofNullable(clazz.cast(createDate()));
        case "WorkforceVpcConfig":
            return Optional.ofNullable(clazz.cast(workforceVpcConfig()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<Workforce, T> g) {
        return obj -> g.apply((Workforce) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Workforce> {
        /**
         * <p>
         * The name of the private workforce.
         * </p>
         * 
         * @param workforceName
         *        The name of the private workforce.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workforceName(String workforceName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the private workforce.
         * </p>
         * 
         * @param workforceArn
         *        The Amazon Resource Name (ARN) of the private workforce.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workforceArn(String workforceArn);

        /**
         * <p>
         * The most recent date that <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateWorkforce.html">UpdateWorkforce</a>
         * was used to successfully add one or more IP address ranges (<a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">CIDRs</a>) to a private workforce's
         * allow list.
         * </p>
         * 
         * @param lastUpdatedDate
         *        The most recent date that <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateWorkforce.html"
         *        >UpdateWorkforce</a> was used to successfully add one or more IP address ranges (<a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">CIDRs</a>) to a private
         *        workforce's allow list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedDate(Instant lastUpdatedDate);

        /**
         * <p>
         * A list of one to ten IP address ranges (<a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">CIDRs</a>) to be added to the
         * workforce allow list. By default, a workforce isn't restricted to specific IP addresses.
         * </p>
         * 
         * @param sourceIpConfig
         *        A list of one to ten IP address ranges (<a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">CIDRs</a>) to be added to the
         *        workforce allow list. By default, a workforce isn't restricted to specific IP addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIpConfig(SourceIpConfig sourceIpConfig);

        /**
         * <p>
         * A list of one to ten IP address ranges (<a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">CIDRs</a>) to be added to the
         * workforce allow list. By default, a workforce isn't restricted to specific IP addresses.
         * </p>
         * This is a convenience method that creates an instance of the {@link SourceIpConfig.Builder} avoiding the need
         * to create one manually via {@link SourceIpConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SourceIpConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #sourceIpConfig(SourceIpConfig)}.
         * 
         * @param sourceIpConfig
         *        a consumer that will call methods on {@link SourceIpConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceIpConfig(SourceIpConfig)
         */
        default Builder sourceIpConfig(Consumer<SourceIpConfig.Builder> sourceIpConfig) {
            return sourceIpConfig(SourceIpConfig.builder().applyMutation(sourceIpConfig).build());
        }

        /**
         * <p>
         * The subdomain for your OIDC Identity Provider.
         * </p>
         * 
         * @param subDomain
         *        The subdomain for your OIDC Identity Provider.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subDomain(String subDomain);

        /**
         * <p>
         * The configuration of an Amazon Cognito workforce. A single Cognito workforce is created using and corresponds
         * to a single <a
         * href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html"> Amazon
         * Cognito user pool</a>.
         * </p>
         * 
         * @param cognitoConfig
         *        The configuration of an Amazon Cognito workforce. A single Cognito workforce is created using and
         *        corresponds to a single <a
         *        href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html">
         *        Amazon Cognito user pool</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cognitoConfig(CognitoConfig cognitoConfig);

        /**
         * <p>
         * The configuration of an Amazon Cognito workforce. A single Cognito workforce is created using and corresponds
         * to a single <a
         * href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html"> Amazon
         * Cognito user pool</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link CognitoConfig.Builder} avoiding the need
         * to create one manually via {@link CognitoConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CognitoConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #cognitoConfig(CognitoConfig)}.
         * 
         * @param cognitoConfig
         *        a consumer that will call methods on {@link CognitoConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cognitoConfig(CognitoConfig)
         */
        default Builder cognitoConfig(Consumer<CognitoConfig.Builder> cognitoConfig) {
            return cognitoConfig(CognitoConfig.builder().applyMutation(cognitoConfig).build());
        }

        /**
         * <p>
         * The configuration of an OIDC Identity Provider (IdP) private workforce.
         * </p>
         * 
         * @param oidcConfig
         *        The configuration of an OIDC Identity Provider (IdP) private workforce.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder oidcConfig(OidcConfigForResponse oidcConfig);

        /**
         * <p>
         * The configuration of an OIDC Identity Provider (IdP) private workforce.
         * </p>
         * This is a convenience method that creates an instance of the {@link OidcConfigForResponse.Builder} avoiding
         * the need to create one manually via {@link OidcConfigForResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OidcConfigForResponse.Builder#build()} is called immediately and
         * its result is passed to {@link #oidcConfig(OidcConfigForResponse)}.
         * 
         * @param oidcConfig
         *        a consumer that will call methods on {@link OidcConfigForResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #oidcConfig(OidcConfigForResponse)
         */
        default Builder oidcConfig(Consumer<OidcConfigForResponse.Builder> oidcConfig) {
            return oidcConfig(OidcConfigForResponse.builder().applyMutation(oidcConfig).build());
        }

        /**
         * <p>
         * The date that the workforce is created.
         * </p>
         * 
         * @param createDate
         *        The date that the workforce is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createDate(Instant createDate);

        /**
         * <p>
         * The configuration of a VPC workforce.
         * </p>
         * 
         * @param workforceVpcConfig
         *        The configuration of a VPC workforce.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workforceVpcConfig(WorkforceVpcConfigResponse workforceVpcConfig);

        /**
         * <p>
         * The configuration of a VPC workforce.
         * </p>
         * This is a convenience method that creates an instance of the {@link WorkforceVpcConfigResponse.Builder}
         * avoiding the need to create one manually via {@link WorkforceVpcConfigResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link WorkforceVpcConfigResponse.Builder#build()} is called immediately
         * and its result is passed to {@link #workforceVpcConfig(WorkforceVpcConfigResponse)}.
         * 
         * @param workforceVpcConfig
         *        a consumer that will call methods on {@link WorkforceVpcConfigResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #workforceVpcConfig(WorkforceVpcConfigResponse)
         */
        default Builder workforceVpcConfig(Consumer<WorkforceVpcConfigResponse.Builder> workforceVpcConfig) {
            return workforceVpcConfig(WorkforceVpcConfigResponse.builder().applyMutation(workforceVpcConfig).build());
        }

        /**
         * <p>
         * The status of your workforce.
         * </p>
         * 
         * @param status
         *        The status of your workforce.
         * @see WorkforceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkforceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of your workforce.
         * </p>
         * 
         * @param status
         *        The status of your workforce.
         * @see WorkforceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkforceStatus
         */
        Builder status(WorkforceStatus status);

        /**
         * <p>
         * The reason your workforce failed.
         * </p>
         * 
         * @param failureReason
         *        The reason your workforce failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);
    }

    static final class BuilderImpl implements Builder {
        private String workforceName;

        private String workforceArn;

        private Instant lastUpdatedDate;

        private SourceIpConfig sourceIpConfig;

        private String subDomain;

        private CognitoConfig cognitoConfig;

        private OidcConfigForResponse oidcConfig;

        private Instant createDate;

        private WorkforceVpcConfigResponse workforceVpcConfig;

        private String status;

        private String failureReason;

        private BuilderImpl() {
        }

        private BuilderImpl(Workforce model) {
            workforceName(model.workforceName);
            workforceArn(model.workforceArn);
            lastUpdatedDate(model.lastUpdatedDate);
            sourceIpConfig(model.sourceIpConfig);
            subDomain(model.subDomain);
            cognitoConfig(model.cognitoConfig);
            oidcConfig(model.oidcConfig);
            createDate(model.createDate);
            workforceVpcConfig(model.workforceVpcConfig);
            status(model.status);
            failureReason(model.failureReason);
        }

        public final String getWorkforceName() {
            return workforceName;
        }

        public final void setWorkforceName(String workforceName) {
            this.workforceName = workforceName;
        }

        @Override
        public final Builder workforceName(String workforceName) {
            this.workforceName = workforceName;
            return this;
        }

        public final String getWorkforceArn() {
            return workforceArn;
        }

        public final void setWorkforceArn(String workforceArn) {
            this.workforceArn = workforceArn;
        }

        @Override
        public final Builder workforceArn(String workforceArn) {
            this.workforceArn = workforceArn;
            return this;
        }

        public final Instant getLastUpdatedDate() {
            return lastUpdatedDate;
        }

        public final void setLastUpdatedDate(Instant lastUpdatedDate) {
            this.lastUpdatedDate = lastUpdatedDate;
        }

        @Override
        public final Builder lastUpdatedDate(Instant lastUpdatedDate) {
            this.lastUpdatedDate = lastUpdatedDate;
            return this;
        }

        public final SourceIpConfig.Builder getSourceIpConfig() {
            return sourceIpConfig != null ? sourceIpConfig.toBuilder() : null;
        }

        public final void setSourceIpConfig(SourceIpConfig.BuilderImpl sourceIpConfig) {
            this.sourceIpConfig = sourceIpConfig != null ? sourceIpConfig.build() : null;
        }

        @Override
        public final Builder sourceIpConfig(SourceIpConfig sourceIpConfig) {
            this.sourceIpConfig = sourceIpConfig;
            return this;
        }

        public final String getSubDomain() {
            return subDomain;
        }

        public final void setSubDomain(String subDomain) {
            this.subDomain = subDomain;
        }

        @Override
        public final Builder subDomain(String subDomain) {
            this.subDomain = subDomain;
            return this;
        }

        public final CognitoConfig.Builder getCognitoConfig() {
            return cognitoConfig != null ? cognitoConfig.toBuilder() : null;
        }

        public final void setCognitoConfig(CognitoConfig.BuilderImpl cognitoConfig) {
            this.cognitoConfig = cognitoConfig != null ? cognitoConfig.build() : null;
        }

        @Override
        public final Builder cognitoConfig(CognitoConfig cognitoConfig) {
            this.cognitoConfig = cognitoConfig;
            return this;
        }

        public final OidcConfigForResponse.Builder getOidcConfig() {
            return oidcConfig != null ? oidcConfig.toBuilder() : null;
        }

        public final void setOidcConfig(OidcConfigForResponse.BuilderImpl oidcConfig) {
            this.oidcConfig = oidcConfig != null ? oidcConfig.build() : null;
        }

        @Override
        public final Builder oidcConfig(OidcConfigForResponse oidcConfig) {
            this.oidcConfig = oidcConfig;
            return this;
        }

        public final Instant getCreateDate() {
            return createDate;
        }

        public final void setCreateDate(Instant createDate) {
            this.createDate = createDate;
        }

        @Override
        public final Builder createDate(Instant createDate) {
            this.createDate = createDate;
            return this;
        }

        public final WorkforceVpcConfigResponse.Builder getWorkforceVpcConfig() {
            return workforceVpcConfig != null ? workforceVpcConfig.toBuilder() : null;
        }

        public final void setWorkforceVpcConfig(WorkforceVpcConfigResponse.BuilderImpl workforceVpcConfig) {
            this.workforceVpcConfig = workforceVpcConfig != null ? workforceVpcConfig.build() : null;
        }

        @Override
        public final Builder workforceVpcConfig(WorkforceVpcConfigResponse workforceVpcConfig) {
            this.workforceVpcConfig = workforceVpcConfig;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(WorkforceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        @Override
        public Workforce build() {
            return new Workforce(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
