/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies which training algorithm to use for training jobs that a hyperparameter tuning job launches and the metrics
 * to monitor.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HyperParameterAlgorithmSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<HyperParameterAlgorithmSpecification.Builder, HyperParameterAlgorithmSpecification> {
    private static final SdkField<String> TRAINING_IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TrainingImage").getter(getter(HyperParameterAlgorithmSpecification::trainingImage))
            .setter(setter(Builder::trainingImage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingImage").build()).build();

    private static final SdkField<String> TRAINING_INPUT_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TrainingInputMode").getter(getter(HyperParameterAlgorithmSpecification::trainingInputModeAsString))
            .setter(setter(Builder::trainingInputMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingInputMode").build()).build();

    private static final SdkField<String> ALGORITHM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlgorithmName").getter(getter(HyperParameterAlgorithmSpecification::algorithmName))
            .setter(setter(Builder::algorithmName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlgorithmName").build()).build();

    private static final SdkField<List<MetricDefinition>> METRIC_DEFINITIONS_FIELD = SdkField
            .<List<MetricDefinition>> builder(MarshallingType.LIST)
            .memberName("MetricDefinitions")
            .getter(getter(HyperParameterAlgorithmSpecification::metricDefinitions))
            .setter(setter(Builder::metricDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricDefinitions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRAINING_IMAGE_FIELD,
            TRAINING_INPUT_MODE_FIELD, ALGORITHM_NAME_FIELD, METRIC_DEFINITIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String trainingImage;

    private final String trainingInputMode;

    private final String algorithmName;

    private final List<MetricDefinition> metricDefinitions;

    private HyperParameterAlgorithmSpecification(BuilderImpl builder) {
        this.trainingImage = builder.trainingImage;
        this.trainingInputMode = builder.trainingInputMode;
        this.algorithmName = builder.algorithmName;
        this.metricDefinitions = builder.metricDefinitions;
    }

    /**
     * <p>
     * The registry path of the Docker image that contains the training algorithm. For information about Docker registry
     * paths for built-in algorithms, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-algo-docker-registry-paths.html">Algorithms
     * Provided by Amazon SageMaker: Common Parameters</a>. SageMaker supports both
     * <code>registry/repository[:tag]</code> and <code>registry/repository[@digest]</code> image path formats. For more
     * information, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own
     * Algorithms with Amazon SageMaker</a>.
     * </p>
     * 
     * @return The registry path of the Docker image that contains the training algorithm. For information about Docker
     *         registry paths for built-in algorithms, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-algo-docker-registry-paths.html"
     *         >Algorithms Provided by Amazon SageMaker: Common Parameters</a>. SageMaker supports both
     *         <code>registry/repository[:tag]</code> and <code>registry/repository[@digest]</code> image path formats.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own Algorithms
     *         with Amazon SageMaker</a>.
     */
    public final String trainingImage() {
        return trainingImage;
    }

    /**
     * Returns the value of the TrainingInputMode property for this object.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trainingInputMode}
     * will return {@link TrainingInputMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #trainingInputModeAsString}.
     * </p>
     * 
     * @return The value of the TrainingInputMode property for this object.
     * @see TrainingInputMode
     */
    public final TrainingInputMode trainingInputMode() {
        return TrainingInputMode.fromValue(trainingInputMode);
    }

    /**
     * Returns the value of the TrainingInputMode property for this object.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trainingInputMode}
     * will return {@link TrainingInputMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #trainingInputModeAsString}.
     * </p>
     * 
     * @return The value of the TrainingInputMode property for this object.
     * @see TrainingInputMode
     */
    public final String trainingInputModeAsString() {
        return trainingInputMode;
    }

    /**
     * <p>
     * The name of the resource algorithm to use for the hyperparameter tuning job. If you specify a value for this
     * parameter, do not specify a value for <code>TrainingImage</code>.
     * </p>
     * 
     * @return The name of the resource algorithm to use for the hyperparameter tuning job. If you specify a value for
     *         this parameter, do not specify a value for <code>TrainingImage</code>.
     */
    public final String algorithmName() {
        return algorithmName;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricDefinitions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMetricDefinitions() {
        return metricDefinitions != null && !(metricDefinitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_MetricDefinition.html">MetricDefinition</a>
     * objects that specify the metrics that the algorithm emits.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricDefinitions} method.
     * </p>
     * 
     * @return An array of <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_MetricDefinition.html">
     *         MetricDefinition</a> objects that specify the metrics that the algorithm emits.
     */
    public final List<MetricDefinition> metricDefinitions() {
        return metricDefinitions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(trainingImage());
        hashCode = 31 * hashCode + Objects.hashCode(trainingInputModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(algorithmName());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricDefinitions() ? metricDefinitions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HyperParameterAlgorithmSpecification)) {
            return false;
        }
        HyperParameterAlgorithmSpecification other = (HyperParameterAlgorithmSpecification) obj;
        return Objects.equals(trainingImage(), other.trainingImage())
                && Objects.equals(trainingInputModeAsString(), other.trainingInputModeAsString())
                && Objects.equals(algorithmName(), other.algorithmName())
                && hasMetricDefinitions() == other.hasMetricDefinitions()
                && Objects.equals(metricDefinitions(), other.metricDefinitions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HyperParameterAlgorithmSpecification").add("TrainingImage", trainingImage())
                .add("TrainingInputMode", trainingInputModeAsString()).add("AlgorithmName", algorithmName())
                .add("MetricDefinitions", hasMetricDefinitions() ? metricDefinitions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TrainingImage":
            return Optional.ofNullable(clazz.cast(trainingImage()));
        case "TrainingInputMode":
            return Optional.ofNullable(clazz.cast(trainingInputModeAsString()));
        case "AlgorithmName":
            return Optional.ofNullable(clazz.cast(algorithmName()));
        case "MetricDefinitions":
            return Optional.ofNullable(clazz.cast(metricDefinitions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("TrainingImage", TRAINING_IMAGE_FIELD);
        map.put("TrainingInputMode", TRAINING_INPUT_MODE_FIELD);
        map.put("AlgorithmName", ALGORITHM_NAME_FIELD);
        map.put("MetricDefinitions", METRIC_DEFINITIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<HyperParameterAlgorithmSpecification, T> g) {
        return obj -> g.apply((HyperParameterAlgorithmSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HyperParameterAlgorithmSpecification> {
        /**
         * <p>
         * The registry path of the Docker image that contains the training algorithm. For information about Docker
         * registry paths for built-in algorithms, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-algo-docker-registry-paths.html">Algorithms
         * Provided by Amazon SageMaker: Common Parameters</a>. SageMaker supports both
         * <code>registry/repository[:tag]</code> and <code>registry/repository[@digest]</code> image path formats. For
         * more information, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using
         * Your Own Algorithms with Amazon SageMaker</a>.
         * </p>
         * 
         * @param trainingImage
         *        The registry path of the Docker image that contains the training algorithm. For information about
         *        Docker registry paths for built-in algorithms, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-algo-docker-registry-paths.html"
         *        >Algorithms Provided by Amazon SageMaker: Common Parameters</a>. SageMaker supports both
         *        <code>registry/repository[:tag]</code> and <code>registry/repository[@digest]</code> image path
         *        formats. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own Algorithms
         *        with Amazon SageMaker</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingImage(String trainingImage);

        /**
         * Sets the value of the TrainingInputMode property for this object.
         *
         * @param trainingInputMode
         *        The new value for the TrainingInputMode property for this object.
         * @see TrainingInputMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInputMode
         */
        Builder trainingInputMode(String trainingInputMode);

        /**
         * Sets the value of the TrainingInputMode property for this object.
         *
         * @param trainingInputMode
         *        The new value for the TrainingInputMode property for this object.
         * @see TrainingInputMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInputMode
         */
        Builder trainingInputMode(TrainingInputMode trainingInputMode);

        /**
         * <p>
         * The name of the resource algorithm to use for the hyperparameter tuning job. If you specify a value for this
         * parameter, do not specify a value for <code>TrainingImage</code>.
         * </p>
         * 
         * @param algorithmName
         *        The name of the resource algorithm to use for the hyperparameter tuning job. If you specify a value
         *        for this parameter, do not specify a value for <code>TrainingImage</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder algorithmName(String algorithmName);

        /**
         * <p>
         * An array of <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_MetricDefinition.html">MetricDefinition
         * </a> objects that specify the metrics that the algorithm emits.
         * </p>
         * 
         * @param metricDefinitions
         *        An array of <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_MetricDefinition.html"
         *        >MetricDefinition</a> objects that specify the metrics that the algorithm emits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricDefinitions(Collection<MetricDefinition> metricDefinitions);

        /**
         * <p>
         * An array of <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_MetricDefinition.html">MetricDefinition
         * </a> objects that specify the metrics that the algorithm emits.
         * </p>
         * 
         * @param metricDefinitions
         *        An array of <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_MetricDefinition.html"
         *        >MetricDefinition</a> objects that specify the metrics that the algorithm emits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricDefinitions(MetricDefinition... metricDefinitions);

        /**
         * <p>
         * An array of <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_MetricDefinition.html">MetricDefinition
         * </a> objects that specify the metrics that the algorithm emits.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.MetricDefinition.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.sagemaker.model.MetricDefinition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.MetricDefinition.Builder#build()} is called
         * immediately and its result is passed to {@link #metricDefinitions(List<MetricDefinition>)}.
         * 
         * @param metricDefinitions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.MetricDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricDefinitions(java.util.Collection<MetricDefinition>)
         */
        Builder metricDefinitions(Consumer<MetricDefinition.Builder>... metricDefinitions);
    }

    static final class BuilderImpl implements Builder {
        private String trainingImage;

        private String trainingInputMode;

        private String algorithmName;

        private List<MetricDefinition> metricDefinitions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(HyperParameterAlgorithmSpecification model) {
            trainingImage(model.trainingImage);
            trainingInputMode(model.trainingInputMode);
            algorithmName(model.algorithmName);
            metricDefinitions(model.metricDefinitions);
        }

        public final String getTrainingImage() {
            return trainingImage;
        }

        public final void setTrainingImage(String trainingImage) {
            this.trainingImage = trainingImage;
        }

        @Override
        public final Builder trainingImage(String trainingImage) {
            this.trainingImage = trainingImage;
            return this;
        }

        public final String getTrainingInputMode() {
            return trainingInputMode;
        }

        public final void setTrainingInputMode(String trainingInputMode) {
            this.trainingInputMode = trainingInputMode;
        }

        @Override
        public final Builder trainingInputMode(String trainingInputMode) {
            this.trainingInputMode = trainingInputMode;
            return this;
        }

        @Override
        public final Builder trainingInputMode(TrainingInputMode trainingInputMode) {
            this.trainingInputMode(trainingInputMode == null ? null : trainingInputMode.toString());
            return this;
        }

        public final String getAlgorithmName() {
            return algorithmName;
        }

        public final void setAlgorithmName(String algorithmName) {
            this.algorithmName = algorithmName;
        }

        @Override
        public final Builder algorithmName(String algorithmName) {
            this.algorithmName = algorithmName;
            return this;
        }

        public final List<MetricDefinition.Builder> getMetricDefinitions() {
            List<MetricDefinition.Builder> result = MetricDefinitionListCopier.copyToBuilder(this.metricDefinitions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMetricDefinitions(Collection<MetricDefinition.BuilderImpl> metricDefinitions) {
            this.metricDefinitions = MetricDefinitionListCopier.copyFromBuilder(metricDefinitions);
        }

        @Override
        public final Builder metricDefinitions(Collection<MetricDefinition> metricDefinitions) {
            this.metricDefinitions = MetricDefinitionListCopier.copy(metricDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricDefinitions(MetricDefinition... metricDefinitions) {
            metricDefinitions(Arrays.asList(metricDefinitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricDefinitions(Consumer<MetricDefinition.Builder>... metricDefinitions) {
            metricDefinitions(Stream.of(metricDefinitions).map(c -> MetricDefinition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public HyperParameterAlgorithmSpecification build() {
            return new HyperParameterAlgorithmSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
