/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration of resources, including compute instances and storage volumes for use in training jobs launched by
 * hyperparameter tuning jobs. <code>HyperParameterTuningResourceConfig</code> is similar to <code>ResourceConfig</code>
 * , but has the additional <code>InstanceConfigs</code> and <code>AllocationStrategy</code> fields to allow for
 * flexible instance management. Specify one or more instance types, count, and the allocation strategy for instance
 * selection.
 * </p>
 * <note>
 * <p>
 * <code>HyperParameterTuningResourceConfig</code> supports the capabilities of <code>ResourceConfig</code> with the
 * exception of <code>KeepAlivePeriodInSeconds</code>. Hyperparameter tuning jobs use warm pools by default, which reuse
 * clusters between training jobs.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HyperParameterTuningResourceConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<HyperParameterTuningResourceConfig.Builder, HyperParameterTuningResourceConfig> {
    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(HyperParameterTuningResourceConfig::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Integer> INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstanceCount").getter(getter(HyperParameterTuningResourceConfig::instanceCount))
            .setter(setter(Builder::instanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceCount").build()).build();

    private static final SdkField<Integer> VOLUME_SIZE_IN_GB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("VolumeSizeInGB").getter(getter(HyperParameterTuningResourceConfig::volumeSizeInGB))
            .setter(setter(Builder::volumeSizeInGB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeSizeInGB").build()).build();

    private static final SdkField<String> VOLUME_KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VolumeKmsKeyId").getter(getter(HyperParameterTuningResourceConfig::volumeKmsKeyId))
            .setter(setter(Builder::volumeKmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeKmsKeyId").build()).build();

    private static final SdkField<String> ALLOCATION_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AllocationStrategy").getter(getter(HyperParameterTuningResourceConfig::allocationStrategyAsString))
            .setter(setter(Builder::allocationStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationStrategy").build())
            .build();

    private static final SdkField<List<HyperParameterTuningInstanceConfig>> INSTANCE_CONFIGS_FIELD = SdkField
            .<List<HyperParameterTuningInstanceConfig>> builder(MarshallingType.LIST)
            .memberName("InstanceConfigs")
            .getter(getter(HyperParameterTuningResourceConfig::instanceConfigs))
            .setter(setter(Builder::instanceConfigs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceConfigs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HyperParameterTuningInstanceConfig> builder(MarshallingType.SDK_POJO)
                                            .constructor(HyperParameterTuningInstanceConfig::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_TYPE_FIELD,
            INSTANCE_COUNT_FIELD, VOLUME_SIZE_IN_GB_FIELD, VOLUME_KMS_KEY_ID_FIELD, ALLOCATION_STRATEGY_FIELD,
            INSTANCE_CONFIGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String instanceType;

    private final Integer instanceCount;

    private final Integer volumeSizeInGB;

    private final String volumeKmsKeyId;

    private final String allocationStrategy;

    private final List<HyperParameterTuningInstanceConfig> instanceConfigs;

    private HyperParameterTuningResourceConfig(BuilderImpl builder) {
        this.instanceType = builder.instanceType;
        this.instanceCount = builder.instanceCount;
        this.volumeSizeInGB = builder.volumeSizeInGB;
        this.volumeKmsKeyId = builder.volumeKmsKeyId;
        this.allocationStrategy = builder.allocationStrategy;
        this.instanceConfigs = builder.instanceConfigs;
    }

    /**
     * <p>
     * The instance type used to run hyperparameter optimization tuning jobs. See <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html"> descriptions of
     * instance types</a> for more information.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link TrainingInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type used to run hyperparameter optimization tuning jobs. See <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html">
     *         descriptions of instance types</a> for more information.
     * @see TrainingInstanceType
     */
    public final TrainingInstanceType instanceType() {
        return TrainingInstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The instance type used to run hyperparameter optimization tuning jobs. See <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html"> descriptions of
     * instance types</a> for more information.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link TrainingInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type used to run hyperparameter optimization tuning jobs. See <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html">
     *         descriptions of instance types</a> for more information.
     * @see TrainingInstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * The number of compute instances of type <code>InstanceType</code> to use. For <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/data-parallel-use-api.html">distributed training</a>,
     * select a value greater than 1.
     * </p>
     * 
     * @return The number of compute instances of type <code>InstanceType</code> to use. For <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/data-parallel-use-api.html">distributed
     *         training</a>, select a value greater than 1.
     */
    public final Integer instanceCount() {
        return instanceCount;
    }

    /**
     * <p>
     * The volume size in GB for the storage volume to be used in processing hyperparameter optimization jobs
     * (optional). These volumes store model artifacts, incremental states and optionally, scratch space for training
     * algorithms. Do not provide a value for this parameter if a value for <code>InstanceConfigs</code> is also
     * specified.
     * </p>
     * <p>
     * Some instance types have a fixed total local storage size. If you select one of these instances for training,
     * <code>VolumeSizeInGB</code> cannot be greater than this total size. For a list of instance types with local
     * instance storage and their sizes, see <a
     * href="http://aws.amazon.com/releasenotes/host-instance-storage-volumes-table/">instance store volumes</a>.
     * </p>
     * <note>
     * <p>
     * SageMaker supports only the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html">General Purpose SSD (gp2)</a>
     * storage volume type.
     * </p>
     * </note>
     * 
     * @return The volume size in GB for the storage volume to be used in processing hyperparameter optimization jobs
     *         (optional). These volumes store model artifacts, incremental states and optionally, scratch space for
     *         training algorithms. Do not provide a value for this parameter if a value for
     *         <code>InstanceConfigs</code> is also specified.</p>
     *         <p>
     *         Some instance types have a fixed total local storage size. If you select one of these instances for
     *         training, <code>VolumeSizeInGB</code> cannot be greater than this total size. For a list of instance
     *         types with local instance storage and their sizes, see <a
     *         href="http://aws.amazon.com/releasenotes/host-instance-storage-volumes-table/">instance store
     *         volumes</a>.
     *         </p>
     *         <note>
     *         <p>
     *         SageMaker supports only the <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html">General Purpose SSD
     *         (gp2)</a> storage volume type.
     *         </p>
     */
    public final Integer volumeSizeInGB() {
        return volumeSizeInGB;
    }

    /**
     * <p>
     * A key used by Amazon Web Services Key Management Service to encrypt data on the storage volume attached to the
     * compute instances used to run the training job. You can use either of the following formats to specify a key.
     * </p>
     * <p>
     * KMS Key ID:
     * </p>
     * <p>
     * <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     * </p>
     * <p>
     * Amazon Resource Name (ARN) of a KMS key:
     * </p>
     * <p>
     * <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     * </p>
     * <p>
     * Some instances use local storage, which use a <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ssd-instance-store.html">hardware module to encrypt</a>
     * storage volumes. If you choose one of these instance types, you cannot request a <code>VolumeKmsKeyId</code>. For
     * a list of instance types that use local storage, see <a
     * href="http://aws.amazon.com/releasenotes/host-instance-storage-volumes-table/">instance store volumes</a>. For
     * more information about Amazon Web Services Key Management Service, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/sms-security-kms-permissions.html">KMS encryption</a> for
     * more information.
     * </p>
     * 
     * @return A key used by Amazon Web Services Key Management Service to encrypt data on the storage volume attached
     *         to the compute instances used to run the training job. You can use either of the following formats to
     *         specify a key.</p>
     *         <p>
     *         KMS Key ID:
     *         </p>
     *         <p>
     *         <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     *         </p>
     *         <p>
     *         Amazon Resource Name (ARN) of a KMS key:
     *         </p>
     *         <p>
     *         <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     *         </p>
     *         <p>
     *         Some instances use local storage, which use a <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ssd-instance-store.html">hardware module to
     *         encrypt</a> storage volumes. If you choose one of these instance types, you cannot request a
     *         <code>VolumeKmsKeyId</code>. For a list of instance types that use local storage, see <a
     *         href="http://aws.amazon.com/releasenotes/host-instance-storage-volumes-table/">instance store
     *         volumes</a>. For more information about Amazon Web Services Key Management Service, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/sms-security-kms-permissions.html">KMS
     *         encryption</a> for more information.
     */
    public final String volumeKmsKeyId() {
        return volumeKmsKeyId;
    }

    /**
     * <p>
     * The strategy that determines the order of preference for resources specified in <code>InstanceConfigs</code> used
     * in hyperparameter optimization.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allocationStrategy} will return {@link HyperParameterTuningAllocationStrategy#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #allocationStrategyAsString}.
     * </p>
     * 
     * @return The strategy that determines the order of preference for resources specified in
     *         <code>InstanceConfigs</code> used in hyperparameter optimization.
     * @see HyperParameterTuningAllocationStrategy
     */
    public final HyperParameterTuningAllocationStrategy allocationStrategy() {
        return HyperParameterTuningAllocationStrategy.fromValue(allocationStrategy);
    }

    /**
     * <p>
     * The strategy that determines the order of preference for resources specified in <code>InstanceConfigs</code> used
     * in hyperparameter optimization.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allocationStrategy} will return {@link HyperParameterTuningAllocationStrategy#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #allocationStrategyAsString}.
     * </p>
     * 
     * @return The strategy that determines the order of preference for resources specified in
     *         <code>InstanceConfigs</code> used in hyperparameter optimization.
     * @see HyperParameterTuningAllocationStrategy
     */
    public final String allocationStrategyAsString() {
        return allocationStrategy;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceConfigs property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInstanceConfigs() {
        return instanceConfigs != null && !(instanceConfigs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list containing the configuration(s) for one or more resources for processing hyperparameter jobs. These
     * resources include compute instances and storage volumes to use in model training jobs launched by hyperparameter
     * tuning jobs. The <code>AllocationStrategy</code> controls the order in which multiple configurations provided in
     * <code>InstanceConfigs</code> are used.
     * </p>
     * <note>
     * <p>
     * If you only want to use a single instance configuration inside the
     * <code>HyperParameterTuningResourceConfig</code> API, do not provide a value for <code>InstanceConfigs</code>.
     * Instead, use <code>InstanceType</code>, <code>VolumeSizeInGB</code> and <code>InstanceCount</code>. If you use
     * <code>InstanceConfigs</code>, do not provide values for <code>InstanceType</code>, <code>VolumeSizeInGB</code> or
     * <code>InstanceCount</code>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceConfigs} method.
     * </p>
     * 
     * @return A list containing the configuration(s) for one or more resources for processing hyperparameter jobs.
     *         These resources include compute instances and storage volumes to use in model training jobs launched by
     *         hyperparameter tuning jobs. The <code>AllocationStrategy</code> controls the order in which multiple
     *         configurations provided in <code>InstanceConfigs</code> are used.</p> <note>
     *         <p>
     *         If you only want to use a single instance configuration inside the
     *         <code>HyperParameterTuningResourceConfig</code> API, do not provide a value for
     *         <code>InstanceConfigs</code>. Instead, use <code>InstanceType</code>, <code>VolumeSizeInGB</code> and
     *         <code>InstanceCount</code>. If you use <code>InstanceConfigs</code>, do not provide values for
     *         <code>InstanceType</code>, <code>VolumeSizeInGB</code> or <code>InstanceCount</code>.
     *         </p>
     */
    public final List<HyperParameterTuningInstanceConfig> instanceConfigs() {
        return instanceConfigs;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(volumeSizeInGB());
        hashCode = 31 * hashCode + Objects.hashCode(volumeKmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(allocationStrategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceConfigs() ? instanceConfigs() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HyperParameterTuningResourceConfig)) {
            return false;
        }
        HyperParameterTuningResourceConfig other = (HyperParameterTuningResourceConfig) obj;
        return Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(instanceCount(), other.instanceCount())
                && Objects.equals(volumeSizeInGB(), other.volumeSizeInGB())
                && Objects.equals(volumeKmsKeyId(), other.volumeKmsKeyId())
                && Objects.equals(allocationStrategyAsString(), other.allocationStrategyAsString())
                && hasInstanceConfigs() == other.hasInstanceConfigs()
                && Objects.equals(instanceConfigs(), other.instanceConfigs());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HyperParameterTuningResourceConfig").add("InstanceType", instanceTypeAsString())
                .add("InstanceCount", instanceCount()).add("VolumeSizeInGB", volumeSizeInGB())
                .add("VolumeKmsKeyId", volumeKmsKeyId()).add("AllocationStrategy", allocationStrategyAsString())
                .add("InstanceConfigs", hasInstanceConfigs() ? instanceConfigs() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "InstanceCount":
            return Optional.ofNullable(clazz.cast(instanceCount()));
        case "VolumeSizeInGB":
            return Optional.ofNullable(clazz.cast(volumeSizeInGB()));
        case "VolumeKmsKeyId":
            return Optional.ofNullable(clazz.cast(volumeKmsKeyId()));
        case "AllocationStrategy":
            return Optional.ofNullable(clazz.cast(allocationStrategyAsString()));
        case "InstanceConfigs":
            return Optional.ofNullable(clazz.cast(instanceConfigs()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("InstanceType", INSTANCE_TYPE_FIELD);
        map.put("InstanceCount", INSTANCE_COUNT_FIELD);
        map.put("VolumeSizeInGB", VOLUME_SIZE_IN_GB_FIELD);
        map.put("VolumeKmsKeyId", VOLUME_KMS_KEY_ID_FIELD);
        map.put("AllocationStrategy", ALLOCATION_STRATEGY_FIELD);
        map.put("InstanceConfigs", INSTANCE_CONFIGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<HyperParameterTuningResourceConfig, T> g) {
        return obj -> g.apply((HyperParameterTuningResourceConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HyperParameterTuningResourceConfig> {
        /**
         * <p>
         * The instance type used to run hyperparameter optimization tuning jobs. See <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html"> descriptions
         * of instance types</a> for more information.
         * </p>
         * 
         * @param instanceType
         *        The instance type used to run hyperparameter optimization tuning jobs. See <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html">
         *        descriptions of instance types</a> for more information.
         * @see TrainingInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The instance type used to run hyperparameter optimization tuning jobs. See <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html"> descriptions
         * of instance types</a> for more information.
         * </p>
         * 
         * @param instanceType
         *        The instance type used to run hyperparameter optimization tuning jobs. See <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html">
         *        descriptions of instance types</a> for more information.
         * @see TrainingInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInstanceType
         */
        Builder instanceType(TrainingInstanceType instanceType);

        /**
         * <p>
         * The number of compute instances of type <code>InstanceType</code> to use. For <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/data-parallel-use-api.html">distributed training</a>,
         * select a value greater than 1.
         * </p>
         * 
         * @param instanceCount
         *        The number of compute instances of type <code>InstanceType</code> to use. For <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/data-parallel-use-api.html">distributed
         *        training</a>, select a value greater than 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceCount(Integer instanceCount);

        /**
         * <p>
         * The volume size in GB for the storage volume to be used in processing hyperparameter optimization jobs
         * (optional). These volumes store model artifacts, incremental states and optionally, scratch space for
         * training algorithms. Do not provide a value for this parameter if a value for <code>InstanceConfigs</code> is
         * also specified.
         * </p>
         * <p>
         * Some instance types have a fixed total local storage size. If you select one of these instances for training,
         * <code>VolumeSizeInGB</code> cannot be greater than this total size. For a list of instance types with local
         * instance storage and their sizes, see <a
         * href="http://aws.amazon.com/releasenotes/host-instance-storage-volumes-table/">instance store volumes</a>.
         * </p>
         * <note>
         * <p>
         * SageMaker supports only the <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html">General Purpose SSD
         * (gp2)</a> storage volume type.
         * </p>
         * </note>
         * 
         * @param volumeSizeInGB
         *        The volume size in GB for the storage volume to be used in processing hyperparameter optimization jobs
         *        (optional). These volumes store model artifacts, incremental states and optionally, scratch space for
         *        training algorithms. Do not provide a value for this parameter if a value for
         *        <code>InstanceConfigs</code> is also specified.</p>
         *        <p>
         *        Some instance types have a fixed total local storage size. If you select one of these instances for
         *        training, <code>VolumeSizeInGB</code> cannot be greater than this total size. For a list of instance
         *        types with local instance storage and their sizes, see <a
         *        href="http://aws.amazon.com/releasenotes/host-instance-storage-volumes-table/">instance store
         *        volumes</a>.
         *        </p>
         *        <note>
         *        <p>
         *        SageMaker supports only the <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html">General Purpose SSD
         *        (gp2)</a> storage volume type.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeSizeInGB(Integer volumeSizeInGB);

        /**
         * <p>
         * A key used by Amazon Web Services Key Management Service to encrypt data on the storage volume attached to
         * the compute instances used to run the training job. You can use either of the following formats to specify a
         * key.
         * </p>
         * <p>
         * KMS Key ID:
         * </p>
         * <p>
         * <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         * </p>
         * <p>
         * Amazon Resource Name (ARN) of a KMS key:
         * </p>
         * <p>
         * <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         * </p>
         * <p>
         * Some instances use local storage, which use a <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ssd-instance-store.html">hardware module to
         * encrypt</a> storage volumes. If you choose one of these instance types, you cannot request a
         * <code>VolumeKmsKeyId</code>. For a list of instance types that use local storage, see <a
         * href="http://aws.amazon.com/releasenotes/host-instance-storage-volumes-table/">instance store volumes</a>.
         * For more information about Amazon Web Services Key Management Service, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/sms-security-kms-permissions.html">KMS encryption</a>
         * for more information.
         * </p>
         * 
         * @param volumeKmsKeyId
         *        A key used by Amazon Web Services Key Management Service to encrypt data on the storage volume
         *        attached to the compute instances used to run the training job. You can use either of the following
         *        formats to specify a key.</p>
         *        <p>
         *        KMS Key ID:
         *        </p>
         *        <p>
         *        <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         *        </p>
         *        <p>
         *        Amazon Resource Name (ARN) of a KMS key:
         *        </p>
         *        <p>
         *        <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         *        </p>
         *        <p>
         *        Some instances use local storage, which use a <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ssd-instance-store.html">hardware module to
         *        encrypt</a> storage volumes. If you choose one of these instance types, you cannot request a
         *        <code>VolumeKmsKeyId</code>. For a list of instance types that use local storage, see <a
         *        href="http://aws.amazon.com/releasenotes/host-instance-storage-volumes-table/">instance store
         *        volumes</a>. For more information about Amazon Web Services Key Management Service, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/sms-security-kms-permissions.html">KMS
         *        encryption</a> for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeKmsKeyId(String volumeKmsKeyId);

        /**
         * <p>
         * The strategy that determines the order of preference for resources specified in <code>InstanceConfigs</code>
         * used in hyperparameter optimization.
         * </p>
         * 
         * @param allocationStrategy
         *        The strategy that determines the order of preference for resources specified in
         *        <code>InstanceConfigs</code> used in hyperparameter optimization.
         * @see HyperParameterTuningAllocationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HyperParameterTuningAllocationStrategy
         */
        Builder allocationStrategy(String allocationStrategy);

        /**
         * <p>
         * The strategy that determines the order of preference for resources specified in <code>InstanceConfigs</code>
         * used in hyperparameter optimization.
         * </p>
         * 
         * @param allocationStrategy
         *        The strategy that determines the order of preference for resources specified in
         *        <code>InstanceConfigs</code> used in hyperparameter optimization.
         * @see HyperParameterTuningAllocationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HyperParameterTuningAllocationStrategy
         */
        Builder allocationStrategy(HyperParameterTuningAllocationStrategy allocationStrategy);

        /**
         * <p>
         * A list containing the configuration(s) for one or more resources for processing hyperparameter jobs. These
         * resources include compute instances and storage volumes to use in model training jobs launched by
         * hyperparameter tuning jobs. The <code>AllocationStrategy</code> controls the order in which multiple
         * configurations provided in <code>InstanceConfigs</code> are used.
         * </p>
         * <note>
         * <p>
         * If you only want to use a single instance configuration inside the
         * <code>HyperParameterTuningResourceConfig</code> API, do not provide a value for <code>InstanceConfigs</code>.
         * Instead, use <code>InstanceType</code>, <code>VolumeSizeInGB</code> and <code>InstanceCount</code>. If you
         * use <code>InstanceConfigs</code>, do not provide values for <code>InstanceType</code>,
         * <code>VolumeSizeInGB</code> or <code>InstanceCount</code>.
         * </p>
         * </note>
         * 
         * @param instanceConfigs
         *        A list containing the configuration(s) for one or more resources for processing hyperparameter jobs.
         *        These resources include compute instances and storage volumes to use in model training jobs launched
         *        by hyperparameter tuning jobs. The <code>AllocationStrategy</code> controls the order in which
         *        multiple configurations provided in <code>InstanceConfigs</code> are used.</p> <note>
         *        <p>
         *        If you only want to use a single instance configuration inside the
         *        <code>HyperParameterTuningResourceConfig</code> API, do not provide a value for
         *        <code>InstanceConfigs</code>. Instead, use <code>InstanceType</code>, <code>VolumeSizeInGB</code> and
         *        <code>InstanceCount</code>. If you use <code>InstanceConfigs</code>, do not provide values for
         *        <code>InstanceType</code>, <code>VolumeSizeInGB</code> or <code>InstanceCount</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceConfigs(Collection<HyperParameterTuningInstanceConfig> instanceConfigs);

        /**
         * <p>
         * A list containing the configuration(s) for one or more resources for processing hyperparameter jobs. These
         * resources include compute instances and storage volumes to use in model training jobs launched by
         * hyperparameter tuning jobs. The <code>AllocationStrategy</code> controls the order in which multiple
         * configurations provided in <code>InstanceConfigs</code> are used.
         * </p>
         * <note>
         * <p>
         * If you only want to use a single instance configuration inside the
         * <code>HyperParameterTuningResourceConfig</code> API, do not provide a value for <code>InstanceConfigs</code>.
         * Instead, use <code>InstanceType</code>, <code>VolumeSizeInGB</code> and <code>InstanceCount</code>. If you
         * use <code>InstanceConfigs</code>, do not provide values for <code>InstanceType</code>,
         * <code>VolumeSizeInGB</code> or <code>InstanceCount</code>.
         * </p>
         * </note>
         * 
         * @param instanceConfigs
         *        A list containing the configuration(s) for one or more resources for processing hyperparameter jobs.
         *        These resources include compute instances and storage volumes to use in model training jobs launched
         *        by hyperparameter tuning jobs. The <code>AllocationStrategy</code> controls the order in which
         *        multiple configurations provided in <code>InstanceConfigs</code> are used.</p> <note>
         *        <p>
         *        If you only want to use a single instance configuration inside the
         *        <code>HyperParameterTuningResourceConfig</code> API, do not provide a value for
         *        <code>InstanceConfigs</code>. Instead, use <code>InstanceType</code>, <code>VolumeSizeInGB</code> and
         *        <code>InstanceCount</code>. If you use <code>InstanceConfigs</code>, do not provide values for
         *        <code>InstanceType</code>, <code>VolumeSizeInGB</code> or <code>InstanceCount</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceConfigs(HyperParameterTuningInstanceConfig... instanceConfigs);

        /**
         * <p>
         * A list containing the configuration(s) for one or more resources for processing hyperparameter jobs. These
         * resources include compute instances and storage volumes to use in model training jobs launched by
         * hyperparameter tuning jobs. The <code>AllocationStrategy</code> controls the order in which multiple
         * configurations provided in <code>InstanceConfigs</code> are used.
         * </p>
         * <note>
         * <p>
         * If you only want to use a single instance configuration inside the
         * <code>HyperParameterTuningResourceConfig</code> API, do not provide a value for <code>InstanceConfigs</code>.
         * Instead, use <code>InstanceType</code>, <code>VolumeSizeInGB</code> and <code>InstanceCount</code>. If you
         * use <code>InstanceConfigs</code>, do not provide values for <code>InstanceType</code>,
         * <code>VolumeSizeInGB</code> or <code>InstanceCount</code>.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.HyperParameterTuningInstanceConfig.Builder} avoiding
         * the need to create one manually via
         * {@link software.amazon.awssdk.services.sagemaker.model.HyperParameterTuningInstanceConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.HyperParameterTuningInstanceConfig.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #instanceConfigs(List<HyperParameterTuningInstanceConfig>)}.
         * 
         * @param instanceConfigs
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.HyperParameterTuningInstanceConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceConfigs(java.util.Collection<HyperParameterTuningInstanceConfig>)
         */
        Builder instanceConfigs(Consumer<HyperParameterTuningInstanceConfig.Builder>... instanceConfigs);
    }

    static final class BuilderImpl implements Builder {
        private String instanceType;

        private Integer instanceCount;

        private Integer volumeSizeInGB;

        private String volumeKmsKeyId;

        private String allocationStrategy;

        private List<HyperParameterTuningInstanceConfig> instanceConfigs = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(HyperParameterTuningResourceConfig model) {
            instanceType(model.instanceType);
            instanceCount(model.instanceCount);
            volumeSizeInGB(model.volumeSizeInGB);
            volumeKmsKeyId(model.volumeKmsKeyId);
            allocationStrategy(model.allocationStrategy);
            instanceConfigs(model.instanceConfigs);
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(TrainingInstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final Integer getInstanceCount() {
            return instanceCount;
        }

        public final void setInstanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
        }

        @Override
        public final Builder instanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
            return this;
        }

        public final Integer getVolumeSizeInGB() {
            return volumeSizeInGB;
        }

        public final void setVolumeSizeInGB(Integer volumeSizeInGB) {
            this.volumeSizeInGB = volumeSizeInGB;
        }

        @Override
        public final Builder volumeSizeInGB(Integer volumeSizeInGB) {
            this.volumeSizeInGB = volumeSizeInGB;
            return this;
        }

        public final String getVolumeKmsKeyId() {
            return volumeKmsKeyId;
        }

        public final void setVolumeKmsKeyId(String volumeKmsKeyId) {
            this.volumeKmsKeyId = volumeKmsKeyId;
        }

        @Override
        public final Builder volumeKmsKeyId(String volumeKmsKeyId) {
            this.volumeKmsKeyId = volumeKmsKeyId;
            return this;
        }

        public final String getAllocationStrategy() {
            return allocationStrategy;
        }

        public final void setAllocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
        }

        @Override
        public final Builder allocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
            return this;
        }

        @Override
        public final Builder allocationStrategy(HyperParameterTuningAllocationStrategy allocationStrategy) {
            this.allocationStrategy(allocationStrategy == null ? null : allocationStrategy.toString());
            return this;
        }

        public final List<HyperParameterTuningInstanceConfig.Builder> getInstanceConfigs() {
            List<HyperParameterTuningInstanceConfig.Builder> result = HyperParameterTuningInstanceConfigsCopier
                    .copyToBuilder(this.instanceConfigs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInstanceConfigs(Collection<HyperParameterTuningInstanceConfig.BuilderImpl> instanceConfigs) {
            this.instanceConfigs = HyperParameterTuningInstanceConfigsCopier.copyFromBuilder(instanceConfigs);
        }

        @Override
        public final Builder instanceConfigs(Collection<HyperParameterTuningInstanceConfig> instanceConfigs) {
            this.instanceConfigs = HyperParameterTuningInstanceConfigsCopier.copy(instanceConfigs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceConfigs(HyperParameterTuningInstanceConfig... instanceConfigs) {
            instanceConfigs(Arrays.asList(instanceConfigs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceConfigs(Consumer<HyperParameterTuningInstanceConfig.Builder>... instanceConfigs) {
            instanceConfigs(Stream.of(instanceConfigs)
                    .map(c -> HyperParameterTuningInstanceConfig.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public HyperParameterTuningResourceConfig build() {
            return new HyperParameterTuningResourceConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
