/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines a container that provides the runtime environment for a model that you deploy with an inference component.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InferenceComponentContainerSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<InferenceComponentContainerSpecification.Builder, InferenceComponentContainerSpecification> {
    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Image")
            .getter(getter(InferenceComponentContainerSpecification::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Image").build()).build();

    private static final SdkField<String> ARTIFACT_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ArtifactUrl").getter(getter(InferenceComponentContainerSpecification::artifactUrl))
            .setter(setter(Builder::artifactUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ArtifactUrl").build()).build();

    private static final SdkField<Map<String, String>> ENVIRONMENT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Environment")
            .getter(getter(InferenceComponentContainerSpecification::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Environment").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_FIELD,
            ARTIFACT_URL_FIELD, ENVIRONMENT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String image;

    private final String artifactUrl;

    private final Map<String, String> environment;

    private InferenceComponentContainerSpecification(BuilderImpl builder) {
        this.image = builder.image;
        this.artifactUrl = builder.artifactUrl;
        this.environment = builder.environment;
    }

    /**
     * <p>
     * The Amazon Elastic Container Registry (Amazon ECR) path where the Docker image for the model is stored.
     * </p>
     * 
     * @return The Amazon Elastic Container Registry (Amazon ECR) path where the Docker image for the model is stored.
     */
    public final String image() {
        return image;
    }

    /**
     * <p>
     * The Amazon S3 path where the model artifacts, which result from model training, are stored. This path must point
     * to a single gzip compressed tar archive (.tar.gz suffix).
     * </p>
     * 
     * @return The Amazon S3 path where the model artifacts, which result from model training, are stored. This path
     *         must point to a single gzip compressed tar archive (.tar.gz suffix).
     */
    public final String artifactUrl() {
        return artifactUrl;
    }

    /**
     * For responses, this returns true if the service returned a value for the Environment property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The environment variables to set in the Docker container. Each key and value in the Environment string-to-string
     * map can have length of up to 1024. We support up to 16 entries in the map.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironment} method.
     * </p>
     * 
     * @return The environment variables to set in the Docker container. Each key and value in the Environment
     *         string-to-string map can have length of up to 1024. We support up to 16 entries in the map.
     */
    public final Map<String, String> environment() {
        return environment;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(artifactUrl());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InferenceComponentContainerSpecification)) {
            return false;
        }
        InferenceComponentContainerSpecification other = (InferenceComponentContainerSpecification) obj;
        return Objects.equals(image(), other.image()) && Objects.equals(artifactUrl(), other.artifactUrl())
                && hasEnvironment() == other.hasEnvironment() && Objects.equals(environment(), other.environment());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InferenceComponentContainerSpecification").add("Image", image())
                .add("ArtifactUrl", artifactUrl()).add("Environment", hasEnvironment() ? environment() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Image":
            return Optional.ofNullable(clazz.cast(image()));
        case "ArtifactUrl":
            return Optional.ofNullable(clazz.cast(artifactUrl()));
        case "Environment":
            return Optional.ofNullable(clazz.cast(environment()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Image", IMAGE_FIELD);
        map.put("ArtifactUrl", ARTIFACT_URL_FIELD);
        map.put("Environment", ENVIRONMENT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InferenceComponentContainerSpecification, T> g) {
        return obj -> g.apply((InferenceComponentContainerSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InferenceComponentContainerSpecification> {
        /**
         * <p>
         * The Amazon Elastic Container Registry (Amazon ECR) path where the Docker image for the model is stored.
         * </p>
         * 
         * @param image
         *        The Amazon Elastic Container Registry (Amazon ECR) path where the Docker image for the model is
         *        stored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * The Amazon S3 path where the model artifacts, which result from model training, are stored. This path must
         * point to a single gzip compressed tar archive (.tar.gz suffix).
         * </p>
         * 
         * @param artifactUrl
         *        The Amazon S3 path where the model artifacts, which result from model training, are stored. This path
         *        must point to a single gzip compressed tar archive (.tar.gz suffix).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder artifactUrl(String artifactUrl);

        /**
         * <p>
         * The environment variables to set in the Docker container. Each key and value in the Environment
         * string-to-string map can have length of up to 1024. We support up to 16 entries in the map.
         * </p>
         * 
         * @param environment
         *        The environment variables to set in the Docker container. Each key and value in the Environment
         *        string-to-string map can have length of up to 1024. We support up to 16 entries in the map.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Map<String, String> environment);
    }

    static final class BuilderImpl implements Builder {
        private String image;

        private String artifactUrl;

        private Map<String, String> environment = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InferenceComponentContainerSpecification model) {
            image(model.image);
            artifactUrl(model.artifactUrl);
            environment(model.environment);
        }

        public final String getImage() {
            return image;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final String getArtifactUrl() {
            return artifactUrl;
        }

        public final void setArtifactUrl(String artifactUrl) {
            this.artifactUrl = artifactUrl;
        }

        @Override
        public final Builder artifactUrl(String artifactUrl) {
            this.artifactUrl = artifactUrl;
            return this;
        }

        public final Map<String, String> getEnvironment() {
            if (environment instanceof SdkAutoConstructMap) {
                return null;
            }
            return environment;
        }

        public final void setEnvironment(Map<String, String> environment) {
            this.environment = EnvironmentMapCopier.copy(environment);
        }

        @Override
        public final Builder environment(Map<String, String> environment) {
            this.environment = EnvironmentMapCopier.copy(environment);
            return this;
        }

        @Override
        public InferenceComponentContainerSpecification build() {
            return new InferenceComponentContainerSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
