/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the location of input model artifacts, the name and shape of the expected data inputs, and
 * the framework in which the model was trained.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InputConfig implements SdkPojo, Serializable, ToCopyableBuilder<InputConfig.Builder, InputConfig> {
    private static final SdkField<String> S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("S3Uri")
            .getter(getter(InputConfig::s3Uri)).setter(setter(Builder::s3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Uri").build()).build();

    private static final SdkField<String> DATA_INPUT_CONFIG_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataInputConfig").getter(getter(InputConfig::dataInputConfig)).setter(setter(Builder::dataInputConfig))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataInputConfig").build()).build();

    private static final SdkField<String> FRAMEWORK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Framework").getter(getter(InputConfig::frameworkAsString)).setter(setter(Builder::framework))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Framework").build()).build();

    private static final SdkField<String> FRAMEWORK_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FrameworkVersion").getter(getter(InputConfig::frameworkVersion))
            .setter(setter(Builder::frameworkVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FrameworkVersion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_URI_FIELD,
            DATA_INPUT_CONFIG_FIELD, FRAMEWORK_FIELD, FRAMEWORK_VERSION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String s3Uri;

    private final String dataInputConfig;

    private final String framework;

    private final String frameworkVersion;

    private InputConfig(BuilderImpl builder) {
        this.s3Uri = builder.s3Uri;
        this.dataInputConfig = builder.dataInputConfig;
        this.framework = builder.framework;
        this.frameworkVersion = builder.frameworkVersion;
    }

    /**
     * <p>
     * The S3 path where the model artifacts, which result from model training, are stored. This path must point to a
     * single gzip compressed tar archive (.tar.gz suffix).
     * </p>
     * 
     * @return The S3 path where the model artifacts, which result from model training, are stored. This path must point
     *         to a single gzip compressed tar archive (.tar.gz suffix).
     */
    public final String s3Uri() {
        return s3Uri;
    }

    /**
     * <p>
     * Specifies the name and shape of the expected data inputs for your trained model with a JSON dictionary form. The
     * data inputs are <code>Framework</code> specific.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TensorFlow</code>: You must specify the name and shape (NHWC format) of the expected data inputs using a
     * dictionary format for your trained model. The dictionary formats required for the console and CLI are different.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Examples for one input:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If using the console, <code>{"input":[1,1024,1024,3]}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If using the CLI, <code>{\"input\":[1,1024,1024,3]}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Examples for two inputs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If using the console, <code>{"data1": [1,28,28,1], "data2":[1,28,28,1]}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If using the CLI, <code>{\"data1\": [1,28,28,1], \"data2\":[1,28,28,1]}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>KERAS</code>: You must specify the name and shape (NCHW format) of expected data inputs using a dictionary
     * format for your trained model. Note that while Keras model artifacts should be uploaded in NHWC (channel-last)
     * format, <code>DataInputConfig</code> should be specified in NCHW (channel-first) format. The dictionary formats
     * required for the console and CLI are different.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Examples for one input:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If using the console, <code>{"input_1":[1,3,224,224]}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If using the CLI, <code>{\"input_1\":[1,3,224,224]}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Examples for two inputs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If using the console, <code>{"input_1": [1,3,224,224], "input_2":[1,3,224,224]} </code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If using the CLI, <code>{\"input_1\": [1,3,224,224], \"input_2\":[1,3,224,224]}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>MXNET/ONNX/DARKNET</code>: You must specify the name and shape (NCHW format) of the expected data inputs in
     * order using a dictionary format for your trained model. The dictionary formats required for the console and CLI
     * are different.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Examples for one input:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If using the console, <code>{"data":[1,3,1024,1024]}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If using the CLI, <code>{\"data\":[1,3,1024,1024]}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Examples for two inputs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If using the console, <code>{"var1": [1,1,28,28], "var2":[1,1,28,28]} </code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If using the CLI, <code>{\"var1\": [1,1,28,28], \"var2\":[1,1,28,28]}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>PyTorch</code>: You can either specify the name and shape (NCHW format) of expected data inputs in order
     * using a dictionary format for your trained model or you can specify the shape only using a list format. The
     * dictionary formats required for the console and CLI are different. The list formats for the console and CLI are
     * the same.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Examples for one input in dictionary format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If using the console, <code>{"input0":[1,3,224,224]}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If using the CLI, <code>{\"input0\":[1,3,224,224]}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Example for one input in list format: <code>[[1,3,224,224]]</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Examples for two inputs in dictionary format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If using the console, <code>{"input0":[1,3,224,224], "input1":[1,3,224,224]}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If using the CLI, <code>{\"input0\":[1,3,224,224], \"input1\":[1,3,224,224]} </code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Example for two inputs in list format: <code>[[1,3,224,224], [1,3,224,224]]</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>XGBOOST</code>: input data name and shape are not needed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>DataInputConfig</code> supports the following parameters for <code>CoreML</code> <code>TargetDevice</code>
     * (ML Model format):
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>shape</code>: Input shape, for example <code>{"input_1": {"shape": [1,224,224,3]}}</code>. In addition to
     * static input shapes, CoreML converter supports Flexible input shapes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Range Dimension. You can use the Range Dimension feature if you know the input shape will be within some specific
     * interval in that dimension, for example: <code>{"input_1": {"shape": ["1..10", 224, 224, 3]}}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Enumerated shapes. Sometimes, the models are trained to work only on a select set of inputs. You can enumerate
     * all supported input shapes, for example:
     * <code>{"input_1": {"shape": [[1, 224, 224, 3], [1, 160, 160, 3]]}}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>default_shape</code>: Default input shape. You can set a default shape during conversion for both Range
     * Dimension and Enumerated Shapes. For example
     * <code>{"input_1": {"shape": ["1..10", 224, 224, 3], "default_shape": [1, 224, 224, 3]}}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>type</code>: Input type. Allowed values: <code>Image</code> and <code>Tensor</code>. By default, the
     * converter generates an ML Model with inputs of type Tensor (MultiArray). User can set input type to be Image.
     * Image input type requires additional input parameters such as <code>bias</code> and <code>scale</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>bias</code>: If the input type is an Image, you need to provide the bias vector.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>scale</code>: If the input type is an Image, you need to provide a scale factor.
     * </p>
     * </li>
     * </ul>
     * <p>
     * CoreML <code>ClassifierConfig</code> parameters can be specified using <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html">OutputConfig</a>
     * <code>CompilerOptions</code>. CoreML converter supports Tensorflow and PyTorch models. CoreML conversion
     * examples:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Tensor type input:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>"DataInputConfig": {"input_1": {"shape": [[1,224,224,3], [1,160,160,3]], "default_shape": [1,224,224,3]}}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Tensor type input without input name (PyTorch):
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>"DataInputConfig": [{"shape": [[1,3,224,224], [1,3,160,160]], "default_shape": [1,3,224,224]}]</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Image type input:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>"DataInputConfig": {"input_1": {"shape": [[1,224,224,3], [1,160,160,3]], "default_shape": [1,224,224,3], "type": "Image", "bias": [-1,-1,-1], "scale": 0.007843137255}}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"CompilerOptions": {"class_labels": "imagenet_labels_1000.txt"}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Image type input without input name (PyTorch):
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>"DataInputConfig": [{"shape": [[1,3,224,224], [1,3,160,160]], "default_shape": [1,3,224,224], "type": "Image", "bias": [-1,-1,-1], "scale": 0.007843137255}]</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"CompilerOptions": {"class_labels": "imagenet_labels_1000.txt"}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * Depending on the model format, <code>DataInputConfig</code> requires the following parameters for
     * <code>ml_eia2</code> <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html#sagemaker-Type-OutputConfig-TargetDevice"
     * >OutputConfig:TargetDevice</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For TensorFlow models saved in the SavedModel format, specify the input names from <code>signature_def_key</code>
     * and the input model shapes for <code>DataInputConfig</code>. Specify the <code>signature_def_key</code> in <a
     * href=
     * "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html#sagemaker-Type-OutputConfig-CompilerOptions"
     * > <code>OutputConfig:CompilerOptions</code> </a> if the model does not use TensorFlow's default signature def
     * key. For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>"DataInputConfig": {"inputs": [1, 224, 224, 3]}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"CompilerOptions": {"signature_def_key": "serving_custom"}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For TensorFlow models saved as a frozen graph, specify the input tensor names and shapes in
     * <code>DataInputConfig</code> and the output tensor names for <code>output_names</code> in <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html#sagemaker-Type-OutputConfig-CompilerOptions"
     * > <code>OutputConfig:CompilerOptions</code> </a>. For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>"DataInputConfig": {"input_tensor:0": [1, 224, 224, 3]}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"CompilerOptions": {"output_names": ["output_tensor:0"]}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * 
     * @return Specifies the name and shape of the expected data inputs for your trained model with a JSON dictionary
     *         form. The data inputs are <code>Framework</code> specific. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>TensorFlow</code>: You must specify the name and shape (NHWC format) of the expected data inputs
     *         using a dictionary format for your trained model. The dictionary formats required for the console and CLI
     *         are different.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Examples for one input:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If using the console, <code>{"input":[1,1024,1024,3]}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If using the CLI, <code>{\"input\":[1,1024,1024,3]}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Examples for two inputs:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If using the console, <code>{"data1": [1,28,28,1], "data2":[1,28,28,1]}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If using the CLI, <code>{\"data1\": [1,28,28,1], \"data2\":[1,28,28,1]}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KERAS</code>: You must specify the name and shape (NCHW format) of expected data inputs using a
     *         dictionary format for your trained model. Note that while Keras model artifacts should be uploaded in
     *         NHWC (channel-last) format, <code>DataInputConfig</code> should be specified in NCHW (channel-first)
     *         format. The dictionary formats required for the console and CLI are different.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Examples for one input:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If using the console, <code>{"input_1":[1,3,224,224]}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If using the CLI, <code>{\"input_1\":[1,3,224,224]}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Examples for two inputs:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If using the console, <code>{"input_1": [1,3,224,224], "input_2":[1,3,224,224]} </code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If using the CLI, <code>{\"input_1\": [1,3,224,224], \"input_2\":[1,3,224,224]}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MXNET/ONNX/DARKNET</code>: You must specify the name and shape (NCHW format) of the expected data
     *         inputs in order using a dictionary format for your trained model. The dictionary formats required for the
     *         console and CLI are different.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Examples for one input:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If using the console, <code>{"data":[1,3,1024,1024]}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If using the CLI, <code>{\"data\":[1,3,1024,1024]}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Examples for two inputs:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If using the console, <code>{"var1": [1,1,28,28], "var2":[1,1,28,28]} </code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If using the CLI, <code>{\"var1\": [1,1,28,28], \"var2\":[1,1,28,28]}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PyTorch</code>: You can either specify the name and shape (NCHW format) of expected data inputs in
     *         order using a dictionary format for your trained model or you can specify the shape only using a list
     *         format. The dictionary formats required for the console and CLI are different. The list formats for the
     *         console and CLI are the same.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Examples for one input in dictionary format:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If using the console, <code>{"input0":[1,3,224,224]}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If using the CLI, <code>{\"input0\":[1,3,224,224]}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Example for one input in list format: <code>[[1,3,224,224]]</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Examples for two inputs in dictionary format:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If using the console, <code>{"input0":[1,3,224,224], "input1":[1,3,224,224]}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If using the CLI, <code>{\"input0\":[1,3,224,224], \"input1\":[1,3,224,224]} </code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Example for two inputs in list format: <code>[[1,3,224,224], [1,3,224,224]]</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>XGBOOST</code>: input data name and shape are not needed.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>DataInputConfig</code> supports the following parameters for <code>CoreML</code>
     *         <code>TargetDevice</code> (ML Model format):
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>shape</code>: Input shape, for example <code>{"input_1": {"shape": [1,224,224,3]}}</code>. In
     *         addition to static input shapes, CoreML converter supports Flexible input shapes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Range Dimension. You can use the Range Dimension feature if you know the input shape will be within some
     *         specific interval in that dimension, for example:
     *         <code>{"input_1": {"shape": ["1..10", 224, 224, 3]}}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Enumerated shapes. Sometimes, the models are trained to work only on a select set of inputs. You can
     *         enumerate all supported input shapes, for example:
     *         <code>{"input_1": {"shape": [[1, 224, 224, 3], [1, 160, 160, 3]]}}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>default_shape</code>: Default input shape. You can set a default shape during conversion for both
     *         Range Dimension and Enumerated Shapes. For example
     *         <code>{"input_1": {"shape": ["1..10", 224, 224, 3], "default_shape": [1, 224, 224, 3]}}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>type</code>: Input type. Allowed values: <code>Image</code> and <code>Tensor</code>. By default,
     *         the converter generates an ML Model with inputs of type Tensor (MultiArray). User can set input type to
     *         be Image. Image input type requires additional input parameters such as <code>bias</code> and
     *         <code>scale</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>bias</code>: If the input type is an Image, you need to provide the bias vector.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>scale</code>: If the input type is an Image, you need to provide a scale factor.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         CoreML <code>ClassifierConfig</code> parameters can be specified using <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html">OutputConfig</a>
     *         <code>CompilerOptions</code>. CoreML converter supports Tensorflow and PyTorch models. CoreML conversion
     *         examples:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Tensor type input:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>"DataInputConfig": {"input_1": {"shape": [[1,224,224,3], [1,160,160,3]], "default_shape": [1,224,224,3]}}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Tensor type input without input name (PyTorch):
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>"DataInputConfig": [{"shape": [[1,3,224,224], [1,3,160,160]], "default_shape": [1,3,224,224]}]</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Image type input:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>"DataInputConfig": {"input_1": {"shape": [[1,224,224,3], [1,160,160,3]], "default_shape": [1,224,224,3], "type": "Image", "bias": [-1,-1,-1], "scale": 0.007843137255}}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"CompilerOptions": {"class_labels": "imagenet_labels_1000.txt"}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Image type input without input name (PyTorch):
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>"DataInputConfig": [{"shape": [[1,3,224,224], [1,3,160,160]], "default_shape": [1,3,224,224], "type": "Image", "bias": [-1,-1,-1], "scale": 0.007843137255}]</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"CompilerOptions": {"class_labels": "imagenet_labels_1000.txt"}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         <p>
     *         Depending on the model format, <code>DataInputConfig</code> requires the following parameters for
     *         <code>ml_eia2</code> <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html#sagemaker-Type-OutputConfig-TargetDevice"
     *         >OutputConfig:TargetDevice</a>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For TensorFlow models saved in the SavedModel format, specify the input names from
     *         <code>signature_def_key</code> and the input model shapes for <code>DataInputConfig</code>. Specify the
     *         <code>signature_def_key</code> in <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html#sagemaker-Type-OutputConfig-CompilerOptions"
     *         > <code>OutputConfig:CompilerOptions</code> </a> if the model does not use TensorFlow's default signature
     *         def key. For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>"DataInputConfig": {"inputs": [1, 224, 224, 3]}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"CompilerOptions": {"signature_def_key": "serving_custom"}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For TensorFlow models saved as a frozen graph, specify the input tensor names and shapes in
     *         <code>DataInputConfig</code> and the output tensor names for <code>output_names</code> in <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html#sagemaker-Type-OutputConfig-CompilerOptions"
     *         > <code>OutputConfig:CompilerOptions</code> </a>. For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>"DataInputConfig": {"input_tensor:0": [1, 224, 224, 3]}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"CompilerOptions": {"output_names": ["output_tensor:0"]}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     */
    public final String dataInputConfig() {
        return dataInputConfig;
    }

    /**
     * <p>
     * Identifies the framework in which the model was trained. For example: TENSORFLOW.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #framework} will
     * return {@link Framework#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #frameworkAsString}.
     * </p>
     * 
     * @return Identifies the framework in which the model was trained. For example: TENSORFLOW.
     * @see Framework
     */
    public final Framework framework() {
        return Framework.fromValue(framework);
    }

    /**
     * <p>
     * Identifies the framework in which the model was trained. For example: TENSORFLOW.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #framework} will
     * return {@link Framework#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #frameworkAsString}.
     * </p>
     * 
     * @return Identifies the framework in which the model was trained. For example: TENSORFLOW.
     * @see Framework
     */
    public final String frameworkAsString() {
        return framework;
    }

    /**
     * <p>
     * Specifies the framework version to use. This API field is only supported for the MXNet, PyTorch, TensorFlow and
     * TensorFlow Lite frameworks.
     * </p>
     * <p>
     * For information about framework versions supported for cloud targets and edge devices, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/neo-supported-cloud.html">Cloud Supported Instance Types
     * and Frameworks</a> and <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/neo-supported-devices-edge-frameworks.html">Edge Supported
     * Frameworks</a>.
     * </p>
     * 
     * @return Specifies the framework version to use. This API field is only supported for the MXNet, PyTorch,
     *         TensorFlow and TensorFlow Lite frameworks.</p>
     *         <p>
     *         For information about framework versions supported for cloud targets and edge devices, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/neo-supported-cloud.html">Cloud Supported Instance
     *         Types and Frameworks</a> and <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/neo-supported-devices-edge-frameworks.html">Edge
     *         Supported Frameworks</a>.
     */
    public final String frameworkVersion() {
        return frameworkVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3Uri());
        hashCode = 31 * hashCode + Objects.hashCode(dataInputConfig());
        hashCode = 31 * hashCode + Objects.hashCode(frameworkAsString());
        hashCode = 31 * hashCode + Objects.hashCode(frameworkVersion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InputConfig)) {
            return false;
        }
        InputConfig other = (InputConfig) obj;
        return Objects.equals(s3Uri(), other.s3Uri()) && Objects.equals(dataInputConfig(), other.dataInputConfig())
                && Objects.equals(frameworkAsString(), other.frameworkAsString())
                && Objects.equals(frameworkVersion(), other.frameworkVersion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InputConfig").add("S3Uri", s3Uri()).add("DataInputConfig", dataInputConfig())
                .add("Framework", frameworkAsString()).add("FrameworkVersion", frameworkVersion()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3Uri":
            return Optional.ofNullable(clazz.cast(s3Uri()));
        case "DataInputConfig":
            return Optional.ofNullable(clazz.cast(dataInputConfig()));
        case "Framework":
            return Optional.ofNullable(clazz.cast(frameworkAsString()));
        case "FrameworkVersion":
            return Optional.ofNullable(clazz.cast(frameworkVersion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("S3Uri", S3_URI_FIELD);
        map.put("DataInputConfig", DATA_INPUT_CONFIG_FIELD);
        map.put("Framework", FRAMEWORK_FIELD);
        map.put("FrameworkVersion", FRAMEWORK_VERSION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InputConfig, T> g) {
        return obj -> g.apply((InputConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InputConfig> {
        /**
         * <p>
         * The S3 path where the model artifacts, which result from model training, are stored. This path must point to
         * a single gzip compressed tar archive (.tar.gz suffix).
         * </p>
         * 
         * @param s3Uri
         *        The S3 path where the model artifacts, which result from model training, are stored. This path must
         *        point to a single gzip compressed tar archive (.tar.gz suffix).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Uri(String s3Uri);

        /**
         * <p>
         * Specifies the name and shape of the expected data inputs for your trained model with a JSON dictionary form.
         * The data inputs are <code>Framework</code> specific.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TensorFlow</code>: You must specify the name and shape (NHWC format) of the expected data inputs using
         * a dictionary format for your trained model. The dictionary formats required for the console and CLI are
         * different.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Examples for one input:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If using the console, <code>{"input":[1,1024,1024,3]}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If using the CLI, <code>{\"input\":[1,1024,1024,3]}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Examples for two inputs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If using the console, <code>{"data1": [1,28,28,1], "data2":[1,28,28,1]}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If using the CLI, <code>{\"data1\": [1,28,28,1], \"data2\":[1,28,28,1]}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>KERAS</code>: You must specify the name and shape (NCHW format) of expected data inputs using a
         * dictionary format for your trained model. Note that while Keras model artifacts should be uploaded in NHWC
         * (channel-last) format, <code>DataInputConfig</code> should be specified in NCHW (channel-first) format. The
         * dictionary formats required for the console and CLI are different.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Examples for one input:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If using the console, <code>{"input_1":[1,3,224,224]}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If using the CLI, <code>{\"input_1\":[1,3,224,224]}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Examples for two inputs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If using the console, <code>{"input_1": [1,3,224,224], "input_2":[1,3,224,224]} </code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If using the CLI, <code>{\"input_1\": [1,3,224,224], \"input_2\":[1,3,224,224]}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>MXNET/ONNX/DARKNET</code>: You must specify the name and shape (NCHW format) of the expected data
         * inputs in order using a dictionary format for your trained model. The dictionary formats required for the
         * console and CLI are different.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Examples for one input:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If using the console, <code>{"data":[1,3,1024,1024]}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If using the CLI, <code>{\"data\":[1,3,1024,1024]}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Examples for two inputs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If using the console, <code>{"var1": [1,1,28,28], "var2":[1,1,28,28]} </code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If using the CLI, <code>{\"var1\": [1,1,28,28], \"var2\":[1,1,28,28]}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>PyTorch</code>: You can either specify the name and shape (NCHW format) of expected data inputs in
         * order using a dictionary format for your trained model or you can specify the shape only using a list format.
         * The dictionary formats required for the console and CLI are different. The list formats for the console and
         * CLI are the same.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Examples for one input in dictionary format:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If using the console, <code>{"input0":[1,3,224,224]}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If using the CLI, <code>{\"input0\":[1,3,224,224]}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Example for one input in list format: <code>[[1,3,224,224]]</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Examples for two inputs in dictionary format:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If using the console, <code>{"input0":[1,3,224,224], "input1":[1,3,224,224]}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If using the CLI, <code>{\"input0\":[1,3,224,224], \"input1\":[1,3,224,224]} </code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Example for two inputs in list format: <code>[[1,3,224,224], [1,3,224,224]]</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>XGBOOST</code>: input data name and shape are not needed.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>DataInputConfig</code> supports the following parameters for <code>CoreML</code>
         * <code>TargetDevice</code> (ML Model format):
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>shape</code>: Input shape, for example <code>{"input_1": {"shape": [1,224,224,3]}}</code>. In addition
         * to static input shapes, CoreML converter supports Flexible input shapes:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Range Dimension. You can use the Range Dimension feature if you know the input shape will be within some
         * specific interval in that dimension, for example: <code>{"input_1": {"shape": ["1..10", 224, 224, 3]}}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Enumerated shapes. Sometimes, the models are trained to work only on a select set of inputs. You can
         * enumerate all supported input shapes, for example:
         * <code>{"input_1": {"shape": [[1, 224, 224, 3], [1, 160, 160, 3]]}}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>default_shape</code>: Default input shape. You can set a default shape during conversion for both Range
         * Dimension and Enumerated Shapes. For example
         * <code>{"input_1": {"shape": ["1..10", 224, 224, 3], "default_shape": [1, 224, 224, 3]}}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>type</code>: Input type. Allowed values: <code>Image</code> and <code>Tensor</code>. By default, the
         * converter generates an ML Model with inputs of type Tensor (MultiArray). User can set input type to be Image.
         * Image input type requires additional input parameters such as <code>bias</code> and <code>scale</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>bias</code>: If the input type is an Image, you need to provide the bias vector.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>scale</code>: If the input type is an Image, you need to provide a scale factor.
         * </p>
         * </li>
         * </ul>
         * <p>
         * CoreML <code>ClassifierConfig</code> parameters can be specified using <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html">OutputConfig</a>
         * <code>CompilerOptions</code>. CoreML converter supports Tensorflow and PyTorch models. CoreML conversion
         * examples:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Tensor type input:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>"DataInputConfig": {"input_1": {"shape": [[1,224,224,3], [1,160,160,3]], "default_shape": [1,224,224,3]}}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Tensor type input without input name (PyTorch):
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>"DataInputConfig": [{"shape": [[1,3,224,224], [1,3,160,160]], "default_shape": [1,3,224,224]}]</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Image type input:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>"DataInputConfig": {"input_1": {"shape": [[1,224,224,3], [1,160,160,3]], "default_shape": [1,224,224,3], "type": "Image", "bias": [-1,-1,-1], "scale": 0.007843137255}}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"CompilerOptions": {"class_labels": "imagenet_labels_1000.txt"}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Image type input without input name (PyTorch):
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>"DataInputConfig": [{"shape": [[1,3,224,224], [1,3,160,160]], "default_shape": [1,3,224,224], "type": "Image", "bias": [-1,-1,-1], "scale": 0.007843137255}]</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"CompilerOptions": {"class_labels": "imagenet_labels_1000.txt"}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <p>
         * Depending on the model format, <code>DataInputConfig</code> requires the following parameters for
         * <code>ml_eia2</code> <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html#sagemaker-Type-OutputConfig-TargetDevice"
         * >OutputConfig:TargetDevice</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For TensorFlow models saved in the SavedModel format, specify the input names from
         * <code>signature_def_key</code> and the input model shapes for <code>DataInputConfig</code>. Specify the
         * <code>signature_def_key</code> in <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html#sagemaker-Type-OutputConfig-CompilerOptions"
         * > <code>OutputConfig:CompilerOptions</code> </a> if the model does not use TensorFlow's default signature def
         * key. For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>"DataInputConfig": {"inputs": [1, 224, 224, 3]}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"CompilerOptions": {"signature_def_key": "serving_custom"}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For TensorFlow models saved as a frozen graph, specify the input tensor names and shapes in
         * <code>DataInputConfig</code> and the output tensor names for <code>output_names</code> in <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html#sagemaker-Type-OutputConfig-CompilerOptions"
         * > <code>OutputConfig:CompilerOptions</code> </a>. For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>"DataInputConfig": {"input_tensor:0": [1, 224, 224, 3]}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"CompilerOptions": {"output_names": ["output_tensor:0"]}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param dataInputConfig
         *        Specifies the name and shape of the expected data inputs for your trained model with a JSON dictionary
         *        form. The data inputs are <code>Framework</code> specific. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TensorFlow</code>: You must specify the name and shape (NHWC format) of the expected data inputs
         *        using a dictionary format for your trained model. The dictionary formats required for the console and
         *        CLI are different.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Examples for one input:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If using the console, <code>{"input":[1,1024,1024,3]}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If using the CLI, <code>{\"input\":[1,1024,1024,3]}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Examples for two inputs:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If using the console, <code>{"data1": [1,28,28,1], "data2":[1,28,28,1]}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If using the CLI, <code>{\"data1\": [1,28,28,1], \"data2\":[1,28,28,1]}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KERAS</code>: You must specify the name and shape (NCHW format) of expected data inputs using a
         *        dictionary format for your trained model. Note that while Keras model artifacts should be uploaded in
         *        NHWC (channel-last) format, <code>DataInputConfig</code> should be specified in NCHW (channel-first)
         *        format. The dictionary formats required for the console and CLI are different.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Examples for one input:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If using the console, <code>{"input_1":[1,3,224,224]}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If using the CLI, <code>{\"input_1\":[1,3,224,224]}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Examples for two inputs:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If using the console, <code>{"input_1": [1,3,224,224], "input_2":[1,3,224,224]} </code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If using the CLI, <code>{\"input_1\": [1,3,224,224], \"input_2\":[1,3,224,224]}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MXNET/ONNX/DARKNET</code>: You must specify the name and shape (NCHW format) of the expected
         *        data inputs in order using a dictionary format for your trained model. The dictionary formats required
         *        for the console and CLI are different.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Examples for one input:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If using the console, <code>{"data":[1,3,1024,1024]}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If using the CLI, <code>{\"data\":[1,3,1024,1024]}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Examples for two inputs:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If using the console, <code>{"var1": [1,1,28,28], "var2":[1,1,28,28]} </code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If using the CLI, <code>{\"var1\": [1,1,28,28], \"var2\":[1,1,28,28]}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PyTorch</code>: You can either specify the name and shape (NCHW format) of expected data inputs
         *        in order using a dictionary format for your trained model or you can specify the shape only using a
         *        list format. The dictionary formats required for the console and CLI are different. The list formats
         *        for the console and CLI are the same.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Examples for one input in dictionary format:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If using the console, <code>{"input0":[1,3,224,224]}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If using the CLI, <code>{\"input0\":[1,3,224,224]}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Example for one input in list format: <code>[[1,3,224,224]]</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Examples for two inputs in dictionary format:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If using the console, <code>{"input0":[1,3,224,224], "input1":[1,3,224,224]}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If using the CLI, <code>{\"input0\":[1,3,224,224], \"input1\":[1,3,224,224]} </code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Example for two inputs in list format: <code>[[1,3,224,224], [1,3,224,224]]</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>XGBOOST</code>: input data name and shape are not needed.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>DataInputConfig</code> supports the following parameters for <code>CoreML</code>
         *        <code>TargetDevice</code> (ML Model format):
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>shape</code>: Input shape, for example <code>{"input_1": {"shape": [1,224,224,3]}}</code>. In
         *        addition to static input shapes, CoreML converter supports Flexible input shapes:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Range Dimension. You can use the Range Dimension feature if you know the input shape will be within
         *        some specific interval in that dimension, for example:
         *        <code>{"input_1": {"shape": ["1..10", 224, 224, 3]}}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Enumerated shapes. Sometimes, the models are trained to work only on a select set of inputs. You can
         *        enumerate all supported input shapes, for example:
         *        <code>{"input_1": {"shape": [[1, 224, 224, 3], [1, 160, 160, 3]]}}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>default_shape</code>: Default input shape. You can set a default shape during conversion for
         *        both Range Dimension and Enumerated Shapes. For example
         *        <code>{"input_1": {"shape": ["1..10", 224, 224, 3], "default_shape": [1, 224, 224, 3]}}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>type</code>: Input type. Allowed values: <code>Image</code> and <code>Tensor</code>. By default,
         *        the converter generates an ML Model with inputs of type Tensor (MultiArray). User can set input type
         *        to be Image. Image input type requires additional input parameters such as <code>bias</code> and
         *        <code>scale</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>bias</code>: If the input type is an Image, you need to provide the bias vector.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>scale</code>: If the input type is an Image, you need to provide a scale factor.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        CoreML <code>ClassifierConfig</code> parameters can be specified using <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html"
         *        >OutputConfig</a> <code>CompilerOptions</code>. CoreML converter supports Tensorflow and PyTorch
         *        models. CoreML conversion examples:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Tensor type input:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>"DataInputConfig": {"input_1": {"shape": [[1,224,224,3], [1,160,160,3]], "default_shape": [1,224,224,3]}}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Tensor type input without input name (PyTorch):
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>"DataInputConfig": [{"shape": [[1,3,224,224], [1,3,160,160]], "default_shape": [1,3,224,224]}]</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Image type input:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>"DataInputConfig": {"input_1": {"shape": [[1,224,224,3], [1,160,160,3]], "default_shape": [1,224,224,3], "type": "Image", "bias": [-1,-1,-1], "scale": 0.007843137255}}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"CompilerOptions": {"class_labels": "imagenet_labels_1000.txt"}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Image type input without input name (PyTorch):
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>"DataInputConfig": [{"shape": [[1,3,224,224], [1,3,160,160]], "default_shape": [1,3,224,224], "type": "Image", "bias": [-1,-1,-1], "scale": 0.007843137255}]</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"CompilerOptions": {"class_labels": "imagenet_labels_1000.txt"}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <p>
         *        Depending on the model format, <code>DataInputConfig</code> requires the following parameters for
         *        <code>ml_eia2</code> <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html#sagemaker-Type-OutputConfig-TargetDevice"
         *        >OutputConfig:TargetDevice</a>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For TensorFlow models saved in the SavedModel format, specify the input names from
         *        <code>signature_def_key</code> and the input model shapes for <code>DataInputConfig</code>. Specify
         *        the <code>signature_def_key</code> in <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html#sagemaker-Type-OutputConfig-CompilerOptions"
         *        > <code>OutputConfig:CompilerOptions</code> </a> if the model does not use TensorFlow's default
         *        signature def key. For example:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>"DataInputConfig": {"inputs": [1, 224, 224, 3]}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"CompilerOptions": {"signature_def_key": "serving_custom"}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For TensorFlow models saved as a frozen graph, specify the input tensor names and shapes in
         *        <code>DataInputConfig</code> and the output tensor names for <code>output_names</code> in <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OutputConfig.html#sagemaker-Type-OutputConfig-CompilerOptions"
         *        > <code>OutputConfig:CompilerOptions</code> </a>. For example:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>"DataInputConfig": {"input_tensor:0": [1, 224, 224, 3]}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"CompilerOptions": {"output_names": ["output_tensor:0"]}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataInputConfig(String dataInputConfig);

        /**
         * <p>
         * Identifies the framework in which the model was trained. For example: TENSORFLOW.
         * </p>
         * 
         * @param framework
         *        Identifies the framework in which the model was trained. For example: TENSORFLOW.
         * @see Framework
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Framework
         */
        Builder framework(String framework);

        /**
         * <p>
         * Identifies the framework in which the model was trained. For example: TENSORFLOW.
         * </p>
         * 
         * @param framework
         *        Identifies the framework in which the model was trained. For example: TENSORFLOW.
         * @see Framework
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Framework
         */
        Builder framework(Framework framework);

        /**
         * <p>
         * Specifies the framework version to use. This API field is only supported for the MXNet, PyTorch, TensorFlow
         * and TensorFlow Lite frameworks.
         * </p>
         * <p>
         * For information about framework versions supported for cloud targets and edge devices, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/neo-supported-cloud.html">Cloud Supported Instance
         * Types and Frameworks</a> and <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/neo-supported-devices-edge-frameworks.html">Edge
         * Supported Frameworks</a>.
         * </p>
         * 
         * @param frameworkVersion
         *        Specifies the framework version to use. This API field is only supported for the MXNet, PyTorch,
         *        TensorFlow and TensorFlow Lite frameworks.</p>
         *        <p>
         *        For information about framework versions supported for cloud targets and edge devices, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/neo-supported-cloud.html">Cloud Supported
         *        Instance Types and Frameworks</a> and <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/neo-supported-devices-edge-frameworks.html">Edge
         *        Supported Frameworks</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder frameworkVersion(String frameworkVersion);
    }

    static final class BuilderImpl implements Builder {
        private String s3Uri;

        private String dataInputConfig;

        private String framework;

        private String frameworkVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(InputConfig model) {
            s3Uri(model.s3Uri);
            dataInputConfig(model.dataInputConfig);
            framework(model.framework);
            frameworkVersion(model.frameworkVersion);
        }

        public final String getS3Uri() {
            return s3Uri;
        }

        public final void setS3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
        }

        @Override
        public final Builder s3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
            return this;
        }

        public final String getDataInputConfig() {
            return dataInputConfig;
        }

        public final void setDataInputConfig(String dataInputConfig) {
            this.dataInputConfig = dataInputConfig;
        }

        @Override
        public final Builder dataInputConfig(String dataInputConfig) {
            this.dataInputConfig = dataInputConfig;
            return this;
        }

        public final String getFramework() {
            return framework;
        }

        public final void setFramework(String framework) {
            this.framework = framework;
        }

        @Override
        public final Builder framework(String framework) {
            this.framework = framework;
            return this;
        }

        @Override
        public final Builder framework(Framework framework) {
            this.framework(framework == null ? null : framework.toString());
            return this;
        }

        public final String getFrameworkVersion() {
            return frameworkVersion;
        }

        public final void setFrameworkVersion(String frameworkVersion) {
            this.frameworkVersion = frameworkVersion;
        }

        @Override
        public final Builder frameworkVersion(String frameworkVersion) {
            this.frameworkVersion = frameworkVersion;
            return this;
        }

        @Override
        public InputConfig build() {
            return new InputConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
