/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Settings for the model quantization technique that's applied by a model optimization job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ModelQuantizationConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<ModelQuantizationConfig.Builder, ModelQuantizationConfig> {
    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Image")
            .getter(getter(ModelQuantizationConfig::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Image").build()).build();

    private static final SdkField<Map<String, String>> OVERRIDE_ENVIRONMENT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("OverrideEnvironment")
            .getter(getter(ModelQuantizationConfig::overrideEnvironment))
            .setter(setter(Builder::overrideEnvironment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OverrideEnvironment").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_FIELD,
            OVERRIDE_ENVIRONMENT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String image;

    private final Map<String, String> overrideEnvironment;

    private ModelQuantizationConfig(BuilderImpl builder) {
        this.image = builder.image;
        this.overrideEnvironment = builder.overrideEnvironment;
    }

    /**
     * <p>
     * The URI of an LMI DLC in Amazon ECR. SageMaker uses this image to run the optimization.
     * </p>
     * 
     * @return The URI of an LMI DLC in Amazon ECR. SageMaker uses this image to run the optimization.
     */
    public final String image() {
        return image;
    }

    /**
     * For responses, this returns true if the service returned a value for the OverrideEnvironment property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasOverrideEnvironment() {
        return overrideEnvironment != null && !(overrideEnvironment instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Environment variables that override the default ones in the model container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOverrideEnvironment} method.
     * </p>
     * 
     * @return Environment variables that override the default ones in the model container.
     */
    public final Map<String, String> overrideEnvironment() {
        return overrideEnvironment;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(hasOverrideEnvironment() ? overrideEnvironment() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ModelQuantizationConfig)) {
            return false;
        }
        ModelQuantizationConfig other = (ModelQuantizationConfig) obj;
        return Objects.equals(image(), other.image()) && hasOverrideEnvironment() == other.hasOverrideEnvironment()
                && Objects.equals(overrideEnvironment(), other.overrideEnvironment());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ModelQuantizationConfig").add("Image", image())
                .add("OverrideEnvironment", hasOverrideEnvironment() ? overrideEnvironment() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Image":
            return Optional.ofNullable(clazz.cast(image()));
        case "OverrideEnvironment":
            return Optional.ofNullable(clazz.cast(overrideEnvironment()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Image", IMAGE_FIELD);
        map.put("OverrideEnvironment", OVERRIDE_ENVIRONMENT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ModelQuantizationConfig, T> g) {
        return obj -> g.apply((ModelQuantizationConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ModelQuantizationConfig> {
        /**
         * <p>
         * The URI of an LMI DLC in Amazon ECR. SageMaker uses this image to run the optimization.
         * </p>
         * 
         * @param image
         *        The URI of an LMI DLC in Amazon ECR. SageMaker uses this image to run the optimization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * Environment variables that override the default ones in the model container.
         * </p>
         * 
         * @param overrideEnvironment
         *        Environment variables that override the default ones in the model container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overrideEnvironment(Map<String, String> overrideEnvironment);
    }

    static final class BuilderImpl implements Builder {
        private String image;

        private Map<String, String> overrideEnvironment = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ModelQuantizationConfig model) {
            image(model.image);
            overrideEnvironment(model.overrideEnvironment);
        }

        public final String getImage() {
            return image;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final Map<String, String> getOverrideEnvironment() {
            if (overrideEnvironment instanceof SdkAutoConstructMap) {
                return null;
            }
            return overrideEnvironment;
        }

        public final void setOverrideEnvironment(Map<String, String> overrideEnvironment) {
            this.overrideEnvironment = OptimizationJobEnvironmentVariablesCopier.copy(overrideEnvironment);
        }

        @Override
        public final Builder overrideEnvironment(Map<String, String> overrideEnvironment) {
            this.overrideEnvironment = OptimizationJobEnvironmentVariablesCopier.copy(overrideEnvironment);
            return this;
        }

        @Override
        public ModelQuantizationConfig build() {
            return new ModelQuantizationConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
