/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Identifies a model that you want to host and the resources chosen to deploy for hosting it. If you are deploying
 * multiple models, tell SageMaker how to distribute traffic among the models by specifying variant weights. For more
 * information on production variants, check <a
 * href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-ab-testing.html"> Production variants</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProductionVariant implements SdkPojo, Serializable,
        ToCopyableBuilder<ProductionVariant.Builder, ProductionVariant> {
    private static final SdkField<String> VARIANT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VariantName").getter(getter(ProductionVariant::variantName)).setter(setter(Builder::variantName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VariantName").build()).build();

    private static final SdkField<String> MODEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelName").getter(getter(ProductionVariant::modelName)).setter(setter(Builder::modelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelName").build()).build();

    private static final SdkField<Integer> INITIAL_INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("InitialInstanceCount").getter(getter(ProductionVariant::initialInstanceCount))
            .setter(setter(Builder::initialInstanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitialInstanceCount").build())
            .build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(ProductionVariant::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Float> INITIAL_VARIANT_WEIGHT_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("InitialVariantWeight").getter(getter(ProductionVariant::initialVariantWeight))
            .setter(setter(Builder::initialVariantWeight))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitialVariantWeight").build())
            .build();

    private static final SdkField<String> ACCELERATOR_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AcceleratorType").getter(getter(ProductionVariant::acceleratorTypeAsString))
            .setter(setter(Builder::acceleratorType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceleratorType").build()).build();

    private static final SdkField<ProductionVariantCoreDumpConfig> CORE_DUMP_CONFIG_FIELD = SdkField
            .<ProductionVariantCoreDumpConfig> builder(MarshallingType.SDK_POJO).memberName("CoreDumpConfig")
            .getter(getter(ProductionVariant::coreDumpConfig)).setter(setter(Builder::coreDumpConfig))
            .constructor(ProductionVariantCoreDumpConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CoreDumpConfig").build()).build();

    private static final SdkField<ProductionVariantServerlessConfig> SERVERLESS_CONFIG_FIELD = SdkField
            .<ProductionVariantServerlessConfig> builder(MarshallingType.SDK_POJO).memberName("ServerlessConfig")
            .getter(getter(ProductionVariant::serverlessConfig)).setter(setter(Builder::serverlessConfig))
            .constructor(ProductionVariantServerlessConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerlessConfig").build()).build();

    private static final SdkField<Integer> VOLUME_SIZE_IN_GB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("VolumeSizeInGB").getter(getter(ProductionVariant::volumeSizeInGB))
            .setter(setter(Builder::volumeSizeInGB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeSizeInGB").build()).build();

    private static final SdkField<Integer> MODEL_DATA_DOWNLOAD_TIMEOUT_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("ModelDataDownloadTimeoutInSeconds")
            .getter(getter(ProductionVariant::modelDataDownloadTimeoutInSeconds))
            .setter(setter(Builder::modelDataDownloadTimeoutInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelDataDownloadTimeoutInSeconds")
                    .build()).build();

    private static final SdkField<Integer> CONTAINER_STARTUP_HEALTH_CHECK_TIMEOUT_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("ContainerStartupHealthCheckTimeoutInSeconds")
            .getter(getter(ProductionVariant::containerStartupHealthCheckTimeoutInSeconds))
            .setter(setter(Builder::containerStartupHealthCheckTimeoutInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("ContainerStartupHealthCheckTimeoutInSeconds").build()).build();

    private static final SdkField<Boolean> ENABLE_SSM_ACCESS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableSSMAccess").getter(getter(ProductionVariant::enableSSMAccess))
            .setter(setter(Builder::enableSSMAccess))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableSSMAccess").build()).build();

    private static final SdkField<ProductionVariantManagedInstanceScaling> MANAGED_INSTANCE_SCALING_FIELD = SdkField
            .<ProductionVariantManagedInstanceScaling> builder(MarshallingType.SDK_POJO).memberName("ManagedInstanceScaling")
            .getter(getter(ProductionVariant::managedInstanceScaling)).setter(setter(Builder::managedInstanceScaling))
            .constructor(ProductionVariantManagedInstanceScaling::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManagedInstanceScaling").build())
            .build();

    private static final SdkField<ProductionVariantRoutingConfig> ROUTING_CONFIG_FIELD = SdkField
            .<ProductionVariantRoutingConfig> builder(MarshallingType.SDK_POJO).memberName("RoutingConfig")
            .getter(getter(ProductionVariant::routingConfig)).setter(setter(Builder::routingConfig))
            .constructor(ProductionVariantRoutingConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoutingConfig").build()).build();

    private static final SdkField<String> INFERENCE_AMI_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InferenceAmiVersion").getter(getter(ProductionVariant::inferenceAmiVersionAsString))
            .setter(setter(Builder::inferenceAmiVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InferenceAmiVersion").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VARIANT_NAME_FIELD,
            MODEL_NAME_FIELD, INITIAL_INSTANCE_COUNT_FIELD, INSTANCE_TYPE_FIELD, INITIAL_VARIANT_WEIGHT_FIELD,
            ACCELERATOR_TYPE_FIELD, CORE_DUMP_CONFIG_FIELD, SERVERLESS_CONFIG_FIELD, VOLUME_SIZE_IN_GB_FIELD,
            MODEL_DATA_DOWNLOAD_TIMEOUT_IN_SECONDS_FIELD, CONTAINER_STARTUP_HEALTH_CHECK_TIMEOUT_IN_SECONDS_FIELD,
            ENABLE_SSM_ACCESS_FIELD, MANAGED_INSTANCE_SCALING_FIELD, ROUTING_CONFIG_FIELD, INFERENCE_AMI_VERSION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String variantName;

    private final String modelName;

    private final Integer initialInstanceCount;

    private final String instanceType;

    private final Float initialVariantWeight;

    private final String acceleratorType;

    private final ProductionVariantCoreDumpConfig coreDumpConfig;

    private final ProductionVariantServerlessConfig serverlessConfig;

    private final Integer volumeSizeInGB;

    private final Integer modelDataDownloadTimeoutInSeconds;

    private final Integer containerStartupHealthCheckTimeoutInSeconds;

    private final Boolean enableSSMAccess;

    private final ProductionVariantManagedInstanceScaling managedInstanceScaling;

    private final ProductionVariantRoutingConfig routingConfig;

    private final String inferenceAmiVersion;

    private ProductionVariant(BuilderImpl builder) {
        this.variantName = builder.variantName;
        this.modelName = builder.modelName;
        this.initialInstanceCount = builder.initialInstanceCount;
        this.instanceType = builder.instanceType;
        this.initialVariantWeight = builder.initialVariantWeight;
        this.acceleratorType = builder.acceleratorType;
        this.coreDumpConfig = builder.coreDumpConfig;
        this.serverlessConfig = builder.serverlessConfig;
        this.volumeSizeInGB = builder.volumeSizeInGB;
        this.modelDataDownloadTimeoutInSeconds = builder.modelDataDownloadTimeoutInSeconds;
        this.containerStartupHealthCheckTimeoutInSeconds = builder.containerStartupHealthCheckTimeoutInSeconds;
        this.enableSSMAccess = builder.enableSSMAccess;
        this.managedInstanceScaling = builder.managedInstanceScaling;
        this.routingConfig = builder.routingConfig;
        this.inferenceAmiVersion = builder.inferenceAmiVersion;
    }

    /**
     * <p>
     * The name of the production variant.
     * </p>
     * 
     * @return The name of the production variant.
     */
    public final String variantName() {
        return variantName;
    }

    /**
     * <p>
     * The name of the model that you want to host. This is the name that you specified when creating the model.
     * </p>
     * 
     * @return The name of the model that you want to host. This is the name that you specified when creating the model.
     */
    public final String modelName() {
        return modelName;
    }

    /**
     * <p>
     * Number of instances to launch initially.
     * </p>
     * 
     * @return Number of instances to launch initially.
     */
    public final Integer initialInstanceCount() {
        return initialInstanceCount;
    }

    /**
     * <p>
     * The ML compute instance type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link ProductionVariantInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The ML compute instance type.
     * @see ProductionVariantInstanceType
     */
    public final ProductionVariantInstanceType instanceType() {
        return ProductionVariantInstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The ML compute instance type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link ProductionVariantInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The ML compute instance type.
     * @see ProductionVariantInstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * Determines initial traffic distribution among all of the models that you specify in the endpoint configuration.
     * The traffic to a production variant is determined by the ratio of the <code>VariantWeight</code> to the sum of
     * all <code>VariantWeight</code> values across all ProductionVariants. If unspecified, it defaults to 1.0.
     * </p>
     * 
     * @return Determines initial traffic distribution among all of the models that you specify in the endpoint
     *         configuration. The traffic to a production variant is determined by the ratio of the
     *         <code>VariantWeight</code> to the sum of all <code>VariantWeight</code> values across all
     *         ProductionVariants. If unspecified, it defaults to 1.0.
     */
    public final Float initialVariantWeight() {
        return initialVariantWeight;
    }

    /**
     * <p>
     * This parameter is no longer supported. Elastic Inference (EI) is no longer available.
     * </p>
     * <p>
     * This parameter was used to specify the size of the EI instance to use for the production variant.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #acceleratorType}
     * will return {@link ProductionVariantAcceleratorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #acceleratorTypeAsString}.
     * </p>
     * 
     * @return This parameter is no longer supported. Elastic Inference (EI) is no longer available.</p>
     *         <p>
     *         This parameter was used to specify the size of the EI instance to use for the production variant.
     * @see ProductionVariantAcceleratorType
     */
    public final ProductionVariantAcceleratorType acceleratorType() {
        return ProductionVariantAcceleratorType.fromValue(acceleratorType);
    }

    /**
     * <p>
     * This parameter is no longer supported. Elastic Inference (EI) is no longer available.
     * </p>
     * <p>
     * This parameter was used to specify the size of the EI instance to use for the production variant.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #acceleratorType}
     * will return {@link ProductionVariantAcceleratorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #acceleratorTypeAsString}.
     * </p>
     * 
     * @return This parameter is no longer supported. Elastic Inference (EI) is no longer available.</p>
     *         <p>
     *         This parameter was used to specify the size of the EI instance to use for the production variant.
     * @see ProductionVariantAcceleratorType
     */
    public final String acceleratorTypeAsString() {
        return acceleratorType;
    }

    /**
     * <p>
     * Specifies configuration for a core dump from the model container when the process crashes.
     * </p>
     * 
     * @return Specifies configuration for a core dump from the model container when the process crashes.
     */
    public final ProductionVariantCoreDumpConfig coreDumpConfig() {
        return coreDumpConfig;
    }

    /**
     * <p>
     * The serverless configuration for an endpoint. Specifies a serverless endpoint configuration instead of an
     * instance-based endpoint configuration.
     * </p>
     * 
     * @return The serverless configuration for an endpoint. Specifies a serverless endpoint configuration instead of an
     *         instance-based endpoint configuration.
     */
    public final ProductionVariantServerlessConfig serverlessConfig() {
        return serverlessConfig;
    }

    /**
     * <p>
     * The size, in GB, of the ML storage volume attached to individual inference instance associated with the
     * production variant. Currently only Amazon EBS gp2 storage volumes are supported.
     * </p>
     * 
     * @return The size, in GB, of the ML storage volume attached to individual inference instance associated with the
     *         production variant. Currently only Amazon EBS gp2 storage volumes are supported.
     */
    public final Integer volumeSizeInGB() {
        return volumeSizeInGB;
    }

    /**
     * <p>
     * The timeout value, in seconds, to download and extract the model that you want to host from Amazon S3 to the
     * individual inference instance associated with this production variant.
     * </p>
     * 
     * @return The timeout value, in seconds, to download and extract the model that you want to host from Amazon S3 to
     *         the individual inference instance associated with this production variant.
     */
    public final Integer modelDataDownloadTimeoutInSeconds() {
        return modelDataDownloadTimeoutInSeconds;
    }

    /**
     * <p>
     * The timeout value, in seconds, for your inference container to pass health check by SageMaker Hosting. For more
     * information about health check, see <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-inference-code.html#your-algorithms-inference-algo-ping-requests"
     * >How Your Container Should Respond to Health Check (Ping) Requests</a>.
     * </p>
     * 
     * @return The timeout value, in seconds, for your inference container to pass health check by SageMaker Hosting.
     *         For more information about health check, see <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-inference-code.html#your-algorithms-inference-algo-ping-requests"
     *         >How Your Container Should Respond to Health Check (Ping) Requests</a>.
     */
    public final Integer containerStartupHealthCheckTimeoutInSeconds() {
        return containerStartupHealthCheckTimeoutInSeconds;
    }

    /**
     * <p>
     * You can use this parameter to turn on native Amazon Web Services Systems Manager (SSM) access for a production
     * variant behind an endpoint. By default, SSM access is disabled for all production variants behind an endpoint.
     * You can turn on or turn off SSM access for a production variant behind an existing endpoint by creating a new
     * endpoint configuration and calling <code>UpdateEndpoint</code>.
     * </p>
     * 
     * @return You can use this parameter to turn on native Amazon Web Services Systems Manager (SSM) access for a
     *         production variant behind an endpoint. By default, SSM access is disabled for all production variants
     *         behind an endpoint. You can turn on or turn off SSM access for a production variant behind an existing
     *         endpoint by creating a new endpoint configuration and calling <code>UpdateEndpoint</code>.
     */
    public final Boolean enableSSMAccess() {
        return enableSSMAccess;
    }

    /**
     * <p>
     * Settings that control the range in the number of instances that the endpoint provisions as it scales up or down
     * to accommodate traffic.
     * </p>
     * 
     * @return Settings that control the range in the number of instances that the endpoint provisions as it scales up
     *         or down to accommodate traffic.
     */
    public final ProductionVariantManagedInstanceScaling managedInstanceScaling() {
        return managedInstanceScaling;
    }

    /**
     * <p>
     * Settings that control how the endpoint routes incoming traffic to the instances that the endpoint hosts.
     * </p>
     * 
     * @return Settings that control how the endpoint routes incoming traffic to the instances that the endpoint hosts.
     */
    public final ProductionVariantRoutingConfig routingConfig() {
        return routingConfig;
    }

    /**
     * <p>
     * Specifies an option from a collection of preconfigured Amazon Machine Image (AMI) images. Each image is
     * configured by Amazon Web Services with a set of software and driver versions. Amazon Web Services optimizes these
     * configurations for different machine learning workloads.
     * </p>
     * <p>
     * By selecting an AMI version, you can ensure that your inference environment is compatible with specific software
     * requirements, such as CUDA driver versions, Linux kernel versions, or Amazon Web Services Neuron driver versions.
     * </p>
     * <p>
     * The AMI version names, and their configurations, are the following:
     * </p>
     * <dl>
     * <dt>al2-ami-sagemaker-inference-gpu-2</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * Accelerator: GPU
     * </p>
     * </li>
     * <li>
     * <p>
     * NVIDIA driver version: 535.54.03
     * </p>
     * </li>
     * <li>
     * <p>
     * CUDA driver version: 12.2
     * </p>
     * </li>
     * <li>
     * <p>
     * Supported instance types: ml.g4dn.*, ml.g5.*, ml.g6.*, ml.p3.*, ml.p4d.*, ml.p4de.*, ml.p5.*
     * </p>
     * </li>
     * </ul>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #inferenceAmiVersion} will return {@link ProductionVariantInferenceAmiVersion#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #inferenceAmiVersionAsString}.
     * </p>
     * 
     * @return Specifies an option from a collection of preconfigured Amazon Machine Image (AMI) images. Each image is
     *         configured by Amazon Web Services with a set of software and driver versions. Amazon Web Services
     *         optimizes these configurations for different machine learning workloads.</p>
     *         <p>
     *         By selecting an AMI version, you can ensure that your inference environment is compatible with specific
     *         software requirements, such as CUDA driver versions, Linux kernel versions, or Amazon Web Services Neuron
     *         driver versions.
     *         </p>
     *         <p>
     *         The AMI version names, and their configurations, are the following:
     *         </p>
     *         <dl>
     *         <dt>al2-ami-sagemaker-inference-gpu-2</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         Accelerator: GPU
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NVIDIA driver version: 535.54.03
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         CUDA driver version: 12.2
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Supported instance types: ml.g4dn.*, ml.g5.*, ml.g6.*, ml.p3.*, ml.p4d.*, ml.p4de.*, ml.p5.*
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     * @see ProductionVariantInferenceAmiVersion
     */
    public final ProductionVariantInferenceAmiVersion inferenceAmiVersion() {
        return ProductionVariantInferenceAmiVersion.fromValue(inferenceAmiVersion);
    }

    /**
     * <p>
     * Specifies an option from a collection of preconfigured Amazon Machine Image (AMI) images. Each image is
     * configured by Amazon Web Services with a set of software and driver versions. Amazon Web Services optimizes these
     * configurations for different machine learning workloads.
     * </p>
     * <p>
     * By selecting an AMI version, you can ensure that your inference environment is compatible with specific software
     * requirements, such as CUDA driver versions, Linux kernel versions, or Amazon Web Services Neuron driver versions.
     * </p>
     * <p>
     * The AMI version names, and their configurations, are the following:
     * </p>
     * <dl>
     * <dt>al2-ami-sagemaker-inference-gpu-2</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * Accelerator: GPU
     * </p>
     * </li>
     * <li>
     * <p>
     * NVIDIA driver version: 535.54.03
     * </p>
     * </li>
     * <li>
     * <p>
     * CUDA driver version: 12.2
     * </p>
     * </li>
     * <li>
     * <p>
     * Supported instance types: ml.g4dn.*, ml.g5.*, ml.g6.*, ml.p3.*, ml.p4d.*, ml.p4de.*, ml.p5.*
     * </p>
     * </li>
     * </ul>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #inferenceAmiVersion} will return {@link ProductionVariantInferenceAmiVersion#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #inferenceAmiVersionAsString}.
     * </p>
     * 
     * @return Specifies an option from a collection of preconfigured Amazon Machine Image (AMI) images. Each image is
     *         configured by Amazon Web Services with a set of software and driver versions. Amazon Web Services
     *         optimizes these configurations for different machine learning workloads.</p>
     *         <p>
     *         By selecting an AMI version, you can ensure that your inference environment is compatible with specific
     *         software requirements, such as CUDA driver versions, Linux kernel versions, or Amazon Web Services Neuron
     *         driver versions.
     *         </p>
     *         <p>
     *         The AMI version names, and their configurations, are the following:
     *         </p>
     *         <dl>
     *         <dt>al2-ami-sagemaker-inference-gpu-2</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         Accelerator: GPU
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NVIDIA driver version: 535.54.03
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         CUDA driver version: 12.2
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Supported instance types: ml.g4dn.*, ml.g5.*, ml.g6.*, ml.p3.*, ml.p4d.*, ml.p4de.*, ml.p5.*
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     * @see ProductionVariantInferenceAmiVersion
     */
    public final String inferenceAmiVersionAsString() {
        return inferenceAmiVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(variantName());
        hashCode = 31 * hashCode + Objects.hashCode(modelName());
        hashCode = 31 * hashCode + Objects.hashCode(initialInstanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(initialVariantWeight());
        hashCode = 31 * hashCode + Objects.hashCode(acceleratorTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(coreDumpConfig());
        hashCode = 31 * hashCode + Objects.hashCode(serverlessConfig());
        hashCode = 31 * hashCode + Objects.hashCode(volumeSizeInGB());
        hashCode = 31 * hashCode + Objects.hashCode(modelDataDownloadTimeoutInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(containerStartupHealthCheckTimeoutInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(enableSSMAccess());
        hashCode = 31 * hashCode + Objects.hashCode(managedInstanceScaling());
        hashCode = 31 * hashCode + Objects.hashCode(routingConfig());
        hashCode = 31 * hashCode + Objects.hashCode(inferenceAmiVersionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProductionVariant)) {
            return false;
        }
        ProductionVariant other = (ProductionVariant) obj;
        return Objects.equals(variantName(), other.variantName())
                && Objects.equals(modelName(), other.modelName())
                && Objects.equals(initialInstanceCount(), other.initialInstanceCount())
                && Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(initialVariantWeight(), other.initialVariantWeight())
                && Objects.equals(acceleratorTypeAsString(), other.acceleratorTypeAsString())
                && Objects.equals(coreDumpConfig(), other.coreDumpConfig())
                && Objects.equals(serverlessConfig(), other.serverlessConfig())
                && Objects.equals(volumeSizeInGB(), other.volumeSizeInGB())
                && Objects.equals(modelDataDownloadTimeoutInSeconds(), other.modelDataDownloadTimeoutInSeconds())
                && Objects.equals(containerStartupHealthCheckTimeoutInSeconds(),
                        other.containerStartupHealthCheckTimeoutInSeconds())
                && Objects.equals(enableSSMAccess(), other.enableSSMAccess())
                && Objects.equals(managedInstanceScaling(), other.managedInstanceScaling())
                && Objects.equals(routingConfig(), other.routingConfig())
                && Objects.equals(inferenceAmiVersionAsString(), other.inferenceAmiVersionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProductionVariant").add("VariantName", variantName()).add("ModelName", modelName())
                .add("InitialInstanceCount", initialInstanceCount()).add("InstanceType", instanceTypeAsString())
                .add("InitialVariantWeight", initialVariantWeight()).add("AcceleratorType", acceleratorTypeAsString())
                .add("CoreDumpConfig", coreDumpConfig()).add("ServerlessConfig", serverlessConfig())
                .add("VolumeSizeInGB", volumeSizeInGB())
                .add("ModelDataDownloadTimeoutInSeconds", modelDataDownloadTimeoutInSeconds())
                .add("ContainerStartupHealthCheckTimeoutInSeconds", containerStartupHealthCheckTimeoutInSeconds())
                .add("EnableSSMAccess", enableSSMAccess()).add("ManagedInstanceScaling", managedInstanceScaling())
                .add("RoutingConfig", routingConfig()).add("InferenceAmiVersion", inferenceAmiVersionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VariantName":
            return Optional.ofNullable(clazz.cast(variantName()));
        case "ModelName":
            return Optional.ofNullable(clazz.cast(modelName()));
        case "InitialInstanceCount":
            return Optional.ofNullable(clazz.cast(initialInstanceCount()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "InitialVariantWeight":
            return Optional.ofNullable(clazz.cast(initialVariantWeight()));
        case "AcceleratorType":
            return Optional.ofNullable(clazz.cast(acceleratorTypeAsString()));
        case "CoreDumpConfig":
            return Optional.ofNullable(clazz.cast(coreDumpConfig()));
        case "ServerlessConfig":
            return Optional.ofNullable(clazz.cast(serverlessConfig()));
        case "VolumeSizeInGB":
            return Optional.ofNullable(clazz.cast(volumeSizeInGB()));
        case "ModelDataDownloadTimeoutInSeconds":
            return Optional.ofNullable(clazz.cast(modelDataDownloadTimeoutInSeconds()));
        case "ContainerStartupHealthCheckTimeoutInSeconds":
            return Optional.ofNullable(clazz.cast(containerStartupHealthCheckTimeoutInSeconds()));
        case "EnableSSMAccess":
            return Optional.ofNullable(clazz.cast(enableSSMAccess()));
        case "ManagedInstanceScaling":
            return Optional.ofNullable(clazz.cast(managedInstanceScaling()));
        case "RoutingConfig":
            return Optional.ofNullable(clazz.cast(routingConfig()));
        case "InferenceAmiVersion":
            return Optional.ofNullable(clazz.cast(inferenceAmiVersionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("VariantName", VARIANT_NAME_FIELD);
        map.put("ModelName", MODEL_NAME_FIELD);
        map.put("InitialInstanceCount", INITIAL_INSTANCE_COUNT_FIELD);
        map.put("InstanceType", INSTANCE_TYPE_FIELD);
        map.put("InitialVariantWeight", INITIAL_VARIANT_WEIGHT_FIELD);
        map.put("AcceleratorType", ACCELERATOR_TYPE_FIELD);
        map.put("CoreDumpConfig", CORE_DUMP_CONFIG_FIELD);
        map.put("ServerlessConfig", SERVERLESS_CONFIG_FIELD);
        map.put("VolumeSizeInGB", VOLUME_SIZE_IN_GB_FIELD);
        map.put("ModelDataDownloadTimeoutInSeconds", MODEL_DATA_DOWNLOAD_TIMEOUT_IN_SECONDS_FIELD);
        map.put("ContainerStartupHealthCheckTimeoutInSeconds", CONTAINER_STARTUP_HEALTH_CHECK_TIMEOUT_IN_SECONDS_FIELD);
        map.put("EnableSSMAccess", ENABLE_SSM_ACCESS_FIELD);
        map.put("ManagedInstanceScaling", MANAGED_INSTANCE_SCALING_FIELD);
        map.put("RoutingConfig", ROUTING_CONFIG_FIELD);
        map.put("InferenceAmiVersion", INFERENCE_AMI_VERSION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ProductionVariant, T> g) {
        return obj -> g.apply((ProductionVariant) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProductionVariant> {
        /**
         * <p>
         * The name of the production variant.
         * </p>
         * 
         * @param variantName
         *        The name of the production variant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variantName(String variantName);

        /**
         * <p>
         * The name of the model that you want to host. This is the name that you specified when creating the model.
         * </p>
         * 
         * @param modelName
         *        The name of the model that you want to host. This is the name that you specified when creating the
         *        model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelName(String modelName);

        /**
         * <p>
         * Number of instances to launch initially.
         * </p>
         * 
         * @param initialInstanceCount
         *        Number of instances to launch initially.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initialInstanceCount(Integer initialInstanceCount);

        /**
         * <p>
         * The ML compute instance type.
         * </p>
         * 
         * @param instanceType
         *        The ML compute instance type.
         * @see ProductionVariantInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantInstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The ML compute instance type.
         * </p>
         * 
         * @param instanceType
         *        The ML compute instance type.
         * @see ProductionVariantInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantInstanceType
         */
        Builder instanceType(ProductionVariantInstanceType instanceType);

        /**
         * <p>
         * Determines initial traffic distribution among all of the models that you specify in the endpoint
         * configuration. The traffic to a production variant is determined by the ratio of the
         * <code>VariantWeight</code> to the sum of all <code>VariantWeight</code> values across all ProductionVariants.
         * If unspecified, it defaults to 1.0.
         * </p>
         * 
         * @param initialVariantWeight
         *        Determines initial traffic distribution among all of the models that you specify in the endpoint
         *        configuration. The traffic to a production variant is determined by the ratio of the
         *        <code>VariantWeight</code> to the sum of all <code>VariantWeight</code> values across all
         *        ProductionVariants. If unspecified, it defaults to 1.0.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initialVariantWeight(Float initialVariantWeight);

        /**
         * <p>
         * This parameter is no longer supported. Elastic Inference (EI) is no longer available.
         * </p>
         * <p>
         * This parameter was used to specify the size of the EI instance to use for the production variant.
         * </p>
         * 
         * @param acceleratorType
         *        This parameter is no longer supported. Elastic Inference (EI) is no longer available.</p>
         *        <p>
         *        This parameter was used to specify the size of the EI instance to use for the production variant.
         * @see ProductionVariantAcceleratorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantAcceleratorType
         */
        Builder acceleratorType(String acceleratorType);

        /**
         * <p>
         * This parameter is no longer supported. Elastic Inference (EI) is no longer available.
         * </p>
         * <p>
         * This parameter was used to specify the size of the EI instance to use for the production variant.
         * </p>
         * 
         * @param acceleratorType
         *        This parameter is no longer supported. Elastic Inference (EI) is no longer available.</p>
         *        <p>
         *        This parameter was used to specify the size of the EI instance to use for the production variant.
         * @see ProductionVariantAcceleratorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantAcceleratorType
         */
        Builder acceleratorType(ProductionVariantAcceleratorType acceleratorType);

        /**
         * <p>
         * Specifies configuration for a core dump from the model container when the process crashes.
         * </p>
         * 
         * @param coreDumpConfig
         *        Specifies configuration for a core dump from the model container when the process crashes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder coreDumpConfig(ProductionVariantCoreDumpConfig coreDumpConfig);

        /**
         * <p>
         * Specifies configuration for a core dump from the model container when the process crashes.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProductionVariantCoreDumpConfig.Builder}
         * avoiding the need to create one manually via {@link ProductionVariantCoreDumpConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ProductionVariantCoreDumpConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #coreDumpConfig(ProductionVariantCoreDumpConfig)}.
         * 
         * @param coreDumpConfig
         *        a consumer that will call methods on {@link ProductionVariantCoreDumpConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #coreDumpConfig(ProductionVariantCoreDumpConfig)
         */
        default Builder coreDumpConfig(Consumer<ProductionVariantCoreDumpConfig.Builder> coreDumpConfig) {
            return coreDumpConfig(ProductionVariantCoreDumpConfig.builder().applyMutation(coreDumpConfig).build());
        }

        /**
         * <p>
         * The serverless configuration for an endpoint. Specifies a serverless endpoint configuration instead of an
         * instance-based endpoint configuration.
         * </p>
         * 
         * @param serverlessConfig
         *        The serverless configuration for an endpoint. Specifies a serverless endpoint configuration instead of
         *        an instance-based endpoint configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverlessConfig(ProductionVariantServerlessConfig serverlessConfig);

        /**
         * <p>
         * The serverless configuration for an endpoint. Specifies a serverless endpoint configuration instead of an
         * instance-based endpoint configuration.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ProductionVariantServerlessConfig.Builder} avoiding the need to create one manually via
         * {@link ProductionVariantServerlessConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ProductionVariantServerlessConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #serverlessConfig(ProductionVariantServerlessConfig)}.
         * 
         * @param serverlessConfig
         *        a consumer that will call methods on {@link ProductionVariantServerlessConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serverlessConfig(ProductionVariantServerlessConfig)
         */
        default Builder serverlessConfig(Consumer<ProductionVariantServerlessConfig.Builder> serverlessConfig) {
            return serverlessConfig(ProductionVariantServerlessConfig.builder().applyMutation(serverlessConfig).build());
        }

        /**
         * <p>
         * The size, in GB, of the ML storage volume attached to individual inference instance associated with the
         * production variant. Currently only Amazon EBS gp2 storage volumes are supported.
         * </p>
         * 
         * @param volumeSizeInGB
         *        The size, in GB, of the ML storage volume attached to individual inference instance associated with
         *        the production variant. Currently only Amazon EBS gp2 storage volumes are supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeSizeInGB(Integer volumeSizeInGB);

        /**
         * <p>
         * The timeout value, in seconds, to download and extract the model that you want to host from Amazon S3 to the
         * individual inference instance associated with this production variant.
         * </p>
         * 
         * @param modelDataDownloadTimeoutInSeconds
         *        The timeout value, in seconds, to download and extract the model that you want to host from Amazon S3
         *        to the individual inference instance associated with this production variant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelDataDownloadTimeoutInSeconds(Integer modelDataDownloadTimeoutInSeconds);

        /**
         * <p>
         * The timeout value, in seconds, for your inference container to pass health check by SageMaker Hosting. For
         * more information about health check, see <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-inference-code.html#your-algorithms-inference-algo-ping-requests"
         * >How Your Container Should Respond to Health Check (Ping) Requests</a>.
         * </p>
         * 
         * @param containerStartupHealthCheckTimeoutInSeconds
         *        The timeout value, in seconds, for your inference container to pass health check by SageMaker Hosting.
         *        For more information about health check, see <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-inference-code.html#your-algorithms-inference-algo-ping-requests"
         *        >How Your Container Should Respond to Health Check (Ping) Requests</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerStartupHealthCheckTimeoutInSeconds(Integer containerStartupHealthCheckTimeoutInSeconds);

        /**
         * <p>
         * You can use this parameter to turn on native Amazon Web Services Systems Manager (SSM) access for a
         * production variant behind an endpoint. By default, SSM access is disabled for all production variants behind
         * an endpoint. You can turn on or turn off SSM access for a production variant behind an existing endpoint by
         * creating a new endpoint configuration and calling <code>UpdateEndpoint</code>.
         * </p>
         * 
         * @param enableSSMAccess
         *        You can use this parameter to turn on native Amazon Web Services Systems Manager (SSM) access for a
         *        production variant behind an endpoint. By default, SSM access is disabled for all production variants
         *        behind an endpoint. You can turn on or turn off SSM access for a production variant behind an existing
         *        endpoint by creating a new endpoint configuration and calling <code>UpdateEndpoint</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableSSMAccess(Boolean enableSSMAccess);

        /**
         * <p>
         * Settings that control the range in the number of instances that the endpoint provisions as it scales up or
         * down to accommodate traffic.
         * </p>
         * 
         * @param managedInstanceScaling
         *        Settings that control the range in the number of instances that the endpoint provisions as it scales
         *        up or down to accommodate traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedInstanceScaling(ProductionVariantManagedInstanceScaling managedInstanceScaling);

        /**
         * <p>
         * Settings that control the range in the number of instances that the endpoint provisions as it scales up or
         * down to accommodate traffic.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ProductionVariantManagedInstanceScaling.Builder} avoiding the need to create one manually via
         * {@link ProductionVariantManagedInstanceScaling#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ProductionVariantManagedInstanceScaling.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #managedInstanceScaling(ProductionVariantManagedInstanceScaling)}.
         * 
         * @param managedInstanceScaling
         *        a consumer that will call methods on {@link ProductionVariantManagedInstanceScaling.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #managedInstanceScaling(ProductionVariantManagedInstanceScaling)
         */
        default Builder managedInstanceScaling(Consumer<ProductionVariantManagedInstanceScaling.Builder> managedInstanceScaling) {
            return managedInstanceScaling(ProductionVariantManagedInstanceScaling.builder().applyMutation(managedInstanceScaling)
                    .build());
        }

        /**
         * <p>
         * Settings that control how the endpoint routes incoming traffic to the instances that the endpoint hosts.
         * </p>
         * 
         * @param routingConfig
         *        Settings that control how the endpoint routes incoming traffic to the instances that the endpoint
         *        hosts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routingConfig(ProductionVariantRoutingConfig routingConfig);

        /**
         * <p>
         * Settings that control how the endpoint routes incoming traffic to the instances that the endpoint hosts.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProductionVariantRoutingConfig.Builder}
         * avoiding the need to create one manually via {@link ProductionVariantRoutingConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ProductionVariantRoutingConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #routingConfig(ProductionVariantRoutingConfig)}.
         * 
         * @param routingConfig
         *        a consumer that will call methods on {@link ProductionVariantRoutingConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #routingConfig(ProductionVariantRoutingConfig)
         */
        default Builder routingConfig(Consumer<ProductionVariantRoutingConfig.Builder> routingConfig) {
            return routingConfig(ProductionVariantRoutingConfig.builder().applyMutation(routingConfig).build());
        }

        /**
         * <p>
         * Specifies an option from a collection of preconfigured Amazon Machine Image (AMI) images. Each image is
         * configured by Amazon Web Services with a set of software and driver versions. Amazon Web Services optimizes
         * these configurations for different machine learning workloads.
         * </p>
         * <p>
         * By selecting an AMI version, you can ensure that your inference environment is compatible with specific
         * software requirements, such as CUDA driver versions, Linux kernel versions, or Amazon Web Services Neuron
         * driver versions.
         * </p>
         * <p>
         * The AMI version names, and their configurations, are the following:
         * </p>
         * <dl>
         * <dt>al2-ami-sagemaker-inference-gpu-2</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * Accelerator: GPU
         * </p>
         * </li>
         * <li>
         * <p>
         * NVIDIA driver version: 535.54.03
         * </p>
         * </li>
         * <li>
         * <p>
         * CUDA driver version: 12.2
         * </p>
         * </li>
         * <li>
         * <p>
         * Supported instance types: ml.g4dn.*, ml.g5.*, ml.g6.*, ml.p3.*, ml.p4d.*, ml.p4de.*, ml.p5.*
         * </p>
         * </li>
         * </ul>
         * </dd>
         * </dl>
         * 
         * @param inferenceAmiVersion
         *        Specifies an option from a collection of preconfigured Amazon Machine Image (AMI) images. Each image
         *        is configured by Amazon Web Services with a set of software and driver versions. Amazon Web Services
         *        optimizes these configurations for different machine learning workloads.</p>
         *        <p>
         *        By selecting an AMI version, you can ensure that your inference environment is compatible with
         *        specific software requirements, such as CUDA driver versions, Linux kernel versions, or Amazon Web
         *        Services Neuron driver versions.
         *        </p>
         *        <p>
         *        The AMI version names, and their configurations, are the following:
         *        </p>
         *        <dl>
         *        <dt>al2-ami-sagemaker-inference-gpu-2</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        Accelerator: GPU
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NVIDIA driver version: 535.54.03
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        CUDA driver version: 12.2
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Supported instance types: ml.g4dn.*, ml.g5.*, ml.g6.*, ml.p3.*, ml.p4d.*, ml.p4de.*, ml.p5.*
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         * @see ProductionVariantInferenceAmiVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantInferenceAmiVersion
         */
        Builder inferenceAmiVersion(String inferenceAmiVersion);

        /**
         * <p>
         * Specifies an option from a collection of preconfigured Amazon Machine Image (AMI) images. Each image is
         * configured by Amazon Web Services with a set of software and driver versions. Amazon Web Services optimizes
         * these configurations for different machine learning workloads.
         * </p>
         * <p>
         * By selecting an AMI version, you can ensure that your inference environment is compatible with specific
         * software requirements, such as CUDA driver versions, Linux kernel versions, or Amazon Web Services Neuron
         * driver versions.
         * </p>
         * <p>
         * The AMI version names, and their configurations, are the following:
         * </p>
         * <dl>
         * <dt>al2-ami-sagemaker-inference-gpu-2</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * Accelerator: GPU
         * </p>
         * </li>
         * <li>
         * <p>
         * NVIDIA driver version: 535.54.03
         * </p>
         * </li>
         * <li>
         * <p>
         * CUDA driver version: 12.2
         * </p>
         * </li>
         * <li>
         * <p>
         * Supported instance types: ml.g4dn.*, ml.g5.*, ml.g6.*, ml.p3.*, ml.p4d.*, ml.p4de.*, ml.p5.*
         * </p>
         * </li>
         * </ul>
         * </dd>
         * </dl>
         * 
         * @param inferenceAmiVersion
         *        Specifies an option from a collection of preconfigured Amazon Machine Image (AMI) images. Each image
         *        is configured by Amazon Web Services with a set of software and driver versions. Amazon Web Services
         *        optimizes these configurations for different machine learning workloads.</p>
         *        <p>
         *        By selecting an AMI version, you can ensure that your inference environment is compatible with
         *        specific software requirements, such as CUDA driver versions, Linux kernel versions, or Amazon Web
         *        Services Neuron driver versions.
         *        </p>
         *        <p>
         *        The AMI version names, and their configurations, are the following:
         *        </p>
         *        <dl>
         *        <dt>al2-ami-sagemaker-inference-gpu-2</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        Accelerator: GPU
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NVIDIA driver version: 535.54.03
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        CUDA driver version: 12.2
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Supported instance types: ml.g4dn.*, ml.g5.*, ml.g6.*, ml.p3.*, ml.p4d.*, ml.p4de.*, ml.p5.*
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         * @see ProductionVariantInferenceAmiVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantInferenceAmiVersion
         */
        Builder inferenceAmiVersion(ProductionVariantInferenceAmiVersion inferenceAmiVersion);
    }

    static final class BuilderImpl implements Builder {
        private String variantName;

        private String modelName;

        private Integer initialInstanceCount;

        private String instanceType;

        private Float initialVariantWeight;

        private String acceleratorType;

        private ProductionVariantCoreDumpConfig coreDumpConfig;

        private ProductionVariantServerlessConfig serverlessConfig;

        private Integer volumeSizeInGB;

        private Integer modelDataDownloadTimeoutInSeconds;

        private Integer containerStartupHealthCheckTimeoutInSeconds;

        private Boolean enableSSMAccess;

        private ProductionVariantManagedInstanceScaling managedInstanceScaling;

        private ProductionVariantRoutingConfig routingConfig;

        private String inferenceAmiVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(ProductionVariant model) {
            variantName(model.variantName);
            modelName(model.modelName);
            initialInstanceCount(model.initialInstanceCount);
            instanceType(model.instanceType);
            initialVariantWeight(model.initialVariantWeight);
            acceleratorType(model.acceleratorType);
            coreDumpConfig(model.coreDumpConfig);
            serverlessConfig(model.serverlessConfig);
            volumeSizeInGB(model.volumeSizeInGB);
            modelDataDownloadTimeoutInSeconds(model.modelDataDownloadTimeoutInSeconds);
            containerStartupHealthCheckTimeoutInSeconds(model.containerStartupHealthCheckTimeoutInSeconds);
            enableSSMAccess(model.enableSSMAccess);
            managedInstanceScaling(model.managedInstanceScaling);
            routingConfig(model.routingConfig);
            inferenceAmiVersion(model.inferenceAmiVersion);
        }

        public final String getVariantName() {
            return variantName;
        }

        public final void setVariantName(String variantName) {
            this.variantName = variantName;
        }

        @Override
        public final Builder variantName(String variantName) {
            this.variantName = variantName;
            return this;
        }

        public final String getModelName() {
            return modelName;
        }

        public final void setModelName(String modelName) {
            this.modelName = modelName;
        }

        @Override
        public final Builder modelName(String modelName) {
            this.modelName = modelName;
            return this;
        }

        public final Integer getInitialInstanceCount() {
            return initialInstanceCount;
        }

        public final void setInitialInstanceCount(Integer initialInstanceCount) {
            this.initialInstanceCount = initialInstanceCount;
        }

        @Override
        public final Builder initialInstanceCount(Integer initialInstanceCount) {
            this.initialInstanceCount = initialInstanceCount;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(ProductionVariantInstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final Float getInitialVariantWeight() {
            return initialVariantWeight;
        }

        public final void setInitialVariantWeight(Float initialVariantWeight) {
            this.initialVariantWeight = initialVariantWeight;
        }

        @Override
        public final Builder initialVariantWeight(Float initialVariantWeight) {
            this.initialVariantWeight = initialVariantWeight;
            return this;
        }

        public final String getAcceleratorType() {
            return acceleratorType;
        }

        public final void setAcceleratorType(String acceleratorType) {
            this.acceleratorType = acceleratorType;
        }

        @Override
        public final Builder acceleratorType(String acceleratorType) {
            this.acceleratorType = acceleratorType;
            return this;
        }

        @Override
        public final Builder acceleratorType(ProductionVariantAcceleratorType acceleratorType) {
            this.acceleratorType(acceleratorType == null ? null : acceleratorType.toString());
            return this;
        }

        public final ProductionVariantCoreDumpConfig.Builder getCoreDumpConfig() {
            return coreDumpConfig != null ? coreDumpConfig.toBuilder() : null;
        }

        public final void setCoreDumpConfig(ProductionVariantCoreDumpConfig.BuilderImpl coreDumpConfig) {
            this.coreDumpConfig = coreDumpConfig != null ? coreDumpConfig.build() : null;
        }

        @Override
        public final Builder coreDumpConfig(ProductionVariantCoreDumpConfig coreDumpConfig) {
            this.coreDumpConfig = coreDumpConfig;
            return this;
        }

        public final ProductionVariantServerlessConfig.Builder getServerlessConfig() {
            return serverlessConfig != null ? serverlessConfig.toBuilder() : null;
        }

        public final void setServerlessConfig(ProductionVariantServerlessConfig.BuilderImpl serverlessConfig) {
            this.serverlessConfig = serverlessConfig != null ? serverlessConfig.build() : null;
        }

        @Override
        public final Builder serverlessConfig(ProductionVariantServerlessConfig serverlessConfig) {
            this.serverlessConfig = serverlessConfig;
            return this;
        }

        public final Integer getVolumeSizeInGB() {
            return volumeSizeInGB;
        }

        public final void setVolumeSizeInGB(Integer volumeSizeInGB) {
            this.volumeSizeInGB = volumeSizeInGB;
        }

        @Override
        public final Builder volumeSizeInGB(Integer volumeSizeInGB) {
            this.volumeSizeInGB = volumeSizeInGB;
            return this;
        }

        public final Integer getModelDataDownloadTimeoutInSeconds() {
            return modelDataDownloadTimeoutInSeconds;
        }

        public final void setModelDataDownloadTimeoutInSeconds(Integer modelDataDownloadTimeoutInSeconds) {
            this.modelDataDownloadTimeoutInSeconds = modelDataDownloadTimeoutInSeconds;
        }

        @Override
        public final Builder modelDataDownloadTimeoutInSeconds(Integer modelDataDownloadTimeoutInSeconds) {
            this.modelDataDownloadTimeoutInSeconds = modelDataDownloadTimeoutInSeconds;
            return this;
        }

        public final Integer getContainerStartupHealthCheckTimeoutInSeconds() {
            return containerStartupHealthCheckTimeoutInSeconds;
        }

        public final void setContainerStartupHealthCheckTimeoutInSeconds(Integer containerStartupHealthCheckTimeoutInSeconds) {
            this.containerStartupHealthCheckTimeoutInSeconds = containerStartupHealthCheckTimeoutInSeconds;
        }

        @Override
        public final Builder containerStartupHealthCheckTimeoutInSeconds(Integer containerStartupHealthCheckTimeoutInSeconds) {
            this.containerStartupHealthCheckTimeoutInSeconds = containerStartupHealthCheckTimeoutInSeconds;
            return this;
        }

        public final Boolean getEnableSSMAccess() {
            return enableSSMAccess;
        }

        public final void setEnableSSMAccess(Boolean enableSSMAccess) {
            this.enableSSMAccess = enableSSMAccess;
        }

        @Override
        public final Builder enableSSMAccess(Boolean enableSSMAccess) {
            this.enableSSMAccess = enableSSMAccess;
            return this;
        }

        public final ProductionVariantManagedInstanceScaling.Builder getManagedInstanceScaling() {
            return managedInstanceScaling != null ? managedInstanceScaling.toBuilder() : null;
        }

        public final void setManagedInstanceScaling(ProductionVariantManagedInstanceScaling.BuilderImpl managedInstanceScaling) {
            this.managedInstanceScaling = managedInstanceScaling != null ? managedInstanceScaling.build() : null;
        }

        @Override
        public final Builder managedInstanceScaling(ProductionVariantManagedInstanceScaling managedInstanceScaling) {
            this.managedInstanceScaling = managedInstanceScaling;
            return this;
        }

        public final ProductionVariantRoutingConfig.Builder getRoutingConfig() {
            return routingConfig != null ? routingConfig.toBuilder() : null;
        }

        public final void setRoutingConfig(ProductionVariantRoutingConfig.BuilderImpl routingConfig) {
            this.routingConfig = routingConfig != null ? routingConfig.build() : null;
        }

        @Override
        public final Builder routingConfig(ProductionVariantRoutingConfig routingConfig) {
            this.routingConfig = routingConfig;
            return this;
        }

        public final String getInferenceAmiVersion() {
            return inferenceAmiVersion;
        }

        public final void setInferenceAmiVersion(String inferenceAmiVersion) {
            this.inferenceAmiVersion = inferenceAmiVersion;
        }

        @Override
        public final Builder inferenceAmiVersion(String inferenceAmiVersion) {
            this.inferenceAmiVersion = inferenceAmiVersion;
            return this;
        }

        @Override
        public final Builder inferenceAmiVersion(ProductionVariantInferenceAmiVersion inferenceAmiVersion) {
            this.inferenceAmiVersion(inferenceAmiVersion == null ? null : inferenceAmiVersion.toString());
            return this;
        }

        @Override
        public ProductionVariant build() {
            return new ProductionVariant(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
