/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The collection of settings used by an AutoML job V2 for the time-series forecasting problem type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TimeSeriesForecastingJobConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<TimeSeriesForecastingJobConfig.Builder, TimeSeriesForecastingJobConfig> {
    private static final SdkField<String> FEATURE_SPECIFICATION_S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FeatureSpecificationS3Uri").getter(getter(TimeSeriesForecastingJobConfig::featureSpecificationS3Uri))
            .setter(setter(Builder::featureSpecificationS3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeatureSpecificationS3Uri").build())
            .build();

    private static final SdkField<AutoMLJobCompletionCriteria> COMPLETION_CRITERIA_FIELD = SdkField
            .<AutoMLJobCompletionCriteria> builder(MarshallingType.SDK_POJO).memberName("CompletionCriteria")
            .getter(getter(TimeSeriesForecastingJobConfig::completionCriteria)).setter(setter(Builder::completionCriteria))
            .constructor(AutoMLJobCompletionCriteria::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionCriteria").build())
            .build();

    private static final SdkField<String> FORECAST_FREQUENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ForecastFrequency").getter(getter(TimeSeriesForecastingJobConfig::forecastFrequency))
            .setter(setter(Builder::forecastFrequency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForecastFrequency").build()).build();

    private static final SdkField<Integer> FORECAST_HORIZON_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ForecastHorizon").getter(getter(TimeSeriesForecastingJobConfig::forecastHorizon))
            .setter(setter(Builder::forecastHorizon))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForecastHorizon").build()).build();

    private static final SdkField<List<String>> FORECAST_QUANTILES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ForecastQuantiles")
            .getter(getter(TimeSeriesForecastingJobConfig::forecastQuantiles))
            .setter(setter(Builder::forecastQuantiles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForecastQuantiles").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<TimeSeriesTransformations> TRANSFORMATIONS_FIELD = SdkField
            .<TimeSeriesTransformations> builder(MarshallingType.SDK_POJO).memberName("Transformations")
            .getter(getter(TimeSeriesForecastingJobConfig::transformations)).setter(setter(Builder::transformations))
            .constructor(TimeSeriesTransformations::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Transformations").build()).build();

    private static final SdkField<TimeSeriesConfig> TIME_SERIES_CONFIG_FIELD = SdkField
            .<TimeSeriesConfig> builder(MarshallingType.SDK_POJO).memberName("TimeSeriesConfig")
            .getter(getter(TimeSeriesForecastingJobConfig::timeSeriesConfig)).setter(setter(Builder::timeSeriesConfig))
            .constructor(TimeSeriesConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeSeriesConfig").build()).build();

    private static final SdkField<List<HolidayConfigAttributes>> HOLIDAY_CONFIG_FIELD = SdkField
            .<List<HolidayConfigAttributes>> builder(MarshallingType.LIST)
            .memberName("HolidayConfig")
            .getter(getter(TimeSeriesForecastingJobConfig::holidayConfig))
            .setter(setter(Builder::holidayConfig))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HolidayConfig").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HolidayConfigAttributes> builder(MarshallingType.SDK_POJO)
                                            .constructor(HolidayConfigAttributes::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<CandidateGenerationConfig> CANDIDATE_GENERATION_CONFIG_FIELD = SdkField
            .<CandidateGenerationConfig> builder(MarshallingType.SDK_POJO).memberName("CandidateGenerationConfig")
            .getter(getter(TimeSeriesForecastingJobConfig::candidateGenerationConfig))
            .setter(setter(Builder::candidateGenerationConfig)).constructor(CandidateGenerationConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CandidateGenerationConfig").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            FEATURE_SPECIFICATION_S3_URI_FIELD, COMPLETION_CRITERIA_FIELD, FORECAST_FREQUENCY_FIELD, FORECAST_HORIZON_FIELD,
            FORECAST_QUANTILES_FIELD, TRANSFORMATIONS_FIELD, TIME_SERIES_CONFIG_FIELD, HOLIDAY_CONFIG_FIELD,
            CANDIDATE_GENERATION_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String featureSpecificationS3Uri;

    private final AutoMLJobCompletionCriteria completionCriteria;

    private final String forecastFrequency;

    private final Integer forecastHorizon;

    private final List<String> forecastQuantiles;

    private final TimeSeriesTransformations transformations;

    private final TimeSeriesConfig timeSeriesConfig;

    private final List<HolidayConfigAttributes> holidayConfig;

    private final CandidateGenerationConfig candidateGenerationConfig;

    private TimeSeriesForecastingJobConfig(BuilderImpl builder) {
        this.featureSpecificationS3Uri = builder.featureSpecificationS3Uri;
        this.completionCriteria = builder.completionCriteria;
        this.forecastFrequency = builder.forecastFrequency;
        this.forecastHorizon = builder.forecastHorizon;
        this.forecastQuantiles = builder.forecastQuantiles;
        this.transformations = builder.transformations;
        this.timeSeriesConfig = builder.timeSeriesConfig;
        this.holidayConfig = builder.holidayConfig;
        this.candidateGenerationConfig = builder.candidateGenerationConfig;
    }

    /**
     * <p>
     * A URL to the Amazon S3 data source containing additional selected features that complement the target, itemID,
     * timestamp, and grouped columns set in <code>TimeSeriesConfig</code>. When not provided, the AutoML job V2
     * includes all the columns from the original dataset that are not already declared in <code>TimeSeriesConfig</code>
     * . If provided, the AutoML job V2 only considers these additional columns as a complement to the ones declared in
     * <code>TimeSeriesConfig</code>.
     * </p>
     * <p>
     * You can input <code>FeatureAttributeNames</code> (optional) in JSON format as shown below:
     * </p>
     * <p>
     * <code>{ "FeatureAttributeNames":["col1", "col2", ...] }</code>.
     * </p>
     * <p>
     * You can also specify the data type of the feature (optional) in the format shown below:
     * </p>
     * <p>
     * <code>{ "FeatureDataTypes":{"col1":"numeric", "col2":"categorical" ... } }</code>
     * </p>
     * <p>
     * Autopilot supports the following data types: <code>numeric</code>, <code>categorical</code>, <code>text</code>,
     * and <code>datetime</code>.
     * </p>
     * <note>
     * <p>
     * These column keys must not include any column set in <code>TimeSeriesConfig</code>.
     * </p>
     * </note>
     * 
     * @return A URL to the Amazon S3 data source containing additional selected features that complement the target,
     *         itemID, timestamp, and grouped columns set in <code>TimeSeriesConfig</code>. When not provided, the
     *         AutoML job V2 includes all the columns from the original dataset that are not already declared in
     *         <code>TimeSeriesConfig</code>. If provided, the AutoML job V2 only considers these additional columns as
     *         a complement to the ones declared in <code>TimeSeriesConfig</code>.</p>
     *         <p>
     *         You can input <code>FeatureAttributeNames</code> (optional) in JSON format as shown below:
     *         </p>
     *         <p>
     *         <code>{ "FeatureAttributeNames":["col1", "col2", ...] }</code>.
     *         </p>
     *         <p>
     *         You can also specify the data type of the feature (optional) in the format shown below:
     *         </p>
     *         <p>
     *         <code>{ "FeatureDataTypes":{"col1":"numeric", "col2":"categorical" ... } }</code>
     *         </p>
     *         <p>
     *         Autopilot supports the following data types: <code>numeric</code>, <code>categorical</code>,
     *         <code>text</code>, and <code>datetime</code>.
     *         </p>
     *         <note>
     *         <p>
     *         These column keys must not include any column set in <code>TimeSeriesConfig</code>.
     *         </p>
     */
    public final String featureSpecificationS3Uri() {
        return featureSpecificationS3Uri;
    }

    /**
     * Returns the value of the CompletionCriteria property for this object.
     * 
     * @return The value of the CompletionCriteria property for this object.
     */
    public final AutoMLJobCompletionCriteria completionCriteria() {
        return completionCriteria;
    }

    /**
     * <p>
     * The frequency of predictions in a forecast.
     * </p>
     * <p>
     * Valid intervals are an integer followed by Y (Year), M (Month), W (Week), D (Day), H (Hour), and min (Minute).
     * For example, <code>1D</code> indicates every day and <code>15min</code> indicates every 15 minutes. The value of
     * a frequency must not overlap with the next larger frequency. For example, you must use a frequency of
     * <code>1H</code> instead of <code>60min</code>.
     * </p>
     * <p>
     * The valid values for each frequency are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Minute - 1-59
     * </p>
     * </li>
     * <li>
     * <p>
     * Hour - 1-23
     * </p>
     * </li>
     * <li>
     * <p>
     * Day - 1-6
     * </p>
     * </li>
     * <li>
     * <p>
     * Week - 1-4
     * </p>
     * </li>
     * <li>
     * <p>
     * Month - 1-11
     * </p>
     * </li>
     * <li>
     * <p>
     * Year - 1
     * </p>
     * </li>
     * </ul>
     * 
     * @return The frequency of predictions in a forecast.</p>
     *         <p>
     *         Valid intervals are an integer followed by Y (Year), M (Month), W (Week), D (Day), H (Hour), and min
     *         (Minute). For example, <code>1D</code> indicates every day and <code>15min</code> indicates every 15
     *         minutes. The value of a frequency must not overlap with the next larger frequency. For example, you must
     *         use a frequency of <code>1H</code> instead of <code>60min</code>.
     *         </p>
     *         <p>
     *         The valid values for each frequency are the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Minute - 1-59
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Hour - 1-23
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Day - 1-6
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Week - 1-4
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Month - 1-11
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Year - 1
     *         </p>
     *         </li>
     */
    public final String forecastFrequency() {
        return forecastFrequency;
    }

    /**
     * <p>
     * The number of time-steps that the model predicts. The forecast horizon is also called the prediction length. The
     * maximum forecast horizon is the lesser of 500 time-steps or 1/4 of the time-steps in the dataset.
     * </p>
     * 
     * @return The number of time-steps that the model predicts. The forecast horizon is also called the prediction
     *         length. The maximum forecast horizon is the lesser of 500 time-steps or 1/4 of the time-steps in the
     *         dataset.
     */
    public final Integer forecastHorizon() {
        return forecastHorizon;
    }

    /**
     * For responses, this returns true if the service returned a value for the ForecastQuantiles property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasForecastQuantiles() {
        return forecastQuantiles != null && !(forecastQuantiles instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The quantiles used to train the model for forecasts at a specified quantile. You can specify quantiles from
     * <code>0.01</code> (p1) to <code>0.99</code> (p99), by increments of 0.01 or higher. Up to five forecast quantiles
     * can be specified. When <code>ForecastQuantiles</code> is not provided, the AutoML job uses the quantiles p10,
     * p50, and p90 as default.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasForecastQuantiles} method.
     * </p>
     * 
     * @return The quantiles used to train the model for forecasts at a specified quantile. You can specify quantiles
     *         from <code>0.01</code> (p1) to <code>0.99</code> (p99), by increments of 0.01 or higher. Up to five
     *         forecast quantiles can be specified. When <code>ForecastQuantiles</code> is not provided, the AutoML job
     *         uses the quantiles p10, p50, and p90 as default.
     */
    public final List<String> forecastQuantiles() {
        return forecastQuantiles;
    }

    /**
     * <p>
     * The transformations modifying specific attributes of the time-series, such as filling strategies for missing
     * values.
     * </p>
     * 
     * @return The transformations modifying specific attributes of the time-series, such as filling strategies for
     *         missing values.
     */
    public final TimeSeriesTransformations transformations() {
        return transformations;
    }

    /**
     * <p>
     * The collection of components that defines the time-series.
     * </p>
     * 
     * @return The collection of components that defines the time-series.
     */
    public final TimeSeriesConfig timeSeriesConfig() {
        return timeSeriesConfig;
    }

    /**
     * For responses, this returns true if the service returned a value for the HolidayConfig property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasHolidayConfig() {
        return holidayConfig != null && !(holidayConfig instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The collection of holiday featurization attributes used to incorporate national holiday information into your
     * forecasting model.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHolidayConfig} method.
     * </p>
     * 
     * @return The collection of holiday featurization attributes used to incorporate national holiday information into
     *         your forecasting model.
     */
    public final List<HolidayConfigAttributes> holidayConfig() {
        return holidayConfig;
    }

    /**
     * Returns the value of the CandidateGenerationConfig property for this object.
     * 
     * @return The value of the CandidateGenerationConfig property for this object.
     */
    public final CandidateGenerationConfig candidateGenerationConfig() {
        return candidateGenerationConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(featureSpecificationS3Uri());
        hashCode = 31 * hashCode + Objects.hashCode(completionCriteria());
        hashCode = 31 * hashCode + Objects.hashCode(forecastFrequency());
        hashCode = 31 * hashCode + Objects.hashCode(forecastHorizon());
        hashCode = 31 * hashCode + Objects.hashCode(hasForecastQuantiles() ? forecastQuantiles() : null);
        hashCode = 31 * hashCode + Objects.hashCode(transformations());
        hashCode = 31 * hashCode + Objects.hashCode(timeSeriesConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasHolidayConfig() ? holidayConfig() : null);
        hashCode = 31 * hashCode + Objects.hashCode(candidateGenerationConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TimeSeriesForecastingJobConfig)) {
            return false;
        }
        TimeSeriesForecastingJobConfig other = (TimeSeriesForecastingJobConfig) obj;
        return Objects.equals(featureSpecificationS3Uri(), other.featureSpecificationS3Uri())
                && Objects.equals(completionCriteria(), other.completionCriteria())
                && Objects.equals(forecastFrequency(), other.forecastFrequency())
                && Objects.equals(forecastHorizon(), other.forecastHorizon())
                && hasForecastQuantiles() == other.hasForecastQuantiles()
                && Objects.equals(forecastQuantiles(), other.forecastQuantiles())
                && Objects.equals(transformations(), other.transformations())
                && Objects.equals(timeSeriesConfig(), other.timeSeriesConfig()) && hasHolidayConfig() == other.hasHolidayConfig()
                && Objects.equals(holidayConfig(), other.holidayConfig())
                && Objects.equals(candidateGenerationConfig(), other.candidateGenerationConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TimeSeriesForecastingJobConfig").add("FeatureSpecificationS3Uri", featureSpecificationS3Uri())
                .add("CompletionCriteria", completionCriteria()).add("ForecastFrequency", forecastFrequency())
                .add("ForecastHorizon", forecastHorizon())
                .add("ForecastQuantiles", hasForecastQuantiles() ? forecastQuantiles() : null)
                .add("Transformations", transformations()).add("TimeSeriesConfig", timeSeriesConfig())
                .add("HolidayConfig", hasHolidayConfig() ? holidayConfig() : null)
                .add("CandidateGenerationConfig", candidateGenerationConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FeatureSpecificationS3Uri":
            return Optional.ofNullable(clazz.cast(featureSpecificationS3Uri()));
        case "CompletionCriteria":
            return Optional.ofNullable(clazz.cast(completionCriteria()));
        case "ForecastFrequency":
            return Optional.ofNullable(clazz.cast(forecastFrequency()));
        case "ForecastHorizon":
            return Optional.ofNullable(clazz.cast(forecastHorizon()));
        case "ForecastQuantiles":
            return Optional.ofNullable(clazz.cast(forecastQuantiles()));
        case "Transformations":
            return Optional.ofNullable(clazz.cast(transformations()));
        case "TimeSeriesConfig":
            return Optional.ofNullable(clazz.cast(timeSeriesConfig()));
        case "HolidayConfig":
            return Optional.ofNullable(clazz.cast(holidayConfig()));
        case "CandidateGenerationConfig":
            return Optional.ofNullable(clazz.cast(candidateGenerationConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("FeatureSpecificationS3Uri", FEATURE_SPECIFICATION_S3_URI_FIELD);
        map.put("CompletionCriteria", COMPLETION_CRITERIA_FIELD);
        map.put("ForecastFrequency", FORECAST_FREQUENCY_FIELD);
        map.put("ForecastHorizon", FORECAST_HORIZON_FIELD);
        map.put("ForecastQuantiles", FORECAST_QUANTILES_FIELD);
        map.put("Transformations", TRANSFORMATIONS_FIELD);
        map.put("TimeSeriesConfig", TIME_SERIES_CONFIG_FIELD);
        map.put("HolidayConfig", HOLIDAY_CONFIG_FIELD);
        map.put("CandidateGenerationConfig", CANDIDATE_GENERATION_CONFIG_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TimeSeriesForecastingJobConfig, T> g) {
        return obj -> g.apply((TimeSeriesForecastingJobConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TimeSeriesForecastingJobConfig> {
        /**
         * <p>
         * A URL to the Amazon S3 data source containing additional selected features that complement the target,
         * itemID, timestamp, and grouped columns set in <code>TimeSeriesConfig</code>. When not provided, the AutoML
         * job V2 includes all the columns from the original dataset that are not already declared in
         * <code>TimeSeriesConfig</code>. If provided, the AutoML job V2 only considers these additional columns as a
         * complement to the ones declared in <code>TimeSeriesConfig</code>.
         * </p>
         * <p>
         * You can input <code>FeatureAttributeNames</code> (optional) in JSON format as shown below:
         * </p>
         * <p>
         * <code>{ "FeatureAttributeNames":["col1", "col2", ...] }</code>.
         * </p>
         * <p>
         * You can also specify the data type of the feature (optional) in the format shown below:
         * </p>
         * <p>
         * <code>{ "FeatureDataTypes":{"col1":"numeric", "col2":"categorical" ... } }</code>
         * </p>
         * <p>
         * Autopilot supports the following data types: <code>numeric</code>, <code>categorical</code>,
         * <code>text</code>, and <code>datetime</code>.
         * </p>
         * <note>
         * <p>
         * These column keys must not include any column set in <code>TimeSeriesConfig</code>.
         * </p>
         * </note>
         * 
         * @param featureSpecificationS3Uri
         *        A URL to the Amazon S3 data source containing additional selected features that complement the target,
         *        itemID, timestamp, and grouped columns set in <code>TimeSeriesConfig</code>. When not provided, the
         *        AutoML job V2 includes all the columns from the original dataset that are not already declared in
         *        <code>TimeSeriesConfig</code>. If provided, the AutoML job V2 only considers these additional columns
         *        as a complement to the ones declared in <code>TimeSeriesConfig</code>.</p>
         *        <p>
         *        You can input <code>FeatureAttributeNames</code> (optional) in JSON format as shown below:
         *        </p>
         *        <p>
         *        <code>{ "FeatureAttributeNames":["col1", "col2", ...] }</code>.
         *        </p>
         *        <p>
         *        You can also specify the data type of the feature (optional) in the format shown below:
         *        </p>
         *        <p>
         *        <code>{ "FeatureDataTypes":{"col1":"numeric", "col2":"categorical" ... } }</code>
         *        </p>
         *        <p>
         *        Autopilot supports the following data types: <code>numeric</code>, <code>categorical</code>,
         *        <code>text</code>, and <code>datetime</code>.
         *        </p>
         *        <note>
         *        <p>
         *        These column keys must not include any column set in <code>TimeSeriesConfig</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureSpecificationS3Uri(String featureSpecificationS3Uri);

        /**
         * Sets the value of the CompletionCriteria property for this object.
         *
         * @param completionCriteria
         *        The new value for the CompletionCriteria property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionCriteria(AutoMLJobCompletionCriteria completionCriteria);

        /**
         * Sets the value of the CompletionCriteria property for this object.
         *
         * This is a convenience method that creates an instance of the {@link AutoMLJobCompletionCriteria.Builder}
         * avoiding the need to create one manually via {@link AutoMLJobCompletionCriteria#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoMLJobCompletionCriteria.Builder#build()} is called
         * immediately and its result is passed to {@link #completionCriteria(AutoMLJobCompletionCriteria)}.
         * 
         * @param completionCriteria
         *        a consumer that will call methods on {@link AutoMLJobCompletionCriteria.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #completionCriteria(AutoMLJobCompletionCriteria)
         */
        default Builder completionCriteria(Consumer<AutoMLJobCompletionCriteria.Builder> completionCriteria) {
            return completionCriteria(AutoMLJobCompletionCriteria.builder().applyMutation(completionCriteria).build());
        }

        /**
         * <p>
         * The frequency of predictions in a forecast.
         * </p>
         * <p>
         * Valid intervals are an integer followed by Y (Year), M (Month), W (Week), D (Day), H (Hour), and min
         * (Minute). For example, <code>1D</code> indicates every day and <code>15min</code> indicates every 15 minutes.
         * The value of a frequency must not overlap with the next larger frequency. For example, you must use a
         * frequency of <code>1H</code> instead of <code>60min</code>.
         * </p>
         * <p>
         * The valid values for each frequency are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Minute - 1-59
         * </p>
         * </li>
         * <li>
         * <p>
         * Hour - 1-23
         * </p>
         * </li>
         * <li>
         * <p>
         * Day - 1-6
         * </p>
         * </li>
         * <li>
         * <p>
         * Week - 1-4
         * </p>
         * </li>
         * <li>
         * <p>
         * Month - 1-11
         * </p>
         * </li>
         * <li>
         * <p>
         * Year - 1
         * </p>
         * </li>
         * </ul>
         * 
         * @param forecastFrequency
         *        The frequency of predictions in a forecast.</p>
         *        <p>
         *        Valid intervals are an integer followed by Y (Year), M (Month), W (Week), D (Day), H (Hour), and min
         *        (Minute). For example, <code>1D</code> indicates every day and <code>15min</code> indicates every 15
         *        minutes. The value of a frequency must not overlap with the next larger frequency. For example, you
         *        must use a frequency of <code>1H</code> instead of <code>60min</code>.
         *        </p>
         *        <p>
         *        The valid values for each frequency are the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Minute - 1-59
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Hour - 1-23
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Day - 1-6
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Week - 1-4
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Month - 1-11
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Year - 1
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastFrequency(String forecastFrequency);

        /**
         * <p>
         * The number of time-steps that the model predicts. The forecast horizon is also called the prediction length.
         * The maximum forecast horizon is the lesser of 500 time-steps or 1/4 of the time-steps in the dataset.
         * </p>
         * 
         * @param forecastHorizon
         *        The number of time-steps that the model predicts. The forecast horizon is also called the prediction
         *        length. The maximum forecast horizon is the lesser of 500 time-steps or 1/4 of the time-steps in the
         *        dataset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastHorizon(Integer forecastHorizon);

        /**
         * <p>
         * The quantiles used to train the model for forecasts at a specified quantile. You can specify quantiles from
         * <code>0.01</code> (p1) to <code>0.99</code> (p99), by increments of 0.01 or higher. Up to five forecast
         * quantiles can be specified. When <code>ForecastQuantiles</code> is not provided, the AutoML job uses the
         * quantiles p10, p50, and p90 as default.
         * </p>
         * 
         * @param forecastQuantiles
         *        The quantiles used to train the model for forecasts at a specified quantile. You can specify quantiles
         *        from <code>0.01</code> (p1) to <code>0.99</code> (p99), by increments of 0.01 or higher. Up to five
         *        forecast quantiles can be specified. When <code>ForecastQuantiles</code> is not provided, the AutoML
         *        job uses the quantiles p10, p50, and p90 as default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastQuantiles(Collection<String> forecastQuantiles);

        /**
         * <p>
         * The quantiles used to train the model for forecasts at a specified quantile. You can specify quantiles from
         * <code>0.01</code> (p1) to <code>0.99</code> (p99), by increments of 0.01 or higher. Up to five forecast
         * quantiles can be specified. When <code>ForecastQuantiles</code> is not provided, the AutoML job uses the
         * quantiles p10, p50, and p90 as default.
         * </p>
         * 
         * @param forecastQuantiles
         *        The quantiles used to train the model for forecasts at a specified quantile. You can specify quantiles
         *        from <code>0.01</code> (p1) to <code>0.99</code> (p99), by increments of 0.01 or higher. Up to five
         *        forecast quantiles can be specified. When <code>ForecastQuantiles</code> is not provided, the AutoML
         *        job uses the quantiles p10, p50, and p90 as default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastQuantiles(String... forecastQuantiles);

        /**
         * <p>
         * The transformations modifying specific attributes of the time-series, such as filling strategies for missing
         * values.
         * </p>
         * 
         * @param transformations
         *        The transformations modifying specific attributes of the time-series, such as filling strategies for
         *        missing values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transformations(TimeSeriesTransformations transformations);

        /**
         * <p>
         * The transformations modifying specific attributes of the time-series, such as filling strategies for missing
         * values.
         * </p>
         * This is a convenience method that creates an instance of the {@link TimeSeriesTransformations.Builder}
         * avoiding the need to create one manually via {@link TimeSeriesTransformations#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TimeSeriesTransformations.Builder#build()} is called immediately
         * and its result is passed to {@link #transformations(TimeSeriesTransformations)}.
         * 
         * @param transformations
         *        a consumer that will call methods on {@link TimeSeriesTransformations.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transformations(TimeSeriesTransformations)
         */
        default Builder transformations(Consumer<TimeSeriesTransformations.Builder> transformations) {
            return transformations(TimeSeriesTransformations.builder().applyMutation(transformations).build());
        }

        /**
         * <p>
         * The collection of components that defines the time-series.
         * </p>
         * 
         * @param timeSeriesConfig
         *        The collection of components that defines the time-series.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeSeriesConfig(TimeSeriesConfig timeSeriesConfig);

        /**
         * <p>
         * The collection of components that defines the time-series.
         * </p>
         * This is a convenience method that creates an instance of the {@link TimeSeriesConfig.Builder} avoiding the
         * need to create one manually via {@link TimeSeriesConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TimeSeriesConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #timeSeriesConfig(TimeSeriesConfig)}.
         * 
         * @param timeSeriesConfig
         *        a consumer that will call methods on {@link TimeSeriesConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #timeSeriesConfig(TimeSeriesConfig)
         */
        default Builder timeSeriesConfig(Consumer<TimeSeriesConfig.Builder> timeSeriesConfig) {
            return timeSeriesConfig(TimeSeriesConfig.builder().applyMutation(timeSeriesConfig).build());
        }

        /**
         * <p>
         * The collection of holiday featurization attributes used to incorporate national holiday information into your
         * forecasting model.
         * </p>
         * 
         * @param holidayConfig
         *        The collection of holiday featurization attributes used to incorporate national holiday information
         *        into your forecasting model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder holidayConfig(Collection<HolidayConfigAttributes> holidayConfig);

        /**
         * <p>
         * The collection of holiday featurization attributes used to incorporate national holiday information into your
         * forecasting model.
         * </p>
         * 
         * @param holidayConfig
         *        The collection of holiday featurization attributes used to incorporate national holiday information
         *        into your forecasting model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder holidayConfig(HolidayConfigAttributes... holidayConfig);

        /**
         * <p>
         * The collection of holiday featurization attributes used to incorporate national holiday information into your
         * forecasting model.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.HolidayConfigAttributes.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.sagemaker.model.HolidayConfigAttributes#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.HolidayConfigAttributes.Builder#build()} is called
         * immediately and its result is passed to {@link #holidayConfig(List<HolidayConfigAttributes>)}.
         * 
         * @param holidayConfig
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.HolidayConfigAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #holidayConfig(java.util.Collection<HolidayConfigAttributes>)
         */
        Builder holidayConfig(Consumer<HolidayConfigAttributes.Builder>... holidayConfig);

        /**
         * Sets the value of the CandidateGenerationConfig property for this object.
         *
         * @param candidateGenerationConfig
         *        The new value for the CandidateGenerationConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder candidateGenerationConfig(CandidateGenerationConfig candidateGenerationConfig);

        /**
         * Sets the value of the CandidateGenerationConfig property for this object.
         *
         * This is a convenience method that creates an instance of the {@link CandidateGenerationConfig.Builder}
         * avoiding the need to create one manually via {@link CandidateGenerationConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CandidateGenerationConfig.Builder#build()} is called immediately
         * and its result is passed to {@link #candidateGenerationConfig(CandidateGenerationConfig)}.
         * 
         * @param candidateGenerationConfig
         *        a consumer that will call methods on {@link CandidateGenerationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #candidateGenerationConfig(CandidateGenerationConfig)
         */
        default Builder candidateGenerationConfig(Consumer<CandidateGenerationConfig.Builder> candidateGenerationConfig) {
            return candidateGenerationConfig(CandidateGenerationConfig.builder().applyMutation(candidateGenerationConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String featureSpecificationS3Uri;

        private AutoMLJobCompletionCriteria completionCriteria;

        private String forecastFrequency;

        private Integer forecastHorizon;

        private List<String> forecastQuantiles = DefaultSdkAutoConstructList.getInstance();

        private TimeSeriesTransformations transformations;

        private TimeSeriesConfig timeSeriesConfig;

        private List<HolidayConfigAttributes> holidayConfig = DefaultSdkAutoConstructList.getInstance();

        private CandidateGenerationConfig candidateGenerationConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(TimeSeriesForecastingJobConfig model) {
            featureSpecificationS3Uri(model.featureSpecificationS3Uri);
            completionCriteria(model.completionCriteria);
            forecastFrequency(model.forecastFrequency);
            forecastHorizon(model.forecastHorizon);
            forecastQuantiles(model.forecastQuantiles);
            transformations(model.transformations);
            timeSeriesConfig(model.timeSeriesConfig);
            holidayConfig(model.holidayConfig);
            candidateGenerationConfig(model.candidateGenerationConfig);
        }

        public final String getFeatureSpecificationS3Uri() {
            return featureSpecificationS3Uri;
        }

        public final void setFeatureSpecificationS3Uri(String featureSpecificationS3Uri) {
            this.featureSpecificationS3Uri = featureSpecificationS3Uri;
        }

        @Override
        public final Builder featureSpecificationS3Uri(String featureSpecificationS3Uri) {
            this.featureSpecificationS3Uri = featureSpecificationS3Uri;
            return this;
        }

        public final AutoMLJobCompletionCriteria.Builder getCompletionCriteria() {
            return completionCriteria != null ? completionCriteria.toBuilder() : null;
        }

        public final void setCompletionCriteria(AutoMLJobCompletionCriteria.BuilderImpl completionCriteria) {
            this.completionCriteria = completionCriteria != null ? completionCriteria.build() : null;
        }

        @Override
        public final Builder completionCriteria(AutoMLJobCompletionCriteria completionCriteria) {
            this.completionCriteria = completionCriteria;
            return this;
        }

        public final String getForecastFrequency() {
            return forecastFrequency;
        }

        public final void setForecastFrequency(String forecastFrequency) {
            this.forecastFrequency = forecastFrequency;
        }

        @Override
        public final Builder forecastFrequency(String forecastFrequency) {
            this.forecastFrequency = forecastFrequency;
            return this;
        }

        public final Integer getForecastHorizon() {
            return forecastHorizon;
        }

        public final void setForecastHorizon(Integer forecastHorizon) {
            this.forecastHorizon = forecastHorizon;
        }

        @Override
        public final Builder forecastHorizon(Integer forecastHorizon) {
            this.forecastHorizon = forecastHorizon;
            return this;
        }

        public final Collection<String> getForecastQuantiles() {
            if (forecastQuantiles instanceof SdkAutoConstructList) {
                return null;
            }
            return forecastQuantiles;
        }

        public final void setForecastQuantiles(Collection<String> forecastQuantiles) {
            this.forecastQuantiles = ForecastQuantilesCopier.copy(forecastQuantiles);
        }

        @Override
        public final Builder forecastQuantiles(Collection<String> forecastQuantiles) {
            this.forecastQuantiles = ForecastQuantilesCopier.copy(forecastQuantiles);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder forecastQuantiles(String... forecastQuantiles) {
            forecastQuantiles(Arrays.asList(forecastQuantiles));
            return this;
        }

        public final TimeSeriesTransformations.Builder getTransformations() {
            return transformations != null ? transformations.toBuilder() : null;
        }

        public final void setTransformations(TimeSeriesTransformations.BuilderImpl transformations) {
            this.transformations = transformations != null ? transformations.build() : null;
        }

        @Override
        public final Builder transformations(TimeSeriesTransformations transformations) {
            this.transformations = transformations;
            return this;
        }

        public final TimeSeriesConfig.Builder getTimeSeriesConfig() {
            return timeSeriesConfig != null ? timeSeriesConfig.toBuilder() : null;
        }

        public final void setTimeSeriesConfig(TimeSeriesConfig.BuilderImpl timeSeriesConfig) {
            this.timeSeriesConfig = timeSeriesConfig != null ? timeSeriesConfig.build() : null;
        }

        @Override
        public final Builder timeSeriesConfig(TimeSeriesConfig timeSeriesConfig) {
            this.timeSeriesConfig = timeSeriesConfig;
            return this;
        }

        public final List<HolidayConfigAttributes.Builder> getHolidayConfig() {
            List<HolidayConfigAttributes.Builder> result = HolidayConfigCopier.copyToBuilder(this.holidayConfig);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setHolidayConfig(Collection<HolidayConfigAttributes.BuilderImpl> holidayConfig) {
            this.holidayConfig = HolidayConfigCopier.copyFromBuilder(holidayConfig);
        }

        @Override
        public final Builder holidayConfig(Collection<HolidayConfigAttributes> holidayConfig) {
            this.holidayConfig = HolidayConfigCopier.copy(holidayConfig);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder holidayConfig(HolidayConfigAttributes... holidayConfig) {
            holidayConfig(Arrays.asList(holidayConfig));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder holidayConfig(Consumer<HolidayConfigAttributes.Builder>... holidayConfig) {
            holidayConfig(Stream.of(holidayConfig).map(c -> HolidayConfigAttributes.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final CandidateGenerationConfig.Builder getCandidateGenerationConfig() {
            return candidateGenerationConfig != null ? candidateGenerationConfig.toBuilder() : null;
        }

        public final void setCandidateGenerationConfig(CandidateGenerationConfig.BuilderImpl candidateGenerationConfig) {
            this.candidateGenerationConfig = candidateGenerationConfig != null ? candidateGenerationConfig.build() : null;
        }

        @Override
        public final Builder candidateGenerationConfig(CandidateGenerationConfig candidateGenerationConfig) {
            this.candidateGenerationConfig = candidateGenerationConfig;
            return this;
        }

        @Override
        public TimeSeriesForecastingJobConfig build() {
            return new TimeSeriesForecastingJobConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
