/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about a training plan offering.
 * </p>
 * <p>
 * For more information about how to reserve GPU capacity for your SageMaker HyperPod clusters using Amazon SageMaker
 * Training Plan, see
 * <code> <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateTrainingPlan.html">CreateTrainingPlan</a> </code>
 * .
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TrainingPlanOffering implements SdkPojo, Serializable,
        ToCopyableBuilder<TrainingPlanOffering.Builder, TrainingPlanOffering> {
    private static final SdkField<String> TRAINING_PLAN_OFFERING_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TrainingPlanOfferingId").getter(getter(TrainingPlanOffering::trainingPlanOfferingId))
            .setter(setter(Builder::trainingPlanOfferingId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingPlanOfferingId").build())
            .build();

    private static final SdkField<List<String>> TARGET_RESOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TargetResources")
            .getter(getter(TrainingPlanOffering::targetResourcesAsStrings))
            .setter(setter(Builder::targetResourcesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> REQUESTED_START_TIME_AFTER_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("RequestedStartTimeAfter").getter(getter(TrainingPlanOffering::requestedStartTimeAfter))
            .setter(setter(Builder::requestedStartTimeAfter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestedStartTimeAfter").build())
            .build();

    private static final SdkField<Instant> REQUESTED_END_TIME_BEFORE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("RequestedEndTimeBefore").getter(getter(TrainingPlanOffering::requestedEndTimeBefore))
            .setter(setter(Builder::requestedEndTimeBefore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestedEndTimeBefore").build())
            .build();

    private static final SdkField<Long> DURATION_HOURS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("DurationHours").getter(getter(TrainingPlanOffering::durationHours))
            .setter(setter(Builder::durationHours))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DurationHours").build()).build();

    private static final SdkField<Long> DURATION_MINUTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("DurationMinutes").getter(getter(TrainingPlanOffering::durationMinutes))
            .setter(setter(Builder::durationMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DurationMinutes").build()).build();

    private static final SdkField<String> UPFRONT_FEE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UpfrontFee").getter(getter(TrainingPlanOffering::upfrontFee)).setter(setter(Builder::upfrontFee))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpfrontFee").build()).build();

    private static final SdkField<String> CURRENCY_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CurrencyCode").getter(getter(TrainingPlanOffering::currencyCode)).setter(setter(Builder::currencyCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CurrencyCode").build()).build();

    private static final SdkField<List<ReservedCapacityOffering>> RESERVED_CAPACITY_OFFERINGS_FIELD = SdkField
            .<List<ReservedCapacityOffering>> builder(MarshallingType.LIST)
            .memberName("ReservedCapacityOfferings")
            .getter(getter(TrainingPlanOffering::reservedCapacityOfferings))
            .setter(setter(Builder::reservedCapacityOfferings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservedCapacityOfferings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ReservedCapacityOffering> builder(MarshallingType.SDK_POJO)
                                            .constructor(ReservedCapacityOffering::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            TRAINING_PLAN_OFFERING_ID_FIELD, TARGET_RESOURCES_FIELD, REQUESTED_START_TIME_AFTER_FIELD,
            REQUESTED_END_TIME_BEFORE_FIELD, DURATION_HOURS_FIELD, DURATION_MINUTES_FIELD, UPFRONT_FEE_FIELD,
            CURRENCY_CODE_FIELD, RESERVED_CAPACITY_OFFERINGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String trainingPlanOfferingId;

    private final List<String> targetResources;

    private final Instant requestedStartTimeAfter;

    private final Instant requestedEndTimeBefore;

    private final Long durationHours;

    private final Long durationMinutes;

    private final String upfrontFee;

    private final String currencyCode;

    private final List<ReservedCapacityOffering> reservedCapacityOfferings;

    private TrainingPlanOffering(BuilderImpl builder) {
        this.trainingPlanOfferingId = builder.trainingPlanOfferingId;
        this.targetResources = builder.targetResources;
        this.requestedStartTimeAfter = builder.requestedStartTimeAfter;
        this.requestedEndTimeBefore = builder.requestedEndTimeBefore;
        this.durationHours = builder.durationHours;
        this.durationMinutes = builder.durationMinutes;
        this.upfrontFee = builder.upfrontFee;
        this.currencyCode = builder.currencyCode;
        this.reservedCapacityOfferings = builder.reservedCapacityOfferings;
    }

    /**
     * <p>
     * The unique identifier for this training plan offering.
     * </p>
     * 
     * @return The unique identifier for this training plan offering.
     */
    public final String trainingPlanOfferingId() {
        return trainingPlanOfferingId;
    }

    /**
     * <p>
     * The target resources (e.g., SageMaker Training Jobs, SageMaker HyperPod) for this training plan offering.
     * </p>
     * <p>
     * Training plans are specific to their target resource.
     * </p>
     * <ul>
     * <li>
     * <p>
     * A training plan designed for SageMaker training jobs can only be used to schedule and run training jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * A training plan for HyperPod clusters can be used exclusively to provide compute resources to a cluster's
     * instance group.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargetResources} method.
     * </p>
     * 
     * @return The target resources (e.g., SageMaker Training Jobs, SageMaker HyperPod) for this training plan
     *         offering.</p>
     *         <p>
     *         Training plans are specific to their target resource.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A training plan designed for SageMaker training jobs can only be used to schedule and run training jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A training plan for HyperPod clusters can be used exclusively to provide compute resources to a cluster's
     *         instance group.
     *         </p>
     *         </li>
     */
    public final List<SageMakerResourceName> targetResources() {
        return SageMakerResourceNamesCopier.copyStringToEnum(targetResources);
    }

    /**
     * For responses, this returns true if the service returned a value for the TargetResources property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTargetResources() {
        return targetResources != null && !(targetResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The target resources (e.g., SageMaker Training Jobs, SageMaker HyperPod) for this training plan offering.
     * </p>
     * <p>
     * Training plans are specific to their target resource.
     * </p>
     * <ul>
     * <li>
     * <p>
     * A training plan designed for SageMaker training jobs can only be used to schedule and run training jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * A training plan for HyperPod clusters can be used exclusively to provide compute resources to a cluster's
     * instance group.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargetResources} method.
     * </p>
     * 
     * @return The target resources (e.g., SageMaker Training Jobs, SageMaker HyperPod) for this training plan
     *         offering.</p>
     *         <p>
     *         Training plans are specific to their target resource.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A training plan designed for SageMaker training jobs can only be used to schedule and run training jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A training plan for HyperPod clusters can be used exclusively to provide compute resources to a cluster's
     *         instance group.
     *         </p>
     *         </li>
     */
    public final List<String> targetResourcesAsStrings() {
        return targetResources;
    }

    /**
     * <p>
     * The requested start time that the user specified when searching for the training plan offering.
     * </p>
     * 
     * @return The requested start time that the user specified when searching for the training plan offering.
     */
    public final Instant requestedStartTimeAfter() {
        return requestedStartTimeAfter;
    }

    /**
     * <p>
     * The requested end time that the user specified when searching for the training plan offering.
     * </p>
     * 
     * @return The requested end time that the user specified when searching for the training plan offering.
     */
    public final Instant requestedEndTimeBefore() {
        return requestedEndTimeBefore;
    }

    /**
     * <p>
     * The number of whole hours in the total duration for this training plan offering.
     * </p>
     * 
     * @return The number of whole hours in the total duration for this training plan offering.
     */
    public final Long durationHours() {
        return durationHours;
    }

    /**
     * <p>
     * The additional minutes beyond whole hours in the total duration for this training plan offering.
     * </p>
     * 
     * @return The additional minutes beyond whole hours in the total duration for this training plan offering.
     */
    public final Long durationMinutes() {
        return durationMinutes;
    }

    /**
     * <p>
     * The upfront fee for this training plan offering.
     * </p>
     * 
     * @return The upfront fee for this training plan offering.
     */
    public final String upfrontFee() {
        return upfrontFee;
    }

    /**
     * <p>
     * The currency code for the upfront fee (e.g., USD).
     * </p>
     * 
     * @return The currency code for the upfront fee (e.g., USD).
     */
    public final String currencyCode() {
        return currencyCode;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReservedCapacityOfferings property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasReservedCapacityOfferings() {
        return reservedCapacityOfferings != null && !(reservedCapacityOfferings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of reserved capacity offerings associated with this training plan offering.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReservedCapacityOfferings} method.
     * </p>
     * 
     * @return A list of reserved capacity offerings associated with this training plan offering.
     */
    public final List<ReservedCapacityOffering> reservedCapacityOfferings() {
        return reservedCapacityOfferings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(trainingPlanOfferingId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetResources() ? targetResourcesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(requestedStartTimeAfter());
        hashCode = 31 * hashCode + Objects.hashCode(requestedEndTimeBefore());
        hashCode = 31 * hashCode + Objects.hashCode(durationHours());
        hashCode = 31 * hashCode + Objects.hashCode(durationMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(upfrontFee());
        hashCode = 31 * hashCode + Objects.hashCode(currencyCode());
        hashCode = 31 * hashCode + Objects.hashCode(hasReservedCapacityOfferings() ? reservedCapacityOfferings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TrainingPlanOffering)) {
            return false;
        }
        TrainingPlanOffering other = (TrainingPlanOffering) obj;
        return Objects.equals(trainingPlanOfferingId(), other.trainingPlanOfferingId())
                && hasTargetResources() == other.hasTargetResources()
                && Objects.equals(targetResourcesAsStrings(), other.targetResourcesAsStrings())
                && Objects.equals(requestedStartTimeAfter(), other.requestedStartTimeAfter())
                && Objects.equals(requestedEndTimeBefore(), other.requestedEndTimeBefore())
                && Objects.equals(durationHours(), other.durationHours())
                && Objects.equals(durationMinutes(), other.durationMinutes()) && Objects.equals(upfrontFee(), other.upfrontFee())
                && Objects.equals(currencyCode(), other.currencyCode())
                && hasReservedCapacityOfferings() == other.hasReservedCapacityOfferings()
                && Objects.equals(reservedCapacityOfferings(), other.reservedCapacityOfferings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TrainingPlanOffering").add("TrainingPlanOfferingId", trainingPlanOfferingId())
                .add("TargetResources", hasTargetResources() ? targetResourcesAsStrings() : null)
                .add("RequestedStartTimeAfter", requestedStartTimeAfter())
                .add("RequestedEndTimeBefore", requestedEndTimeBefore()).add("DurationHours", durationHours())
                .add("DurationMinutes", durationMinutes()).add("UpfrontFee", upfrontFee()).add("CurrencyCode", currencyCode())
                .add("ReservedCapacityOfferings", hasReservedCapacityOfferings() ? reservedCapacityOfferings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TrainingPlanOfferingId":
            return Optional.ofNullable(clazz.cast(trainingPlanOfferingId()));
        case "TargetResources":
            return Optional.ofNullable(clazz.cast(targetResourcesAsStrings()));
        case "RequestedStartTimeAfter":
            return Optional.ofNullable(clazz.cast(requestedStartTimeAfter()));
        case "RequestedEndTimeBefore":
            return Optional.ofNullable(clazz.cast(requestedEndTimeBefore()));
        case "DurationHours":
            return Optional.ofNullable(clazz.cast(durationHours()));
        case "DurationMinutes":
            return Optional.ofNullable(clazz.cast(durationMinutes()));
        case "UpfrontFee":
            return Optional.ofNullable(clazz.cast(upfrontFee()));
        case "CurrencyCode":
            return Optional.ofNullable(clazz.cast(currencyCode()));
        case "ReservedCapacityOfferings":
            return Optional.ofNullable(clazz.cast(reservedCapacityOfferings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("TrainingPlanOfferingId", TRAINING_PLAN_OFFERING_ID_FIELD);
        map.put("TargetResources", TARGET_RESOURCES_FIELD);
        map.put("RequestedStartTimeAfter", REQUESTED_START_TIME_AFTER_FIELD);
        map.put("RequestedEndTimeBefore", REQUESTED_END_TIME_BEFORE_FIELD);
        map.put("DurationHours", DURATION_HOURS_FIELD);
        map.put("DurationMinutes", DURATION_MINUTES_FIELD);
        map.put("UpfrontFee", UPFRONT_FEE_FIELD);
        map.put("CurrencyCode", CURRENCY_CODE_FIELD);
        map.put("ReservedCapacityOfferings", RESERVED_CAPACITY_OFFERINGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TrainingPlanOffering, T> g) {
        return obj -> g.apply((TrainingPlanOffering) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TrainingPlanOffering> {
        /**
         * <p>
         * The unique identifier for this training plan offering.
         * </p>
         * 
         * @param trainingPlanOfferingId
         *        The unique identifier for this training plan offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingPlanOfferingId(String trainingPlanOfferingId);

        /**
         * <p>
         * The target resources (e.g., SageMaker Training Jobs, SageMaker HyperPod) for this training plan offering.
         * </p>
         * <p>
         * Training plans are specific to their target resource.
         * </p>
         * <ul>
         * <li>
         * <p>
         * A training plan designed for SageMaker training jobs can only be used to schedule and run training jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * A training plan for HyperPod clusters can be used exclusively to provide compute resources to a cluster's
         * instance group.
         * </p>
         * </li>
         * </ul>
         * 
         * @param targetResources
         *        The target resources (e.g., SageMaker Training Jobs, SageMaker HyperPod) for this training plan
         *        offering.</p>
         *        <p>
         *        Training plans are specific to their target resource.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A training plan designed for SageMaker training jobs can only be used to schedule and run training
         *        jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A training plan for HyperPod clusters can be used exclusively to provide compute resources to a
         *        cluster's instance group.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetResourcesWithStrings(Collection<String> targetResources);

        /**
         * <p>
         * The target resources (e.g., SageMaker Training Jobs, SageMaker HyperPod) for this training plan offering.
         * </p>
         * <p>
         * Training plans are specific to their target resource.
         * </p>
         * <ul>
         * <li>
         * <p>
         * A training plan designed for SageMaker training jobs can only be used to schedule and run training jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * A training plan for HyperPod clusters can be used exclusively to provide compute resources to a cluster's
         * instance group.
         * </p>
         * </li>
         * </ul>
         * 
         * @param targetResources
         *        The target resources (e.g., SageMaker Training Jobs, SageMaker HyperPod) for this training plan
         *        offering.</p>
         *        <p>
         *        Training plans are specific to their target resource.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A training plan designed for SageMaker training jobs can only be used to schedule and run training
         *        jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A training plan for HyperPod clusters can be used exclusively to provide compute resources to a
         *        cluster's instance group.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetResourcesWithStrings(String... targetResources);

        /**
         * <p>
         * The target resources (e.g., SageMaker Training Jobs, SageMaker HyperPod) for this training plan offering.
         * </p>
         * <p>
         * Training plans are specific to their target resource.
         * </p>
         * <ul>
         * <li>
         * <p>
         * A training plan designed for SageMaker training jobs can only be used to schedule and run training jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * A training plan for HyperPod clusters can be used exclusively to provide compute resources to a cluster's
         * instance group.
         * </p>
         * </li>
         * </ul>
         * 
         * @param targetResources
         *        The target resources (e.g., SageMaker Training Jobs, SageMaker HyperPod) for this training plan
         *        offering.</p>
         *        <p>
         *        Training plans are specific to their target resource.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A training plan designed for SageMaker training jobs can only be used to schedule and run training
         *        jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A training plan for HyperPod clusters can be used exclusively to provide compute resources to a
         *        cluster's instance group.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetResources(Collection<SageMakerResourceName> targetResources);

        /**
         * <p>
         * The target resources (e.g., SageMaker Training Jobs, SageMaker HyperPod) for this training plan offering.
         * </p>
         * <p>
         * Training plans are specific to their target resource.
         * </p>
         * <ul>
         * <li>
         * <p>
         * A training plan designed for SageMaker training jobs can only be used to schedule and run training jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * A training plan for HyperPod clusters can be used exclusively to provide compute resources to a cluster's
         * instance group.
         * </p>
         * </li>
         * </ul>
         * 
         * @param targetResources
         *        The target resources (e.g., SageMaker Training Jobs, SageMaker HyperPod) for this training plan
         *        offering.</p>
         *        <p>
         *        Training plans are specific to their target resource.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A training plan designed for SageMaker training jobs can only be used to schedule and run training
         *        jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A training plan for HyperPod clusters can be used exclusively to provide compute resources to a
         *        cluster's instance group.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetResources(SageMakerResourceName... targetResources);

        /**
         * <p>
         * The requested start time that the user specified when searching for the training plan offering.
         * </p>
         * 
         * @param requestedStartTimeAfter
         *        The requested start time that the user specified when searching for the training plan offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedStartTimeAfter(Instant requestedStartTimeAfter);

        /**
         * <p>
         * The requested end time that the user specified when searching for the training plan offering.
         * </p>
         * 
         * @param requestedEndTimeBefore
         *        The requested end time that the user specified when searching for the training plan offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedEndTimeBefore(Instant requestedEndTimeBefore);

        /**
         * <p>
         * The number of whole hours in the total duration for this training plan offering.
         * </p>
         * 
         * @param durationHours
         *        The number of whole hours in the total duration for this training plan offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder durationHours(Long durationHours);

        /**
         * <p>
         * The additional minutes beyond whole hours in the total duration for this training plan offering.
         * </p>
         * 
         * @param durationMinutes
         *        The additional minutes beyond whole hours in the total duration for this training plan offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder durationMinutes(Long durationMinutes);

        /**
         * <p>
         * The upfront fee for this training plan offering.
         * </p>
         * 
         * @param upfrontFee
         *        The upfront fee for this training plan offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder upfrontFee(String upfrontFee);

        /**
         * <p>
         * The currency code for the upfront fee (e.g., USD).
         * </p>
         * 
         * @param currencyCode
         *        The currency code for the upfront fee (e.g., USD).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currencyCode(String currencyCode);

        /**
         * <p>
         * A list of reserved capacity offerings associated with this training plan offering.
         * </p>
         * 
         * @param reservedCapacityOfferings
         *        A list of reserved capacity offerings associated with this training plan offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservedCapacityOfferings(Collection<ReservedCapacityOffering> reservedCapacityOfferings);

        /**
         * <p>
         * A list of reserved capacity offerings associated with this training plan offering.
         * </p>
         * 
         * @param reservedCapacityOfferings
         *        A list of reserved capacity offerings associated with this training plan offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservedCapacityOfferings(ReservedCapacityOffering... reservedCapacityOfferings);

        /**
         * <p>
         * A list of reserved capacity offerings associated with this training plan offering.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.ReservedCapacityOffering.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.sagemaker.model.ReservedCapacityOffering#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.ReservedCapacityOffering.Builder#build()} is called
         * immediately and its result is passed to {@link #reservedCapacityOfferings(List<ReservedCapacityOffering>)}.
         * 
         * @param reservedCapacityOfferings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.ReservedCapacityOffering.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #reservedCapacityOfferings(java.util.Collection<ReservedCapacityOffering>)
         */
        Builder reservedCapacityOfferings(Consumer<ReservedCapacityOffering.Builder>... reservedCapacityOfferings);
    }

    static final class BuilderImpl implements Builder {
        private String trainingPlanOfferingId;

        private List<String> targetResources = DefaultSdkAutoConstructList.getInstance();

        private Instant requestedStartTimeAfter;

        private Instant requestedEndTimeBefore;

        private Long durationHours;

        private Long durationMinutes;

        private String upfrontFee;

        private String currencyCode;

        private List<ReservedCapacityOffering> reservedCapacityOfferings = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TrainingPlanOffering model) {
            trainingPlanOfferingId(model.trainingPlanOfferingId);
            targetResourcesWithStrings(model.targetResources);
            requestedStartTimeAfter(model.requestedStartTimeAfter);
            requestedEndTimeBefore(model.requestedEndTimeBefore);
            durationHours(model.durationHours);
            durationMinutes(model.durationMinutes);
            upfrontFee(model.upfrontFee);
            currencyCode(model.currencyCode);
            reservedCapacityOfferings(model.reservedCapacityOfferings);
        }

        public final String getTrainingPlanOfferingId() {
            return trainingPlanOfferingId;
        }

        public final void setTrainingPlanOfferingId(String trainingPlanOfferingId) {
            this.trainingPlanOfferingId = trainingPlanOfferingId;
        }

        @Override
        public final Builder trainingPlanOfferingId(String trainingPlanOfferingId) {
            this.trainingPlanOfferingId = trainingPlanOfferingId;
            return this;
        }

        public final Collection<String> getTargetResources() {
            if (targetResources instanceof SdkAutoConstructList) {
                return null;
            }
            return targetResources;
        }

        public final void setTargetResources(Collection<String> targetResources) {
            this.targetResources = SageMakerResourceNamesCopier.copy(targetResources);
        }

        @Override
        public final Builder targetResourcesWithStrings(Collection<String> targetResources) {
            this.targetResources = SageMakerResourceNamesCopier.copy(targetResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetResourcesWithStrings(String... targetResources) {
            targetResourcesWithStrings(Arrays.asList(targetResources));
            return this;
        }

        @Override
        public final Builder targetResources(Collection<SageMakerResourceName> targetResources) {
            this.targetResources = SageMakerResourceNamesCopier.copyEnumToString(targetResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetResources(SageMakerResourceName... targetResources) {
            targetResources(Arrays.asList(targetResources));
            return this;
        }

        public final Instant getRequestedStartTimeAfter() {
            return requestedStartTimeAfter;
        }

        public final void setRequestedStartTimeAfter(Instant requestedStartTimeAfter) {
            this.requestedStartTimeAfter = requestedStartTimeAfter;
        }

        @Override
        public final Builder requestedStartTimeAfter(Instant requestedStartTimeAfter) {
            this.requestedStartTimeAfter = requestedStartTimeAfter;
            return this;
        }

        public final Instant getRequestedEndTimeBefore() {
            return requestedEndTimeBefore;
        }

        public final void setRequestedEndTimeBefore(Instant requestedEndTimeBefore) {
            this.requestedEndTimeBefore = requestedEndTimeBefore;
        }

        @Override
        public final Builder requestedEndTimeBefore(Instant requestedEndTimeBefore) {
            this.requestedEndTimeBefore = requestedEndTimeBefore;
            return this;
        }

        public final Long getDurationHours() {
            return durationHours;
        }

        public final void setDurationHours(Long durationHours) {
            this.durationHours = durationHours;
        }

        @Override
        public final Builder durationHours(Long durationHours) {
            this.durationHours = durationHours;
            return this;
        }

        public final Long getDurationMinutes() {
            return durationMinutes;
        }

        public final void setDurationMinutes(Long durationMinutes) {
            this.durationMinutes = durationMinutes;
        }

        @Override
        public final Builder durationMinutes(Long durationMinutes) {
            this.durationMinutes = durationMinutes;
            return this;
        }

        public final String getUpfrontFee() {
            return upfrontFee;
        }

        public final void setUpfrontFee(String upfrontFee) {
            this.upfrontFee = upfrontFee;
        }

        @Override
        public final Builder upfrontFee(String upfrontFee) {
            this.upfrontFee = upfrontFee;
            return this;
        }

        public final String getCurrencyCode() {
            return currencyCode;
        }

        public final void setCurrencyCode(String currencyCode) {
            this.currencyCode = currencyCode;
        }

        @Override
        public final Builder currencyCode(String currencyCode) {
            this.currencyCode = currencyCode;
            return this;
        }

        public final List<ReservedCapacityOffering.Builder> getReservedCapacityOfferings() {
            List<ReservedCapacityOffering.Builder> result = ReservedCapacityOfferingsCopier
                    .copyToBuilder(this.reservedCapacityOfferings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setReservedCapacityOfferings(Collection<ReservedCapacityOffering.BuilderImpl> reservedCapacityOfferings) {
            this.reservedCapacityOfferings = ReservedCapacityOfferingsCopier.copyFromBuilder(reservedCapacityOfferings);
        }

        @Override
        public final Builder reservedCapacityOfferings(Collection<ReservedCapacityOffering> reservedCapacityOfferings) {
            this.reservedCapacityOfferings = ReservedCapacityOfferingsCopier.copy(reservedCapacityOfferings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder reservedCapacityOfferings(ReservedCapacityOffering... reservedCapacityOfferings) {
            reservedCapacityOfferings(Arrays.asList(reservedCapacityOfferings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder reservedCapacityOfferings(Consumer<ReservedCapacityOffering.Builder>... reservedCapacityOfferings) {
            reservedCapacityOfferings(Stream.of(reservedCapacityOfferings)
                    .map(c -> ReservedCapacityOffering.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public TrainingPlanOffering build() {
            return new TrainingPlanOffering(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
