/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies an algorithm that was used to create the model package. The algorithm must be either an algorithm resource
 * in your SageMaker account or an algorithm in Amazon Web Services Marketplace that you are subscribed to.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SourceAlgorithm implements SdkPojo, Serializable, ToCopyableBuilder<SourceAlgorithm.Builder, SourceAlgorithm> {
    private static final SdkField<String> MODEL_DATA_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelDataUrl").getter(getter(SourceAlgorithm::modelDataUrl)).setter(setter(Builder::modelDataUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelDataUrl").build()).build();

    private static final SdkField<ModelDataSource> MODEL_DATA_SOURCE_FIELD = SdkField
            .<ModelDataSource> builder(MarshallingType.SDK_POJO).memberName("ModelDataSource")
            .getter(getter(SourceAlgorithm::modelDataSource)).setter(setter(Builder::modelDataSource))
            .constructor(ModelDataSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelDataSource").build()).build();

    private static final SdkField<String> MODEL_DATA_E_TAG_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelDataETag").getter(getter(SourceAlgorithm::modelDataETag)).setter(setter(Builder::modelDataETag))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelDataETag").build()).build();

    private static final SdkField<String> ALGORITHM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlgorithmName").getter(getter(SourceAlgorithm::algorithmName)).setter(setter(Builder::algorithmName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlgorithmName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MODEL_DATA_URL_FIELD,
            MODEL_DATA_SOURCE_FIELD, MODEL_DATA_E_TAG_FIELD, ALGORITHM_NAME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String modelDataUrl;

    private final ModelDataSource modelDataSource;

    private final String modelDataETag;

    private final String algorithmName;

    private SourceAlgorithm(BuilderImpl builder) {
        this.modelDataUrl = builder.modelDataUrl;
        this.modelDataSource = builder.modelDataSource;
        this.modelDataETag = builder.modelDataETag;
        this.algorithmName = builder.algorithmName;
    }

    /**
     * <p>
     * The Amazon S3 path where the model artifacts, which result from model training, are stored. This path must point
     * to a single <code>gzip</code> compressed tar archive (<code>.tar.gz</code> suffix).
     * </p>
     * <note>
     * <p>
     * The model artifacts must be in an S3 bucket that is in the same Amazon Web Services region as the algorithm.
     * </p>
     * </note>
     * 
     * @return The Amazon S3 path where the model artifacts, which result from model training, are stored. This path
     *         must point to a single <code>gzip</code> compressed tar archive (<code>.tar.gz</code> suffix).</p> <note>
     *         <p>
     *         The model artifacts must be in an S3 bucket that is in the same Amazon Web Services region as the
     *         algorithm.
     *         </p>
     */
    public final String modelDataUrl() {
        return modelDataUrl;
    }

    /**
     * <p>
     * Specifies the location of ML model data to deploy during endpoint creation.
     * </p>
     * 
     * @return Specifies the location of ML model data to deploy during endpoint creation.
     */
    public final ModelDataSource modelDataSource() {
        return modelDataSource;
    }

    /**
     * <p>
     * The ETag associated with Model Data URL.
     * </p>
     * 
     * @return The ETag associated with Model Data URL.
     */
    public final String modelDataETag() {
        return modelDataETag;
    }

    /**
     * <p>
     * The name of an algorithm that was used to create the model package. The algorithm must be either an algorithm
     * resource in your SageMaker account or an algorithm in Amazon Web Services Marketplace that you are subscribed to.
     * </p>
     * 
     * @return The name of an algorithm that was used to create the model package. The algorithm must be either an
     *         algorithm resource in your SageMaker account or an algorithm in Amazon Web Services Marketplace that you
     *         are subscribed to.
     */
    public final String algorithmName() {
        return algorithmName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(modelDataUrl());
        hashCode = 31 * hashCode + Objects.hashCode(modelDataSource());
        hashCode = 31 * hashCode + Objects.hashCode(modelDataETag());
        hashCode = 31 * hashCode + Objects.hashCode(algorithmName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SourceAlgorithm)) {
            return false;
        }
        SourceAlgorithm other = (SourceAlgorithm) obj;
        return Objects.equals(modelDataUrl(), other.modelDataUrl()) && Objects.equals(modelDataSource(), other.modelDataSource())
                && Objects.equals(modelDataETag(), other.modelDataETag())
                && Objects.equals(algorithmName(), other.algorithmName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SourceAlgorithm").add("ModelDataUrl", modelDataUrl()).add("ModelDataSource", modelDataSource())
                .add("ModelDataETag", modelDataETag()).add("AlgorithmName", algorithmName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ModelDataUrl":
            return Optional.ofNullable(clazz.cast(modelDataUrl()));
        case "ModelDataSource":
            return Optional.ofNullable(clazz.cast(modelDataSource()));
        case "ModelDataETag":
            return Optional.ofNullable(clazz.cast(modelDataETag()));
        case "AlgorithmName":
            return Optional.ofNullable(clazz.cast(algorithmName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ModelDataUrl", MODEL_DATA_URL_FIELD);
        map.put("ModelDataSource", MODEL_DATA_SOURCE_FIELD);
        map.put("ModelDataETag", MODEL_DATA_E_TAG_FIELD);
        map.put("AlgorithmName", ALGORITHM_NAME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SourceAlgorithm, T> g) {
        return obj -> g.apply((SourceAlgorithm) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SourceAlgorithm> {
        /**
         * <p>
         * The Amazon S3 path where the model artifacts, which result from model training, are stored. This path must
         * point to a single <code>gzip</code> compressed tar archive (<code>.tar.gz</code> suffix).
         * </p>
         * <note>
         * <p>
         * The model artifacts must be in an S3 bucket that is in the same Amazon Web Services region as the algorithm.
         * </p>
         * </note>
         * 
         * @param modelDataUrl
         *        The Amazon S3 path where the model artifacts, which result from model training, are stored. This path
         *        must point to a single <code>gzip</code> compressed tar archive (<code>.tar.gz</code> suffix).</p>
         *        <note>
         *        <p>
         *        The model artifacts must be in an S3 bucket that is in the same Amazon Web Services region as the
         *        algorithm.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelDataUrl(String modelDataUrl);

        /**
         * <p>
         * Specifies the location of ML model data to deploy during endpoint creation.
         * </p>
         * 
         * @param modelDataSource
         *        Specifies the location of ML model data to deploy during endpoint creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelDataSource(ModelDataSource modelDataSource);

        /**
         * <p>
         * Specifies the location of ML model data to deploy during endpoint creation.
         * </p>
         * This is a convenience method that creates an instance of the {@link ModelDataSource.Builder} avoiding the
         * need to create one manually via {@link ModelDataSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ModelDataSource.Builder#build()} is called immediately and its
         * result is passed to {@link #modelDataSource(ModelDataSource)}.
         * 
         * @param modelDataSource
         *        a consumer that will call methods on {@link ModelDataSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modelDataSource(ModelDataSource)
         */
        default Builder modelDataSource(Consumer<ModelDataSource.Builder> modelDataSource) {
            return modelDataSource(ModelDataSource.builder().applyMutation(modelDataSource).build());
        }

        /**
         * <p>
         * The ETag associated with Model Data URL.
         * </p>
         * 
         * @param modelDataETag
         *        The ETag associated with Model Data URL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelDataETag(String modelDataETag);

        /**
         * <p>
         * The name of an algorithm that was used to create the model package. The algorithm must be either an algorithm
         * resource in your SageMaker account or an algorithm in Amazon Web Services Marketplace that you are subscribed
         * to.
         * </p>
         * 
         * @param algorithmName
         *        The name of an algorithm that was used to create the model package. The algorithm must be either an
         *        algorithm resource in your SageMaker account or an algorithm in Amazon Web Services Marketplace that
         *        you are subscribed to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder algorithmName(String algorithmName);
    }

    static final class BuilderImpl implements Builder {
        private String modelDataUrl;

        private ModelDataSource modelDataSource;

        private String modelDataETag;

        private String algorithmName;

        private BuilderImpl() {
        }

        private BuilderImpl(SourceAlgorithm model) {
            modelDataUrl(model.modelDataUrl);
            modelDataSource(model.modelDataSource);
            modelDataETag(model.modelDataETag);
            algorithmName(model.algorithmName);
        }

        public final String getModelDataUrl() {
            return modelDataUrl;
        }

        public final void setModelDataUrl(String modelDataUrl) {
            this.modelDataUrl = modelDataUrl;
        }

        @Override
        public final Builder modelDataUrl(String modelDataUrl) {
            this.modelDataUrl = modelDataUrl;
            return this;
        }

        public final ModelDataSource.Builder getModelDataSource() {
            return modelDataSource != null ? modelDataSource.toBuilder() : null;
        }

        public final void setModelDataSource(ModelDataSource.BuilderImpl modelDataSource) {
            this.modelDataSource = modelDataSource != null ? modelDataSource.build() : null;
        }

        @Override
        public final Builder modelDataSource(ModelDataSource modelDataSource) {
            this.modelDataSource = modelDataSource;
            return this;
        }

        public final String getModelDataETag() {
            return modelDataETag;
        }

        public final void setModelDataETag(String modelDataETag) {
            this.modelDataETag = modelDataETag;
        }

        @Override
        public final Builder modelDataETag(String modelDataETag) {
            this.modelDataETag = modelDataETag;
            return this;
        }

        public final String getAlgorithmName() {
            return algorithmName;
        }

        public final void setAlgorithmName(String algorithmName) {
            this.algorithmName = algorithmName;
        }

        @Override
        public final Builder algorithmName(String algorithmName) {
            this.algorithmName = algorithmName;
            return this;
        }

        @Override
        public SourceAlgorithm build() {
            return new SourceAlgorithm(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
