/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details of an instance group in a SageMaker HyperPod cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ClusterInstanceGroupDetails implements SdkPojo, Serializable,
        ToCopyableBuilder<ClusterInstanceGroupDetails.Builder, ClusterInstanceGroupDetails> {
    private static final SdkField<Integer> CURRENT_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("CurrentCount").getter(getter(ClusterInstanceGroupDetails::currentCount))
            .setter(setter(Builder::currentCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CurrentCount").build()).build();

    private static final SdkField<Integer> TARGET_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TargetCount").getter(getter(ClusterInstanceGroupDetails::targetCount))
            .setter(setter(Builder::targetCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetCount").build()).build();

    private static final SdkField<String> INSTANCE_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceGroupName").getter(getter(ClusterInstanceGroupDetails::instanceGroupName))
            .setter(setter(Builder::instanceGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceGroupName").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(ClusterInstanceGroupDetails::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<ClusterLifeCycleConfig> LIFE_CYCLE_CONFIG_FIELD = SdkField
            .<ClusterLifeCycleConfig> builder(MarshallingType.SDK_POJO).memberName("LifeCycleConfig")
            .getter(getter(ClusterInstanceGroupDetails::lifeCycleConfig)).setter(setter(Builder::lifeCycleConfig))
            .constructor(ClusterLifeCycleConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LifeCycleConfig").build()).build();

    private static final SdkField<String> EXECUTION_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionRole").getter(getter(ClusterInstanceGroupDetails::executionRole))
            .setter(setter(Builder::executionRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionRole").build()).build();

    private static final SdkField<Integer> THREADS_PER_CORE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ThreadsPerCore").getter(getter(ClusterInstanceGroupDetails::threadsPerCore))
            .setter(setter(Builder::threadsPerCore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ThreadsPerCore").build()).build();

    private static final SdkField<List<ClusterInstanceStorageConfig>> INSTANCE_STORAGE_CONFIGS_FIELD = SdkField
            .<List<ClusterInstanceStorageConfig>> builder(MarshallingType.LIST)
            .memberName("InstanceStorageConfigs")
            .getter(getter(ClusterInstanceGroupDetails::instanceStorageConfigs))
            .setter(setter(Builder::instanceStorageConfigs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceStorageConfigs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ClusterInstanceStorageConfig> builder(MarshallingType.SDK_POJO)
                                            .constructor(ClusterInstanceStorageConfig::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ON_START_DEEP_HEALTH_CHECKS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("OnStartDeepHealthChecks")
            .getter(getter(ClusterInstanceGroupDetails::onStartDeepHealthChecksAsStrings))
            .setter(setter(Builder::onStartDeepHealthChecksWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OnStartDeepHealthChecks").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ClusterInstanceGroupDetails::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> TRAINING_PLAN_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TrainingPlanArn").getter(getter(ClusterInstanceGroupDetails::trainingPlanArn))
            .setter(setter(Builder::trainingPlanArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingPlanArn").build()).build();

    private static final SdkField<String> TRAINING_PLAN_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TrainingPlanStatus").getter(getter(ClusterInstanceGroupDetails::trainingPlanStatus))
            .setter(setter(Builder::trainingPlanStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingPlanStatus").build())
            .build();

    private static final SdkField<VpcConfig> OVERRIDE_VPC_CONFIG_FIELD = SdkField.<VpcConfig> builder(MarshallingType.SDK_POJO)
            .memberName("OverrideVpcConfig").getter(getter(ClusterInstanceGroupDetails::overrideVpcConfig))
            .setter(setter(Builder::overrideVpcConfig)).constructor(VpcConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OverrideVpcConfig").build()).build();

    private static final SdkField<ScheduledUpdateConfig> SCHEDULED_UPDATE_CONFIG_FIELD = SdkField
            .<ScheduledUpdateConfig> builder(MarshallingType.SDK_POJO).memberName("ScheduledUpdateConfig")
            .getter(getter(ClusterInstanceGroupDetails::scheduledUpdateConfig)).setter(setter(Builder::scheduledUpdateConfig))
            .constructor(ScheduledUpdateConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduledUpdateConfig").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CURRENT_COUNT_FIELD,
            TARGET_COUNT_FIELD, INSTANCE_GROUP_NAME_FIELD, INSTANCE_TYPE_FIELD, LIFE_CYCLE_CONFIG_FIELD, EXECUTION_ROLE_FIELD,
            THREADS_PER_CORE_FIELD, INSTANCE_STORAGE_CONFIGS_FIELD, ON_START_DEEP_HEALTH_CHECKS_FIELD, STATUS_FIELD,
            TRAINING_PLAN_ARN_FIELD, TRAINING_PLAN_STATUS_FIELD, OVERRIDE_VPC_CONFIG_FIELD, SCHEDULED_UPDATE_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer currentCount;

    private final Integer targetCount;

    private final String instanceGroupName;

    private final String instanceType;

    private final ClusterLifeCycleConfig lifeCycleConfig;

    private final String executionRole;

    private final Integer threadsPerCore;

    private final List<ClusterInstanceStorageConfig> instanceStorageConfigs;

    private final List<String> onStartDeepHealthChecks;

    private final String status;

    private final String trainingPlanArn;

    private final String trainingPlanStatus;

    private final VpcConfig overrideVpcConfig;

    private final ScheduledUpdateConfig scheduledUpdateConfig;

    private ClusterInstanceGroupDetails(BuilderImpl builder) {
        this.currentCount = builder.currentCount;
        this.targetCount = builder.targetCount;
        this.instanceGroupName = builder.instanceGroupName;
        this.instanceType = builder.instanceType;
        this.lifeCycleConfig = builder.lifeCycleConfig;
        this.executionRole = builder.executionRole;
        this.threadsPerCore = builder.threadsPerCore;
        this.instanceStorageConfigs = builder.instanceStorageConfigs;
        this.onStartDeepHealthChecks = builder.onStartDeepHealthChecks;
        this.status = builder.status;
        this.trainingPlanArn = builder.trainingPlanArn;
        this.trainingPlanStatus = builder.trainingPlanStatus;
        this.overrideVpcConfig = builder.overrideVpcConfig;
        this.scheduledUpdateConfig = builder.scheduledUpdateConfig;
    }

    /**
     * <p>
     * The number of instances that are currently in the instance group of a SageMaker HyperPod cluster.
     * </p>
     * 
     * @return The number of instances that are currently in the instance group of a SageMaker HyperPod cluster.
     */
    public final Integer currentCount() {
        return currentCount;
    }

    /**
     * <p>
     * The number of instances you specified to add to the instance group of a SageMaker HyperPod cluster.
     * </p>
     * 
     * @return The number of instances you specified to add to the instance group of a SageMaker HyperPod cluster.
     */
    public final Integer targetCount() {
        return targetCount;
    }

    /**
     * <p>
     * The name of the instance group of a SageMaker HyperPod cluster.
     * </p>
     * 
     * @return The name of the instance group of a SageMaker HyperPod cluster.
     */
    public final String instanceGroupName() {
        return instanceGroupName;
    }

    /**
     * <p>
     * The instance type of the instance group of a SageMaker HyperPod cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link ClusterInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type of the instance group of a SageMaker HyperPod cluster.
     * @see ClusterInstanceType
     */
    public final ClusterInstanceType instanceType() {
        return ClusterInstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The instance type of the instance group of a SageMaker HyperPod cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link ClusterInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type of the instance group of a SageMaker HyperPod cluster.
     * @see ClusterInstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * Details of LifeCycle configuration for the instance group.
     * </p>
     * 
     * @return Details of LifeCycle configuration for the instance group.
     */
    public final ClusterLifeCycleConfig lifeCycleConfig() {
        return lifeCycleConfig;
    }

    /**
     * <p>
     * The execution role for the instance group to assume.
     * </p>
     * 
     * @return The execution role for the instance group to assume.
     */
    public final String executionRole() {
        return executionRole;
    }

    /**
     * <p>
     * The number you specified to <code>TreadsPerCore</code> in <code>CreateCluster</code> for enabling or disabling
     * multithreading. For instance types that support multithreading, you can specify 1 for disabling multithreading
     * and 2 for enabling multithreading. For more information, see the reference table of <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/cpu-options-supported-instances-values.html">CPU cores
     * and threads per CPU core per instance type</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * 
     * @return The number you specified to <code>TreadsPerCore</code> in <code>CreateCluster</code> for enabling or
     *         disabling multithreading. For instance types that support multithreading, you can specify 1 for disabling
     *         multithreading and 2 for enabling multithreading. For more information, see the reference table of <a
     *         href
     *         ="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/cpu-options-supported-instances-values.html">CPU
     *         cores and threads per CPU core per instance type</a> in the <i>Amazon Elastic Compute Cloud User
     *         Guide</i>.
     */
    public final Integer threadsPerCore() {
        return threadsPerCore;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceStorageConfigs property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasInstanceStorageConfigs() {
        return instanceStorageConfigs != null && !(instanceStorageConfigs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The additional storage configurations for the instances in the SageMaker HyperPod cluster instance group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceStorageConfigs} method.
     * </p>
     * 
     * @return The additional storage configurations for the instances in the SageMaker HyperPod cluster instance group.
     */
    public final List<ClusterInstanceStorageConfig> instanceStorageConfigs() {
        return instanceStorageConfigs;
    }

    /**
     * <p>
     * A flag indicating whether deep health checks should be performed when the cluster instance group is created or
     * updated.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOnStartDeepHealthChecks} method.
     * </p>
     * 
     * @return A flag indicating whether deep health checks should be performed when the cluster instance group is
     *         created or updated.
     */
    public final List<DeepHealthCheckType> onStartDeepHealthChecks() {
        return OnStartDeepHealthChecksCopier.copyStringToEnum(onStartDeepHealthChecks);
    }

    /**
     * For responses, this returns true if the service returned a value for the OnStartDeepHealthChecks property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasOnStartDeepHealthChecks() {
        return onStartDeepHealthChecks != null && !(onStartDeepHealthChecks instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A flag indicating whether deep health checks should be performed when the cluster instance group is created or
     * updated.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOnStartDeepHealthChecks} method.
     * </p>
     * 
     * @return A flag indicating whether deep health checks should be performed when the cluster instance group is
     *         created or updated.
     */
    public final List<String> onStartDeepHealthChecksAsStrings() {
        return onStartDeepHealthChecks;
    }

    /**
     * <p>
     * The current status of the cluster instance group.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>InService</code>: The instance group is active and healthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Creating</code>: The instance group is being provisioned.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Updating</code>: The instance group is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code>: The instance group has failed to provision or is no longer healthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Degraded</code>: The instance group is degraded, meaning that some instances have failed to provision or
     * are no longer healthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Deleting</code>: The instance group is being deleted.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link InstanceGroupStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the cluster instance group.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>InService</code>: The instance group is active and healthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Creating</code>: The instance group is being provisioned.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Updating</code>: The instance group is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code>: The instance group has failed to provision or is no longer healthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Degraded</code>: The instance group is degraded, meaning that some instances have failed to
     *         provision or are no longer healthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Deleting</code>: The instance group is being deleted.
     *         </p>
     *         </li>
     * @see InstanceGroupStatus
     */
    public final InstanceGroupStatus status() {
        return InstanceGroupStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the cluster instance group.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>InService</code>: The instance group is active and healthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Creating</code>: The instance group is being provisioned.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Updating</code>: The instance group is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code>: The instance group has failed to provision or is no longer healthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Degraded</code>: The instance group is degraded, meaning that some instances have failed to provision or
     * are no longer healthy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Deleting</code>: The instance group is being deleted.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link InstanceGroupStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the cluster instance group.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>InService</code>: The instance group is active and healthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Creating</code>: The instance group is being provisioned.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Updating</code>: The instance group is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code>: The instance group has failed to provision or is no longer healthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Degraded</code>: The instance group is degraded, meaning that some instances have failed to
     *         provision or are no longer healthy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Deleting</code>: The instance group is being deleted.
     *         </p>
     *         </li>
     * @see InstanceGroupStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN); of the training plan associated with this cluster instance group.
     * </p>
     * <p>
     * For more information about how to reserve GPU capacity for your SageMaker HyperPod clusters using Amazon
     * SageMaker Training Plan, see
     * <code> <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateTrainingPlan.html">CreateTrainingPlan</a> </code>
     * .
     * </p>
     * 
     * @return The Amazon Resource Name (ARN); of the training plan associated with this cluster instance group.</p>
     *         <p>
     *         For more information about how to reserve GPU capacity for your SageMaker HyperPod clusters using Amazon
     *         SageMaker Training Plan, see
     *         <code> <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateTrainingPlan.html">CreateTrainingPlan</a> </code>.
     */
    public final String trainingPlanArn() {
        return trainingPlanArn;
    }

    /**
     * <p>
     * The current status of the training plan associated with this cluster instance group.
     * </p>
     * 
     * @return The current status of the training plan associated with this cluster instance group.
     */
    public final String trainingPlanStatus() {
        return trainingPlanStatus;
    }

    /**
     * <p>
     * The customized Amazon VPC configuration at the instance group level that overrides the default Amazon VPC
     * configuration of the SageMaker HyperPod cluster.
     * </p>
     * 
     * @return The customized Amazon VPC configuration at the instance group level that overrides the default Amazon VPC
     *         configuration of the SageMaker HyperPod cluster.
     */
    public final VpcConfig overrideVpcConfig() {
        return overrideVpcConfig;
    }

    /**
     * <p>
     * The configuration object of the schedule that SageMaker follows when updating the AMI.
     * </p>
     * 
     * @return The configuration object of the schedule that SageMaker follows when updating the AMI.
     */
    public final ScheduledUpdateConfig scheduledUpdateConfig() {
        return scheduledUpdateConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(currentCount());
        hashCode = 31 * hashCode + Objects.hashCode(targetCount());
        hashCode = 31 * hashCode + Objects.hashCode(instanceGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lifeCycleConfig());
        hashCode = 31 * hashCode + Objects.hashCode(executionRole());
        hashCode = 31 * hashCode + Objects.hashCode(threadsPerCore());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceStorageConfigs() ? instanceStorageConfigs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasOnStartDeepHealthChecks() ? onStartDeepHealthChecksAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(trainingPlanArn());
        hashCode = 31 * hashCode + Objects.hashCode(trainingPlanStatus());
        hashCode = 31 * hashCode + Objects.hashCode(overrideVpcConfig());
        hashCode = 31 * hashCode + Objects.hashCode(scheduledUpdateConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClusterInstanceGroupDetails)) {
            return false;
        }
        ClusterInstanceGroupDetails other = (ClusterInstanceGroupDetails) obj;
        return Objects.equals(currentCount(), other.currentCount()) && Objects.equals(targetCount(), other.targetCount())
                && Objects.equals(instanceGroupName(), other.instanceGroupName())
                && Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(lifeCycleConfig(), other.lifeCycleConfig())
                && Objects.equals(executionRole(), other.executionRole())
                && Objects.equals(threadsPerCore(), other.threadsPerCore())
                && hasInstanceStorageConfigs() == other.hasInstanceStorageConfigs()
                && Objects.equals(instanceStorageConfigs(), other.instanceStorageConfigs())
                && hasOnStartDeepHealthChecks() == other.hasOnStartDeepHealthChecks()
                && Objects.equals(onStartDeepHealthChecksAsStrings(), other.onStartDeepHealthChecksAsStrings())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(trainingPlanArn(), other.trainingPlanArn())
                && Objects.equals(trainingPlanStatus(), other.trainingPlanStatus())
                && Objects.equals(overrideVpcConfig(), other.overrideVpcConfig())
                && Objects.equals(scheduledUpdateConfig(), other.scheduledUpdateConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ClusterInstanceGroupDetails").add("CurrentCount", currentCount())
                .add("TargetCount", targetCount()).add("InstanceGroupName", instanceGroupName())
                .add("InstanceType", instanceTypeAsString()).add("LifeCycleConfig", lifeCycleConfig())
                .add("ExecutionRole", executionRole()).add("ThreadsPerCore", threadsPerCore())
                .add("InstanceStorageConfigs", hasInstanceStorageConfigs() ? instanceStorageConfigs() : null)
                .add("OnStartDeepHealthChecks", hasOnStartDeepHealthChecks() ? onStartDeepHealthChecksAsStrings() : null)
                .add("Status", statusAsString()).add("TrainingPlanArn", trainingPlanArn())
                .add("TrainingPlanStatus", trainingPlanStatus()).add("OverrideVpcConfig", overrideVpcConfig())
                .add("ScheduledUpdateConfig", scheduledUpdateConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CurrentCount":
            return Optional.ofNullable(clazz.cast(currentCount()));
        case "TargetCount":
            return Optional.ofNullable(clazz.cast(targetCount()));
        case "InstanceGroupName":
            return Optional.ofNullable(clazz.cast(instanceGroupName()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "LifeCycleConfig":
            return Optional.ofNullable(clazz.cast(lifeCycleConfig()));
        case "ExecutionRole":
            return Optional.ofNullable(clazz.cast(executionRole()));
        case "ThreadsPerCore":
            return Optional.ofNullable(clazz.cast(threadsPerCore()));
        case "InstanceStorageConfigs":
            return Optional.ofNullable(clazz.cast(instanceStorageConfigs()));
        case "OnStartDeepHealthChecks":
            return Optional.ofNullable(clazz.cast(onStartDeepHealthChecksAsStrings()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "TrainingPlanArn":
            return Optional.ofNullable(clazz.cast(trainingPlanArn()));
        case "TrainingPlanStatus":
            return Optional.ofNullable(clazz.cast(trainingPlanStatus()));
        case "OverrideVpcConfig":
            return Optional.ofNullable(clazz.cast(overrideVpcConfig()));
        case "ScheduledUpdateConfig":
            return Optional.ofNullable(clazz.cast(scheduledUpdateConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CurrentCount", CURRENT_COUNT_FIELD);
        map.put("TargetCount", TARGET_COUNT_FIELD);
        map.put("InstanceGroupName", INSTANCE_GROUP_NAME_FIELD);
        map.put("InstanceType", INSTANCE_TYPE_FIELD);
        map.put("LifeCycleConfig", LIFE_CYCLE_CONFIG_FIELD);
        map.put("ExecutionRole", EXECUTION_ROLE_FIELD);
        map.put("ThreadsPerCore", THREADS_PER_CORE_FIELD);
        map.put("InstanceStorageConfigs", INSTANCE_STORAGE_CONFIGS_FIELD);
        map.put("OnStartDeepHealthChecks", ON_START_DEEP_HEALTH_CHECKS_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("TrainingPlanArn", TRAINING_PLAN_ARN_FIELD);
        map.put("TrainingPlanStatus", TRAINING_PLAN_STATUS_FIELD);
        map.put("OverrideVpcConfig", OVERRIDE_VPC_CONFIG_FIELD);
        map.put("ScheduledUpdateConfig", SCHEDULED_UPDATE_CONFIG_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ClusterInstanceGroupDetails, T> g) {
        return obj -> g.apply((ClusterInstanceGroupDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ClusterInstanceGroupDetails> {
        /**
         * <p>
         * The number of instances that are currently in the instance group of a SageMaker HyperPod cluster.
         * </p>
         * 
         * @param currentCount
         *        The number of instances that are currently in the instance group of a SageMaker HyperPod cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentCount(Integer currentCount);

        /**
         * <p>
         * The number of instances you specified to add to the instance group of a SageMaker HyperPod cluster.
         * </p>
         * 
         * @param targetCount
         *        The number of instances you specified to add to the instance group of a SageMaker HyperPod cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetCount(Integer targetCount);

        /**
         * <p>
         * The name of the instance group of a SageMaker HyperPod cluster.
         * </p>
         * 
         * @param instanceGroupName
         *        The name of the instance group of a SageMaker HyperPod cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGroupName(String instanceGroupName);

        /**
         * <p>
         * The instance type of the instance group of a SageMaker HyperPod cluster.
         * </p>
         * 
         * @param instanceType
         *        The instance type of the instance group of a SageMaker HyperPod cluster.
         * @see ClusterInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterInstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The instance type of the instance group of a SageMaker HyperPod cluster.
         * </p>
         * 
         * @param instanceType
         *        The instance type of the instance group of a SageMaker HyperPod cluster.
         * @see ClusterInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterInstanceType
         */
        Builder instanceType(ClusterInstanceType instanceType);

        /**
         * <p>
         * Details of LifeCycle configuration for the instance group.
         * </p>
         * 
         * @param lifeCycleConfig
         *        Details of LifeCycle configuration for the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifeCycleConfig(ClusterLifeCycleConfig lifeCycleConfig);

        /**
         * <p>
         * Details of LifeCycle configuration for the instance group.
         * </p>
         * This is a convenience method that creates an instance of the {@link ClusterLifeCycleConfig.Builder} avoiding
         * the need to create one manually via {@link ClusterLifeCycleConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ClusterLifeCycleConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #lifeCycleConfig(ClusterLifeCycleConfig)}.
         * 
         * @param lifeCycleConfig
         *        a consumer that will call methods on {@link ClusterLifeCycleConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lifeCycleConfig(ClusterLifeCycleConfig)
         */
        default Builder lifeCycleConfig(Consumer<ClusterLifeCycleConfig.Builder> lifeCycleConfig) {
            return lifeCycleConfig(ClusterLifeCycleConfig.builder().applyMutation(lifeCycleConfig).build());
        }

        /**
         * <p>
         * The execution role for the instance group to assume.
         * </p>
         * 
         * @param executionRole
         *        The execution role for the instance group to assume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRole(String executionRole);

        /**
         * <p>
         * The number you specified to <code>TreadsPerCore</code> in <code>CreateCluster</code> for enabling or
         * disabling multithreading. For instance types that support multithreading, you can specify 1 for disabling
         * multithreading and 2 for enabling multithreading. For more information, see the reference table of <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/cpu-options-supported-instances-values.html">CPU
         * cores and threads per CPU core per instance type</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * 
         * @param threadsPerCore
         *        The number you specified to <code>TreadsPerCore</code> in <code>CreateCluster</code> for enabling or
         *        disabling multithreading. For instance types that support multithreading, you can specify 1 for
         *        disabling multithreading and 2 for enabling multithreading. For more information, see the reference
         *        table of <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/cpu-options-supported-instances-values.html">CPU
         *        cores and threads per CPU core per instance type</a> in the <i>Amazon Elastic Compute Cloud User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threadsPerCore(Integer threadsPerCore);

        /**
         * <p>
         * The additional storage configurations for the instances in the SageMaker HyperPod cluster instance group.
         * </p>
         * 
         * @param instanceStorageConfigs
         *        The additional storage configurations for the instances in the SageMaker HyperPod cluster instance
         *        group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceStorageConfigs(Collection<ClusterInstanceStorageConfig> instanceStorageConfigs);

        /**
         * <p>
         * The additional storage configurations for the instances in the SageMaker HyperPod cluster instance group.
         * </p>
         * 
         * @param instanceStorageConfigs
         *        The additional storage configurations for the instances in the SageMaker HyperPod cluster instance
         *        group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceStorageConfigs(ClusterInstanceStorageConfig... instanceStorageConfigs);

        /**
         * <p>
         * The additional storage configurations for the instances in the SageMaker HyperPod cluster instance group.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.ClusterInstanceStorageConfig.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.sagemaker.model.ClusterInstanceStorageConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.ClusterInstanceStorageConfig.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #instanceStorageConfigs(List<ClusterInstanceStorageConfig>)}.
         * 
         * @param instanceStorageConfigs
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.ClusterInstanceStorageConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceStorageConfigs(java.util.Collection<ClusterInstanceStorageConfig>)
         */
        Builder instanceStorageConfigs(Consumer<ClusterInstanceStorageConfig.Builder>... instanceStorageConfigs);

        /**
         * <p>
         * A flag indicating whether deep health checks should be performed when the cluster instance group is created
         * or updated.
         * </p>
         * 
         * @param onStartDeepHealthChecks
         *        A flag indicating whether deep health checks should be performed when the cluster instance group is
         *        created or updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder onStartDeepHealthChecksWithStrings(Collection<String> onStartDeepHealthChecks);

        /**
         * <p>
         * A flag indicating whether deep health checks should be performed when the cluster instance group is created
         * or updated.
         * </p>
         * 
         * @param onStartDeepHealthChecks
         *        A flag indicating whether deep health checks should be performed when the cluster instance group is
         *        created or updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder onStartDeepHealthChecksWithStrings(String... onStartDeepHealthChecks);

        /**
         * <p>
         * A flag indicating whether deep health checks should be performed when the cluster instance group is created
         * or updated.
         * </p>
         * 
         * @param onStartDeepHealthChecks
         *        A flag indicating whether deep health checks should be performed when the cluster instance group is
         *        created or updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder onStartDeepHealthChecks(Collection<DeepHealthCheckType> onStartDeepHealthChecks);

        /**
         * <p>
         * A flag indicating whether deep health checks should be performed when the cluster instance group is created
         * or updated.
         * </p>
         * 
         * @param onStartDeepHealthChecks
         *        A flag indicating whether deep health checks should be performed when the cluster instance group is
         *        created or updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder onStartDeepHealthChecks(DeepHealthCheckType... onStartDeepHealthChecks);

        /**
         * <p>
         * The current status of the cluster instance group.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>InService</code>: The instance group is active and healthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Creating</code>: The instance group is being provisioned.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Updating</code>: The instance group is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code>: The instance group has failed to provision or is no longer healthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Degraded</code>: The instance group is degraded, meaning that some instances have failed to provision
         * or are no longer healthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Deleting</code>: The instance group is being deleted.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current status of the cluster instance group.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>InService</code>: The instance group is active and healthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Creating</code>: The instance group is being provisioned.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Updating</code>: The instance group is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code>: The instance group has failed to provision or is no longer healthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Degraded</code>: The instance group is degraded, meaning that some instances have failed to
         *        provision or are no longer healthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Deleting</code>: The instance group is being deleted.
         *        </p>
         *        </li>
         * @see InstanceGroupStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceGroupStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the cluster instance group.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>InService</code>: The instance group is active and healthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Creating</code>: The instance group is being provisioned.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Updating</code>: The instance group is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code>: The instance group has failed to provision or is no longer healthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Degraded</code>: The instance group is degraded, meaning that some instances have failed to provision
         * or are no longer healthy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Deleting</code>: The instance group is being deleted.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current status of the cluster instance group.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>InService</code>: The instance group is active and healthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Creating</code>: The instance group is being provisioned.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Updating</code>: The instance group is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code>: The instance group has failed to provision or is no longer healthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Degraded</code>: The instance group is degraded, meaning that some instances have failed to
         *        provision or are no longer healthy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Deleting</code>: The instance group is being deleted.
         *        </p>
         *        </li>
         * @see InstanceGroupStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceGroupStatus
         */
        Builder status(InstanceGroupStatus status);

        /**
         * <p>
         * The Amazon Resource Name (ARN); of the training plan associated with this cluster instance group.
         * </p>
         * <p>
         * For more information about how to reserve GPU capacity for your SageMaker HyperPod clusters using Amazon
         * SageMaker Training Plan, see
         * <code> <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateTrainingPlan.html">CreateTrainingPlan</a> </code>
         * .
         * </p>
         * 
         * @param trainingPlanArn
         *        The Amazon Resource Name (ARN); of the training plan associated with this cluster instance group.</p>
         *        <p>
         *        For more information about how to reserve GPU capacity for your SageMaker HyperPod clusters using
         *        Amazon SageMaker Training Plan, see
         *        <code> <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateTrainingPlan.html">CreateTrainingPlan</a> </code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingPlanArn(String trainingPlanArn);

        /**
         * <p>
         * The current status of the training plan associated with this cluster instance group.
         * </p>
         * 
         * @param trainingPlanStatus
         *        The current status of the training plan associated with this cluster instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingPlanStatus(String trainingPlanStatus);

        /**
         * <p>
         * The customized Amazon VPC configuration at the instance group level that overrides the default Amazon VPC
         * configuration of the SageMaker HyperPod cluster.
         * </p>
         * 
         * @param overrideVpcConfig
         *        The customized Amazon VPC configuration at the instance group level that overrides the default Amazon
         *        VPC configuration of the SageMaker HyperPod cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overrideVpcConfig(VpcConfig overrideVpcConfig);

        /**
         * <p>
         * The customized Amazon VPC configuration at the instance group level that overrides the default Amazon VPC
         * configuration of the SageMaker HyperPod cluster.
         * </p>
         * This is a convenience method that creates an instance of the {@link VpcConfig.Builder} avoiding the need to
         * create one manually via {@link VpcConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VpcConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #overrideVpcConfig(VpcConfig)}.
         * 
         * @param overrideVpcConfig
         *        a consumer that will call methods on {@link VpcConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #overrideVpcConfig(VpcConfig)
         */
        default Builder overrideVpcConfig(Consumer<VpcConfig.Builder> overrideVpcConfig) {
            return overrideVpcConfig(VpcConfig.builder().applyMutation(overrideVpcConfig).build());
        }

        /**
         * <p>
         * The configuration object of the schedule that SageMaker follows when updating the AMI.
         * </p>
         * 
         * @param scheduledUpdateConfig
         *        The configuration object of the schedule that SageMaker follows when updating the AMI.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduledUpdateConfig(ScheduledUpdateConfig scheduledUpdateConfig);

        /**
         * <p>
         * The configuration object of the schedule that SageMaker follows when updating the AMI.
         * </p>
         * This is a convenience method that creates an instance of the {@link ScheduledUpdateConfig.Builder} avoiding
         * the need to create one manually via {@link ScheduledUpdateConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ScheduledUpdateConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #scheduledUpdateConfig(ScheduledUpdateConfig)}.
         * 
         * @param scheduledUpdateConfig
         *        a consumer that will call methods on {@link ScheduledUpdateConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scheduledUpdateConfig(ScheduledUpdateConfig)
         */
        default Builder scheduledUpdateConfig(Consumer<ScheduledUpdateConfig.Builder> scheduledUpdateConfig) {
            return scheduledUpdateConfig(ScheduledUpdateConfig.builder().applyMutation(scheduledUpdateConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Integer currentCount;

        private Integer targetCount;

        private String instanceGroupName;

        private String instanceType;

        private ClusterLifeCycleConfig lifeCycleConfig;

        private String executionRole;

        private Integer threadsPerCore;

        private List<ClusterInstanceStorageConfig> instanceStorageConfigs = DefaultSdkAutoConstructList.getInstance();

        private List<String> onStartDeepHealthChecks = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String trainingPlanArn;

        private String trainingPlanStatus;

        private VpcConfig overrideVpcConfig;

        private ScheduledUpdateConfig scheduledUpdateConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(ClusterInstanceGroupDetails model) {
            currentCount(model.currentCount);
            targetCount(model.targetCount);
            instanceGroupName(model.instanceGroupName);
            instanceType(model.instanceType);
            lifeCycleConfig(model.lifeCycleConfig);
            executionRole(model.executionRole);
            threadsPerCore(model.threadsPerCore);
            instanceStorageConfigs(model.instanceStorageConfigs);
            onStartDeepHealthChecksWithStrings(model.onStartDeepHealthChecks);
            status(model.status);
            trainingPlanArn(model.trainingPlanArn);
            trainingPlanStatus(model.trainingPlanStatus);
            overrideVpcConfig(model.overrideVpcConfig);
            scheduledUpdateConfig(model.scheduledUpdateConfig);
        }

        public final Integer getCurrentCount() {
            return currentCount;
        }

        public final void setCurrentCount(Integer currentCount) {
            this.currentCount = currentCount;
        }

        @Override
        public final Builder currentCount(Integer currentCount) {
            this.currentCount = currentCount;
            return this;
        }

        public final Integer getTargetCount() {
            return targetCount;
        }

        public final void setTargetCount(Integer targetCount) {
            this.targetCount = targetCount;
        }

        @Override
        public final Builder targetCount(Integer targetCount) {
            this.targetCount = targetCount;
            return this;
        }

        public final String getInstanceGroupName() {
            return instanceGroupName;
        }

        public final void setInstanceGroupName(String instanceGroupName) {
            this.instanceGroupName = instanceGroupName;
        }

        @Override
        public final Builder instanceGroupName(String instanceGroupName) {
            this.instanceGroupName = instanceGroupName;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(ClusterInstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final ClusterLifeCycleConfig.Builder getLifeCycleConfig() {
            return lifeCycleConfig != null ? lifeCycleConfig.toBuilder() : null;
        }

        public final void setLifeCycleConfig(ClusterLifeCycleConfig.BuilderImpl lifeCycleConfig) {
            this.lifeCycleConfig = lifeCycleConfig != null ? lifeCycleConfig.build() : null;
        }

        @Override
        public final Builder lifeCycleConfig(ClusterLifeCycleConfig lifeCycleConfig) {
            this.lifeCycleConfig = lifeCycleConfig;
            return this;
        }

        public final String getExecutionRole() {
            return executionRole;
        }

        public final void setExecutionRole(String executionRole) {
            this.executionRole = executionRole;
        }

        @Override
        public final Builder executionRole(String executionRole) {
            this.executionRole = executionRole;
            return this;
        }

        public final Integer getThreadsPerCore() {
            return threadsPerCore;
        }

        public final void setThreadsPerCore(Integer threadsPerCore) {
            this.threadsPerCore = threadsPerCore;
        }

        @Override
        public final Builder threadsPerCore(Integer threadsPerCore) {
            this.threadsPerCore = threadsPerCore;
            return this;
        }

        public final List<ClusterInstanceStorageConfig.Builder> getInstanceStorageConfigs() {
            List<ClusterInstanceStorageConfig.Builder> result = ClusterInstanceStorageConfigsCopier
                    .copyToBuilder(this.instanceStorageConfigs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInstanceStorageConfigs(Collection<ClusterInstanceStorageConfig.BuilderImpl> instanceStorageConfigs) {
            this.instanceStorageConfigs = ClusterInstanceStorageConfigsCopier.copyFromBuilder(instanceStorageConfigs);
        }

        @Override
        public final Builder instanceStorageConfigs(Collection<ClusterInstanceStorageConfig> instanceStorageConfigs) {
            this.instanceStorageConfigs = ClusterInstanceStorageConfigsCopier.copy(instanceStorageConfigs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceStorageConfigs(ClusterInstanceStorageConfig... instanceStorageConfigs) {
            instanceStorageConfigs(Arrays.asList(instanceStorageConfigs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceStorageConfigs(Consumer<ClusterInstanceStorageConfig.Builder>... instanceStorageConfigs) {
            instanceStorageConfigs(Stream.of(instanceStorageConfigs)
                    .map(c -> ClusterInstanceStorageConfig.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getOnStartDeepHealthChecks() {
            if (onStartDeepHealthChecks instanceof SdkAutoConstructList) {
                return null;
            }
            return onStartDeepHealthChecks;
        }

        public final void setOnStartDeepHealthChecks(Collection<String> onStartDeepHealthChecks) {
            this.onStartDeepHealthChecks = OnStartDeepHealthChecksCopier.copy(onStartDeepHealthChecks);
        }

        @Override
        public final Builder onStartDeepHealthChecksWithStrings(Collection<String> onStartDeepHealthChecks) {
            this.onStartDeepHealthChecks = OnStartDeepHealthChecksCopier.copy(onStartDeepHealthChecks);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder onStartDeepHealthChecksWithStrings(String... onStartDeepHealthChecks) {
            onStartDeepHealthChecksWithStrings(Arrays.asList(onStartDeepHealthChecks));
            return this;
        }

        @Override
        public final Builder onStartDeepHealthChecks(Collection<DeepHealthCheckType> onStartDeepHealthChecks) {
            this.onStartDeepHealthChecks = OnStartDeepHealthChecksCopier.copyEnumToString(onStartDeepHealthChecks);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder onStartDeepHealthChecks(DeepHealthCheckType... onStartDeepHealthChecks) {
            onStartDeepHealthChecks(Arrays.asList(onStartDeepHealthChecks));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(InstanceGroupStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getTrainingPlanArn() {
            return trainingPlanArn;
        }

        public final void setTrainingPlanArn(String trainingPlanArn) {
            this.trainingPlanArn = trainingPlanArn;
        }

        @Override
        public final Builder trainingPlanArn(String trainingPlanArn) {
            this.trainingPlanArn = trainingPlanArn;
            return this;
        }

        public final String getTrainingPlanStatus() {
            return trainingPlanStatus;
        }

        public final void setTrainingPlanStatus(String trainingPlanStatus) {
            this.trainingPlanStatus = trainingPlanStatus;
        }

        @Override
        public final Builder trainingPlanStatus(String trainingPlanStatus) {
            this.trainingPlanStatus = trainingPlanStatus;
            return this;
        }

        public final VpcConfig.Builder getOverrideVpcConfig() {
            return overrideVpcConfig != null ? overrideVpcConfig.toBuilder() : null;
        }

        public final void setOverrideVpcConfig(VpcConfig.BuilderImpl overrideVpcConfig) {
            this.overrideVpcConfig = overrideVpcConfig != null ? overrideVpcConfig.build() : null;
        }

        @Override
        public final Builder overrideVpcConfig(VpcConfig overrideVpcConfig) {
            this.overrideVpcConfig = overrideVpcConfig;
            return this;
        }

        public final ScheduledUpdateConfig.Builder getScheduledUpdateConfig() {
            return scheduledUpdateConfig != null ? scheduledUpdateConfig.toBuilder() : null;
        }

        public final void setScheduledUpdateConfig(ScheduledUpdateConfig.BuilderImpl scheduledUpdateConfig) {
            this.scheduledUpdateConfig = scheduledUpdateConfig != null ? scheduledUpdateConfig.build() : null;
        }

        @Override
        public final Builder scheduledUpdateConfig(ScheduledUpdateConfig scheduledUpdateConfig) {
            this.scheduledUpdateConfig = scheduledUpdateConfig;
            return this;
        }

        @Override
        public ClusterInstanceGroupDetails build() {
            return new ClusterInstanceGroupDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
