/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines the input needed to run a transform job using the inference specification specified in the algorithm.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TransformJobDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<TransformJobDefinition.Builder, TransformJobDefinition> {
    private static final SdkField<Integer> MAX_CONCURRENT_TRANSFORMS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxConcurrentTransforms").getter(getter(TransformJobDefinition::maxConcurrentTransforms))
            .setter(setter(Builder::maxConcurrentTransforms))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConcurrentTransforms").build())
            .build();

    private static final SdkField<Integer> MAX_PAYLOAD_IN_MB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxPayloadInMB").getter(getter(TransformJobDefinition::maxPayloadInMB))
            .setter(setter(Builder::maxPayloadInMB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxPayloadInMB").build()).build();

    private static final SdkField<String> BATCH_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BatchStrategy").getter(getter(TransformJobDefinition::batchStrategyAsString))
            .setter(setter(Builder::batchStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BatchStrategy").build()).build();

    private static final SdkField<Map<String, String>> ENVIRONMENT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Environment")
            .getter(getter(TransformJobDefinition::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Environment").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<TransformInput> TRANSFORM_INPUT_FIELD = SdkField
            .<TransformInput> builder(MarshallingType.SDK_POJO).memberName("TransformInput")
            .getter(getter(TransformJobDefinition::transformInput)).setter(setter(Builder::transformInput))
            .constructor(TransformInput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransformInput").build()).build();

    private static final SdkField<TransformOutput> TRANSFORM_OUTPUT_FIELD = SdkField
            .<TransformOutput> builder(MarshallingType.SDK_POJO).memberName("TransformOutput")
            .getter(getter(TransformJobDefinition::transformOutput)).setter(setter(Builder::transformOutput))
            .constructor(TransformOutput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransformOutput").build()).build();

    private static final SdkField<TransformResources> TRANSFORM_RESOURCES_FIELD = SdkField
            .<TransformResources> builder(MarshallingType.SDK_POJO).memberName("TransformResources")
            .getter(getter(TransformJobDefinition::transformResources)).setter(setter(Builder::transformResources))
            .constructor(TransformResources::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransformResources").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            MAX_CONCURRENT_TRANSFORMS_FIELD, MAX_PAYLOAD_IN_MB_FIELD, BATCH_STRATEGY_FIELD, ENVIRONMENT_FIELD,
            TRANSFORM_INPUT_FIELD, TRANSFORM_OUTPUT_FIELD, TRANSFORM_RESOURCES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer maxConcurrentTransforms;

    private final Integer maxPayloadInMB;

    private final String batchStrategy;

    private final Map<String, String> environment;

    private final TransformInput transformInput;

    private final TransformOutput transformOutput;

    private final TransformResources transformResources;

    private TransformJobDefinition(BuilderImpl builder) {
        this.maxConcurrentTransforms = builder.maxConcurrentTransforms;
        this.maxPayloadInMB = builder.maxPayloadInMB;
        this.batchStrategy = builder.batchStrategy;
        this.environment = builder.environment;
        this.transformInput = builder.transformInput;
        this.transformOutput = builder.transformOutput;
        this.transformResources = builder.transformResources;
    }

    /**
     * <p>
     * The maximum number of parallel requests that can be sent to each instance in a transform job. The default value
     * is 1.
     * </p>
     * 
     * @return The maximum number of parallel requests that can be sent to each instance in a transform job. The default
     *         value is 1.
     */
    public final Integer maxConcurrentTransforms() {
        return maxConcurrentTransforms;
    }

    /**
     * <p>
     * The maximum payload size allowed, in MB. A payload is the data portion of a record (without metadata).
     * </p>
     * 
     * @return The maximum payload size allowed, in MB. A payload is the data portion of a record (without metadata).
     */
    public final Integer maxPayloadInMB() {
        return maxPayloadInMB;
    }

    /**
     * <p>
     * A string that determines the number of records included in a single mini-batch.
     * </p>
     * <p>
     * <code>SingleRecord</code> means only one record is used per mini-batch. <code>MultiRecord</code> means a
     * mini-batch is set to contain as many records that can fit within the <code>MaxPayloadInMB</code> limit.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #batchStrategy}
     * will return {@link BatchStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #batchStrategyAsString}.
     * </p>
     * 
     * @return A string that determines the number of records included in a single mini-batch.</p>
     *         <p>
     *         <code>SingleRecord</code> means only one record is used per mini-batch. <code>MultiRecord</code> means a
     *         mini-batch is set to contain as many records that can fit within the <code>MaxPayloadInMB</code> limit.
     * @see BatchStrategy
     */
    public final BatchStrategy batchStrategy() {
        return BatchStrategy.fromValue(batchStrategy);
    }

    /**
     * <p>
     * A string that determines the number of records included in a single mini-batch.
     * </p>
     * <p>
     * <code>SingleRecord</code> means only one record is used per mini-batch. <code>MultiRecord</code> means a
     * mini-batch is set to contain as many records that can fit within the <code>MaxPayloadInMB</code> limit.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #batchStrategy}
     * will return {@link BatchStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #batchStrategyAsString}.
     * </p>
     * 
     * @return A string that determines the number of records included in a single mini-batch.</p>
     *         <p>
     *         <code>SingleRecord</code> means only one record is used per mini-batch. <code>MultiRecord</code> means a
     *         mini-batch is set to contain as many records that can fit within the <code>MaxPayloadInMB</code> limit.
     * @see BatchStrategy
     */
    public final String batchStrategyAsString() {
        return batchStrategy;
    }

    /**
     * For responses, this returns true if the service returned a value for the Environment property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The environment variables to set in the Docker container. We support up to 16 key and values entries in the map.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironment} method.
     * </p>
     * 
     * @return The environment variables to set in the Docker container. We support up to 16 key and values entries in
     *         the map.
     */
    public final Map<String, String> environment() {
        return environment;
    }

    /**
     * <p>
     * A description of the input source and the way the transform job consumes it.
     * </p>
     * 
     * @return A description of the input source and the way the transform job consumes it.
     */
    public final TransformInput transformInput() {
        return transformInput;
    }

    /**
     * <p>
     * Identifies the Amazon S3 location where you want Amazon SageMaker to save the results from the transform job.
     * </p>
     * 
     * @return Identifies the Amazon S3 location where you want Amazon SageMaker to save the results from the transform
     *         job.
     */
    public final TransformOutput transformOutput() {
        return transformOutput;
    }

    /**
     * <p>
     * Identifies the ML compute instances for the transform job.
     * </p>
     * 
     * @return Identifies the ML compute instances for the transform job.
     */
    public final TransformResources transformResources() {
        return transformResources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrentTransforms());
        hashCode = 31 * hashCode + Objects.hashCode(maxPayloadInMB());
        hashCode = 31 * hashCode + Objects.hashCode(batchStrategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(transformInput());
        hashCode = 31 * hashCode + Objects.hashCode(transformOutput());
        hashCode = 31 * hashCode + Objects.hashCode(transformResources());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TransformJobDefinition)) {
            return false;
        }
        TransformJobDefinition other = (TransformJobDefinition) obj;
        return Objects.equals(maxConcurrentTransforms(), other.maxConcurrentTransforms())
                && Objects.equals(maxPayloadInMB(), other.maxPayloadInMB())
                && Objects.equals(batchStrategyAsString(), other.batchStrategyAsString())
                && hasEnvironment() == other.hasEnvironment() && Objects.equals(environment(), other.environment())
                && Objects.equals(transformInput(), other.transformInput())
                && Objects.equals(transformOutput(), other.transformOutput())
                && Objects.equals(transformResources(), other.transformResources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TransformJobDefinition").add("MaxConcurrentTransforms", maxConcurrentTransforms())
                .add("MaxPayloadInMB", maxPayloadInMB()).add("BatchStrategy", batchStrategyAsString())
                .add("Environment", hasEnvironment() ? environment() : null).add("TransformInput", transformInput())
                .add("TransformOutput", transformOutput()).add("TransformResources", transformResources()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MaxConcurrentTransforms":
            return Optional.ofNullable(clazz.cast(maxConcurrentTransforms()));
        case "MaxPayloadInMB":
            return Optional.ofNullable(clazz.cast(maxPayloadInMB()));
        case "BatchStrategy":
            return Optional.ofNullable(clazz.cast(batchStrategyAsString()));
        case "Environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "TransformInput":
            return Optional.ofNullable(clazz.cast(transformInput()));
        case "TransformOutput":
            return Optional.ofNullable(clazz.cast(transformOutput()));
        case "TransformResources":
            return Optional.ofNullable(clazz.cast(transformResources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("MaxConcurrentTransforms", MAX_CONCURRENT_TRANSFORMS_FIELD);
        map.put("MaxPayloadInMB", MAX_PAYLOAD_IN_MB_FIELD);
        map.put("BatchStrategy", BATCH_STRATEGY_FIELD);
        map.put("Environment", ENVIRONMENT_FIELD);
        map.put("TransformInput", TRANSFORM_INPUT_FIELD);
        map.put("TransformOutput", TRANSFORM_OUTPUT_FIELD);
        map.put("TransformResources", TRANSFORM_RESOURCES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TransformJobDefinition, T> g) {
        return obj -> g.apply((TransformJobDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TransformJobDefinition> {
        /**
         * <p>
         * The maximum number of parallel requests that can be sent to each instance in a transform job. The default
         * value is 1.
         * </p>
         * 
         * @param maxConcurrentTransforms
         *        The maximum number of parallel requests that can be sent to each instance in a transform job. The
         *        default value is 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrentTransforms(Integer maxConcurrentTransforms);

        /**
         * <p>
         * The maximum payload size allowed, in MB. A payload is the data portion of a record (without metadata).
         * </p>
         * 
         * @param maxPayloadInMB
         *        The maximum payload size allowed, in MB. A payload is the data portion of a record (without metadata).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxPayloadInMB(Integer maxPayloadInMB);

        /**
         * <p>
         * A string that determines the number of records included in a single mini-batch.
         * </p>
         * <p>
         * <code>SingleRecord</code> means only one record is used per mini-batch. <code>MultiRecord</code> means a
         * mini-batch is set to contain as many records that can fit within the <code>MaxPayloadInMB</code> limit.
         * </p>
         * 
         * @param batchStrategy
         *        A string that determines the number of records included in a single mini-batch.</p>
         *        <p>
         *        <code>SingleRecord</code> means only one record is used per mini-batch. <code>MultiRecord</code> means
         *        a mini-batch is set to contain as many records that can fit within the <code>MaxPayloadInMB</code>
         *        limit.
         * @see BatchStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BatchStrategy
         */
        Builder batchStrategy(String batchStrategy);

        /**
         * <p>
         * A string that determines the number of records included in a single mini-batch.
         * </p>
         * <p>
         * <code>SingleRecord</code> means only one record is used per mini-batch. <code>MultiRecord</code> means a
         * mini-batch is set to contain as many records that can fit within the <code>MaxPayloadInMB</code> limit.
         * </p>
         * 
         * @param batchStrategy
         *        A string that determines the number of records included in a single mini-batch.</p>
         *        <p>
         *        <code>SingleRecord</code> means only one record is used per mini-batch. <code>MultiRecord</code> means
         *        a mini-batch is set to contain as many records that can fit within the <code>MaxPayloadInMB</code>
         *        limit.
         * @see BatchStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BatchStrategy
         */
        Builder batchStrategy(BatchStrategy batchStrategy);

        /**
         * <p>
         * The environment variables to set in the Docker container. We support up to 16 key and values entries in the
         * map.
         * </p>
         * 
         * @param environment
         *        The environment variables to set in the Docker container. We support up to 16 key and values entries
         *        in the map.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Map<String, String> environment);

        /**
         * <p>
         * A description of the input source and the way the transform job consumes it.
         * </p>
         * 
         * @param transformInput
         *        A description of the input source and the way the transform job consumes it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transformInput(TransformInput transformInput);

        /**
         * <p>
         * A description of the input source and the way the transform job consumes it.
         * </p>
         * This is a convenience method that creates an instance of the {@link TransformInput.Builder} avoiding the need
         * to create one manually via {@link TransformInput#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TransformInput.Builder#build()} is called immediately and its
         * result is passed to {@link #transformInput(TransformInput)}.
         * 
         * @param transformInput
         *        a consumer that will call methods on {@link TransformInput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transformInput(TransformInput)
         */
        default Builder transformInput(Consumer<TransformInput.Builder> transformInput) {
            return transformInput(TransformInput.builder().applyMutation(transformInput).build());
        }

        /**
         * <p>
         * Identifies the Amazon S3 location where you want Amazon SageMaker to save the results from the transform job.
         * </p>
         * 
         * @param transformOutput
         *        Identifies the Amazon S3 location where you want Amazon SageMaker to save the results from the
         *        transform job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transformOutput(TransformOutput transformOutput);

        /**
         * <p>
         * Identifies the Amazon S3 location where you want Amazon SageMaker to save the results from the transform job.
         * </p>
         * This is a convenience method that creates an instance of the {@link TransformOutput.Builder} avoiding the
         * need to create one manually via {@link TransformOutput#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TransformOutput.Builder#build()} is called immediately and its
         * result is passed to {@link #transformOutput(TransformOutput)}.
         * 
         * @param transformOutput
         *        a consumer that will call methods on {@link TransformOutput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transformOutput(TransformOutput)
         */
        default Builder transformOutput(Consumer<TransformOutput.Builder> transformOutput) {
            return transformOutput(TransformOutput.builder().applyMutation(transformOutput).build());
        }

        /**
         * <p>
         * Identifies the ML compute instances for the transform job.
         * </p>
         * 
         * @param transformResources
         *        Identifies the ML compute instances for the transform job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transformResources(TransformResources transformResources);

        /**
         * <p>
         * Identifies the ML compute instances for the transform job.
         * </p>
         * This is a convenience method that creates an instance of the {@link TransformResources.Builder} avoiding the
         * need to create one manually via {@link TransformResources#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TransformResources.Builder#build()} is called immediately and its
         * result is passed to {@link #transformResources(TransformResources)}.
         * 
         * @param transformResources
         *        a consumer that will call methods on {@link TransformResources.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transformResources(TransformResources)
         */
        default Builder transformResources(Consumer<TransformResources.Builder> transformResources) {
            return transformResources(TransformResources.builder().applyMutation(transformResources).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Integer maxConcurrentTransforms;

        private Integer maxPayloadInMB;

        private String batchStrategy;

        private Map<String, String> environment = DefaultSdkAutoConstructMap.getInstance();

        private TransformInput transformInput;

        private TransformOutput transformOutput;

        private TransformResources transformResources;

        private BuilderImpl() {
        }

        private BuilderImpl(TransformJobDefinition model) {
            maxConcurrentTransforms(model.maxConcurrentTransforms);
            maxPayloadInMB(model.maxPayloadInMB);
            batchStrategy(model.batchStrategy);
            environment(model.environment);
            transformInput(model.transformInput);
            transformOutput(model.transformOutput);
            transformResources(model.transformResources);
        }

        public final Integer getMaxConcurrentTransforms() {
            return maxConcurrentTransforms;
        }

        public final void setMaxConcurrentTransforms(Integer maxConcurrentTransforms) {
            this.maxConcurrentTransforms = maxConcurrentTransforms;
        }

        @Override
        public final Builder maxConcurrentTransforms(Integer maxConcurrentTransforms) {
            this.maxConcurrentTransforms = maxConcurrentTransforms;
            return this;
        }

        public final Integer getMaxPayloadInMB() {
            return maxPayloadInMB;
        }

        public final void setMaxPayloadInMB(Integer maxPayloadInMB) {
            this.maxPayloadInMB = maxPayloadInMB;
        }

        @Override
        public final Builder maxPayloadInMB(Integer maxPayloadInMB) {
            this.maxPayloadInMB = maxPayloadInMB;
            return this;
        }

        public final String getBatchStrategy() {
            return batchStrategy;
        }

        public final void setBatchStrategy(String batchStrategy) {
            this.batchStrategy = batchStrategy;
        }

        @Override
        public final Builder batchStrategy(String batchStrategy) {
            this.batchStrategy = batchStrategy;
            return this;
        }

        @Override
        public final Builder batchStrategy(BatchStrategy batchStrategy) {
            this.batchStrategy(batchStrategy == null ? null : batchStrategy.toString());
            return this;
        }

        public final Map<String, String> getEnvironment() {
            if (environment instanceof SdkAutoConstructMap) {
                return null;
            }
            return environment;
        }

        public final void setEnvironment(Map<String, String> environment) {
            this.environment = TransformEnvironmentMapCopier.copy(environment);
        }

        @Override
        public final Builder environment(Map<String, String> environment) {
            this.environment = TransformEnvironmentMapCopier.copy(environment);
            return this;
        }

        public final TransformInput.Builder getTransformInput() {
            return transformInput != null ? transformInput.toBuilder() : null;
        }

        public final void setTransformInput(TransformInput.BuilderImpl transformInput) {
            this.transformInput = transformInput != null ? transformInput.build() : null;
        }

        @Override
        public final Builder transformInput(TransformInput transformInput) {
            this.transformInput = transformInput;
            return this;
        }

        public final TransformOutput.Builder getTransformOutput() {
            return transformOutput != null ? transformOutput.toBuilder() : null;
        }

        public final void setTransformOutput(TransformOutput.BuilderImpl transformOutput) {
            this.transformOutput = transformOutput != null ? transformOutput.build() : null;
        }

        @Override
        public final Builder transformOutput(TransformOutput transformOutput) {
            this.transformOutput = transformOutput;
            return this;
        }

        public final TransformResources.Builder getTransformResources() {
            return transformResources != null ? transformResources.toBuilder() : null;
        }

        public final void setTransformResources(TransformResources.BuilderImpl transformResources) {
            this.transformResources = transformResources != null ? transformResources.build() : null;
        }

        @Override
        public final Builder transformResources(TransformResources transformResources) {
            this.transformResources = transformResources;
            return this;
        }

        @Override
        public TransformJobDefinition build() {
            return new TransformJobDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
