/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an available upgrade for a SageMaker Partner AI App, including the version number and
 * release notes.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AvailableUpgrade implements SdkPojo, Serializable,
        ToCopyableBuilder<AvailableUpgrade.Builder, AvailableUpgrade> {
    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(AvailableUpgrade::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<List<String>> RELEASE_NOTES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ReleaseNotes")
            .getter(getter(AvailableUpgrade::releaseNotes))
            .setter(setter(Builder::releaseNotes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReleaseNotes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VERSION_FIELD,
            RELEASE_NOTES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String version;

    private final List<String> releaseNotes;

    private AvailableUpgrade(BuilderImpl builder) {
        this.version = builder.version;
        this.releaseNotes = builder.releaseNotes;
    }

    /**
     * <p>
     * The semantic version number of the available upgrade for the SageMaker Partner AI App.
     * </p>
     * 
     * @return The semantic version number of the available upgrade for the SageMaker Partner AI App.
     */
    public final String version() {
        return version;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReleaseNotes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasReleaseNotes() {
        return releaseNotes != null && !(releaseNotes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of release notes describing the changes and improvements included in the available upgrade version.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReleaseNotes} method.
     * </p>
     * 
     * @return A list of release notes describing the changes and improvements included in the available upgrade
     *         version.
     */
    public final List<String> releaseNotes() {
        return releaseNotes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(hasReleaseNotes() ? releaseNotes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AvailableUpgrade)) {
            return false;
        }
        AvailableUpgrade other = (AvailableUpgrade) obj;
        return Objects.equals(version(), other.version()) && hasReleaseNotes() == other.hasReleaseNotes()
                && Objects.equals(releaseNotes(), other.releaseNotes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AvailableUpgrade").add("Version", version())
                .add("ReleaseNotes", hasReleaseNotes() ? releaseNotes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "ReleaseNotes":
            return Optional.ofNullable(clazz.cast(releaseNotes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Version", VERSION_FIELD);
        map.put("ReleaseNotes", RELEASE_NOTES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AvailableUpgrade, T> g) {
        return obj -> g.apply((AvailableUpgrade) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AvailableUpgrade> {
        /**
         * <p>
         * The semantic version number of the available upgrade for the SageMaker Partner AI App.
         * </p>
         * 
         * @param version
         *        The semantic version number of the available upgrade for the SageMaker Partner AI App.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * A list of release notes describing the changes and improvements included in the available upgrade version.
         * </p>
         * 
         * @param releaseNotes
         *        A list of release notes describing the changes and improvements included in the available upgrade
         *        version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder releaseNotes(Collection<String> releaseNotes);

        /**
         * <p>
         * A list of release notes describing the changes and improvements included in the available upgrade version.
         * </p>
         * 
         * @param releaseNotes
         *        A list of release notes describing the changes and improvements included in the available upgrade
         *        version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder releaseNotes(String... releaseNotes);
    }

    static final class BuilderImpl implements Builder {
        private String version;

        private List<String> releaseNotes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AvailableUpgrade model) {
            version(model.version);
            releaseNotes(model.releaseNotes);
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final Collection<String> getReleaseNotes() {
            if (releaseNotes instanceof SdkAutoConstructList) {
                return null;
            }
            return releaseNotes;
        }

        public final void setReleaseNotes(Collection<String> releaseNotes) {
            this.releaseNotes = ReleaseNotesListCopier.copy(releaseNotes);
        }

        @Override
        public final Builder releaseNotes(Collection<String> releaseNotes) {
            this.releaseNotes = ReleaseNotesListCopier.copy(releaseNotes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder releaseNotes(String... releaseNotes) {
            releaseNotes(Arrays.asList(releaseNotes));
            return this;
        }

        @Override
        public AvailableUpgrade build() {
            return new AvailableUpgrade(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
