/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Metadata information about a HyperPod cluster showing information about the cluster level operations, such as
 * creating, updating, and deleting.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ClusterMetadata implements SdkPojo, Serializable, ToCopyableBuilder<ClusterMetadata.Builder, ClusterMetadata> {
    private static final SdkField<String> FAILURE_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureMessage").getter(getter(ClusterMetadata::failureMessage)).setter(setter(Builder::failureMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureMessage").build()).build();

    private static final SdkField<List<String>> EKS_ROLE_ACCESS_ENTRIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EksRoleAccessEntries")
            .getter(getter(ClusterMetadata::eksRoleAccessEntries))
            .setter(setter(Builder::eksRoleAccessEntries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EksRoleAccessEntries").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SLR_ACCESS_ENTRY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SlrAccessEntry").getter(getter(ClusterMetadata::slrAccessEntry)).setter(setter(Builder::slrAccessEntry))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SlrAccessEntry").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FAILURE_MESSAGE_FIELD,
            EKS_ROLE_ACCESS_ENTRIES_FIELD, SLR_ACCESS_ENTRY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String failureMessage;

    private final List<String> eksRoleAccessEntries;

    private final String slrAccessEntry;

    private ClusterMetadata(BuilderImpl builder) {
        this.failureMessage = builder.failureMessage;
        this.eksRoleAccessEntries = builder.eksRoleAccessEntries;
        this.slrAccessEntry = builder.slrAccessEntry;
    }

    /**
     * <p>
     * An error message describing why the cluster level operation (such as creating, updating, or deleting) failed.
     * </p>
     * 
     * @return An error message describing why the cluster level operation (such as creating, updating, or deleting)
     *         failed.
     */
    public final String failureMessage() {
        return failureMessage;
    }

    /**
     * For responses, this returns true if the service returned a value for the EksRoleAccessEntries property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEksRoleAccessEntries() {
        return eksRoleAccessEntries != null && !(eksRoleAccessEntries instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of Amazon EKS IAM role ARNs associated with the cluster. This is created by HyperPod on your behalf and
     * only applies for EKS orchestrated clusters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEksRoleAccessEntries} method.
     * </p>
     * 
     * @return A list of Amazon EKS IAM role ARNs associated with the cluster. This is created by HyperPod on your
     *         behalf and only applies for EKS orchestrated clusters.
     */
    public final List<String> eksRoleAccessEntries() {
        return eksRoleAccessEntries;
    }

    /**
     * <p>
     * The Service-Linked Role (SLR) associated with the cluster. This is created by HyperPod on your behalf and only
     * applies for EKS orchestrated clusters.
     * </p>
     * 
     * @return The Service-Linked Role (SLR) associated with the cluster. This is created by HyperPod on your behalf and
     *         only applies for EKS orchestrated clusters.
     */
    public final String slrAccessEntry() {
        return slrAccessEntry;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(failureMessage());
        hashCode = 31 * hashCode + Objects.hashCode(hasEksRoleAccessEntries() ? eksRoleAccessEntries() : null);
        hashCode = 31 * hashCode + Objects.hashCode(slrAccessEntry());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClusterMetadata)) {
            return false;
        }
        ClusterMetadata other = (ClusterMetadata) obj;
        return Objects.equals(failureMessage(), other.failureMessage())
                && hasEksRoleAccessEntries() == other.hasEksRoleAccessEntries()
                && Objects.equals(eksRoleAccessEntries(), other.eksRoleAccessEntries())
                && Objects.equals(slrAccessEntry(), other.slrAccessEntry());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ClusterMetadata").add("FailureMessage", failureMessage())
                .add("EksRoleAccessEntries", hasEksRoleAccessEntries() ? eksRoleAccessEntries() : null)
                .add("SlrAccessEntry", slrAccessEntry()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FailureMessage":
            return Optional.ofNullable(clazz.cast(failureMessage()));
        case "EksRoleAccessEntries":
            return Optional.ofNullable(clazz.cast(eksRoleAccessEntries()));
        case "SlrAccessEntry":
            return Optional.ofNullable(clazz.cast(slrAccessEntry()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("FailureMessage", FAILURE_MESSAGE_FIELD);
        map.put("EksRoleAccessEntries", EKS_ROLE_ACCESS_ENTRIES_FIELD);
        map.put("SlrAccessEntry", SLR_ACCESS_ENTRY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ClusterMetadata, T> g) {
        return obj -> g.apply((ClusterMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ClusterMetadata> {
        /**
         * <p>
         * An error message describing why the cluster level operation (such as creating, updating, or deleting) failed.
         * </p>
         * 
         * @param failureMessage
         *        An error message describing why the cluster level operation (such as creating, updating, or deleting)
         *        failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureMessage(String failureMessage);

        /**
         * <p>
         * A list of Amazon EKS IAM role ARNs associated with the cluster. This is created by HyperPod on your behalf
         * and only applies for EKS orchestrated clusters.
         * </p>
         * 
         * @param eksRoleAccessEntries
         *        A list of Amazon EKS IAM role ARNs associated with the cluster. This is created by HyperPod on your
         *        behalf and only applies for EKS orchestrated clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eksRoleAccessEntries(Collection<String> eksRoleAccessEntries);

        /**
         * <p>
         * A list of Amazon EKS IAM role ARNs associated with the cluster. This is created by HyperPod on your behalf
         * and only applies for EKS orchestrated clusters.
         * </p>
         * 
         * @param eksRoleAccessEntries
         *        A list of Amazon EKS IAM role ARNs associated with the cluster. This is created by HyperPod on your
         *        behalf and only applies for EKS orchestrated clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eksRoleAccessEntries(String... eksRoleAccessEntries);

        /**
         * <p>
         * The Service-Linked Role (SLR) associated with the cluster. This is created by HyperPod on your behalf and
         * only applies for EKS orchestrated clusters.
         * </p>
         * 
         * @param slrAccessEntry
         *        The Service-Linked Role (SLR) associated with the cluster. This is created by HyperPod on your behalf
         *        and only applies for EKS orchestrated clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slrAccessEntry(String slrAccessEntry);
    }

    static final class BuilderImpl implements Builder {
        private String failureMessage;

        private List<String> eksRoleAccessEntries = DefaultSdkAutoConstructList.getInstance();

        private String slrAccessEntry;

        private BuilderImpl() {
        }

        private BuilderImpl(ClusterMetadata model) {
            failureMessage(model.failureMessage);
            eksRoleAccessEntries(model.eksRoleAccessEntries);
            slrAccessEntry(model.slrAccessEntry);
        }

        public final String getFailureMessage() {
            return failureMessage;
        }

        public final void setFailureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
        }

        @Override
        public final Builder failureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
            return this;
        }

        public final Collection<String> getEksRoleAccessEntries() {
            if (eksRoleAccessEntries instanceof SdkAutoConstructList) {
                return null;
            }
            return eksRoleAccessEntries;
        }

        public final void setEksRoleAccessEntries(Collection<String> eksRoleAccessEntries) {
            this.eksRoleAccessEntries = EksRoleAccessEntriesCopier.copy(eksRoleAccessEntries);
        }

        @Override
        public final Builder eksRoleAccessEntries(Collection<String> eksRoleAccessEntries) {
            this.eksRoleAccessEntries = EksRoleAccessEntriesCopier.copy(eksRoleAccessEntries);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eksRoleAccessEntries(String... eksRoleAccessEntries) {
            eksRoleAccessEntries(Arrays.asList(eksRoleAccessEntries));
            return this;
        }

        public final String getSlrAccessEntry() {
            return slrAccessEntry;
        }

        public final void setSlrAccessEntry(String slrAccessEntry) {
            this.slrAccessEntry = slrAccessEntry;
        }

        @Override
        public final Builder slrAccessEntry(String slrAccessEntry) {
            this.slrAccessEntry = slrAccessEntry;
            return this;
        }

        @Override
        public ClusterMetadata build() {
            return new ClusterMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
