/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Metadata information about an instance group in a HyperPod cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceGroupMetadata implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceGroupMetadata.Builder, InstanceGroupMetadata> {
    private static final SdkField<String> FAILURE_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureMessage").getter(getter(InstanceGroupMetadata::failureMessage))
            .setter(setter(Builder::failureMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureMessage").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZoneId").getter(getter(InstanceGroupMetadata::availabilityZoneId))
            .setter(setter(Builder::availabilityZoneId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZoneId").build())
            .build();

    private static final SdkField<CapacityReservation> CAPACITY_RESERVATION_FIELD = SdkField
            .<CapacityReservation> builder(MarshallingType.SDK_POJO).memberName("CapacityReservation")
            .getter(getter(InstanceGroupMetadata::capacityReservation)).setter(setter(Builder::capacityReservation))
            .constructor(CapacityReservation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityReservation").build())
            .build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubnetId").getter(getter(InstanceGroupMetadata::subnetId)).setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupIds")
            .getter(getter(InstanceGroupMetadata::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> AMI_OVERRIDE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AmiOverride").getter(getter(InstanceGroupMetadata::amiOverride)).setter(setter(Builder::amiOverride))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AmiOverride").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(FAILURE_MESSAGE_FIELD, AVAILABILITY_ZONE_ID_FIELD, CAPACITY_RESERVATION_FIELD, SUBNET_ID_FIELD,
                    SECURITY_GROUP_IDS_FIELD, AMI_OVERRIDE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String failureMessage;

    private final String availabilityZoneId;

    private final CapacityReservation capacityReservation;

    private final String subnetId;

    private final List<String> securityGroupIds;

    private final String amiOverride;

    private InstanceGroupMetadata(BuilderImpl builder) {
        this.failureMessage = builder.failureMessage;
        this.availabilityZoneId = builder.availabilityZoneId;
        this.capacityReservation = builder.capacityReservation;
        this.subnetId = builder.subnetId;
        this.securityGroupIds = builder.securityGroupIds;
        this.amiOverride = builder.amiOverride;
    }

    /**
     * <p>
     * An error message describing why the instance group level operation (such as creating, scaling, or deleting)
     * failed.
     * </p>
     * 
     * @return An error message describing why the instance group level operation (such as creating, scaling, or
     *         deleting) failed.
     */
    public final String failureMessage() {
        return failureMessage;
    }

    /**
     * <p>
     * The ID of the Availability Zone where the instance group is located.
     * </p>
     * 
     * @return The ID of the Availability Zone where the instance group is located.
     */
    public final String availabilityZoneId() {
        return availabilityZoneId;
    }

    /**
     * <p>
     * Information about the Capacity Reservation used by the instance group.
     * </p>
     * 
     * @return Information about the Capacity Reservation used by the instance group.
     */
    public final CapacityReservation capacityReservation() {
        return capacityReservation;
    }

    /**
     * <p>
     * The ID of the subnet where the instance group is located.
     * </p>
     * 
     * @return The ID of the subnet where the instance group is located.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of security group IDs associated with the instance group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupIds} method.
     * </p>
     * 
     * @return A list of security group IDs associated with the instance group.
     */
    public final List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * <p>
     * If you use a custom Amazon Machine Image (AMI) for the instance group, this field shows the ID of the custom AMI.
     * </p>
     * 
     * @return If you use a custom Amazon Machine Image (AMI) for the instance group, this field shows the ID of the
     *         custom AMI.
     */
    public final String amiOverride() {
        return amiOverride;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(failureMessage());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZoneId());
        hashCode = 31 * hashCode + Objects.hashCode(capacityReservation());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupIds() ? securityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(amiOverride());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceGroupMetadata)) {
            return false;
        }
        InstanceGroupMetadata other = (InstanceGroupMetadata) obj;
        return Objects.equals(failureMessage(), other.failureMessage())
                && Objects.equals(availabilityZoneId(), other.availabilityZoneId())
                && Objects.equals(capacityReservation(), other.capacityReservation())
                && Objects.equals(subnetId(), other.subnetId()) && hasSecurityGroupIds() == other.hasSecurityGroupIds()
                && Objects.equals(securityGroupIds(), other.securityGroupIds())
                && Objects.equals(amiOverride(), other.amiOverride());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceGroupMetadata").add("FailureMessage", failureMessage())
                .add("AvailabilityZoneId", availabilityZoneId()).add("CapacityReservation", capacityReservation())
                .add("SubnetId", subnetId()).add("SecurityGroupIds", hasSecurityGroupIds() ? securityGroupIds() : null)
                .add("AmiOverride", amiOverride()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FailureMessage":
            return Optional.ofNullable(clazz.cast(failureMessage()));
        case "AvailabilityZoneId":
            return Optional.ofNullable(clazz.cast(availabilityZoneId()));
        case "CapacityReservation":
            return Optional.ofNullable(clazz.cast(capacityReservation()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "SecurityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        case "AmiOverride":
            return Optional.ofNullable(clazz.cast(amiOverride()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("FailureMessage", FAILURE_MESSAGE_FIELD);
        map.put("AvailabilityZoneId", AVAILABILITY_ZONE_ID_FIELD);
        map.put("CapacityReservation", CAPACITY_RESERVATION_FIELD);
        map.put("SubnetId", SUBNET_ID_FIELD);
        map.put("SecurityGroupIds", SECURITY_GROUP_IDS_FIELD);
        map.put("AmiOverride", AMI_OVERRIDE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InstanceGroupMetadata, T> g) {
        return obj -> g.apply((InstanceGroupMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceGroupMetadata> {
        /**
         * <p>
         * An error message describing why the instance group level operation (such as creating, scaling, or deleting)
         * failed.
         * </p>
         * 
         * @param failureMessage
         *        An error message describing why the instance group level operation (such as creating, scaling, or
         *        deleting) failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureMessage(String failureMessage);

        /**
         * <p>
         * The ID of the Availability Zone where the instance group is located.
         * </p>
         * 
         * @param availabilityZoneId
         *        The ID of the Availability Zone where the instance group is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneId(String availabilityZoneId);

        /**
         * <p>
         * Information about the Capacity Reservation used by the instance group.
         * </p>
         * 
         * @param capacityReservation
         *        Information about the Capacity Reservation used by the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityReservation(CapacityReservation capacityReservation);

        /**
         * <p>
         * Information about the Capacity Reservation used by the instance group.
         * </p>
         * This is a convenience method that creates an instance of the {@link CapacityReservation.Builder} avoiding the
         * need to create one manually via {@link CapacityReservation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CapacityReservation.Builder#build()} is called immediately and
         * its result is passed to {@link #capacityReservation(CapacityReservation)}.
         * 
         * @param capacityReservation
         *        a consumer that will call methods on {@link CapacityReservation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #capacityReservation(CapacityReservation)
         */
        default Builder capacityReservation(Consumer<CapacityReservation.Builder> capacityReservation) {
            return capacityReservation(CapacityReservation.builder().applyMutation(capacityReservation).build());
        }

        /**
         * <p>
         * The ID of the subnet where the instance group is located.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet where the instance group is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * A list of security group IDs associated with the instance group.
         * </p>
         * 
         * @param securityGroupIds
         *        A list of security group IDs associated with the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * A list of security group IDs associated with the instance group.
         * </p>
         * 
         * @param securityGroupIds
         *        A list of security group IDs associated with the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * If you use a custom Amazon Machine Image (AMI) for the instance group, this field shows the ID of the custom
         * AMI.
         * </p>
         * 
         * @param amiOverride
         *        If you use a custom Amazon Machine Image (AMI) for the instance group, this field shows the ID of the
         *        custom AMI.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder amiOverride(String amiOverride);
    }

    static final class BuilderImpl implements Builder {
        private String failureMessage;

        private String availabilityZoneId;

        private CapacityReservation capacityReservation;

        private String subnetId;

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private String amiOverride;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceGroupMetadata model) {
            failureMessage(model.failureMessage);
            availabilityZoneId(model.availabilityZoneId);
            capacityReservation(model.capacityReservation);
            subnetId(model.subnetId);
            securityGroupIds(model.securityGroupIds);
            amiOverride(model.amiOverride);
        }

        public final String getFailureMessage() {
            return failureMessage;
        }

        public final void setFailureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
        }

        @Override
        public final Builder failureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
            return this;
        }

        public final String getAvailabilityZoneId() {
            return availabilityZoneId;
        }

        public final void setAvailabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
        }

        @Override
        public final Builder availabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
            return this;
        }

        public final CapacityReservation.Builder getCapacityReservation() {
            return capacityReservation != null ? capacityReservation.toBuilder() : null;
        }

        public final void setCapacityReservation(CapacityReservation.BuilderImpl capacityReservation) {
            this.capacityReservation = capacityReservation != null ? capacityReservation.build() : null;
        }

        @Override
        public final Builder capacityReservation(CapacityReservation capacityReservation) {
            this.capacityReservation = capacityReservation;
            return this;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final Collection<String> getSecurityGroupIds() {
            if (securityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupIds;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdsCopier.copy(securityGroupIds);
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdsCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final String getAmiOverride() {
            return amiOverride;
        }

        public final void setAmiOverride(String amiOverride) {
            this.amiOverride = amiOverride;
        }

        @Override
        public final Builder amiOverride(String amiOverride) {
            this.amiOverride = amiOverride;
            return this;
        }

        @Override
        public InstanceGroupMetadata build() {
            return new InstanceGroupMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
