/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides summary information for an SageMaker AI notebook instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NotebookInstanceSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<NotebookInstanceSummary.Builder, NotebookInstanceSummary> {
    private static final SdkField<String> NOTEBOOK_INSTANCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NotebookInstanceName").getter(getter(NotebookInstanceSummary::notebookInstanceName))
            .setter(setter(Builder::notebookInstanceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotebookInstanceName").build())
            .build();

    private static final SdkField<String> NOTEBOOK_INSTANCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NotebookInstanceArn").getter(getter(NotebookInstanceSummary::notebookInstanceArn))
            .setter(setter(Builder::notebookInstanceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotebookInstanceArn").build())
            .build();

    private static final SdkField<String> NOTEBOOK_INSTANCE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NotebookInstanceStatus").getter(getter(NotebookInstanceSummary::notebookInstanceStatusAsString))
            .setter(setter(Builder::notebookInstanceStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotebookInstanceStatus").build())
            .build();

    private static final SdkField<String> URL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Url")
            .getter(getter(NotebookInstanceSummary::url)).setter(setter(Builder::url))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Url").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(NotebookInstanceSummary::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(NotebookInstanceSummary::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedTime").getter(getter(NotebookInstanceSummary::lastModifiedTime))
            .setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime").build()).build();

    private static final SdkField<String> NOTEBOOK_INSTANCE_LIFECYCLE_CONFIG_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NotebookInstanceLifecycleConfigName")
            .getter(getter(NotebookInstanceSummary::notebookInstanceLifecycleConfigName))
            .setter(setter(Builder::notebookInstanceLifecycleConfigName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("NotebookInstanceLifecycleConfigName").build()).build();

    private static final SdkField<String> DEFAULT_CODE_REPOSITORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DefaultCodeRepository").getter(getter(NotebookInstanceSummary::defaultCodeRepository))
            .setter(setter(Builder::defaultCodeRepository))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultCodeRepository").build())
            .build();

    private static final SdkField<List<String>> ADDITIONAL_CODE_REPOSITORIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AdditionalCodeRepositories")
            .getter(getter(NotebookInstanceSummary::additionalCodeRepositories))
            .setter(setter(Builder::additionalCodeRepositories))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalCodeRepositories").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NOTEBOOK_INSTANCE_NAME_FIELD,
            NOTEBOOK_INSTANCE_ARN_FIELD, NOTEBOOK_INSTANCE_STATUS_FIELD, URL_FIELD, INSTANCE_TYPE_FIELD, CREATION_TIME_FIELD,
            LAST_MODIFIED_TIME_FIELD, NOTEBOOK_INSTANCE_LIFECYCLE_CONFIG_NAME_FIELD, DEFAULT_CODE_REPOSITORY_FIELD,
            ADDITIONAL_CODE_REPOSITORIES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String notebookInstanceName;

    private final String notebookInstanceArn;

    private final String notebookInstanceStatus;

    private final String url;

    private final String instanceType;

    private final Instant creationTime;

    private final Instant lastModifiedTime;

    private final String notebookInstanceLifecycleConfigName;

    private final String defaultCodeRepository;

    private final List<String> additionalCodeRepositories;

    private NotebookInstanceSummary(BuilderImpl builder) {
        this.notebookInstanceName = builder.notebookInstanceName;
        this.notebookInstanceArn = builder.notebookInstanceArn;
        this.notebookInstanceStatus = builder.notebookInstanceStatus;
        this.url = builder.url;
        this.instanceType = builder.instanceType;
        this.creationTime = builder.creationTime;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.notebookInstanceLifecycleConfigName = builder.notebookInstanceLifecycleConfigName;
        this.defaultCodeRepository = builder.defaultCodeRepository;
        this.additionalCodeRepositories = builder.additionalCodeRepositories;
    }

    /**
     * <p>
     * The name of the notebook instance that you want a summary for.
     * </p>
     * 
     * @return The name of the notebook instance that you want a summary for.
     */
    public final String notebookInstanceName() {
        return notebookInstanceName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the notebook instance.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the notebook instance.
     */
    public final String notebookInstanceArn() {
        return notebookInstanceArn;
    }

    /**
     * <p>
     * The status of the notebook instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #notebookInstanceStatus} will return {@link NotebookInstanceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #notebookInstanceStatusAsString}.
     * </p>
     * 
     * @return The status of the notebook instance.
     * @see NotebookInstanceStatus
     */
    public final NotebookInstanceStatus notebookInstanceStatus() {
        return NotebookInstanceStatus.fromValue(notebookInstanceStatus);
    }

    /**
     * <p>
     * The status of the notebook instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #notebookInstanceStatus} will return {@link NotebookInstanceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #notebookInstanceStatusAsString}.
     * </p>
     * 
     * @return The status of the notebook instance.
     * @see NotebookInstanceStatus
     */
    public final String notebookInstanceStatusAsString() {
        return notebookInstanceStatus;
    }

    /**
     * <p>
     * The URL that you use to connect to the Jupyter notebook running in your notebook instance.
     * </p>
     * 
     * @return The URL that you use to connect to the Jupyter notebook running in your notebook instance.
     */
    public final String url() {
        return url;
    }

    /**
     * <p>
     * The type of ML compute instance that the notebook instance is running on.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The type of ML compute instance that the notebook instance is running on.
     * @see InstanceType
     */
    public final InstanceType instanceType() {
        return InstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The type of ML compute instance that the notebook instance is running on.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The type of ML compute instance that the notebook instance is running on.
     * @see InstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * A timestamp that shows when the notebook instance was created.
     * </p>
     * 
     * @return A timestamp that shows when the notebook instance was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * A timestamp that shows when the notebook instance was last modified.
     * </p>
     * 
     * @return A timestamp that shows when the notebook instance was last modified.
     */
    public final Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * <p>
     * The name of a notebook instance lifecycle configuration associated with this notebook instance.
     * </p>
     * <p>
     * For information about notebook instance lifestyle configurations, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebook-lifecycle-config.html">Step 2.1: (Optional)
     * Customize a Notebook Instance</a>.
     * </p>
     * 
     * @return The name of a notebook instance lifecycle configuration associated with this notebook instance.</p>
     *         <p>
     *         For information about notebook instance lifestyle configurations, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebook-lifecycle-config.html">Step 2.1:
     *         (Optional) Customize a Notebook Instance</a>.
     */
    public final String notebookInstanceLifecycleConfigName() {
        return notebookInstanceLifecycleConfigName;
    }

    /**
     * <p>
     * The Git repository associated with the notebook instance as its default code repository. This can be either the
     * name of a Git repository stored as a resource in your account, or the URL of a Git repository in <a
     * href="https://docs.aws.amazon.com/codecommit/latest/userguide/welcome.html">Amazon Web Services CodeCommit</a> or
     * in any other Git repository. When you open a notebook instance, it opens in the directory that contains this
     * repository. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/nbi-git-repo.html">Associating Git Repositories with
     * SageMaker AI Notebook Instances</a>.
     * </p>
     * 
     * @return The Git repository associated with the notebook instance as its default code repository. This can be
     *         either the name of a Git repository stored as a resource in your account, or the URL of a Git repository
     *         in <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/welcome.html">Amazon Web Services
     *         CodeCommit</a> or in any other Git repository. When you open a notebook instance, it opens in the
     *         directory that contains this repository. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/nbi-git-repo.html">Associating Git Repositories
     *         with SageMaker AI Notebook Instances</a>.
     */
    public final String defaultCodeRepository() {
        return defaultCodeRepository;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdditionalCodeRepositories property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAdditionalCodeRepositories() {
        return additionalCodeRepositories != null && !(additionalCodeRepositories instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of up to three Git repositories associated with the notebook instance. These can be either the names of
     * Git repositories stored as resources in your account, or the URL of Git repositories in <a
     * href="https://docs.aws.amazon.com/codecommit/latest/userguide/welcome.html">Amazon Web Services CodeCommit</a> or
     * in any other Git repository. These repositories are cloned at the same level as the default repository of your
     * notebook instance. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/nbi-git-repo.html">Associating Git Repositories with
     * SageMaker AI Notebook Instances</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdditionalCodeRepositories} method.
     * </p>
     * 
     * @return An array of up to three Git repositories associated with the notebook instance. These can be either the
     *         names of Git repositories stored as resources in your account, or the URL of Git repositories in <a
     *         href="https://docs.aws.amazon.com/codecommit/latest/userguide/welcome.html">Amazon Web Services
     *         CodeCommit</a> or in any other Git repository. These repositories are cloned at the same level as the
     *         default repository of your notebook instance. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/nbi-git-repo.html">Associating Git Repositories
     *         with SageMaker AI Notebook Instances</a>.
     */
    public final List<String> additionalCodeRepositories() {
        return additionalCodeRepositories;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(notebookInstanceName());
        hashCode = 31 * hashCode + Objects.hashCode(notebookInstanceArn());
        hashCode = 31 * hashCode + Objects.hashCode(notebookInstanceStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(url());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(notebookInstanceLifecycleConfigName());
        hashCode = 31 * hashCode + Objects.hashCode(defaultCodeRepository());
        hashCode = 31 * hashCode + Objects.hashCode(hasAdditionalCodeRepositories() ? additionalCodeRepositories() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NotebookInstanceSummary)) {
            return false;
        }
        NotebookInstanceSummary other = (NotebookInstanceSummary) obj;
        return Objects.equals(notebookInstanceName(), other.notebookInstanceName())
                && Objects.equals(notebookInstanceArn(), other.notebookInstanceArn())
                && Objects.equals(notebookInstanceStatusAsString(), other.notebookInstanceStatusAsString())
                && Objects.equals(url(), other.url()) && Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && Objects.equals(notebookInstanceLifecycleConfigName(), other.notebookInstanceLifecycleConfigName())
                && Objects.equals(defaultCodeRepository(), other.defaultCodeRepository())
                && hasAdditionalCodeRepositories() == other.hasAdditionalCodeRepositories()
                && Objects.equals(additionalCodeRepositories(), other.additionalCodeRepositories());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NotebookInstanceSummary").add("NotebookInstanceName", notebookInstanceName())
                .add("NotebookInstanceArn", notebookInstanceArn())
                .add("NotebookInstanceStatus", notebookInstanceStatusAsString()).add("Url", url())
                .add("InstanceType", instanceTypeAsString()).add("CreationTime", creationTime())
                .add("LastModifiedTime", lastModifiedTime())
                .add("NotebookInstanceLifecycleConfigName", notebookInstanceLifecycleConfigName())
                .add("DefaultCodeRepository", defaultCodeRepository())
                .add("AdditionalCodeRepositories", hasAdditionalCodeRepositories() ? additionalCodeRepositories() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NotebookInstanceName":
            return Optional.ofNullable(clazz.cast(notebookInstanceName()));
        case "NotebookInstanceArn":
            return Optional.ofNullable(clazz.cast(notebookInstanceArn()));
        case "NotebookInstanceStatus":
            return Optional.ofNullable(clazz.cast(notebookInstanceStatusAsString()));
        case "Url":
            return Optional.ofNullable(clazz.cast(url()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "NotebookInstanceLifecycleConfigName":
            return Optional.ofNullable(clazz.cast(notebookInstanceLifecycleConfigName()));
        case "DefaultCodeRepository":
            return Optional.ofNullable(clazz.cast(defaultCodeRepository()));
        case "AdditionalCodeRepositories":
            return Optional.ofNullable(clazz.cast(additionalCodeRepositories()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("NotebookInstanceName", NOTEBOOK_INSTANCE_NAME_FIELD);
        map.put("NotebookInstanceArn", NOTEBOOK_INSTANCE_ARN_FIELD);
        map.put("NotebookInstanceStatus", NOTEBOOK_INSTANCE_STATUS_FIELD);
        map.put("Url", URL_FIELD);
        map.put("InstanceType", INSTANCE_TYPE_FIELD);
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("LastModifiedTime", LAST_MODIFIED_TIME_FIELD);
        map.put("NotebookInstanceLifecycleConfigName", NOTEBOOK_INSTANCE_LIFECYCLE_CONFIG_NAME_FIELD);
        map.put("DefaultCodeRepository", DEFAULT_CODE_REPOSITORY_FIELD);
        map.put("AdditionalCodeRepositories", ADDITIONAL_CODE_REPOSITORIES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<NotebookInstanceSummary, T> g) {
        return obj -> g.apply((NotebookInstanceSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NotebookInstanceSummary> {
        /**
         * <p>
         * The name of the notebook instance that you want a summary for.
         * </p>
         * 
         * @param notebookInstanceName
         *        The name of the notebook instance that you want a summary for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notebookInstanceName(String notebookInstanceName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the notebook instance.
         * </p>
         * 
         * @param notebookInstanceArn
         *        The Amazon Resource Name (ARN) of the notebook instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notebookInstanceArn(String notebookInstanceArn);

        /**
         * <p>
         * The status of the notebook instance.
         * </p>
         * 
         * @param notebookInstanceStatus
         *        The status of the notebook instance.
         * @see NotebookInstanceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotebookInstanceStatus
         */
        Builder notebookInstanceStatus(String notebookInstanceStatus);

        /**
         * <p>
         * The status of the notebook instance.
         * </p>
         * 
         * @param notebookInstanceStatus
         *        The status of the notebook instance.
         * @see NotebookInstanceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotebookInstanceStatus
         */
        Builder notebookInstanceStatus(NotebookInstanceStatus notebookInstanceStatus);

        /**
         * <p>
         * The URL that you use to connect to the Jupyter notebook running in your notebook instance.
         * </p>
         * 
         * @param url
         *        The URL that you use to connect to the Jupyter notebook running in your notebook instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder url(String url);

        /**
         * <p>
         * The type of ML compute instance that the notebook instance is running on.
         * </p>
         * 
         * @param instanceType
         *        The type of ML compute instance that the notebook instance is running on.
         * @see InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The type of ML compute instance that the notebook instance is running on.
         * </p>
         * 
         * @param instanceType
         *        The type of ML compute instance that the notebook instance is running on.
         * @see InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceType
         */
        Builder instanceType(InstanceType instanceType);

        /**
         * <p>
         * A timestamp that shows when the notebook instance was created.
         * </p>
         * 
         * @param creationTime
         *        A timestamp that shows when the notebook instance was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * A timestamp that shows when the notebook instance was last modified.
         * </p>
         * 
         * @param lastModifiedTime
         *        A timestamp that shows when the notebook instance was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * The name of a notebook instance lifecycle configuration associated with this notebook instance.
         * </p>
         * <p>
         * For information about notebook instance lifestyle configurations, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebook-lifecycle-config.html">Step 2.1: (Optional)
         * Customize a Notebook Instance</a>.
         * </p>
         * 
         * @param notebookInstanceLifecycleConfigName
         *        The name of a notebook instance lifecycle configuration associated with this notebook instance.</p>
         *        <p>
         *        For information about notebook instance lifestyle configurations, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebook-lifecycle-config.html">Step 2.1:
         *        (Optional) Customize a Notebook Instance</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notebookInstanceLifecycleConfigName(String notebookInstanceLifecycleConfigName);

        /**
         * <p>
         * The Git repository associated with the notebook instance as its default code repository. This can be either
         * the name of a Git repository stored as a resource in your account, or the URL of a Git repository in <a
         * href="https://docs.aws.amazon.com/codecommit/latest/userguide/welcome.html">Amazon Web Services
         * CodeCommit</a> or in any other Git repository. When you open a notebook instance, it opens in the directory
         * that contains this repository. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/nbi-git-repo.html">Associating Git Repositories with
         * SageMaker AI Notebook Instances</a>.
         * </p>
         * 
         * @param defaultCodeRepository
         *        The Git repository associated with the notebook instance as its default code repository. This can be
         *        either the name of a Git repository stored as a resource in your account, or the URL of a Git
         *        repository in <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/welcome.html">Amazon
         *        Web Services CodeCommit</a> or in any other Git repository. When you open a notebook instance, it
         *        opens in the directory that contains this repository. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/nbi-git-repo.html">Associating Git Repositories
         *        with SageMaker AI Notebook Instances</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultCodeRepository(String defaultCodeRepository);

        /**
         * <p>
         * An array of up to three Git repositories associated with the notebook instance. These can be either the names
         * of Git repositories stored as resources in your account, or the URL of Git repositories in <a
         * href="https://docs.aws.amazon.com/codecommit/latest/userguide/welcome.html">Amazon Web Services
         * CodeCommit</a> or in any other Git repository. These repositories are cloned at the same level as the default
         * repository of your notebook instance. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/nbi-git-repo.html">Associating Git Repositories with
         * SageMaker AI Notebook Instances</a>.
         * </p>
         * 
         * @param additionalCodeRepositories
         *        An array of up to three Git repositories associated with the notebook instance. These can be either
         *        the names of Git repositories stored as resources in your account, or the URL of Git repositories in
         *        <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/welcome.html">Amazon Web Services
         *        CodeCommit</a> or in any other Git repository. These repositories are cloned at the same level as the
         *        default repository of your notebook instance. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/nbi-git-repo.html">Associating Git Repositories
         *        with SageMaker AI Notebook Instances</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalCodeRepositories(Collection<String> additionalCodeRepositories);

        /**
         * <p>
         * An array of up to three Git repositories associated with the notebook instance. These can be either the names
         * of Git repositories stored as resources in your account, or the URL of Git repositories in <a
         * href="https://docs.aws.amazon.com/codecommit/latest/userguide/welcome.html">Amazon Web Services
         * CodeCommit</a> or in any other Git repository. These repositories are cloned at the same level as the default
         * repository of your notebook instance. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/nbi-git-repo.html">Associating Git Repositories with
         * SageMaker AI Notebook Instances</a>.
         * </p>
         * 
         * @param additionalCodeRepositories
         *        An array of up to three Git repositories associated with the notebook instance. These can be either
         *        the names of Git repositories stored as resources in your account, or the URL of Git repositories in
         *        <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/welcome.html">Amazon Web Services
         *        CodeCommit</a> or in any other Git repository. These repositories are cloned at the same level as the
         *        default repository of your notebook instance. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/nbi-git-repo.html">Associating Git Repositories
         *        with SageMaker AI Notebook Instances</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalCodeRepositories(String... additionalCodeRepositories);
    }

    static final class BuilderImpl implements Builder {
        private String notebookInstanceName;

        private String notebookInstanceArn;

        private String notebookInstanceStatus;

        private String url;

        private String instanceType;

        private Instant creationTime;

        private Instant lastModifiedTime;

        private String notebookInstanceLifecycleConfigName;

        private String defaultCodeRepository;

        private List<String> additionalCodeRepositories = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NotebookInstanceSummary model) {
            notebookInstanceName(model.notebookInstanceName);
            notebookInstanceArn(model.notebookInstanceArn);
            notebookInstanceStatus(model.notebookInstanceStatus);
            url(model.url);
            instanceType(model.instanceType);
            creationTime(model.creationTime);
            lastModifiedTime(model.lastModifiedTime);
            notebookInstanceLifecycleConfigName(model.notebookInstanceLifecycleConfigName);
            defaultCodeRepository(model.defaultCodeRepository);
            additionalCodeRepositories(model.additionalCodeRepositories);
        }

        public final String getNotebookInstanceName() {
            return notebookInstanceName;
        }

        public final void setNotebookInstanceName(String notebookInstanceName) {
            this.notebookInstanceName = notebookInstanceName;
        }

        @Override
        public final Builder notebookInstanceName(String notebookInstanceName) {
            this.notebookInstanceName = notebookInstanceName;
            return this;
        }

        public final String getNotebookInstanceArn() {
            return notebookInstanceArn;
        }

        public final void setNotebookInstanceArn(String notebookInstanceArn) {
            this.notebookInstanceArn = notebookInstanceArn;
        }

        @Override
        public final Builder notebookInstanceArn(String notebookInstanceArn) {
            this.notebookInstanceArn = notebookInstanceArn;
            return this;
        }

        public final String getNotebookInstanceStatus() {
            return notebookInstanceStatus;
        }

        public final void setNotebookInstanceStatus(String notebookInstanceStatus) {
            this.notebookInstanceStatus = notebookInstanceStatus;
        }

        @Override
        public final Builder notebookInstanceStatus(String notebookInstanceStatus) {
            this.notebookInstanceStatus = notebookInstanceStatus;
            return this;
        }

        @Override
        public final Builder notebookInstanceStatus(NotebookInstanceStatus notebookInstanceStatus) {
            this.notebookInstanceStatus(notebookInstanceStatus == null ? null : notebookInstanceStatus.toString());
            return this;
        }

        public final String getUrl() {
            return url;
        }

        public final void setUrl(String url) {
            this.url = url;
        }

        @Override
        public final Builder url(String url) {
            this.url = url;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(InstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final String getNotebookInstanceLifecycleConfigName() {
            return notebookInstanceLifecycleConfigName;
        }

        public final void setNotebookInstanceLifecycleConfigName(String notebookInstanceLifecycleConfigName) {
            this.notebookInstanceLifecycleConfigName = notebookInstanceLifecycleConfigName;
        }

        @Override
        public final Builder notebookInstanceLifecycleConfigName(String notebookInstanceLifecycleConfigName) {
            this.notebookInstanceLifecycleConfigName = notebookInstanceLifecycleConfigName;
            return this;
        }

        public final String getDefaultCodeRepository() {
            return defaultCodeRepository;
        }

        public final void setDefaultCodeRepository(String defaultCodeRepository) {
            this.defaultCodeRepository = defaultCodeRepository;
        }

        @Override
        public final Builder defaultCodeRepository(String defaultCodeRepository) {
            this.defaultCodeRepository = defaultCodeRepository;
            return this;
        }

        public final Collection<String> getAdditionalCodeRepositories() {
            if (additionalCodeRepositories instanceof SdkAutoConstructList) {
                return null;
            }
            return additionalCodeRepositories;
        }

        public final void setAdditionalCodeRepositories(Collection<String> additionalCodeRepositories) {
            this.additionalCodeRepositories = AdditionalCodeRepositoryNamesOrUrlsCopier.copy(additionalCodeRepositories);
        }

        @Override
        public final Builder additionalCodeRepositories(Collection<String> additionalCodeRepositories) {
            this.additionalCodeRepositories = AdditionalCodeRepositoryNamesOrUrlsCopier.copy(additionalCodeRepositories);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder additionalCodeRepositories(String... additionalCodeRepositories) {
            additionalCodeRepositories(Arrays.asList(additionalCodeRepositories));
            return this;
        }

        @Override
        public NotebookInstanceSummary build() {
            return new NotebookInstanceSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
