/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration of an <code>OfflineStore</code>.
 * </p>
 * <p>
 * Provide an <code>OfflineStoreConfig</code> in a request to <code>CreateFeatureGroup</code> to create an
 * <code>OfflineStore</code>.
 * </p>
 * <p>
 * To encrypt an <code>OfflineStore</code> using at rest data encryption, specify Amazon Web Services Key Management
 * Service (KMS) key ID, or <code>KMSKeyId</code>, in <code>S3StorageConfig</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OfflineStoreConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<OfflineStoreConfig.Builder, OfflineStoreConfig> {
    private static final SdkField<S3StorageConfig> S3_STORAGE_CONFIG_FIELD = SdkField
            .<S3StorageConfig> builder(MarshallingType.SDK_POJO).memberName("S3StorageConfig")
            .getter(getter(OfflineStoreConfig::s3StorageConfig)).setter(setter(Builder::s3StorageConfig))
            .constructor(S3StorageConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3StorageConfig").build()).build();

    private static final SdkField<Boolean> DISABLE_GLUE_TABLE_CREATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("DisableGlueTableCreation")
            .getter(getter(OfflineStoreConfig::disableGlueTableCreation)).setter(setter(Builder::disableGlueTableCreation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisableGlueTableCreation").build())
            .build();

    private static final SdkField<DataCatalogConfig> DATA_CATALOG_CONFIG_FIELD = SdkField
            .<DataCatalogConfig> builder(MarshallingType.SDK_POJO).memberName("DataCatalogConfig")
            .getter(getter(OfflineStoreConfig::dataCatalogConfig)).setter(setter(Builder::dataCatalogConfig))
            .constructor(DataCatalogConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataCatalogConfig").build()).build();

    private static final SdkField<String> TABLE_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TableFormat").getter(getter(OfflineStoreConfig::tableFormatAsString))
            .setter(setter(Builder::tableFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TableFormat").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_STORAGE_CONFIG_FIELD,
            DISABLE_GLUE_TABLE_CREATION_FIELD, DATA_CATALOG_CONFIG_FIELD, TABLE_FORMAT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final S3StorageConfig s3StorageConfig;

    private final Boolean disableGlueTableCreation;

    private final DataCatalogConfig dataCatalogConfig;

    private final String tableFormat;

    private OfflineStoreConfig(BuilderImpl builder) {
        this.s3StorageConfig = builder.s3StorageConfig;
        this.disableGlueTableCreation = builder.disableGlueTableCreation;
        this.dataCatalogConfig = builder.dataCatalogConfig;
        this.tableFormat = builder.tableFormat;
    }

    /**
     * <p>
     * The Amazon Simple Storage (Amazon S3) location of <code>OfflineStore</code>.
     * </p>
     * 
     * @return The Amazon Simple Storage (Amazon S3) location of <code>OfflineStore</code>.
     */
    public final S3StorageConfig s3StorageConfig() {
        return s3StorageConfig;
    }

    /**
     * <p>
     * Set to <code>True</code> to disable the automatic creation of an Amazon Web Services Glue table when configuring
     * an <code>OfflineStore</code>. If set to <code>False</code>, Feature Store will name the <code>OfflineStore</code>
     * Glue table following <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/tables-databases-columns-names.html">Athena's naming
     * recommendations</a>.
     * </p>
     * <p>
     * The default value is <code>False</code>.
     * </p>
     * 
     * @return Set to <code>True</code> to disable the automatic creation of an Amazon Web Services Glue table when
     *         configuring an <code>OfflineStore</code>. If set to <code>False</code>, Feature Store will name the
     *         <code>OfflineStore</code> Glue table following <a
     *         href="https://docs.aws.amazon.com/athena/latest/ug/tables-databases-columns-names.html">Athena's naming
     *         recommendations</a>.</p>
     *         <p>
     *         The default value is <code>False</code>.
     */
    public final Boolean disableGlueTableCreation() {
        return disableGlueTableCreation;
    }

    /**
     * <p>
     * The meta data of the Glue table that is autogenerated when an <code>OfflineStore</code> is created.
     * </p>
     * 
     * @return The meta data of the Glue table that is autogenerated when an <code>OfflineStore</code> is created.
     */
    public final DataCatalogConfig dataCatalogConfig() {
        return dataCatalogConfig;
    }

    /**
     * <p>
     * Format for the offline store table. Supported formats are Glue (Default) and <a
     * href="https://iceberg.apache.org/">Apache Iceberg</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tableFormat} will
     * return {@link TableFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tableFormatAsString}.
     * </p>
     * 
     * @return Format for the offline store table. Supported formats are Glue (Default) and <a
     *         href="https://iceberg.apache.org/">Apache Iceberg</a>.
     * @see TableFormat
     */
    public final TableFormat tableFormat() {
        return TableFormat.fromValue(tableFormat);
    }

    /**
     * <p>
     * Format for the offline store table. Supported formats are Glue (Default) and <a
     * href="https://iceberg.apache.org/">Apache Iceberg</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tableFormat} will
     * return {@link TableFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tableFormatAsString}.
     * </p>
     * 
     * @return Format for the offline store table. Supported formats are Glue (Default) and <a
     *         href="https://iceberg.apache.org/">Apache Iceberg</a>.
     * @see TableFormat
     */
    public final String tableFormatAsString() {
        return tableFormat;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3StorageConfig());
        hashCode = 31 * hashCode + Objects.hashCode(disableGlueTableCreation());
        hashCode = 31 * hashCode + Objects.hashCode(dataCatalogConfig());
        hashCode = 31 * hashCode + Objects.hashCode(tableFormatAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OfflineStoreConfig)) {
            return false;
        }
        OfflineStoreConfig other = (OfflineStoreConfig) obj;
        return Objects.equals(s3StorageConfig(), other.s3StorageConfig())
                && Objects.equals(disableGlueTableCreation(), other.disableGlueTableCreation())
                && Objects.equals(dataCatalogConfig(), other.dataCatalogConfig())
                && Objects.equals(tableFormatAsString(), other.tableFormatAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OfflineStoreConfig").add("S3StorageConfig", s3StorageConfig())
                .add("DisableGlueTableCreation", disableGlueTableCreation()).add("DataCatalogConfig", dataCatalogConfig())
                .add("TableFormat", tableFormatAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3StorageConfig":
            return Optional.ofNullable(clazz.cast(s3StorageConfig()));
        case "DisableGlueTableCreation":
            return Optional.ofNullable(clazz.cast(disableGlueTableCreation()));
        case "DataCatalogConfig":
            return Optional.ofNullable(clazz.cast(dataCatalogConfig()));
        case "TableFormat":
            return Optional.ofNullable(clazz.cast(tableFormatAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("S3StorageConfig", S3_STORAGE_CONFIG_FIELD);
        map.put("DisableGlueTableCreation", DISABLE_GLUE_TABLE_CREATION_FIELD);
        map.put("DataCatalogConfig", DATA_CATALOG_CONFIG_FIELD);
        map.put("TableFormat", TABLE_FORMAT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OfflineStoreConfig, T> g) {
        return obj -> g.apply((OfflineStoreConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OfflineStoreConfig> {
        /**
         * <p>
         * The Amazon Simple Storage (Amazon S3) location of <code>OfflineStore</code>.
         * </p>
         * 
         * @param s3StorageConfig
         *        The Amazon Simple Storage (Amazon S3) location of <code>OfflineStore</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3StorageConfig(S3StorageConfig s3StorageConfig);

        /**
         * <p>
         * The Amazon Simple Storage (Amazon S3) location of <code>OfflineStore</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3StorageConfig.Builder} avoiding the
         * need to create one manually via {@link S3StorageConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3StorageConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #s3StorageConfig(S3StorageConfig)}.
         * 
         * @param s3StorageConfig
         *        a consumer that will call methods on {@link S3StorageConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3StorageConfig(S3StorageConfig)
         */
        default Builder s3StorageConfig(Consumer<S3StorageConfig.Builder> s3StorageConfig) {
            return s3StorageConfig(S3StorageConfig.builder().applyMutation(s3StorageConfig).build());
        }

        /**
         * <p>
         * Set to <code>True</code> to disable the automatic creation of an Amazon Web Services Glue table when
         * configuring an <code>OfflineStore</code>. If set to <code>False</code>, Feature Store will name the
         * <code>OfflineStore</code> Glue table following <a
         * href="https://docs.aws.amazon.com/athena/latest/ug/tables-databases-columns-names.html">Athena's naming
         * recommendations</a>.
         * </p>
         * <p>
         * The default value is <code>False</code>.
         * </p>
         * 
         * @param disableGlueTableCreation
         *        Set to <code>True</code> to disable the automatic creation of an Amazon Web Services Glue table when
         *        configuring an <code>OfflineStore</code>. If set to <code>False</code>, Feature Store will name the
         *        <code>OfflineStore</code> Glue table following <a
         *        href="https://docs.aws.amazon.com/athena/latest/ug/tables-databases-columns-names.html">Athena's
         *        naming recommendations</a>.</p>
         *        <p>
         *        The default value is <code>False</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disableGlueTableCreation(Boolean disableGlueTableCreation);

        /**
         * <p>
         * The meta data of the Glue table that is autogenerated when an <code>OfflineStore</code> is created.
         * </p>
         * 
         * @param dataCatalogConfig
         *        The meta data of the Glue table that is autogenerated when an <code>OfflineStore</code> is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataCatalogConfig(DataCatalogConfig dataCatalogConfig);

        /**
         * <p>
         * The meta data of the Glue table that is autogenerated when an <code>OfflineStore</code> is created.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataCatalogConfig.Builder} avoiding the
         * need to create one manually via {@link DataCatalogConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataCatalogConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #dataCatalogConfig(DataCatalogConfig)}.
         * 
         * @param dataCatalogConfig
         *        a consumer that will call methods on {@link DataCatalogConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataCatalogConfig(DataCatalogConfig)
         */
        default Builder dataCatalogConfig(Consumer<DataCatalogConfig.Builder> dataCatalogConfig) {
            return dataCatalogConfig(DataCatalogConfig.builder().applyMutation(dataCatalogConfig).build());
        }

        /**
         * <p>
         * Format for the offline store table. Supported formats are Glue (Default) and <a
         * href="https://iceberg.apache.org/">Apache Iceberg</a>.
         * </p>
         * 
         * @param tableFormat
         *        Format for the offline store table. Supported formats are Glue (Default) and <a
         *        href="https://iceberg.apache.org/">Apache Iceberg</a>.
         * @see TableFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TableFormat
         */
        Builder tableFormat(String tableFormat);

        /**
         * <p>
         * Format for the offline store table. Supported formats are Glue (Default) and <a
         * href="https://iceberg.apache.org/">Apache Iceberg</a>.
         * </p>
         * 
         * @param tableFormat
         *        Format for the offline store table. Supported formats are Glue (Default) and <a
         *        href="https://iceberg.apache.org/">Apache Iceberg</a>.
         * @see TableFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TableFormat
         */
        Builder tableFormat(TableFormat tableFormat);
    }

    static final class BuilderImpl implements Builder {
        private S3StorageConfig s3StorageConfig;

        private Boolean disableGlueTableCreation;

        private DataCatalogConfig dataCatalogConfig;

        private String tableFormat;

        private BuilderImpl() {
        }

        private BuilderImpl(OfflineStoreConfig model) {
            s3StorageConfig(model.s3StorageConfig);
            disableGlueTableCreation(model.disableGlueTableCreation);
            dataCatalogConfig(model.dataCatalogConfig);
            tableFormat(model.tableFormat);
        }

        public final S3StorageConfig.Builder getS3StorageConfig() {
            return s3StorageConfig != null ? s3StorageConfig.toBuilder() : null;
        }

        public final void setS3StorageConfig(S3StorageConfig.BuilderImpl s3StorageConfig) {
            this.s3StorageConfig = s3StorageConfig != null ? s3StorageConfig.build() : null;
        }

        @Override
        public final Builder s3StorageConfig(S3StorageConfig s3StorageConfig) {
            this.s3StorageConfig = s3StorageConfig;
            return this;
        }

        public final Boolean getDisableGlueTableCreation() {
            return disableGlueTableCreation;
        }

        public final void setDisableGlueTableCreation(Boolean disableGlueTableCreation) {
            this.disableGlueTableCreation = disableGlueTableCreation;
        }

        @Override
        public final Builder disableGlueTableCreation(Boolean disableGlueTableCreation) {
            this.disableGlueTableCreation = disableGlueTableCreation;
            return this;
        }

        public final DataCatalogConfig.Builder getDataCatalogConfig() {
            return dataCatalogConfig != null ? dataCatalogConfig.toBuilder() : null;
        }

        public final void setDataCatalogConfig(DataCatalogConfig.BuilderImpl dataCatalogConfig) {
            this.dataCatalogConfig = dataCatalogConfig != null ? dataCatalogConfig.build() : null;
        }

        @Override
        public final Builder dataCatalogConfig(DataCatalogConfig dataCatalogConfig) {
            this.dataCatalogConfig = dataCatalogConfig;
            return this;
        }

        public final String getTableFormat() {
            return tableFormat;
        }

        public final void setTableFormat(String tableFormat) {
            this.tableFormat = tableFormat;
        }

        @Override
        public final Builder tableFormat(String tableFormat) {
            this.tableFormat = tableFormat;
            return this;
        }

        @Override
        public final Builder tableFormat(TableFormat tableFormat) {
            this.tableFormat(tableFormat == null ? null : tableFormat.toString());
            return this;
        }

        @Override
        public OfflineStoreConfig build() {
            return new OfflineStoreConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
