/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the S3 data source.
 * </p>
 * <p>
 * Your input bucket must be in the same Amazon Web Services region as your training job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3DataSource implements SdkPojo, Serializable, ToCopyableBuilder<S3DataSource.Builder, S3DataSource> {
    private static final SdkField<String> S3_DATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3DataType").getter(getter(S3DataSource::s3DataTypeAsString)).setter(setter(Builder::s3DataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3DataType").build()).build();

    private static final SdkField<String> S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("S3Uri")
            .getter(getter(S3DataSource::s3Uri)).setter(setter(Builder::s3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Uri").build()).build();

    private static final SdkField<String> S3_DATA_DISTRIBUTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3DataDistributionType").getter(getter(S3DataSource::s3DataDistributionTypeAsString))
            .setter(setter(Builder::s3DataDistributionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3DataDistributionType").build())
            .build();

    private static final SdkField<List<String>> ATTRIBUTE_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AttributeNames")
            .getter(getter(S3DataSource::attributeNames))
            .setter(setter(Builder::attributeNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> INSTANCE_GROUP_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InstanceGroupNames")
            .getter(getter(S3DataSource::instanceGroupNames))
            .setter(setter(Builder::instanceGroupNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceGroupNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ModelAccessConfig> MODEL_ACCESS_CONFIG_FIELD = SdkField
            .<ModelAccessConfig> builder(MarshallingType.SDK_POJO).memberName("ModelAccessConfig")
            .getter(getter(S3DataSource::modelAccessConfig)).setter(setter(Builder::modelAccessConfig))
            .constructor(ModelAccessConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelAccessConfig").build()).build();

    private static final SdkField<HubAccessConfig> HUB_ACCESS_CONFIG_FIELD = SdkField
            .<HubAccessConfig> builder(MarshallingType.SDK_POJO).memberName("HubAccessConfig")
            .getter(getter(S3DataSource::hubAccessConfig)).setter(setter(Builder::hubAccessConfig))
            .constructor(HubAccessConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HubAccessConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_DATA_TYPE_FIELD,
            S3_URI_FIELD, S3_DATA_DISTRIBUTION_TYPE_FIELD, ATTRIBUTE_NAMES_FIELD, INSTANCE_GROUP_NAMES_FIELD,
            MODEL_ACCESS_CONFIG_FIELD, HUB_ACCESS_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String s3DataType;

    private final String s3Uri;

    private final String s3DataDistributionType;

    private final List<String> attributeNames;

    private final List<String> instanceGroupNames;

    private final ModelAccessConfig modelAccessConfig;

    private final HubAccessConfig hubAccessConfig;

    private S3DataSource(BuilderImpl builder) {
        this.s3DataType = builder.s3DataType;
        this.s3Uri = builder.s3Uri;
        this.s3DataDistributionType = builder.s3DataDistributionType;
        this.attributeNames = builder.attributeNames;
        this.instanceGroupNames = builder.instanceGroupNames;
        this.modelAccessConfig = builder.modelAccessConfig;
        this.hubAccessConfig = builder.hubAccessConfig;
    }

    /**
     * <p>
     * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all objects
     * that match the specified key name prefix for model training.
     * </p>
     * <p>
     * If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
     * containing a list of object keys that you want SageMaker to use for model training.
     * </p>
     * <p>
     * If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an augmented
     * manifest file in JSON lines format. This file contains the data you want to use for model training.
     * <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is <code>Pipe</code>.
     * </p>
     * <p>
     * If you choose <code>Converse</code>, <code>S3Uri</code> identifies an Amazon S3 location that contains data
     * formatted according to Converse format. This format structures conversational messages with specific roles and
     * content types used for training and fine-tuning foundational models.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3DataType} will
     * return {@link S3DataType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3DataTypeAsString}.
     * </p>
     * 
     * @return If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all
     *         objects that match the specified key name prefix for model training. </p>
     *         <p>
     *         If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
     *         containing a list of object keys that you want SageMaker to use for model training.
     *         </p>
     *         <p>
     *         If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an
     *         augmented manifest file in JSON lines format. This file contains the data you want to use for model
     *         training. <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is
     *         <code>Pipe</code>.
     *         </p>
     *         <p>
     *         If you choose <code>Converse</code>, <code>S3Uri</code> identifies an Amazon S3 location that contains
     *         data formatted according to Converse format. This format structures conversational messages with specific
     *         roles and content types used for training and fine-tuning foundational models.
     * @see S3DataType
     */
    public final S3DataType s3DataType() {
        return S3DataType.fromValue(s3DataType);
    }

    /**
     * <p>
     * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all objects
     * that match the specified key name prefix for model training.
     * </p>
     * <p>
     * If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
     * containing a list of object keys that you want SageMaker to use for model training.
     * </p>
     * <p>
     * If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an augmented
     * manifest file in JSON lines format. This file contains the data you want to use for model training.
     * <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is <code>Pipe</code>.
     * </p>
     * <p>
     * If you choose <code>Converse</code>, <code>S3Uri</code> identifies an Amazon S3 location that contains data
     * formatted according to Converse format. This format structures conversational messages with specific roles and
     * content types used for training and fine-tuning foundational models.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3DataType} will
     * return {@link S3DataType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3DataTypeAsString}.
     * </p>
     * 
     * @return If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all
     *         objects that match the specified key name prefix for model training. </p>
     *         <p>
     *         If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
     *         containing a list of object keys that you want SageMaker to use for model training.
     *         </p>
     *         <p>
     *         If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an
     *         augmented manifest file in JSON lines format. This file contains the data you want to use for model
     *         training. <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is
     *         <code>Pipe</code>.
     *         </p>
     *         <p>
     *         If you choose <code>Converse</code>, <code>S3Uri</code> identifies an Amazon S3 location that contains
     *         data formatted according to Converse format. This format structures conversational messages with specific
     *         roles and content types used for training and fine-tuning foundational models.
     * @see S3DataType
     */
    public final String s3DataTypeAsString() {
        return s3DataType;
    }

    /**
     * <p>
     * Depending on the value specified for the <code>S3DataType</code>, identifies either a key name prefix or a
     * manifest. For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A key name prefix might look like this: <code>s3://bucketname/exampleprefix/</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * A manifest might look like this: <code>s3://bucketname/example.manifest</code>
     * </p>
     * <p>
     * A manifest is an S3 object which is a JSON file consisting of an array of elements. The first element is a prefix
     * which is followed by one or more suffixes. SageMaker appends the suffix elements to the prefix to get a full set
     * of <code>S3Uri</code>. Note that the prefix must be a valid non-empty <code>S3Uri</code> that precludes users
     * from specifying a manifest whose individual <code>S3Uri</code> is sourced from different S3 buckets.
     * </p>
     * <p>
     * The following code example shows a valid manifest format:
     * </p>
     * <p>
     * <code>[ {"prefix": "s3://customer_bucket/some/prefix/"},</code>
     * </p>
     * <p>
     * <code> "relative/path/to/custdata-1",</code>
     * </p>
     * <p>
     * <code> "relative/path/custdata-2",</code>
     * </p>
     * <p>
     * <code> ...</code>
     * </p>
     * <p>
     * <code> "relative/path/custdata-N"</code>
     * </p>
     * <p>
     * <code>]</code>
     * </p>
     * <p>
     * This JSON is equivalent to the following <code>S3Uri</code> list:
     * </p>
     * <p>
     * <code>s3://customer_bucket/some/prefix/relative/path/to/custdata-1</code>
     * </p>
     * <p>
     * <code>s3://customer_bucket/some/prefix/relative/path/custdata-2</code>
     * </p>
     * <p>
     * <code>...</code>
     * </p>
     * <p>
     * <code>s3://customer_bucket/some/prefix/relative/path/custdata-N</code>
     * </p>
     * <p>
     * The complete set of <code>S3Uri</code> in this manifest is the input data for the channel for this data source.
     * The object that each <code>S3Uri</code> points to must be readable by the IAM role that SageMaker uses to perform
     * tasks on your behalf.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Your input bucket must be located in same Amazon Web Services region as your training job.
     * </p>
     * 
     * @return Depending on the value specified for the <code>S3DataType</code>, identifies either a key name prefix or
     *         a manifest. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A key name prefix might look like this: <code>s3://bucketname/exampleprefix/</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A manifest might look like this: <code>s3://bucketname/example.manifest</code>
     *         </p>
     *         <p>
     *         A manifest is an S3 object which is a JSON file consisting of an array of elements. The first element is
     *         a prefix which is followed by one or more suffixes. SageMaker appends the suffix elements to the prefix
     *         to get a full set of <code>S3Uri</code>. Note that the prefix must be a valid non-empty
     *         <code>S3Uri</code> that precludes users from specifying a manifest whose individual <code>S3Uri</code> is
     *         sourced from different S3 buckets.
     *         </p>
     *         <p>
     *         The following code example shows a valid manifest format:
     *         </p>
     *         <p>
     *         <code>[ {"prefix": "s3://customer_bucket/some/prefix/"},</code>
     *         </p>
     *         <p>
     *         <code> "relative/path/to/custdata-1",</code>
     *         </p>
     *         <p>
     *         <code> "relative/path/custdata-2",</code>
     *         </p>
     *         <p>
     *         <code> ...</code>
     *         </p>
     *         <p>
     *         <code> "relative/path/custdata-N"</code>
     *         </p>
     *         <p>
     *         <code>]</code>
     *         </p>
     *         <p>
     *         This JSON is equivalent to the following <code>S3Uri</code> list:
     *         </p>
     *         <p>
     *         <code>s3://customer_bucket/some/prefix/relative/path/to/custdata-1</code>
     *         </p>
     *         <p>
     *         <code>s3://customer_bucket/some/prefix/relative/path/custdata-2</code>
     *         </p>
     *         <p>
     *         <code>...</code>
     *         </p>
     *         <p>
     *         <code>s3://customer_bucket/some/prefix/relative/path/custdata-N</code>
     *         </p>
     *         <p>
     *         The complete set of <code>S3Uri</code> in this manifest is the input data for the channel for this data
     *         source. The object that each <code>S3Uri</code> points to must be readable by the IAM role that SageMaker
     *         uses to perform tasks on your behalf.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Your input bucket must be located in same Amazon Web Services region as your training job.
     */
    public final String s3Uri() {
        return s3Uri;
    }

    /**
     * <p>
     * If you want SageMaker to replicate the entire dataset on each ML compute instance that is launched for model
     * training, specify <code>FullyReplicated</code>.
     * </p>
     * <p>
     * If you want SageMaker to replicate a subset of data on each ML compute instance that is launched for model
     * training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances launched for a training
     * job, each instance gets approximately 1/<i>n</i> of the number of S3 objects. In this case, model training on
     * each machine uses only the subset of training data.
     * </p>
     * <p>
     * Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes won't get
     * any data and you will pay for nodes that aren't getting any training data. This applies in both File and Pipe
     * modes. Keep this in mind when developing algorithms.
     * </p>
     * <p>
     * In distributed training, where you use multiple ML compute EC2 instances, you might choose
     * <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume (when
     * <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the number of objects.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #s3DataDistributionType} will return {@link S3DataDistribution#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #s3DataDistributionTypeAsString}.
     * </p>
     * 
     * @return If you want SageMaker to replicate the entire dataset on each ML compute instance that is launched for
     *         model training, specify <code>FullyReplicated</code>. </p>
     *         <p>
     *         If you want SageMaker to replicate a subset of data on each ML compute instance that is launched for
     *         model training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances launched
     *         for a training job, each instance gets approximately 1/<i>n</i> of the number of S3 objects. In this
     *         case, model training on each machine uses only the subset of training data.
     *         </p>
     *         <p>
     *         Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes
     *         won't get any data and you will pay for nodes that aren't getting any training data. This applies in both
     *         File and Pipe modes. Keep this in mind when developing algorithms.
     *         </p>
     *         <p>
     *         In distributed training, where you use multiple ML compute EC2 instances, you might choose
     *         <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume
     *         (when <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the number
     *         of objects.
     * @see S3DataDistribution
     */
    public final S3DataDistribution s3DataDistributionType() {
        return S3DataDistribution.fromValue(s3DataDistributionType);
    }

    /**
     * <p>
     * If you want SageMaker to replicate the entire dataset on each ML compute instance that is launched for model
     * training, specify <code>FullyReplicated</code>.
     * </p>
     * <p>
     * If you want SageMaker to replicate a subset of data on each ML compute instance that is launched for model
     * training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances launched for a training
     * job, each instance gets approximately 1/<i>n</i> of the number of S3 objects. In this case, model training on
     * each machine uses only the subset of training data.
     * </p>
     * <p>
     * Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes won't get
     * any data and you will pay for nodes that aren't getting any training data. This applies in both File and Pipe
     * modes. Keep this in mind when developing algorithms.
     * </p>
     * <p>
     * In distributed training, where you use multiple ML compute EC2 instances, you might choose
     * <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume (when
     * <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the number of objects.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #s3DataDistributionType} will return {@link S3DataDistribution#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #s3DataDistributionTypeAsString}.
     * </p>
     * 
     * @return If you want SageMaker to replicate the entire dataset on each ML compute instance that is launched for
     *         model training, specify <code>FullyReplicated</code>. </p>
     *         <p>
     *         If you want SageMaker to replicate a subset of data on each ML compute instance that is launched for
     *         model training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances launched
     *         for a training job, each instance gets approximately 1/<i>n</i> of the number of S3 objects. In this
     *         case, model training on each machine uses only the subset of training data.
     *         </p>
     *         <p>
     *         Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes
     *         won't get any data and you will pay for nodes that aren't getting any training data. This applies in both
     *         File and Pipe modes. Keep this in mind when developing algorithms.
     *         </p>
     *         <p>
     *         In distributed training, where you use multiple ML compute EC2 instances, you might choose
     *         <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume
     *         (when <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the number
     *         of objects.
     * @see S3DataDistribution
     */
    public final String s3DataDistributionTypeAsString() {
        return s3DataDistributionType;
    }

    /**
     * For responses, this returns true if the service returned a value for the AttributeNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAttributeNames() {
        return attributeNames != null && !(attributeNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of one or more attribute names to use that are found in a specified augmented manifest file.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributeNames} method.
     * </p>
     * 
     * @return A list of one or more attribute names to use that are found in a specified augmented manifest file.
     */
    public final List<String> attributeNames() {
        return attributeNames;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceGroupNames property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInstanceGroupNames() {
        return instanceGroupNames != null && !(instanceGroupNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of names of instance groups that get data from the S3 data source.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceGroupNames} method.
     * </p>
     * 
     * @return A list of names of instance groups that get data from the S3 data source.
     */
    public final List<String> instanceGroupNames() {
        return instanceGroupNames;
    }

    /**
     * Returns the value of the ModelAccessConfig property for this object.
     * 
     * @return The value of the ModelAccessConfig property for this object.
     */
    public final ModelAccessConfig modelAccessConfig() {
        return modelAccessConfig;
    }

    /**
     * <p>
     * The configuration for a private hub model reference that points to a SageMaker JumpStart public hub model.
     * </p>
     * 
     * @return The configuration for a private hub model reference that points to a SageMaker JumpStart public hub
     *         model.
     */
    public final HubAccessConfig hubAccessConfig() {
        return hubAccessConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3DataTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(s3Uri());
        hashCode = 31 * hashCode + Objects.hashCode(s3DataDistributionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributeNames() ? attributeNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceGroupNames() ? instanceGroupNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(modelAccessConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hubAccessConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3DataSource)) {
            return false;
        }
        S3DataSource other = (S3DataSource) obj;
        return Objects.equals(s3DataTypeAsString(), other.s3DataTypeAsString()) && Objects.equals(s3Uri(), other.s3Uri())
                && Objects.equals(s3DataDistributionTypeAsString(), other.s3DataDistributionTypeAsString())
                && hasAttributeNames() == other.hasAttributeNames() && Objects.equals(attributeNames(), other.attributeNames())
                && hasInstanceGroupNames() == other.hasInstanceGroupNames()
                && Objects.equals(instanceGroupNames(), other.instanceGroupNames())
                && Objects.equals(modelAccessConfig(), other.modelAccessConfig())
                && Objects.equals(hubAccessConfig(), other.hubAccessConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3DataSource").add("S3DataType", s3DataTypeAsString()).add("S3Uri", s3Uri())
                .add("S3DataDistributionType", s3DataDistributionTypeAsString())
                .add("AttributeNames", hasAttributeNames() ? attributeNames() : null)
                .add("InstanceGroupNames", hasInstanceGroupNames() ? instanceGroupNames() : null)
                .add("ModelAccessConfig", modelAccessConfig()).add("HubAccessConfig", hubAccessConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3DataType":
            return Optional.ofNullable(clazz.cast(s3DataTypeAsString()));
        case "S3Uri":
            return Optional.ofNullable(clazz.cast(s3Uri()));
        case "S3DataDistributionType":
            return Optional.ofNullable(clazz.cast(s3DataDistributionTypeAsString()));
        case "AttributeNames":
            return Optional.ofNullable(clazz.cast(attributeNames()));
        case "InstanceGroupNames":
            return Optional.ofNullable(clazz.cast(instanceGroupNames()));
        case "ModelAccessConfig":
            return Optional.ofNullable(clazz.cast(modelAccessConfig()));
        case "HubAccessConfig":
            return Optional.ofNullable(clazz.cast(hubAccessConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("S3DataType", S3_DATA_TYPE_FIELD);
        map.put("S3Uri", S3_URI_FIELD);
        map.put("S3DataDistributionType", S3_DATA_DISTRIBUTION_TYPE_FIELD);
        map.put("AttributeNames", ATTRIBUTE_NAMES_FIELD);
        map.put("InstanceGroupNames", INSTANCE_GROUP_NAMES_FIELD);
        map.put("ModelAccessConfig", MODEL_ACCESS_CONFIG_FIELD);
        map.put("HubAccessConfig", HUB_ACCESS_CONFIG_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<S3DataSource, T> g) {
        return obj -> g.apply((S3DataSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3DataSource> {
        /**
         * <p>
         * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all
         * objects that match the specified key name prefix for model training.
         * </p>
         * <p>
         * If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
         * containing a list of object keys that you want SageMaker to use for model training.
         * </p>
         * <p>
         * If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an
         * augmented manifest file in JSON lines format. This file contains the data you want to use for model training.
         * <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is <code>Pipe</code>.
         * </p>
         * <p>
         * If you choose <code>Converse</code>, <code>S3Uri</code> identifies an Amazon S3 location that contains data
         * formatted according to Converse format. This format structures conversational messages with specific roles
         * and content types used for training and fine-tuning foundational models.
         * </p>
         * 
         * @param s3DataType
         *        If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses
         *        all objects that match the specified key name prefix for model training. </p>
         *        <p>
         *        If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest
         *        file containing a list of object keys that you want SageMaker to use for model training.
         *        </p>
         *        <p>
         *        If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an
         *        augmented manifest file in JSON lines format. This file contains the data you want to use for model
         *        training. <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is
         *        <code>Pipe</code>.
         *        </p>
         *        <p>
         *        If you choose <code>Converse</code>, <code>S3Uri</code> identifies an Amazon S3 location that contains
         *        data formatted according to Converse format. This format structures conversational messages with
         *        specific roles and content types used for training and fine-tuning foundational models.
         * @see S3DataType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3DataType
         */
        Builder s3DataType(String s3DataType);

        /**
         * <p>
         * If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses all
         * objects that match the specified key name prefix for model training.
         * </p>
         * <p>
         * If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest file
         * containing a list of object keys that you want SageMaker to use for model training.
         * </p>
         * <p>
         * If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an
         * augmented manifest file in JSON lines format. This file contains the data you want to use for model training.
         * <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is <code>Pipe</code>.
         * </p>
         * <p>
         * If you choose <code>Converse</code>, <code>S3Uri</code> identifies an Amazon S3 location that contains data
         * formatted according to Converse format. This format structures conversational messages with specific roles
         * and content types used for training and fine-tuning foundational models.
         * </p>
         * 
         * @param s3DataType
         *        If you choose <code>S3Prefix</code>, <code>S3Uri</code> identifies a key name prefix. SageMaker uses
         *        all objects that match the specified key name prefix for model training. </p>
         *        <p>
         *        If you choose <code>ManifestFile</code>, <code>S3Uri</code> identifies an object that is a manifest
         *        file containing a list of object keys that you want SageMaker to use for model training.
         *        </p>
         *        <p>
         *        If you choose <code>AugmentedManifestFile</code>, <code>S3Uri</code> identifies an object that is an
         *        augmented manifest file in JSON lines format. This file contains the data you want to use for model
         *        training. <code>AugmentedManifestFile</code> can only be used if the Channel's input mode is
         *        <code>Pipe</code>.
         *        </p>
         *        <p>
         *        If you choose <code>Converse</code>, <code>S3Uri</code> identifies an Amazon S3 location that contains
         *        data formatted according to Converse format. This format structures conversational messages with
         *        specific roles and content types used for training and fine-tuning foundational models.
         * @see S3DataType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3DataType
         */
        Builder s3DataType(S3DataType s3DataType);

        /**
         * <p>
         * Depending on the value specified for the <code>S3DataType</code>, identifies either a key name prefix or a
         * manifest. For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A key name prefix might look like this: <code>s3://bucketname/exampleprefix/</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * A manifest might look like this: <code>s3://bucketname/example.manifest</code>
         * </p>
         * <p>
         * A manifest is an S3 object which is a JSON file consisting of an array of elements. The first element is a
         * prefix which is followed by one or more suffixes. SageMaker appends the suffix elements to the prefix to get
         * a full set of <code>S3Uri</code>. Note that the prefix must be a valid non-empty <code>S3Uri</code> that
         * precludes users from specifying a manifest whose individual <code>S3Uri</code> is sourced from different S3
         * buckets.
         * </p>
         * <p>
         * The following code example shows a valid manifest format:
         * </p>
         * <p>
         * <code>[ {"prefix": "s3://customer_bucket/some/prefix/"},</code>
         * </p>
         * <p>
         * <code> "relative/path/to/custdata-1",</code>
         * </p>
         * <p>
         * <code> "relative/path/custdata-2",</code>
         * </p>
         * <p>
         * <code> ...</code>
         * </p>
         * <p>
         * <code> "relative/path/custdata-N"</code>
         * </p>
         * <p>
         * <code>]</code>
         * </p>
         * <p>
         * This JSON is equivalent to the following <code>S3Uri</code> list:
         * </p>
         * <p>
         * <code>s3://customer_bucket/some/prefix/relative/path/to/custdata-1</code>
         * </p>
         * <p>
         * <code>s3://customer_bucket/some/prefix/relative/path/custdata-2</code>
         * </p>
         * <p>
         * <code>...</code>
         * </p>
         * <p>
         * <code>s3://customer_bucket/some/prefix/relative/path/custdata-N</code>
         * </p>
         * <p>
         * The complete set of <code>S3Uri</code> in this manifest is the input data for the channel for this data
         * source. The object that each <code>S3Uri</code> points to must be readable by the IAM role that SageMaker
         * uses to perform tasks on your behalf.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Your input bucket must be located in same Amazon Web Services region as your training job.
         * </p>
         * 
         * @param s3Uri
         *        Depending on the value specified for the <code>S3DataType</code>, identifies either a key name prefix
         *        or a manifest. For example: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A key name prefix might look like this: <code>s3://bucketname/exampleprefix/</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A manifest might look like this: <code>s3://bucketname/example.manifest</code>
         *        </p>
         *        <p>
         *        A manifest is an S3 object which is a JSON file consisting of an array of elements. The first element
         *        is a prefix which is followed by one or more suffixes. SageMaker appends the suffix elements to the
         *        prefix to get a full set of <code>S3Uri</code>. Note that the prefix must be a valid non-empty
         *        <code>S3Uri</code> that precludes users from specifying a manifest whose individual <code>S3Uri</code>
         *        is sourced from different S3 buckets.
         *        </p>
         *        <p>
         *        The following code example shows a valid manifest format:
         *        </p>
         *        <p>
         *        <code>[ {"prefix": "s3://customer_bucket/some/prefix/"},</code>
         *        </p>
         *        <p>
         *        <code> "relative/path/to/custdata-1",</code>
         *        </p>
         *        <p>
         *        <code> "relative/path/custdata-2",</code>
         *        </p>
         *        <p>
         *        <code> ...</code>
         *        </p>
         *        <p>
         *        <code> "relative/path/custdata-N"</code>
         *        </p>
         *        <p>
         *        <code>]</code>
         *        </p>
         *        <p>
         *        This JSON is equivalent to the following <code>S3Uri</code> list:
         *        </p>
         *        <p>
         *        <code>s3://customer_bucket/some/prefix/relative/path/to/custdata-1</code>
         *        </p>
         *        <p>
         *        <code>s3://customer_bucket/some/prefix/relative/path/custdata-2</code>
         *        </p>
         *        <p>
         *        <code>...</code>
         *        </p>
         *        <p>
         *        <code>s3://customer_bucket/some/prefix/relative/path/custdata-N</code>
         *        </p>
         *        <p>
         *        The complete set of <code>S3Uri</code> in this manifest is the input data for the channel for this
         *        data source. The object that each <code>S3Uri</code> points to must be readable by the IAM role that
         *        SageMaker uses to perform tasks on your behalf.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Your input bucket must be located in same Amazon Web Services region as your training job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Uri(String s3Uri);

        /**
         * <p>
         * If you want SageMaker to replicate the entire dataset on each ML compute instance that is launched for model
         * training, specify <code>FullyReplicated</code>.
         * </p>
         * <p>
         * If you want SageMaker to replicate a subset of data on each ML compute instance that is launched for model
         * training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances launched for a
         * training job, each instance gets approximately 1/<i>n</i> of the number of S3 objects. In this case, model
         * training on each machine uses only the subset of training data.
         * </p>
         * <p>
         * Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes won't
         * get any data and you will pay for nodes that aren't getting any training data. This applies in both File and
         * Pipe modes. Keep this in mind when developing algorithms.
         * </p>
         * <p>
         * In distributed training, where you use multiple ML compute EC2 instances, you might choose
         * <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume (when
         * <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the number of objects.
         * </p>
         * 
         * @param s3DataDistributionType
         *        If you want SageMaker to replicate the entire dataset on each ML compute instance that is launched for
         *        model training, specify <code>FullyReplicated</code>. </p>
         *        <p>
         *        If you want SageMaker to replicate a subset of data on each ML compute instance that is launched for
         *        model training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances
         *        launched for a training job, each instance gets approximately 1/<i>n</i> of the number of S3 objects.
         *        In this case, model training on each machine uses only the subset of training data.
         *        </p>
         *        <p>
         *        Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes
         *        won't get any data and you will pay for nodes that aren't getting any training data. This applies in
         *        both File and Pipe modes. Keep this in mind when developing algorithms.
         *        </p>
         *        <p>
         *        In distributed training, where you use multiple ML compute EC2 instances, you might choose
         *        <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume
         *        (when <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the
         *        number of objects.
         * @see S3DataDistribution
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3DataDistribution
         */
        Builder s3DataDistributionType(String s3DataDistributionType);

        /**
         * <p>
         * If you want SageMaker to replicate the entire dataset on each ML compute instance that is launched for model
         * training, specify <code>FullyReplicated</code>.
         * </p>
         * <p>
         * If you want SageMaker to replicate a subset of data on each ML compute instance that is launched for model
         * training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances launched for a
         * training job, each instance gets approximately 1/<i>n</i> of the number of S3 objects. In this case, model
         * training on each machine uses only the subset of training data.
         * </p>
         * <p>
         * Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes won't
         * get any data and you will pay for nodes that aren't getting any training data. This applies in both File and
         * Pipe modes. Keep this in mind when developing algorithms.
         * </p>
         * <p>
         * In distributed training, where you use multiple ML compute EC2 instances, you might choose
         * <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume (when
         * <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the number of objects.
         * </p>
         * 
         * @param s3DataDistributionType
         *        If you want SageMaker to replicate the entire dataset on each ML compute instance that is launched for
         *        model training, specify <code>FullyReplicated</code>. </p>
         *        <p>
         *        If you want SageMaker to replicate a subset of data on each ML compute instance that is launched for
         *        model training, specify <code>ShardedByS3Key</code>. If there are <i>n</i> ML compute instances
         *        launched for a training job, each instance gets approximately 1/<i>n</i> of the number of S3 objects.
         *        In this case, model training on each machine uses only the subset of training data.
         *        </p>
         *        <p>
         *        Don't choose more ML compute instances for training than available S3 objects. If you do, some nodes
         *        won't get any data and you will pay for nodes that aren't getting any training data. This applies in
         *        both File and Pipe modes. Keep this in mind when developing algorithms.
         *        </p>
         *        <p>
         *        In distributed training, where you use multiple ML compute EC2 instances, you might choose
         *        <code>ShardedByS3Key</code>. If the algorithm requires copying training data to the ML storage volume
         *        (when <code>TrainingInputMode</code> is set to <code>File</code>), this copies 1/<i>n</i> of the
         *        number of objects.
         * @see S3DataDistribution
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3DataDistribution
         */
        Builder s3DataDistributionType(S3DataDistribution s3DataDistributionType);

        /**
         * <p>
         * A list of one or more attribute names to use that are found in a specified augmented manifest file.
         * </p>
         * 
         * @param attributeNames
         *        A list of one or more attribute names to use that are found in a specified augmented manifest file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeNames(Collection<String> attributeNames);

        /**
         * <p>
         * A list of one or more attribute names to use that are found in a specified augmented manifest file.
         * </p>
         * 
         * @param attributeNames
         *        A list of one or more attribute names to use that are found in a specified augmented manifest file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeNames(String... attributeNames);

        /**
         * <p>
         * A list of names of instance groups that get data from the S3 data source.
         * </p>
         * 
         * @param instanceGroupNames
         *        A list of names of instance groups that get data from the S3 data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGroupNames(Collection<String> instanceGroupNames);

        /**
         * <p>
         * A list of names of instance groups that get data from the S3 data source.
         * </p>
         * 
         * @param instanceGroupNames
         *        A list of names of instance groups that get data from the S3 data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGroupNames(String... instanceGroupNames);

        /**
         * Sets the value of the ModelAccessConfig property for this object.
         *
         * @param modelAccessConfig
         *        The new value for the ModelAccessConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelAccessConfig(ModelAccessConfig modelAccessConfig);

        /**
         * Sets the value of the ModelAccessConfig property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ModelAccessConfig.Builder} avoiding the
         * need to create one manually via {@link ModelAccessConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ModelAccessConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #modelAccessConfig(ModelAccessConfig)}.
         * 
         * @param modelAccessConfig
         *        a consumer that will call methods on {@link ModelAccessConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modelAccessConfig(ModelAccessConfig)
         */
        default Builder modelAccessConfig(Consumer<ModelAccessConfig.Builder> modelAccessConfig) {
            return modelAccessConfig(ModelAccessConfig.builder().applyMutation(modelAccessConfig).build());
        }

        /**
         * <p>
         * The configuration for a private hub model reference that points to a SageMaker JumpStart public hub model.
         * </p>
         * 
         * @param hubAccessConfig
         *        The configuration for a private hub model reference that points to a SageMaker JumpStart public hub
         *        model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hubAccessConfig(HubAccessConfig hubAccessConfig);

        /**
         * <p>
         * The configuration for a private hub model reference that points to a SageMaker JumpStart public hub model.
         * </p>
         * This is a convenience method that creates an instance of the {@link HubAccessConfig.Builder} avoiding the
         * need to create one manually via {@link HubAccessConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link HubAccessConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #hubAccessConfig(HubAccessConfig)}.
         * 
         * @param hubAccessConfig
         *        a consumer that will call methods on {@link HubAccessConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hubAccessConfig(HubAccessConfig)
         */
        default Builder hubAccessConfig(Consumer<HubAccessConfig.Builder> hubAccessConfig) {
            return hubAccessConfig(HubAccessConfig.builder().applyMutation(hubAccessConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String s3DataType;

        private String s3Uri;

        private String s3DataDistributionType;

        private List<String> attributeNames = DefaultSdkAutoConstructList.getInstance();

        private List<String> instanceGroupNames = DefaultSdkAutoConstructList.getInstance();

        private ModelAccessConfig modelAccessConfig;

        private HubAccessConfig hubAccessConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(S3DataSource model) {
            s3DataType(model.s3DataType);
            s3Uri(model.s3Uri);
            s3DataDistributionType(model.s3DataDistributionType);
            attributeNames(model.attributeNames);
            instanceGroupNames(model.instanceGroupNames);
            modelAccessConfig(model.modelAccessConfig);
            hubAccessConfig(model.hubAccessConfig);
        }

        public final String getS3DataType() {
            return s3DataType;
        }

        public final void setS3DataType(String s3DataType) {
            this.s3DataType = s3DataType;
        }

        @Override
        public final Builder s3DataType(String s3DataType) {
            this.s3DataType = s3DataType;
            return this;
        }

        @Override
        public final Builder s3DataType(S3DataType s3DataType) {
            this.s3DataType(s3DataType == null ? null : s3DataType.toString());
            return this;
        }

        public final String getS3Uri() {
            return s3Uri;
        }

        public final void setS3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
        }

        @Override
        public final Builder s3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
            return this;
        }

        public final String getS3DataDistributionType() {
            return s3DataDistributionType;
        }

        public final void setS3DataDistributionType(String s3DataDistributionType) {
            this.s3DataDistributionType = s3DataDistributionType;
        }

        @Override
        public final Builder s3DataDistributionType(String s3DataDistributionType) {
            this.s3DataDistributionType = s3DataDistributionType;
            return this;
        }

        @Override
        public final Builder s3DataDistributionType(S3DataDistribution s3DataDistributionType) {
            this.s3DataDistributionType(s3DataDistributionType == null ? null : s3DataDistributionType.toString());
            return this;
        }

        public final Collection<String> getAttributeNames() {
            if (attributeNames instanceof SdkAutoConstructList) {
                return null;
            }
            return attributeNames;
        }

        public final void setAttributeNames(Collection<String> attributeNames) {
            this.attributeNames = AttributeNamesCopier.copy(attributeNames);
        }

        @Override
        public final Builder attributeNames(Collection<String> attributeNames) {
            this.attributeNames = AttributeNamesCopier.copy(attributeNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributeNames(String... attributeNames) {
            attributeNames(Arrays.asList(attributeNames));
            return this;
        }

        public final Collection<String> getInstanceGroupNames() {
            if (instanceGroupNames instanceof SdkAutoConstructList) {
                return null;
            }
            return instanceGroupNames;
        }

        public final void setInstanceGroupNames(Collection<String> instanceGroupNames) {
            this.instanceGroupNames = InstanceGroupNamesCopier.copy(instanceGroupNames);
        }

        @Override
        public final Builder instanceGroupNames(Collection<String> instanceGroupNames) {
            this.instanceGroupNames = InstanceGroupNamesCopier.copy(instanceGroupNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceGroupNames(String... instanceGroupNames) {
            instanceGroupNames(Arrays.asList(instanceGroupNames));
            return this;
        }

        public final ModelAccessConfig.Builder getModelAccessConfig() {
            return modelAccessConfig != null ? modelAccessConfig.toBuilder() : null;
        }

        public final void setModelAccessConfig(ModelAccessConfig.BuilderImpl modelAccessConfig) {
            this.modelAccessConfig = modelAccessConfig != null ? modelAccessConfig.build() : null;
        }

        @Override
        public final Builder modelAccessConfig(ModelAccessConfig modelAccessConfig) {
            this.modelAccessConfig = modelAccessConfig;
            return this;
        }

        public final HubAccessConfig.Builder getHubAccessConfig() {
            return hubAccessConfig != null ? hubAccessConfig.toBuilder() : null;
        }

        public final void setHubAccessConfig(HubAccessConfig.BuilderImpl hubAccessConfig) {
            this.hubAccessConfig = hubAccessConfig != null ? hubAccessConfig.build() : null;
        }

        @Override
        public final Builder hubAccessConfig(HubAccessConfig hubAccessConfig) {
            this.hubAccessConfig = hubAccessConfig;
            return this;
        }

        @Override
        public S3DataSource build() {
            return new S3DataSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
