/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines the training jobs launched by a hyperparameter tuning job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HyperParameterTrainingJobDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<HyperParameterTrainingJobDefinition.Builder, HyperParameterTrainingJobDefinition> {
    private static final SdkField<Map<String, String>> STATIC_HYPER_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(HyperParameterTrainingJobDefinition::staticHyperParameters))
            .setter(setter(Builder::staticHyperParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StaticHyperParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<HyperParameterAlgorithmSpecification> ALGORITHM_SPECIFICATION_FIELD = SdkField
            .<HyperParameterAlgorithmSpecification> builder(MarshallingType.SDK_POJO)
            .getter(getter(HyperParameterTrainingJobDefinition::algorithmSpecification))
            .setter(setter(Builder::algorithmSpecification)).constructor(HyperParameterAlgorithmSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlgorithmSpecification").build())
            .build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(HyperParameterTrainingJobDefinition::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<List<Channel>> INPUT_DATA_CONFIG_FIELD = SdkField
            .<List<Channel>> builder(MarshallingType.LIST)
            .getter(getter(HyperParameterTrainingJobDefinition::inputDataConfig))
            .setter(setter(Builder::inputDataConfig))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputDataConfig").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Channel> builder(MarshallingType.SDK_POJO)
                                            .constructor(Channel::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<VpcConfig> VPC_CONFIG_FIELD = SdkField.<VpcConfig> builder(MarshallingType.SDK_POJO)
            .getter(getter(HyperParameterTrainingJobDefinition::vpcConfig)).setter(setter(Builder::vpcConfig))
            .constructor(VpcConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConfig").build()).build();

    private static final SdkField<OutputDataConfig> OUTPUT_DATA_CONFIG_FIELD = SdkField
            .<OutputDataConfig> builder(MarshallingType.SDK_POJO)
            .getter(getter(HyperParameterTrainingJobDefinition::outputDataConfig)).setter(setter(Builder::outputDataConfig))
            .constructor(OutputDataConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputDataConfig").build()).build();

    private static final SdkField<ResourceConfig> RESOURCE_CONFIG_FIELD = SdkField
            .<ResourceConfig> builder(MarshallingType.SDK_POJO)
            .getter(getter(HyperParameterTrainingJobDefinition::resourceConfig)).setter(setter(Builder::resourceConfig))
            .constructor(ResourceConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceConfig").build()).build();

    private static final SdkField<StoppingCondition> STOPPING_CONDITION_FIELD = SdkField
            .<StoppingCondition> builder(MarshallingType.SDK_POJO)
            .getter(getter(HyperParameterTrainingJobDefinition::stoppingCondition)).setter(setter(Builder::stoppingCondition))
            .constructor(StoppingCondition::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StoppingCondition").build()).build();

    private static final SdkField<Boolean> ENABLE_NETWORK_ISOLATION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(HyperParameterTrainingJobDefinition::enableNetworkIsolation))
            .setter(setter(Builder::enableNetworkIsolation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableNetworkIsolation").build())
            .build();

    private static final SdkField<Boolean> ENABLE_INTER_CONTAINER_TRAFFIC_ENCRYPTION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(HyperParameterTrainingJobDefinition::enableInterContainerTrafficEncryption))
            .setter(setter(Builder::enableInterContainerTrafficEncryption))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("EnableInterContainerTrafficEncryption").build()).build();

    private static final SdkField<Boolean> ENABLE_MANAGED_SPOT_TRAINING_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(HyperParameterTrainingJobDefinition::enableManagedSpotTraining))
            .setter(setter(Builder::enableManagedSpotTraining))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableManagedSpotTraining").build())
            .build();

    private static final SdkField<CheckpointConfig> CHECKPOINT_CONFIG_FIELD = SdkField
            .<CheckpointConfig> builder(MarshallingType.SDK_POJO)
            .getter(getter(HyperParameterTrainingJobDefinition::checkpointConfig)).setter(setter(Builder::checkpointConfig))
            .constructor(CheckpointConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CheckpointConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATIC_HYPER_PARAMETERS_FIELD,
            ALGORITHM_SPECIFICATION_FIELD, ROLE_ARN_FIELD, INPUT_DATA_CONFIG_FIELD, VPC_CONFIG_FIELD, OUTPUT_DATA_CONFIG_FIELD,
            RESOURCE_CONFIG_FIELD, STOPPING_CONDITION_FIELD, ENABLE_NETWORK_ISOLATION_FIELD,
            ENABLE_INTER_CONTAINER_TRAFFIC_ENCRYPTION_FIELD, ENABLE_MANAGED_SPOT_TRAINING_FIELD, CHECKPOINT_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final Map<String, String> staticHyperParameters;

    private final HyperParameterAlgorithmSpecification algorithmSpecification;

    private final String roleArn;

    private final List<Channel> inputDataConfig;

    private final VpcConfig vpcConfig;

    private final OutputDataConfig outputDataConfig;

    private final ResourceConfig resourceConfig;

    private final StoppingCondition stoppingCondition;

    private final Boolean enableNetworkIsolation;

    private final Boolean enableInterContainerTrafficEncryption;

    private final Boolean enableManagedSpotTraining;

    private final CheckpointConfig checkpointConfig;

    private HyperParameterTrainingJobDefinition(BuilderImpl builder) {
        this.staticHyperParameters = builder.staticHyperParameters;
        this.algorithmSpecification = builder.algorithmSpecification;
        this.roleArn = builder.roleArn;
        this.inputDataConfig = builder.inputDataConfig;
        this.vpcConfig = builder.vpcConfig;
        this.outputDataConfig = builder.outputDataConfig;
        this.resourceConfig = builder.resourceConfig;
        this.stoppingCondition = builder.stoppingCondition;
        this.enableNetworkIsolation = builder.enableNetworkIsolation;
        this.enableInterContainerTrafficEncryption = builder.enableInterContainerTrafficEncryption;
        this.enableManagedSpotTraining = builder.enableManagedSpotTraining;
        this.checkpointConfig = builder.checkpointConfig;
    }

    /**
     * <p>
     * Specifies the values of hyperparameters that do not change for the tuning job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Specifies the values of hyperparameters that do not change for the tuning job.
     */
    public Map<String, String> staticHyperParameters() {
        return staticHyperParameters;
    }

    /**
     * <p>
     * The <a>HyperParameterAlgorithmSpecification</a> object that specifies the resource algorithm to use for the
     * training jobs that the tuning job launches.
     * </p>
     * 
     * @return The <a>HyperParameterAlgorithmSpecification</a> object that specifies the resource algorithm to use for
     *         the training jobs that the tuning job launches.
     */
    public HyperParameterAlgorithmSpecification algorithmSpecification() {
        return algorithmSpecification;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role associated with the training jobs that the tuning job launches.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role associated with the training jobs that the tuning job
     *         launches.
     */
    public String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * An array of <a>Channel</a> objects that specify the input for the training jobs that the tuning job launches.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of <a>Channel</a> objects that specify the input for the training jobs that the tuning job
     *         launches.
     */
    public List<Channel> inputDataConfig() {
        return inputDataConfig;
    }

    /**
     * <p>
     * The <a>VpcConfig</a> object that specifies the VPC that you want the training jobs that this hyperparameter
     * tuning job launches to connect to. Control access to and from your training container by configuring the VPC. For
     * more information, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/train-vpc.html">Protect Training
     * Jobs by Using an Amazon Virtual Private Cloud</a>.
     * </p>
     * 
     * @return The <a>VpcConfig</a> object that specifies the VPC that you want the training jobs that this
     *         hyperparameter tuning job launches to connect to. Control access to and from your training container by
     *         configuring the VPC. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/train-vpc.html">Protect Training Jobs by Using an
     *         Amazon Virtual Private Cloud</a>.
     */
    public VpcConfig vpcConfig() {
        return vpcConfig;
    }

    /**
     * <p>
     * Specifies the path to the Amazon S3 bucket where you store model artifacts from the training jobs that the tuning
     * job launches.
     * </p>
     * 
     * @return Specifies the path to the Amazon S3 bucket where you store model artifacts from the training jobs that
     *         the tuning job launches.
     */
    public OutputDataConfig outputDataConfig() {
        return outputDataConfig;
    }

    /**
     * <p>
     * The resources, including the compute instances and storage volumes, to use for the training jobs that the tuning
     * job launches.
     * </p>
     * <p>
     * Storage volumes store model artifacts and incremental states. Training algorithms might also use storage volumes
     * for scratch space. If you want Amazon SageMaker to use the storage volume to store the training data, choose
     * <code>File</code> as the <code>TrainingInputMode</code> in the algorithm specification. For distributed training
     * algorithms, specify an instance count greater than 1.
     * </p>
     * 
     * @return The resources, including the compute instances and storage volumes, to use for the training jobs that the
     *         tuning job launches.</p>
     *         <p>
     *         Storage volumes store model artifacts and incremental states. Training algorithms might also use storage
     *         volumes for scratch space. If you want Amazon SageMaker to use the storage volume to store the training
     *         data, choose <code>File</code> as the <code>TrainingInputMode</code> in the algorithm specification. For
     *         distributed training algorithms, specify an instance count greater than 1.
     */
    public ResourceConfig resourceConfig() {
        return resourceConfig;
    }

    /**
     * <p>
     * Specifies a limit to how long a model hyperparameter training job can run. It also specifies how long you are
     * willing to wait for a managed spot training job to complete. When the job reaches the a limit, Amazon SageMaker
     * ends the training job. Use this API to cap model training costs.
     * </p>
     * 
     * @return Specifies a limit to how long a model hyperparameter training job can run. It also specifies how long you
     *         are willing to wait for a managed spot training job to complete. When the job reaches the a limit, Amazon
     *         SageMaker ends the training job. Use this API to cap model training costs.
     */
    public StoppingCondition stoppingCondition() {
        return stoppingCondition;
    }

    /**
     * <p>
     * Isolates the training container. No inbound or outbound network calls can be made, except for calls between peers
     * within a training cluster for distributed training. If network isolation is used for training jobs that are
     * configured to use a VPC, Amazon SageMaker downloads and uploads customer data and model artifacts through the
     * specified VPC, but the training container does not have network access.
     * </p>
     * <note>
     * <p>
     * The Semantic Segmentation built-in algorithm does not support network isolation.
     * </p>
     * </note>
     * 
     * @return Isolates the training container. No inbound or outbound network calls can be made, except for calls
     *         between peers within a training cluster for distributed training. If network isolation is used for
     *         training jobs that are configured to use a VPC, Amazon SageMaker downloads and uploads customer data and
     *         model artifacts through the specified VPC, but the training container does not have network access.</p>
     *         <note>
     *         <p>
     *         The Semantic Segmentation built-in algorithm does not support network isolation.
     *         </p>
     */
    public Boolean enableNetworkIsolation() {
        return enableNetworkIsolation;
    }

    /**
     * <p>
     * To encrypt all communications between ML compute instances in distributed training, choose <code>True</code>.
     * Encryption provides greater security for distributed training, but training might take longer. How long it takes
     * depends on the amount of communication between compute instances, especially if you use a deep learning algorithm
     * in distributed training.
     * </p>
     * 
     * @return To encrypt all communications between ML compute instances in distributed training, choose
     *         <code>True</code>. Encryption provides greater security for distributed training, but training might take
     *         longer. How long it takes depends on the amount of communication between compute instances, especially if
     *         you use a deep learning algorithm in distributed training.
     */
    public Boolean enableInterContainerTrafficEncryption() {
        return enableInterContainerTrafficEncryption;
    }

    /**
     * <p>
     * A Boolean indicating whether managed spot training is enabled (<code>True</code>) or not (<code>False</code>).
     * </p>
     * 
     * @return A Boolean indicating whether managed spot training is enabled (<code>True</code>) or not (
     *         <code>False</code>).
     */
    public Boolean enableManagedSpotTraining() {
        return enableManagedSpotTraining;
    }

    /**
     * Returns the value of the CheckpointConfig property for this object.
     * 
     * @return The value of the CheckpointConfig property for this object.
     */
    public CheckpointConfig checkpointConfig() {
        return checkpointConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(staticHyperParameters());
        hashCode = 31 * hashCode + Objects.hashCode(algorithmSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(inputDataConfig());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfig());
        hashCode = 31 * hashCode + Objects.hashCode(outputDataConfig());
        hashCode = 31 * hashCode + Objects.hashCode(resourceConfig());
        hashCode = 31 * hashCode + Objects.hashCode(stoppingCondition());
        hashCode = 31 * hashCode + Objects.hashCode(enableNetworkIsolation());
        hashCode = 31 * hashCode + Objects.hashCode(enableInterContainerTrafficEncryption());
        hashCode = 31 * hashCode + Objects.hashCode(enableManagedSpotTraining());
        hashCode = 31 * hashCode + Objects.hashCode(checkpointConfig());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HyperParameterTrainingJobDefinition)) {
            return false;
        }
        HyperParameterTrainingJobDefinition other = (HyperParameterTrainingJobDefinition) obj;
        return Objects.equals(staticHyperParameters(), other.staticHyperParameters())
                && Objects.equals(algorithmSpecification(), other.algorithmSpecification())
                && Objects.equals(roleArn(), other.roleArn()) && Objects.equals(inputDataConfig(), other.inputDataConfig())
                && Objects.equals(vpcConfig(), other.vpcConfig()) && Objects.equals(outputDataConfig(), other.outputDataConfig())
                && Objects.equals(resourceConfig(), other.resourceConfig())
                && Objects.equals(stoppingCondition(), other.stoppingCondition())
                && Objects.equals(enableNetworkIsolation(), other.enableNetworkIsolation())
                && Objects.equals(enableInterContainerTrafficEncryption(), other.enableInterContainerTrafficEncryption())
                && Objects.equals(enableManagedSpotTraining(), other.enableManagedSpotTraining())
                && Objects.equals(checkpointConfig(), other.checkpointConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("HyperParameterTrainingJobDefinition").add("StaticHyperParameters", staticHyperParameters())
                .add("AlgorithmSpecification", algorithmSpecification()).add("RoleArn", roleArn())
                .add("InputDataConfig", inputDataConfig()).add("VpcConfig", vpcConfig())
                .add("OutputDataConfig", outputDataConfig()).add("ResourceConfig", resourceConfig())
                .add("StoppingCondition", stoppingCondition()).add("EnableNetworkIsolation", enableNetworkIsolation())
                .add("EnableInterContainerTrafficEncryption", enableInterContainerTrafficEncryption())
                .add("EnableManagedSpotTraining", enableManagedSpotTraining()).add("CheckpointConfig", checkpointConfig())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StaticHyperParameters":
            return Optional.ofNullable(clazz.cast(staticHyperParameters()));
        case "AlgorithmSpecification":
            return Optional.ofNullable(clazz.cast(algorithmSpecification()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "InputDataConfig":
            return Optional.ofNullable(clazz.cast(inputDataConfig()));
        case "VpcConfig":
            return Optional.ofNullable(clazz.cast(vpcConfig()));
        case "OutputDataConfig":
            return Optional.ofNullable(clazz.cast(outputDataConfig()));
        case "ResourceConfig":
            return Optional.ofNullable(clazz.cast(resourceConfig()));
        case "StoppingCondition":
            return Optional.ofNullable(clazz.cast(stoppingCondition()));
        case "EnableNetworkIsolation":
            return Optional.ofNullable(clazz.cast(enableNetworkIsolation()));
        case "EnableInterContainerTrafficEncryption":
            return Optional.ofNullable(clazz.cast(enableInterContainerTrafficEncryption()));
        case "EnableManagedSpotTraining":
            return Optional.ofNullable(clazz.cast(enableManagedSpotTraining()));
        case "CheckpointConfig":
            return Optional.ofNullable(clazz.cast(checkpointConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<HyperParameterTrainingJobDefinition, T> g) {
        return obj -> g.apply((HyperParameterTrainingJobDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HyperParameterTrainingJobDefinition> {
        /**
         * <p>
         * Specifies the values of hyperparameters that do not change for the tuning job.
         * </p>
         * 
         * @param staticHyperParameters
         *        Specifies the values of hyperparameters that do not change for the tuning job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder staticHyperParameters(Map<String, String> staticHyperParameters);

        /**
         * <p>
         * The <a>HyperParameterAlgorithmSpecification</a> object that specifies the resource algorithm to use for the
         * training jobs that the tuning job launches.
         * </p>
         * 
         * @param algorithmSpecification
         *        The <a>HyperParameterAlgorithmSpecification</a> object that specifies the resource algorithm to use
         *        for the training jobs that the tuning job launches.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder algorithmSpecification(HyperParameterAlgorithmSpecification algorithmSpecification);

        /**
         * <p>
         * The <a>HyperParameterAlgorithmSpecification</a> object that specifies the resource algorithm to use for the
         * training jobs that the tuning job launches.
         * </p>
         * This is a convenience that creates an instance of the {@link HyperParameterAlgorithmSpecification.Builder}
         * avoiding the need to create one manually via {@link HyperParameterAlgorithmSpecification#builder()}.
         *
         * When the {@link Consumer} completes, {@link HyperParameterAlgorithmSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #algorithmSpecification(HyperParameterAlgorithmSpecification)}
         * .
         * 
         * @param algorithmSpecification
         *        a consumer that will call methods on {@link HyperParameterAlgorithmSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #algorithmSpecification(HyperParameterAlgorithmSpecification)
         */
        default Builder algorithmSpecification(Consumer<HyperParameterAlgorithmSpecification.Builder> algorithmSpecification) {
            return algorithmSpecification(HyperParameterAlgorithmSpecification.builder().applyMutation(algorithmSpecification)
                    .build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role associated with the training jobs that the tuning job
         * launches.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of the IAM role associated with the training jobs that the tuning job
         *        launches.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * An array of <a>Channel</a> objects that specify the input for the training jobs that the tuning job launches.
         * </p>
         * 
         * @param inputDataConfig
         *        An array of <a>Channel</a> objects that specify the input for the training jobs that the tuning job
         *        launches.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputDataConfig(Collection<Channel> inputDataConfig);

        /**
         * <p>
         * An array of <a>Channel</a> objects that specify the input for the training jobs that the tuning job launches.
         * </p>
         * 
         * @param inputDataConfig
         *        An array of <a>Channel</a> objects that specify the input for the training jobs that the tuning job
         *        launches.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputDataConfig(Channel... inputDataConfig);

        /**
         * <p>
         * An array of <a>Channel</a> objects that specify the input for the training jobs that the tuning job launches.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Channel>.Builder} avoiding the need to
         * create one manually via {@link List<Channel>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Channel>.Builder#build()} is called immediately and its
         * result is passed to {@link #inputDataConfig(List<Channel>)}.
         * 
         * @param inputDataConfig
         *        a consumer that will call methods on {@link List<Channel>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputDataConfig(List<Channel>)
         */
        Builder inputDataConfig(Consumer<Channel.Builder>... inputDataConfig);

        /**
         * <p>
         * The <a>VpcConfig</a> object that specifies the VPC that you want the training jobs that this hyperparameter
         * tuning job launches to connect to. Control access to and from your training container by configuring the VPC.
         * For more information, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/train-vpc.html">Protect
         * Training Jobs by Using an Amazon Virtual Private Cloud</a>.
         * </p>
         * 
         * @param vpcConfig
         *        The <a>VpcConfig</a> object that specifies the VPC that you want the training jobs that this
         *        hyperparameter tuning job launches to connect to. Control access to and from your training container
         *        by configuring the VPC. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/train-vpc.html">Protect Training Jobs by Using
         *        an Amazon Virtual Private Cloud</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfig(VpcConfig vpcConfig);

        /**
         * <p>
         * The <a>VpcConfig</a> object that specifies the VPC that you want the training jobs that this hyperparameter
         * tuning job launches to connect to. Control access to and from your training container by configuring the VPC.
         * For more information, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/train-vpc.html">Protect
         * Training Jobs by Using an Amazon Virtual Private Cloud</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link VpcConfig.Builder} avoiding the need to create
         * one manually via {@link VpcConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link VpcConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #vpcConfig(VpcConfig)}.
         * 
         * @param vpcConfig
         *        a consumer that will call methods on {@link VpcConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfig(VpcConfig)
         */
        default Builder vpcConfig(Consumer<VpcConfig.Builder> vpcConfig) {
            return vpcConfig(VpcConfig.builder().applyMutation(vpcConfig).build());
        }

        /**
         * <p>
         * Specifies the path to the Amazon S3 bucket where you store model artifacts from the training jobs that the
         * tuning job launches.
         * </p>
         * 
         * @param outputDataConfig
         *        Specifies the path to the Amazon S3 bucket where you store model artifacts from the training jobs that
         *        the tuning job launches.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputDataConfig(OutputDataConfig outputDataConfig);

        /**
         * <p>
         * Specifies the path to the Amazon S3 bucket where you store model artifacts from the training jobs that the
         * tuning job launches.
         * </p>
         * This is a convenience that creates an instance of the {@link OutputDataConfig.Builder} avoiding the need to
         * create one manually via {@link OutputDataConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link OutputDataConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #outputDataConfig(OutputDataConfig)}.
         * 
         * @param outputDataConfig
         *        a consumer that will call methods on {@link OutputDataConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputDataConfig(OutputDataConfig)
         */
        default Builder outputDataConfig(Consumer<OutputDataConfig.Builder> outputDataConfig) {
            return outputDataConfig(OutputDataConfig.builder().applyMutation(outputDataConfig).build());
        }

        /**
         * <p>
         * The resources, including the compute instances and storage volumes, to use for the training jobs that the
         * tuning job launches.
         * </p>
         * <p>
         * Storage volumes store model artifacts and incremental states. Training algorithms might also use storage
         * volumes for scratch space. If you want Amazon SageMaker to use the storage volume to store the training data,
         * choose <code>File</code> as the <code>TrainingInputMode</code> in the algorithm specification. For
         * distributed training algorithms, specify an instance count greater than 1.
         * </p>
         * 
         * @param resourceConfig
         *        The resources, including the compute instances and storage volumes, to use for the training jobs that
         *        the tuning job launches.</p>
         *        <p>
         *        Storage volumes store model artifacts and incremental states. Training algorithms might also use
         *        storage volumes for scratch space. If you want Amazon SageMaker to use the storage volume to store the
         *        training data, choose <code>File</code> as the <code>TrainingInputMode</code> in the algorithm
         *        specification. For distributed training algorithms, specify an instance count greater than 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceConfig(ResourceConfig resourceConfig);

        /**
         * <p>
         * The resources, including the compute instances and storage volumes, to use for the training jobs that the
         * tuning job launches.
         * </p>
         * <p>
         * Storage volumes store model artifacts and incremental states. Training algorithms might also use storage
         * volumes for scratch space. If you want Amazon SageMaker to use the storage volume to store the training data,
         * choose <code>File</code> as the <code>TrainingInputMode</code> in the algorithm specification. For
         * distributed training algorithms, specify an instance count greater than 1.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceConfig.Builder} avoiding the need to
         * create one manually via {@link ResourceConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceConfig(ResourceConfig)}.
         * 
         * @param resourceConfig
         *        a consumer that will call methods on {@link ResourceConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceConfig(ResourceConfig)
         */
        default Builder resourceConfig(Consumer<ResourceConfig.Builder> resourceConfig) {
            return resourceConfig(ResourceConfig.builder().applyMutation(resourceConfig).build());
        }

        /**
         * <p>
         * Specifies a limit to how long a model hyperparameter training job can run. It also specifies how long you are
         * willing to wait for a managed spot training job to complete. When the job reaches the a limit, Amazon
         * SageMaker ends the training job. Use this API to cap model training costs.
         * </p>
         * 
         * @param stoppingCondition
         *        Specifies a limit to how long a model hyperparameter training job can run. It also specifies how long
         *        you are willing to wait for a managed spot training job to complete. When the job reaches the a limit,
         *        Amazon SageMaker ends the training job. Use this API to cap model training costs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppingCondition(StoppingCondition stoppingCondition);

        /**
         * <p>
         * Specifies a limit to how long a model hyperparameter training job can run. It also specifies how long you are
         * willing to wait for a managed spot training job to complete. When the job reaches the a limit, Amazon
         * SageMaker ends the training job. Use this API to cap model training costs.
         * </p>
         * This is a convenience that creates an instance of the {@link StoppingCondition.Builder} avoiding the need to
         * create one manually via {@link StoppingCondition#builder()}.
         *
         * When the {@link Consumer} completes, {@link StoppingCondition.Builder#build()} is called immediately and its
         * result is passed to {@link #stoppingCondition(StoppingCondition)}.
         * 
         * @param stoppingCondition
         *        a consumer that will call methods on {@link StoppingCondition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stoppingCondition(StoppingCondition)
         */
        default Builder stoppingCondition(Consumer<StoppingCondition.Builder> stoppingCondition) {
            return stoppingCondition(StoppingCondition.builder().applyMutation(stoppingCondition).build());
        }

        /**
         * <p>
         * Isolates the training container. No inbound or outbound network calls can be made, except for calls between
         * peers within a training cluster for distributed training. If network isolation is used for training jobs that
         * are configured to use a VPC, Amazon SageMaker downloads and uploads customer data and model artifacts through
         * the specified VPC, but the training container does not have network access.
         * </p>
         * <note>
         * <p>
         * The Semantic Segmentation built-in algorithm does not support network isolation.
         * </p>
         * </note>
         * 
         * @param enableNetworkIsolation
         *        Isolates the training container. No inbound or outbound network calls can be made, except for calls
         *        between peers within a training cluster for distributed training. If network isolation is used for
         *        training jobs that are configured to use a VPC, Amazon SageMaker downloads and uploads customer data
         *        and model artifacts through the specified VPC, but the training container does not have network
         *        access.</p> <note>
         *        <p>
         *        The Semantic Segmentation built-in algorithm does not support network isolation.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableNetworkIsolation(Boolean enableNetworkIsolation);

        /**
         * <p>
         * To encrypt all communications between ML compute instances in distributed training, choose <code>True</code>.
         * Encryption provides greater security for distributed training, but training might take longer. How long it
         * takes depends on the amount of communication between compute instances, especially if you use a deep learning
         * algorithm in distributed training.
         * </p>
         * 
         * @param enableInterContainerTrafficEncryption
         *        To encrypt all communications between ML compute instances in distributed training, choose
         *        <code>True</code>. Encryption provides greater security for distributed training, but training might
         *        take longer. How long it takes depends on the amount of communication between compute instances,
         *        especially if you use a deep learning algorithm in distributed training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableInterContainerTrafficEncryption(Boolean enableInterContainerTrafficEncryption);

        /**
         * <p>
         * A Boolean indicating whether managed spot training is enabled (<code>True</code>) or not (<code>False</code>
         * ).
         * </p>
         * 
         * @param enableManagedSpotTraining
         *        A Boolean indicating whether managed spot training is enabled (<code>True</code>) or not (
         *        <code>False</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableManagedSpotTraining(Boolean enableManagedSpotTraining);

        /**
         * Sets the value of the CheckpointConfig property for this object.
         *
         * @param checkpointConfig
         *        The new value for the CheckpointConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checkpointConfig(CheckpointConfig checkpointConfig);

        /**
         * Sets the value of the CheckpointConfig property for this object.
         *
         * This is a convenience that creates an instance of the {@link CheckpointConfig.Builder} avoiding the need to
         * create one manually via {@link CheckpointConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link CheckpointConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #checkpointConfig(CheckpointConfig)}.
         * 
         * @param checkpointConfig
         *        a consumer that will call methods on {@link CheckpointConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #checkpointConfig(CheckpointConfig)
         */
        default Builder checkpointConfig(Consumer<CheckpointConfig.Builder> checkpointConfig) {
            return checkpointConfig(CheckpointConfig.builder().applyMutation(checkpointConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Map<String, String> staticHyperParameters = DefaultSdkAutoConstructMap.getInstance();

        private HyperParameterAlgorithmSpecification algorithmSpecification;

        private String roleArn;

        private List<Channel> inputDataConfig = DefaultSdkAutoConstructList.getInstance();

        private VpcConfig vpcConfig;

        private OutputDataConfig outputDataConfig;

        private ResourceConfig resourceConfig;

        private StoppingCondition stoppingCondition;

        private Boolean enableNetworkIsolation;

        private Boolean enableInterContainerTrafficEncryption;

        private Boolean enableManagedSpotTraining;

        private CheckpointConfig checkpointConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(HyperParameterTrainingJobDefinition model) {
            staticHyperParameters(model.staticHyperParameters);
            algorithmSpecification(model.algorithmSpecification);
            roleArn(model.roleArn);
            inputDataConfig(model.inputDataConfig);
            vpcConfig(model.vpcConfig);
            outputDataConfig(model.outputDataConfig);
            resourceConfig(model.resourceConfig);
            stoppingCondition(model.stoppingCondition);
            enableNetworkIsolation(model.enableNetworkIsolation);
            enableInterContainerTrafficEncryption(model.enableInterContainerTrafficEncryption);
            enableManagedSpotTraining(model.enableManagedSpotTraining);
            checkpointConfig(model.checkpointConfig);
        }

        public final Map<String, String> getStaticHyperParameters() {
            return staticHyperParameters;
        }

        @Override
        public final Builder staticHyperParameters(Map<String, String> staticHyperParameters) {
            this.staticHyperParameters = HyperParametersCopier.copy(staticHyperParameters);
            return this;
        }

        public final void setStaticHyperParameters(Map<String, String> staticHyperParameters) {
            this.staticHyperParameters = HyperParametersCopier.copy(staticHyperParameters);
        }

        public final HyperParameterAlgorithmSpecification.Builder getAlgorithmSpecification() {
            return algorithmSpecification != null ? algorithmSpecification.toBuilder() : null;
        }

        @Override
        public final Builder algorithmSpecification(HyperParameterAlgorithmSpecification algorithmSpecification) {
            this.algorithmSpecification = algorithmSpecification;
            return this;
        }

        public final void setAlgorithmSpecification(HyperParameterAlgorithmSpecification.BuilderImpl algorithmSpecification) {
            this.algorithmSpecification = algorithmSpecification != null ? algorithmSpecification.build() : null;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        public final Collection<Channel.Builder> getInputDataConfig() {
            return inputDataConfig != null ? inputDataConfig.stream().map(Channel::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder inputDataConfig(Collection<Channel> inputDataConfig) {
            this.inputDataConfig = InputDataConfigCopier.copy(inputDataConfig);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputDataConfig(Channel... inputDataConfig) {
            inputDataConfig(Arrays.asList(inputDataConfig));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputDataConfig(Consumer<Channel.Builder>... inputDataConfig) {
            inputDataConfig(Stream.of(inputDataConfig).map(c -> Channel.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setInputDataConfig(Collection<Channel.BuilderImpl> inputDataConfig) {
            this.inputDataConfig = InputDataConfigCopier.copyFromBuilder(inputDataConfig);
        }

        public final VpcConfig.Builder getVpcConfig() {
            return vpcConfig != null ? vpcConfig.toBuilder() : null;
        }

        @Override
        public final Builder vpcConfig(VpcConfig vpcConfig) {
            this.vpcConfig = vpcConfig;
            return this;
        }

        public final void setVpcConfig(VpcConfig.BuilderImpl vpcConfig) {
            this.vpcConfig = vpcConfig != null ? vpcConfig.build() : null;
        }

        public final OutputDataConfig.Builder getOutputDataConfig() {
            return outputDataConfig != null ? outputDataConfig.toBuilder() : null;
        }

        @Override
        public final Builder outputDataConfig(OutputDataConfig outputDataConfig) {
            this.outputDataConfig = outputDataConfig;
            return this;
        }

        public final void setOutputDataConfig(OutputDataConfig.BuilderImpl outputDataConfig) {
            this.outputDataConfig = outputDataConfig != null ? outputDataConfig.build() : null;
        }

        public final ResourceConfig.Builder getResourceConfig() {
            return resourceConfig != null ? resourceConfig.toBuilder() : null;
        }

        @Override
        public final Builder resourceConfig(ResourceConfig resourceConfig) {
            this.resourceConfig = resourceConfig;
            return this;
        }

        public final void setResourceConfig(ResourceConfig.BuilderImpl resourceConfig) {
            this.resourceConfig = resourceConfig != null ? resourceConfig.build() : null;
        }

        public final StoppingCondition.Builder getStoppingCondition() {
            return stoppingCondition != null ? stoppingCondition.toBuilder() : null;
        }

        @Override
        public final Builder stoppingCondition(StoppingCondition stoppingCondition) {
            this.stoppingCondition = stoppingCondition;
            return this;
        }

        public final void setStoppingCondition(StoppingCondition.BuilderImpl stoppingCondition) {
            this.stoppingCondition = stoppingCondition != null ? stoppingCondition.build() : null;
        }

        public final Boolean getEnableNetworkIsolation() {
            return enableNetworkIsolation;
        }

        @Override
        public final Builder enableNetworkIsolation(Boolean enableNetworkIsolation) {
            this.enableNetworkIsolation = enableNetworkIsolation;
            return this;
        }

        public final void setEnableNetworkIsolation(Boolean enableNetworkIsolation) {
            this.enableNetworkIsolation = enableNetworkIsolation;
        }

        public final Boolean getEnableInterContainerTrafficEncryption() {
            return enableInterContainerTrafficEncryption;
        }

        @Override
        public final Builder enableInterContainerTrafficEncryption(Boolean enableInterContainerTrafficEncryption) {
            this.enableInterContainerTrafficEncryption = enableInterContainerTrafficEncryption;
            return this;
        }

        public final void setEnableInterContainerTrafficEncryption(Boolean enableInterContainerTrafficEncryption) {
            this.enableInterContainerTrafficEncryption = enableInterContainerTrafficEncryption;
        }

        public final Boolean getEnableManagedSpotTraining() {
            return enableManagedSpotTraining;
        }

        @Override
        public final Builder enableManagedSpotTraining(Boolean enableManagedSpotTraining) {
            this.enableManagedSpotTraining = enableManagedSpotTraining;
            return this;
        }

        public final void setEnableManagedSpotTraining(Boolean enableManagedSpotTraining) {
            this.enableManagedSpotTraining = enableManagedSpotTraining;
        }

        public final CheckpointConfig.Builder getCheckpointConfig() {
            return checkpointConfig != null ? checkpointConfig.toBuilder() : null;
        }

        @Override
        public final Builder checkpointConfig(CheckpointConfig checkpointConfig) {
            this.checkpointConfig = checkpointConfig;
            return this;
        }

        public final void setCheckpointConfig(CheckpointConfig.BuilderImpl checkpointConfig) {
            this.checkpointConfig = checkpointConfig != null ? checkpointConfig.build() : null;
        }

        @Override
        public HyperParameterTrainingJobDefinition build() {
            return new HyperParameterTrainingJobDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
