/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An array element of <a>DescribeTrainingJobResponse$SecondaryStatusTransitions</a>. It provides additional details
 * about a status that the training job has transitioned through. A training job can be in one of several states, for
 * example, starting, downloading, training, or uploading. Within each state, there are a number of intermediate states.
 * For example, within the starting state, Amazon SageMaker could be starting the training job or launching the ML
 * instances. These transitional states are referred to as the job's secondary status.
 * </p>
 * <p/>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SecondaryStatusTransition implements SdkPojo, Serializable,
        ToCopyableBuilder<SecondaryStatusTransition.Builder, SecondaryStatusTransition> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SecondaryStatusTransition::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(SecondaryStatusTransition::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(SecondaryStatusTransition::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SecondaryStatusTransition::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            START_TIME_FIELD, END_TIME_FIELD, STATUS_MESSAGE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String status;

    private final Instant startTime;

    private final Instant endTime;

    private final String statusMessage;

    private SecondaryStatusTransition(BuilderImpl builder) {
        this.status = builder.status;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.statusMessage = builder.statusMessage;
    }

    /**
     * <p>
     * Contains a secondary status information from a training job.
     * </p>
     * <p>
     * Status might be one of the following secondary statuses:
     * </p>
     * <dl>
     * <dt>InProgress</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * <code>Starting</code> - Starting the training job.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Downloading</code> - An optional stage for algorithms that support <code>File</code> training input mode.
     * It indicates that data is being downloaded to the ML storage volumes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Training</code> - Training is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Uploading</code> - Training is complete and the model artifacts are being uploaded to the S3 location.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>Completed</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * <code>Completed</code> - The training job has completed.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>Failed</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * <code>Failed</code> - The training job has failed. The reason for the failure is returned in the
     * <code>FailureReason</code> field of <code>DescribeTrainingJobResponse</code>.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>Stopped</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * <code>MaxRuntimeExceeded</code> - The job stopped because it exceeded the maximum allowed runtime.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Stopped</code> - The training job has stopped.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>Stopping</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * <code>Stopping</code> - Stopping the training job.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * </dl>
     * <p>
     * We no longer support the following secondary statuses:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LaunchingMLInstances</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PreparingTrainingStack</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DownloadingTrainingImage</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link SecondaryStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Contains a secondary status information from a training job.</p>
     *         <p>
     *         Status might be one of the following secondary statuses:
     *         </p>
     *         <dl>
     *         <dt>InProgress</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Starting</code> - Starting the training job.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Downloading</code> - An optional stage for algorithms that support <code>File</code> training input
     *         mode. It indicates that data is being downloaded to the ML storage volumes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Training</code> - Training is in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Uploading</code> - Training is complete and the model artifacts are being uploaded to the S3
     *         location.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         <dt>Completed</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Completed</code> - The training job has completed.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         <dt>Failed</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Failed</code> - The training job has failed. The reason for the failure is returned in the
     *         <code>FailureReason</code> field of <code>DescribeTrainingJobResponse</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         <dt>Stopped</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MaxRuntimeExceeded</code> - The job stopped because it exceeded the maximum allowed runtime.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Stopped</code> - The training job has stopped.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         <dt>Stopping</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Stopping</code> - Stopping the training job.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         </dl>
     *         <p>
     *         We no longer support the following secondary statuses:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>LaunchingMLInstances</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PreparingTrainingStack</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DownloadingTrainingImage</code>
     *         </p>
     *         </li>
     * @see SecondaryStatus
     */
    public SecondaryStatus status() {
        return SecondaryStatus.fromValue(status);
    }

    /**
     * <p>
     * Contains a secondary status information from a training job.
     * </p>
     * <p>
     * Status might be one of the following secondary statuses:
     * </p>
     * <dl>
     * <dt>InProgress</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * <code>Starting</code> - Starting the training job.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Downloading</code> - An optional stage for algorithms that support <code>File</code> training input mode.
     * It indicates that data is being downloaded to the ML storage volumes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Training</code> - Training is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Uploading</code> - Training is complete and the model artifacts are being uploaded to the S3 location.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>Completed</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * <code>Completed</code> - The training job has completed.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>Failed</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * <code>Failed</code> - The training job has failed. The reason for the failure is returned in the
     * <code>FailureReason</code> field of <code>DescribeTrainingJobResponse</code>.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>Stopped</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * <code>MaxRuntimeExceeded</code> - The job stopped because it exceeded the maximum allowed runtime.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Stopped</code> - The training job has stopped.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>Stopping</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * <code>Stopping</code> - Stopping the training job.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * </dl>
     * <p>
     * We no longer support the following secondary statuses:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LaunchingMLInstances</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PreparingTrainingStack</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DownloadingTrainingImage</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link SecondaryStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Contains a secondary status information from a training job.</p>
     *         <p>
     *         Status might be one of the following secondary statuses:
     *         </p>
     *         <dl>
     *         <dt>InProgress</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Starting</code> - Starting the training job.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Downloading</code> - An optional stage for algorithms that support <code>File</code> training input
     *         mode. It indicates that data is being downloaded to the ML storage volumes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Training</code> - Training is in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Uploading</code> - Training is complete and the model artifacts are being uploaded to the S3
     *         location.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         <dt>Completed</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Completed</code> - The training job has completed.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         <dt>Failed</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Failed</code> - The training job has failed. The reason for the failure is returned in the
     *         <code>FailureReason</code> field of <code>DescribeTrainingJobResponse</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         <dt>Stopped</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MaxRuntimeExceeded</code> - The job stopped because it exceeded the maximum allowed runtime.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Stopped</code> - The training job has stopped.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         <dt>Stopping</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Stopping</code> - Stopping the training job.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         </dl>
     *         <p>
     *         We no longer support the following secondary statuses:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>LaunchingMLInstances</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PreparingTrainingStack</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DownloadingTrainingImage</code>
     *         </p>
     *         </li>
     * @see SecondaryStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A timestamp that shows when the training job transitioned to the current secondary status state.
     * </p>
     * 
     * @return A timestamp that shows when the training job transitioned to the current secondary status state.
     */
    public Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * A timestamp that shows when the training job transitioned out of this secondary status state into another
     * secondary status state or when the training job has ended.
     * </p>
     * 
     * @return A timestamp that shows when the training job transitioned out of this secondary status state into another
     *         secondary status state or when the training job has ended.
     */
    public Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * A detailed description of the progress within a secondary status.
     * </p>
     * <p>
     * Amazon SageMaker provides secondary statuses and status messages that apply to each of them:
     * </p>
     * <dl>
     * <dt>Starting</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * Starting the training job.
     * </p>
     * </li>
     * <li>
     * <p>
     * Launching requested ML instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * Insufficient capacity error from EC2 while launching instances, retrying!
     * </p>
     * </li>
     * <li>
     * <p>
     * Launched instance was unhealthy, replacing it!
     * </p>
     * </li>
     * <li>
     * <p>
     * Preparing the instances for training.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>Training</dt>
     * <dd>
     * <ul>
     * <li>
     * <p>
     * Downloading the training image.
     * </p>
     * </li>
     * <li>
     * <p>
     * Training image download completed. Training in progress.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * </dl>
     * <important>
     * <p>
     * Status messages are subject to change. Therefore, we recommend not including them in code that programmatically
     * initiates actions. For examples, don't use status messages in if statements.
     * </p>
     * </important>
     * <p>
     * To have an overview of your training job's progress, view <code>TrainingJobStatus</code> and
     * <code>SecondaryStatus</code> in <a>DescribeTrainingJob</a>, and <code>StatusMessage</code> together. For example,
     * at the start of a training job, you might see the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TrainingJobStatus</code> - InProgress
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SecondaryStatus</code> - Training
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StatusMessage</code> - Downloading the training image
     * </p>
     * </li>
     * </ul>
     * 
     * @return A detailed description of the progress within a secondary status. </p>
     *         <p>
     *         Amazon SageMaker provides secondary statuses and status messages that apply to each of them:
     *         </p>
     *         <dl>
     *         <dt>Starting</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         Starting the training job.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Launching requested ML instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Insufficient capacity error from EC2 while launching instances, retrying!
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Launched instance was unhealthy, replacing it!
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Preparing the instances for training.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         <dt>Training</dt>
     *         <dd>
     *         <ul>
     *         <li>
     *         <p>
     *         Downloading the training image.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Training image download completed. Training in progress.
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         </dl>
     *         <important>
     *         <p>
     *         Status messages are subject to change. Therefore, we recommend not including them in code that
     *         programmatically initiates actions. For examples, don't use status messages in if statements.
     *         </p>
     *         </important>
     *         <p>
     *         To have an overview of your training job's progress, view <code>TrainingJobStatus</code> and
     *         <code>SecondaryStatus</code> in <a>DescribeTrainingJob</a>, and <code>StatusMessage</code> together. For
     *         example, at the start of a training job, you might see the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>TrainingJobStatus</code> - InProgress
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SecondaryStatus</code> - Training
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>StatusMessage</code> - Downloading the training image
     *         </p>
     *         </li>
     */
    public String statusMessage() {
        return statusMessage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SecondaryStatusTransition)) {
            return false;
        }
        SecondaryStatusTransition other = (SecondaryStatusTransition) obj;
        return Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(statusMessage(), other.statusMessage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SecondaryStatusTransition").add("Status", statusAsString()).add("StartTime", startTime())
                .add("EndTime", endTime()).add("StatusMessage", statusMessage()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SecondaryStatusTransition, T> g) {
        return obj -> g.apply((SecondaryStatusTransition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SecondaryStatusTransition> {
        /**
         * <p>
         * Contains a secondary status information from a training job.
         * </p>
         * <p>
         * Status might be one of the following secondary statuses:
         * </p>
         * <dl>
         * <dt>InProgress</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * <code>Starting</code> - Starting the training job.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Downloading</code> - An optional stage for algorithms that support <code>File</code> training input
         * mode. It indicates that data is being downloaded to the ML storage volumes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Training</code> - Training is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Uploading</code> - Training is complete and the model artifacts are being uploaded to the S3 location.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * <dt>Completed</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * <code>Completed</code> - The training job has completed.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * <dt>Failed</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * <code>Failed</code> - The training job has failed. The reason for the failure is returned in the
         * <code>FailureReason</code> field of <code>DescribeTrainingJobResponse</code>.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * <dt>Stopped</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * <code>MaxRuntimeExceeded</code> - The job stopped because it exceeded the maximum allowed runtime.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Stopped</code> - The training job has stopped.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * <dt>Stopping</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * <code>Stopping</code> - Stopping the training job.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * </dl>
         * <p>
         * We no longer support the following secondary statuses:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>LaunchingMLInstances</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PreparingTrainingStack</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DownloadingTrainingImage</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Contains a secondary status information from a training job.</p>
         *        <p>
         *        Status might be one of the following secondary statuses:
         *        </p>
         *        <dl>
         *        <dt>InProgress</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Starting</code> - Starting the training job.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Downloading</code> - An optional stage for algorithms that support <code>File</code> training
         *        input mode. It indicates that data is being downloaded to the ML storage volumes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Training</code> - Training is in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Uploading</code> - Training is complete and the model artifacts are being uploaded to the S3
         *        location.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        <dt>Completed</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Completed</code> - The training job has completed.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        <dt>Failed</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Failed</code> - The training job has failed. The reason for the failure is returned in the
         *        <code>FailureReason</code> field of <code>DescribeTrainingJobResponse</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        <dt>Stopped</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MaxRuntimeExceeded</code> - The job stopped because it exceeded the maximum allowed runtime.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Stopped</code> - The training job has stopped.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        <dt>Stopping</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Stopping</code> - Stopping the training job.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        </dl>
         *        <p>
         *        We no longer support the following secondary statuses:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LaunchingMLInstances</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PreparingTrainingStack</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DownloadingTrainingImage</code>
         *        </p>
         *        </li>
         * @see SecondaryStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SecondaryStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Contains a secondary status information from a training job.
         * </p>
         * <p>
         * Status might be one of the following secondary statuses:
         * </p>
         * <dl>
         * <dt>InProgress</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * <code>Starting</code> - Starting the training job.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Downloading</code> - An optional stage for algorithms that support <code>File</code> training input
         * mode. It indicates that data is being downloaded to the ML storage volumes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Training</code> - Training is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Uploading</code> - Training is complete and the model artifacts are being uploaded to the S3 location.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * <dt>Completed</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * <code>Completed</code> - The training job has completed.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * <dt>Failed</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * <code>Failed</code> - The training job has failed. The reason for the failure is returned in the
         * <code>FailureReason</code> field of <code>DescribeTrainingJobResponse</code>.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * <dt>Stopped</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * <code>MaxRuntimeExceeded</code> - The job stopped because it exceeded the maximum allowed runtime.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Stopped</code> - The training job has stopped.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * <dt>Stopping</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * <code>Stopping</code> - Stopping the training job.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * </dl>
         * <p>
         * We no longer support the following secondary statuses:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>LaunchingMLInstances</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PreparingTrainingStack</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DownloadingTrainingImage</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Contains a secondary status information from a training job.</p>
         *        <p>
         *        Status might be one of the following secondary statuses:
         *        </p>
         *        <dl>
         *        <dt>InProgress</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Starting</code> - Starting the training job.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Downloading</code> - An optional stage for algorithms that support <code>File</code> training
         *        input mode. It indicates that data is being downloaded to the ML storage volumes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Training</code> - Training is in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Uploading</code> - Training is complete and the model artifacts are being uploaded to the S3
         *        location.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        <dt>Completed</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Completed</code> - The training job has completed.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        <dt>Failed</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Failed</code> - The training job has failed. The reason for the failure is returned in the
         *        <code>FailureReason</code> field of <code>DescribeTrainingJobResponse</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        <dt>Stopped</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MaxRuntimeExceeded</code> - The job stopped because it exceeded the maximum allowed runtime.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Stopped</code> - The training job has stopped.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        <dt>Stopping</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Stopping</code> - Stopping the training job.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        </dl>
         *        <p>
         *        We no longer support the following secondary statuses:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LaunchingMLInstances</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PreparingTrainingStack</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DownloadingTrainingImage</code>
         *        </p>
         *        </li>
         * @see SecondaryStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SecondaryStatus
         */
        Builder status(SecondaryStatus status);

        /**
         * <p>
         * A timestamp that shows when the training job transitioned to the current secondary status state.
         * </p>
         * 
         * @param startTime
         *        A timestamp that shows when the training job transitioned to the current secondary status state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * A timestamp that shows when the training job transitioned out of this secondary status state into another
         * secondary status state or when the training job has ended.
         * </p>
         * 
         * @param endTime
         *        A timestamp that shows when the training job transitioned out of this secondary status state into
         *        another secondary status state or when the training job has ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * A detailed description of the progress within a secondary status.
         * </p>
         * <p>
         * Amazon SageMaker provides secondary statuses and status messages that apply to each of them:
         * </p>
         * <dl>
         * <dt>Starting</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * Starting the training job.
         * </p>
         * </li>
         * <li>
         * <p>
         * Launching requested ML instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * Insufficient capacity error from EC2 while launching instances, retrying!
         * </p>
         * </li>
         * <li>
         * <p>
         * Launched instance was unhealthy, replacing it!
         * </p>
         * </li>
         * <li>
         * <p>
         * Preparing the instances for training.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * <dt>Training</dt>
         * <dd>
         * <ul>
         * <li>
         * <p>
         * Downloading the training image.
         * </p>
         * </li>
         * <li>
         * <p>
         * Training image download completed. Training in progress.
         * </p>
         * </li>
         * </ul>
         * </dd>
         * </dl>
         * <important>
         * <p>
         * Status messages are subject to change. Therefore, we recommend not including them in code that
         * programmatically initiates actions. For examples, don't use status messages in if statements.
         * </p>
         * </important>
         * <p>
         * To have an overview of your training job's progress, view <code>TrainingJobStatus</code> and
         * <code>SecondaryStatus</code> in <a>DescribeTrainingJob</a>, and <code>StatusMessage</code> together. For
         * example, at the start of a training job, you might see the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TrainingJobStatus</code> - InProgress
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SecondaryStatus</code> - Training
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>StatusMessage</code> - Downloading the training image
         * </p>
         * </li>
         * </ul>
         * 
         * @param statusMessage
         *        A detailed description of the progress within a secondary status. </p>
         *        <p>
         *        Amazon SageMaker provides secondary statuses and status messages that apply to each of them:
         *        </p>
         *        <dl>
         *        <dt>Starting</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        Starting the training job.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Launching requested ML instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Insufficient capacity error from EC2 while launching instances, retrying!
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Launched instance was unhealthy, replacing it!
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Preparing the instances for training.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        <dt>Training</dt>
         *        <dd>
         *        <ul>
         *        <li>
         *        <p>
         *        Downloading the training image.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Training image download completed. Training in progress.
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        </dl>
         *        <important>
         *        <p>
         *        Status messages are subject to change. Therefore, we recommend not including them in code that
         *        programmatically initiates actions. For examples, don't use status messages in if statements.
         *        </p>
         *        </important>
         *        <p>
         *        To have an overview of your training job's progress, view <code>TrainingJobStatus</code> and
         *        <code>SecondaryStatus</code> in <a>DescribeTrainingJob</a>, and <code>StatusMessage</code> together.
         *        For example, at the start of a training job, you might see the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TrainingJobStatus</code> - InProgress
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SecondaryStatus</code> - Training
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>StatusMessage</code> - Downloading the training image
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private Instant startTime;

        private Instant endTime;

        private String statusMessage;

        private BuilderImpl() {
        }

        private BuilderImpl(SecondaryStatusTransition model) {
            status(model.status);
            startTime(model.startTime);
            endTime(model.endTime);
            statusMessage(model.statusMessage);
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(SecondaryStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public SecondaryStatusTransition build() {
            return new SecondaryStatusTransition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
