/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides summary information for an endpoint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EndpointSummary implements SdkPojo, Serializable, ToCopyableBuilder<EndpointSummary.Builder, EndpointSummary> {
    private static final SdkField<String> ENDPOINT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EndpointSummary::endpointName)).setter(setter(Builder::endpointName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointName").build()).build();

    private static final SdkField<String> ENDPOINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EndpointSummary::endpointArn)).setter(setter(Builder::endpointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointArn").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(EndpointSummary::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(EndpointSummary::lastModifiedTime)).setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime").build()).build();

    private static final SdkField<String> ENDPOINT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EndpointSummary::endpointStatusAsString)).setter(setter(Builder::endpointStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointStatus").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENDPOINT_NAME_FIELD,
            ENDPOINT_ARN_FIELD, CREATION_TIME_FIELD, LAST_MODIFIED_TIME_FIELD, ENDPOINT_STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String endpointName;

    private final String endpointArn;

    private final Instant creationTime;

    private final Instant lastModifiedTime;

    private final String endpointStatus;

    private EndpointSummary(BuilderImpl builder) {
        this.endpointName = builder.endpointName;
        this.endpointArn = builder.endpointArn;
        this.creationTime = builder.creationTime;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.endpointStatus = builder.endpointStatus;
    }

    /**
     * <p>
     * The name of the endpoint.
     * </p>
     * 
     * @return The name of the endpoint.
     */
    public String endpointName() {
        return endpointName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the endpoint.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the endpoint.
     */
    public String endpointArn() {
        return endpointArn;
    }

    /**
     * <p>
     * A timestamp that shows when the endpoint was created.
     * </p>
     * 
     * @return A timestamp that shows when the endpoint was created.
     */
    public Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * A timestamp that shows when the endpoint was last modified.
     * </p>
     * 
     * @return A timestamp that shows when the endpoint was last modified.
     */
    public Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * <p>
     * The status of the endpoint.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>OutOfService</code>: Endpoint is not available to take incoming requests.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Creating</code>: <a>CreateEndpoint</a> is executing.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Updating</code>: <a>UpdateEndpoint</a> or <a>UpdateEndpointWeightsAndCapacities</a> is executing.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SystemUpdating</code>: Endpoint is undergoing maintenance and cannot be updated or deleted or re-scaled
     * until it has completed. This maintenance operation does not change any customer-specified values such as VPC
     * config, KMS encryption, model, instance type, or instance count.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RollingBack</code>: Endpoint fails to scale up or down or change its variant weight and is in the process
     * of rolling back to its previous configuration. Once the rollback completes, endpoint returns to an
     * <code>InService</code> status. This transitional status only applies to an endpoint that has autoscaling enabled
     * and is undergoing variant weight or capacity changes as part of an <a>UpdateEndpointWeightsAndCapacities</a> call
     * or when the <a>UpdateEndpointWeightsAndCapacities</a> operation is called explicitly.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>InService</code>: Endpoint is available to process incoming requests.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Deleting</code>: <a>DeleteEndpoint</a> is executing.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code>: Endpoint could not be created, updated, or re-scaled. Use
     * <a>DescribeEndpointOutput$FailureReason</a> for information about the failure. <a>DeleteEndpoint</a> is the only
     * operation that can be performed on a failed endpoint.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get a list of endpoints with a specified status, use the <a>ListEndpointsInput$StatusEquals</a> filter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointStatus}
     * will return {@link EndpointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #endpointStatusAsString}.
     * </p>
     * 
     * @return The status of the endpoint.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>OutOfService</code>: Endpoint is not available to take incoming requests.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Creating</code>: <a>CreateEndpoint</a> is executing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Updating</code>: <a>UpdateEndpoint</a> or <a>UpdateEndpointWeightsAndCapacities</a> is executing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SystemUpdating</code>: Endpoint is undergoing maintenance and cannot be updated or deleted or
     *         re-scaled until it has completed. This maintenance operation does not change any customer-specified
     *         values such as VPC config, KMS encryption, model, instance type, or instance count.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RollingBack</code>: Endpoint fails to scale up or down or change its variant weight and is in the
     *         process of rolling back to its previous configuration. Once the rollback completes, endpoint returns to
     *         an <code>InService</code> status. This transitional status only applies to an endpoint that has
     *         autoscaling enabled and is undergoing variant weight or capacity changes as part of an
     *         <a>UpdateEndpointWeightsAndCapacities</a> call or when the <a>UpdateEndpointWeightsAndCapacities</a>
     *         operation is called explicitly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>InService</code>: Endpoint is available to process incoming requests.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Deleting</code>: <a>DeleteEndpoint</a> is executing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code>: Endpoint could not be created, updated, or re-scaled. Use
     *         <a>DescribeEndpointOutput$FailureReason</a> for information about the failure. <a>DeleteEndpoint</a> is
     *         the only operation that can be performed on a failed endpoint.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To get a list of endpoints with a specified status, use the <a>ListEndpointsInput$StatusEquals</a>
     *         filter.
     * @see EndpointStatus
     */
    public EndpointStatus endpointStatus() {
        return EndpointStatus.fromValue(endpointStatus);
    }

    /**
     * <p>
     * The status of the endpoint.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>OutOfService</code>: Endpoint is not available to take incoming requests.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Creating</code>: <a>CreateEndpoint</a> is executing.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Updating</code>: <a>UpdateEndpoint</a> or <a>UpdateEndpointWeightsAndCapacities</a> is executing.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SystemUpdating</code>: Endpoint is undergoing maintenance and cannot be updated or deleted or re-scaled
     * until it has completed. This maintenance operation does not change any customer-specified values such as VPC
     * config, KMS encryption, model, instance type, or instance count.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RollingBack</code>: Endpoint fails to scale up or down or change its variant weight and is in the process
     * of rolling back to its previous configuration. Once the rollback completes, endpoint returns to an
     * <code>InService</code> status. This transitional status only applies to an endpoint that has autoscaling enabled
     * and is undergoing variant weight or capacity changes as part of an <a>UpdateEndpointWeightsAndCapacities</a> call
     * or when the <a>UpdateEndpointWeightsAndCapacities</a> operation is called explicitly.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>InService</code>: Endpoint is available to process incoming requests.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Deleting</code>: <a>DeleteEndpoint</a> is executing.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code>: Endpoint could not be created, updated, or re-scaled. Use
     * <a>DescribeEndpointOutput$FailureReason</a> for information about the failure. <a>DeleteEndpoint</a> is the only
     * operation that can be performed on a failed endpoint.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get a list of endpoints with a specified status, use the <a>ListEndpointsInput$StatusEquals</a> filter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointStatus}
     * will return {@link EndpointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #endpointStatusAsString}.
     * </p>
     * 
     * @return The status of the endpoint.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>OutOfService</code>: Endpoint is not available to take incoming requests.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Creating</code>: <a>CreateEndpoint</a> is executing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Updating</code>: <a>UpdateEndpoint</a> or <a>UpdateEndpointWeightsAndCapacities</a> is executing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SystemUpdating</code>: Endpoint is undergoing maintenance and cannot be updated or deleted or
     *         re-scaled until it has completed. This maintenance operation does not change any customer-specified
     *         values such as VPC config, KMS encryption, model, instance type, or instance count.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RollingBack</code>: Endpoint fails to scale up or down or change its variant weight and is in the
     *         process of rolling back to its previous configuration. Once the rollback completes, endpoint returns to
     *         an <code>InService</code> status. This transitional status only applies to an endpoint that has
     *         autoscaling enabled and is undergoing variant weight or capacity changes as part of an
     *         <a>UpdateEndpointWeightsAndCapacities</a> call or when the <a>UpdateEndpointWeightsAndCapacities</a>
     *         operation is called explicitly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>InService</code>: Endpoint is available to process incoming requests.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Deleting</code>: <a>DeleteEndpoint</a> is executing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code>: Endpoint could not be created, updated, or re-scaled. Use
     *         <a>DescribeEndpointOutput$FailureReason</a> for information about the failure. <a>DeleteEndpoint</a> is
     *         the only operation that can be performed on a failed endpoint.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To get a list of endpoints with a specified status, use the <a>ListEndpointsInput$StatusEquals</a>
     *         filter.
     * @see EndpointStatus
     */
    public String endpointStatusAsString() {
        return endpointStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(endpointName());
        hashCode = 31 * hashCode + Objects.hashCode(endpointArn());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(endpointStatusAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EndpointSummary)) {
            return false;
        }
        EndpointSummary other = (EndpointSummary) obj;
        return Objects.equals(endpointName(), other.endpointName()) && Objects.equals(endpointArn(), other.endpointArn())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && Objects.equals(endpointStatusAsString(), other.endpointStatusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("EndpointSummary").add("EndpointName", endpointName()).add("EndpointArn", endpointArn())
                .add("CreationTime", creationTime()).add("LastModifiedTime", lastModifiedTime())
                .add("EndpointStatus", endpointStatusAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EndpointName":
            return Optional.ofNullable(clazz.cast(endpointName()));
        case "EndpointArn":
            return Optional.ofNullable(clazz.cast(endpointArn()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "EndpointStatus":
            return Optional.ofNullable(clazz.cast(endpointStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EndpointSummary, T> g) {
        return obj -> g.apply((EndpointSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EndpointSummary> {
        /**
         * <p>
         * The name of the endpoint.
         * </p>
         * 
         * @param endpointName
         *        The name of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointName(String endpointName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the endpoint.
         * </p>
         * 
         * @param endpointArn
         *        The Amazon Resource Name (ARN) of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointArn(String endpointArn);

        /**
         * <p>
         * A timestamp that shows when the endpoint was created.
         * </p>
         * 
         * @param creationTime
         *        A timestamp that shows when the endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * A timestamp that shows when the endpoint was last modified.
         * </p>
         * 
         * @param lastModifiedTime
         *        A timestamp that shows when the endpoint was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * The status of the endpoint.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>OutOfService</code>: Endpoint is not available to take incoming requests.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Creating</code>: <a>CreateEndpoint</a> is executing.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Updating</code>: <a>UpdateEndpoint</a> or <a>UpdateEndpointWeightsAndCapacities</a> is executing.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SystemUpdating</code>: Endpoint is undergoing maintenance and cannot be updated or deleted or re-scaled
         * until it has completed. This maintenance operation does not change any customer-specified values such as VPC
         * config, KMS encryption, model, instance type, or instance count.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RollingBack</code>: Endpoint fails to scale up or down or change its variant weight and is in the
         * process of rolling back to its previous configuration. Once the rollback completes, endpoint returns to an
         * <code>InService</code> status. This transitional status only applies to an endpoint that has autoscaling
         * enabled and is undergoing variant weight or capacity changes as part of an
         * <a>UpdateEndpointWeightsAndCapacities</a> call or when the <a>UpdateEndpointWeightsAndCapacities</a>
         * operation is called explicitly.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>InService</code>: Endpoint is available to process incoming requests.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Deleting</code>: <a>DeleteEndpoint</a> is executing.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code>: Endpoint could not be created, updated, or re-scaled. Use
         * <a>DescribeEndpointOutput$FailureReason</a> for information about the failure. <a>DeleteEndpoint</a> is the
         * only operation that can be performed on a failed endpoint.
         * </p>
         * </li>
         * </ul>
         * <p>
         * To get a list of endpoints with a specified status, use the <a>ListEndpointsInput$StatusEquals</a> filter.
         * </p>
         * 
         * @param endpointStatus
         *        The status of the endpoint.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>OutOfService</code>: Endpoint is not available to take incoming requests.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Creating</code>: <a>CreateEndpoint</a> is executing.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Updating</code>: <a>UpdateEndpoint</a> or <a>UpdateEndpointWeightsAndCapacities</a> is
         *        executing.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SystemUpdating</code>: Endpoint is undergoing maintenance and cannot be updated or deleted or
         *        re-scaled until it has completed. This maintenance operation does not change any customer-specified
         *        values such as VPC config, KMS encryption, model, instance type, or instance count.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RollingBack</code>: Endpoint fails to scale up or down or change its variant weight and is in
         *        the process of rolling back to its previous configuration. Once the rollback completes, endpoint
         *        returns to an <code>InService</code> status. This transitional status only applies to an endpoint that
         *        has autoscaling enabled and is undergoing variant weight or capacity changes as part of an
         *        <a>UpdateEndpointWeightsAndCapacities</a> call or when the <a>UpdateEndpointWeightsAndCapacities</a>
         *        operation is called explicitly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>InService</code>: Endpoint is available to process incoming requests.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Deleting</code>: <a>DeleteEndpoint</a> is executing.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code>: Endpoint could not be created, updated, or re-scaled. Use
         *        <a>DescribeEndpointOutput$FailureReason</a> for information about the failure. <a>DeleteEndpoint</a>
         *        is the only operation that can be performed on a failed endpoint.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To get a list of endpoints with a specified status, use the <a>ListEndpointsInput$StatusEquals</a>
         *        filter.
         * @see EndpointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointStatus
         */
        Builder endpointStatus(String endpointStatus);

        /**
         * <p>
         * The status of the endpoint.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>OutOfService</code>: Endpoint is not available to take incoming requests.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Creating</code>: <a>CreateEndpoint</a> is executing.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Updating</code>: <a>UpdateEndpoint</a> or <a>UpdateEndpointWeightsAndCapacities</a> is executing.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SystemUpdating</code>: Endpoint is undergoing maintenance and cannot be updated or deleted or re-scaled
         * until it has completed. This maintenance operation does not change any customer-specified values such as VPC
         * config, KMS encryption, model, instance type, or instance count.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RollingBack</code>: Endpoint fails to scale up or down or change its variant weight and is in the
         * process of rolling back to its previous configuration. Once the rollback completes, endpoint returns to an
         * <code>InService</code> status. This transitional status only applies to an endpoint that has autoscaling
         * enabled and is undergoing variant weight or capacity changes as part of an
         * <a>UpdateEndpointWeightsAndCapacities</a> call or when the <a>UpdateEndpointWeightsAndCapacities</a>
         * operation is called explicitly.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>InService</code>: Endpoint is available to process incoming requests.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Deleting</code>: <a>DeleteEndpoint</a> is executing.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code>: Endpoint could not be created, updated, or re-scaled. Use
         * <a>DescribeEndpointOutput$FailureReason</a> for information about the failure. <a>DeleteEndpoint</a> is the
         * only operation that can be performed on a failed endpoint.
         * </p>
         * </li>
         * </ul>
         * <p>
         * To get a list of endpoints with a specified status, use the <a>ListEndpointsInput$StatusEquals</a> filter.
         * </p>
         * 
         * @param endpointStatus
         *        The status of the endpoint.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>OutOfService</code>: Endpoint is not available to take incoming requests.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Creating</code>: <a>CreateEndpoint</a> is executing.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Updating</code>: <a>UpdateEndpoint</a> or <a>UpdateEndpointWeightsAndCapacities</a> is
         *        executing.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SystemUpdating</code>: Endpoint is undergoing maintenance and cannot be updated or deleted or
         *        re-scaled until it has completed. This maintenance operation does not change any customer-specified
         *        values such as VPC config, KMS encryption, model, instance type, or instance count.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RollingBack</code>: Endpoint fails to scale up or down or change its variant weight and is in
         *        the process of rolling back to its previous configuration. Once the rollback completes, endpoint
         *        returns to an <code>InService</code> status. This transitional status only applies to an endpoint that
         *        has autoscaling enabled and is undergoing variant weight or capacity changes as part of an
         *        <a>UpdateEndpointWeightsAndCapacities</a> call or when the <a>UpdateEndpointWeightsAndCapacities</a>
         *        operation is called explicitly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>InService</code>: Endpoint is available to process incoming requests.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Deleting</code>: <a>DeleteEndpoint</a> is executing.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code>: Endpoint could not be created, updated, or re-scaled. Use
         *        <a>DescribeEndpointOutput$FailureReason</a> for information about the failure. <a>DeleteEndpoint</a>
         *        is the only operation that can be performed on a failed endpoint.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To get a list of endpoints with a specified status, use the <a>ListEndpointsInput$StatusEquals</a>
         *        filter.
         * @see EndpointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointStatus
         */
        Builder endpointStatus(EndpointStatus endpointStatus);
    }

    static final class BuilderImpl implements Builder {
        private String endpointName;

        private String endpointArn;

        private Instant creationTime;

        private Instant lastModifiedTime;

        private String endpointStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(EndpointSummary model) {
            endpointName(model.endpointName);
            endpointArn(model.endpointArn);
            creationTime(model.creationTime);
            lastModifiedTime(model.lastModifiedTime);
            endpointStatus(model.endpointStatus);
        }

        public final String getEndpointName() {
            return endpointName;
        }

        @Override
        public final Builder endpointName(String endpointName) {
            this.endpointName = endpointName;
            return this;
        }

        public final void setEndpointName(String endpointName) {
            this.endpointName = endpointName;
        }

        public final String getEndpointArn() {
            return endpointArn;
        }

        @Override
        public final Builder endpointArn(String endpointArn) {
            this.endpointArn = endpointArn;
            return this;
        }

        public final void setEndpointArn(String endpointArn) {
            this.endpointArn = endpointArn;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        public final String getEndpointStatusAsString() {
            return endpointStatus;
        }

        @Override
        public final Builder endpointStatus(String endpointStatus) {
            this.endpointStatus = endpointStatus;
            return this;
        }

        @Override
        public final Builder endpointStatus(EndpointStatus endpointStatus) {
            this.endpointStatus(endpointStatus == null ? null : endpointStatus.toString());
            return this;
        }

        public final void setEndpointStatus(String endpointStatus) {
            this.endpointStatus = endpointStatus;
        }

        @Override
        public EndpointSummary build() {
            return new EndpointSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
